/* tyCoDrv.c - The Force-31 DUSCC serial driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;


/*
modification history
--------------------
01a,17feb89,jcf  written.
*/

/*
DESCRIPTION
This is the driver for the Force-31 DUSCC chip.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    STATUS tyCoDrv ()
.CE
The status returned is OK.

This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.

*/

#include "vxWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "tyLib.h"
#include "config.h"


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char xmtVal;	/* rate to write to the Transmitter Timing Register */
    char rcvVal;	/* rate to write to the Reciever Timing Register */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *rdr;		/* receiver data port */
    char *xdr;		/* transmitter data port */
    char *ttr;		/* transmitter timer register */
    char *rtr;		/* receiver timer register */
    char *ier;		/* interrupt enable register */
    char *gsr;		/* general status register */
    char rrdy; 		/* receiver ready bit */
    char xrdy;		/* transmitter ready bit */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE,
	DUSCC_RXFIFOA     (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_TXFIFOA     (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_TTRA        (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_RTRA        (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_IERA 	  (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_GSR 	  (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_GSR_A_RXRDY,
	DUSCC_GSR_A_TXRDY },
    {{{{NULL}}}, FALSE,
	DUSCC_RXFIFOB     (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_TXFIFOB     (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_TTRB        (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_RTRB        (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_IERB        (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_GSR 	  (FRC31_DUSCC1_BASE_ADRS),
	DUSCC_GSR_B_RXRDY,
	DUSCC_GSR_B_TXRDY },
    };

#define CHANNEL_A	0		/* for structure references */
#define CHANNEL_B	1		/* for structure references */

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =
    {
    {50,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_50,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_50 },
    {75,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_75,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_75 },
    {110,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_110,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_110},
    {134,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_134_5,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_134_5},
    {150,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_150,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_150},
    {200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_200},
    {300,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_300,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_300},
    {600,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_600,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_600},
    {1050,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_1050,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_1050},
    {1200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_1200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_1200},
    {2000,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_2000,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_2000},
    {2400,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_2400,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_2400},
    {4800,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_4800,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_4800},
    {9600,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_9600,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_9600},
    {19200,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_19200,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_19200},
    {38400,
	DUSCC_TTR_CLK_BRG | DUSCC_TTR_BAUD_38400,
	DUSCC_RTR_CLK_BRG | DUSCC_RTR_BAUD_38400} };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoTxInt ();
LOCAL VOID tyCoRxInt ();

/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the driver, sets up interrupt vectors,
* and performs hardware initialization of the ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS tyCoDrv ()

    {
    /* check if driver already installed */
     
    if (tyCoDrvNum > 0)
	return (OK);
		    
    /* Connect channel A's vectors */

    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC1 + DUSCC_INT_A_RXRDY),
		tyCoRxInt, CHANNEL_A);
    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC1 + DUSCC_INT_A_TXRDY),
		tyCoTxInt, CHANNEL_A);

    /* Connect channel B's vectors */

    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC1 + DUSCC_INT_B_RXRDY),
		tyCoRxInt, CHANNEL_B);
    intConnect (INUM_TO_IVEC(INT_VEC_DUSCC1 + DUSCC_INT_B_TXRDY),
		tyCoTxInt, CHANNEL_B);

    tyCoHrdInit ((ULONG) FRC31_DUSCC1_BASE_ADRS);	/* initialize DUSCC1 */

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);


    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}
				     
    /* if this device already exists, don't create it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)

	return (ERROR);

    *(tyCoDv [channel].ier) = DUSCC_IER_RXRDY;

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyCoDv [channel].created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the DUSCC
*
* This routine initializes the DUSCC for the VxWorks environment. 
*
* This routine must be called in supervisor mode, since it accesses I/O space.
*/

LOCAL VOID tyCoHrdInit (dusccBaseAdrs)
    ULONG dusccBaseAdrs;

    {
    int oldlevel;		/* current interrupt level mask */

    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity, set for 9600 baud */

	/* channel A initialiazation */

    *DUSCC_CMR1A (dusccBaseAdrs)= DUSCC_CMR1_ASYNC;
    *DUSCC_CMR2A (dusccBaseAdrs)= DUSCC_CMR2_DTI_POLL_OR_INT;
    *DUSCC_TPRA (dusccBaseAdrs)	= DUSCC_TPR_ASYNC_1 | DUSCC_TPR_8BITS;
    *DUSCC_TTRA (dusccBaseAdrs)	= DUSCC_TTR_CLK_BRG |
				  DUSCC_TTR_BAUD_9600;
    *DUSCC_RPRA (dusccBaseAdrs)	= DUSCC_RPR_8BITS;
    *DUSCC_RTRA (dusccBaseAdrs)	= DUSCC_RTR_CLK_BRG |
				  DUSCC_RTR_BAUD_9600;
    *DUSCC_OMRA (dusccBaseAdrs)	= DUSCC_OMR_TXRDY_FIFO_EMPTY |
			 	  DUSCC_OMR_TX_RES_CHAR_LENGTH_TPR;
    *DUSCC_CCRA (dusccBaseAdrs)	= DUSCC_CCR_TX_ENABLE_TX;
    *DUSCC_CCRA (dusccBaseAdrs)	= DUSCC_CCR_RX_ENABLE_RX;

	/* channel B initialiazation */

    *DUSCC_CMR1B (dusccBaseAdrs)= DUSCC_CMR1_ASYNC;
    *DUSCC_CMR2B (dusccBaseAdrs)= DUSCC_CMR2_DTI_POLL_OR_INT;
    *DUSCC_TPRB (dusccBaseAdrs)	= DUSCC_TPR_ASYNC_1 | DUSCC_TPR_8BITS;
    *DUSCC_TTRB (dusccBaseAdrs)	= DUSCC_TTR_CLK_BRG |
				  DUSCC_TTR_BAUD_9600;
    *DUSCC_RPRB (dusccBaseAdrs)	= DUSCC_RPR_8BITS;
    *DUSCC_RTRB (dusccBaseAdrs)	= DUSCC_RTR_CLK_BRG |
                                  DUSCC_RTR_BAUD_9600;
    *DUSCC_OMRB (dusccBaseAdrs)	= DUSCC_OMR_TXRDY_FIFO_EMPTY |
				  DUSCC_OMR_TX_RES_CHAR_LENGTH_TPR;
    *DUSCC_CCRB	(dusccBaseAdrs)	= DUSCC_CCR_TX_ENABLE_TX;
    *DUSCC_CCRB	(dusccBaseAdrs)	= DUSCC_CCR_RX_ENABLE_RX;

    /* all interrupts are masked out: the receiver interrupt will be enabled
       in the tyCoDevCreate */

    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to DUSCC
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for DUSCC
*
* This routine fields all read calls to the DUSCC.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for DUSCC
*
* This routine fields all write calls to the DUSCC.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to tyIoctl.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int i;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (i = 0; i < NELEMENTS (baudTable); i++)
		{
		if (baudTable [i].rate == arg)	/* lookup baud rate value */
		    {
		    *pTyCoDv->ttr = baudTable [i].xmtVal;
		    *pTyCoDv->rtr = baudTable [i].rcvVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}
    return (status);
    }

/*******************************************************************************
*
* tyCoRxInt - handle a receiver interrupt
*
* This routine gets called at interrupt level to handle a receiver A interrupt.
*/

LOCAL VOID tyCoRxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], (char) *tyCoDv [channel].rdr);
    *(tyCoDv [channel].gsr) = tyCoDv [channel].rrdy;
    }
/*******************************************************************************
*
* tyCoTxInt - handle a transmitter interrupt
*
* This routine gets called from interrupt level to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID tyCoTxInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))
	{
	*(tyCoDv [channel].xdr) = outChar;
	*(tyCoDv [channel].gsr) = tyCoDv [channel].xrdy;
	}
    else
	{
	/* turn off the transmitter */

	*(tyCoDv [channel].ier) &= ~DUSCC_IER_TXRDY;
	}
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine for DUSCC.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    char outChar;

    if ((pTyCoDv->created) &&
	(tyITx ((TY_DEV_ID) pTyCoDv, &outChar) == OK))
	{
	*(pTyCoDv->ier) |= DUSCC_IER_TXRDY;
	*(pTyCoDv->xdr) = outChar;
	*(pTyCoDv->gsr) = pTyCoDv->xrdy;
	}
    }
