/* sysLib.c - Integrated Solutions - Eagle-5 system dependent library */

static char    *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
 * DESCRIPTION This library contains a set of routines to manipulate the
 * primary functions of the CPU board.  The goal is to provide a
 * board-independant interface on which vxWorks and application code can be
 * built in a system-independant way. Not every feature of every board is
 * supported by this library; a particular board may have various extensions
 * to the capabilities described here. Also not every board will support all
 * the functions provided by this library. And some boards provide some of
 * the functions of this library with hardware switches, jumpers, or PALs,
 * instead of software controllable registers. 
 *
 * The funtions addressed here include: 
 *
 * initialization functions: - initialize hardware to known state - identify the
 * system 
 *
 * memory/address space functions: - get on-board memory limit - map from local
 * to bus and bus to local address spaces - enable/disable cache memory -
 * set/get non-volatile RAM 
 *
 * bus interrupt functions: - enable/disable bus interrupt levels - generate bus
 * interrupts 
 *
 * serial channel functions (see tyCoDrv): - enable/disable serial channel
 * interrupts - set serial channel baud rates - get/put bytes from a serial
 * channel 
 *
 * clock/timer functions: - enable/disable timer interrupts - set timer periodic
 * rate 
 *
 * mailbox/location monitor functions: - enable mailbox/location monitor
 * interrupts 
 *
 * NOT IMPLEMENTED: 
 *	RTS/CTS control of sio port 2 through PIT chip
 *	VME Sysfail interrupts
 *
 */

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "ioLib.h"
#include "config.h"

IMPORT char     end;		/* end of system, inserted automatically by
				 * ld */

/* globals */

int             sysBus = BUS;	/* system bus type (VME_BUS, etc) */
int             sysCpu = CPU;	/* system cpu type (MC680x0) */
int             sysCommType = USE_POLL;	/* system inter-processor comm. type */
char           *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char           *sysExcMsg = EXC_MSG_ADRS;	/* catastrophic message area */
int             sysProcNum;	/* processor number of this cpu */
int             sysFlags;	/* boot flags */
char            sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we
						 * booted */
char            sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we
						 * booted */
/* locals */

LOCAL int       clkTicksPerSecond = 60;	
LOCAL int       sysClkRunning = FALSE;
LOCAL int       auxClkTicksPerSecond = 60;
LOCAL FUNCPTR   sysAuxClkRoutine = NULL;
LOCAL int       sysAuxClkArg = NULL;
LOCAL int       auxClkRunning = FALSE;

LOCAL char     *memTop = NULL;
LOCAL FUNCPTR   usartTxInt = NULL;
LOCAL FUNCPTR   usartRxInt = NULL;
LOCAL FUNCPTR   mailboxRoutine = NULL;
LOCAL int   	mailboxArg = NULL;
LOCAL int   	mailboxEnabled = 0;
LOCAL FUNCPTR   ipiRoutine = NULL;
LOCAL int 	ipiArg = 0;



/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "ISI Eagle-5"
*/
char *
sysModel()
{
    return ("ISI Eagle-5");
}

/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* Save rom monitor address.
* Initialize Memory Management Unit (MMU).
*/
VOID 
sysHwInit()
{
    (void) sysMemTop();		/* to make sure memTop gets calculated */

    /* turn off interrupts on the DUART 
     */
    sysImrSet(0, 0);

    /* temporarily connect the interprocessor interrupt
     * because we get these on reset
     */
    nmi_connect();

    /*
     * initialize the PIT and IRQ handler
     */
    sysPitInit();
    sysIhanInit();

    /*
     * initialize VSB
     */
    sysVsbInit(); 

    /* initialize LED display to 0 */
    sysSetDisplay (sysProcNumGet());
    sysDisplayEnable();
}

/*******************************************************************************
*
* sysVsbInit - Initialize any VSB boards
*
* This does only the most basic initialization
*    1. zero memory
*    2. set parity
*
* Any further functionality for a VSB board is board dependent
*/
LOCAL VOID
sysVsbInit ()
{
    /* compute size of VSB memory, initialize any other VSB features
     * if a board exists
     */
    if (vsbSize() > 0) {
    }
}

/*******************************************************************************
*
* sysPitInit - Initialize the Parallel Interface and clock chip
*/
LOCAL VOID
sysPitInit ()
{
    unsigned char tmp = *PIT_GCR;

    /* reset GCR
     */
    *PIT_GCR = tmp & 0xcf;
    *PIT_GCR = 0x00;

    *PIT_SRR = 0x08; 		/* level 3 semaphore interrupt */
				/* disable vector */
    *PIT_PIVR = INT_NUM_SEM; 

    *PIT_ACR = 0xa8;		/* sub 1x H2 asserted */
    *PIT_BCR = 0x82;		/* sub 1x H3 enabled */

    *PIT_ADDR = 0xff;		/* port A all output */
    *PIT_BDDR = 0x67;		/* port B some output */
    *PIT_BDR = SEM7_ENAB | ABORT_ENAB | ARB_RWD | AM5_NORM | AM4_NORM;
    *PIT_CDDR = 0x2f;

    *PIT_GCR = 0x20;
 
    *PIT_TCR = TCR_TMR_DIS;	/* disable timer */

    return OK;
}

/*******************************************************************************
*
* sysIhanInit - Initialize the Interrupt handler chip
* 
* This initializes the interrupt handler chip and enables 
* all VME and onboard interrupts at this junction
*/
LOCAL VOID
sysIhanInit()
{
    register int count;

    /* initialize each control register for local interrupts */

    for (count = 1; count < 7; count++) {
	*IHAN_PR = (char)count & 0x7;
	switch (count) {
	case 1:		/* Level 1 SCSI */
	    *IHAN_CR = CR_ACTIVE_LO | CR_LEVEL_SENSE | CR_VEC_ENAB;
	    break;
	case 2:		/* level 2 VSB interrupt */
	    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_ENAB;
	    break;
	case 3:		/* level 3 semaphore interrupt */
	    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_ENAB;
	    break;
	case 4:		/* level 4 DUART interrupt */
	    *IHAN_CR = CR_ACTIVE_LO | CR_LEVEL_SENSE | CR_VEC_ENAB;
	    break;
	case 5:		/* Levels 5 IPIN */
	    *IHAN_CR = CR_ACTIVE_LO | CR_LEVEL_SENSE | CR_VEC_DIS;
	    break;
	case 6:		/* clock interrupt, edge sensitive for square wave */
	    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_ENAB;
	    break;
	case 7:		/* nmi interrupt, edge sensitive */
	    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_ENAB;
	    break;
	}
    }

    /* set the local vector area */
    *IHAN_VR = IHAN_INT_BASE;

    /* enable all local interrupts except the clock and IPIN, which
     * are enabled by sysClkEnable(), and sysIPINIntEnable()
     */
    *IHAN_LMR = 0x9f;

    /* if server, enable all VME bus interrupts */
    /* if cluster, disable all VME bus interrupts */
    if (!sysProcNumGet())
	*IHAN_BMR = 0xff;
    else
	*IHAN_BMR = 0;

    return OK;
}

/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* It starts probing at the end of bss, then tries to read a byte
* at every 4K boundary until it finds one that can't be read.
* This routine must be called in supervisor mode, the first time only.
* After that, it can be called in user mode as well.
*
* RETURNS: address of the first missing byte of memory
*/
char *
sysMemTop()
{
    char            bitBucket;
    char            orig;

    if (memTop == NULL) {
	/*
	 * Look for the top of memory.  Starting at the first even page
	 * boundary after _end, probe each page, then write it and read it
	 * back, just to be sure. 
	 */

	memTop = (char *) ((((int) &end) + (VME_PAGE_SIZE - 1)) &
			   (~(VME_PAGE_SIZE - 1)));

	FOREVER
	{
	    if (vxMemProbe(memTop, READ, 1, &bitBucket) != OK)
		break;

	    orig = *memTop;

	    *memTop = 0x55;
	    if (*memTop != 0x55)
		break;

	    *memTop = orig;

	    /*
	     * just look for local memory 
	     */
	    if (((int) memTop + VME_PAGE_SIZE) > (int) 0x3FFFFE)
		break;

	    memTop += VME_PAGE_SIZE;
	}

	/* memTop is now pointing to the first non-existent address */

    }
    return (memTop);
}

/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*/
STATUS 
sysToMonitor(startType)
int             startType;	/* parameter is passed to ROM to tell it how
				 * to boot * The possible type are defined in
				 * h/sysLib.h */
{
    (*((FUNCPTR) (ROM_BASE_ADRS + 0x8))) (startType);

    return (OK);		/* in case we ever continue from rom monitor */
}

/**********************************************************************
* sysIPINTrap
*
* catch IPIN interrupts
*/
sysIPINTrap()
{
    logMsg ("LEVEL 5: IPIN Interrupt\n");
    return 0;
}

/**********************************************************************
* sysSCSITrap
*
* catch SCSI interrupts
*/
sysSCSITrap()
{
    logMsg ("LEVEL 1: SCSI Interrupt\n");
    return 0;
}

/**********************************************************************
*
* sysVSBTrap
*
* catch VSB interrupts
*/
sysVSBTrap()
{
    logMsg ("LEVEL 2: VSB Interrupt\n");
    return 0;
}

/**********************************************************************
*
* sysNmiTrap
*
* catch nmi interrupts and reboot if we are not processor 0
*/
sysNmiTrap ()
{
    register char tmp;

    tmp = *PIT_BDR;	/* get NMI information */
    tmp &= NMI_MASK;

    if (tmp == NMI_MASK) {		/* level 7 semaphore interrupt */
	logMsg ("NMI: SEMAPHORE IRQ\n");
	if (mailboxEnabled && mailboxRoutine)
	    (*mailboxRoutine) (mailboxArg);
	return;
    }
    if (!(tmp & NMI_ABORT)) {
	logMsg ("NMI: ABORT\n");
    }
    if (!(tmp & NMI_ACFAIL)) {
	logMsg ("NMI: ACFAIL\n");
    }
    return OK;
}

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* System clock interrupts are not enabled.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/
STATUS sysClkConnect(routine, arg)
FUNCPTR         routine;	/* routine called at each system clock
				 * interrupt */
int             arg;		/* argument with which to call routine */

{
    return (intConnect(INUM_TO_IVEC(INT_NUM_CLOCK), routine, arg));
}

/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/
VOID 
sysClkDisable()
{
    *PIT_TCR = TCR_TMR_DIS;
    *IHAN_LMR &= ~0x40; 
    sysClkRunning = FALSE;
    return(OK);
}

/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
* 
* The PIT chip is setup for SQUARE_WAVE mode, with the SW input
* to the IHAN chip which supplies the vector for the clock
* interrupt.  This way we don't have to explicitly clear the
* PIT chip every time we get a clock interrupt.
*/
VOID 
sysClkEnable()
{
    unsigned long tmp = 8000000/clkTicksPerSecond/32/2;

    /* setup preload register */

    *PIT_CPRH = (tmp >> 16) & 0xff;
    *PIT_CPRM = (tmp >> 8) & 0xff;
    *PIT_CPRL = tmp & 0xff;

    /*
     * Enable the clock. 
     */
    *PIT_TIVR = INT_NUM_CLOCK;
    *PIT_TCR = TCR_TMR_ENA; 

    *IHAN_LMR |= 0x40; 

    sysClkRunning = TRUE;
    return(OK);
}

/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/
int 
sysClkRateGet()
{
    return (clkTicksPerSecond);
}

/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkRateGet (2)
*
* ARGSUSED
*/
VOID 
sysClkRateSet(ticksPerSecond)
int             ticksPerSecond;	/* number of clock interrupts per second */
{
    clkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning == TRUE) {/* restart clock with new rate */
	sysClkDisable();
	sysClkEnable();
    }
}

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect(routine, arg)
FUNCPTR         routine;	/* routine called at each auxiliary clock
				 * interrupt */
int             arg;		/* argument with which to call routine */
{
    sysAuxClkRoutine = routine;
    sysAuxClkArg = arg;

    return (OK);
}

/*******************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine.
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/
VOID 
sysAuxClkDisconnect()
{
    IMPORT VOID     logMsg();

    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable();

    /* connect dummy routine, just in case */

    sysAuxClkConnect(logMsg, (int) "auxiliary clock interrupt\n");
}

/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/
VOID 
sysAuxClkDisable()
{
    char            dummy;

    sysImrSet(0, VME_U_INT_TIMER);
    auxClkRunning = FALSE;

    /* can't stop C/T -- change clock source */
    dummy = (char) VME_U_ACR_CTR_EXT;
    *VME_U_ACR = dummy;

    if (*VME_U_STOP_TMR);	/* EOI on the timer */
}

/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/
int 
sysAuxClkEnable()
{
    unsigned int    ctr;

    /*
     * calculate the divide ration, and write it to the timer chip.  If
     * ticksPerSecond == 0, write a vl of 0 to stop the clock. 
     */

    ctr = (auxClkTicksPerSecond == 0) ? 0
	: (3686400 / (2 * auxClkTicksPerSecond));

    if ((ctr & 0xffff0000) == 0) {
	*VME_U_ACR = (char) VME_U_ACR_TMR_XTAL;
	*VME_U_CTUR = (ctr & 0xff00) >> 8;
	*VME_U_CTLR = (ctr & 0x00ff);

	sysImrSet(VME_U_INT_TIMER, 0);
	if (*VME_U_START_TMR);	/* EOI on the timer */

	/* set up USART timer to use crystal */
	*VME_U_ACR = VME_U_ACR_TMR_XTAL;

	auxClkRunning = TRUE;
	return (OK);
    } else {
	printErr("sysAuxClkEnable: invalid clock rate (%d)\n",
		 auxClkTicksPerSecond);
	auxClkRunning = FALSE;
	return (ERROR);
    }
}

/*******************************************************************************
*
* sysAuxClkRateGet - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/
int 
sysAuxClkRateGet()
{
    return (auxClkTicksPerSecond);
}

/*******************************************************************************
*
* sysAuxClkRateSet - set the auxiliary timer frequency
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkRateGet (2), sysAuxClkEnable (2)
*/
VOID 
sysAuxClkRateSet(ticksPerSecond)
int             ticksPerSecond;	/* number of clock interrupts per second; 0
				 * stops the clock */

{
    if (ticksPerSecond == 0) {
	auxClkRunning = FALSE;
	sysClkDisable();
	return (OK);
    }
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning) {
	sysAuxClkDisable();
	sysAuxClkEnable();
    }
}

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/
STATUS sysLocalToBusAdrs(adrsSpace, localAdrs, pBusAdrs)
int             adrsSpace;	/* bus address space in which busAdrs
				 * resides; use address modifier codes as
				 * defined in vme.h, such as
				 * VME_AM_STD_SUP_DATA */
char           *localAdrs;	/* local address to convert */
char          **pBusAdrs;	/* where to return bus address */

{
    switch (adrsSpace) {
    case VME_AM_SUP_SHORT_IO:
    case VME_AM_USR_SHORT_IO:
	/* can't be done, no shortio onboard */
	return (ERROR);

    case VME_AM_STD_SUP_ASCENDING:
    case VME_AM_STD_SUP_PGM:
    case VME_AM_STD_SUP_DATA:
    case VME_AM_STD_USR_ASCENDING:
    case VME_AM_STD_USR_PGM:
    case VME_AM_STD_USR_DATA:
	/* standard addressing is decoded by a PAL */
	if (localAdrs > (char *) sysMemTop())
	    return (ERROR);
#ifdef DOOLD
	if (!sysProcNumGet())
	    *pBusAdrs = (char *) (VME_PAL_ADRS | (int) localAdrs);
	else
	    return (ERROR);
#endif DOOLD
	*pBusAdrs = (char *) 
	    (((sysProcNumGet()&3) << 22) | (int) localAdrs & 0x3fffff);
	break;

    case VME_AM_EXT_SUP_ASCENDING:
    case VME_AM_EXT_SUP_PGM:
    case VME_AM_EXT_SUP_DATA:
    case VME_AM_EXT_USR_ASCENDING:
    case VME_AM_EXT_USR_PGM:
    case VME_AM_EXT_USR_DATA:
	if (localAdrs > (char *) sysMemTop())
	    return (ERROR);
	*pBusAdrs = (char *) 
	    (((sysProcNumGet()&3) << 28) | (int) localAdrs & 0x0fffffff); 
	break;

    default:
	return (ERROR);
    }
    return (OK);
}

/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/
STATUS 
sysBusToLocalAdrs(adrsSpace, busAdrs, pLocalAdrs)
int             adrsSpace;	/* bus address space in which busAdrs
				 * resides; use address modifier codes as
				 * defined in vme.h, such as
				 * VME_AM_STD_SUP_DATA */
char           *busAdrs;	/* bus address to convert */
char          **pLocalAdrs;	/* where to return local address */

{
    switch (adrsSpace) {
    case VME_AM_SUP_SHORT_IO:
    case VME_AM_USR_SHORT_IO:
	*pLocalAdrs = (char *) (VME_SHORTIO | (int) busAdrs);
	return (OK);

    case VME_AM_STD_SUP_ASCENDING:
    case VME_AM_STD_SUP_PGM:
    case VME_AM_STD_SUP_DATA:
    case VME_AM_STD_USR_ASCENDING:
    case VME_AM_STD_USR_PGM:
    case VME_AM_STD_USR_DATA:
	*pLocalAdrs = (char *) (VME_STANDARD | (int) busAdrs);
	return (OK);

    case VME_AM_EXT_SUP_ASCENDING:
    case VME_AM_EXT_SUP_PGM:
    case VME_AM_EXT_SUP_DATA:
    case VME_AM_EXT_USR_ASCENDING:
    case VME_AM_EXT_USR_PGM:
    case VME_AM_EXT_USR_DATA:
	/* can't generate extended addresses below 4 Mbytes */
	/* or above 0xeffeffff */
	if (busAdrs < (char *) 0x00400000 || busAdrs > (char *) 0xeffeffff)
	    return (ERROR);

	*pLocalAdrs = (char *) (VME_EXTENDED | (int) busAdrs);
	return (OK);

    default:
	return (ERROR);
    }
}

/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level by masking
* the interrupt at the IHAN chip.
*
* RETURNS: OK
*
* ARGSUSED
*/
STATUS 
sysIntDisable(intLevel)
int             intLevel;	/* interrupt level to disable */

{
    register char tmp;

    if (intLevel > 7 || intLevel < 1)
	return (ERROR);
    tmp = *IHAN_BMR;
    *IHAN_BMR = tmp & ~(1 << intLevel);
    return (OK);
}

/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level by unmasking
* the interrupt at the IHAN chip.
*
* RETURNS: OK
*
* ARGSUSED
*/
STATUS 
sysIntEnable(intLevel)
int             intLevel;	/* interrupt level to enable */
{
    register char tmp;

    if (intLevel > 7 || intLevel < 1)
	return (ERROR);
    tmp = *IHAN_BMR;
    *IHAN_BMR = tmp | (1 << intLevel);
    return (OK);
}

/*******************************************************************************
*
* sysBusIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE EC5:
* If jumpered correctly, the VME interrupts will be acknowledged automatically.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/
int 
sysBusIntAck(intLevel)
int             intLevel;	/* interrupt level to acknowledge */

{
    return (NULL);
}

/*******************************************************************************
*
* sysIpiConnect - connect routine to the interprocessor interrupt
*
* This routine connects the given function to the interprocessor interrupt.
* Level 3 Semaphores are used for this interprocessor interrupt.
* Level 7 NMI Semaphores are used for the generic mailbox interrupts.
* It is expected that this will be used only with the vb driver, so
* there is no corresponding disconnect function.
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/
sysIpiConnect(routine, arg)
FUNCPTR routine;
int arg;
{
    ipiRoutine = routine;
    ipiArg = arg;
    return;
}

/**********************************************************************
*
* sysSemInt
*
* catch level 3 semaphore interrupts used for vb interprocessor 
* communication
*/
sysSemInt (arg)
int arg;
{
    *PIT_PSR = 0x04;		/* clear interrupt */
    if (ipiRoutine)
	(*ipiRoutine) (ipiArg);
    return;
}

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
* Level 7 NMI Semaphores are used for this generic mailbox interrupt.
* Level 3 Semaphores are used for the interprocessor interrupts.
*
* SEE ALSO: intConnect (2)
*
* ARGSUSED
*/
STATUS sysMailboxConnect(routine, arg)
FUNCPTR         routine;	/* routine called at each mailbox interrupt */
int             arg;		/* argument with which to call routine */

{
    mailboxRoutine = routine;
    mailboxArg = arg;
    return (OK);
}

/*******************************************************************************
*
* sysMailboxDisconnect - disconnect routine from the mailbox interrupt
*
*/
STATUS sysMailboxDisconnect()

{
    mailboxRoutine = NULL;
    mailboxArg = NULL;
    return (OK);
}

/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* ARGSUSED
*/
STATUS 
sysMailboxEnable(mailboxAdrs)
char           *mailboxAdrs;	/* mailbox address */

{
    mailboxEnabled = 1;
    return (OK);
}

/*******************************************************************************
*
* sysMailboxDisable - disable mailbox interrupt
*
* This routine disables the mailbox interrupt.
*
*/
STATUS 
sysMailboxDisable()

{
    mailboxEnabled = 0;
    return (OK);
}

/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/
sysProcNumGet()
{
    char    n = 0;

    nvram_readbytes(NVRAM_PROC_NUMBER, &n, 1);
    return (int)n;
}

/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
*/
VOID 
sysProcNumSet(procNum)
int             procNum;	/* processor number */

{
    char    n = 0;

    n = (char)procNum;
    nvram_writebytes(NVRAM_PROC_NUMBER, &n, 1);
    sysProcNum = procNum;
    return procNum;
}

/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/
BOOL 
sysBusTas(addr)
char           *addr;		/* address to be tested and set */

{
    return (vxTas(addr));
}

/*******************************************************************************
*
* sysImrSet - set and clear bits in the USART's interrupt mask register
*
* This routine sets and clears bits in the USART's IMR.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the USART.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* NOTE: This routine must be called in supervisor mode.
*/
VOID sysImrSet(setBits, clearBits)
char            setBits;	/* which bits to set in the IMR */
char            clearBits;	/* which bits to clear in the IMR */

{
    static char     imr = 0;	/* our copy of the interrupt mask reg */

    imr = (imr | setBits) & (~clearBits);
    *VME_U_IMR = imr;
}

/*******************************************************************************
*
* sysUsartInt - interrupt level processing for VME68k20 USART
*
* This routine handles an interrupt from the on-board USART.  The interrupt
* is decoded by checking the uart's interrupt status reg, and the appropriate 
* routine invoked.  
*
* This routine is located here because we use the extra timer on the USART
* chip as the auxiliary timer, which is handled here.
*/
LOCAL VOID 
sysUsartInt()
{
    FAST char       isr = *VME_U_ISR;

    if (isr & VME_U_INT_TXA) {
	/*
	 * transmitter channel A interrupt.  call USART routine, if there is
	 * one, else issue an EOI 
	 */

	if (usartTxInt == NULL)
	    sysImrSet(0, VME_U_INT_TXA);	/* EOI */
	else
	    (*usartTxInt) (0);
    }
    if (isr & VME_U_INT_TXB) {
	/*
	 * transmitter channel B interrupt.  call USART routine, if there is
	 * one, else issue an EOI 
	 */

	if (usartTxInt == NULL)
	    sysImrSet(0, VME_U_INT_TXB);	/* EOI */
	else
	    (*usartTxInt) (1);
    }
    if ((isr & VME_U_INT_RXA) && (usartRxInt != NULL))
	(*usartRxInt) (0);	/* receiver channel A */

    if ((isr & VME_U_INT_RXB) && (usartRxInt != NULL))
	(*usartRxInt) (1);	/* receiver channel B */

    if (isr & VME_U_INT_TIMER) {/* on-chip timer */
	if (sysAuxClkRoutine != NULL)
	    (*sysAuxClkRoutine) (sysAuxClkArg);

	if (*VME_U_STOP_TMR);	/* EOI on the timer */
    }
}

/*******************************************************************************
*
* sysUsartConnect - set interrupt routines for the USART
*
* This routine connects the serial driver interrupt routines to the
* interrupt routine.  It should be called once, from the serial driver.
*/
VOID 
sysUsartConnect(recvRoutine, xmitRoutine)
FUNCPTR         recvRoutine;	/* receive routine */
FUNCPTR         xmitRoutine;	/* transmit routine */
{
    static BOOL     usartConnected = FALSE;

    if (!usartConnected)
	(void) intConnect(INUM_TO_IVEC(INT_NUM_CONSOLE), sysUsartInt, NULL);

    usartConnected = TRUE;
    usartTxInt = xmitRoutine;
    usartRxInt = recvRoutine;
}

/************************************************************************/
/*
 * vdma resource allocator for the Eagle-5 board under vxWorks 
 *
 * the Eagle-5 doesn't have vdma capabilities, so these routines are really
 * no-ops 
 * 
 */

/************************************************************************/
/*
 * sysInitVdma 
 */
sysInitVdma()
{
    return (0);
}

/************************************************************************/
/*
 * sysSetVdma 
 *
 * set the vdma registers for a buffer at location buff of size nbytes 
 *
 * returns address of window requested which is the same as the address
 * requested with the standard vme window offset
 */
char           *
sysSetVdma(buff, nbytes)
char           *buff;
int             nbytes;
{
    char *pBusAdrs;

    if (sysLocalToBusAdrs(VME_AM_STD_SUP_DATA, buff, &pBusAdrs) == ERROR) {
	logMsg ("sysSetVdma: Error!!\n");
	return NULL;
    }
    return pBusAdrs;
}

/************************************************************************/
/*
 * sysGiveVdma 
 *
 */
sysGiveVdma(vmeoff)
int             vmeoff;
{
    return (0);
}

/************************************************************************/
/*
 * sysDataCacheFlush 
 */
sysDataCacheFlush()
{
    return (OK);
}

/************************************************************************/
/*
 * sysDataCacheFlushEntry 
 */
sysDataCacheFlushEntry(addr, num)
char           *addr;
int             num;
{
    return (OK);
}


/************************************************************************
 * 
 * Non-volatile RAM manipulation 
 * utilities for onboard non-volatile ram of Eagle-5 CPU 
 */

char            nvram_image[NVRAM_BYTES];
int             image_valid = 0;

/*
 * read all 128 bytes of the nvram to location 'addr' 
 */
nvram_read(addr)
char           *addr;
{
    register int    count;
    register unsigned char tmp;

    if (!image_valid) {
	for (count = 0; count < NVRAM_BYTES; count++) {
	    tmp = *((char *) (NVRAM_BASE + (2 * count))) & 0xf0;
	    tmp += (0xf0 & *((char *) (NVRAM_BASE + (2 * count + 1)))) >> 4;
	    nvram_image[count] = tmp;
	}
	image_valid = 1;
    }
    bcopy(nvram_image, addr, NVRAM_BYTES);
    return (0);
}

/*
 * write all 128 bytes to the nvram from location 'addr' 
 */
nvram_write(addr)
char           *addr;
{
    register int    count;

    for (count = 0; count < NVRAM_BYTES; count++) {
	*((char *) (NVRAM_BASE + (2 * count))) = addr[count];
	*((char *) (NVRAM_BASE + (2 * count + 1))) = addr[count] << 4;
    }
    *((char *) (NVRAM_SAVE)) = 0xff;	/* save contents of ram */
    image_valid = 0;
    return (0);
}

/*
 * read 'nbytes' from the nvram starting at location 'offset' into address
 * 'addr' 
 */
nvram_readbytes(offset, addr, nbytes)
int             offset, nbytes;
char           *addr;
{
    register int    count;

    if (!image_valid) {
	nvram_read(nvram_image);
	image_valid = 1;
    }
    for (count = 0; count < nbytes; count++)
	addr[count] = nvram_image[offset + count];
    return (0);
}

/*
 * write 'nbytes' to the nvram starting at location 'offset' from address
 * 'addr' 
 */
nvram_writebytes(offset, addr, nbytes)
int             offset, nbytes;
char           *addr;
{
    register int    count;

    if (!image_valid)
	nvram_read(nvram_image);

    for (count = 0; count < nbytes; count++)
	nvram_image[offset + count] = addr[count];
    nvram_write(nvram_image);
    image_valid = 1;
    return (0);
}

/* compatibility */
sysSetNvRam(pnvaddr)
char           *pnvaddr;
{
    return nvram_write(pnvaddr);
}

sysGetNvRam(pnvaddr)
char           *pnvaddr;
{
    return nvram_read(pnvaddr);
}

/* 
 * access real-time clock (battery backup)
 */

binary_to_bcd(i)
{
    return((i / 10) << 4 | (i % 10));
}
	
bcd_to_binary(i)
unsigned char i;
{
    return((i >> 4) * 10 + (i & 0x0f));
}

char *day_string[] =  {
	"",
	"Sunday",
	"Monday",
	"Tuesday",
	"Wednesday",
	"Thursday",
	"Friday",
	"Saturday",
};

char *month_string[] = {
	"",
	"January",
	"Feburary",
	"March",
	"April",
	"May",
	"June",
	"July",
	"August",
	"September",
	"October",
	"November",
	"December",
};

/* wait for the real-time clock chip to become
 * not busy
 */
clkWait ()
{
    char tmp;

    while (1) {
	clkDelay ();
	tmp = *PIT_ADR;		/* data register A, bit 7 has busy info */
	if (0x80 & tmp)		/* bit will be zero when chip is busy */
	    break;
    }
}

clkDelay ()
{
    int i = 100;
    while (--i);
}


/* select address to write to on real-time clock chip
 */
clkSelAddr (addr)
char addr;
{
    *PIT_ADDR = 0x70;		/* select all bits input Data/Dir A*/
    *PIT_ADR =  0x40;		/* select all bits input Data A*/
    clkWait ();
    addr &= 0xf;
    *PIT_ADR =  addr;		/* Set Data A to chip register number */
    *PIT_ADDR = 0x7f;		/* select bits output Data/Dir A*/
    addr |= 0x10;
    *PIT_ADR =  addr;		/* Set Data A to write address  */
    clkDelay ();
    addr &= 0xf;
    *PIT_ADR =  addr;		/* Set Data A to write address  */
    clkDelay ();
}

/* read and write nibbles to real-time clock chip
 * with offset
 * can't be sure what these do, they came from E5's assembler code
 */
writeClkNib (offset, nib)
char offset, nib;
{
   clkSelAddr (offset);		/* select clock register */
   nib &= 0xf;			/* mask data bits */
   *PIT_ADR = nib;		/* write data */
   nib |= 0x20;			/* select write */
   *PIT_ADR = nib;		/* write data */
   clkDelay();
   *PIT_ADR = 0;		/* clear */
   return 0;
}

readClkNib (offset, pnib)
char offset, *pnib;
{
   clkSelAddr (offset);		/* select clock register */
   *PIT_ADDR = 0x70;
   ;
   *PIT_ADR = 0x40;
   clkDelay ();
   *pnib = *PIT_ADR;
   *pnib &= 0xf;
   clkDelay ();
   *PIT_ADR = 0;		/* clear */
   clkDelay ();
   return 0;
}

sysSetDate(syear, smonth, sdate, sday, shour, sminute, ssecond)
int syear, smonth, sdate, sday, shour, sminute, ssecond;
{
    unsigned char year, month, date, day, hour, minute, second;
    unsigned char leapyear = 0;

    if ((syear > 99) || (smonth < 1 || smonth > 12)
	|| (sdate < 1 || sdate > 31) || (sday < 1 || sday > 7)
	|| (shour > 23) || (sminute > 59) || (ssecond > 59))
	    return ERROR;

    year = binary_to_bcd(syear);
    month = binary_to_bcd(smonth);
    date = binary_to_bcd(sdate);
    day = binary_to_bcd(sday-1);
    hour = binary_to_bcd(shour);
    minute = binary_to_bcd(sminute);
    second = binary_to_bcd(ssecond);
    leapyear = (unsigned char)(syear & 0x3);
    if (leapyear & 1) {
	leapyear ^= 0x2;	/* EXOR ? */
	leapyear <<= 2;
    }

    writeClkNib (RTC_YRHI, year >> 4);
    writeClkNib (RTC_YRLO, year);
    writeClkNib (RTC_MOHI, month >> 4);
    writeClkNib (RTC_MOLO, month);
    writeClkNib (RTC_DYHI, ((date >> 4) & 0x3) | leapyear);
    writeClkNib (RTC_DYLO, date);
    writeClkNib (RTC_WEEK, day);
    writeClkNib (RTC_HRHI, ((hour >> 4) & 0x3) | RTC_24HRS);
    writeClkNib (RTC_HRLO, hour);
    writeClkNib (RTC_MIHI, minute >> 4);
    writeClkNib (RTC_MILO, minute);
    writeClkNib (RTC_SEHI, second >> 4);
    writeClkNib (RTC_SELO, second);
    return OK;
}

sysGetDate (year, month, date, day, hour, minute, second)
int *year, *month, *date, *day, *hour, *minute, *second;
{
    unsigned char tmp1, tmp2;

    readClkNib (RTC_YRHI, &tmp1);
    readClkNib (RTC_YRLO, &tmp2);
    *year = bcd_to_binary((tmp1 << 4) | tmp2);
    readClkNib (RTC_MOHI, &tmp1);
    readClkNib (RTC_MOLO, &tmp2);
    *month = bcd_to_binary((tmp1 << 4) | tmp2);
    readClkNib (RTC_DYHI, &tmp1);
    readClkNib (RTC_DYLO, &tmp2);
    *date = bcd_to_binary(((0x3 & tmp1) << 4) | tmp2);
    readClkNib (RTC_WEEK, &tmp2);
    *day = bcd_to_binary(tmp2+1);
    readClkNib (RTC_HRHI, &tmp1);
    readClkNib (RTC_HRLO, &tmp2);
    *hour = bcd_to_binary(((0x7 & tmp1) << 4) | tmp2);
    readClkNib (RTC_MIHI, &tmp1);
    readClkNib (RTC_MILO, &tmp2);
    *minute = bcd_to_binary((tmp1 << 4) | tmp2);
    readClkNib (RTC_SEHI, &tmp1);
    readClkNib (RTC_SELO, &tmp2);
    *second = bcd_to_binary((tmp1 << 4) | tmp2);
    return OK;
}

#define CENTURY "19"
sysDispDate()
{
    int year, month, date, day, hour, minute, second;

    sysGetDate (&year, &month, &date, &day, &hour, &minute, &second);
    printf("  %s %s %d %s%2d   %2d:%02d:%02d\n",
	day_string[day],
	month_string[month],
	date, CENTURY, year, hour, minute, second);
    return OK;
}

sysSetBootArg(buf)
char *buf;
{
    return nvram_writebytes(NVRAM_BOOT_ARG, buf, NVRAM_BOOT_ARGSIZE);
}

sysGetBootArg(buf)
char *buf;
{
    return nvram_readbytes(NVRAM_BOOT_ARG, buf, NVRAM_BOOT_ARGSIZE);
}

sysDispBootArg()
{
    char buff[NVRAM_BOOT_ARGSIZE+1];

    printf("  Boot arg = ");
    bzero (buff, NVRAM_BOOT_ARGSIZE+1);
    sysGetBootArg (buff);
    printf ("%s\n", buff);
    return OK;
}

/*
 * drive 7-segment display 
 * 
 * 'ch' should be a number between 0x0 and 0xf
 */
sysSetDisplay(ch)
char            ch;
{
    *((char *) (SEG7_ADDR)) = ch << 4;
}

sysDisplayDisable()
{
    char tmp = *PIT_CDR;

    tmp |= DISPLAY_OFF;
    *PIT_CDR = tmp;
}

sysDisplayEnable()
{
    char tmp = *PIT_CDR;

    tmp &= DISPLAY_ON;
    *PIT_CDR = tmp;
}


/* stuff for the LANCE chip */
/*char lnEnetAddr[6] = {0xff, 0x00, 0xff, 0x00, 0xff, 0x00}; /**/

sysIPINIntEnable ()
{
    *IHAN_PR = 0x5;
    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_DIS;
    *IHAN_LMR |= 0x20;
}

sysIPINIntDisable()
{
    *IHAN_PR = 0x5;
    *IHAN_CR = CR_ACTIVE_LO | CR_EDGE_SENSE | CR_VEC_DIS;
    *IHAN_LMR &= ~0x20;
}

sysLanIntEnable  (level) { sysIPINIntEnable (); }
sysLanIntDisable (level) { sysIPINIntDisable (); }

sysLanIntConnect (inum, func, arg)
int inum;
FUNCPTR func;
int arg;
{
    *((char *)(0xfe401401)) = (char)inum;
    (void) intConnect (INUM_TO_IVEC (inum), func, (int)arg); 
}

/*#define SYSDEBUG 	/**/
#ifdef SYSDEBUG
/* pre-kernel startup debugging tools 
 */

k_putdata (string, datum, type)
char *string;
int datum;
char type;
{
    k_puts (string);
    if (type == 'h') {
	k_puts ("0x");
	k_puthex (datum);
    } else
	k_putint (datum);
    k_puts ("\r\n");
}

k_puts (string)
char *string;
{
    int i;
    char *p;

    for (p = string, i = 0; i < 256; p++, i++) {
	if (!*p) return 0;
	k_putc (*p);
    }
    return (-1);
}


k_putint (num)
int num;
{
    char digs[15];
    int i;

    if (num == 0)  {
	k_putc ('0');
	return;
    }

    bzero (digs, 15);
    for (i = 0; i < 15; i++) {
	digs [i] = num % 10 + 0x30;
	num /= 10;
    }
    for (i = 14; i >= 0; i--) {
	if (digs[i] != 0x30)
	    break;
    }
    for (; i >= 0; i--)
	k_putc (digs[i]);
}

k_puthex (num)
unsigned int num;
{
    char digs[15];
    int i;
    static char hdigs [] = "0123456789ABCDEF";

    if (num == 0)  {
	k_putc ('0');
	return;
    }

    bzero (digs, 15);

    for (i = 0; i < 15; i++) {
	digs [i] = hdigs[num & 0xf];
	num >>= 4;
    }
    for (i = 14; i >= 0; i--) {
	if (digs[i] != 0x30)
	    break;
    }
    for (; i >= 0; i--)
	k_putc (digs[i]);
}

k_putc(ch)
char            ch;
{
    register int    i;

    *(char *) 0xfec00003 = ch & 0x7f;
    for (i = 0; i < 10000; i++);
}

#else SYSDEBUG
k_puts(){}
#endif SYSDEBUG

/*******************************************************************************
*
* sysLocalIntDisable - disable local interrupt level
*
* This routine disables the specified LOCAL interrupt level by masking
* the interrupt at the IHAN chip.
*
* RETURNS: OK
*
* ARGSUSED
*/
STATUS 
sysLocalIntDisable(intLevel)
int             intLevel;	/* interrupt level to disable */

{
    register char tmp;

    if (intLevel > 6 || intLevel < 1)
	return (ERROR);		/* NMI stuff in bits 0,7 */
    tmp = *IHAN_LMR;
    *IHAN_LMR = tmp & ~(1 << intLevel);
    return (OK);

} /* end sysLocalIntDisable () */

/*******************************************************************************
*
* sysLocalIntEnable - enable local interrupt level
*
* This routine enables the specified LOCAL interrupt level by unmasking
* the interrupt at the IHAN chip.
*
* RETURNS: OK
*
* ARGSUSED
*/
STATUS 
sysLocalIntEnable(intLevel)
int             intLevel;	/* interrupt level to enable */
{
    register char tmp;

    if (intLevel > 6 || intLevel < 1)
	return (ERROR);		/* NMI stuff in bits 0,7 */
    tmp = *IHAN_LMR;
    *IHAN_LMR = tmp | (1 << intLevel);
    return (OK);
}
