/* sysALib.s - system dependent assembly language routines - ISI version */

/*
modification history
--------------------
*/

/*
DESCRIPTION
This module contains system-dependent routines which must be written
in assembly language.  They perform functions such as locking and
unlocking interrupts, trapping to the ROM monitor, etc.  This module
has conditional compiles to work with either the ISI 68010 or 68020
boards.

INTERNAL: Many routines in this module "link" and "unlk" the "c" frame pointer
a6@ although they don't use it in any way!  This only for the benefit of
the stacktrace facility to allow it to properly trace tasks executing within
these routines.

*/

#define ASMLANGUAGE

#include "vxWorks.h"
#include "sysLib.h"
#include "config.h"

	/* internals */

	.globl	_sysInit	/* start of system code */
	.globl  _sysKernelTrap
	.globl  _sysVwTrap
	.globl  _sysVwTrapRtn
	.globl	_sysCacheEnable

	/* externals */

	.globl	_usrInit	/* system initialization routine */

	.globl  _vme_short
        .globl  _vme_std
        .globl  _vme_ext
        .globl  _vme_window_std
        .globl  _vme_window_ext

        .set    _vme_short,VME_SHORTIO
        .set    _vme_std,  VME_STANDARD
        .set    _vme_ext,  VME_EXTENDED
        .set    _vme_window_std,VME_WINDOW_STD
        .set    _vme_window_ext,VME_WINDOW_EXT

	.globl	_sysTarget
	.set	_sysTarget,TARGET_E5

	.text
	.even

/***********************************************************************
*
* sysInit - start after boot
*
* This is the system start-up entry point for vxWorks in ram.
* It is the first code executed after booting.
* It disables the interrupts, sets up the stack,
* and jumps to the C routine usrInit in usrConfig.c.
*
* The initial stack is set to grow down from sysInit.
* Note that this initial stack is used only by usrInit,
* then is never used again.
* Memory for the stack needs to be accounted for when determining the load
* address of the system.
*
* THIS IS NOT A CALLABLE ROUTINE.

* sysInit ()	/* THIS IS NOT A CALLABLE ROUTINE *

*/

_sysInit:
	movew	#0x3700,sr		/* disable interrupts, turn on M bit */
	movel	#_sysInit,a7		/* set stack to grow down from code */
	movel	#BOOT_WARM_AUTOBOOT,a7@- /* push start type arg = WARM_BOOT */
	jsr	_usrInit		/* never returns - starts up kernel */

/***********************************************************************
*
* sysKernelTrap - trap to kernel function
*
* This routine traps to the kernel. It can only be called from assembly
* language.
* The trap number for the kernel is defined in config.h.
*
* NOMANUAL

* sysKernelTrap ()

*/
_sysKernelTrap:
	trap	#TRAP_KERNEL
	rts

/***********************************************************************
*
* sysVwTrap - trap to vxWorks function
*
* This routine traps to vxWorks. It can only be called from assembly
* language.
* The trap number for vxWorks is defined in config.h.
*
* NOMANUAL

* sysVwTrap ()

*/
_sysVwTrap:
	trap	#TRAP_VXWORKS
	rts

/*
 * This routine calls the routine whose address is in a0.  Since the system
 * is now in supervisor state (since we got here from a trap) this can be
 * used to call a routine in supervisor state.
 * THIS IS NOT A C CALLABLE ROUTINE!!
 */

_sysVwTrapRtn:
	jsr	a0@		/* vector to proper routine. */
	rte			/* return from trap */


/***********************************************************************
*
* sysCacheEnable - enable cache memory
*
* Enable or disable the 68020 on-chip cache.
*
* SEE ALSO: "MC68020 32-Bit Microprocessor User's Manual"

* VOID sysCacheEnable (enable)
*     BOOL enable;		/* enable if TRUE, otherwise disable *

*/
_sysCacheEnable:

	link	a6,#0
	movel	a6@(0x08),d1		/* put enable in d1 */

	cmpl	#1,d1			/* does enable == TRUE? */
	bne	sce0			/* if not, disable cache */
	movel	#0x09,d0		/* else, clear and enable cache */
	bra	sce1

sce0:	movel	#0,d0			/* disable cache */

	/* Enable or disable the cache by writing to the cache control
	   register.  Since the assemblers differ on the syntax for this one
	   (if they even have it), it's done with two .word's.  Yucko. */

sce1:
	.word	0x4e7b		/* movec ... 			*/
	.word	0x0002		/* .....	d0,cacr 	*/

	unlk	a6
	rts

/*
 * connect interprocessor interrupt vector
 */
	.globl	_nmi_connect
_nmi_connect:
	movel	#nmi_hnd,INT_VEC_NMI
	movel	#nmi_hnd,INT_VEC_IPI
	rts
nmi_hnd:
	rte

/*
 * DELAY(n)
 *
 * Busy loop delay for roughly n microseconds
 */
	.globl	_DELAY
_DELAY:	movl	_cpuspeed,d1
    1:	movl	sp@(4),d0
    2:	dbf	d0,2b
	addqw	#1,d0
	subql	#1,d0
	jge	2b
	dbf	d1,1b
	addqw	#1,d1
	subql	#1,d1
	jge	1b
	rts


	.globl	_ts_cmd_buf
	.globl	_ts_cmd_buf_end
_ts_cmd_buf:
	.space	64
_ts_cmd_buf_end:

	.data
	.globl	_cpuspeed
_cpuspeed:	.long	14


/*
* int vsbSize ()
*
* returns the number of bytes of VSB memory
*/
	.text
	.globl  _vsbSize
_vsbSize:
	tstl	_sysVsbDone		/* check for already done */
	jeq	1f
	movl	_sysVsbSize,d0		/* return size of VSB memory */
	rts
1:
	addql	#1,_sysVsbDone		/* mark as done */
	movl	d2,sp@-			/* save d2 on stack*/
	movl	sp,d2			/* save sp in d2 */
	movl	0x8,a1			/* save buserr vector in a1 */
	movl	#_vsb_vec,0x8		/* point buserr to new handler */
	movl	#_sysVsbAddr,a0		/* starting vsb address in a0 */
	movl	#0,d0
	movl	#0,d1
2:
	movl	d0,a0@			/* write to doubtful location */
	nop				/* wait for write to happen */
3:
	cmpl	#1,d1			/* if d1 == 1, we got buserr */
	beq	4f
	addql	#4,a0			/* bump counter and address ptr */
	addql	#4,_sysVsbSize
	bra     2b			/* keep it going */
4:					/* got buserr */
	movl	a1,0x8			/* restore buserr vector from a1 */
	movl	_sysVsbSize,d0		/* return size of VSB memory */
	movl	sp@+,d2			/* restore d2 */
	rts

_vsb_vec:
	movl	#1,d1			/* indicate buserr */
	movl	d2,sp			/* restore stack pointer */
	jmp	3b			/* continue after bad memory */


	.data
	.globl	_sysVsbAddr
	.globl	_sysVsbSize
	.globl	_sysVsbDone

	.set	_sysVsbAddr,VSB_START
_sysVsbSize:	.long	0
_sysVsbDone:	.long	0
	
