/* sysLib.c - DY-4 DVME-134 system dependent library */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
01n,09sep88,gae  documentation.
01m,08jun88,gae  Revised code/documentation for uniformity across targets.
		 sysClkConnect now returns STATUS.
		 fixed sysClk{Disable,Enable}, sysClkRateSet() doesn't enable.
		 fixed sysAuxClk{Disable,Enable}.
		 changed name of sysGetProcNum() to sysProcNumGet().
01i,30may88,dnw  changed to v4 names.
01h,27apr88,ecs  added sysExcMsg.
01l,30may88,dnw  changed to v4 names.
01k,30apr88,gae  fixed MFP usage.  Added sysClk{Dis,En}able().
01j,24mar88,ecs  added sysExcMsg.
01i,20nov87,dnw  changed sysIntGen to take interrupt vec num instead of vec adrs
		 changed to use new defines in config.h
01h,19nov87,ecs  lint.
01g,14nov87,jcf  changed sysToMonitor to take and pass an argument to bootrom
01f,10nov87,ecs  documentation.
01e,05nov87,jcf  removed reset of Excellan board in sysToMon
01d,01nov87,jcf  hushed lint
01c,23oct87,jcf  added sysMailboxConnect, mailboxInt, sysMailboxAck
		 fixed bug in sysMailboxEnable
01b,03oct87,jcf  added sysClkConnect(), sysIntGen(), sysMailboxEnable(),
		   sysBusToLocalAdrs().
		 added more sys... global variables.
		 added sysToMonitor().
		 deleted sysMailBoxAdrs().
		 deleted sysGetProcNum().
		 changed sysLocalToBusAdrs() to take adrs space arg.
		 changed aux clock to log msg if int occurs while unconnected.
01a,13jul87,jcf  created by modifying version 01k of config/hkv2f/sysLib.c
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which VxWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "dy4_134.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

#define	HZ	2457600		/* clock rate */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

/* forward declarations */

LOCAL VOID sysMailboxInt ();	/* mailbox interrupt handler */

/* locals */

LOCAL int sysClkTicksPerSecond  = 60;
LOCAL int auxClkTicksPerSecond  = 60;
LOCAL int sysClkRunning         = FALSE;
LOCAL int auxClkRunning         = FALSE;
LOCAL FUNCPTR sysMailboxRoutine = logMsg;
LOCAL int sysMailboxArg         = (int) "mailbox interrupt\n";


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "DY-4 DVME-134"
*/

char *sysModel ()
    {
    return ("DY-4 DVME-134");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
* The Multi-Function Peripheral chip is initialized.
* The control register is setup.
* The board is taken out of Bus Isolation (BI) mode.
*/

VOID sysHwInit ()

    {
    /* if the id-code is odd, then we have no access to standard VME space,
     * so we flash the LED to tell the user of problem */

    if (*DY4_SR & 1)
	{
	FOREVER
	    sysFlash ((int) 100000);
	}

    /* disable interrupts */

    *MFP_IERA(MFP_BASE_ADR) = 0;	/* clear interrupt enable register A */
    *MFP_IERB(MFP_BASE_ADR) = 0;	/* clear interrupt enable register B */

    *MFP_IMRA(MFP_BASE_ADR) = 0;	/* clear interrupt mask register A */
    *MFP_IMRB(MFP_BASE_ADR) = 0;	/* clear interrupt mask register B */

    /* clear interrupt pending and in-service registers */

    *MFP_IPRA(MFP_BASE_ADR) = 0;	/* interrupt pending register A */
    *MFP_IPRB(MFP_BASE_ADR) = 0;	/* interrupt pending register B */
    *MFP_ISRA(MFP_BASE_ADR) = 0;	/* interrupt in-service register A */
    *MFP_ISRB(MFP_BASE_ADR) = 0;	/* interrupt in-service register B */

    /* initialize vector register on MFP chip */

    *MFP_VR(MFP_BASE_ADR) = MFP_INT_VECT_NUM;

    /* initialize timers */

    *MFP_TACR(MFP_BASE_ADR) = 0x07;	/* timer A delay mode, 200 prescale */
    *MFP_TBCR(MFP_BASE_ADR) = 0;	/* clear timer B control register */
    *MFP_TCDCR(MFP_BASE_ADR) = 0;	/* clear timer C and D control regs. */

    *MFP_TADR(MFP_BASE_ADR) = 205;	/* timer A data register */
					/* - 205==60Hz w/200 prescale */
    *MFP_TBDR(MFP_BASE_ADR) = 0;	/* clear timer B data register */
    *MFP_TCDR(MFP_BASE_ADR) = 0;	/* clear timer C data register */
    *MFP_TDDR(MFP_BASE_ADR) = 0;	/* clear timer D data register */

    /* setup the control register */

    *DY4_CR = CR_INIT;

    /* take the board out of BI mode by writing to an odd word of the LM */

    *DY4_NO_BI = 0;
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* NOTE DY-4 DVME-134:
* This board has 1M of onboard RAM, the top 1024 bytes of which are
* the location monitor addresses (mail boxes).  Sizing memory is, therefore,
* difficult.  We don't want the location monitor residing in the free memory
* pool!
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    return ((char *)(LOCAL_MEM_LOCAL_ADRS + 0x000ffc00));
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility could be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot */
                    /* possible types are defined in sysLib.h            */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
* System clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    int temp;

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int temp;

    /* enable the clock interrupt */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* Write the timer value (using a 200 prescale) */

    *MFP_TACR(MFP_BASE_ADR) = 0x07;	/* timer A delay mode, 200 prescale */
    *MFP_TADR(MFP_BASE_ADR) = HZ / (200 * sysClkTicksPerSecond);

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkRateGet - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkRateSet (2)
*/

int sysClkRateGet ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkRateSet - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkRateGet (2), sysClkEnable (2)
*/

VOID sysClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    return (intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM + MFP_INT_TIMER_B),
			routine, arg));
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    int temp;

    *MFP_TBCR(MFP_BASE_ADR) = 0x00;	/* timer B stopped */

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    /* mask out */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    int temp;

    /* write the timer value (assuming a 200 prescale) */

    *MFP_TBCR(MFP_BASE_ADR) = 0x07;	/* timer B delay mode, 200 prescale */
    *MFP_TBDR(MFP_BASE_ADR) = HZ / (200 * auxClkTicksPerSecond);

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkRateGet - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkRateSet (2)
*/

int sysAuxClkRateGet ()
    
    {
    return (auxClkTicksPerSecond);
    }
/********************************************************************************
* sysAuxClkRateSet - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkRateGet (2)
*/

VOID sysAuxClkRateSet (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
			/* use address modifier codes as defined in vme.h, */
			/* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    char *lowByteStdAdrs;		/* ram location in standard VME space */
    char *lowByteExtAdrs;		/* ram location in extended VME space */

    if ((localAdrs < DY4_RAM_BASE_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */
	*pBusAdrs = localAdrs;
	return (OK);
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 * there is no way to keep the dy4 from dual porting its ram so it
	 * is up to the user to strap multiple dy4's in separate address
	 * regions.  BI mode is inadequate because you lose VME bus access
	 * as well as dual portedness.  So regardless of processor number
	 * we read the status register to determine how the board has been
	 * strapped and place the dual ported ram in STD space at the address
	 * read.
	 */

	lowByteStdAdrs = ((char *) ((UINT)((*DY4_SR) & 0x1f) << 19));
	lowByteExtAdrs = ((char *) ((int) (lowByteStdAdrs) | 0x02000000));

	switch (adrsSpace)
	    {
	    case VME_AM_SUP_SHORT_IO:
	    case VME_AM_USR_SHORT_IO:
		return (ERROR);			/* no short i/o */

	    case VME_AM_STD_SUP_ASCENDING:
	    case VME_AM_STD_SUP_PGM:
	    case VME_AM_STD_SUP_DATA:
	    case VME_AM_STD_USR_ASCENDING:
	    case VME_AM_STD_USR_PGM:
	    case VME_AM_STD_USR_DATA:
		*pBusAdrs = localAdrs + (int) (lowByteStdAdrs);
		return (OK);

	    case VME_AM_EXT_SUP_ASCENDING:
	    case VME_AM_EXT_SUP_PGM:
	    case VME_AM_EXT_SUP_DATA:
	    case VME_AM_EXT_USR_ASCENDING:
	    case VME_AM_EXT_USR_PGM:
	    case VME_AM_EXT_USR_DATA:
		*pBusAdrs = localAdrs + (int) (lowByteExtAdrs);
		return (OK);

	    default:
		return (ERROR);
	    }
	}
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0xffff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (0x01000000 | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    *pLocalAdrs = (char *) (0x02000000 | (int) busAdrs);
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE DY-4 DVME-134:
* Interrupt levels are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* NOTE DY-4 DVME-134:
* VME interrupts are jumperable but not software controllable.
* This routine has no effect.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*******************************************************************************
*
* sysBusIntAck - acknowledge VME bus interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE DY-4 DVME-134:
* If jumpered correctly, the VME interrupts will be acknowledged automatically.
* This routine has no effect.
*
* RETURNS: NULL
*
* ARGSUSED
*/

int sysBusIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*******************************************************************************
*
* sysBusIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* NOTE DY-4 DVME-134:
* This board can't generate a VME bus interrupt.
* This routine has no effect.
*
* RETURNS: ERROR a board is unable to generate a VME bus interrupt
*
* ARGSUSED
*/

STATUS sysBusIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/*******************************************************************************
*
* sysMailboxAck - acknowledge mailbox interrupt
*
* This routine acknowledges the mailbox interrupt.
* The mailbox interrupt must be reset for the next interrupt.
*
* RETURNS: location modified
*/

LOCAL char *sysMailboxAck ()

    {
    return ((char *)(0xffc00 + ((*(UTINY *) DY4_LMF) << 2)));
    }
/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    sysMailboxRoutine = routine;
    sysMailboxArg      = arg;

    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* disable interrupts */
    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* mask interrupts */

    while (!(*MFP_GPIP(MFP_BASE_ADR) & 0x4))	/* empty the Location Monitor */
	(void) sysMailboxAck ();

    return (intConnect (INUM_TO_IVEC (INT_VEC_BP), sysMailboxInt, 0));
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* NOTE DY-4 DVME-134:
* The mailbox address is not settable.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    /* enable mailbox interrupt */

    *MFP_IERB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* enable */
    *MFP_IMRB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* mask */

    if (!(*MFP_GPIP(MFP_BASE_ADR) & 0x4))	/* if an interrupt is pending */
	sysMailboxInt ();			/* do not lose this interrupt */

    return (OK);
    }
/*******************************************************************************
*
* sysMailboxInt - the mailbox interrupt handler
*/

LOCAL VOID sysMailboxInt ()

    {
    if (sysMailboxRoutine != NULL)
	(*(FUNCPTR) sysMailboxRoutine) (sysMailboxArg);

    (void) sysMailboxAck ();
    }
/*******************************************************************************
*
* sysProcNumGet - get processor number
*
* This routine returns the processor number previously set with 
* sysProcNumSet (2).
*
* RETURNS: processor number
*
* SEE ALSO: sysProcNumSet (2)
*/

int sysProcNumGet ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysProcNumSet - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* SEE ALSO: sysProcNumGet (2)
*/

VOID sysProcNumSet (procNum)
    int procNum;		/* processor number */

    {
    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    return (vxTas (addr));
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysMemParityError - memory parity error interrupt routine
*
* This routine handles the memory parity error on the DY-4 DVME-134 CPU.
* It simply logs a message.
*
* NOTE DY-4 DVME-134:
* Memory parity errors on this CPU are jumperable.  They indicate
* that memory that has not been written to has been read.  In
* order to get around this problem, VxWorks zeroes all of RAM when
* the system is initialized.  Therefore, memory parity errors
* should never occur unless we have a bona fide faulty RAM.
*/

VOID sysMemParityError ()

    {
    logMsg ("memory parity error\n");
    }
/*******************************************************************************
*
* sysFlash - blink LEDs
*
* This routine blinks the CPU board LEDs for a period defined by the parameter.
* A good choice for the parameter is about 100,000.  This gives a flash
* duration of about 1 second.
*
* INTERNAL
* This routine is independent of the interrupt clock because
* there is no guarantee that that it is on when this routine is called.
*
* NOMANUAL
*/

VOID sysFlash (time)
    int time;	/* length of flash, units of ~ 10 microseconds */

    {
    int i;
    short oldCR;
    
    oldCR = *DY4_CR;			/* save the CR */

    *DY4_CR &=  0xfff3;			/* lights on */

    for (i = 0; i < time; i++);		/* burn some time */

    *DY4_CR |=  0x000c;			/* lights out */

    for (i = 0; i < time; i++);		/* burn some time */

    *DY4_CR = oldCR;			/* restore CR */
    }
