/* ramDrv.c - ram disk driver */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/* 
modification history
--------------------
01v,08jul88,jcf  lint.
01u,06jun88,dnw  changed rtLib to rt11Lib.
01t,30may88,dnw  changed to v4 names.
01s,06nov87,ecs  documentation.
01r,04aug87,gae  made ramClose() return STATUS.
		 Changed "static" stuff in ramDrv().
01q,25mar87,jlf  documentation
01p,13mar87,gae  documentation and fixed/improved version 01o.
01o,11mar87,gae  changed raw device open to not use "-" any more.
		 made ramMkfs() return STATUS not (char *).
		 made ramDrv() be a NOP on successive calls.
01n,17feb87,gae  fixed rtDevInit() call.
01m,04feb87,llk  changed call to rtDevInit.  Specifies number of files.
01l,21dec86,dnw  changed to not get include files from default directories.
01k,05sep86,jlf  documentation
01j,15aug86,ecs  renamed ramIoctrl to ramIoctl, changed to call rtIoctl.
01i,02jul86,jlf  oops.
01h,02jul86,jlf  documentation.
01g,03mar86,jlf  changed ioctrl calls to ioctl.
01f,11nov85,jlf  made ramOpen, ramCreate, and ramDelete do a rtReadyChange.
		   The is a kludge so when multiple CPU's share a ram
		   device, the directory will be reread when any of them
		   try to create, delete, or open a file.
01e,19sep85,jlf  got rid of checkMount, and removed member `mounted' from
		 dev descriptor.
01d,12sep85,jlf  updated calling sequence to rtDevInit.
01c,14aug85,dnw  added ramMkfs.
01b,21jul85,jlf  documentation
01a,24may85,jlf  written, by modifying v03k of fd208Drv.c.
*/

/*
DESCRIPTION
This driver emulates a disk driver, but actually keeps all data in memory.
The memory location, and the size of the disk, may be specified on the 
ramDevCreate () call.  This is useful for data that might need to be
kept around between boots of VxWorks, or for sharing data between CPU's.

The RT-11 file and directory structure is used.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, ramDrv () to
initialize the driver, and ramDevCreate () to create devices.  Also,
ramMkfs () provides a somewhat easier way to create devices.

RAMDRV
Before using the driver, it must be initialized by calling the routine:
.CS
    ramDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
ramDevCreates.  It may be called from usrRoot(2) in usrConfig.c, or
at some later point.

CREATING RAM-DISKS
Before a ram disk can be used, it must be created.  This is done
with the ramDevCreate call.  The format of this call is:

.CS
    STATUS ramDevCreate (name, where, bytesSec, secTrack,
			 nSectors, rtFmt, secOffset)
	char *name;	/* Device name *
	char *where;	/* Where it is in memory (0=malloc) *
	int bytesSec;	/* Number of bytes per sector *
	int secTrack;	/* Number of sectors per track *
	int nSectors;	/* Number of sectors on this device *
	BOOL rtFmt;	/* TRUE = RT-11 skew and intereleave *
	int secOffset;	/* Number of sectors to skip at *
			 * beginning of physical device *
.CE

The memory for the ram disk may be allocated separately;
the `where' parameter points to it.
The formatting parameters (bytesSec, secTrack, nSectors, rtFmt) are critical
only if the ramdisk memory already
contains an image of a disk, created elsewhere.  In that case, the
formatting parameters must be identical to those used when the image
was created.  Otherwise, they may be any convenient number.
In this case, 512 is a good sector size,
secTrack could be == nSectors, rtFmt could be FALSE, and secOffset == 0.

For instance, to create the device "/ram/0/", a 200KB ram disk with
these parameters, the proper call would be:
.CS
    ramDevCreate ("/ram/0/", malloc (200000), 512, 2000/512,
		  2000/512, FALSE, 0)
.CE
Note that when the disk is created in this fashion, it is not yet formatted.
This must be done with the FIODISKINIT ioctl call.

RAMMKFS
RamMkfs provides a convenient way to create and initialize
ram disks with standard parameters. The format of the call is:

.CS
    STATUS ramMkfs (name, nbytes, where, dontInit)
	char *name;	/* Device name *
	int nbytes;	/* Number of bytes total for ram disk *
	char *where;	/* Where it is in memory (0=malloc) *
	BOOL dontInit;	/* FALSE = initialize RT-11 directory, *
			 * TRUE  = don't *
.CE

The device is created with the specified name and size.  If no
memory address is specified, then memory for the ram disk will
be allocated from the pool.
Unless `dontInit' is specified TRUE, an RT-11 directory will be
initialized on the newly created device.

The ram devices created by this routine have 512 byte "sectors",
1 "track", do not have "RT-11 compatible skew and interleave",
and have 0 sector offset (see ramDevCreate for details).

For example, the shell command:
.CS
	-> ramMkfs "/ram/",200000
.CE
creates a ram device named "/ram/" with 200000 bytes of memory
allocated from the pool, and initializes an RT-11 directory on it.

The shell command:
.CS
	-> ramMkfs "/ram/",200000,0xc0000,1
.CE
creates a ram device whose starting address is 0xc0000 and does
NOT initialize an RT-11 directory on it.  This might be useful
if a ram disk was created at that same address in a previous
boot of VxWorks.  The contents of the ram disk would be unchanged.

IOCTL
The ram driver responds to all the same ioctl codes as a normal disk driver.
These are described in the "I/O System" chapter, and in rt11Ioctl (2).

SEE ALSO: "I/O System", rt11Ioctl (2)

LINTLIBRARY
*/

#include "vxWorks.h"
#include "ioLib.h"
#include "iosLib.h"
#include "rt11Lib.h"
#include "memLib.h"


typedef struct		/* RAM_DEV - ram disk device descriptor */
    {
    RT_VOL_DESC rtvol;	/* RT-11 volume descriptor */
    char *where;	/* memory location of the ram disk */
    int secOffset;	/* Sector offset of this device from where */
    } RAM_DEV;

LOCAL int ramDrvNum;		/* driver number of ram disk driver */

/* forward declarations */

LOCAL int ramCreate ();
LOCAL STATUS ramDelete ();
LOCAL int ramOpen ();
LOCAL STATUS ramClose ();
LOCAL int ramRead ();
LOCAL int ramWrite ();
LOCAL STATUS ramIoctl ();
LOCAL STATUS ramRdSec ();
LOCAL STATUS ramWrtSec ();


/*******************************************************************************
*
* ramDrv - install ram disk driver
*
* This routine initializes the ram disk driver.  It must be called
* before any other routine in this driver.
*
* RETURNS:
*    OK | ERROR if I/O system unable to install driver
*/

STATUS ramDrv ()

    {
    static BOOL done;	/* FALSE = not done, TRUE = done */

    if (!done)
	{
	done = TRUE;

	ramDrvNum = iosDrvInstall (ramCreate, ramDelete, ramOpen, ramClose,
				    ramRead, ramWrite, ramIoctl);
	}

    return (ramDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* ramDevCreate - create a ram disk device
*
* This routine creates a ram disk device.
* The memory for the ram disk may be allocated separately by providing `where'
* or, if the `where' parameter is NULL then memory will be automatically
* allocated.
*
* The formatting parameters (bytesSec, secTrack, nSectors, rtFmt) are critical
* only if the ramdisk memory already
* contains an image of a disk, created elsewhere.  In that case, the
* formatting parameters must be identical to those used when the image
* was created.  Otherwise, they may be any convenient number.
* In this case, 512 is a good sector size,
* secTrack could be == nSectors, rtFmt could be FALSE, and secOffset == 0.
*
* RETURNS: OK | ERROR if out of memory or I/O system unable to add device
*/

STATUS ramDevCreate (name, where, bytesSec, secTrack, nSectors,
		     rtFmt, secOffset)
    char *name;		/* Device name				*/
    char *where;	/* Where it is in memory (0 = malloc)	*/
    int bytesSec;	/* Number of bytes per sector		*/
    int secTrack;	/* Number of sectors per track		*/
    int nSectors;	/* Number of sectors on this device	*/
    BOOL rtFmt;		/* True if RT-11 skew and intereleave	*/
    int secOffset;	/* Number of sectors to skip at		*/
			/* beginning of physical device		*/

    {
    STATUS status;
    FAST RAM_DEV *pRamDv = (RAM_DEV *) malloc (sizeof (RAM_DEV));

    if (pRamDv == NULL)
	return (ERROR);

    rt11DevInit ((RT_VOL_DESC *) pRamDv, bytesSec, secTrack, nSectors, rtFmt,
	       RT_FILES_FOR_2_BLOCK_SEG, ramRdSec, ramWrtSec, (FUNCPTR) NULL);

    if (where == NULL)
	{
	pRamDv->where = malloc ((unsigned) (bytesSec * nSectors));

	if (pRamDv->where == NULL)
	    {
	    free ((char *) pRamDv);
	    return (ERROR);
	    }
	}
    else
	pRamDv->where = where;

    
    pRamDv->secOffset = secOffset;

    status = iosDevAdd ((DEV_HDR *) pRamDv, name, ramDrvNum);

    if (status == ERROR)
	{
	if (where == NULL)
	    free (pRamDv->where);	/* only free 'where' if we malloc'ed */

	free ((char *) pRamDv);
	}

    return (status);
    }
/*******************************************************************************
*
* ramMkfs - create a ram disk device with an RT-11 file system
*
* This routine provides a convenient way to create and initialize
* ram disks with standard parameters.  The device is created with the
* specified name and size.  If no memory address is specified,
* then memory for the ram disk will be allocated from the pool.
* Unless 'dontInit' is specified TRUE, an RT-11 directory will be
* initialized on the newly created device.
*
* The ram devices created by this routine have 512 byte "sectors",
* 1 "track", do not have "RT-11 compatible skew and interleave",
* and have 0 sector offset (see ramDevCreate for details).
*
* EXAMPLE:
* The shell command:
*	-> ramMkfs "/ram/",200000
* creates a ram device named "/ram/" with 200000 bytes of memory
* allocated from the pool, and initializes an RT-11 directory on it.
*
* The shell command:
*	-> ramMkfs "/ram/",200000,0xc0000,1
* creates a ram device whose starting address is 0xc0000 and does
* NOT initialize an RT-11 directory on it.  This might be useful
* if a ram disk was created at that same address in a previous
* boot of VxWorks.  The contents of the ram disk would be unchanged.
*
* RETURNS: OK | ERROR
*/

STATUS ramMkfs (name, nbytes, where, dontInit)
    char *name;		/* Device name				*/
    int nbytes;		/* Number of bytes total for ram disk	*/
    char *where;	/* Where it is in memory (0 = malloc)	*/
    BOOL dontInit;	/* FALSE = initialize RT-11 directory;	*/
			/* TRUE  = don't			*/

    {
    int fd;
    int nSectors = (nbytes + RT_BYTES_PER_BLOCK - 1) / RT_BYTES_PER_BLOCK;

    /* create the ram device with RT_BYTES_PER_BLOCK byte "sectors" */

    if (ramDevCreate (name, where, RT_BYTES_PER_BLOCK, nSectors, nSectors,
			FALSE, 0) == ERROR)
	{
	printf ("ramMkfs: error creating ram device %s.\n", name);
	return (ERROR);
	}

    /* initialize the RT-11 directory unless specified not to */

    if (!dontInit)
	{
	fd = open (name, WRITE);

	if (fd == ERROR)
	    {
	    printf ("ramMkfs: error opening ram device %s.\n", name);
	    return (ERROR);
	    }

	if (ioctl (fd, FIODISKINIT) == ERROR)
	    {
	    printf ("ramMkfs: error initializing ram device %s.\n", name);
	    close (fd);
	    return (ERROR);
	    }
	
	close (fd);
	}
    
    return (OK);
    }
/*******************************************************************************
*
* ramCreate - create a ram disk file
*
* RETURNS: file descriptor number, or ERROR.
*/

LOCAL int ramCreate (pRamDv, name, mode)
    RAM_DEV *pRamDv;		/* pointer to ram disk device descriptor */
    char *name;			/* name of file to create */
    int mode;			/* access mode */

    {
    RT_FILE_DESC *pfd;

    rt11ReadyChange ((RT_VOL_DESC *) pRamDv);	/* kludge */

    if ((pfd = rt11Create ((RT_VOL_DESC *) pRamDv, name, mode)) == NULL)
	{
	return (ERROR);
	}

    return ((int) pfd);
    }
/*******************************************************************************
*
* ramDelete - delete a ram disk file
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS ramDelete (pRamDv, name)
    RAM_DEV *pRamDv;		/* pointer to ram disk device descriptor */
    char *name;			/* name of file to delete */

    {
    rt11ReadyChange ((RT_VOL_DESC *) pRamDv);	/* kludge */

    return (rt11Delete ((RT_VOL_DESC *) pRamDv, name));
    }
/*******************************************************************************
*
* ramOpen - open a ram disk file
*
* INTERNAL
* This routine calls rt11ReadyChange before opening a file.  This is a kludge,
* to force rtLib to re-read the directory before trying to open the file.
* this is necessary when the ram device is used to transfer files between
* CPU's.
*
* RETURNS: file descriptor number, or ERROR.
*/

LOCAL int ramOpen (pRamDv, name, mode)
    RAM_DEV *pRamDv;		/* pointer to ram disk device descriptor */
    char *name;			/* name of file to open */
    int mode;			/* access mode */

    {
    RT_FILE_DESC *pfd;

    rt11ReadyChange ((RT_VOL_DESC *) pRamDv);	/* kludge */

    if ((pfd = rt11Open ((RT_VOL_DESC *) pRamDv, name, mode)) == NULL)
	{
	return (ERROR);
	}

    return ((int) pfd);
    }
/*******************************************************************************
*
* ramClose - close a ram disk file
*/

LOCAL STATUS ramClose (pfd)
    RT_FILE_DESC *pfd;		/* file descriptor of file to close */

    {
    return (rt11Close (pfd));
    }
/*******************************************************************************
*
* ramRead - read from a ram disk file
*
* RETURNS: number of bytes read, or 0 if end of file, or ERROR if read error.
*/

LOCAL int ramRead (pfd, buffer, maxbytes)
    RT_FILE_DESC *pfd;	/* file descriptor of file to close */
    char *buffer;	/* buffer to receive data */
    int maxbytes;	/* max bytes to read in to buffer */

    {
    return (rt11Read (pfd, buffer, maxbytes));
    }
/*******************************************************************************
*
* ramWrite - write to a ram disk file
*
* RETURNS: number of bytes written, or ERROR.
*/

LOCAL int ramWrite (pfd, buffer, nbytes)
    RT_FILE_DESC *pfd;	/* file descriptor of file to close */
    char *buffer;	/* buffer to be written */
    int nbytes;		/* number of bytes to write from buffer */

    {
    return (rt11Write (pfd, buffer, nbytes));
    }
/*******************************************************************************
*
* ramIoctl - do device specific control function
*
* This routine just passes request on to the rtLib ioctl function.
*/

LOCAL STATUS ramIoctl (pfd, function, arg)
    FAST RT_FILE_DESC *pfd;	/* file descriptor of file to control */
    FAST int function;		/* function code */
    int	arg;			/* some argument */

    {
    switch (function)
	{
	default:
	    return (rt11Ioctl (pfd, function, arg));
	}
    }
/*******************************************************************************
*
* ramRdSec - read a sector from a ram disk volume
*
* This routine reads the specified sector from the specified volume.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS ramRdSec (pRamDv, secNum, buffer)
    FAST RAM_DEV *pRamDv;	/* pointer to device desriptor */
    int secNum;			/* number of sector to read */
    char *buffer;		/* buffer to receive block */

    {
    FAST int bps = pRamDv->rtvol.vd_bytesPerSec;

    /* add in the sector offset */

    secNum += pRamDv->secOffset;

    /* read the sector */

    bcopy (pRamDv->where + (secNum * bps), buffer, bps);

    return (OK);
    }
/*******************************************************************************
*
* ramWrtSec - write a block to a ram disk volume
*
* This routine writes the specified block to the specified volume.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS ramWrtSec (pRamDv, secNum, buffer)
    FAST RAM_DEV *pRamDv;	/* pointer to device desriptor */
    int secNum;			/* number of sector to write */
    char *buffer;		/* buffer to write to block */

    {
    FAST int bps = pRamDv->rtvol.vd_bytesPerSec;

    /* add in the sector offset */

    secNum += pRamDv->secOffset;

    /* write the sector */

    bcopy (buffer, pRamDv->where + (secNum * bps), bps);

    return (OK);
    }
