/* bootConfig.c - system configuration module for boot ROMs */

/* Copyright 1984,1985,1986,1987,1988,1989 Wind River Systems, Inc. */
extern char copyright_wind_river[]; static char *copyright=copyright_wind_river;

/*
modification history
--------------------
02p,13feb89,jcf  changed the calling sequence of lnattach.
		 added 't' command for bcopy.
		 fixed 'g' command to take hex.
		 addes 's' command for frc30/frc31.
02o,15oct88,dnw  fixed printing of exception msg.
		 changed to auto-print new exception msg.
		 added SYS_NO_AUTOBOOT and SYS_QUICK_AUTOBOOT options.
		 improved parsing of command args:
		   improved error reporting,
		   non-adrs args are decimal unless prepended with "0x" or "$"
02n,29oct88,gae  updated copyright date; added ie to netDevList; documentation.
02m,05aug88,gae  documentation.
02l,10aug88,gae  changed lnattach to have usual number of parms.
02k,08aug88,rdc  added support for lance ethernet interface.
	   +dfm  
02j,07jul88,jcf  removed include of non-existant configPsos.h.
02i,05jul88,jcf  made more kernel independent.
		 added sigInit, so exceptions are handled correctly.
		 fixed bug in bootExcepHandler, introduced in v2a.
		   changed its name to bootExcHandler.
02h,22jun88,dnw  name tweaks.
02g,09jun88,dnw  fixed bug in "l" load command (thank you Terry Arden!).
02f,06jun88,dnw  changed taskSpawn/taskCreate args.
		 changed ldLib to loadLib.
02e,30may88,dnw  changed to v4 names.
02d,29may88,dnw  changed call to atoi() to sscanf().
02c,28may88,dnw  made m and d LOCAL.
		 moved skipSpace and skipHexSpecifier here from fioLib.
		 changed calls to fioSetStd... to ioSetGlobalStd (...).
		 changed calls to fioStd... to STD_....
02b,02may88,gae  fixed up VRTX/VRTX32 includes.
02a,09mar88,gae  made kernel independent.  Fixed bug in sysNvSet call.
           +ak
01x,14mar88,gae  changed sysNv{G,S}et() parameters, now test for defn of NV_RAM.
		 caught some lint.
01w,28feb88,dnw  fixed bug in default backplane anchor address.
01v,19feb88,dnw  added subnet-mask option to boot parameters.
01u,12jan88,gae  added remLibInit().
01t,15dec87,gae  fixed usage of BP_ANCHOR_ADRS so it needn't be defined if
		   it isn't used.
01s,24nov87,jcf  added pause before all reboot calls.
01s,23nov87,ecs  lint.
01r,20nov87,dnw  fixed problem with getting alternative bp anchor adrs.
		 fixed to restart correctly after bus error.
		 changed to use new defines in config.h.
01q,17nov87,dnw  changed parameters to bpattach.
01p,14nov87,jcf  changed startType so we only printLogo on BOOT_COLD.
		 If a BERR or error in bootLoad, we 
		   reboot (BOOT_WARM_NO_AUTOBOOT).
01o,12nov87,dnw  changed bus error handler to only print stack trace & regs.
01n,01nov87,llk  deleted call to addNetRoute().  Extract network address from
		   host address with inet_netof_string() and use routeAdd().
01m,26oct87,jcf  added SUN IE ethernet controller conditionals
		 changed call to bpattach to not take the interrupt vector
01l,22oct87,dnw  added flags field to boot line.
		 removed task delete hook in vrtx configuration table.
		 changed bpattach() call to always use polling mode.
		 removed "vb" net driver since this is never used for booting.
		 changed references to tcb_spbottom to tcb_extension->botOfStack
01k,22apr87,dnw  removed call to ifinit() now done in netStart().
		 added echo, tandem (^s/^q), and 7 bit options to console
		   while waiting for initial character during autoboot timeout.
		 fixed parsing of nwords arg in "d" command.
		 changed to call sysClkConnect() instead of direct intConnect().
		 added zeroing of VRTX workspace required by VRTX despite
		   VRTX documentation claims to the contrary.
		 changed args to bpattach() as required by new bp driver.
		 moved ISR_STACK_SIZE here from config.h
		 changed to get vrtx config tbl vector from definition in vrtx.
		 removed call to sysSetLowAdrs().
		 added flush of std in to get rid of junk in usart
		   (to allow hkv2f to boot w/o terminal attached).
		 added initialization of ex, enp, nw, pn, bp, and vb net drivers
		   if INCLUDE_{EX,ENP,NW,PN,BP} respectively are defined.
		 changed to re-initialize boot line to default on re-starts
		   caused by software rather than hardware reset.
01j,08apr87,llk  deleted ISR_STACK_SIZE definition.
		 removed unused param on call sysHwInit().
		 added ROOT_STACK_SIZE.
		 changed to not add usrRoot's stack to memory pool until
		   just before end of usrRoot.
		 added additional memory to pool for Ironics 1624 only.
01i,25mar87,jlf  changed bootLib names to new format
		 documentation
		 added autoboot.
01h,25mar87,rdc  modifications for new backplane driver and gateways.
01g,18feb87,dnw  modifications for vrtx 3.2, removed conditional compilation
		 of exattach, added sysSetLowAdrs.
01f,16jan87,dnw  Fixed bug in clearing memory for heurikon.
01e,15jan87,dnw  Fixed bug in interpreting commands.
01d,10jan87,dnw  Changed to not get include files from default directories.
		 Added l (load), g (go), and ? (help) commands.
		 Simplified user interface.
		 Simplified interaction with non-volitile ram.
		 Added ftp.
01c,08jan87,rdc  fixed memory initialization for heurikon, and changed exattach
		 to use level 2 interrupts for excelan.  removed reset of
		 excelan board in usrInit.
01b,18dec86,llk  Reworked to be common for all ports.  Moved routines for
		    reading boot line to bootLib.c.
		 Added sysModel.
		 Acknowledge clock interrupts on the Ironics 1624 in usrClock.
01a,03nov86,llk  written for Heurikon, modified is20 bootConfig.c version 01a.
*/

/*
DESCRIPTION
This is the WRS-supplied configuration module for the VxWorks boot ROM.
It is a stripped down version of usrConfig.c,
having no "shell" or debugging facilities.
Its primary function is to load an object module
over the network with either rsh or ftp protocols.
Additionally, a simple set of single letter commands is provided
for displaying and modifying memory contents.
Use this module as a starting point for ROMing applications.
*/

/* LINTLIBRARY */

#include "vxWorks.h"
#include "a_out.h"
#include "ctype.h"
#include "ftpLib.h"
#include "ioLib.h"
#include "iosLib.h"
#include "inetLib.h"
#include "loadLib.h"
#include "strLib.h"
#include "sysLib.h"
#include "taskLib.h"
#include "sysSymTbl.h"
#include "version.h"

#include "config.h"


IMPORT VOID sysVwTrapRtn ();	/* vxWorks trap handler entry */
IMPORT VOID sysMemParityError();/* memory parity err handling routine */
IMPORT char *bootStringToParams();	/* interpret boot line */
IMPORT char *sysModel ();	/* model name of system CPU */
IMPORT ULONG tickGet ();	/* get current time in ticks */

IMPORT char edata;		/* automatically defined by the loader */
IMPORT char end;		/* automatically defined by the loader */
IMPORT int  remLastResvPort;	/* XXX */


#define TIMEOUT		7	/* number of seconds before auto-boot */
#define MAX_LINE        160	/* max line length for input to 'm' routine */

#define RSHD		514	/* rshd service */

#define DEC		FALSE	/* getArg parameters */
#define HEX		TRUE
#define OPT		TRUE

int idleTaskPriority     = 254;
int idleTaskOptions      = VX_SUPERVISOR_MODE | VX_UNBREAKABLE;
int idleTaskStackSize    = 1000;

int bootCmdTaskPriority  = 1;
int bootCmdTaskOptions   = VX_SUPERVISOR_MODE;
int bootCmdTaskStackSize = 5000;

SYMTAB_ID sysSymTbl;

LOCAL int sysStartType;		/* i.e. BOOT_COLD, BOOT_WARM, etc. */
LOCAL int consoleFd;

LOCAL char *netDevList [] =
    {
#ifdef	INCLUDE_EX
    "ex",
#endif	INCLUDE_EX
#ifdef	INCLUDE_ENP
    "enp",
#endif	INCLUDE_ENP
#ifdef	INCLUDE_NW
    "nw",
#endif	INCLUDE_NW
#ifdef	INCLUDE_IE
    "ie",
#endif	INCLUDE_IE
#ifdef	INCLUDE_PN
    "pn",
#endif	INCLUDE_PN
#ifdef	INCLUDE_BP
    "bp",
#endif	INCLUDE_BP
#ifdef	INCLUDE_LN
    "ln",
#endif	INCLUDE_LN
    NULL
    };


/* forward declarations */

VOID usrClock ();
VOID usrRoot ();
LOCAL VOID bootCmdLoop ();
LOCAL VOID bootExcHandler ();


/*******************************************************************************
*
* usrInit - user defined system initialization routine
*
* THIS IS THE FIRST "C" CODE EXECUTED AFTER POWER ON. This routine is
* called by the assembly language start-up code in romInit.  It is
* called before kernel multi-tasking is enabled, in system mode, with
* the interrupts locked out.
*
* It starts by clearing BSS, so all variables are initialized to 0 as
* per the C specification, then copies the DATA segment from ROM into RAM.
* Then it sets up exception vectors,
* initializes the hardware by calling sysHwInit,
* and finally starts the kernel with
* the usrRoot task to do the remainder of the initialization.
*
* NOMANUAL
*/

VOID usrInit (startType)
    int startType;		/* BOOT_COLD, BOOT_WARM, BOOT_WARM_NO_AUTO */

    {

#if (CPU==MC68020)
    sysCacheEnable (TRUE);		/* enable cache memory */
#endif

    bzero (&edata, &end - &edata);	/* clear bss */

    sysStartType = startType;

    /* set the vector base register (except on 68000);
     * initialize exception/interrupt vectors */

#if ((CPU==MC68010) || (CPU==MC68020))
    intVecBaseSet ((FUNCPTR *) VEC_BASE_ADRS);
#endif

    excVecInit ();
    intVecSet (TRAPNUM_TO_IVEC (TRAP_VXWORKS), sysVwTrapRtn);  /* vxWks trap */


    /* do system dependent hardware initialization */

    sysHwInit ();

    /* start the kernel specifying usrRoot as the root task */

    kernelInit (TRAP_KERNEL, usrRoot, ROOT_STACK_SIZE,
		FREE_RAM_ADRS, sysMemTop (), ISR_STACK_SIZE, INT_LOCK_LEVEL);
    }
/*******************************************************************************
*
* usrRoot - user defined root task
*
* The root task performs any initialization that should be done
* subsequent to the kernel initialization.
*
* It initializes the I/O system, install drivers, create devices,
* sets up the network, etc., as necessary for the particular configuration.
* It may also create the system symbol table if one is to be included.
* Finally, it spawns the boot command loop task.
*
* NOMANUAL
*/

VOID usrRoot ()

    {
    /* set up parity error interrupt routine, if interrupt vector is defined */

#ifdef INT_VEC_PARITY_ERROR
    intConnect (INUM_TO_IVEC (INT_VEC_PARITY_ERROR), sysMemParityError, 0);
#endif

    /* set up system timer */

    wdLibInit ();			/* init watchdog lib */
    sysClkConnect (usrClock, 0);	/* connect clock interrupt routine */
    sysClkRateSet (60);			/* set sys clock rate */
    sysClkEnable ();			/* start it */

    tickSet ((ULONG) 0);		/* set initial time to zero */


    /* initialize I/O and file system */

    iosInit (NUM_DRIVERS, NUM_FILES, "/null");

    /* install driver for on-board ports and make devices */

    tyCoDrv ();

    /* create console device */
#if defined (TARGET_UBAR_68K2) || defined (TARGET_FRC_30)
    tyCoDevCreate (CONSOLE_DEVICE, 1, 512, 512);
#else
    tyCoDevCreate (CONSOLE_DEVICE, 0, 512, 512);
#endif

    /* make specified console device be standard in/out */

    consoleFd = open (CONSOLE_DEVICE, UPDATE);	/* open console device */
    ioctl (consoleFd, FIOBAUDRATE, 9600);	/* set to 9600 baud */
    ioctl (consoleFd, FIOOPTIONS,
	   OPT_ECHO | OPT_CRMOD | OPT_TANDEM | OPT_7_BIT); /* set raw mode */
    ioGlobalStdSet (STD_IN, consoleFd);		/* set as std input */
    ioGlobalStdSet (STD_OUT, consoleFd);	/* set as std output */
    ioGlobalStdSet (STD_ERR, consoleFd);	/* set as std error */


    taskSpawn ("idle", idleTaskPriority, idleTaskOptions, idleTaskStackSize,
		idle);

    /* install pipe driver and 
     * initialize exception reporting, debugging, and logging */

    pipeDrv ();				/* init pipe driver */
    excInit ();				/* init exception reporting/handling */
    excHookAdd (bootExcHandler);	/* set bootrom exception handler */
    logInit (consoleFd, 5);		/* init logging */
    sigInit ();				/* initialize signals */

    taskSpawn ("boot", bootCmdTaskPriority, bootCmdTaskOptions,
		bootCmdTaskStackSize, bootCmdLoop);
    }
/*******************************************************************************
*
* usrClock - user defined system clock interrupt routine
*
* This routine is called at interrupt level on each clock interrupt.
* It is installed in sysInit by a sysClkConnect call.
* It calls all the other packages that need to know about clock ticks,
* including the kernel itself.
*
* If the application needs anything to happen at clock interrupt level,
* it should be added to this routine.
*
* NOMANUAL
*/

VOID usrClock ()

    {
    wdTick ();		/* check watchdog timers */
    tickAnnounce ();	/* announce system tick to kernel */
    }
/*******************************************************************************
*
* bootCmdLoop - read and execute user commands forever (until boot)
*/

LOCAL VOID bootCmdLoop ()

    {
    char line [MAX_LINE];
    char *pLine;
    int nwords;
    int nbytes;
    int value;
    int adr;
    int adr2;
    FUNCPTR entry;


    /* flush std in to get rid of any garbage;
     * (i.e. Heurikon HKV2F gets junk in usart if no terminal connected). */

    ioctl (STD_IN, FIOFLUSH);


    if (sysStartType == BOOT_COLD)
	{
	printBootLogo ();

	/* this is a cold boot so get the default boot line */

#ifdef	NV_RAM_SIZE
	if (sysNvRamGet (BOOT_LINE_ADRS, NV_RAM_SIZE, 0) == ERROR)
	    {
	    strcpy (BOOT_LINE_ADRS, DEFAULT_BOOT_LINE);
	    *(BOOT_LINE_ADRS + strlen (BOOT_LINE_ADRS) + 1) = EOS; /* 2nd EOS */
	    }
#else	NV_RAM_SIZE
	strcpy (BOOT_LINE_ADRS, DEFAULT_BOOT_LINE);
	*(BOOT_LINE_ADRS + strlen (BOOT_LINE_ADRS) + 1) = EOS; /* 2nd EOS */
#endif	NV_RAM_SIZE
	}


    /* print out any new exception msg -
     * the first byte is zeroed after printing so that we won't print
     * it again automatically.  However, 'e' command will still print out
     * the remainder. */

    printExcMsg (sysExcMsg);
    *sysExcMsg = EOS;		/* indicate exception msg is old */

	
    /* start autoboot, unless no-autoboot specified */

    sysFlags = bootFlagsGet (BOOT_LINE_ADRS);

    if ((sysStartType != BOOT_WARM_NO_AUTOBOOT) &&
	(!(sysFlags & SYSFLG_NO_AUTOBOOT)))
	{
	int timeout = TIMEOUT;

	/* would like to make QUICK timeout be 0 but this seems to
	 * occasionally give ENP controller problems and also makes
	 * it hard to regain control over boot roms, so for now it is 1 sec.
	 */
	if ((sysStartType == BOOT_WARM_QUICK_AUTOBOOT) ||
	    (sysFlags & SYSFLG_QUICK_AUTOBOOT))
	    timeout = 1;

	autoboot (timeout);
	}


    /* If we're here, either we aren't auto-booting, or we got an error
     * auto-booting, or the auto-booting was stopped. */

    /* read and execute the ROM commands */

    ioctl (consoleFd, FIOOPTIONS, OPT_TERMINAL);	/* put console 
							 * in line mode */
    printf ("\n");

    FOREVER
	{
        printf ("[VxWorks Boot]: ");

        fioRdString (STD_IN, line, sizeof (line));

	adr = adr2 = 0;
	nwords = 0;

	pLine = line;
	skipSpace (&pLine);

	switch (*(pLine++))
	    {
	    case EOS:		/* blank line */
		break;

	    case '$':		/* explicit boot line */
		if (bootLoad (&pLine [1], &entry) == OK)
		    go (entry);
		else
		    {
		    taskDelay (sysClkRateGet ());	/* pause a second */
		    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		    }
		break;

	    case 'd':		/* display */
		if ((getArg (&pLine, &adr, HEX, OPT) == OK) &&
		    (getArg (&pLine, &nwords, DEC, OPT) == OK))
		    d ((char *) adr, nwords);
		break;

	    case 'e':		/* exception */
		printExcMsg (sysExcMsg + 1);
		break;

	    case 'f':		/* fill */
		if ((getArg (&pLine, &adr, HEX, !OPT) == OK) &&
		    (getArg (&pLine, &nbytes, DEC, !OPT) == OK) &&
		    (getArg (&pLine, &value, DEC, !OPT) == OK))
		    bfill ((char *) adr, nbytes, value);
		break;

	    case 't':		/* transpose(?) (running out of letters!) */
		if ((getArg (&pLine, &adr, HEX, !OPT) == OK) &&
		    (getArg (&pLine, &adr2, HEX, !OPT) == OK) &&
		    (getArg (&pLine, &nbytes, HEX, !OPT) == OK))
		    bcopy ((char *) adr, adr2, nbytes);
		break;

	    case 'm':		/* modify */
		if (getArg (&pLine, &adr, HEX, !OPT) == OK)
		    m ((char *) adr);
		break;

#ifdef TARGET_HK_V2F
	    case 's':		/* system controller */
		if (getArg (&pLine, &value, DEC, !OPT) == OK)
		    {
		    if (value != 0)
			{
			sysBCLSet (HK_BCL_SYS_CONTROLLER,HK_BCL_SYS_CONTROLLER);
			printf ("System controller on.\n");
			}
		    else
			{
			sysBCLSet (HK_BCL_SYS_CONTROLLER, 0);
			printf ("System controller off.\n");
			}
		    }
		break;
#endif TARGET_HK_V2F

#ifdef TARGET_FRC_30
	    case 's':		/* system controller */
		if (getArg (&pLine, &value, DEC, !OPT) == OK)
		    {
		    if (value != 0)
			{
			*FGA_CTL1(FRC30_FGA_BASE_ADRS) |= FGA_CTL1_SCON;
			printf ("System controller on.\n");
			}
		    else
			{
			*FGA_CTL1(FRC30_FGA_BASE_ADRS) &= ~FGA_CTL1_SCON;
			printf ("System controller off.\n");
			}
		    }
		break;
#endif TARGET_FRC_30

#ifdef TARGET_FRC_31
	    case 's':		/* system controller */
		if (getArg (&pLine, &value, DEC, !OPT) == OK)
		    {
		    if (value != 0)
			{
			*FGA_CTL1(FRC31_FGA_BASE_ADRS) |= FGA_CTL1_SCON;
			printf ("System controller on.\n");
			}
		    else
			{
			*FGA_CTL1(FRC31_FGA_BASE_ADRS) &= ~FGA_CTL1_SCON;
			printf ("System controller off.\n");
			}
		    }
		break;
#endif TARGET_FRC_31

	    case 'p':		/* print boot params */
		bootParamsShow (BOOT_LINE_ADRS);
		break;

	    case 'c':		/* change boot params */
		bootParamsPrompt (BOOT_LINE_ADRS);
#ifdef	NV_RAM_SIZE
		sysNvRamSet (BOOT_LINE_ADRS, strlen (BOOT_LINE_ADRS) + 1, 0);
#endif	NV_RAM_SIZE
		break;

	    case 'l':		/* load */
		if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
		    printf ("entry = 0x%x\n", entry);
		else
		    {
		    taskDelay (sysClkRateGet ());	/* pause a second */
		    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		    }
		break;
		    
	    case 'g':		/* go */
		if (getArg (&pLine, (int *) &entry, HEX, !OPT) == OK)
		    go (entry);
		break;
			
            case '?':			/* help */
            case 'h':			/* help */
		bootHelp ();
		break;

            case '@':			/* autoboot */
		if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
		    go (entry);
		else
		    {
		    taskDelay (sysClkRateGet ());	/* pause a second */
		    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
		    }
		break;

	    default:
		printf ("Unrecognized command. Type '?' for help.\n");
		break;

            } /* switch */
        } /* FOREVER */
    }
/******************************************************************************
*
* autoboot - do automatic boot sequence
*
* RETURNS: doesn't return if successful (starts execution of booted system)
*/  

LOCAL VOID autoboot (timeout)
    int timeout;		/* timeout time in seconds */

    {
    int autoBootTime;
    int timeLeft;
    int timeMarker;
    int bytesRead = 0;
    FUNCPTR entry;

    if (timeout > 0)
	{
	printf ("\nPress any key to stop auto-boot...\n");

	/* Loop looking for a char, or timeout after specified seconds */

	autoBootTime = (int) tickGet () + sysClkRateGet () * timeout;
	timeMarker = (int) tickGet ();
	timeLeft = timeout;

	printf ("%2d\r", timeLeft);

	while (((int) tickGet () < autoBootTime) && (bytesRead == 0))
	    {
	    ioctl (consoleFd, FIONREAD, &bytesRead);
	    if ((int) tickGet () == timeMarker + sysClkRateGet ())
		{
		timeMarker = (int) tickGet ();
		printf ("%2d\r", --timeLeft);
		}
	    }
	}

    /* put the console back in line mode so it echoes (so's you can bang
     * on it to see if it's still alive) */

    ioctl (consoleFd, FIOOPTIONS, OPT_TERMINAL);

    if (bytesRead == 0)    /* nothing typed so auto-boot */
	{
	printf ("\nauto-booting...\n\n");
	if (bootLoad (BOOT_LINE_ADRS, &entry) == OK)
	    go (entry);		/* ... and never return */
	else
	    {
	    printf ("Can't load boot file!!\n");
	    taskDelay (sysClkRateGet ());	/* pause a second */
	    reboot (BOOT_WARM_NO_AUTOBOOT);	/* something is awry */
	    }
	}
    }
/******************************************************************************
*
* printBootLogo - print initial boot banner page
*/  

LOCAL VOID printBootLogo ()
    {
    printf ("\n\n\n\n\n\n\n\n\n\n\n\n");
    printf ("                         VxWorks System Boot \n\n\n");
    printf ("                 %s\n", copyright);
    printf ("\n\n\n\n\n");
    printf ("CPU: %s\n", sysModel ());
    printf ("Version: %s\n", vxWorksVersion);
    printf ("Creation date: %s\n\n", creationDate);
    }
/*******************************************************************************
*
* bootHelp - print brief help list
*/

LOCAL VOID bootHelp ()
    {
    static char *helpMsg[] =
	{
	"?",                      "- print this list",
	"@",                      "- boot (load and go)",
	"p",                      "- print boot params",
	"c",                      "- change boot params",
	"l",                      "- load boot file",
	"g adrs",                 "- go to adrs",
	"d adrs[,n]",             "- display memory",
	"m adrs",                 "- modify memory",
	"f adrs, nbytes, value",  "- fill memory",
	"t adrs, adrs, nbytes",   "- copy memory",
	"e",                      "- print fatal exception",
#if defined (TARGET_HK_V2F) || defined (TARGET_FRC_30) || \
    defined (TARGET_FRC_31)
	"s [0/1]",                "- system controller 0 = off, 1 = on",
#endif TARGET_HK_V2F
	"$dev(0,procnum)host:/file h=# e=# b=# g=# u=usr [pw=passwd] f=#", "",
	NULL
	};

    FAST char **pMsg;

    printf ("\n");

    for (pMsg = helpMsg; *pMsg != NULL; pMsg += 2)
	printf (" %-21s %s\n", *pMsg, *(pMsg + 1));

    printf ("\navailable boot devices:");
    for (pMsg = netDevList; *pMsg != NULL; ++pMsg)
	printf (" %s", *pMsg);

    printf ("\n");
    }
/******************************************************************************
*
* bootFlagsGet - extract boot flags from boot line
*
* RETURNS: boot flags
*/  

LOCAL int bootFlagsGet (paramString)
    char *paramString;

    {
    char dum [BOOT_FIELD_LEN];	/* bit bucket */
    int procNum;
    int flags = 0;

    bootStringToParams (paramString, dum, dum, dum, dum, dum,
			dum, dum, dum, dum, &procNum, &flags);
    return (flags);
    }
/*******************************************************************************
*
* bootLoad - load a module into memory
*/

LOCAL STATUS bootLoad (paramString, pEntry)
    char *paramString;
    FUNCPTR *pEntry;

    {
    char bootDev [BOOT_FIELD_LEN];	/* boot device code */
    char hostName [BOOT_FIELD_LEN];	/* name of host */
    char bootFile [BOOT_FIELD_LEN];	/* name of file to boot */
    char ead [BOOT_FIELD_LEN];		/* ethernet internet addr */
    char bad [BOOT_FIELD_LEN];		/* backplane internet addr */
    char had [BOOT_FIELD_LEN];		/* host internet addr */
    char gad [BOOT_FIELD_LEN];		/* gateway internet addr */
    char usr [BOOT_FIELD_LEN];		/* user id */
    char passwd [BOOT_FIELD_LEN];	/* password */
    char nad [BOOT_FIELD_LEN];		/* host's network internet addr */
    int procNum;			/* where to return processor number */
    char *pS;
    char ifname [20];
    char *inetAdrs;
    int netmask;
    FAST STATUS status;
    FAST char **pMsg;

    /* interpret boot command */

    pS = bootStringToParams (paramString, bootDev, hostName, bootFile, ead, bad,
			     had, gad, usr, passwd, &procNum, &sysFlags);
    if (*pS != EOS)
	{
	/* print error msg with '^' where parse failed */

	printf ("Error in boot command:\n%s\n%*c\n", paramString,
		pS - paramString + 1, '^');
	return (ERROR);
	}

    bootParamsShow (paramString);


    /* set our processor number: may establish vme access, etc. */

    sysProcNumSet (procNum);


    /* put boot command at address expected by usrConfig */

    strcpy (BOOT_LINE_ADRS, paramString);
    *(BOOT_LINE_ADRS + strlen (paramString) + 1) = EOS;	/* 2nd EOS for ISI */


    hostTblInit ();		/* initialize remote command libary */

    /* start the network */
    
    remLastResvPort = 1010;	/* XXX kludge to shorten opening delay */
    netLibInit ();


    /* attach boot device */

    inetAdrs = ead;	/* assume we will use enet inet adrs */

    if (strncmp (bootDev, "bp", 2) != 0)
	printf ("Attaching network interface %s0... ", bootDev);

    if (FALSE)		/* so INCLUDEs will work with 'else if's */
	;
#ifdef INCLUDE_EX
    else if (strcmp (bootDev, "ex") == 0)
	status = exattach (0, IO_ADRS_EX, INT_VEC_EX, INT_LVL_EX);
#endif
#ifdef INCLUDE_ENP
    else if (strcmp (bootDev, "enp") == 0)
	status = enpattach (0, IO_ADRS_ENP, INT_VEC_ENP, INT_LVL_ENP);
#endif
#ifdef INCLUDE_IE
    else if (strcmp (bootDev, "ie") == 0)
	status = ieattach (0, IO_ADRS_IE, INT_VEC_IE, INT_LVL_IE);
#endif
#ifdef INCLUDE_NW
    else if (strcmp (bootDev, "nw") == 0)
	status = nwattach (0, IO_ADRS_NW, INT_VEC_NW, INT_LVL_NW);
#endif
#ifdef INCLUDE_LN
    else if (strcmp (bootDev, "ln") == 0)
	{
#if defined (TARGET_FRC_37) || defined (TARGET_FRC_30)
	/* Force 30 & 37 are supposed to use the board serial number as the
	 * last two bytes of the ethernet address, but this is not currently
	 * available at run-time (i.e. in non-volitile RAM), so we use the
	 * bottom two bytes of the internet address instead.
	 */
	IMPORT char lnEnetAddr [];      /* ethernet addr to load into lance */

	u_long inet = inet_addr (ead);
	lnEnetAddr [4] = (inet >> 8) & 0xff;
	lnEnetAddr [5] = inet & 0xff;
#endif
	status = lnattach (0, IO_ADRS_LN, INT_VEC_LN, INT_LVL_LN, 
			   LN_POOL_ADRS, LN_POOL_SIZE, LN_DATA_WIDTH);
	}
#endif
#ifdef INCLUDE_PN
    else if (strcmp (bootDev, "pn") == 0)
	status = pnattach (0, IO_ADRS_PN, INT_VEC_PN, INT_LVL_PN);
#endif
#ifdef INCLUDE_BP
    else if ((strcmp (bootDev, "bp") == 0) ||
	     (strncmp (bootDev, "bp=", 3) == 0))
	{
	char *bpAnchor = BP_ANCHOR_ADRS;

	if (bootBpAnchorExtract (bootDev, &bpAnchor) < 0)
	    {
	    printf ("invalid bp anchor address: %s\n", bootDev);
	    return (ERROR);
	    }

	printf ("Attaching backplane interface bp0 at 0x%x... ", bpAnchor);

	status = bpattach (0, bpAnchor, sysProcNum, BP_INT_NONE, 0, 0, 0);
	inetAdrs = bad;		/* use bp inet adrs instead */
	}
#endif

    else
	{
	printf ("\nBoot device \"%s\" unknown; known devices:", bootDev);
	for (pMsg = netDevList; *pMsg != NULL; ++pMsg)
	    printf (" %s", *pMsg);
	printf ("\n");

	return (ERROR);
	}

    if (status != OK)
	{
	if (errnoGet () == S_iosLib_CONTROLLER_NOT_PRESENT)
	    printf ("failed: S_iosLib_CONTROLLER_NOT_PRESENT.\n");
	else
	    printf ("failed: status = 0x%x.\n", errnoGet ());

	return (ERROR);
	}
    else
	printf ("done.\n");


    /* configure the boot device with the specified inet address and net mask */

    if (inetAdrs[0] == EOS)
	{
	printf ("Inet address not specified for boot device \"%s\".\n",bootDev);
	return (ERROR);
	}

    strcpy (ifname, bootDev);
    strcat (ifname, "0");

    netmask = 0;
    if (bootNetmaskExtract (inetAdrs, &netmask) < 0)
	{
	printf ("invalid netmask specified: %s\n", inetAdrs);
	return (ERROR);
	}

    if (netmask != 0)
	ifMaskSet (ifname, netmask);

    if (ifAddrSet (ifname, inetAdrs) != OK)
	{
	printf ("invalid inet address specified: %s\n", inetAdrs);
	return (ERROR);
	}


    /* if a gateway was specified, extract the network part of the host's
     * address and add a route to this network */

    if (gad[0] != EOS)
        {
	inet_netof_string (had, nad);
	routeAdd (nad, gad);
        }

    /* associate hostName with the specified host address */

    hostAdd (hostName, had);


    /* load specified file */

    if (netLoad (hostName, bootFile, usr, passwd, pEntry) != OK)
	{
	printf ("error loading file: status = 0x%x.\n", errnoGet ());
	return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* netLoad - downLoad a file from a remote machine via the network.
*
* The remote shell daemon on the machine 'host' is used to download
* the given file to the specified previously opened network file descriptor.
* The remote userId should have been set previously by a call to remIam.
* If the file does not exist, the error message from the Unix 'host' 
* is printed to the VxWorks standard error fd and ERROR is returned.
*
* RETURNS: OK | ERROR
*/

LOCAL STATUS netLoad (hostName, fileName, usr, passwd, pEntry)
    char *hostName;
    char *fileName;
    char *usr;
    char *passwd;
    FUNCPTR *pEntry;

    {
    int fd;
    int errFd;		/* for receiving standard error messages from Unix */
    char command[100];
    struct exec header;
    FAST int nBytes;
    BOOL bootFtp = (passwd[0] != EOS);

    printf ("Loading... ");

    if (bootFtp)
	{
	if (ftpXfer (hostName, usr, passwd, "", "RETR %s", "", fileName,
		     &errFd, &fd) == ERROR)
	    return (ERROR);
	}
    else
	{
	sprintf (command, "cat %s", fileName);

	fd = rcmd (hostName, RSHD, usr, usr, command, &errFd);
	if (fd == ERROR)
	    return (ERROR);
	}

    nBytes = fioRead (fd, (char *) &header, sizeof (struct exec));
    if (nBytes < sizeof (struct exec))
	goto readErr;

    printf ("%d", header.a_text);
    nBytes = fioRead (fd, (char *) header.a_entry, (int) header.a_text);
    if (nBytes < header.a_text)
	goto readErr;

    printf (" + %d", header.a_data);
    nBytes = fioRead (fd, (char *) (header.a_entry + header.a_text),
		      (int) header.a_data);
    if (nBytes < header.a_data)
	goto readErr;

    printf (" + %d\n", header.a_bss);

    close (fd);
    close (errFd);

    *pEntry = (FUNCPTR) header.a_entry;
    return (OK);

readErr:
    /* check standard error on Unix */
    if (bootFtp)
	(void) ftpReplyGet (errFd, FALSE); /* error message on std. err */
    else
	{
	char buf [100];
	int errBytesRecv = fioRead (errFd, buf, sizeof (buf));

	if (errBytesRecv > 0)
	    {
	    /* print error message on standard error fd */

	    buf [errBytesRecv] = EOS;
	    printf ("%s:%s: %s", hostName, fileName, buf);
	    }
	}

    close (fd);
    close (errFd);
    return (ERROR);
    }
/*******************************************************************************
*
* go - start at specified address
*/

LOCAL VOID go (entry)
    FUNCPTR entry;

    {
    printf ("Starting at 0x%x...\n\n", entry);

    taskDelay (sysClkRateGet () / 4); /* give the network a moment to close */

    ifreset ();		/* reset any network board so it doesn't interrupt */

    (entry) ();		/* go to entry point - never to return */
    }

/*******************************************************************************
*
* m - modify memory
*
* This routine prompts the user for modifications to memory, starting at the
* specified address.  It prints each address, and the current contents of
* that address, in turn.  The user can respond in one of several ways:
* .CS
*	RETURN   - No change to that address, but continue
*		   prompting at next address.
*	<number> - Set the contents to <number>.
*	. (dot)	 - No change to that address, and quit.
*	<EOF>	 - No change to that address, and quit.
* .CE
* All numbers entered and displayed are in hexadecimal.
* Memory is treated as 16-bit words.
*
* SEE ALSO: mRegs(2)
*/

LOCAL VOID m (adrs)
    char *adrs;		/* address to change */

    {
    char line[MAX_LINE + 1];	/* leave room for EOS */
    char *pLine;		/* ptr to current position in line */
    int value;			/* value found in line */
    char excess;

    /* round down to word boundary */

    for (adrs = (char *) ((int) adrs & 0xfffffffe);	/* start on even addr */
         ;						/* FOREVER */
	 adrs = (char *) (((short *) adrs) + 1))	/* bump as short ptr */
	{
	/* prompt for substitution */

	printf ("%06x:  %04x-", adrs, (*(short *)adrs) & 0x0000ffff);

	/* get substitution value:
	 *   skip empty lines (CR only);
	 *   quit on end of file or invalid input;
	 *   otherwise put specified value at address */

	if (fioRdString (STD_IN, line, MAX_LINE) == EOF)
	    break;

	line[MAX_LINE] = EOS;	/* make sure input line has EOS */

	for (pLine = line; isspace (*pLine); ++pLine)	/* skip leading spaces*/
	    ;

	if (*pLine == EOS)			/* skip field if just CR */
	    continue;

	if (sscanf (pLine, "%x%1s", &value, &excess) != 1)
	    break;				/* quit if not number */

	* (short *) adrs = value;		/* assign new value */
	}

    printf ("\n");
    }
/*******************************************************************************
*
* d - display memory
*
* Display contents of memory, starting at adrs.  Memory is displayed in
* words.  The number of words displayed defaults to 64.  If
* nwords is non-zero, that number of words is printed, rounded up to
* the nearest number of full lines.  That number then becomes the default.
*/

LOCAL VOID d (adrs, nwords)
    FAST char *adrs;	/* address to display */
    int nwords;		/* number of words to print. */
			/* If 0, print 64 or last specified. */

    {
    static char *last_adrs;
    static int dNbytes = 128;

    FAST int nbytes;
    FAST int byte;
    char ascii [17];

    ascii [16] = EOS;			/* put an EOS on the string */

    nbytes = 2 * nwords;

    if (nbytes == 0)
	nbytes = dNbytes;	/* no count specified: use current byte count */
    else
	dNbytes = nbytes;	/* change current byte count */

    if (adrs == 0)
	adrs = last_adrs;	/* no address specified: use last address */

    adrs = (char *) ((int) adrs & ~1);	/* round adrs down to word boundary */


    /* print leading spaces on first line */

    bfill ((char *) ascii, 16, '.');

    printf ("%06x:  ", (int) adrs & ~0xf);

    for (byte = 0; byte < ((int) adrs & 0xf); byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}


    /* print out all the words */

    while (nbytes-- > 0)
	{
	if (byte == 16)
	    {
	    /* end of line:
	     *   print out ascii format values and address of next line */

	    printf ("  *%16s*\n%06x:  ", ascii, (int) adrs);

	    bfill ((char *) ascii, 16, '.');	/* clear out ascii buffer */
	    byte = 0;				/* reset word count */
	    }

	printf ("%02x", *adrs & 0x000000ff);
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	if (*adrs == ' ' || (isascii (*adrs) && isprint (*adrs)))
	    ascii[byte] = *adrs;

	adrs++;
	byte++;
	}


    /* print remainder of last line */

    for (; byte < 16; byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}

    printf ("  *%16s*\n", ascii);	/* print out ascii format values */

    last_adrs = adrs;
    }
/*******************************************************************************
*
* bootExcHandler - bootrom exception handling routine
*/

LOCAL VOID bootExcHandler (tid)
    int tid;		/* task id */

    {
    int dregs[8];	/* task's data registers */
    int aregs[7];	/* task's address registers */
    char *sp;		/* task's stack pointer */
    USHORT sr;		/* task's status register */
    INSTR *pc;		/* task's pc */

    /* get tcb of task to be traced */

    if (taskRegsGet (tid, dregs, aregs, &sp, &sr, &pc) != ERROR)
	{
	trcStack ((int *) aregs[6], (int *) sp, pc, (FUNCPTR) NULL);
	taskRegsShow (tid);

	taskDelay (sysClkRateGet ());	/* pause a second */
	}

    reboot (BOOT_WARM_NO_AUTOBOOT);
    }
/*******************************************************************************
*
* skipSpace - advance pointer past white space
*
* Increments the string pointer passed as a parameter to the next
* non-white-space character in the string.
*/

LOCAL VOID skipSpace (strptr)
    FAST char **strptr;	/* pointer to pointer to string */

    {
    while (isspace (**strptr))
	++*strptr;
    }
/*******************************************************************************
*
* printExcMsg - print exception message
*
* Avoid printing possible control characters in exception message area.
*/

LOCAL VOID printExcMsg (string)
    char *string;

    {
    printf ("\n");
    while (isprint (*string) || isspace (*string))
	printf ("%c", *string++);
    printf ("\n");
    }
/******************************************************************************
*
* getArg - get argument from command line
*
* This routine gets the next numerical argument from the command line.
* If the argument is not optional, then an error is reported if no argument
* is found.  ppString will be updated to point to the new position in the
* command line.
*
* RETURNS: OK or ERROR
*/  

LOCAL STATUS getArg (ppString, pValue, defaultHex, optional)
    FAST char **ppString;	/* ptr to ptr to current position in line */
    int *pValue;		/* ptr where to return value */
    BOOL defaultHex;		/* TRUE = arg is hex (even w/o 0x) */
    BOOL optional;		/* TRUE = ok if end of line */

    {
    skipSpace (ppString);


    /* if nothing left, complain if arg is not optional */

    if (**ppString == EOS)
	{
	if (!optional)
	    {
	    printf ("missing parameter\n"); 
	    return (ERROR);
	    }
	else
	    return (OK);
	}


    /* scan arg */

    if (scanNum (ppString, pValue, defaultHex) != OK)
	{
	printf ("invalid parameter\n");
	return (ERROR);
	}

    skipSpace (ppString);

    /* if we encountered ',' delimeter, step over it */

    if (**ppString == ',')
	{
	++*ppString;
	return (OK);
	}

    /* if end of line, scan is ok */

    if (**ppString == EOS)
	return (OK);

    /* we got stopped by something else */

    printf ("invalid parameter\n");
    return (ERROR);
    }
