/* tyCoDrv.c - The MVME135 tty handler */

/*
modification history
--------------------
01b,24jun88,gae  lint and cleanup.
01a,11apr88,ak   a modification of v01a of mz7120 tyCoDrv.c
*/

/*
DESCRIPTION
This is the driver for MVME135's SIO 68681 DUART.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, tyCoDrv to
initialize the driver, and tyCoDevCreate to create devices.

TYCODRV
Before using the driver, it must be initialized by calling the routine:
.CS
    tyCoDrv ()
.CE
This routine should be called exactly once, before any reads, writes, or
tyCoDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the tyCoDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS
    STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device
	int channel;	/* Physical channel for this device, always 0
	int rdBufSize;	/* Read buffer size, in bytes
	int wrtBufSize;	/* Write buffer size, in bytes
.CE

For instance, to create the device "/tyCo/0", with buffer sizes of 512 bytes,
the proper call would be:
.CS
   tyCoDevCreate ("/tyCo/0", 0, 512, 512);
.CE
IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available in the mode that we use the chip in are 75, 110, 134.5,
150, 300, 600, 1200, 2000, 2400, 4800, 1800, 9600, 19200.  In the other mode,
you lose 19200 but gain 38400; others are also affected.

*/

#include "UniWorks.h"
#include "iv68k.h"
#include "ioLib.h"
#include "iosLib.h"
#include "memLib.h"
#include "tyLib.h"
#include "mv135.h"


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    char csrVal;	/* rate to write to the csr reg to get that baud rate */
    } BAUD;

typedef struct			/* TY_CO_DEV */
    {
    TY_DEV tyDev;
    BOOL created;	/* true if this device has really been created */
    char *dr;		/* data port */
    char *csr;		/* clock select register */
    char *cr;		/* control reg */
    char rem;		/* bit for receive-enable mask */
    char tem;		/* bit for transmit-enable mask */
    } TY_CO_DEV;

LOCAL TY_CO_DEV tyCoDv [N_CHANNELS] =	/* device descriptors */
    {
    {{{{NULL}}}, FALSE, DUART_RHRA, DUART_CSRA, DUART_CRA,
	       RX_RDY_A_INT, TX_RDY_A_INT},
    {{{{NULL}}}, FALSE, DUART_RHRB, DUART_CSRB, DUART_CRB,
	       RX_RDY_B_INT, TX_RDY_B_INT}
    };

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD baudTable [] =		/* BRG bit = 1 */
    {
    {75,	RX_CLK_75    | TX_CLK_75 },
    {110,	RX_CLK_110   | TX_CLK_110 },
    {134,	RX_CLK_134_5 | TX_CLK_134_5 },
    {150,	RX_CLK_150   | TX_CLK_150 },
    {300,	RX_CLK_300   | TX_CLK_300 },
    {600,	RX_CLK_600   | TX_CLK_600 },
    {1200,	RX_CLK_1200  | TX_CLK_1200 },
    {2000,	RX_CLK_2000  | TX_CLK_2000 },
    {2400,	RX_CLK_2400  | TX_CLK_2400 },
    {4800,	RX_CLK_4800  | TX_CLK_4800 },
    {1800,	RX_CLK_1800  | TX_CLK_1800 },
    {9600,	RX_CLK_9600  | TX_CLK_9600 },
    {19200,	RX_CLK_19200 | TX_CLK_19200}
    };

LOCAL int tyCoDrvNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID tyCoStartup ();
LOCAL int tyCoOpen ();
LOCAL int tyCoRead ();
LOCAL int tyCoWrite ();
LOCAL STATUS tyCoIoctl ();
LOCAL VOID tyCoInt ();


/*******************************************************************************
*
* tyCoDrv - ty driver initialization routine
*
* This routine initializes the driver, sets up interrupt vectors,
* and performs hardware initialization of the ports.
*
* This routine must be called in supervisor state.
*
* RETURNS: OK or ERROR
*/

STATUS tyCoDrv ()

    {
    if (tyCoDrvNum > 0)
	return (OK);

    (void) intConnect (INUM_TO_IVEC (SIO_INT_VEC), tyCoInt, 0);

    tyCoHrdInit ();

    tyCoDrvNum = iosDrvInstall (tyCoOpen, (FUNCPTR) NULL, tyCoOpen,
				(FUNCPTR) NULL, tyCoRead, tyCoWrite, tyCoIoctl);

    return (tyCoDrvNum == ERROR ? ERROR : OK);
    }
/*******************************************************************************
*
* tyCoDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the ports.  Each port
* to be used should have exactly one device associated with it, by calling
* this routine.
*/

STATUS tyCoDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

    {
    if (tyCoDrvNum <= 0)
	{
	errnoSet (S_ioLib_NO_DRIVER);
	return (ERROR);
	}

    /* if there is a device already on this channel, don't do it */

    if (tyCoDv [channel].created)
	return (ERROR);

    /* initialize the ty descriptor, and turn on the bit for this
     * receiver in the interrupt mask */

    if (tyDevInit ((TY_DEV_ID) &tyCoDv [channel], 
		    rdBufSize, wrtBufSize, tyCoStartup) != OK)

	return (ERROR);

    tyImrSet (tyCoDv [channel].rem, 0);

    /* Mark the device as having been created, and add the device to
     * the io sys */

    tyCoDv [channel].created = TRUE;

    return (iosDevAdd ((DEV_HDR *) &tyCoDv [channel], name, tyCoDrvNum));
    }
/*******************************************************************************
*
* tyCoHrdInit - initialize the duart
*
* This routine initializes the MVME135's SIO 68681 DUART.
*/

LOCAL VOID tyCoHrdInit ()

    {
    int oldlevel;		/* current interrupt level mask */
    
    oldlevel = intLock ();

    /* 8 data bits, 1 stop bit, no parity, set for 9600 baud */

    *DUART_CRA  = RST_BRK_INT_CMD; 
    *DUART_CRA  = RST_ERR_STS_CMD; 
    *DUART_CRA  = RST_TX_CMD;
    *DUART_CRA  = RST_RX_CMD;
    *DUART_CRA  = RST_MR_PTR_CMD;
    *DUART_MRA  = PAR_MODE_NO | BITS_CHAR_8;	/* mode A: 1 */
    *DUART_MRA  = STOP_BITS_1;			/* mode A: 2 */
    *DUART_CSRA = RX_CLK_9600 | TX_CLK_9600;	/* clock A */

    *DUART_CRB  = RST_BRK_INT_CMD;
    *DUART_CRB  = RST_ERR_STS_CMD;
    *DUART_CRB  = RST_TX_CMD;
    *DUART_CRB  = RST_RX_CMD;
    *DUART_CRB  = RST_MR_PTR_CMD;
    *DUART_MRB  = PAR_MODE_NO | BITS_CHAR_8;	/* mode B: 1 */
    *DUART_MRB  = STOP_BITS_1;			/* mode B: 2 */
    *DUART_CSRB = RX_CLK_9600 | TX_CLK_9600;	/* clock B */

    *DUART_SOPBC = 3;			/* drive both A & B CTS* low */

    /* enable the receivers and transmitters on both channels */

    *DUART_CRA  = RX_ENABLE | TX_ENABLE;	/* command register A */
    *DUART_CRB  = RX_ENABLE | TX_ENABLE;	/* command register B */

    /* all interrupts are masked out: the receiver interrupt will be enabled
       in the tyCoDevCreate */
    intUnlock (oldlevel);
    } 
/*******************************************************************************
*
* tyCoOpen - open file to duart
*
* ARGSUSED
*/

LOCAL int tyCoOpen (pTyCoDv, name, mode)
    TY_CO_DEV *pTyCoDv;
    char *name;
    int mode;

    {
    return ((int) pTyCoDv);
    }
/*******************************************************************************
*
* tyCoRead - task level read routine for duart
*
* This routine fields all read calls to the duart.
*/

LOCAL int tyCoRead (pTyCoDv, buffer, maxbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int maxbytes;

    {
    return (tyRead ((TY_DEV_ID) pTyCoDv, buffer, maxbytes));
    }
/*******************************************************************************
*
* tyCoWrite - task level write routine for duart
*
* This routine fields all write calls to the duart.
*/

LOCAL int tyCoWrite (pTyCoDv, buffer, nbytes)
    TY_CO_DEV *pTyCoDv;
    char *buffer;
    int nbytes;

    {
    return (tyWrite ((TY_DEV_ID) pTyCoDv, buffer, nbytes));
    }
/*******************************************************************************
*
* tyCoIoctl - special device control
*
* This driver responds to the same ioctl codes as a normal ty driver.
*/

LOCAL STATUS tyCoIoctl (pTyCoDv, request, arg)
    TY_CO_DEV *pTyCoDv;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

    {
    int ix;
    STATUS status;

    switch (request)
	{
	case FIOBAUDRATE:
	    status = ERROR;
	    for (ix = 0; ix < NELEMENTS (baudTable); ix++)
		{
		if (baudTable [ix].rate == arg)
		    {
		    *pTyCoDv->csr = baudTable [ix].csrVal;
		    status = OK;
		    break;
		    }
		}
	    break;

	default:
	    status = tyIoctl ((TY_DEV_ID) pTyCoDv, request, arg);
	    break;
	}

    return (status);
    }

/*******************************************************************************
*
* tyImrSet - set and clear bits in the m68681's interrupt mask register
*
* This routine sets and clears bits in the duart's IMR.
*
* This routine sets and clears bits in a local copy of the IMR, then
* writes that local copy to the duart.  This means that all changes to
* the IMR must go through this routine.  Otherwise, any direct changes
* to the IMR would be lost the next time this routine is called.
*
* This routine must be called in supervisor mode.
*/

LOCAL VOID tyImrSet (setBits, clearBits)
    char setBits;	/* which bits to set in the IMR */
    char clearBits;	/* which bits to clear in the IMR */

    {
    static int imr;	/* current value of duart imr register */

    imr = (imr | setBits) & (~clearBits);
    *DUART_IMR = imr;
    }
/*******************************************************************************
*
* tyCoInt - handle duart interrupts
*
* This routine simply sees if it is transmit or receive, and 
* does the appropriate handoff.
*/

LOCAL VOID tyCoInt ()

    {
    if (*DUART_ISR & TX_RDY_A)
	txInt (0);		/* transmitter channel A interrupt */

    if (*DUART_ISR & TX_RDY_B)
	txInt (1);		/* transmitter channel B interrupt */

    if (*DUART_ISR & RX_RDY_A) 
	rxInt (0);		/* receiver channel A */

    if (*DUART_ISR & RX_RDY_B)
	rxInt (1);		/* receiver channel B */
    }
/*******************************************************************************
*
* rxInt - handle a receiver interrupt
*
* This routine gets called by tyCoInt to handle a receiver interrupt.
*/

LOCAL VOID rxInt (channel)
    FAST int channel;

    {
    tyIRd ((TY_DEV_ID) &tyCoDv [channel], *tyCoDv [channel].dr);
    }
/*******************************************************************************
*
* txInt - handle a transmitter interrupt
*
* This routine gets called by tyCoInt to handle a xmitter interrupt.
* If there is another character to be transmitted, it sends it.  If
* not, or if a device has never been created for this channel, we just
* disable the interrupt.
*/

LOCAL VOID txInt (channel)
    FAST int channel;

    {
    char outChar;

    if ((tyCoDv [channel].created) &&
	(tyITx ((TY_DEV_ID) &tyCoDv [channel], &outChar) == OK))
	{
	*(tyCoDv [channel].dr) = outChar;
       *DUART_CRA  = RX_ENABLE;	/* command register A */
       *DUART_CRA  = RX_ENABLE | TX_ENABLE;	/* command register A */
	}
    else
	tyImrSet (0, tyCoDv [channel].tem); /* turn off the transmitter */
    }
/*******************************************************************************
*
* tyCoStartup - transmitter startup routine
*
* Call interrupt level character output routine.
*/

LOCAL VOID tyCoStartup (pTyCoDv)
    TY_CO_DEV *pTyCoDv;		/* ty device to start up */

    {
    /* all we gotta do is enable the transmitter and it should interrupt
     * to write the next char out. */

    tyImrSet (pTyCoDv->tem, 0);
    }
