/* sysLib.c - Heurikon HK68/V2F system dependent library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

    mailbox/location monitor functions:
       - enable mailbox/location monitor interrupts
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "hkv2f.h"
#include "iv68k.h"

IMPORT char end;		/* end of system, created automatically by ld */

IMPORT VOID logMsg ();

#define	HZ	2457600		/* clock rate */

/* globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;	/* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */


/* locals */

LOCAL int sysClkTicksPerSecond = 60;
LOCAL int auxClkTicksPerSecond = 60;
LOCAL int sysClkRunning = FALSE;
LOCAL int auxClkRunning = FALSE;

LOCAL ULONG sysBcl;		/* current value of bus control latch */


/*******************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "Heurikon HK68/V2F"
*/

char *sysModel ()
    {
    return ("Heurikon HK68/V2F");
    }
/*******************************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
* This routine must be called in supervisor mode.
*
* The timers are initialized and turned off.
* The Multi-Function Peripheral chip is initialized.
* The bus control register is set forn "normal" addressing.
*/

VOID sysHwInit ()

    {
    /* disable interrupts during init */

    *MFP_IERA(MFP_BASE_ADR) = 0;	/* clear interrupt enable register A */
    *MFP_IERB(MFP_BASE_ADR) = 0;	/* clear interrupt enable register B */

    *MFP_IMRA(MFP_BASE_ADR) = 0;	/* clear interrupt mask register A */
    *MFP_IMRB(MFP_BASE_ADR) = 0;	/* clear interrupt mask register B */

    /* clear interrupt pending and in-service registers */

    *MFP_IPRA(MFP_BASE_ADR) = 0;	/* interrupt pending register A */
    *MFP_IPRB(MFP_BASE_ADR) = 0;	/* interrupt pending register B */
    *MFP_ISRA(MFP_BASE_ADR) = 0;	/* interrupt in-service register A */
    *MFP_ISRB(MFP_BASE_ADR) = 0;	/* interrupt in-service register B */

    /* initialize vector register on MFP chip */

    *MFP_VR(MFP_BASE_ADR) = MFP_INT_VECT_NUM;

    /* initialize timers */

    *MFP_TACR(MFP_BASE_ADR)  = 0x07;	/* timer A delay mode, 200 prescale */
    *MFP_TBCR(MFP_BASE_ADR)  = 0;	/* clear timer B control register */
    *MFP_TCDCR(MFP_BASE_ADR) = 0;	/* clear timer C and D control regs. */

    *MFP_TADR(MFP_BASE_ADR)  = 205;	/* timer A data register */
					/* - 205==60Hz w/200 prescale */
    *MFP_TBDR(MFP_BASE_ADR)  = 0;	/* clear timer B data register */
    *MFP_TCDR(MFP_BASE_ADR)  = 0;	/* clear timer C data register */
    *MFP_TDDR(MFP_BASE_ADR)  = 0;	/* clear timer D data register */

    /* set up Heurikon's VME bus control register:
     *   release bus when done, std addressing;
     * NOTE:
     *   everything on the hkv2f is set in software instead of
     *   with hardware jumpers, which is great, except for the system
     *   controller option, which ought to be a hardware jumper.
     *   Anyway, we explicitly DON'T affect the system controller bit
     *   in the BCL here because it may have been turned on by the bootroms
     *   during booting, and turning it off here could cause the backplane
     *   to hang.  Anyway, it gets set in sysSetProcNum() when we know
     *   if it should be on or off, and it also gets turned off in romInit
     *   when starting from a cold start.
     */

    sysBCLSet ((ULONG) ~HK_BCL_SYS_CONTROLLER,
	       (ULONG) (HK_BCL_BC_RWD|HK_BCL_RELEASE_HOLD));

    *HK_RELEASE_SYSFAIL = 1;

    sysLED (FALSE);		/* turn off user LED */
    }
/*******************************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
*
* NOTE HK68/V2F:
* Memory reappears throughout the the address range 0x200000 - 0x300000.
* The algorithm to determine memory size is described
* on page 18 of the HK68/V2F Manual.
*
* RETURNS: address of the first missing byte of memory
*/

char *sysMemTop ()

    {
    static char memDummy;	/* used for calculating memTop */
    static char *memTop = NULL;	/* top of memory address */
    char *m0, *m1, *m2, *m3;	/* pointers in memory */
    int  memSize;
    char testValue = 55;

    if (memTop == NULL)
	{
        /* m0 points to a dummy byte in RAM */

        m0 = &memDummy;

        /* get addresses of memory locations instrumental 
	   in calculating RAM size */

        m1 = m0 + 0x40000;		/* get address 256 K after m0 */
        m2 = m0 + 0x100000;		/* get address 1 M after m0 */
        m3 = m0 + 0x400000;		/* get address 4 M after m0 */

        *m1 = *m2 = *m3 = 0;		/* clear out key memory locations */

        *m0 = testValue;

        /* we can determine the size of memory by checking the 
           key memory locations */

        if (*m1 == testValue)
	    memSize = 0x40000;
        else if (*m2 == testValue)
	    memSize = 0x100000;
        else if (*m3 == testValue)
	    memSize = 0x400000;
        else
	    memSize = 0x1000000;

        memTop = HK_RAM_BASE_ADRS + memSize;

	/* memTop is now pointing to the first non-existent address */
        }

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-X, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor)
*
* INTERNAL
* Note that the "WARM" restart address is at (ROM_BASE_ADRS + 16) bytes.
*/

STATUS sysToMonitor (startType)
    int startType;  /* parameter is passed to ROM to tell it how to boot, */
                    /* possible types are defined in sysLib.h             */

    {
    (* ((FUNCPTR) (ROM_BASE_ADRS + 16))) (startType);

    return (OK);	/* in case we ever continue from rom monitor */
    }

/*******************************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* RETURN: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

STATUS sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each system clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    return (intConnect (INUM_TO_IVEC (INT_VEC_CLOCK), routine, arg));
    }
/*******************************************************************************
*
* sysClkDisable - turn off system clock interrupts
*/

VOID sysClkDisable ()

    {
    int temp;

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysClkEnable - turn system clock interrupts on
*/

VOID sysClkEnable ()

    {
    int temp;

    /* write the timer value (using a 200 prescale) */

    *MFP_TACR(MFP_BASE_ADR) = 0x07;	/* timer A delay mode, 200 prescale */
    *MFP_TADR(MFP_BASE_ADR) = HZ / (200 * sysClkTicksPerSecond);

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TIMER_A;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    sysClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: number of ticks per second of the system clock
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()
    
    {
    return (sysClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysClkSetRate - set rate of system clock
*
* This routine sets the clock rate of the system clock.
* System clock interrupts are not enabled.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* SEE ALSO: sysClkGetRate (2), sysClkEnable (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	sysClkTicksPerSecond = ticksPerSecond;

    if (sysClkRunning)
	{
	sysClkDisable ();
	sysClkEnable ();
	}
    }

/*******************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt.
* Auxiliary clock interrupts are not enabled.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each auxiliary clock interrupt */
    int arg;		/* argument with which to call routine */

    {
    /* connect the routine to the interrupt vector */

    return (intConnect (INUM_TO_IVEC (MFP_INT_VECT_NUM + MFP_INT_TIMER_B),
			routine, arg));
    }
/*******************************************************************************
*
* sysAuxClkDisconnect - disconnect a routine from the auxiliary clock interrupt
*
* This routine disables the auxiliary clock interrupt and disconnects
* the routine currently connected to the auxiliary clock interrupt.
*
* SEE ALSO: sysAuxClkConnect (2), sysAuxClkDisable (2)
*/

VOID sysAuxClkDisconnect ()

    {
    /* disable the auxiliary clock interrupt */

    sysAuxClkDisable ();

    /* connect dummy routine, just in case */

    sysAuxClkConnect (logMsg, (int) "auxiliary clock interrupt\n");
    }
/*******************************************************************************
*
* sysAuxClkDisable - turn off auxiliary clock interrupts
*/

VOID sysAuxClkDisable ()

    {
    int temp;

    *MFP_TBCR(MFP_BASE_ADR) = 0x00;	/* timer B stopped */

    /* disable interrupts */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    /* mask out interrupts  */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp &= ~MFP_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    auxClkRunning = FALSE;
    }
/*******************************************************************************
*
* sysAuxClkEnable - turn auxiliary clock interrupts on
*/

VOID sysAuxClkEnable ()

    {
    int temp;

    /* Write the timer value (assuming a 200 prescale) */

    *MFP_TBDR(MFP_BASE_ADR) = HZ / (200 * auxClkTicksPerSecond);
    *MFP_TBCR(MFP_BASE_ADR) = 0x07;	/* timer B delay mode, 200 prescale */

    /* set mask */

    temp = *MFP_IMRA(MFP_BASE_ADR);
    temp |= MFP_TIMER_B;
    *MFP_IMRA(MFP_BASE_ADR) = temp;

    /* enable interrupt */

    temp = *MFP_IERA(MFP_BASE_ADR);
    temp |= MFP_TIMER_B;
    *MFP_IERA(MFP_BASE_ADR) = temp;

    auxClkRunning = TRUE;
    }
/*******************************************************************************
*
* sysAuxClkGetRate - get rate of auxiliary clock
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: number of ticks per second of the auxiliary clock
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()
    
    {
    return (auxClkTicksPerSecond);
    }
/*******************************************************************************
*
* sysAuxClkSetRate - set rate of auxiliary clock
*
* This routine sets the clock rate of the auxiliary clock.
* Auxiliary clock interrupts are not enabled.
*
* SEE ALSO: sysAuxClkGetRate (2), sysAuxClkEnable (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
    if (ticksPerSecond > 0)
	auxClkTicksPerSecond = ticksPerSecond;

    if (auxClkRunning)
	{
	sysAuxClkDisable ();
	sysAuxClkEnable ();
	}
    }

/*******************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus
*
* SEE ALSO: sysBusToLocalAdrs (2)
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *localAdrs;    /* local address to convert                        */
    char **pBusAdrs;    /* where to return bus address                     */

    {
    if ((localAdrs < HK_RAM_BASE_ADRS) || (localAdrs >= sysMemTop ()))
	{
	/* this is off-board memory - just return local address */

	*pBusAdrs = localAdrs;
	}
    else
	{
	/* this is on-board memory - map to bus address space;
	 *   the following memory mapping is established in sysSetProcNum():
	 *   - only processor 0 has memory on bus,
	 *   - the memory is placed in STD space at address LOCAL_MEM_BUS_ADRS.
	 */

	if ((sysProcNum != 0) ||
	    ((adrsSpace != VME_AM_STD_SUP_PGM) &&
	     (adrsSpace != VME_AM_STD_SUP_DATA) &&
	     (adrsSpace != VME_AM_STD_USR_PGM) &&
	     (adrsSpace != VME_AM_STD_USR_DATA)))
	    return (ERROR);

	*pBusAdrs = localAdrs + (LOCAL_MEM_BUS_ADRS - LOCAL_MEM_LOCAL_ADRS);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;      /* bus address space in which busAdrs resides,     */
                        /* use address modifier codes as defined in vme.h, */
                        /* such as VME_AM_STD_SUP_DATA                     */
    char *busAdrs;      /* bus address to convert                          */
    char **pLocalAdrs;  /* where to return local address                   */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (0x00ff0000 | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (0x01000000 | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if (busAdrs < (char *) 0x03000000)
		return (ERROR);

	    *pLocalAdrs = busAdrs;
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* NOTE HK68/V2F:
* Interrupt levels 2, 4, and 7 are handled by the MFP chip.
* Interrupt levels 1, 3, 5, and 6 are handled by the bus control latch.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    switch (intLevel)
	{
	case 1:
	    /* disable level 1 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ1_ENABLE, (ULONG) 0);
	    break;

	case 2:
	    /* disable level 2 interrupts on the mfp */
	    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_2;	/* disable */
	    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_2;	/* mask */
	    break;

	case 3:
	    /* disable level 3 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ3_ENABLE, (ULONG) 0);
	    break;

	case 4:
	    /* disable level 4 interrupts on the mfp */
	    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_4;	/* disable */
	    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_4;	/* mask */
	    break;

	case 5:
	    /* disable level 5 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ5_ENABLE, (ULONG) 0);
	    break;

	case 6:
	    /* disable level 6 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ6_ENABLE, (ULONG) 0);
	    break;

	case 7:
	    /* disable level 7 interrupts on the mfp */
	    *MFP_IERA(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_7;	/* disable */
	    *MFP_IMRA(MFP_BASE_ADR) &= ~MFP_INT_LEVEL_7;	/* mask */
	    break;

	default:
	    return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified vme interrupt level.
*
* NOTE HK68/V2F:
* Interrupt levels 2, 4, and 7 are handled by the MFP chip.
* Interrupt levels 1, 3, 5, and 6 are handled by the bus control latch.
*
* RETURNS: OK, or ERROR if intLevel not in range 1-7
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    switch (intLevel)
	{
	case 1:
	    /* enable level 1 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ1_ENABLE, (ULONG)HK_BCL_IRQ1_ENABLE);
	    break;

	case 2:
	    /* enable level 2 interrupts on the mfp */
	    *MFP_IERB(MFP_BASE_ADR) |= MFP_INT_LEVEL_2;	/* enable */
	    *MFP_IMRB(MFP_BASE_ADR) |= MFP_INT_LEVEL_2;	/* mask */
	    break;

	case 3:
	    /* enable level 3 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ3_ENABLE, (ULONG)HK_BCL_IRQ3_ENABLE);
	    break;

	case 4:
	    /* enable level 4 interrupts on the mfp */
	    *MFP_IERB(MFP_BASE_ADR) |= MFP_INT_LEVEL_4;	/* enable */
	    *MFP_IMRB(MFP_BASE_ADR) |= MFP_INT_LEVEL_4;	/* mask */
	    break;

	case 5:
	    /* enable level 5 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ5_ENABLE, (ULONG)HK_BCL_IRQ5_ENABLE);
	    break;

	case 6:
	    /* enable level 6 interrupts with the bus control latch */
	    sysBCLSet ((ULONG)HK_BCL_IRQ6_ENABLE, (ULONG)HK_BCL_IRQ6_ENABLE);
	    break;

	case 7:
	    /* enable level 7 interrupts on the mfp */
	    *MFP_IERA(MFP_BASE_ADR) |= MFP_INT_LEVEL_7;	/* enable */
	    *MFP_IMRA(MFP_BASE_ADR) |= MFP_INT_LEVEL_7;	/* mask */
	    break;

	default:
	    return (ERROR);
	}

    return (OK);
    }
/*******************************************************************************
*
* sysIntAck - acknowledge interrupt
*
* This routine acknowledges the specified interrupt.
*
* NOTE HK68/V2F:
* Certain interrupts (IRQ2, IRQ4, and IRQ7) require that
* particular memory locations be read to acknowledge the interrupt and
* to retreive the vector number put on the bus by the interrupting device.
* For other interrupts, this routine performs no action since they
* are acknowledged automatically by the hardware.
*
* RETURNS:
*   For 2,4,7:  the vector number put on the bus
*		by the interrupting device,
*   For others: ERROR
*
* SEE ALSO: Heurikon specific documentation in Chapter 5
*/

int sysIntAck (intLevel)
    int intLevel;	/* interrupt level to acknowledge */

    {
    switch (intLevel)
	{
	case 2:  return (*HK_VEC_IRQ2);
	case 4:  return (*HK_VEC_IRQ4);
	case 7:  return (*HK_VEC_IRQ7);
	default: return (ERROR);
	}
    }
/*******************************************************************************
*
* sysIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* RETURNS: OK or ERROR if invalid level
*/

STATUS sysIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    if (level < 1 || level > 7)
	return (ERROR);

    *(char *) (0xfea000 + (level << 2)) = vector;

    return (OK);
    }

/*******************************************************************************
*
* sysMailboxConnect - connect routine to the mailbox interrupt
*
* This routine connects the given function to the mailbox interrupt.
*
* RETURNS: OK or ERROR if unable to connect to interrupt
*
* SEE ALSO: intConnect (2)
*/

STATUS sysMailboxConnect (routine, arg)
    FUNCPTR routine;	/* routine called at each mailbox interrupt */
    int arg;		/* argument with which to call routine */

    {
    *MFP_IERB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* disable interrupts */
    *MFP_IMRB(MFP_BASE_ADR) &= ~MFP_MBOX_INT;	/* mask interrupts */

    return (intConnect (INUM_TO_IVEC (INT_VEC_BP), routine, arg));
    }
/*******************************************************************************
*
* sysMailboxEnable - enable mailbox interrupt
*
* This routine enables the mailbox interrupt.
*
* RETURNS: OK (always)
*/

STATUS sysMailboxEnable (mailboxAdrs)
    char *mailboxAdrs;		/* mailbox address */

    {
    int mboxBaseCode;

    /* set mailbox address */

    mboxBaseCode = ((((int) mailboxAdrs & 0xE000) >> 8) |
		    (((int) mailboxAdrs & 0x00F8) >> 3))
			<< HK_BCL_MBOX_BASE_SHIFT;

    sysBCLSet ((ULONG)(HK_BCL_MBOX_BASE_MASK | HK_BCL_MBOX_ENABLE),
    	       (ULONG) (mboxBaseCode | HK_BCL_MBOX_ENABLE));

    /* enable mailbox interrupt */

    *MFP_IERB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* enable */
    *MFP_IMRB(MFP_BASE_ADR) |= MFP_MBOX_INT;	/* mask */

    return (OK);
    }
/*******************************************************************************
*
* sysGetProcNum - get processor number
*
* This routine returns the processor number previously set with 
* sysSetProcNum (2).
*
* RETURNS: processor number
*/

int sysGetProcNum ()

    {
    return (sysProcNum);
    }
/*******************************************************************************
*
* sysSetProcNum - set processor number
*
* Set the processor number for this CPU.  Processor numbers should be
* unique on a single backplane.
*
* NOTE HK68/V2F:
* This routine performs several functions:
*   - turns on the bus arbitor if we are processor 0
*     (unless SYSFLG_NO_SYS_CONTROLLER flag is set in sysFlags).
*   - maps our memory onto the vme bus if we are processor 0.
*/

VOID sysSetProcNum (procNum)
    int procNum;		/* processor number */

    {
    int memSize;
    ULONG value;

    if (procNum == 0)
	{
	/* set on-board memory in Standard 24 bit address space */

	value = HK_BCL_MP1_STD | HK_BCL_MP2_24A;


	/* enable system controller for processor 0 unless
	 * SYSFLG_NO_SYS_CONTROLLER flag is set in sysFlags
	 */

	if ((sysFlags & SYSFLG_NO_SYS_CONTROLLER) == 0)
	    value |= HK_BCL_SYS_CONTROLLER;


	/* enable bus access to all local memory */

	memSize = sysMemTop () - HK_RAM_BASE_ADRS;

	if (memSize == 0x1000000)
	    value |= HK_BCL_MP3_16M_0V_24A_00;
	else if (memSize == 0x400000)
	    value |= HK_BCL_MP3_4M_0V_24A_00;
	else
	    value |= HK_BCL_MP3_1M_0V_24A_00;
	}
    else
	value = 0;


    sysBCLSet ((ULONG) (HK_BCL_SYS_CONTROLLER | HK_BCL_VSMAP_MASK), value);

    sysProcNum = procNum;
    }
/*******************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* NOTE HK68/V2F:
* On most CPU's this is equivalent to the vxTas(2) routine, but
* this board has problems doing TAS instructions on the bus - apparently
* the onboard arbiter has problems with read-modify-write cycles.
* To get around the problem we get exclusive access to the
* bus for the duration of the TAS instruction.
*
* RETURNS: TRUE (successful set) or FALSE (failure)
*/

BOOL sysBusTas (addr)
    char *addr;		/* address to be tested and set */

    {
    int retVal;

    /* get exclusive access to bus */

    sysBCLSet ((ULONG) HK_BCL_BC_NO_RELEASE, (ULONG) HK_BCL_BC_NO_RELEASE);

    if (*addr);		/* access location */

    retVal = vxTas (addr);

    /* release bus */

    sysBCLSet ((ULONG) HK_BCL_BC_NO_RELEASE, (ULONG) 0);

    if (*addr);		/* access location to release hold */

    return (retVal);
    }


/* miscellaneous support routines */

/*******************************************************************************
*
* sysBCLSet - set bits in the bus control latch
*
* This routine sets bits in the bus control latch on the HK68/V2F board.
* The parameter "mask" determines which bits will be set, and "value"
* determines the value to which those bits will be set.
* In  other words, newBCL = (oldBCL & ~mask) | (value & mask).
*/

VOID sysBCLSet (mask, value)
    ULONG mask;		/* which bits to change */
    ULONG value;	/* what to change bits to */

    {
    sysBcl = (sysBcl & (~mask)) | (value & mask);
    *HK_BUS_CONTROL_LATCH = sysBcl;
    }
/*******************************************************************************
*
* sysBCLGet - return the current value of the bus control latch
*
* This routine returns the last value to which the bus control latch
* was set.  It is only effective if all changes to the bus control
* latch are done through sysBCLSet (2).
*/

ULONG sysBCLGet ()

    {
    return (sysBcl);
    }
/*******************************************************************************
*
* sysMemParityError - memory parity error interrupt routine
*
* This routine handles the memory parity error.  It simply logs a message.
*
* NOTE HK68/V2F:
* Memory parity errors on this CPU cannot be disabled!  They generally
* indicate that memory that has not been initialized has been read.  In
* order to get around this problem, UniWorks zeroes all of RAM when
* the system is initialized.  Therefore, memory parity errors should, in
* theory, never occur.
*/

VOID sysMemParityError ()

    {
    logMsg ("memory parity error\n");
    }
/*******************************************************************************
*
* sysNvRamGet - get contents out of non-volatile RAM
*
* Copies non-volatile memory into string.
* The string will be terminated with an EOS.
*
* RETURNS: OK (always)
*/

STATUS sysNvRamGet (string, strLen, offset)
    FAST char *string;	/* where to copy non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */
    int offset;		/* (even) byte offset into non-volatile RAM */

    {
    FAST char *pNvRam;
    FAST int ix;

    /* read from NVRAM into shadow RAM */

    pNvRam = (char *) HK_NV_RECALL;
    pNvRam = (char *) (HK_NVRAM_ADDR + ((offset + 1) & ~1));

    /* construct the character string from NV nibbles; nibbles are ordered as:
     *
     * <lower addrses> (high order nibble)(low order nibble)... <higher addrses>
     *
     * Odd addresses contain low order nibbles, even addresses contain
     * high order nibbles. */

    for (ix = 0; ix < HK_NVRAM_SIZE && ix < strLen; ix++)
	{
	*(string++) = (*pNvRam << 4) | (*(pNvRam + 1) & 0xf);
	pNvRam += 2;
	}

    *string = EOS;

    return (OK);
    }
/*******************************************************************************
*
* sysNvRamSet - write to non-volatile RAM
*
* Copy string into non-volatile RAM.
*/

VOID sysNvRamSet (string, strLen, offset)
    FAST char *string;	/* string to be copied into non-volatile RAM */
    int strLen;		/* maximum number of bytes to copy */
    int offset;		/* (even) byte offset into non-volatile RAM */

    {
    FAST char *pNvRam = (char *) (HK_NVRAM_ADDR + ((offset + 1) & ~1));
    FAST int ix;

    strLen--;	/* leave room for terminating EOS */

    /* construct the NV nibbles from string; nibbles are ordered as:
     *
     * <lower addrses> (high order nibble)(low order nibble)... <higher addrses>
     *
     * Odd addresses contain low order nibbles, even addresses contain
     * high order nibbles. */

    for (ix = 0; ix < HK_NVRAM_SIZE; ix++)
	{
	*(pNvRam++) = (string [ix] >> 4) & 0xf;
	*(pNvRam++) = string [ix] & 0xf;
	}

    /* load from the shadow RAM to the NVRAM */

    vxTas ((char *) HK_NV_STORE);
    }
/*******************************************************************************
*
* sysLED - turn user LED on or off
*/

VOID sysLED (on)
    BOOL on;	/* TRUE = turn LED on, FALSE = turn LED off */

    {
    sysBCLSet ((ULONG)HK_BCL_USER_LED, on ? (ULONG)0 : (ULONG)HK_BCL_USER_LED);
    }
