/*
	8/87 S. Chaplin
*/
/* icp.c - The Integrated Solutions 68020 ICP-board tty handler */

/*
modification history
--------------------
*/

/*
DESCRIPTION
This is the driver for the Integrated Solutions 68020 ICP-board serial ports.

USER CALLABLE ROUTINES
Most of the routines in this driver are accessible only through the I/O
system.  Two routines, however, must be called directly, cpDrv () to
initialize the driver, and cpDevCreate to create devices.

ICPCRV
Before using the driver, it must be initialized by calling the routine:

.CS
    cpDrv ()
.CE

This routine should be called exactly once, before any reads, writes, or
cpDevCreates.  Normally, it is called from usrRoot.

CREATING TERMINAL DEVICES
Before a terminal can be used, it must be created.  This is done
with the cpDevCreate call.
Each port to be used should have exactly one device associated with it,
by calling this routine.

.CS

    STATUS cpDevCreate (name, channel, rdBufSize, wrtBufSize)
	char *name;	/* Name to use for this device *
	int channel;	/* Physical channel for this device (0 or 1) *
	int rdBufSize;	/* Read buffer size, in bytes *
	int wrtBufSize;	/* Write buffer size, in bytes *

.CE

For instance, to create the device "/cp/0", with buffer sizes of 512 bytes,
the proper call would be:

.CS
   cpDevCreate ("/cp/0", 0, 512, 512);

.CE

IOCTL
This driver responds to all the same ioctl codes as a normal ty driver.
The baud rates available are 50, 110, 134, 200, 300, 600, 1200, 1050,
2400, 4800, 7200, 9600, 38400.

*/
/*
#include "/usr/sys/h/types.h"
#include "/usr/sys/machine/machparam.h"
*/
#include "../../../h/vxWorks.h"
#include "icp.h"
#include "icpreg.h"
#include "../../../h/ioLib.h"
#include "../../../h/iosLib.h"
#include "../../../h/memLib.h"
#include "../../../h/wdLib.h"
#include "../../../h/tyLib.h"
#include "../../../h/rngLib.h"
#include "../../../h/vme68k20.h"


#define loword(X)   ( (unsigned long)(X) & 0xffff )
#define hiword(X)   (((unsigned long)(X)>>16) & 0xffff)


typedef struct			/* BAUD */
    {
    int rate;		/* a baud rate */
    USHORT csrVal;	/* rate to write to the csr reg to get that baud rate */
    } BAUD;

typedef struct {
        BOOL    alive;
        struct 	cpdevice *cpaddr;		/* pointer to controller */
	BAUD	*baud_conv;		/* baud rate conversion table */
	int	nlines;			/* number of lines on controller */
} CP_CONT;
LOCAL CP_CONT cp_cont[NICP];

typedef struct			/* ICP_DEV */
{
	TY_DEV 	tyDev;
    	BOOL 	created;	/* true if this device has really been created */
    	BOOL	open;		/* true if this device has been opened */
    	int	unit;		/* unit number for this structure */
    	int	task;		/* task which opened channel */
	int	state;		/* channel state */
	int	putpt;		/* put pointer for ring buffer after transfer */
	int	options;	/* tty channel options */
    	CP_CONT	*cont;		/* ptr to controller */
} ICP_DEV;

/* declare a device structure for each channel */
LOCAL ICP_DEV cp_dev[N_ICP_CHANNELS]; 

/* baudTable is a table of the available baud rates, and the values to write
   to the csr reg to get those rates */

LOCAL BAUD bconv0 [] =
    {
    {75,	0x00}, {110,	0x11}, {134,	0x22}, {200,	0x33},
    {300,	0x44}, {600,	0x55}, {1200,	0x66}, {2000,	0x77},
    {2400,	0x88}, {4800,	0x99}, {1800,	0xaa}, {9600,	0xbb},
    {19200,	0xcc}, {0, 0}
    };
LOCAL BAUD bconv1 [] =
    {
    {50,	0x00}, {110,	0x11}, {134,	0x22}, {200,	0x33},
    {300,	0x44}, {600,	0x55}, {1200,	0x66}, {1050,	0x77},
    {2400,	0x88}, {4800,	0x99}, {7200,	0xaa}, {9600,	0xbb},
    {0,	0}
    };

LOCAL int cpNum;		/* driver number assigned to this driver */

/* forward declarations */

LOCAL VOID cpStartup ();
LOCAL int cpOpen ();
LOCAL int cpRead ();
LOCAL int cpWrite ();
LOCAL STATUS cpIoctrl ();
LOCAL VOID txInt ();
LOCAL VOID sxInt ();


typedef struct {
        USHORT *csr;
        char    vec;
}DEVADDS;

extern char     vme_std[];
extern char vme_window_ext[];
extern char vme_window_std[];

DEVADDS cpaddrs[] = {           /* Bus and vector address of ICPs */
        (USHORT *)&vme_std[0xfff520], (char)0x70, 
		(USHORT *)&vme_std[0], (char)0
};

LOCAL BOOL device_present;

LOCAL   int     cpOpen(), cpRead(), cpInterror(), cpWrite(), cpClose(),
		cpIoctl(), cpStartup(), cpSint(), cptxInt();

#define DELAY(n)        { register int N = (n); while (--N > 0); }
#define AWT		15

/***********************************************************************
*
* cpDrv - ty driver initialization routine
*
* This routine initializes the serial driver, sets up interrupt vectors,
* and performs hardware initialization of the serial ports.
*
* This routine must be called in supervisor state, since it does
* physical I/O directly.
*/

STATUS cpDrv ()

{
register struct cpdevice *cpaddr;
register int 	i, j;
BOOL	someone_alive;
int	conf;


	for (i = 0, someone_alive = FALSE; i < NICP; i++)
	{
		cpaddr = (struct cpdevice*)cpaddrs[i].csr;
        /* issue a master clear */
       	 	cpaddr->cp_sel = SEL_MC | SEL_MV;
					
        	while (cpaddr->cp_sel & SEL_MC)
               		 vxTdelay(1);

        /*
         * cause interrupt by setting small silo age time and writing
         * something in the silo
         */
	/* connect controller to interrupt routine */
		intConnect(cpaddrs[i].vec << 2, cpInterror, i);
		intConnect((cpaddrs[i].vec << 2) + 4, cpSint, i);
		intConnect((cpaddrs[i].vec << 2) + 8, cptxInt, i);
		intConnect((cpaddrs[i].vec << 2) + 12, cpInterror, i);
		intConnect((cpaddrs[i].vec << 2) + 16, cpInterror, i);
		cp_cont[i].cpaddr = cpaddr;

        	cpaddr->cp_isr = (ISR_IE << 8) | cpaddrs[i].vec;
				
        	cpaddr->cp_sel = SEL_SILO;
				
        	cpaddr->cp_pr = 1;           /* silo age time */
				
		device_present = FALSE;

 	/* simulate reception of character */
        	cpaddr->cp_swr = SWR_VDP | 0xAB; 
        		vxTdelay(1);

		if (device_present)
		{
			someone_alive = TRUE;
			cp_cont[i].alive = TRUE;

		/* get configuration from sel register on controller */
        	/* issue a master clear */
       	 		cpaddr->cp_sel = SEL_MC | SEL_MV;
					
        		while (cpaddr->cp_sel & SEL_MC)
               		 vxTdelay(1);

		        conf = cpaddr->cp_sel;
			cp_cont[i].nlines = conf & SEL_CONF_NLINES;
#ifdef DEBUG
			logMsg("number of lines on controller = %d\n",
				cp_cont[i].nlines);
#endif
        		if (conf & SEL_CONF_BR) 
			{
#ifdef DEBUG
                		logMsg("baud rate table 1\n");
#endif
				cp_cont[i].baud_conv = bconv1; 	
			}
        		else
			{
#ifdef DEBUG
				logMsg("baud rate table 0\n");
#endif
				cp_cont[i].baud_conv = bconv0; 	
			}

		/* point to silo */
        		cpaddr->cp_sel = SEL_SILO;
					

		/* silo fill level | silo age time */
			cpaddr->cp_pr = (40 << 8) | 11; 
					

		/* enable all interrupts on ICP using single vector */
        		cpaddr->cp_isr = (ISR_IE << 8) | cpaddrs[i].vec;

		/* set default line parameters */
			for (j = 0; j <  cp_cont[i].nlines; j++)
			{
				cpaddr->cp_sel = j << SEL_SHIFT;
					
			/* 8 bits, no parity, 1 stop bit 9600 baud */
				cpaddr->cp_pr = PR_BITS8 | 0xbb;
					
			}

		}
		else
		{
#ifdef DEBUG
			logMsg("controller %d dead\n", i);
#endif
			cp_cont[i].alive = FALSE;
		}
	/* initialize data structures for all devices on controller */
		for (j = 0; j < cp_cont[i].nlines; j++)
		{
			cp_dev[j].created = FALSE;
			cp_dev[j].open = FALSE;
			cp_dev[j].unit = j;
			cp_dev[j].task = 0;
			cp_dev[j].state = IDLE;
			cp_dev[j].cont = &cp_cont[i]; 
		}

	}	

	if (someone_alive)
	{
#ifdef DEBUG
			logMsg("\nICP data structure locations\n");
	for (i = 0; i < NICP; i++)
	{
		logMsg("  controller %d at %x\n", i, &cp_cont[i]);
		for (j = 0; j < cp_cont[i].nlines; j++)
		{
			logMsg("	device %d at %x\n", j, &cp_dev[j]);
		}
	}
#endif
		cpNum = iosDrvInstall(cpOpen, (FUNCPTR)NULL, cpOpen, cpClose,
					cpRead, cpWrite, cpIoctl);
		return(OK);
	}
	else
	{
#ifdef DEBUG
			logMsg("no ICP boards operational\n", i);
#endif
	}
		return(ERROR);
}


/******************************************************************
*
* cpDevCreate - create a device for the onboard ports
*
* This routine creates a device on one of the serial ports.  Each port
* to be used should have exactly one device associated with it, 
* by calling this routine.
*/

STATUS cpDevCreate (name, channel, rdBufSize, wrtBufSize)
    char *name;		/* Name to use for this device */
    int channel;	/* Physical channel for this device (0 or 1) */
    int rdBufSize;	/* Read buffer size, in bytes */
    int wrtBufSize;	/* Write buffer size, in bytes */

{
#ifdef DEBUG
	logMsg("device create unit %d\n", channel);
#endif
 
/* check to see if a controller exists for this device */
	if (!(cp_dev[channel].cont->alive))
	{
		logMsg("ERROR - cpDevCreate - controller not alive\n");
		return(ERROR);
	}

/* if there is a device already on this channel, don't do it */
	if (cp_dev[channel].created)
	{
		logMsg("ERROR - cpDevCreate - device already on this channel\n");
		return (ERROR);
	}

/* initialize the ty descriptor, and turn on the bit for this
 * receiver in the interrupt mask */

	if (tyDevInit ((TY_DEV_ID) &cp_dev[channel], 
		    rdBufSize, wrtBufSize, cpStartup) != OK)
	{
		logMsg("ERROR - cpDevCreate - tyDevInit failed\n");
		return (ERROR);
	}

/* mark device as created */
	cp_dev[channel].created = TRUE;

#ifdef DEBUG
	logMsg("calling device add\n");
#endif

    return (iosDevAdd ((DEV_HDR *) &cp_dev[channel], name, cpNum));
}
	

/**********************************************************************
*
* cpOpen - open file to isi ICP line
*
*/

LOCAL int cpOpen (pcp_dev, name, mode)
    ICP_DEV  *pcp_dev;
    char *name;
    int mode;

{
register USHORT line;

#ifdef DEBUG
	logMsg("open cp unit = %d\n", pcp_dev->unit);
#endif

	if (pcp_dev->open)
	{
		logMsg("ERROR - cpOpen -cp device unit %d already open\n", pcp_dev->unit);
		return(ERROR);
	}
		
/* note which task has opened the channel */
	pcp_dev->task = vxMyTaskId();

/* enable serial line and mark device as open */
	line = 1 << pcp_dev->unit;
	pcp_dev->cont->cpaddr->cp_ler |= line;
	pcp_dev->open = TRUE;
	pcp_dev->state = IDLE;
	return ((int) pcp_dev);
}

/**********************************************************************
*
* cpClose - close file to isi ICP line
*
*/

LOCAL int cpClose (pcp_dev)
    ICP_DEV *pcp_dev;

{
register USHORT line;

#ifdef DEBUG
	logMsg("close cp channel\n");
#endif

	if (!pcp_dev->open)
	{
		logMsg("ERROR -  cpClose - cp device unit %d not open\n", pcp_dev->unit);
		return(ERROR);
	}

/* disable serial line and mark device as closed */
	while(pcp_dev->state != IDLE)
		vxTdelay(60);	/* keep open long enough to finish a trans. */
	line = 1 << pcp_dev->unit;
	pcp_dev->cont->cpaddr->cp_ler &= ~line;
	pcp_dev->open = FALSE;
	return(0);
}

/**********************************************************************
*
* cpRead - Task level read routine for isi ICP line.
*
* This routine fields all read calls to the isi usart.  It
* calls tyRead with a pointer to the appropriate element of ty_dev.
*/

LOCAL int cpRead (pcp_dev, buffer, maxbytes)
    ICP_DEV *pcp_dev;
    char *buffer;
    int maxbytes;

{
#ifdef DEBUG
	logMsg("cp read maxbytes = %d\n", maxbytes);
#endif
 
    return (tyRead ((TY_DEV_ID)pcp_dev, buffer, maxbytes));
}

/**********************************************************************
*
* cpWrite - Task level write routine for isi ICP line.
*
* This routine fields all write calls to the isi ICP line.  It
* calls tyWrite with a pointer to the appropriate element of ty_dev.
*/

LOCAL int cpWrite (pcp_dev, buffer, nbytes)
    ICP_DEV *pcp_dev;
    char *buffer;
    int nbytes;

{
#ifdef DEBUG
	logMsg("cp write bytes = %d\n", nbytes);
#endif
 
    return (tyWrite ((TY_DEV_ID)pcp_dev, buffer, nbytes));
}

/***********************************************************************
*
* cpIoctrl - special device control
*
* This routine handles baud rate requests, and passes all other requests
* to cpIoctl.
*/

LOCAL STATUS cpIoctl (pcp_dev, request, arg)
    ICP_DEV *pcp_dev;	/* device to control */
    int request;	/* request code */
    int arg;		/* some argument */

{   
int i = 0;
USHORT lpr;
STATUS status;

#ifdef  DEBUG
	logMsg("cp ioctl req = %d\n", request);
#endif  

/* get line parameter register */
	lpr = pcp_dev->cont->cpaddr->cp_pr;

	switch (request)
	{
	  case FIOBAUDRATE:
	    	status = ERROR;
	    	while (pcp_dev->cont->baud_conv[i].rate != 0)
		{
			if (pcp_dev->cont->baud_conv[i].rate == arg)
			{
#ifdef  DEBUG
	logMsg("setting baud rate unit %d to %x\n", pcp_dev->unit, 
			pcp_dev->cont->baud_conv[i].csrVal);
#endif  
				lpr = (lpr & 0xFF00) | 
					pcp_dev->cont->baud_conv[i].csrVal;
			    	status = OK;
			    	break;
			}
			i++;
		}
		if (status != OK)
			logMsg("ERROR - cpIoctl - Invalid baud rate\n");
		break;
	  case FIOOPTIONS:
		if (arg == OPT_TANDEM)
        		lpr |= PR_XOFF;
		else
	  		status = tyIoctl ((TY_DEV_ID)pcp_dev, request, arg);
		pcp_dev->options = arg;
		break;
	  default:
	  	status = tyIoctl ((TY_DEV_ID)pcp_dev, request, arg);
	    	break;
	}

/* set line parameter register */
        pcp_dev->cont->cpaddr->cp_sel =
               pcp_dev->unit << SEL_SHIFT;
		
        pcp_dev->cont->cpaddr->cp_pr = lpr;

#ifdef DEBUG
	logMsg("lpr line %d = %x\n", pcp_dev->unit, lpr);
#endif
    return (status);
}

/**********************************************************************
*
* cpInterrupt - handle interrupts
*
* This routine gets called to all interrupts 
*/

LOCAL VOID cpInterror(cont)
    FAST int cont;

{
USHORT	isr;
	

	isr = cp_cont[cont].cpaddr->cp_isr;

#ifdef  DEBUG
	logMsg("cp int cont = %d, status = %x\n", cont, isr);
#endif  

        switch (isr & 0xFF) 
	{
/*
                case ISR_SI:
                        cpSint(cont);
                        break;

                case ISR_TI:
                        cptxInt(cont);
                        break;
		case ISR_CI:;
                case ISR_PI:;
                case ISR_RI:;
                case ISR_NI:;
*/
		default:
#ifdef DEBUG3
			logMsg("ERROR - cpInt - cp error type %x\n", isr);
#endif
                        break;
        }
}


/**********************************************************************
*
* cpSint - handle a receiver interrupt
*
* This routine gets called to handle a receiver interrupt.
*/

LOCAL VOID cpSint (cont)
    FAST int cont;

{
register struct cpdevice *cpaddr;
register int overrun = 0;
register ICP_DEV *pcp_dev; 
register int c;
register int line;
USHORT	isr;

 
#ifdef DEBUG
	logMsg("cp Rx int \n");
#endif

	isr = cp_cont[cont].cpaddr->cp_isr;

#ifdef  DEBUG
        if ((isr & 0xFF) != ISR_SI) 
		logMsg("cp int not correct, status = %x\n", cont, isr);
#endif  

	cpaddr = cp_cont[cont].cpaddr;  
        /* Loop fetching characters from the silo for this cp.  */
        while ((c = cpaddr->cp_swr) & SWR_VDP) 
	{
		pcp_dev = &cp_dev[(c & SWR_LN_MASK) >> SWR_LN_SHIFT];  
                if (!pcp_dev->open)
		{
#ifdef DEBUG
			logMsg("device not open\n");
#endif
			device_present = TRUE;
			continue;
		}
		if ((c & SWR_DO) && overrun++ == 0)
			if (c & SWR_PE)
				logMsg("ERROR - cp %d silo overflow\n", cont);
			else
				logMsg("ERROR - cp %d line %d overflow\n", cont,
					(c & SWR_LN_MASK) >> SWR_LN_SHIFT);
		if (c & SWR_PE)
			logMsg("ERROR - cp %d line %d parity error\n", cont,   
					(c & SWR_LN_MASK) >> SWR_LN_SHIFT);
		if (c & SWR_FE)
			logMsg("ERROR - cp %d line %d framing error\n", cont,   
					(c & SWR_LN_MASK) >> SWR_LN_SHIFT);
					
		c = c & SWR_CH_MASK;

		
	/* if XON recieved and in tandem mode restart device */
		if (c == '\021' && (OPT_TANDEM & pcp_dev->options) != 0)
		{
#ifdef DEBUG2
			logMsg("XON - restarting line %d\n", pcp_dev->unit); 
#endif
		/* start I/O */
       	      	 	cpaddr->cp_sel = pcp_dev->unit << SEL_SHIFT;
               		line = 1 << pcp_dev->unit;
					
               		cpaddr->cp_tcr = line;
		}
#ifdef DEBUG2
		if (c == '\023')
			logMsg("XOFF received\n");
#endif
			
		tyIRd ((TY_DEV_ID)pcp_dev, c);
	}
}

/**********************************************************************
*
* cptxInt - handle a transmitter interrupt
*
* This routine gets handles a xmitter interrupt.
* If there is more characters to be transmitted, cpStart is called. 
*/

LOCAL VOID cptxInt(cont)
    FAST int cont;

{
char outChar;
register USHORT tcr;
register struct cpdevice *cpaddr;
register int	line;
register ICP_DEV *pcp_dev;
USHORT	isr;

/***********/
int	bytes;
RING_ID bufpt;


	isr = cp_cont[cont].cpaddr->cp_isr;

#ifdef DEBUG
	logMsg("cp Tx int\n");
#endif
 
	cpaddr = cp_cont[cont].cpaddr;
	
	tcr = cpaddr->cp_tcr;
	

/* check all lines for transmit termination */

        for (line = 0; tcr ; line++) 
	{
                if (tcr & (1 << line)) 
		{
           	cpaddr->cp_sel = 0;
			
            tcr ^= (1 << line);
			pcp_dev = &cp_dev[line];
			if ((bytes = cpaddr->cp_bc) > 0)
			{
#ifdef DEBUG3
				logMsg("bytes left = %d\n", cpaddr->cp_bc);
#endif
				pcp_dev->state = XOFFED;
			}
			else
			{
#ifdef DEBUG
				logMsg("restarting line %d\n", line);
#endif  
#ifdef DEBUG3
	bufpt = cp_dev[line].tyDev.wrtBuf;
		logMsg("txInt - to, from = %d, %d\n", bufpt->pToBuf, bufpt->pFromBuf);	
#endif
		/* update the put pointer now that the transfer is done */
				pcp_dev->tyDev.wrtBuf->pFromBuf = pcp_dev->putpt;
				
		/* some ring buffer space is availible so give semaphore */
				if(pcp_dev->tyDev.wrtState.semNeeded == 1)
				{
				       pcp_dev->tyDev.wrtState.semNeeded = 0;
				       semGive(&cp_dev[line].tyDev.wrtSem);
				}
	
		/* mark device as idle and try to restart */
				pcp_dev->state = IDLE;
				pcp_dev->tyDev.wrtState.busy = 0;	
				cpStartup(&cp_dev[line]);
			}
				
		}
	}
}


/**********************************************************************
*
* cpStartup - Transmitter startup routine
*
* Call interrupt level character output routine for isi ICP line.
*/

LOCAL VOID cpStartup (pcp_dev)
    ICP_DEV *pcp_dev;		/* ty device to start up */

{
register struct cpdevice *cpaddr;
USHORT line;
register RING_ID bufpt;
char	*wrtbuf;
register bytes;
	
#ifdef DEBUG
	logMsg("cp startup\n");
#endif
 
/* do dma out of ring buffer */
	bufpt = pcp_dev->tyDev.wrtBuf;


/* if the device is idle check for data in the transmit buffer */
	if (pcp_dev->state == IDLE)
	{
/* check if anything is in the ring buffer */
		if(bufpt->pFromBuf != bufpt->pToBuf)
		{
			wrtbuf = &bufpt->buf[bufpt->pFromBuf];
	/* do dma to put buffer point or end of buffer whichever is first */
			if(bufpt->pToBuf > bufpt->pFromBuf)
			{
				bytes = bufpt->pToBuf - bufpt->pFromBuf;
				pcp_dev->putpt = bufpt->pToBuf;	
			}
			else
			{
				bytes = bufpt->bufSize - bufpt->pFromBuf;
				pcp_dev->putpt = 0;
			}

		/* setup transmit registers */
			cpaddr = pcp_dev->cont->cpaddr;
       	      	 	cpaddr->cp_sel = pcp_dev->unit << SEL_SHIFT;
					
       	       		cpaddr->cp_bah = hiword((int)wrtbuf - 
											(int)vme_window_std);
					
               		cpaddr->cp_bal = loword ((int)wrtbuf - 
											(int)vme_window_std);
					
               		cpaddr->cp_bc = bytes;

		/* mark channel as busy and start I/O */
			pcp_dev->state = BUSY;
			pcp_dev->tyDev.wrtState.busy = 1;	
            line = 1 << pcp_dev->unit;
			
            cpaddr->cp_tcr = line;
	
		}
#ifdef DEBUG
		else
			logMsg("cp start no characters in write buffer\n");
#endif

	}
	else
		logMsg("channel busy\n");

#ifdef DEBUG
	logMsg("unit %d, bytes %d, from %d, to %d, size %d\n", pcp_dev->unit, 
			bytes, bufpt->pFromBuf, bufpt->pToBuf, bufpt->bufSize);
#endif
}
