/* windALib.s - internal WRS kernel assembly library */

	.data
	.asciz	"Copyright 1988, Wind River Systems, Inc."

/*
modification history
--------------------
*/

/*
DESCRIPTION
This module contains internals to the WRS kernel.
These routines have been coded in assembler because they are either
specific to this processor, or they have been optimized for performance.
*/

#define ASMLANGUAGE
#include "UniWorks.h"
#include "asm.h"
#include "wind.h"

	/* locals */
	.globl	_windExit		/* routine to exit mutual exclusion */
	.globl	_vxTaskEntry		/* task entry wrapper */
	.globl	_intEnt			/* interrupt entrance routine */
	.globl	_intExit		/* interrupt exit routine */
	.globl	_windIntStackSet	/* interrupt stack set routine */

	/* globals */
	.globl	_taskIdCurrent		/* current executing task */
	.globl	_reschedule		/* resheduling routine */
	.globl	_idle			/* idle routine */
	.globl	_exit			/* task exit routine */
	.globl	_ispBot			/* bottom of interrupt stack */

	.text
	.even

/*******************************************************************************
*
* windLoadContext - load the register context from the control block
*
* The registers of the current executing task, (the one reschedule chose),
* are restored from the control block.  Then the appropriate exception frame
* for the architecture being used is constructed.  To unlock interrupts and
* enter the new context we simply use the instruction rte.
*
* NOMANUAL

* VOID windLoadContext ()

*/

_windLoadContext:
	movel	_taskIdCurrent,a0	/* current tid */
	movel	a0@(TCB_SSP),a7		/* push dummy except. */

#if (CPU == 68000)			/* 68000 EXC FRAME */
	movew	a0@(EXC_FRAME2),a7@-	/* push dummy except. */
	movel	a0@(EXC_FRAME1),a7@-	/* onto stack */

#else (CPU != 68000)			/* 68010/68020 EXC FRAME */
	movew	a0@(TCB_FOR),a7@-	/* push format/offset */
	movel	a0@(TCB_PC),a7@-	/* push new pc and sr */
	movew	a0@(TCB_SR),a7@-	/* onto stack */
#endif

	moveml	a0@(TCB_REGS),d0-d7/a0-a6	/* load register set */
	rte				/* enter new task's context. */
					/* The interrupt level will now be */
				 	/* interrupt level in the new SR */

/******************************************************************************
*
* windExit - task level exit from kernel
*
* This is the way out of kernel mutual exclusion.  If a higher priority task
* than the current task is ready, then we invoke the rescheduler.  We
* also invoke the rescheduler if any interrupts have occured which have added
* work to the windWorkList.  If rescheduling is necessary,
* the context of the calling task is saved with the PC pointing at the
* next instruction after the jsr to this routine.  The SSP in the tcb is
* modified to ignore the return address on the stack.  Thus the context saved
* is as if this routine was never called.
*
* NOMANUAL

* VOID windExit ()

*/  

_windExit:
	tstl	_intCnt		/* if intCnt == 0 we're exiting task code */
	beq	taskCode	/* else we're exiting interrupt code */
	clrl	_kernelState	/* kernelState = FALSE */
	rts			/* intExit will take care of rescheduling */
taskCode:
	movew	sr,d0		/* store sr */
	movew	#0x3700,sr	/* lock interrupts */
	movel	#_workList,a0	/* find out if anything is on work q */
	tstl	a0@		/* if pointer to first node is not NULL */
	bne	workToDo	/* then we have to reschdule */
	movel	#_readyList,a0	/* find out if ready task is us */
	movel	a0@,a0		/* get pointer to first node */
	cmpl	_taskIdCurrent,a0
	beq	noWorkToDo	/* if same then don't reschedule */
	tstl	_contextSwitching
	beq	noWorkToDo	/* if no rescheduling then don't */
workToDo:
	movew	#0x3000,sr			/* unlock interrupts to zero */
	movel	_taskIdCurrent,a0		/* current tid */
	movel	a7@,a0@(TCB_PC)			/* save PC */
	movew	d0,a0@(TCB_SR)			/* save SR */
	clrw	a0@(TCB_FOR)			/* clear the format/offset */
	moveml	d2-d7,a0@(TCB_DREGS + 8)	/* d2-d7; d0,d1 are volatile  */
	moveml	a2-a7,a0@(TCB_AREGS + 8)	/* a2-a7; a0,a1 are volatile  */
	addl	#4,a0@(TCB_SSP)			/* pop ret pc off stack ptr */
	jsr	_reschedule			/* goto rescheduler */
noWorkToDo:
	clrl	_kernelState	/* release mutual exclusion to lists */
	movew	d0,sr		/* unlock interrupts by restoring sr */
	rts			/* back to calling task */

/*******************************************************************************
*
* vxTaskEntry - task startup code following spawn
*
* This hunk of code is the initial entry point to every task created via
* the "spawn" routines.  taskCreate(2) has put the true entry point of the
* task into the tcb extension before creating the task,
* and then pushed exactly ten arguments (although the task may use
* fewer) onto the stack.  This code picks up the real entry point and calls it.
* Upon return, the 10 task args are popped, and the result of the main
* routine is passed to "exit" which terminates the task.
* This way of doing things has several purposes.  First a task is easily
* "restartable" via the routine taskRestart(2) since the real
* entry point is available in the tcb extension.  Second, the call to the main
* routine is a normal call including the usual stack clean-up afterwards,
* which means that debugging stack trace facilities will handle the call of
* the main routine properly.  
*
* NOMANUAL

* VOID vxTaskEntry () 

*/

_vxTaskEntry:
	movel	#0,a6		/* make sure frame pointer is 0 */
	movel	_taskIdCurrent,a0 /* get current task id */
	movel	a0@(TCB_EXT),a0	/* entry point is first entry in tcbx */
	jsr	a0@		/* call main routine */
	addl	#40,a7		/* pop args to main routine */
	movel	d0,a7@-		/* pass result to exit */
	jsr	_exit		/* gone for good */

/*******************************************************************************
*
* intEnt - enter interrupt level
*
* intEnt should be called when entering an interrupt.
* This normally happens automatically, from the stub built by intConnect (2).
* This routine should NEVER be called from C.
*
* SEE ALSO: intConnect(2)

* VOID intEnt ()

*/

_intEnt:
	addql	#1,_intCnt	/* Bump the counter */
	rts

/*******************************************************************************
*
* intExit - leave interrupt level
*
* This routine should be jumped to when exiting an interrupt service routine.
* This normally happens automatically, from the stub built by intConnect (2).
* This routine can NEVER be called from C.  It can only be jumped to from 
* assembly code.
*
* SEE ALSO: intConnect(2)

* VOID intExit ()

*/

#if ((CPU == MC68020) || (CPU == MC68030))

_intExit:
	movel	a0,a7@-		/* interrupt stack now has a0 on it */
	subl	#1,_intCnt	/* intCnt --; */
	tstl	_kernelState	/* if kernelState == TRUE */
	bne	intRte		/*     then just clean up and rte */ 
	btst	#4,a7@(4)	/* if M bit is set, then we were in an isr */
	beq	intRte		/*     so just clean up and rte */ 
	movew	#0x2700,sr	/* lock interrupts */
	movel	#_workList,a0	/* find out if anything is on work q */
	tstl	a0@		/* if pointer to first node is not NULL */
	bne	intMoreToDo	/* then we have to reschedule */
	movel	#_readyList,a0	/* find out if ready task is us */
	movel	a0@,a0		/* get pointer to first node */
	cmpl	_taskIdCurrent,a0
	beq	intRte		/* if same then don't reschedule */
	tstl	_contextSwitching
	bne	intMoreToDo	/* if context switch ok then exit via kernel */
intRte:
	movel	a7@+,a0		/* restore a0 */
	rte			/* unlocks interrupts if locked */

/* We come here if we have decided that rescheduling is a distinct possibility.
 * The context must be gathered and stored in the current task's tcb.
 * The stored stack pointers must be modified to clean up the stacks (ISP, MSP).
 */  

intMoreToDo:
	/* interrupts are still locked out */
	movel	#1,_kernelState			/* kernelState = TRUE; */
	movel	_taskIdCurrent,a0		/* tcb to be fixed up */
	movel	a7@+,a0@(TCB_AREGS)		/* store a0 in tcb */
	movel	_ispBot,a7			/* fix up isp */
	movew	#0x3000,sr			/* unlock int., switch to msp */

	/* interrupts unlocked and using master stack*/
	movew	a7@,a0@(TCB_SR)			/* save sr in tcb */
	movel	a7@(2),a0@(TCB_PC)		/* save pc in tcb */
	movew	a7@(6),a0@(TCB_FOR)		/* save format/offset */
	moveml	d0-d7,a0@(TCB_DREGS)		/* d0-d7 */
	moveml	a1-a7,a0@(TCB_AREGS + 4)	/* a1-a7 */

	addl	#8,a0@(TCB_SSP)			/* adj master stack ptr */
	jsr	_reschedule			/* goto rescheduler */

#else ((CPU != MC68020) && (CPU != MC68030))

_intExit:
	movel	d0,a7@-		/* interrupt stack now has d0 on it */
	movel	a0,a7@-		/* interrupt stack now has a0 on it */
	subl	#1,_intCnt	/* intCnt --; */
	tstl	_kernelState	/* if kernelState == TRUE */
	bne	intRte		/*     then just clean up and rte */ 

	movew	a7@(8),d0	/* get the sr off the stack and into a0 */
	andw	#0x0700,d0	/* if the interrupt level > 0 */
	bne	intRte		/*     then just clean up and rte */ 

	movew	#0x2700,sr	/* lock interrupts */
	movel	#_workList,a0	/* find out if anything is on work q */
	tstl	a0@		/* if pointer to first node is not NULL */
	bne	intMoreToDo	/* then we have to reschedule */
	movel	#_readyList,a0	/* find out if ready task is us */
	movel	a0@,a0		/* get pointer to first node */
	cmpl	_taskIdCurrent,a0
	beq	intRte		/* if same then don't reschedule */
	tstl	_contextSwitching
	bne	intMoreToDo	/* if context switching then exit via kernel */
intRte:
	movel	a7@+,a0		/* restore a0 */
	movel	a7@+,d0		/* restore d0 */
	rte			/* unlocks interrupts if locked */

/* We come here if we have decided that rescheduling is a distinct possibility.
 * The context must be gathered and stored in the current task's tcb.
 * The stored stack pointer must be modified to clean up the SSP.
 */  

intMoreToDo:
	/* interrupts are still locked out */
	movel	#1,_kernelState			/* kernelState = TRUE; */
	movel	_taskIdCurrent,a0		/* tcb to be fixed up */
	movel	a7@+,a0@(TCB_AREGS)		/* store a0 in tcb */
	movel	a7@+,a0@(TCB_DREGS)		/* store d0 in tcb */
	movew	#0x3000,sr			/* unlock int., switch to msp */

	/* interrupts unlocked and using master stack*/
	movew	a7@,a0@(TCB_SR)			/* save sr in tcb */
	movel	a7@(2),a0@(TCB_PC)		/* save pc in tcb */
	moveml	d1-d7,a0@(TCB_DREGS + 4)	/* d1-d7 */
	moveml	a1-a7,a0@(TCB_AREGS + 4)	/* a1-a7 */
#if (CPU == MC68010)
	movew	a7@(6),a0@(TCB_FOR)		/* save format/offset */
	addl	#8,a0@(TCB_SSP)			/* adj master stack ptr */
#else (CPU == MC68000)
	addl	#6,a0@(TCB_SSP)			/* adj master stack ptr */
#endif	MC68010
	jsr	_reschedule			/* goto rescheduler */

#endif	!68020

/*******************************************************************************
*
* windIntStackSet - set the interrupt stack pointer
*
* This routine sets the interrupt stack pointer to the specified address.
* It is only valid on architectures with an interrupt stack pointer.
*
* NOMANUAL

* VOID windIntStackSet (pBotStack)
*     char *pBotStack;	/* pointer to bottom of interrupt stack *

*/  

_windIntStackSet:
	link	a6,#0
#if ((CPU == MC68020) || (CPU == MC68030))
	movel	a6@(ARG1),d1	/* get new botOfIsp */
	movew	sr,d0		/* save sr */
	movew	#0x2700,sr	/* get ISP to SSP and lock interrupts */
	movel	d1,a7		/* set stack */
	movew	d0,sr		/* restore sr and unlock interrupts */
	movel	d1,_ispBot	/* update the global variable */
#endif	MC68020
	unlk	a6
	rts

/*******************************************************************************
*
* reschedule - rescheduler for WRS kernel
*
* This routine is called when either intExit, or windExit, thinks the
* context might change.  All of the contexts of all of the tasks are
* accurately stored in the task control blocks when entering this function.
* The status register is 0x3000. (Supervisor, Master Stack, Interrupts UNLOCKED) 
* At the conclusion of this routine, taskIdCurrent will equal the highest
* priority task eligible to run, the kernel work queue will be empty, and
* vxLastTask will equal the last task that ran.  Note that vxLastTask will
* often equal taskIdCurrent as in the case of resuming a lower priority task.
* vxLastTask is used to decide whether or not switchHooks should be invoked.
*
* NOMANUAL

* VOID reschedule ()

*/

_reschedule:
	bra	while		/* typical while construct */
whileInards:
	movel	a4@(0x10),a7@-	/* push arg2 */
	movel	a4@(0xc),a7@-	/* push arg1 */
	movel	a4@(0x8),a1	/* load function ptr. */
	jsr	a1@		/* invoke function */
	addl	#8,a7		/* clean up stack */
	movel	a4,a7@-		/* push pNode */
	pea	_freeJobList	/* push &freeJobList */
	movew	sr,d4		/* save sr */
	movew	#0x3700,sr	/* LOCK INTERRUPTS */
	jsr	_lstAdd		/* add old job to freeJobList */
	movew	d4,sr		/* UNLOCK INTERRUPTS */
	addl	#8,a7		/* clean up stack */
while:
	pea	_workList	/* deque work from workList */
	movew	sr,d4		/* save sr */
	movew	#0x3700,sr	/* LOCK INTERRUPTS */
	jsr	_lstGet
	movew	d4,sr		/* UNLOCK INTERRUPTS */
	addl	#4,a7		/* clean up stack */
	movel	d0,a4		/* put in a4 expected for whileInards */
	movel	d0,d0		/* set condition codes */
	bne	whileInards	/* while work to be done, do it */

workListEmpty:
	movel	#_taskIdCurrent,a2
	movel	#_vxLastTask,a3
	movel	a2@,a3@			/* vxLastTask = _taskIdCurrent */
	tstl	_contextSwitching	/* If no context switching */
	beq	noSwitch		/* then don't switch */
	movel	#_readyList,a0		/* Get highest ready task. */
	movel	a0@,a2@			/* Store in taskIdCurrent. */
	tstl	a2@			/* If NULL we're in troble */
	bne	notIdle			/* because the idle task should */
	clrl	_kernelState		/* never die */
	jsr	_idle
notIdle:
	movel	a3@,d0			/* Compare vxLastTask to */
	cmpl	a2@,d0			/* to taskIdCurrent */
	beq	noSwitch		/* if the same then no context switch */
	movel	a2@,a2			/* get pTcbs into regs */
	movel	a3@,a3
	addl	#TCB_EXT,a2		/* add offset to the tcb_extension */
	addl	#TCB_EXT,a3
	movel	a2,a7@-			/* push pointer to new tcbX */
	movel	a3,a7@-			/* push pointer to old tcbX */

	movel	#_taskSwitchTable,a2	/* get adrs of task switch rtn list */

doHook:
	movel	a2@,d0			/* get next task switch rtn */
	beq	noMoreHooks		/* quit if end of list */

	movel	d0,a1
	jsr	a1@			/* call routine */

	addl	#4,a2			/* bump to next task switch routine */
	bra	doHook			/* loop */

noMoreHooks:
	addl	#8,a7			/* clean up stack */

noSwitch:
	movew	sr,d4			/* save sr */
	movew	#0x3700,sr		/* LOCK INTERRUPTS */
	movel	#_workList,a0		/* did anyone q more work */
	tstl	a0@			/* If no work */
	beq	goGo			/* then all clear to load context */
	movew	d4,sr			/* else UNLOCK INTERRUPTS and */
	bra	while			/* go back to the while statement */
goGo:
	clrl	_kernelState		/* release mutual exclusion to lists */
	jsr	_windLoadContext	/* load taskIdCurrent */


#if	FALSE

/*******************************************************************************
*
* reschedule - C version of reschedule
*
* This was the original implemenation of the reschedule program... somewhat
* out of date now.
*
* NOMANUAL
*/

VOID reschedule ()
    {
    JOB *pJob;
    int oldLevel;

unlucky:
    oldLevel = intLock ();			/* LOCK INTERRUPTS */
    while ((int)(pJob = (JOB *) lstGet (&workList)) != NULL)
	{
	intUnlock (oldLevel);			/* UNLOCK INTERRUPTS */
        (FUNCPTR *)(pJob->function) (pJob->arg1, pJob->arg2);	/* do work */
	oldLevel = intLock ();			/* LOCK INTERRUPTS */
	lstAdd (&freeJobList, (NODE *) pJob);
	intUnlock (oldLevel);			/* UNLOCK INTERRUPTS */
	}

    vxLastTask = taskIdCurrent;

    if (contextSwitching)	/* if not, we got here just to empty work q */
	taskIdCurrent = (int) lstFirst (&readyList);

    if (taskIdCurrent == NULL)	/* nobody is left to run, the idle task died?!*/
	{
	kernelState --;		/* release mutual exclusion */
	idle ();		/* go idle */
	}

    if (taskIdCurrent != vxLastTask)		/* call the switch hook */
	switchHook ((WIND_TCB *) vxLastTask, (WIND_TCB *) taskIdCurrent);

    oldLevel = intLock ();			/* LOCK INTERRUPTS */
    if (lstFirst (&workList) != NULL)
	{
	intUnlock (oldLevel);			/* UNLOCK INTERRUPTS */
	goto unlucky;				/* back to while */
	}
    else
	{
	kernelState --;			/* release mutual exclusion */
	windLoadContext ();		/* switches context too. */
	}
    }
#endif	FALSE
