/* semLib.c - WRS kernel semaphore library */

static char *copyright = "Copyright 1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION
This library provides a generic interface to the WRS kernel semaphores.
Semaphore management includes:

    - create/delete a semaphore,
    - initialize a semaphore,
    - take a semaphore,
    - take a semaphore if available,
    - give a semaphore.

SEMAPHORES
This implementation provides a binary semaphore facility.  Semaphores can
be used to control access to shared devices or data structures, and
to synchronize multiple tasks, or task level and interrupt level processes.

A semaphore is just a cell in memory which contents is globally available.
The semaphore must be initialized (by semInit) before it
is ever used.  Thereafter, every time the semaphore is given (semGive) it
becomes full, and every time it is taken (semTake), it becomes empty.

TASK STATE CHANGES
When a task takes a semaphore that is empty, that task will pend on the
filling of the semaphore.  Any number of tasks may be simultaneously pended on
the same semaphore.  If the semaphore is full when the semTake is done,
the semaphore is emptied and the calling task continues running.  At most,
a single task may have an instance of a semaphore at one time.

When a task gives a semaphore, that semaphore becomes full.  If any tasks are
pending on this semaphore the kernel unblocks the highest priority task waiting
on this semaphore, and the semaphore is emptied once again.  Otherwise, the
semaphore is simply filled and processing continues with the calling task.
Note that if a semaphore is given, and there is a task pending on this semaphore
of equal or higher priority than the task doing the semGive, that task will
unblock and execute, and the giving task will be preempted.  Unblocking a task
of lower priority, puts that task on the ready list but will not preempt the
calling task.

INTERRUPT LEVEL USE
Semaphores may be given, but not taken, from interrupt level.  An interrupt
level process that gives a semaphore will not be preempted, no matter who
becomes ready, but the task that had been running when the interrupt
occurred may be preempted when an interrupt service routine completes.

PROTOCOL HINTS
Although there is no restriction to many tasks giving, getting, and initializing
the same semaphore, one must be careful to ensure the proper functionality of
this construct.

There is no danger in any number of processes taking a semaphore as suited.
SemGives and semInits, however, should be more carefully controlled.  If
semGives are done only by one task, there is no problem.  If they are done by
multiple tasks, only a task that already has the semaphore should
give it.  Otherwise, multiple copies of the semaphore could be running
around, and mutual-exclusion will no longer be in effect.  SemInit's should
only be performed if some task (perhaps the same one that does the semInit)
is going to do a semGive without doing a semTake first.

DEFICIENCIES
There is no mechanism to give back semaphores automatically
when tasks are suspended or deleted.  If a task ceases execution unexpectedly
(a bus error, deleted from the shell, etc.) any semaphores that it currently
has taken will never be given again, unless there is a watchdog or something
in operation that can do it.  Since the I/O system uses semaphores, this
can cause a device to get hung up if a task doing I/O to that device has
blown up.

SEE ALSO: "Architecture", taskLib (1), kernelLib (1)
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "semLib.h"
#include "memLib.h"
#include "wind.h"

IMPORT BOOL kernelState;	/* for use in kernel mutual exclusion */
IMPORT VOID windSemGive ();
IMPORT VOID windSemTake ();


/*******************************************************************************
*
* semCreate - create and initialize a semaphore
*
* Create a semaphore.  The semaphore is initially empty.  After
* initialization, it must be given before it can be taken.
*
* RETURNS: SEM_ID or NULL if unable to allocate memory
*
* SEE ALSO: semInit (2)
*/

SEM_ID semCreate ()

    {
    FAST SEM_ID semId = (SEM_ID) malloc ((unsigned) sizeof (SEMAPHORE));

    if (semId != NULL)
	{
	lstInit (&semId->semList);	/* initialize blocked list */
	semId->count = 0;		/* semaphore initially taken */
	}

    return (semId);
    }
/*******************************************************************************
*
* semGive - give semaphore
*
* Gives the semaphore.  If a higher priority task has already taken
* the semaphore (so that it is now pended waiting for it), that task
* will now become ready to run, and preempt the task that does the semGive.
* If the semaphore is already full (it has been given but not taken) this
* call is essentially a no-op.
*/

VOID semGive (semId)
    FAST SEM_ID semId;	/* semaphore id to give */

    {
    if (semId->count == 1)	/* already available, so just return */
	return;

    if (kernelState)		/* if TRUE then we interrupted the kernel */
	windWorkAdd (windSemGive, (int) semId);
    else
	{
	kernelState = TRUE;			/* KERNEL_ENT */
	windSemGive (semId);			/* resort the kernel lists */
	windExit ();				/* KERNEL_EXIT */
	}
    }
/*******************************************************************************
*
* semTake - take semaphore
*
* Takes the semaphore.  If the semaphore is empty (it has not been given
* since the last semTake or semInit), this task will become pended until
* the semaphore becomes available by some other task doing a semGive
* of it.  If the semaphore is already available, this call will empty
* the semaphore, so that no other task can take it until this task gives
* it back, and this task will continue running.
*
* WARNING
* This routine may not be used from interrupt level.
*/

VOID semTake (semId)
    SEM_ID semId;	/* semaphore id to take */

    {
    if (kernelState)		/* if TRUE then we interrupted the kernel */
	logMsg ("semTake: Can't take a semaphore from interrupt level.\n");
    else
	{
	kernelState = TRUE;			/* KERNEL_ENT */
	windSemTake (semId);			/* resort the kernel lists */
	windExit ();				/* KERNEL_EXIT */
	}
    }
/*******************************************************************************
*
* semDelete - delete semaphore
*
* Frees the memory allocated for the semaphore.  A deleted semaphore can
* not be resurrected.  If the semaphore was declared as a global variable
* and not created via semCreate then this should NOT be called.
*
* WARNING
* This routine may not be used from interrupt level.
*/  

VOID semDelete (semId)
    SEM_ID semId;	/* semaphore id to delete */

    {
    /* someday we should signal the tasks pended on the semaphore */
    free ((char *) semId);
    }
/******************************************************************************
*
* semClear - take semaphore if semaphore is available
* 
* If the semaphore is available, it is taken, otherwise no action is taken.
* This routine never preempts the caller.  If semaphore was not available
* this routine returns ERROR.
*
* RETURNS: OK or ERROR if semaphore was unavailable.
*/  

STATUS semClear (semId)
    SEM_ID semId;	/* semaphore id to empty */

    {
    STATUS status = ERROR;		/* assume the worst */

    kernelState = TRUE;			/* KERNEL_ENT */

    if (semId->count != 0)
	{
	semId->count = 0;	/* clear the semaphore count */
	status = OK;		/* we are the only one who has it */
	}

    windExit ();			/* KERNEL_EXIT */

    return (status);
    }
/******************************************************************************
*
* semInit - initialize a declared semaphore
*
* In some instances a semaphore cannot be created with semCreate (2)
* but is global variable.  This routine must be called to initialize it.
*
* SEE ALSO: semCreate (2)
*/  

VOID semInit (semId)
    SEM_ID semId;	/* semaphore id to initialize */

    {
    lstInit (&semId->semList);
    semId->count = 0;
    }
