/* paslib.c - UniWorks Pascal support library */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/*
modification history
--------------------
*/

/*
DESCRIPTION

Since each task which uses Pascal needs it's own _input and _output structure, 
_input and _output have been implemented as task variables, similar to 
UniWorks stdio's stdin, stdout, and stderr. Each task which uses Pascal
should be spawned with both the VX_PASCAL option and the VX_STDIO option,
since the Pascal library is built on top of stdio. 

CAVEATS

Two problems occured in implementing this Pascal library. First, the address 
of _input and _output are sent to the Pascal library routines. These routines 
are also passed pointers to structures of the same type as _input and _output, 
when file I/O is used. Since _input and _output are now pointers, these 
routines sometimes recieve a pointer to a struct and sometimes a pointer to 
a pointer to a struct. So I had to add a condition to each routine to check 
for the pointer being equal to _input or _output and then adjusting the 
pointer if not.

Second, both the _input and _output structs have an entry for stdin and 
stdout respectively. Since stdin and stdout are task variables, they are 
not defined when the task creation routine is run and the _input and _output 
structs are initialized (the task variables get set in the task variable
switch hook routine which is run after the create hook). So I've added a 
condition to each of the checks mentioned above, to set the stdin and stdout 
variables in each routine.

*/

#include "UniWorks.h"
#include "stdioLib.h"
#include "pasLib.h"
#include "types.h"
#include "taskLib.h"
#include "memLib.h"

/*#define	DEBUG	/* */
/*#define	DEBUG1	/* */

struct pasftype *_input;
struct pasftype *_output;
int             pasdbg = 1;

/* forward declarations */

VOID pasCreateHook ();
VOID pasDeleteHook ();
LOCAL struct pasftype *makeFile ();

#define	PINPUT	0
#define	POUTPUT	1

/****************************************************************************
*
* pascalInit - initialize Pascal support
*
* This routine must be called before using the pasLib.
* It is usually called by the root task, usrRoot(2), in usrConfig(1).
*
*/

VOID 
pascalInit()
{
	static BOOL     pasInitialized = FALSE;

#ifdef	DEBUG1
	printf("pascalInit, &_input=0x%x, &_output=0x%x\n", &_input, &_output);
#endif	DEBUG1

	if (!pasInitialized) {
		if (taskCreateHookAdd(pasCreateHook) == ERROR ||
		    taskDeleteHookAdd(pasDeleteHook) == ERROR) {
			logMsg("pascalInit: unable to add create/delete hooks!\n");
		} else
			pasInitialized = TRUE;
	}
}

/****************************************************************************
*
* pasCreateHook - init Pascal support for task
*
* This routine creates input and output pasftype structs for tasks
* with VX_STDIO and VX_PASCAL options set.
*
*/

STATUS pasCreateHook(pTcbx)
FAST TCBX      *pTcbx;
{

#ifdef	DEBUG1
	logMsg("pasCreateHook, pTcbx=0x%x\n",pTcbx);
#endif	DEBUG1

	/* check for option bits */
	if ((pTcbx->options & VX_STDIO) && (pTcbx->options & VX_PASCAL)) {

		if (((pasInitFile(pTcbx->taskId, &_input, PINPUT)) == ERROR) ||
		    ((pasInitFile(pTcbx->taskId, &_output, POUTPUT)) == ERROR))
			return (ERROR);
	}
	errnoSet(0);
	return (OK);
}

/****************************************************************************
*
* pasDeleteHook - delete Pascal support for task
*
*/

STATUS pasDeleteHook(pTcbx)
FAST TCBX      *pTcbx;
{

#ifdef	DEBUG1
	logMsg("pasDeleteHook, pTcbx=0x%x\n",pTcbx);
#endif	DEBUG1

    /* check for option bit */

    if (pTcbx->options & VX_PASCAL) {
	pasExitFile (pTcbx->taskId, &_input, PINPUT);
	pasExitFile (pTcbx->taskId, &_output, POUTPUT);
	}
}

/****************************************************************************
*
* pasInitFile - init a task's input struct
*
*/
STATUS pasInitFile(tid, ppFtype, ioflag)
	FAST int        tid;
	struct pasftype **ppFtype;
	FAST int        ioflag;
{
	struct pasftype *fptr;

#ifdef	DEBUG
	fprintf(stderr, "pasInitFile, tid=%d, ppFtype=0x%x, ioflag=%d\n",
		tid, ppFtype, ioflag);
#endif	DEBUG

	if ((taskVarAdd(tid, (int *) ppFtype) != OK) ||
	    ((fptr = makeFile()) == NULL) ||
	    taskVarSet(tid, (int *) ppFtype, (int) fptr) == ERROR) {
		logMsg("pasLib: error initializing file struct - status = %#x\n",
		       errnoGet());
		return (ERROR);
	}

	/* Initialize the file structs */

	fptr->elementsize = 1;
	fptr->open = 1;
	fptr->linemark = 1;
	fptr->valid = 0;
	fptr->text = 1;
	fptr->temp = 0;
	fptr->filename = NULL;
	fptr->tid = tid;

	/* 
	 * stdin and stdout are not defined for this task yet, since
	 * we're not in the task's context yet. stdin and stdout get
	 * set for this task in the taskVarSwitchHook routine which
	 * follows the create hooks.
	 */

	if (ioflag == PINPUT) {
		fptr->read = 1;
		fptr->eof = 0;
		fptr->filevar = stdin;
	} else {
		fptr->read = 0;
		fptr->eof = 1;
		fptr->filevar = stdout;
	}

	return (OK);
}

/****************************************************************************
*
* pasExitFile - deallocate task's file structs
*
*/

VOID pasExitFile (tid, ppFtype, ioflag)
	FAST int        tid;
	struct pasftype **ppFtype;
	FAST int        ioflag;
{
	struct pasftype *pFile;

	pFile = (struct pasftype *) taskVarGet (tid, (int *) ppFtype);

	if (pFile == (struct pasftype *) ERROR) {
		logMsg ("pasLib: can't access task variable - status = %#x\n",
		errnoGet());
	} else {
		if (pFile == NULL)
	    		logMsg ("pasLib: task variable is NULL!\n");
		taskVarDelete (tid, (int *) ppFtype);
	}
}

static char *copy(str) 
char *str;
{
	char           *malloc();
	register char  *pt;

	pt = malloc(strlen(str) + 1);
	strcpy(pt, str);
	return (pt);
}

_rewrite(pfptr, filename, elementsize)
	struct pasftype **pfptr;
	char           *filename;
{
	FILE           *fopen();
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_rewrite, f=0x%x, fd=%d, filename=%s\n",
	       f, f->filevar->fd, filename);
#endif	DEBUG

	if (filename == 0) {
		paserr("Can't open NULL filename\n", S_pasLib_BAD_FILENAME);
		return (-1);
	} else
		filename = copy(filename);
	if (f->open) {
		if (!f->linemark && f->text)
			_writeln(f);
		fclose(f->filevar);
	}
	f->filename = filename;
	if ((f->filevar = fopen(filename, "w")) == NULL) {
		sprintf(str, "Error opening file %s\n", filename);
		paserr(str, S_pasLib_FILE_OPEN_ERR);
		return (-1);
	} else {
		f->open = TRUE;
		f->read = FALSE;
		f->elementsize = elementsize;
		f->linemark = TRUE;
		f->valid = FALSE;
		f->text = FALSE;
		f->eof = TRUE;
	}
}

_put(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_putf, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	} else if (f->read) {
		paserr("File not open for writing\n", S_pasLib_FILE_READ_ONLY);
		return (-1);
	} else if (!f->valid) {
		paserr("Buffer variable undefined\n", S_pasLib_BUF_VAR_ERR);
		return (-1);
	} else
		fwrite(&f->buffervar, f->elementsize, 1, f->filevar);
	f->valid = FALSE;
	f->linemark = FALSE;
}

_writeln(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	}
	else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr, "_writeln, pfptr=0x%x, f=0x%x, fd=%d\n", 
		pfptr, f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	putc('\n', f->filevar);
	f->linemark = TRUE;
	f->text = TRUE;
}

_page(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_page, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (!f->linemark)
		_writeln(f);
	putc('\014', f->filevar);
}

_writechr(pfptr, c, totalwidth)
	struct pasftype **pfptr;
	char            c;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_writechr, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (totalwidth <= 0) {
		paserr("Width or fracdig is not positive\n",
		       S_pasLib_FORMAT_ERR);
		return (-1);
	}
	fprintf(f->filevar, "%*c", totalwidth, c);
	f->text = TRUE;
}

_writeint(pfptr, i, totalwidth)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_writeint, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (totalwidth <= 0) {
		paserr("Width or fracdig is not positive\n",
		       S_pasLib_FORMAT_ERR);
		return (-1);
	}
	fprintf(f->filevar, "%*d", totalwidth, i);
	f->text = TRUE;
}

_writeexp(pfptr, r, totalwidth)
	struct pasftype **pfptr;
	double          r;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_writeexp, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (totalwidth <= 0) {
		paserr("Width or fracdig is not positive\n",
		       S_pasLib_FORMAT_ERR);
		return (-1);
	}
	fprintf(f->filevar, " %*e", totalwidth, r);
	f->text = TRUE;
}

_writefix(pfptr, r, totalwidth, fracdig)
	struct pasftype **pfptr;
	double          r;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_writefix, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (totalwidth <= 0) {
		paserr("Width or fracdig is not positive\n",
		       S_pasLib_FORMAT_ERR);
		return (-1);
	}
	fprintf(f->filevar, "%*.*f", totalwidth, fracdig, r);
	f->text = TRUE;
}

_writeboo(pfptr, b, totalwidth)
	struct pasftype **pfptr;
	char            b;
{
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

	if (b == 1)
		_writestr(f, "TRUE", totalwidth, 4);
	else
		_writestr(f, "FALSE", totalwidth, 5);
}

_writestr(pfptr, s, totalwidth, strlen)
	struct pasftype **pfptr;
	char           *s;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_writestr, pfptr=0x%x, f=0x%x, fd=%d\n", 
		&_output,_output,pfptr, f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	if (totalwidth <= 0) {
		paserr("Width or fracdig is not positive\n",
		       S_pasLib_FORMAT_ERR);
		return (-1);
	}
	if (totalwidth < strlen)
		strlen = totalwidth;
	fprintf(f->filevar, "%*.*s", totalwidth, strlen, s);
	f->text = TRUE;
}

_reset(pfptr, filename, elementsize)
	struct pasftype **pfptr;
	char           *filename;
{
	FILE           *fopen();
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_reset, f=0x%x, filename=%s, fd=%d\n",
	       f, filename, f->filevar->fd);
#endif	DEBUG

	if (!f->linemark && f->open && f->text)
		_writeln(f);
	if (!f->open)
		f->text = FALSE;
	if (filename != 0) {
		if (f->open) {
			fclose(f->filevar);
		}
		if ((f->filevar = fopen(filename, "r")) == 0) {
			sprintf(str, "Error opening file %s\n", filename);
			paserr(str, S_pasLib_FILE_OPEN_ERR);
			return (-1);
		} else {
			f->open = TRUE;
			f->read = TRUE;
			f->elementsize = elementsize;
			f->temp = FALSE;
			f->filename = copy(filename);
		}
	} else if (f->open) {
		fclose(f->filevar);
		f->filevar = fopen(f->filename, "r");
		f->read = TRUE;
	} else {
		paserr("Cannot reset undefined file variable\n",
		       S_pasLib_UNDEF_FILE_VAR);
		return (-1);
	}
	f->linemark = FALSE;	/* pascal says so */
	f->valid = FALSE;
	f->text = FALSE;
	f->eof = FALSE;
}

_setupfil(pfptr)			/* evar(f) */
	struct pasftype **pfptr;
{
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_setupfil, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (f->read && !f->valid) {
		_read(f, &f->buffervar);
		if (!f->eof)
			f->valid = TRUE;
		f->linemark = FALSE;
	} else if (!f->read)
		f->valid = TRUE;
}

_eof(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_eof, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	_setupfil(f);
	return (f->eof);
}

_get(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_get, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	/* this code is repeated in gettext */
	if (!f->read) {
		sprintf(str, "File %s not open for reading\n", f->filename);
		paserr(str, S_pasLib_FILE_WRT_ONLY);
		return (-1);
	} else if (f->valid)
		f->valid = FALSE;
	else if (f->eof) {
		paserr("Attempt to read past end of file\n", S_pasLib_EOF);
		return (-1);
	} else
		_read(f, &f->buffervar);
}

_read(pfptr, v)
	struct pasftype **pfptr;
	char           *v;
{
	register int    cnt;
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_read, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	} else if (!f->read) {
		sprintf(str, "File %s not open for reading\n", f->filename);
		paserr(str, S_pasLib_FILE_WRT_ONLY);
		return (-1);
	} else if (f->eof) {
		paserr("Attempt to read past end of file\n", S_pasLib_EOF);
		return (-1);
	} else if (feof(f->filevar))
		f->eof = TRUE;
	else if ((cnt = fread(v, f->elementsize, 1, f->filevar)) == -1) {
		sprintf(str, "File %s is not an even number of records long\n",
			f->filename);
		paserr(str, S_pasLib_FILE_REC_ERR);
		return (-1);
	} else if (cnt == 0 && f->text && !f->linemark)
		*v = '\n';
	else if (cnt == 0)
		f->eof = TRUE;
	f->valid = FALSE;	/* since the buffervar doesn't contain it, v
				 * does */
}

_eoln(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_eoln, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	}
	_setuptex(f);
	if (f->eof) {
		paserr("Attempt to read beyond end of file\n", S_pasLib_EOF);
		return (-1);
	}
	return (f->linemark);
}

_setuptex(pfptr)			/* tfilevar(f) */
	struct pasftype **pfptr;
{
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_setuptex, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (f->valid) {
		if (f->buffervar == '\n') {
			f->buffervar = ' ';
			f->linemark = TRUE;
		}
		return;
	}
	f->text = TRUE;
	_setupfil(f);
	if (f->buffervar == '\n') {
		f->buffervar = ' ';
		f->linemark = TRUE;
	} else
		f->linemark = FALSE;
}

_gettext(pfptr)
	struct pasftype **pfptr;
{
	char            str[80];
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_gettext, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	/* this code is a copy of get */
	if (!f->open) {
		sprintf(str, "File %s not open\n", f->filename);
		paserr(str, S_pasLib_FILE_NOT_OPEN);
		return (-1);
	} else if (f->valid)
		f->valid = FALSE;
	else if (f->eof) {
		paserr("Attempt to read past end of file\n", S_pasLib_EOF);
		return (-1);
	} else
		_read(f, &f->buffervar);
}

_readln(pfptr)
	struct pasftype **pfptr;
{
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_readln, pfptr=0x%x, f=0x%x, fd=%d\n", 
		pfptr, f, f->filevar->fd);
#endif	DEBUG

	while (!_eoln(f))
		_gettext(f);
	_gettext(f);
}

_readchr(pfptr)
	struct pasftype **pfptr;
{
	char            ret;
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_readchr, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (f->valid) {
		ungetc(f->buffervar, f->filevar);
		f->valid = FALSE;
	}
	if (fscanf(f->filevar, "%c", &ret) != 1) {
		paserr("Invalid character\n", S_pasLib_INPUT_ERR);
		return (-1);
	}
	return ret;
}

_readint(pfptr)
	struct pasftype **pfptr;
{
	int             ret;
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_readint, pfptr=0x%x, &_input=0x%x, f=0x%x, fd=%d, stdin=0x%x\n", 
		pfptr, &_input, f, f->filevar->fd, stdin);
#endif	DEBUG

	if (f->valid) {
		ungetc(f->buffervar, f->filevar);
		f->valid = FALSE;
	}
	if (fscanf(f->filevar, "%d", &ret) != 1) {
		paserr("Invalid integer\n", S_pasLib_INPUT_ERR);
		return (-1);
	}
	return ret;
}

double
_readreal(pfptr)
	struct pasftype **pfptr;
{
	double          ret;
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else
		f = (struct pasftype *) pfptr;

#ifdef	DEBUG
	fprintf(stderr,"_readreal, f=0x%x, fd=%d\n", f, f->filevar->fd);
#endif	DEBUG

	if (f->valid) {
		ungetc(f->buffervar, f->filevar);
		f->valid = FALSE;
	}
	if (fscanf(f->filevar, "%lf", &ret) != 1) {
		paserr("Invalid real\n", S_pasLib_INPUT_ERR);
		return (-1);
	}
	return ret;
}

NEW(pt, size)
	char          **pt;
	int             size;
{
	char           *calloc();

	*pt = calloc(1, size);
	if (*pt == NULL) {
		paserr("Memory limit exceeded\n", S_pasLib_NEW_ERR);
		return (-1);
	}
}

DISPOSE(pt, size)
	char           *pt;
	int             size;
{

	if (pt == 0) {
		paserr("Attempted to dispose NIL object\n",
		       S_pasLib_DISPOSE_ERR);
		return (-1);
	} else
		cfree(pt);
}
double 
_log(x)
	double          x;
{
	double          log();

	if (x <= 0.0) {
		paserr("Negative or zero argument to ln\n",
		       S_pasLib_NEG_OR_0_ERR);
		return (-1);
	}
	return (log(x));
}

double 
_sqrt(x)
	double          x;
{
	double          sqrt();

	if (x < 0.0) {
		paserr("Negative argument to sqrt\n",
		       S_pasLib_NEG_OR_0_ERR);
		return (-1);
	}
	return (sqrt(x));
}

__exit()
{
	exit(0);
}

pclose(pfptr)
	struct pasftype **pfptr;
{
	struct pasftype *f;

	if(*pfptr == _input) {
		f = *pfptr;
		f->filevar = stdin;
	} else if(*pfptr == _output) {
		f = *pfptr;
		f->filevar = stdout;
	} else
		f = (struct pasftype *) pfptr;

	if (f->filevar == NULL) {
		paserr("Can't close NULL file variable\n",
		       S_pasLib_BAD_FILEVARIABLE);
		return (-1);
	}
	fclose(f->filevar);
}

pasFileShow(f)
	struct pasftype *f;
{
	fprintf(stderr,"elementsize=%d\n",f->elementsize);
	fprintf(stderr,"open=%d\n",f->open);
	fprintf(stderr,"read=%d\n",f->read);
	fprintf(stderr,"linemark=%d\n",f->linemark);
	fprintf(stderr,"valid=%d\n",f->valid);
	fprintf(stderr,"text=%d\n",f->text);
	fprintf(stderr,"eof=%d\n",f->eof);
	fprintf(stderr,"temp=%d\n",f->temp);
	fprintf(stderr,"filename=%s\n",f->filename);
	fprintf(stderr,"filevar=0x%x\n",f->filevar);
	fprintf(stderr,"fd=%d\n",f->filevar->fd);
	fprintf(stderr,"tid=%d\n",f->tid);
}

pasDbg(level)
{
	pasdbg = level;
	return (level);
}

paserr(msg, err)
	char           *msg;
	int             err;
{
	switch (pasdbg) {
	case PAS_DBG_LEV_ERR_ONLY:	/* set errno only */
		errnoSet(err);
		break;

	case PAS_DBG_LEV_ERR_MSG:	/* set errno and print msg */
		errnoSet(err);
		printf("%s", msg);
		break;

	case PAS_DBG_LEV_ERR_MSG_DIE:	/* set errno, print msg and die */
		errnoSet(err);
		printf("%s", msg);
		exit();
		break;

	case PAS_DBG_LEV_ERR_DIE:	/* set errno and die */
		errnoSet(err);
		exit();
		break;
	}
}

/****************************************************************************
*
* makeFile - make a new file buffer
*
* RETURNS: pointer to newly malloc'd file, or NULL if out of memory
*/

LOCAL struct pasftype *makeFile()
{
	FAST struct pasftype *fp;

	fp = (struct pasftype *) malloc(sizeof(struct pasftype));

	if(fp == NULL)
		return(NULL);
	return (fp);
}

SETPAIR(s,low,high)
char s[];
{
    unsigned i;

    for (i = low; i <= high; i++)
	s[i>>3] |= (1 << (i & 7));
}
PMOD(x, y)
register x,y;
{
    x = x % y;
    if (x != 0)
	if (x < 0 && y > 0 || x > 0 && y < 0)
	    x += y;
    return(x);
}

PDIV(x, y)
register x,y;
{
    return((x - PMOD(x, y)) / y);
}
