/* qs.c - ISI lower level SCSI driver for Liberator/WD33C93 */

static char *copyright = "Copyright 1988, Integrated Solutions, Inc.";

/* 
modification history
--------------------
*/

/*
DESCRIPTION
*/

#define	SCSI_LIB

/*#define	DEBUG		/**/
/*#define	DEBUG_PHASE		/**/

#include "UniWorks.h"
#include "types.h"
#include "semLib.h"

#include "iv68k.h"
#include "openchip.h"
#include "qs.h"
#include "qsreg.h"
#include "qsvar.h"
#include "scsi.h"

/* The Whole Story ... 

            --------                        --------
            | CONT |                        | CONT |
            --------                        --------            . . .
        / / / | | \ \ \                 / / / | | \ \ \ 
  TARG0(CTLR0) ...  TARG7(CTLR7)  TARG0(CTLR8)...   TARG7(CTLR15)
    ///||\\\         ///||\\\      ///||\\\          ///||\\\
   LUN ... LUN      LUN ... LUN   LUN ... LUN       LUN ... LUN

Only one CONT for the Liberator board, but stay consistent with other drivers.

*/

/* Macros */

#define	QS_CTLR(cont,targ)	(((cont) << 3) | (targ)) 
#define GETMST()	(OPENCHIP->op_scstat & OP_ST_MST_MASK)
#define GETPHASE()	(OPENCHIP->op_scstat & OP_ST_PHSMASK)
#define GETSTATE()	(OPENCHIP->op_scstat & OP_ST_STATEMASK)
#define NEWMST(x)	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_MST_MASK) | x)
#define NEWPHASE(x)	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_PHSMASK) | x)
#define NEWSTATE(x)	(OPENCHIP->op_scstat = (OPENCHIP->op_scstat & ~OP_ST_STATEMASK) | x)

/* QS addresses */

struct addrst {
	struct qsdevice *qsaddr;
	int vec;
};

struct addrst qsaddrlist[] = {
	(struct qsdevice *) IO_ADRS_SCSI, VEC_SCSI };
	
/* Global variables */

u_char		qsmsginbuf;
u_char		qsmsgout;
struct qsdevice *QSaddr[NQS];
struct scsi_cmnd qs_cmds[NQS];
SEMAPHORE 	qs_sems[NQS];
SEMAPHORE	qs_intr_sem;

#ifdef	DEBUG
int prevstat;
int prevopstat;
#endif	DEBUG

/* Forward Declarations */

int qsintr();


/*************************************************************************
*
* qsgetcmd - return ptr to cmd struct
*
*/

struct scsi_cmnd *qsgetcmd(cont)
{
	return(&qs_cmds[cont]);
}

/*************************************************************************
*
* qsprobe - Assume that controller is there, connect interrupt and call
*	init routine.
*
*/

qsprobe(cont)
{

	if(QSaddr[cont]) 
		return(OK);

	QSaddr[cont] = qsaddrlist[cont].qsaddr;
	intConnect(INUM_TO_IVEC(qsaddrlist[cont].vec),qsintr,cont);
	semInit(&qs_sems[cont]);
	semInit(&qs_intr_sem);
	semGive(&qs_sems[cont]);
	if(qsinit(qsaddrlist[cont].qsaddr) != OK)
		return(ERROR);
	printf("    QS%d  at address 0x%08x  vector %02d\n",
		cont,QSaddr[cont],qsaddrlist[cont].vec);
	return(OK);
}


/*************************************************************************
*
* qsinit - Intialize controller and data structures.
*
*/

qsinit(addr)
struct qsdevice *addr;
{

	register int i,target,lun;
	register short size;

	OPENCHIP->op_sccntrl = 0;
	while(OPENCHIP->op_sccntrl & OP_SC_DACT) 
		taskDelay(1);

	/* Initialize the WD33C93 chip */

	if((addr->wd_addr_stat &
		(WD_AUX_BSY | WD_AUX_CIP)) == (WD_AUX_BSY | WD_AUX_CIP)) {
		IO_ADRS_PORT1 = OP1_SET_RESSCSI;
		taskDelay(1);
		IO_ADRS_PORT1 = OP1_RES_RESSCSI;
		taskDelay(1);
	}
	do {
		OPENCHIP->op_scsintr = 0xff;
		taskDelay(1);
	} while (OPENCHIP->op_scsintr);

	/*
	 * Set the SCSI ID and issue a reset cmd. Then field the
	 * interrupt which results from the reset.
	 */

	addr->wd_addr_stat = WD_OWNID;
	addr->wd_data = QS_ID;
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_RESET;
	while (!(OPENCHIP->op_scsintr)) {
		taskDelay(1);
	}

	/* Set the source id register and the timeout */
	addr->wd_addr_stat = WD_SOURCE_ID;
	addr->wd_data = 0;
	addr->wd_addr_stat = WD_CONTROL;
	addr->wd_data = WCNT_DMA; 
	addr->wd_addr_stat = WD_TIMEOUT;
	addr->wd_data = QS_TIMEOUT;

	/*
	 * Reset the interrupt, get the OpenChip to start handling
	 * interrupts.
	 */

	OPENCHIP->op_scsintr = 0xff;
	OPENCHIP->op_sccntrl = OP_SC_INEA;	

	return(OK);
}

/*************************************************************************
*
* qscmdwait - Wait for command to finish.
*
*/
qscmdwait(cont,targ,cmd,timeout)
	unsigned char cont, targ;
	register struct scsi_cmnd *cmd;
{
	qsrun(cont,targ);
	semTake(&qs_intr_sem);
#ifdef	DOIT
	while(cmd->qs_finstat == SFN_NOT_FINISHED) {
		DELAY(400);
		if(timeout-- == 0)
			return(ERROR);
	}
#endif	DOIT
	/* Caller processes errors */
	return(OK);
}

DELAY(n)
{
	while(n--);
}

/*************************************************************************
*
* qsintr - Handle interrupts.
*
*/

qsintr(cont)
{
	unsigned char targ;
	unsigned char status,rstatus;
	unsigned char   c;
	struct scsi_cmnd *cmd;
	register struct qsdevice *wdptr;
	char i,str,tmpstr;
	char *cptr;
	
	wdptr = QSaddr[cont];
	status = OPENCHIP->op_scsistatus;
	wdptr->wd_addr_stat = WD_DEST_ID;
	targ = wdptr->wd_data; 
	cmd = &qs_cmds[cont];

	status = rstatus = OPENCHIP->op_scsistatus;

	/*
	 * There are several different versions of status which encode the
	 * new phase. In order to reduce the number of case's in a switch
	 * such multiple choices are reduced to one. 
	 */
	if (status & 0x08)
		status &= 0x0f;
	switch (GETSTATE()) {
	case SC_SELECTING:
		switch (status) {
		case WDST_SEL_CMPLT:	/* select completed successfully */
			if(OPENCHIP->op_sccntrl & OP_SC_SELATN) 
				NEWSTATE(SC_SELMSGOUT);
			else
				NEWSTATE(SC_CONNECTED);
			break;
		case WDST_TIMEOUT_CMD_TERM:	/* selection timed out */
#ifdef	DEBUG_PHASE
			logMsg("WDST_TIMEOUT_CMD_TERM\n");
#endif	DEBUG_PHASE
			qscmddone(cont,targ,cmd,SFN_SLCT_TIMEOUT);
			break;
		default:
			logMsg("SELECTING with status 0x%x\n", rstatus);
#ifdef	DEBUG
			logMsg("op_scsistatus=0x%x, op_scstat=0x%x, prevstat=0x%x, prevopstat=0x%x\n",
				OPENCHIP->op_scsistatus, OPENCHIP->op_scstat,
				prevstat,prevopstat);
#endif	DEBUG
			break;
		}
		break;

	case SC_SELMSGOUT:
		switch(status) {
			case WDST_REQ_SERV_MSG_OUT:
				qsmsgout = MSG_IDENTIFY | cmd->qs_lun;
				setupDMA(wdptr,&qsmsgout, 1, OP_ST_PHS_MSG_OUT);
				break;
			default:
				NEWSTATE(SC_CONNECTED);
				goto connected;
				
		}

	case SC_CONNECTED:
		/*
		 * If DMA was in use, it must be tidied up. 
		 */
connected:	
		if (OPENCHIP->op_sccntrl & OP_SC_DACT) {
			unsigned long   lngth;
			char           *cptr;

			OPENCHIP->op_sccntrl &= ~OP_SC_DACT;
			/*
			 * wait for DACT to read as reset. Perform a slight
			 * delay in the loop so as not to access the OpenChip
			 * too much. 
			 */
			do {
				DELAY(10);
			} while (OPENCHIP->op_sccntrl & OP_SC_DACT);

			/*
			 * get the remaining count out of the SCSI chip. 
			 */
			wdptr->wd_addr_stat = WD_TRANS_COUNT0;
			lngth = (unsigned char) wdptr->wd_data;
			lngth <<= 8;
			lngth |= (unsigned char) wdptr->wd_data;
			lngth <<= 8;
			lngth |= (unsigned char) wdptr->wd_data;
			/*
			 * calculate the corrected DMA pointer. 
			 */
			cptr = OPENCHIP->op_scdmaptr;
			cptr -= (unsigned char) ((unsigned char) cptr +
				(unsigned char) lngth) - OPENCHIP->op_scsum;

			switch (GETPHASE()) {
			case OP_ST_PHS_DATA_IN:
			case OP_ST_PHS_DATA_OUT:
				if (!(cmd->qs_flaqs & SCF_DATA_OVFL)) {
					cmd->qs_datlnk.sl_cnt = lngth;
					cmd->qs_datlnk.sl_ptr = cptr;
				}
				break;
			case OP_ST_PHS_STATUS:
				if (!(cmd->qs_flaqs & SCF_STATUS_OVFL)) {
					cmd->qs_nstat = lngth;
					cmd->qs_pstat = cptr;
				}
				break;
			case OP_ST_PHS_COMMAND:
				if (!(cmd->qs_flaqs & SCF_COMM_OVFL)) {
					cmd->qs_ncmnd = lngth;
					cmd->qs_pcmnd = cptr;
				}
				break;
			}
		}
		switch (status) {
			/*
			 * statuses 0x1? indicate previous transfer completed
			 * normally. 0x4? indicates transfer completed with
			 * count != 0. 0x8? indicates first transfer after
			 * connection. 
			 */
		case 0x08 | PHS_MSG_IN:
#ifdef	DEBUG_PHASE
			logMsg("PHS_MSG_IN\n");
#endif	DEBUG_PHASE
			setupDMA(wdptr,&qsmsginbuf, 1, OP_ST_PHS_MSG_IN);
			break;
		case 0x08 | PHS_MSG_OUT:
#ifdef	DEBUG_PHASE
			logMsg("PHS_MSG_OUT\n");
#endif	DEBUG_PHASE
			qsmsgout = MSG_IDENTIFY | cmd->qs_lun;
			setupDMA(wdptr,&qsmsgout, 1, OP_ST_PHS_MSG_OUT);
			break;
		case 0x08 | PHS_STATUS:
#ifdef	DEBUG_PHASE
			logMsg("PHS_STATUS\n");
#endif	DEBUG_PHASE
			if (cmd->qs_nstat) {
			    setupDMA(wdptr,cmd->qs_pstat, cmd->qs_nstat, 
				OP_ST_PHS_STATUS);
			} else {
			    cmd->qs_flaqs |= SCF_STATUS_OVFL;
			    setmanual(wdptr, OP_ST_PHS_STATUS);
			}
			break;
		case 0x08 | PHS_COMMAND:
#ifdef	DEBUG_PHASE
			logMsg("PHS_COMMAND\n");
#endif	DEBUG_PHASE
			if (cmd->qs_ncmnd) {

#ifdef	DEBUG
			    for(i=0,cptr=cmd->qs_pcmnd;i<cmd->qs_ncmnd;i++) {
			    	logMsg("PHS_COMMAND cdb[%d]=0x%x\n",i,*cptr++);
			    }
#endif	DEBUG
			    setupDMA(wdptr,cmd->qs_pcmnd, cmd->qs_ncmnd, 
				OP_ST_PHS_COMMAND);
			} else {
			    setmanual(wdptr, OP_ST_PHS_COMMAND);
			}
			break;
		case 0x08 | PHS_DATA_IN:
		case 0x08 | PHS_DATA_OUT:
#ifdef	DEBUG_PHASE
			logMsg("PHS_DATA_IN/PHS_DATA_OUT\n");
#endif	DEBUG_PHASE
			if (cmd->qs_datlnk.sl_cnt == 0) {
			    cmd->qs_flaqs |= SCF_DATA_OVFL;
			    if ((status & PHS_MASK) == PHS_DATA_IN) 
				setmanual(wdptr, OP_ST_PHS_DATA_IN);
			    else
				setmanual(wdptr, OP_ST_PHS_DATA_OUT);
			} else {
			    if ((status & PHS_MASK) == PHS_DATA_IN) {
				setupDMA(wdptr,cmd->qs_datlnk.sl_ptr, 
				    cmd->qs_datlnk.sl_cnt, OP_ST_PHS_DATA_IN);
			    } else {
				setupDMA(wdptr,cmd->qs_datlnk.sl_ptr, 
				    cmd->qs_datlnk.sl_cnt, OP_ST_PHS_DATA_OUT);
			    }
			}
			break;
		case WDST_DISCONN_SERV:	/* normal disconnect */
		case WDST_DISCONN_CMD_TERM:	/* unexpected disconnect */
			/*
			 * Notice that l_state is guaranteed to be L_ST_NULL
			 * since it gets set so when a reselection takes
			 * place. 
			 */
			switch (GETMST()) {
			case MST_NULL:
			case MST_DISC:
#ifdef	DEBUG
				logMsg("UNEXPECTED DISCONNECT\n");
#endif	DEBUG
				qscmddone(cont,targ,cmd,SFN_NO_CMND_CMPLT);
				break;
			case MST_CMPLT:
			case MST_CMPLT2:
				qscmddone(cont,targ,cmd,SFN_NORMAL_END);
				break;
			}
			/*
			 * The code may now go ahead and try to issue another
			 * command. 
			 */
			break;
		case WDST_MSG_IN_PAUSE:	/* message in has paused */
			switch (cmd->qs_status[0]) {
			case MSG_CMD_COMPLETE:
				NEWMST(MST_CMPLT);
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_EXTENDED:
#ifdef	DEBUG_PHASE
				logMsg("WDST_MSG_IN_PAUSE: MSG_EXTENDED\n");
#endif	DEBUG_PHASE
				setmanual(wdptr, OP_ST_PHS_MSG_IN);
			case MSG_SAVE_DPTR:
			case MSG_RESTORE_PTRS:
			default:
#ifdef	DEBUG_PHASE
				logMsg("WDST_MSG_IN_PAUSE: MSG_SAVE_DPTR/MSG_RESTORE_PTRS\n");
#endif	DEBUG_PHASE
				cmd->qs_status[0] = MSG_MSG_REJECT;
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_DISCONNECT:
#ifdef	DEBUG_PHASE
				logMsg("WDST_MSG_IN_PAUSE: MSG_DISCONNECT\n");
#endif	DEBUG_PHASE
				NEWMST(MST_DISC);
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			case MSG_MSG_REJECT:
#ifdef	DEBUG_PHASE
				logMsg("WDST_MSG_IN_PAUSE: MSG_REJECT\n");
#endif	DEBUG_PHASE
				/*
				 * It is hard to see what to do if this is
				 * received but the specification says that a
				 * message reject message may not be rejected
				 * itself. 
				 */
				wdptr->wd_addr_stat = WD_COMMAND;
				wdptr->wd_data = WD_CMD_NEG_ACK;
				break;
			}
			break;
		default:
			logMsg("CONNECTED with status 0x%x\n", rstatus); 
#ifdef	DEBUG 
				logMsg("op_scsistatus=0x%x, op_scstat=0x%x, prevstat=0x%x, prevopstat=0x%x\n",
				OPENCHIP->op_scsistatus, OPENCHIP->op_scstat,
				prevstat,prevopstat);
#endif	DEBUG
			break;
		}
		break;
	}
#ifdef	DEBUG
	prevstat = OPENCHIP->op_scsistatus;
	prevopstat = OPENCHIP->op_scstat;
#endif	DEBUG
	OPENCHIP->op_scsintr = 0xff;
}

/*************************************************************************
*
* qscmddone - wrap up command processing (this routine may do more
		when interrupts are working)
*
*/

qscmddone(cont,targ,cmd,finstat)
	unsigned char cont, targ;
	struct scsi_cmnd *cmd;
{
	register unsigned int ctlr;
	NEWSTATE(SC_DISCONNECT); 
	cmd->qs_finstat = finstat;
	semGive(&qs_intr_sem);
	return(OK);
}

/*************************************************************************
*
* qsrun - start a command going
*
*/

qsrun(cont,targ)
	unsigned char cont,targ;
{
	register struct qsdevice *addr;

	addr = QSaddr[cont];

	addr->wd_addr_stat = WD_DEST_ID;	
	addr->wd_data = targ;

	addr->wd_addr_stat = WD_SYNCHRNS;
	addr->wd_data = 0;
	addr->wd_addr_stat = WD_CONTROL;
	addr->wd_data = WCNT_DMA;
	OPENCHIP->op_sccntrl |= OP_SC_SELATN;
	NEWSTATE(SC_SELECTING);
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_SEL_ATN;
}

/*************************************************************************
*
* setupDMA - setup an OpenChip DMA transfer
*
*/

setupDMA(addr,ptr,cnt,phase)
	struct qsdevice *addr;
	char *ptr;
	char phase;
{
	short i;

	addr->wd_addr_stat = WD_TRANS_COUNT0;
	addr->wd_data = cnt >> 16;
	addr->wd_data = cnt >> 8;
	addr->wd_data = cnt;
	NEWPHASE(phase);
	OPENCHIP->op_scdmaptr = ptr;
	OPENCHIP->op_scsum = (char) ptr + (char) cnt;
	addr->wd_addr_stat = WD_COMMAND;
	addr->wd_data = WD_CMD_TRANS_INFO;
	while(addr->wd_addr_stat & WD_AUX_CIP)
		for(i=0;i<8;i++) ;
	OPENCHIP->op_sccntrl |= OP_SC_DACT;
}


/*************************************************************************
*
* setmanual - finish command protocol
*
*/

setmanual(wdptr,phase)
	struct qsdevice *wdptr;
	char            phase;
{
	static char     non_val;

	wdptr->wd_addr_stat = WD_TRANS_COUNT0;
	wdptr->wd_data = 0xff;
	/*
	 * It isn't really worth writing a value to any but the most
	 * significant byte of the transfer counter. 
	 */
	wdptr->wd_addr_stat = WD_COMMAND;
	wdptr->wd_data = WD_CMD_TRANS_PAD;
	NEWPHASE(phase);
	if (!(phase & OP_ST_PHS_IO)) {
		OPENCHIP->op_scdmaptr = &non_val;
		OPENCHIP->op_sccntrl |= OP_SC_DACT;
	}
}

op_stat()
{
	register struct qsdevice *addr;

	addr = qsaddrlist[0].qsaddr;

	logMsg("OpenChip Status\n\n");
	logMsg("WDaux_stat=0x%x, WDscsi_stat=0x%x\n",addr->wd_addr_stat,	
		OPENCHIP->op_scsistatus);
	addr->wd_addr_stat = WD_CONTROL;
	logMsg("WDcntrl=0x%x\n",addr->wd_data);
	logMsg("op_scratch=0x%x, op_lunbase=0x%x\n",
		OPENCHIP->op_scratch,OPENCHIP->op_lunbase);
	logMsg("scstat=0x%x  scsintr=0x%x  sccntrl=0x%x, scdmaptr=0x%x, scsum=%d\n",
		OPENCHIP->op_scstat, OPENCHIP->op_scsintr, 
		OPENCHIP->op_sccntrl, OPENCHIP->op_scdmaptr, OPENCHIP->op_scsum);
	logMsg("\n");
}

printcmd(cmd)
	register struct scsi_cmnd *cmd;
{
	register unsigned char i;
	register unsigned char *pcmd;

	printf("\n printcmd, cmd=0x%x, dbgmsg=%s\n",cmd,cmd->dbgmsg);
	printf("datlnk.sl_ptr=0x%x, datlnk.sl_cnt=%d, datlnk.sl_lnptr=0x%x\n",
		cmd->qs_datlnk.sl_ptr,cmd->qs_datlnk.sl_cnt,cmd->qs_datlnk.sl_lnptr);
	printf("pcmnd=0x%x, ncmnd=%d, pstat=0x%x, nstat=%d, finstat=0x%x\n",
		cmd->qs_pcmnd,cmd->qs_ncmnd,cmd->qs_pstat,cmd->qs_nstat,cmd->qs_finstat);
	printf("cdb=");
	for(i=0, pcmd=cmd->qs_pcmnd; i<cmd->qs_ncmnd; i++, pcmd++)
		printf(" %x",*pcmd);
	printf("\n\n");
}

#ifdef	DEBUG
printcdb(cdb,n)
{
	register unsigned char i;
	register unsigned char *pcmd;

	printf("cdb=");
	for(i=0, pcmd=cdb; i<n; i++, pcmd++)
		printf(" %x",*pcmd);
	printf("\n\n");
}
#endif	DEBUG
