/* sysLib.c - ISI Liberator system dependent library */

/*
DESCRIPTION
This library contains a set of routines to manipulate the primary functions
of the CPU board.  The goal is to provide a board-independant interface on
which UniWorks and application code can be built in a system-independant way.
Not every feature of every board is supported by this library; a particular
board may have various extensions to the capabilities described here.
Also not every board will support all the functions provided by this library.
And some boards provide some of the functions of this library with hardware
switches, jumpers, or PALs, instead of software controllable registers.

The funtions addressed here include:

    initialization functions:
	- initialize hardware to known state
	- identify the system

    memory/address space functions:
	- get on-board memory limit
	- map from local to bus and bus to local address spaces
	- enable/disable cache memory
	- set/get non-volatile RAM

    bus interrupt functions:
	- enable/disable bus interrupt levels
	- generate bus interrupts

    serial channel functions (see tyCoDrv):
	- enable/disable serial channel interrupts
	- set serial channel baud rates
	- get/put bytes from a serial channel

    clock/timer functions:
       - enable/disable timer interrupts
       - set timer periodic rate

*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "types.h"
#include "vme.h"
#include "memLib.h"
#include "sysLib.h"
#include "config.h"
#include "scn68562.h"
#include "openchip.h"
#include "liberator.h"

IMPORT VOID logMsg ();
IMPORT char end;	/* end of system, inserted automatically by ld */
IMPORT char vme_window_std[];

/* Forward Declarations */

sysAClkDummy();

/* Globals */

int   sysBus      = BUS;		/* system bus type (VME_BUS, etc) */
int   sysCpu      = CPU;		/* system cpu type (MC680x0) */
int   sysCommType = USE_POLL;		/* system inter-processor comm. type */
char *sysBootLine = BOOT_LINE_ADRS;	/* address of boot line */
char *sysExcMsg   = EXC_MSG_ADRS;       /* catastrophic message area */
int   sysProcNum;			/* processor number of this cpu */
int   sysFlags;				/* boot flags */
char  sysBootHost[BOOT_FIELD_LEN];	/* name of host from which we booted */
char  sysBootFile[BOOT_FIELD_LEN];	/* name of file from which we booted */

/*
 * Since the OPENCHIP scratch area must start on a 256 byte boundary 
 * extra room must be allocated.
 */
char opcbbuf[255 + sizeof(struct oc_scratch)];
struct oc_scratch *openscratch;

/* locals */

LOCAL 	char *memTop = NULL;

#define	SYS_CLK_RATE	60

LOCAL 	int 	sysClkFreq;			/* ticks per second */
LOCAL 	int 	sysSClkFreq;			/* sys clock frequency */
LOCAL	int	sysSClkTick = 0;		/* sys clock counter */
LOCAL	int	sysSClkFire = 0;		/* counter for firing tick */
LOCAL	FUNCPTR sysAClkRoutine = sysAClkDummy;	/* aux clock intr routine ptr */
LOCAL	int	sysAClkFreq = 0;		/* aux clock frequency */
LOCAL	int	sysAClkParam;			/* aux clock intr rtn param */
LOCAL	int	sysAClkTick = 0;		/* aux clock counter */
LOCAL	int	sysAClkFire = 0;		/* counter for firing tick */
LOCAL	int	sysAClkRunning = 0;		/* aux clock running */

/************************************************************************
*
* sysModel - return model name of the system CPU
*
* Use this routine to find the model name of the system CPU.
*
* RETURNS: pointer to string "ISI Liberator".
*/

char *sysModel ()

    {
    return ("ISI Liberator");
    }
/**********************************************************************
*
* sysHwInit - initialize hardware
*
* This routine initializes various features of the board.
* It is normally called from usrInit (2) in usrConfig (1).
*
*/

VOID sysHwInit ()

    {
	OPENCHIP->op_reset = 0;
	OPENCHIP->op_genvec = VEC_GENBASE;
	openscratch = (struct oc_scratch *)(((long)opcbbuf + 255) & ~0xff);
	OPENCHIP->op_scratch = openscratch;
	OPENCHIP->op_bustimeout = DEFBUSTIMEOUT;
    }
/**********************************************************************
*
* sysMemTop - get top of memory address
*
* This routine returns the address of the first missing byte of memory.
* It starts probing at 0, then tries to read a byte at every 4K boundary
* until it finds one that can't be read.
*
* This routine must be called in system mode, the first time only.  After
* that, it can be called in user mode as well.
*/

char *sysMemTop()
    {

    char orig;

    if (memTop == NULL)
	{
	/* Look for the top of memory.  Starting at the first even page
	   boundary after _end, probe each page, then write it and
	   read it back, just to be sure. */

	memTop = (char *) ((((int) &end) + (VME_PAGE_SIZE - 1)) &
			    (~ (VME_PAGE_SIZE - 1)));

	FOREVER
	    {
	    if (sysMemProbe (memTop) != OK)
		break;

	    orig = *memTop;

	    *memTop = 0x55;
	    if (*memTop != 0x55)
		break;

	    *memTop = orig;

	    memTop += VME_PAGE_SIZE;
	    }
	/* memTop is now pointing to the first non-existent address */
	}

    return (memTop);
    }
/*******************************************************************************
*
* sysToMonitor - transfer to rom monitor
*
* This routine transfers control to the rom monitor.  It is usually called
* only by the routine reboot, which services control-x, and bus errors at
* interrupt level.  In special circumstances, however, the user may wish
* to introduce a new startType such that a special bootrom facility would be
* enabled.
*
* RETURNS: OK (if we ever continue from the rom monitor).
*/

STATUS sysToMonitor (startType)
    int startType;	/* parameter is passed to ROM to tell it how to boot *
			   The possible type are defined in h/sysLib.h */
    {
	static FUNCPTR	*foo;

	foo = (FUNCPTR)(ROM_BASE_ADRS + 4);

	OPENCHIP->op_reset = 0;
	(*foo)(startType);	/**/

    	return (OK);	/* in case we ever continue from rom monitor */
    }

/**********************************************************************
*
* sysClkInit - initialize the clock
*
* This routine sets the rate of the clock interrupts. Two clocks are
* derived from this clock rate, the system clock and the aux clock.
* It is normally called by usrRoot (2) in usrConfig (1).
*
* NOTE: 
*
*	MIN/MAX values:
*
*	If ticks per second is less than 31, the value computed by the 
* 	equation below exceeds 16 bits (the size of the preset register). 
*
*	According to the manual, the smallest value that can be contained 
*	in the preset register is 32. In the equation below, 60,000 gives 
*	you 32, but I found the max reasonable frequency to be
*	~6000 (60,000 causes the CPU to be clobbered with interrupts).
*
* SEE ALSO:
*	sysClkSetRate
*	sysAuxClkSetRate
*
*/

VOID sysClkInit (ticksPerSecond)
	int ticksPerSecond;	    /* number of clock interrupts per second */
{
	if((ticksPerSecond < 31) || (ticksPerSecond > 6000)) {
		printf("ticksPerSecond must be greater than 31 and less than 6000\n");
		return(ERROR);
	}
    	sysClkFreq = ticksPerSecond;
	if(ticksPerSecond >= SYS_CLK_RATE) {
		sysSClkFreq = SYS_CLK_RATE;
		sysSClkFire = sysClkFreq / SYS_CLK_RATE;
	} else {
		sysSClkFreq = ticksPerSecond;
		sysSClkFire = sysClkFreq / ticksPerSecond;
	}
	sysSClkTick = sysSClkFire;
	OPENCHIP->op_clkpreset =
	    ((32 * OPENCHIP->op_sysclk)/(ticksPerSecond * OPENCHIP->op_sysperiod));
	OPENCHIP->op_clkstat = OP_CLK_INEA;

}

/************************************************************************
*
* sysTick - clock tick received
*
* The system clock is set to run at 1200. This routine performs the 
* system clock functions at 60 HZ. If there is an aux clock
* running, the aux clock routine is called at the rate of the aux clock. 
*
*/

sysTick()
{
	/* Check system clock */
	if((--sysSClkTick) == 0) {
		wdTick ();		/* check watchdog timers */
		tickAnnounce ();	/* announce system tick to VRTX */
		sysSClkTick = sysSClkFire;
	}

	/* Check aux clock */
	if(sysAClkRunning) {
		if((--sysAClkTick) == 0) {
			sysAClkTick = sysAClkFire;
			(*sysAClkRoutine) (sysAClkParam);
		}
	}
}

/**********************************************************************
*
* sysClkConnect - connect routine to system clock interrupt
*
* This routine connects the given function to the system clock interrupt.
* It is normally called from usrRoot (2) in usrConfig (1) to connect
* usrClock (2) to the system clock interrupt.
*
* SEE ALSO: intConnect (2), usrClock (2)
*/

VOID sysClkConnect (routine, arg)
    FUNCPTR routine;	/* routine to be called at each system clock
			 * interrupt */
    int arg;		/* argument with which to call routine */

    {
    intConnect (INUM_TO_IVEC (VEC_CLOCK), routine, arg);
    }
/**********************************************************************
*
* sysClkSetRate - set rate of system clock
*
*
* SEE ALSO: sysClkGetRate (2)
*/

VOID sysClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second */
    
    {
	if(ticksPerSecond > sysClkFreq) {
		printf("Cannot set the system clock to be faster than %d\n",
			sysClkFreq);
		return(ERROR);
	}
	sysSClkFreq = ticksPerSecond;
	sysSClkFire = sysClkFreq / sysSClkFreq;
	sysSClkTick = sysSClkFire;
    }
/**********************************************************************
*
* sysClkEnable - enable system clock
*
*
* NOTE: this routine is only here for compatibility
*/

VOID sysClkEnable ()
    {
    }
/**********************************************************************
*
* sysClkGetRate - get rate of system clock
*
* This routine is used to find out the system clock speed.
*
* RETURNS: The number of ticks of the system clock per second.
*
* SEE ALSO: sysClkSetRate (2)
*/

int sysClkGetRate ()

    {
    return (sysSClkFreq);
    }
/**************************************************************************
*
* sysAuxClkSetRate - set the auxiliary timer frequency
*
* This routine sets the clock rate of the auxiliary clock.
*
* SEE ALSO: sysAuxClkGetRate (2)
*/

VOID sysAuxClkSetRate (ticksPerSecond)
    int ticksPerSecond;	    /* number of clock interrupts per second;
			     * 0 stops the clock */

    {
	if(ticksPerSecond > sysClkFreq) {
		printf("Cannot set aux clock frequency higher than %d\n",
			sysClkFreq);
		return(ERROR);
	}
	sysAClkFreq = ticksPerSecond;
	sysAClkFire = sysClkFreq / ticksPerSecond;
	sysAClkTick = sysAClkFire;
	sysAClkRunning = TRUE;
	return(OK);
    }
/**************************************************************************
*
* sysAuxClkGetRate - get the auxiliary timer frequency
*
* This routine finds out the auxiliary clock speed.
*
* RETURNS: The number of ticks of the auxiliary clock per second.
*
* SEE ALSO: sysAuxClkSetRate (2)
*/

int sysAuxClkGetRate ()

    {
    return (sysAClkFreq);
    }
/**************************************************************************
*
* sysAuxClkConnect - connect a routine to the auxiliary clock interrupt
*
* This routine connects a user routine to the auxiliary clock interrupt,
* and enables the auxiliary clock interrupt.
*
* SEE ALSO: intConnect (2), sysAuxClkDisconnect (2)
*/

STATUS sysAuxClkConnect (routine, parm)
    FUNCPTR routine;	/* routine to be called at each auxiliary clock
			 * interrupt */
    int parm;		/* argument with which to call routine */

    {

    sysAClkRoutine = routine;
    sysAClkParam = parm;

    return(OK);
    }
/**********************************************************************
*
* sysAuxClkEnable - enable aux clock
*
*
* NOTE: this routine is only here for compatibility
*/

VOID sysAuxClkEnable ()
    {
    }
/**************************************************************************
*
* sysAuxClkDisconnect - clear the auxiliary clock routine.
* 
* This routine disables the auxiliary clock interrupt, stops the timer,
* and disconnects the routine currently connected to the auxiliary clock
* interrupt.
*
* Liberator: no aux clock yet.
*
* SEE ALSO: sysAuxClkConnect (2)
*/

VOID sysAuxClkDisconnect ()

    {
    sysAClkRoutine = sysAClkDummy;
    sysAClkRunning = FALSE;
    sysAClkParam = 0;
    }
/**************************************************************************
*
* sysAClkDummy - a do nothing routine.
*
*/
VOID sysAClkDummy()
{
}
/************************************************************************
*
* sysLocalToBusAdrs - convert local address to bus address
*
* Given a local memory address, this routine returns the VME address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unable to get to that local address from the bus.
*
* SEE ALSO: sysBusToLocalAdrs (2)
*
* ARGSUSED
*/

STATUS sysLocalToBusAdrs (adrsSpace, localAdrs, pBusAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *localAdrs;	/* local address to convert */
    char **pBusAdrs;	/* where to return bus address */

    {
    pBusAdrs = localAdrs - RAM_LOCAL_ADRS;
    return (ERROR);
    }
/*************************************************************************
*
* sysBusToLocalAdrs - convert bus address to local address
*
* Given a VME memory address, this routine returns the local address
* that would have to be accessed to get to that byte.
*
* RETURNS: OK, or ERROR if unknown address space.
*
* SEE ALSO: sysLocalToBusAdrs (2)
*/

STATUS sysBusToLocalAdrs (adrsSpace, busAdrs, pLocalAdrs)
    int adrsSpace;	/* bus address space in which busAdrs resides;
			 * use address modifier codes as defined in vme.h,
			 * such as VME_AM_STD_SUP_DATA */
    char *busAdrs;	/* bus address to convert */
    char **pLocalAdrs;	/* where to return local address */

    {
    switch (adrsSpace)
	{
	case VME_AM_SUP_SHORT_IO:
	case VME_AM_USR_SHORT_IO:
	    *pLocalAdrs = (char *) (VME_SHORTIO | (int) busAdrs);
	    return (OK);

	case VME_AM_STD_SUP_ASCENDING:
	case VME_AM_STD_SUP_PGM:
	case VME_AM_STD_SUP_DATA:
	case VME_AM_STD_USR_ASCENDING:
	case VME_AM_STD_USR_PGM:
	case VME_AM_STD_USR_DATA:
	    *pLocalAdrs = (char *) (VME_STANDARD | (int) busAdrs);
	    return (OK);

	case VME_AM_EXT_SUP_ASCENDING:
	case VME_AM_EXT_SUP_PGM:
	case VME_AM_EXT_SUP_DATA:
	case VME_AM_EXT_USR_ASCENDING:
	case VME_AM_EXT_USR_PGM:
	case VME_AM_EXT_USR_DATA:
	    if (busAdrs < (char *) 0x03000000)
		return (ERROR);

	    *pLocalAdrs = (char *) (VME_EXTENDED | (int) busAdrs);
	    return (OK);

	default:
	    return (ERROR);
	}
    }
/*************************************************************************
*
* sysIntEnable - enable interrupt level
*
* This routine enables the specified VME interrupt level.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntEnable (intLevel)
    int intLevel;	/* interrupt level to enable */

    {
    return (OK);
    }
/*************************************************************************
*
* sysIntDisable - disable interrupt level
*
* This routine disables the specified interrupt level.
*
* RETURNS: OK
*
* ARGSUSED
*/

STATUS sysIntDisable (intLevel)
    int intLevel;	/* interrupt level to disable */

    {
    return (OK);
    }
/*************************************************************************
*
* sysIntAck - acknowledge interrupt
*
* Acknowledges the specified interrupt.
*
* RETURNS: NULL.
*
* ARGSUSED
*/

int sysIntAck (intLevel)
    int intLevel;		/* interrupt level to acknowledge */

    {
    return (NULL);
    }
/*************************************************************************
*
* sysIntGen - generate interrupt
*
* This routine generates a VME bus interrupt.
*
* ARGSUSED
*/

STATUS sysIntGen (level, vector)
    int level;		/* VME bus interrupt level to generate (1-7) */
    int vector;		/* interrupt vector to generate (0-255) */

    {
    return (ERROR);
    }

/************************************************************************
*
* sysSetProcNum - set the processor number
*
*/

VOID sysSetProcNum (procNum)
    int procNum;                /* processor number */
    {
    struct nv_ram nv_ram;

    /* for the sake of the hm driver, this ignores the procNum
     * param and sets sysProcNum to the procnum stored in nv_ram
    sysProcNum = procNum;
     */
    if (sysGetNvRam(&nv_ram))
	    sysProcNum = 0xff;
    else
	    sysProcNum = nv_ram.cpu_num;
    }

/************************************************************************
*
* sysGetProcNum - get the processor number from nv-ram
*
*/

int sysGetProcNum()
{
	struct nv_ram nv_ram;

	if (sysGetNvRam(&nv_ram))
		return(ERROR);
	else
		return(nv_ram.cpu_num);
}

/************************************************************************
*
* sysBusTas - test and set across VME bus
*
* This routine does a 680x0 test-and-set instruction across the backplane.
*
* Some CPUs have problems doing TAS instructions on the bus -
* ISI doesn't, so this routine simply calls vxTas.
*
* RETURNS: TRUE (successful set) or FALSE (failure).
*/

#if 0
BOOL sysBusTas (addr)
    char *addr;         /* address to be tested and set */
    {
    /*
    return (vxTas (addr));
    */
    /* this will eventually kill us */
    if (*addr)
	return 0;
    return (*addr = 0x1);
    }
#endif 0

/************************************************************************
*
* Non-volatile RAM functions:
*
* sysGetNvRam(p) copies entire contents of nv ram to memory at p.  Returns
* ERROR if checksum error, else returns OK.
*
* sysSetNvRam(p) copies 16 shorts at p into nvram.
*/


#define	RAM_SEL_DELAY	10
#define RAM_LOW_DELAY	10
#define RAM_HIGH_DELAY	10

#define	SETSTROBE	(Q_OP_PORT2 = OP2_SET_STROBE)
#define	RESETSTROBE	(Q_OP_PORT2 = OP2_RES_STROBE)
#define	SETDATAOUT	(Q_OP_PORT2 = OP2_SET_DATOUT)
#define	RESETDATAOUT	(Q_OP_PORT2 = OP2_RES_DATOUT)

STATUS sysGetNvRam(pnvram)
	unsigned char *pnvram;
{
	int	i;
	unsigned short *q;

	/*
	 * send an RCL instruction to the non-volatile RAM chip.
	 * The chip may not then be accessed again for 2.5 microseconds. 
	 */
	ramselect();
	shiftout(8, 0x85, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
	delaycycles(100);
	q = (short *)pnvram;
	for (i=0;i<16;++i)
		*q++ = nv_readw(i);
	return(chkram(pnvram));
}

/************************************************************************
*
* sysSetNvRam
*
* Write all 16x16 bits of memory at p into the non-volatile ram.
*/

sysSetNvRam(pnvram)
unsigned char *pnvram;
{
	int	i;
	unsigned short *q;

	q = (short *)pnvram;
	for (i=0;i<16;++i)
		nv_writew(i, *q++);
	/*
	 * send an STO instruction to the ram.
	 */
	ramselect();
	shiftout(8, 0x81, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
}

/************************************************************************
*
* chkram - check nv-ram
*
*/

chkram(p)
unsigned char	*p;
{
	int             i;
	unsigned char	cksum;

	cksum = 0;
	for (i = 0; i < 31; ++i)
		cksum += *p++;
	if (*p == cksum)
		return (OK);
	else
		return (ERROR);
}

/************************************************************************
*
* nv_writew - set the write enable latch so that writes are enabled. 
*
*/

nv_writew(ind, data)
{
	ramselect();
	shiftout(8, 0x84, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
	/*
	 * Output the byte 1AAAA011 where the A's give the address. Then
	 * write the 16 bits of data. 
	 */
	ramselect();
	shiftout(8, 0x83 | (ind << 3), RAM_LOW_DELAY, RAM_HIGH_DELAY);
	shiftout(16, data, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();

	return (0);
}

/************************************************************************
*
* nv_readw - read nv_ram
*
* Output the byte 1AAAA11X where the A's give the address. The X value
* should be a 1 for the QX. Then read 16 bits of data. 
*
*/

nv_readw(ind)
{
	unsigned short  res;

	ramselect();
	shiftout(8, 0x87 | (ind << 3), RAM_LOW_DELAY, RAM_HIGH_DELAY);
	res = shiftin(16, RAM_LOW_DELAY, RAM_HIGH_DELAY);
	ramdeselect();
	return res;
}

/************************************************************************
*
* ramselect
*
*/

ramselect()
{
	SETDATAOUT;		/* Always have DATAOUT at logic 1, except for
				 * reading */
	Q_OP_PORT2 = OP2_SET_RAMSEL;	/* enable RAM */
	delaycycles(RAM_SEL_DELAY);
}

/************************************************************************
*
* ramdeselect
*
*/

ramdeselect()
{
	Q_OP_PORT2 = OP2_RES_RAMSEL;	/* disable RAM */
}

/************************************************************************
*
* shiftout - write bits to serial RAM
*
* write a number of bits to the serial RAM or clock chip. Data is written
* most signiicant bit first. Data is clocked on the positive going edge. 
*
*/

shiftout(nbits, shftreg, clklowdelay, clkhighdelay)
	register        nbits;
	register unsigned shftreg;
{
	register unsigned mask;

	for (mask = 1 << (nbits - 1); nbits-- > 0; mask >>= 1) {
		if (shftreg & mask) {
			SETDATAOUT;
		} else {
			RESETDATAOUT;
		}
		delaycycles(clklowdelay);
		SETSTROBE;
		delaycycles(clkhighdelay);
		RESETSTROBE;
	}
	SETDATAOUT;		/* Always leave DATAOUT set */
}


/************************************************************************
*
* shiftin - read bits from serial RAM
*
* read a number of bits from the serial RAM or clock chip. Data is read most
* significant bit first. Data is read on the negative going edge. 
*
*/

shiftin(nbits, clklowdel, clkhighdel)
{
	register int    shftreg;

	for (shftreg = 0; nbits-- > 0;) {
		shftreg <<= 1;
		delaycycles(clklowdel);
		if (DATIN)
			shftreg++;
		SETSTROBE;
		delaycycles(clkhighdel);
		RESETSTROBE;
	}
	return (shftreg);
}

/************************************************************************
*
* lshiftin - read bits from serial RAM
*
* read a number of bits from the serial RAM or clock chip. Data is read on
* the negative going edge. This routine differs from that above in that data
* is read least significant bit first. 
*
*/

lshiftin(nbits, clklowdel, clkhighdel)
{
	register        shftreg;
	register        mask;

	shftreg = 0;
	for (mask = 1; nbits-- > 0; mask <<= 1) {
		delaycycles(clklowdel);
		if (DATIN)
			shftreg |= mask;
		SETSTROBE;
		delaycycles(clkhighdel);
		RESETSTROBE;
	}
	return shftreg;
}

/************************************************************************
*
* delaycycles
*
*/

delaycycles(del)
	register short  del;
{
	del /= 3;		/* Assuming best case, cached DBRA in loop */
	if (del > 1) {
		del--;
		do {
		} while (--del != -1);
	}
}

/*
 * vdma resource allocator for the 68020 board 
 * under uniworks
 */

/* the liberator doesn't have vdma capabilities, so
 * these routines are really no-ops
 */

/************************************************************************/
/* 
 * sysInitVdma
 */
sysInitVdma ()
{
    return (0);
}

/************************************************************************/
/* 
 * sysSetVdma
 *
 * set the vdma registers for a buffer at
 * location buff of size nbytes
 *
 * returns address of window requested which is always
 * the same as the address requested
 */
char *sysSetVdma (buff, nbytes)
char *buff;
int nbytes;
{
    return ((char *)((int)buff - (int)vme_window_std));
}

/************************************************************************/
/* 
 * sysGiveVdma
 *
 */
sysGiveVdma (vmeoff)
int vmeoff;
{
    return (0);
}

