/* usrLib.c - some useful user interface subroutines */

static char *copyright = "Copyright 1984-1988, Wind River Systems, Inc.";

/*
modification history
--------------------
*/

/*
This library consists of routines meant to be executed from the UniWorks
shell.  It provides useful utilities for task monitoring and execution,
system information, symbol table management, etc.  

Many routines here are little more than human interfaces to more general
routines contained elsewhere in UniWorks.  Users should feel free to
modify or extend this library.  It might be preferable, however, to
create a private library, using this one as a model, and link that into
a system, to customize capabilities.

Many routines here have optional parameters.  If those parameters are 0,
which is what the shell provides if no argument is typed, default
actions occur.  

SEE ALSO: "Shell", "Debugging"
*/

/* LINTLIBRARY */

#include "UniWorks.h"
#include "a_out.h"
#include "ctype.h"
#include "usrLib.h"
#include "fioLib.h"
#include "ioLib.h"
#include "ldLib.h"
#include "memLib.h"
#include "remLib.h"
#include "rtLib.h"
#include "strLib.h"
#include "symLib.h"
#include "sysLib.h"
#include "taskLib.h"
#include "sysSymTbl.h"
#include "version.h"

IMPORT SYMTAB_ID statSymTbl;	/* table of status code names and values */

#define MAXLINE         80	/* max line length for input to 'm' routine */
#define MAX_DSP_TASKS	500	/* max tasks that can be displayed */

/* sp parameters: change the help message, if these change! */

int spTaskPriority	= 100;
int spTaskOptions	= VX_SUPERVISOR_MODE | VX_FP_TASK | VX_STDIO;
int spTaskStackSize	= 7000;
int defaultStackSize	= 3000;

/* symbol types in system symbol table */

LOCAL char *typeName [] =
    {
    "UNDF",
    "ABS",
    "text",
    "data",
    "bss",
    "COMM",
    "FILE",
    "????",
    };

LOCAL char infoHdr [] = "\n\
    NAME        ENTRY       TID    PRI  STATUS     PC       SP     ERRNO  DELAY\n\
------------ ------------ -------- --- -------- -------- -------- ------- -----\n";

LOCAL char checkStackHdr [] = "\
    NAME        ENTRY       TID    SIZE   CUR  HIGH  MARGIN\n\
------------ ------------ -------- ----- ----- ----- ------\n";



/* things used by lkAddr, lkAddrFind & lkAddrNext */

#define NUM_SYMBLS	3

typedef struct		/* SYMBL - symbol table entry used by lkAddr */
    {
    unsigned int addr;	/* address associated with symbol */
    char *name;		/* points to name in system symbol table */
    UTINY type;		/* type of this symbol table entry */
    } SYMBL;

typedef struct		/* LKADDR_ARG */
    {
    int count;			/* number of entries printed by lkAddr */
    unsigned int addr;		/* target address */
    SYMBL symbl[NUM_SYMBLS];
    } LKADDR_ARG;


/* forward declarations */

LOCAL BOOL lkupPrint ();
LOCAL BOOL lkAddrFind ();
LOCAL BOOL lkAddrNext ();
LOCAL BOOL lkAddrPrintSame ();
LOCAL char *strMatch ();
LOCAL VOID taskIdListSort ();

/*******************************************************************************
*
* help - print a synopsis of selected routines
*
* Print the following list of the calling sequences to selected commonly used
* routines, mostly contained in usrLib (2).
* .CS
*  help                       Print this list
*  dbghelp                    Print debug help info
*  nfshelp                    Print nfs help info
*  nethelp                    Print network help info
*  tahelp                     Print spy help info
*  tihelp                     Print execution timer help info
*  h         [n]              Print (or set) shell history
*  i         [id]             Summary of tasks' TCBs
*  ti        id               Complete info on TCB for task
*  sp        adr,args...      Spawn a task, pri=100, opt=0, stk=7000
*  taskSpawn name,pri,opt,stk,adr,args... Spawn a task
*  td        id               Delete a task
*  ts        id               Suspend a task
*  tr        id               Resume a task
*  d         [adr,[nwords]]   Display memory
*  m         adr              Modify memory
*  mRegs     [task]           Modify a task's registers interactively
*  d0-d7,a0-a7,sr,pc  [task]  Display a register of a task
*  version                    Print UniWorks version info, and boot line
*  iam       "user","passwd"  Set user name and passwd
*  whoami                     Print user name
*  cd        "path"           Set current working path
*  pwd                        Print working path
*  devs                       List devices
*  ls        ["path"]         List contents of directory
*  rename    "old","new"      Change name of file
*  copy                       Copy std in to std out
*  ld        [syms,[noAbort]] Load std in into memory
*                               (syms = add symbols to table:
*                                -1 = none, 0 = globals, 1 = all)
*  lkup      ["substr"]       List symbols in system symbol table
*  lkAddr    adr              List symbol table entries near address
*  checkStack  [id]           List task stack sizes and usage
*  printErrno value          Print the name of a status value
*  period    secs,adr,args... Spawn task to call function periodically
*  repeat    n,adr,args...    Spawn task to call function n times
*                               (0=forever)
*  diskinit  "device"         Format and initialize RT-11 device
*  squeeze   "device"         Squeeze free space on RT-11 device
* .CE
*/

VOID help ()
    {
    static char *help_msg [] = {
    /* page 1 */
    "help                       Print this list",
    "dbghelp                    Print debugger help info",
    "nfshelp                    Print nfs help info",
    "nethelp                    Print network help info",
    "tihelp                     Print execution timer help info",
    "tahelp                     Print spy help info",
    "h         [n]              Print (or set) shell history",
    "i         [id]             Summary of tasks' TCBs",
    "ti        id               Complete info on TCB for task",
    "sp        adr,args...      Spawn a task, pri=100, opt=0, stk=7000",
    "taskSpawn name,pri,opt,stk,adr,args... Spawn a task",
    "td        id               Delete a task",
    "ts        id               Suspend a task",
    "tr        id               Resume a task",
    "d         [adr,[nwords]]   Display memory",
    "m         adr              Modify memory",
    "mRegs     [task]           Modify a task's registers interactively",
    "d0-d7,a0-a7,sr,pc  [task]  Display a register of a task",
    "version                    Print UniWorks version info, and boot line",
    "iam       \"user\",\"passwd\"  Set user name and passwd",
    "whoami                     Print user name",
    "devs                       List devices",
    /* page 2 */
    "cd        \"path\"           Set current working path",
    "pwd                        Print working path",
    "ls        [\"path\"]         List contents of directory",
    "rename    \"old\",\"new\"      Change name of file",
    "copy                       Copy std in to std out",
    "ld        [syms,[noAbort]] Load std in into memory",
    "                             (syms = add symbols to table:",
    "                              -1 = none, 0 = globals, 1 = all)",
    "lkup      [\"substr\"]       List symbols in system symbol table",
    "lkAddr    adr              List symbol table entries near address",
    "checkStack  [id]           List task stack sizes and usage",
    "printErrno value           Print the name of a status value",
    "period    secs,adr,args... Spawn task to call function periodically",
    "repeat    n,adr,args...    Spawn task to call function n times (0=forever)",
    "diskinit  \"device\"         Format and initialize RT-11 device",
    "squeeze   \"device\"         Squeeze free space on RT-11 device",
    NULL
    };
    FAST int i;
    char ch;

    printf ("\n");
    for (i = 0; help_msg [i] != NULL; i++)
	{
	printf ("%s\n", help_msg [i]);
	if ((i+1) % 20 == 0)
	    {
	    printf ("\nType <CR> to continue, Q<CR> to stop: ");
	    fioRdString (STD_IN, &ch, 1);
	    if (ch == 'q' || ch == 'Q')
		break;
	    else
		printf ("\n");
	    }
	}
    printf ("\n");
    }
/*******************************************************************************
*
* bootChange - change the bootLine
*
* This routine is called to change the boot line used in the bootroms.
* It is useful to change the boot line, prior to typing control-X, if
* one is using UniWorks remotely and thus has no access to the console.
*/

VOID bootChange ()

    {
    bootPromptForParams (sysBootLine);
    }
/*******************************************************************************
*
* periodRun - call a function periodically
*
* This routine calls the specified function with the specified args
* repeatedly, forever, delaying the specified number of seconds
* between calls.  It is intended to be used primarily by being spawned
* by the period (2) routine.
*
* Normally, this routine is called only by period (2), which spawns
* it as a task.
*
* SEE ALSO: period (2)
*/

VOID periodRun (secs, func, arg1, arg2, arg3, arg4, arg5,
		arg6, arg7, arg8)
    int secs;		/* number of seconds to delay between calls */
    FUNCPTR func;		/* function to call repeatedly */
    int arg1, arg2, arg3;	/* args to pass to func */
    int arg4, arg5, arg6;	/* args to pass to func */
    int arg7, arg8;		/* args to pass to func */

    {
    FOREVER
	{
	(* func) (arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);

	taskDelay (secs * sysClkGetRate ());
	}
    }
/*******************************************************************************
*
* period - spawn a task to call a function periodically
*
* This routine spawns a task that will repeatedly call the specified
* function with the specified arguments, delaying the specified number of
* seconds between calls.  It provides an easy way to have things
* happen periodically.
*
* For example, to have the task info displayed every 5 seconds,
* just type:
*	-> period 5, i
*
* Note: The task is spawned using the sp (2) routine.  See description
* of sp (2) for details about priority, options, stacksize, and task id.
*
* SEE ALSO: periodRun (2), sp (2)
*/

STATUS period (secs, func, arg1, arg2, arg3, arg4, arg5,
	       arg6, arg7, arg8)
    int secs;		   /* period in seconds */
    FUNCPTR func;	   /* pointer to function to call */
    int arg1, arg2, arg3;  /* args to pass to func */
    int arg4, arg5, arg6;  /* args to pass to func */
    int arg7, arg8; 	   /* args to pass to func */

    {
    if (sp (periodRun, secs, func, arg1, arg2,
	    arg3, arg4, arg5, arg6, arg7, arg8) == ERROR)
	return (ERROR);
    else
	return (OK);
    }
/*******************************************************************************
*
* repeatRun - call a function repeatedly
*
* This routine calls the specified function with the specified args
* the specified number of times.  If the specified number is 0,
* the routine is called endlessly.
*
* This routine is intended to be used primarily by being spawned by repeat (2).
*
* SEE ALSO: repeat (2)
*/

VOID repeatRun (n, func, arg1, arg2, arg3, arg4, arg5,
		arg6, arg7, arg8)
    FAST int n;			/* Number of times to call func */
    FAST FUNCPTR func;		/* Function to call repeatedly */
    int arg1, arg2, arg3;	/* args to pass to func */
    int arg4, arg5, arg6;	/* args to pass to func */
    int arg7, arg8;		/* args to pass to func */

    {
    FAST BOOL infinite = (n == 0);

    while (infinite || (--n >= 0))
	(* func) (arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
    }
/*******************************************************************************
*
* repeat - spawn a task to call a function repeatedly
*
* This routine spawns a task that will call the specified function 
* with the specified args, the specified number of times.
* If specified number is 0, the routine will be called forever,
* or until the spawned task is deleted.
*
* Note: The task is spawned using the sp (2) routine.  See description
* of sp (2) for details about priority, options, stacksize, and task id.
*
* RETURNS: OK, or ERROR if unable to spawn task.
*
* SEE ALSO: repeatRun (2), sp (2)
*/

STATUS repeat (n, func, arg1, arg2, arg3, arg4, arg5,
	       arg6, arg7, arg8)
    int n;		/* number of times to call function (0 = forever) */
    FUNCPTR func;	/* function to call */
    int arg1, arg2, arg3;	/* args to pass to func */
    int arg4, arg5, arg6;	/* args to pass to func */
    int arg7, arg8;		/* args to pass to func */

    {
    if (sp (repeatRun, n, func, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8)
	== ERROR)
	return (ERROR);
    else
	return (OK);
    }
/*******************************************************************************
*
* sp - spawn a task with default parameters
*
* This routine spawns the specified function as a task with priority
* 100, options VX_SUPERVISOR_MODE|VX_FP_TASK|VX_STDIO,
* a 7000 byte stack, and the highest task ID currently not used.
* The ID used is printed after the task is spawned.
*
* This routine is a short form of the underlying taskSpawn (2) routine,
* convenient for spawning tasks in which the above mentioned parameters 
* are not critical.  If those parameters are not acceptable, taskSpawn (2)
* should be called directly.
*
* RETURNS: task ID, or ERROR if unable to spawn task.
*
* SEE ALSO: taskLib (1), taskSpawn (2)
*
* VARARGS1
*/

int sp (func, arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9)
    FUNCPTR func;		/* function to call */
    int arg1, arg2, arg3;	/* passed to spawned task */
    int arg4, arg5, arg6;	/* passed to spawned task */
    int arg7, arg8, arg9;	/* passed to spawned task */

    {
    int id;
    int symaddr;
    int type;
    char name1[MAX_SYS_SYM_LEN + 1], name2[MAX_SYS_SYM_LEN +1];

    if (func == (FUNCPTR)0)
	{
	printf ("sorry, won't spawn task at PC = 0.\n");
	return (ERROR);
	}

    /* get the name from the entry point and get rid of the leading '_' */

    symValFind (sysSymTbl, (int) func, name1, (int *)&symaddr, &type);
    strcpy(name2,(char *)&name1[1]);

    /* spawn task, let taskSpawn pick the task id, report success or failure */

    id = taskSpawn (name2, spTaskPriority, spTaskOptions,
		    spTaskStackSize, func, arg1, arg2, arg3, arg4, arg5,
		    arg6, arg7, arg8, arg9, 0);

    if (id == ERROR)
	{
	printf ("not able to spawn task.\n");
	return (ERROR);
	}
    else
	{
	printf ("task spawned: id = %#x, name = %s\n", id, taskName (id));
	return (id);
	}
    }
/*******************************************************************************
*
* squeeze - reclaim fragmented free space on RT-11 volume
*
* This routine moves data around on an RT-11 volume so that all the
* little bits of free space left on the device are joined together.
*
* Caveat: there MUST NOT be any files open on the device when this procedure
* is called - if there are then their condition after the call will
* be unknown - almost certainly damaged, and writing to a file may
* corrupt the entire disk.
*
* RETURNS: OK, or ERROR if unable to open or squeeze device.
*/

STATUS squeeze (devName)
    char *devName;	/* RT-11 device to squeeze, eg. "/fd0/" */

    {
    FAST char *name = (devName == NULL) ? "." : devName;
    FAST int fd = open (name, WRITE, 0);

    if (fd < 0)
	{
	printf ("couldn't open \"%s\".\n", name);
	return (ERROR);
	}

    if (ioctl (fd, FIOSQUEEZE, 0) < OK)
	{
	printf ("couldn't squeeze \"%s\".\n", name);
	close (fd);
	return (ERROR);
	}

    printf ("\"%s\" squeezed.\n", name);
    close (fd);
    return (OK);
    }
/******************************************************************************
*
* taskIdFigure - translate a super name to a task id
*/

int taskIdFigure (superName)
    int superName;	/* id, number, or name */

    {
    char name [10];	/* superName number to name */
    int id;		/* id */
    int bb;		/* bit bucket */

    if (superName == 0)				/* default task id */
	return (taskIdDefault (0));
    else if (taskIdVerify (superName) == OK)	/* try explicit id */
	return (superName);

    sprintf (name, "%x", superName);
    
    if ((id = taskNameToId (name)) != ERROR)
	return (id);				/* name is a number */
    else if (vxMemProbe ((char *)superName, READ, 4, (char *)&bb) == OK)
	return (taskNameToId ((char *) superName)); /* name is a string */
    else
	return (ERROR);			/* unreasonable name, bus error! */
    }
/*******************************************************************************
*
* checkStack - print a summary of each task's stack usage
*
* This routine prints a summary of each task's stack usage.
* The summary includes the total stack size (SIZE), the current number
* of stack bytes used (CUR), the maximum number of stack bytes used (HIGH), and
* the number of bytes never used at the top of the stack
* (MARGIN [== SIZE - HIGH]).
*
* The maximum stack usage is determined by scanning down from the top of
* the stack for the first byte whose value is not 0xee.  The UniWorks spawn 
* routine initializes all bytes of a task's stack to 0xee before starting 
* a task.
*
* DEFICIENCIES
* It is possible for a task to write beyond the end of its stack, but
* not write into the last part of its stack.  CheckStack will not detect
* this.
*/

VOID checkStack (superName)
    int superName;	/* super name, summary for all tasks if 0 */

    {
    int idList [MAX_DSP_TASKS];	/* list of active task ids */
    int id;			/* task id */
    FAST int nTasks;
    FAST int ix;

    if (superName != 0)
	{
	/* do specified task */

	id = taskIdFigure (superName);

	if (id == ERROR)
	    printErr ("Task not found.\n");
	else
	    {
	    printf (checkStackHdr);
	    printStackSummary (id);
	    }
	}
    else
	{
	/* do all tasks */

	printf (checkStackHdr);

	nTasks = taskIdListGet (idList, NELEMENTS (idList));
	taskIdListSort (idList, nTasks);

	for (ix = 0; ix < nTasks; ++ix)
	    printStackSummary (idList [ix]);
	}
    }
/*******************************************************************************
*
* printStackSummary - print task stack summary line
*
* This routine is used by checkStack, to print each tasks stack summary line.
*/

LOCAL VOID printStackSummary (id)
    int id;	/* task id */

    {
    FUNCPTR entry;		/* task's initial entry point */
    char name[MAX_SYS_SYM_LEN + 1];	/* main routine names go here */
    FUNCPTR symboladdr;		/* address associated with 'name' */
    UTINY type;			/* symbol type */
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    UTINY *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */
    UTINY *spbottom;		/* task's bottom of stack */
    UTINY *sptop;		/* task's top of stack */
    UTINY *sphigh;		/* high point used in stack */
    TCBX *pTcbX = taskTcbX (id);	/* tcb extension */

    spbottom = (UTINY *) pTcbX->botOfStack;
    sptop    = (UTINY *) pTcbX->topOfStack;

    /* find high stack used */

    for (sphigh = sptop; sphigh < spbottom; sphigh++)
	{
	if (*sphigh != 0xee)	/* has this byte been changed? */
	    break;
	}

    /* Print the summary of the stack */

    printf ("%-12.12s", pTcbX->name);

    entry = pTcbX->entry; 
    symValFind (sysSymTbl, (int)entry, name, (int *)&symboladdr, &type);
    if (entry == symboladdr)		/* entry address (symbolic if poss.) */
	printf (" %-12.12s", name);
    else
	printf (" %-12.12x", entry);

    taskRegsGet (id, dregs, aregs, (char **)&sp, &sr, &pc);

    printf (" %8x %5d %5d %5d %6d\n",
	    id,
	    spbottom - sptop,	/* stack size */
	    spbottom - sp,	/* current stack ptr */
	    spbottom - sphigh,	/* high stack usage */
	    sphigh - sptop);	/* unused */
    }
/******************************************************************************
*
* taskIdListSort - sort the id list by priority
*
* This routine sorts the idList by task priority.
*/  

LOCAL VOID taskIdListSort (idList, nTasks)
    int idList[];
    int nTasks;

    {
    FAST int temp;
    int prevPri;
    int curPri;
    FAST int *pCurId;
    BOOL change = TRUE;
    FAST int *pEndId = &idList [nTasks];

    if (nTasks == 0)
	return;

    while (change)
	{
	change = FALSE;

	taskPriorityGet (idList[0], &prevPri);

	for (pCurId = &idList[1]; pCurId < pEndId; ++pCurId, prevPri = curPri)
	    {
	    taskPriorityGet (*pCurId, &curPri);

	    if (prevPri > curPri)
		{
		temp = *pCurId;
		*pCurId = *(pCurId - 1);
		*(pCurId - 1) = temp;
		change = TRUE;
		}
	    }
	}
    }
/*******************************************************************************
*
* i - print a summary of each task's TCB
*
* More complete info about a specific task can be gotten with ti (2).
*
* SEE ALSO: ti (2)
*/

VOID i (superName)
    int superName;	/* superName, summary for all tasks if 0 */

    {
    int idList [MAX_DSP_TASKS];		/* list of active ids */
    TASK_DESC td;			/* task info structure */
    FAST int nTasks;
    FAST int ix;

    if (superName != 0)
	{
	/* do specified task */

	if (taskInfoGet (taskIdFigure (superName), &td) == ERROR)
	    printErr ("Task not found.\n");
	else
	    {
	    printf (infoHdr);
	    printTaskSummary (&td);
	    }
	}
    else
	{
	/* do all tasks */

	printf (infoHdr);

	nTasks = taskIdListGet (idList, NELEMENTS (idList));
	taskIdListSort (idList, nTasks);

	for (ix = 0; ix < nTasks; ++ix)
	    {
	    if (taskInfoGet (idList [ix], &td) == OK)
		printTaskSummary (&td);
	    }
	}
    }
/*******************************************************************************
*
* printTaskSummary - print task summary line
*
* This routine is used by i and ti, to print each tasks summary line.
*/

LOCAL VOID printTaskSummary (pTd)
    TASK_DESC *pTd;

    {
    char name[MAX_SYS_SYM_LEN + 1];	/* main routine names go here */
    FUNCPTR entry;		/* task's initial entry point */
    FUNCPTR symboladdr;		/* address associated with 'name' */
    char statusString[10];	/* status string goes here */
    UTINY type;			/* symbol type */
    int dregs[8];		/* task's data registers */
    int aregs[7];		/* task's address registers */
    char *sp;			/* task's stack pointer */
    USHORT sr;			/* task's status register */
    INSTR *pc;			/* task's pc */

    /* find task's initial entry point, name in symbol table, and
     * current kernel status */

    entry = pTd->entry; 
    symValFind (sysSymTbl, (int)entry, name, (int *)&symboladdr, &type);
    taskStatusString (pTd->td_id, statusString);

    /* Print the summary of the TCB */

    printf ("%-12.12s", pTd->td_name);	/* print the name of the task */

    if (entry == symboladdr)		/* entry address (symbolic if poss.) */
	printf (" %-12.12s", name);
    else
	printf (" %-12.12x", entry);

    /* get tasks sp, sr, and pc;  if the tcb being printed is the
     * calling task's tcb, then vxTaskRegs will return garbage for pc 
     * so we fudge it a little so it won't look bad */

    taskRegsGet (pTd->td_id, dregs, aregs, &sp, &sr, &pc);

    if (taskIdSelf () == pTd->td_id)	/* caller's tcb is being printed? */
	{
	int dummy = (int)printTaskSummary;	/* lint's sake */
	pc = (INSTR *)dummy;
	}

    printf (" %8d %3d %-8s %8x %8x %7x %5d\n",
	    pTd->td_id,
	    pTd->td_priority,
	    statusString,
	    pc,
	    (int) sp,
	    pTd->errorStatus, 
	    pTd->td_delay);
    }
/*******************************************************************************
*
* ts - suspend a task, using taskSuspend
*
* This routine suspends execution of the specified task.
*
* SEE ALSO: tr(2), taskSuspend(2)
*/

VOID ts (superName)
    int superName;		/* superName */

    {
    int id = taskIdFigure (superName);

    if (id == ERROR)		/* no id found */
	{
	printErr ("Task not found.\n");
	return;
	}

    if (id == 0)		/* default task id never set */
	printf ("sorry, the shell can't suspend itself.\n");
    else
	{
	if (taskSuspend (id) != OK)
	    printErrno (0);
	}
    }
/*******************************************************************************
*
* tr - resume a task, using taskResume
*
* This routine resumes execution of the specified (suspended) task.
*
* SEE ALSO: ts(2), taskResume(2)
*/

VOID tr (superName)
    int superName;		/* superName */

    {
    int id = taskIdFigure (superName);
    
    if (id == ERROR)		/* no id found */
	{
	printErr ("Task not found.\n");
	return;
	}

    if (taskResume (id) != OK)
	printErrno (0);
    }
/*******************************************************************************
*
* td - delete a task, using taskDelete
*
* Delete the specified task.
*/

VOID td (superName)
    int superName;		/* superName */

    {
    int id = taskIdFigure (superName);

    if (id == ERROR)		/* no id found */
	{
	printErr ("Task not found.\n");
	return;
	}

    if (id == 0)
	printf ("sorry, the shell can't delete itself.\n");
    else
	{
	if (taskDelete (id) != OK)
	    printErrno (0);
	}
    }
/*******************************************************************************
*
* ti - print complete info from a task's TCB
*
* This routine prints all the TCB info, including registers, for the
* task whose id is specified.
*
* If id==0, tell about the defult task; that is, the last task to hit
* a breakpoint or an exception, etc.
*
* SEE ALSO: "Debugging"
*/

VOID ti (superName)
    int superName;	/* superName to tell about. If 0, default task */

    {
    TASK_DESC td;
    TCBX *pTcbx;
    FAST int id = taskIdFigure (superName);
    
    if (id == ERROR)			/* couldn't figure the super name */
	{
	printErr ("Task not found.\n");
	return;
	}

    id = taskIdDefault (id);		/* set the default task id */

    if (taskInfoGet (id, &td) != OK)	/* get the task's td */
	{
	printErr ("Task not found.\n");
	return;
	}

    /* Print the summary as in all_task_info, then all the regs. */

    printf (infoHdr);
    printTaskSummary (&td);

    printf ("\nSPBOTTOM = %6x, SPTOP = %6x, ", td.td_spbottom, td.topOfStack);
    printf ("OPTIONS = %2x\n",td.options);

    if (id != taskIdSelf ())	/* task can't get own registers */
	{
	taskRegsShow (id);	/* print register contents */
	fppPrintRegs (id);	/* print fp register contents */
	}

    /* print exception info if any */

    if ((pTcbx = taskTcbX (id)) != NULL)
	excInfoShow (&pTcbx->excInfo);
    }
/*******************************************************************************
*
* version - print UniWorks version info
*
* Prints the UniWorks version number, the date this UniWorks was made,
* and other interesting information.
*/

#include "version.h"
VOID version ()

    {
    printf ("\nUniWorks Development System: Version %s\n",UniWorksVersion);
    printf ("    Target: %s.\n", sysModel());
    printf ("    Kernel: %s.\n", kernelVersion ());
    printf ("    Made on %s.\n", creationDate);
    printf ("    Boot line:\n%s\n", sysBootLine);
    }
/*******************************************************************************
*
* m - modify memory
*
* This routine prompts the user for modifications to memory, starting at the
* specified address.  It prints each address, and the current contents of
* that address, in turn.  The user can respond in one of several ways:
* .CS
*	RETURN   - No change to that address, but continue
*		   prompting at next address.
*	<number> - Set the contents to <number>.
*	. (dot)	 - No change to that address, and quit.
*	<EOF>	 - No change to that address, and quit.
* .CE
* All numbers entered and displayed are in hexadecimal.
* Memory is treated as 16-bit words.
*
* SEE ALSO: mRegs(2)
*/

VOID m (adrs)
    char *adrs;		/* address to change */

    {
    char line[MAXLINE + 1];	/* leave room for EOS */
    char *pLine;		/* ptr to current position in line */
    int value;			/* value found in line */
    char excess;

    /* round down to word boundary */

    for (adrs = (char *) ((int) adrs & 0xfffffffe);	/* start on even addr */
         ;						/* FOREVER */
	 adrs = (char *) (((short *) adrs) + 1))	/* bump as short ptr */
	{
	/* prompt for substitution */

	printf ("%06x:  %04x-", adrs, (*(short *)adrs) & 0x0000ffff);

	/* get substitution value:
	 *   skip empty lines (CR only);
	 *   quit on end of file or invalid input;
	 *   otherwise put specified value at address */

	if (fioRdString (STD_IN, line, MAXLINE) == EOF)
	    break;

	line[MAXLINE] = EOS;	/* make sure input line has EOS */

	for (pLine = line; isspace (*pLine); ++pLine)	/* skip leading spaces*/
	    ;

	if (*pLine == EOS)			/* skip field if just CR */
	    continue;

	if (sscanf (pLine, "%x%1s", &value, &excess) != 1)
	    break;				/* quit if not number */

	* (short *) adrs = value;		/* assign new value */
	}

    printf ("\n");
    }
/*******************************************************************************
*
* d - display memory
*
* Display contents of memory, starting at adrs.  Memory is displayed in
* words.  The number of words displayed defaults to 64.  If
* nwords is non-zero, that number of words is printed, rounded up to
* the nearest number of full lines.  That number then becomes the default.
*
* SEE ALSO: m(2)
*/

VOID d (adrs, nwords)
    FAST char *adrs;	/* address to display */
    int nwords;		/* number of words to print */
			/* If 0, print 64 or last specified */
    {
    static char *last_adrs;
    static int dNbytes = 128;

    FAST int nbytes;
    FAST int byte;
    char ascii [17];

    ascii [16] = EOS;			/* Put an EOS on the string */

    nbytes = 2 * nwords;

    if (nbytes == 0)
	nbytes = dNbytes;	/* no count specified: use current byte count */
    else
	dNbytes = nbytes;	/* change current byte count */

    if (adrs == 0)
	adrs = last_adrs;	/* no address specified: use last address */

    adrs = (char *) ((int) adrs & ~1);	/* round adrs down to word boundary */


    /* print leading spaces on first line */

    bfill (ascii, 16, '.');

    printf ("%06x:  ", (int) adrs & ~0xf);

    for (byte = 0; byte < ((int) adrs & 0xf); byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}

    /* print out all the words */

    while (nbytes-- > 0)
	{
	if (byte == 16)
	    {
	    /* end of line:
	     *   print out ascii format values and address of next line */

	    printf ("  *%16s*\n%06x:  ", ascii, adrs);

	    bfill (ascii, 16, '.');		/* clear out ascii buffer */
	    byte = 0;				/* reset word count */
	    }

	printf ("%02x", *adrs & 0x000000ff);
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	if (*adrs == ' ' || (isascii (*adrs) && isprint (*adrs)))
	    ascii[byte] = *adrs;

	adrs++;
	byte++;
	}

    /* print remainder of last line */

    for (; byte < 16; byte++)
	{
	printf ("  ");
	if (byte & 1)
	    printf (" ");	/* space between words */
	if (byte == 7)
	    printf (" ");	/* extra space between words 3 and 4 */

	ascii[byte] = ' ';
	}

    printf ("  *%16s*\n", ascii);	/* print out ascii format values */

    last_adrs = adrs;
    }
/*******************************************************************************
*
* cd - change current default directory
*
* This routine sets the current default directory to the specified name.
*
* In this case, the default directory means a device name optionally followed
* by a directory local to that device.
*
* Changes to a new directory can be made by one of the following ways:
*
*         - specifying an entire path name with a device name
*              possibly followed by a directory name.
*              The entire path name would change.
*         - specifying a directory name starting with a '~' or '/' or '$'.
*              The directory part of the path (right after the device name)
*              would be replaced with the new directory.
*         - specifying a directory name to be appended to the current
*              default directory.
*              The directory would get appended on the current default
*              directory.
*
* An instance of ".." indicates to go up one level in the directory tree.
* Paths are condensed to reduce instances of ".." and ".".
*
*
* EXAMPLES
* .CS
*  cd "/fd0/"              - change to device "/fd0/"
*  cd "wrs:~leslie/vw"     - change to device "wrs:" with local directory
*                            "~leslie/vw"
*  cd "config"             - after the previous step, the new directory would
*                            be "wrs:~leslie/vw/config"
*  cd "../demo"             - after the previous step, the new directory would
*                            be "wrs:~leslie/vw/demo"
*  cd "/etc"                - after the previous step, the new directory would
*                            be "wrs:/etc"
*
*                            NOTE: '~' is only used on net devices
* .CE
*
* SEE ALSO: pwd (2)
*/

STATUS cd (name)
    char *name;		/* new directory name */

    {
    char *pPathDir;
    char defPath [MAX_FILENAME_LENGTH];
    char newPath [MAX_FILENAME_LENGTH];

    if (name == NULL)		/* if no new directory, do nothing */
	return (OK);

    if (iosDevFind (name, &pPathDir) == NULL)
	return (ERROR);
    
    if (name == pPathDir)
	{
	/* no device in 'name', use the default device */

	ioDefPathGet (defPath);
	pathCat (defPath, name, newPath);
	}
    else
	{
	/* 'name' begins with a device name */

	strcpy (newPath, name);
	}

    pathCondense (newPath);	/* resolve '..'s, '.'s, '/'s */
    (void) ioDefPathSet (newPath);
    return (OK);
    }
/*******************************************************************************
*
* pwd - print current default directory
*
* Prints the current working device/directory on standard out.
*
* SEE ALSO: cd (2)
*/

VOID pwd ()

    {
    char name [MAX_FILENAME_LENGTH];

    ioDefPathGet (name);
    printf ("%s\n", name);
    }
/*******************************************************************************
*
* copy - copy file1 (or std in) to file2 (or std out)
*
* This routine copies from standard in to standard out, until an endfile
* is reached. Some examples (as typed from the shell):
*
*.CS
*	copy <dog		prints the file dog, on default
*				file device, on the console.
*	copy >/ct0/dog		copies from the console to file
*				dog, on device /ct0/, until an
*				<EOF> (default ^D) is typed.
*	copy <dog >/ct0/dog	copies the file dog, on default
*				file device, to device /ct0/.
*	copy "file1", "file2"	conventional copy from file named
*				'file1' to file named 'file2'.
*.CE
*
* Remember that standard in and out are global, so that spawning
* the first three constructs will not work as expected.
*
* RETURNS:
*    OK, or
*    ERROR if unable to open/create in or out, or
*    error in copying from in to out.
*
* SEE ALSO: copyStreams (2), tyEOFSet (2)
*/

STATUS copy (in, out)
    char *in;	/* name of file to read  (if NULL assume stdin)  */
    char *out;	/* name of file to write (if NULL assume stdout) */

    {
    int inFd = in ? open (in, READ, 0) : STD_IN;
    int outFd = out ? creat (out, WRITE) : STD_OUT;
    int status;

    if (inFd < OK)
	{
	printErr ("Can't open \"%s\"\n", in);
	return (ERROR);
	}
    if (outFd < OK)
	{
	if (in)
	    close (inFd);
	printErr ("Can't write to \"%s\"\n", out);
	return (ERROR);
	}

    status = copyStreams (inFd, outFd);

    if (in)
	close (inFd);
    if (out && close (outFd) == ERROR)
	status = ERROR;

    return (status);
    }
/*******************************************************************************
*
* copyStreams - copy from/to specified streams
*
* This routine copies from the stream identified by inFd to the stream
* identified by outFd until an end of file is reached on inFd.
* This routine is used by copy (2).
*
* RETURNS: OK, or ERROR if error reading from inFd or writing to outFd.
*
* SEE ALSO: copy (2)
*
* INTERNAL
* Note use of printErr since printf's would probably go to the output stream
* outFd! Also note use of fioRead and large buffer (6 RT-11 blocks which is
* approximately one SS-SD disk track).
*/

STATUS copyStreams (inFd, outFd)
    int inFd;		/* file descriptor of stream to copy from */
    int outFd;		/* file descriptor of stream to copy to */

    {
    char buffer [2 * RT_BYTES_PER_BLOCK];
    int totalBytes = 0;
    int nbytes;

    while ((nbytes = fioRead (inFd, buffer, sizeof (buffer))) > 0)
	{
	if (write (outFd, buffer, nbytes) != nbytes)
	    {
	    printErr ("copy: error writing file.\n");
	    return (ERROR);
	    }

	totalBytes += nbytes;

	/* fioRead returns with less than full buffer only if EOF */

	if (nbytes != sizeof (buffer))
	    break;	/* EOF */
	}

    if (nbytes < 0)
	{
	printErr ("copy: error reading file after copying %d bytes.\n", 
		 totalBytes);
	return (ERROR);
	}

    printErr ("%d bytes copied.\n", totalBytes);

    return (OK);
    }
/*******************************************************************************
*
* diskinit - format and initialize a disk
*
* This routine formats and initializes an RT-11 file device.
* It makes calls to ioctl(2) to do the functions FIODISKINIT and FIOFORMAT.
*
* EXAMPLE
* From the shell:
*	diskinit "/fd0/"
*
* RETURNS: OK, or ERROR if unable to open the device, or to format or initialize
* it.
*
* SEE ALSO: rtLib(1)
*/

STATUS diskinit (devName)
    char *devName;	/* Name of the device to initialize */

    {
    FAST int fd;
    FAST char *name;

    /* If no directory name specified, use current working directory (".") */

    name = (devName == NULL) ? "." : devName;

    /* Open the device, format it, and initialize it. */

    if ((fd = open (name, WRITE, 0)) < 0)
	{
	printErr ("Couldn't open \"%s\".\n", name);
	return (ERROR);
	}

    if (ioctl (fd, FIODISKFORMAT, 0) < OK)
	{
	printErr ("Couldn't format \"%s\".\n", name);
	close (fd);
	return (ERROR);
	}

    if (ioctl (fd, FIODISKINIT, 0) < OK)
	{
	printErr ("Couldn't initialize directory on \"%s\".\n", name);
	close (fd);
	return (ERROR);
	}

    printf ("\"%s\" initialized.\n", name);
    close (fd);
    return (OK);
    }
/*******************************************************************************
*
* ld - load object module into memory
*
* This routine loads an object module from standard in.  That object module is
* expected to be in UNIX a.out format.  As the module is loaded, any external
* references in it will be resolved. If (syms == 0), global symbols defined
* in the module will be added to the system symbol table.  If (syms == 1),
* globals and locals will be added to the table.  If (syms == -1), no
* symbols will be added.
*
* If there is an error on the load (externals undefined, too many symbols,
* etc), shellAbortScript will normally be called, to stop any script that
* this routine was called from.  If noAbort == TRUE, errors are noted but
* ignored.
*
* The normal way of using ld is to load all symbols (syms == 1) during debug,
* and only global symbols later.
*
* EXAMPLE
* .CS
*  ld <module	Loads a.out file "module" from the default file
*		device into memory, and puts any global symbols
*		it defines into the table.
* .CE
*
* RETURNS:
*    OK, or
*    ERROR if too many symbols,
*    invalid object file format, or
*    error reading file.
*
* SEE ALSO: ldLib (1)
*/

STATUS ld (syms, noAbort)
    int syms;		/* -1, 0, or 1 */
    BOOL noAbort;	/* TRUE = don't abort script on error */

    {
    int status = ldLoad (STD_IN, syms);

    if (status == ERROR)
	{
	switch (errnoGet ())
	    {
	    case S_ldLib_TOO_MANY_SYMBOLS:
		printErr ("ld error: too many symbols.\n");
		break;
	    case S_fioLib_UNEXPECTED_EOF:
		printErr ("ld error: invalid object module format.\n");
		break;
	    default:
		printErr ("ld error: error reading file.\n");
		break;
	    }
	if (! noAbort)
	    shellAbortScript ();
	}

    return (status);
    }
/*******************************************************************************
*
* ls - list contents of current directory
*
* This routine is a mere shadow of UNIX's ls.  It lists the directory,
* always in the same format, of the specified device.  If device is not
* given, it lists the default file device.
*
* 'ls "."' or 'ls' lists the current default directory gotten to by cd.
*
* RETURNS: OK, or ERROR if couldn't open directory.
*/

STATUS ls (dirName)
    char *dirName;	/* device to list */

    {
    static char *month[] = {"   ", "Jan", "Feb", "Mar", "Apr", "May", "Jun",
			    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"};
    int fd;
    REQ_DIR_ENTRY rde;
    char curDirName [MAX_FILENAME_LENGTH];
    char *name;
    char *pTail;

    /* If no directory name specified or directory specified as ".",
     * use current working directory */

    if ((dirName == NULL) || (strcmp (dirName, ".") == 0))
	{
	ioDefPathGet (curDirName);
	name = curDirName;
	}
    else
	{
	/* does directory name start with a device name? */

	if (iosDevFind (dirName, &pTail) == NULL)
	    {
	    /* there is no default device for the specified directory */
	    return (ERROR);
	    }

	if (dirName == pTail)
	    {
	    /* dirName does not start with a device name,
	     * use the default directory */

	    FAST int dirNameLen;

	    ioDefPathGet (curDirName);
	    dirNameLen = strlen (curDirName);
	    if ((dirNameLen > 0) && (curDirName [dirNameLen - 1] != '/')
		&& (dirName [0] != '/'))
		strcat (curDirName, "/");
	    strcat (curDirName, dirName);
	    name = curDirName;
	    }
	else
	    name = dirName;
	}

    /* try to open file */

    if ((fd = open (name, READ, 0)) < OK)
	{
	printErr ("Can't open directory \"%s\"\n", name);
	return (ERROR);
	}

    for (rde.entryNum = 0;
	 ioctl (fd, FIODIRENTRY, &rde) == OK;
	 ++rde.entryNum)
	{
	if (rde.entryNum == 0)
	    {
	    /* don't print header if no directory entries are found */

	    printf (" size     date     name\n");
	    printf (" ----     ----     ----\n");
	    }

	if (rde.name[0] == EOS)
	    printf ("%6d             (EMPTY)\n", rde.nChars);
	else
	    printf ("%6d %s %2d %4d %s\n", rde.nChars, month[rde.month],
		   rde.day, rde.year, rde.name);
	}

    close (fd);
    return (OK);
    }

/*******************************************************************************
*
* rename - change name of file
*
* Change a file name from oldname to newname.
*
* RETURNS: OK, or ERROR if couldn't open or rename.
*/

STATUS rename (oldname, newname)
    char *oldname;		/* name of file to rename */
    char *newname;		/* name with which to rename file */

    {
    int fd;
    STATUS status;

    if ((oldname == NULL) || (newname == NULL))
	{
	printf ("usage: rename \"oldname\" \"newname\"\n");
	errnoSet (S_usrLib_NOT_ENOUGH_ARGS);
	return (ERROR);
	}

    /* try to open file */

    if ((fd = open (oldname, READ, 0)) < OK)
	{
	printf ("can't open: %s.\n", oldname);
	return (ERROR);
	}

    /* rename it */

    if (ioctl (fd, FIORENAME, newname) != OK)
	{
	printf ("unable to rename file %s.\n", oldname);
	status = ERROR;
	}
    else
	status = OK;

    close (fd);
    return (status);
    }

/*******************************************************************************
*
* mkdir - make a directory
*
* RETURNS: OK, or ERROR if couldn't make directory.
*/

STATUS mkdir (dirName)
    char *dirName;		/* directory name */

    {
    int fd;

    if ((fd = open (dirName, UPDATE | O_CREAT,
		    FSTAT_DIR | DEFAULT_DIR_PERM))
	 == ERROR)
	return (ERROR);
    return (close (fd));
    }

/*  This should go into usrLib since it can be done with other drivers,
 *  but, mkdir is here because of its NFS constants */

/*******************************************************************************
*
* rmdir - remove a directory
*
* RETURNS: OK, or ERROR if couldn't make directory.
*/
STATUS rmdir (dirName)
    char *dirName;

    {
    return (delete (dirName));
    }

/*******************************************************************************
*
* rm - remove a file
*
* RETURNS: OK, or ERROR if couldn't make directory.
*/
STATUS rm (fileName)
    char *fileName;

    {
    return (delete (fileName));
    }




/*******************************************************************************
*
* devs - list all system-known devices
*
* This routine prints, on standard out, a list of all devices which the
* I/O system knows about.
*
* SEE ALSO: iosDevList (2)
*/

VOID devs ()

    {
    iosDevList ();
    }
/*******************************************************************************
*
* lkup - list global symbols
*
* This utility routine lists all symbols in the system symbol table whose
* names contain the string substr.  If substr is not given
* (i. e. substr = NULL, which is the shell's default if a parameter is not
* specified), or if substr is an empty string (""), all symbols in the
* table will be listed.
*
* Symbols which are local (in the system symbol table only because their
* module was loaded by 'ld (1)') are so indicated.
*/

VOID lkup (substr)
    char *substr;	/* substring to match */

    {
    /* call lkupPrint for each symbol */
    symEach (sysSymTbl, lkupPrint, substr);
    }
/*******************************************************************************
*
* lkupPrint - support routine for lkup
*
* This routine is called by symEach to deal with each symbol in the table.
* If the symbol's name contains substr, this routine prints the symbol.  
* Otherwise, it doesn't.  If substr==NULL, every symbol is printed.
*/

LOCAL BOOL lkupPrint (name, val, type, substr)
    char *name;
    int val;
    TINY type;
    char *substr;

    {
    if (substr == NULL || strMatch (name, substr) != NULL)
	{
	printf ("%-15s 0x%08x %-8s", name, val, typeName [(type >> 1) & 7]);

	if ((type & N_EXT) == 0)
	    printf (" (local)");

	printf ("\n");
	}
    return (TRUE);
    }
/*******************************************************************************
*
* strMatch - find an occurence of a string in another string
*
* This is a simple pattern matcher used by lkupPrint.  It looks for an
* occurence of str2 in str1 and returns a pointer to that occurence in str1.
* If it doesn't find one, it returns 0.
*/

LOCAL char *strMatch (str1, str2)
    FAST char *str1;		/* where to look for match */
    FAST char *str2;		/* string to find a match for */

    {
    FAST int str2Length = strlen (str2);
    FAST int ntries = strlen (str1) - str2Length;

    for (; ntries >= 0; str1++, --ntries)
	{
	if (strncmp (str1, str2, str2Length) == 0)
	    return (str1);	/* we've found a match */
	}

    return (NULL);
    }
/*******************************************************************************
*
* lkAddr - list symbols with value near given value
*
* This utility routine lists the symbols in the system symbol table that
* are near the given value. The symbols that get displayed are:
*	symbols whose value is immediately less than the target value,
*	symbols with the target value,
*	succeeding symbols until at least 12 symbols have been displayed.
*
* Symbols which are local (in the system symbol table only because their
* module was loaded by 'ld (1)') are so indicated.
*
* INTERNAL
* The LKADDR_ARG structure is used to hold the target address, count, and
* the `best' symbols found by lkAddrFind and lkAddrNext, which are the
* routines that get called by symEach for each symbol in the system
* symbol table.
*
*/

VOID lkAddr (addr)
    unsigned int addr;		/* address around which to look */

    {
    FAST int ix;
    LKADDR_ARG arg;

    arg.count = 0;			/* haven't printed anything yet */
    arg.addr  = addr;

    for (ix = 0; ix < NUM_SYMBLS; ++ix)
	arg.symbl[ix].addr = NULL;		/* clear little symbol table */

    /* call lkAddrFind for each symbol */

    symEach (sysSymTbl, lkAddrFind, &arg);

    /* print out the symbols found */

    for (ix = 0; ix < NUM_SYMBLS; ix++)
	{
	if (arg.symbl[ix].addr != NULL)
	    {
	    arg.addr = arg.symbl[ix].addr;
	    symEach (sysSymTbl, lkAddrPrintSame, &arg);
	    }
	}

    if (arg.symbl[2].addr == NULL)
	return;			/* couldn't find anything greater */

    /* print until there are enough entries */

    while (arg.count < 12)
	{
	arg.addr = arg.symbl[2].addr;

	arg.symbl[2].addr = NULL;

	/* find next entry */

	symEach (sysSymTbl, lkAddrNext, &arg);

	if (arg.symbl[2].addr == NULL)
	    break;			/* couldn't find anything greater */

	/* print them */

	arg.addr = arg.symbl[2].addr;
	symEach (sysSymTbl, lkAddrPrintSame, &arg);
	}
    }
/*******************************************************************************
*
* lkAddrFind - support routine for lkAddr
*
* This routine is called by symEach to deal with each symbol in the table.
* If the value associated with the symbol is equal to lkAddrTarg, or closer
* to it than previous close values, the appropriate slot in the array 'symbl'
* is filled with the data for this symbol.
*
* RETURNS: TRUE
*/

LOCAL BOOL lkAddrFind (name, value, type, arg)
    char *name;		/* pointer to name in system symbol table */
    unsigned int value;	/* value associated with name */
    UTINY type;		/* type of this symbol table entry */
    LKADDR_ARG *arg;	/* handle for search parameters */

    {
    FAST unsigned int addr = value;

    if (addr < arg->addr)
	{
	if (addr > arg->symbl[0].addr)
	    {
	    /* found closer symbol that is less than target */

	    arg->symbl[0].addr = addr;
	    arg->symbl[0].name = name;
	    arg->symbl[0].type = type;
	    }
	}
    else if (addr == arg->addr)
	{
	/* found target, fill in target entry */

	arg->symbl[1].addr = addr;
	arg->symbl[1].name = name;
	arg->symbl[1].type = type;
	}
    else if (addr > arg->addr)
	{
	if ((addr < arg->symbl[2].addr) || (arg->symbl[2].addr == NULL))
	    {
	    /* found closer symbol that is greater than target */

	    arg->symbl[2].addr = addr;
	    arg->symbl[2].name = name;
	    arg->symbl[2].type = type;
	    }
	}

    return (TRUE);
    }
/*******************************************************************************
*
* lkAddrNext - another support routine for lkAddr
*
* This routine is called by symEach to deal with each symbol in the table.
* If the value associated with the symbol is greater than target, but less
* than symbl[2].addr, it replaces symbl[2].addr.
*
* RETURNS: TRUE
*/

LOCAL BOOL lkAddrNext (name, value, type, arg)
    char *name;		/* pointer to name in system symbol table */
    unsigned int value;	/* value associated with name */
    UTINY type;		/* type of this symbol table entry */
    LKADDR_ARG *arg;	/* handle for search parameters */

    {
    FAST unsigned int addr = value;

    if (addr > arg->addr)
	{
	if (addr < arg->symbl[2].addr || arg->symbl[2].addr == NULL)
	    {
	    /* found closer symbol that is greater than target */

	    arg->symbl[2].addr = addr;
	    arg->symbl[2].name = name;
	    arg->symbl[2].type = type;
	    }
	}

    return (TRUE);
    }
/*******************************************************************************
*
* lkAddrPrintSame - yet another support routine for lkAddr
*
* This routine is called by symEach to deal with each symbol in the table.
* If the value associated with the symbol is equal to target, print it.
*
* RETURNS: TRUE
*/

LOCAL BOOL lkAddrPrintSame (name, value, type, arg)
    char *name;		/* pointer to name in system symbol table */
    unsigned int value;	/* value associated with name */
    UTINY type;		/* type of this symbol table entry */
    LKADDR_ARG *arg;	/* handle for search parameters */

    {
    if (value == arg->addr)
	{
	printSTE (value, name, type);
	arg->count++;
	}

    return (TRUE);
    }
/*******************************************************************************
*
* printSTE - print symbol table entry
*/

LOCAL VOID printSTE (addr, name, type)
    unsigned int addr;	/* address associated with name */
    char *name;		/* pointer to name in system symbol table */
    UTINY type;		/* type of this symbol table entry */

    {
    printf ("0x%08x %-25s %-8s", addr, name, typeName[(type >> 1) & 7]);

    if ((type & N_EXT) == 0)
	printf (" (local)");

    printf ("\n");
    }
/*******************************************************************************
*
* a0 - return the contents of register a0 (also "a1-7", "d0-7")
*
* The contents of register a0 for the given task are extracted from the
* task's TCB. If no parameter is given (taskId = 0), the current
* default task is used.
*
* Similar routines exist for all data registers (d0 - d7), address registers
* (a0 - a7), the status register (sr), and the program counter (pc).  The
* stack pointer is accessed via a7 - if the task is in supervisor mode, the
* system stack pointer is returned; otherwise the user stack pointer is
* returned.
*
* RETURNS: Contents of register a0 (or requested register).
*
* SEE ALSO: "Debugging"
*/

int a0 (taskId)
    int taskId;			/* task's id, 0 means default task */
    {
    return (getOneReg (taskId, A0));
    }

int a1 (taskId) int taskId; { return (getOneReg (taskId, A1)); }
int a2 (taskId) int taskId; { return (getOneReg (taskId, A2)); }
int a3 (taskId) int taskId; { return (getOneReg (taskId, A3)); }
int a4 (taskId) int taskId; { return (getOneReg (taskId, A4)); }
int a5 (taskId) int taskId; { return (getOneReg (taskId, A5)); }
int a6 (taskId) int taskId; { return (getOneReg (taskId, A6)); }
int a7 (taskId) int taskId; { return (getOneReg (taskId, A7)); }
int d0 (taskId) int taskId; { return (getOneReg (taskId, D0)); }
int d1 (taskId) int taskId; { return (getOneReg (taskId, D1)); }
int d2 (taskId) int taskId; { return (getOneReg (taskId, D2)); }
int d3 (taskId) int taskId; { return (getOneReg (taskId, D3)); }
int d4 (taskId) int taskId; { return (getOneReg (taskId, D4)); }
int d5 (taskId) int taskId; { return (getOneReg (taskId, D5)); }
int d6 (taskId) int taskId; { return (getOneReg (taskId, D6)); }
int d7 (taskId) int taskId; { return (getOneReg (taskId, D7)); }

/*******************************************************************************
*
* sr - return the contents of the status register 
*
* The contents of the status register for the given task are extracted from the
* task's TCB. If no parameter is given (param = 0), the default task is used.
*
* RETURNS: Contents of status register.
*/

int sr (taskId)
    int taskId;			/* task's id, 0 means default task */
    {
    return (getOneReg (taskId, SR));
    }
/*******************************************************************************
*
* pc - return the contents of the program counter 
*
* The contents of the program counter for the given task are extracted from the
* task's TCB. If no parameter is given (param = 0), the default task is used.
*
* RETURNS: contents of program counter.
*/

int pc (taskId)
    int taskId;			/* task's id, 0 means default task */
    {
    return (getOneReg (taskId, PC));
    }
/*******************************************************************************
*
* getOneReg - return the contents of one register
*
* Given a task's ID, this routine returns the contents of the register 
* specified by the register code.  This routine is used by a0, d0, sr, etc.
*
* RETURNS: register contents, or ERROR.
*/

LOCAL int getOneReg (taskId, regCode)
    int taskId;			/* task's id, 0 means default task */
    int regCode;		/* code for specifying register */

    {
    int		dregs [8];	/* data registers */
    int		aregs [7];	/* address registers */
    char	*sp;		/* stack pointer */
    USHORT	sr;		/* status register */
    INSTR	*pc;		/* program counter */
    
    taskId = taskIdFigure (taskId);	/* translate super name to id */

    if (taskId == ERROR)		/* couldn't figure out super name */
	return (ERROR);
    taskId = taskIdDefault (taskId);	/* set the default id */

    if (taskRegsGet (taskId, dregs, aregs, &sp, &sr, &pc) != OK)
	return (ERROR);

    switch (regCode)
	{
	case D0: return (dregs [0]);	/* data registers */
	case D1: return (dregs [1]);
	case D2: return (dregs [2]);
	case D3: return (dregs [3]);
	case D4: return (dregs [4]);
	case D5: return (dregs [5]);
	case D6: return (dregs [6]);
	case D7: return (dregs [7]);

	case A0: return (aregs [0]);	/* address registers */
	case A1: return (aregs [1]);
	case A2: return (aregs [2]);
	case A3: return (aregs [3]);
	case A4: return (aregs [4]);
	case A5: return (aregs [5]);
	case A6: return (aregs [6]);

	case A7: return ((int) sp);	/* a7 is the stack pointer */

	case SR: return ((int) sr);
	case PC: return ((int) pc);
	}

    return (ERROR);		/* unknown regCode */
    }
/*******************************************************************************
*
* mRegs - modify registers
*
* This routine sequentially prompts the user for new values for a
* task's registers.
* If no parameter is given, the default task is used.  
* This routine prompts the user for modifications starting at register d0.
* It prints each register, and the current contents of that register, 
* in turn.  The user can respond in one of several ways:
* .CS
*	RETURN   - No change to that register, but continue
*		   prompting at next register.
*	<number> - Set the register to <number>.
*	. (dot)	 - No change to that register, and quit.
*	<EOF>	 - No change to that register, and quit.
* .CE
* All numbers entered and displayed are in hexadecimal,
* except floating-point values which may be entered in double precision.
*
* RETURNS: OK, or ERROR if no such task.
*
* SEE ALSO: m(2)
*/

STATUS mRegs (taskId)
    int taskId;			/* task's id, 0 means default task */

    {
    static char *dregm[]={"d0:", "d1:", "d2:", "d3:", "d4:", "d5:","d6:","d7:"};
    static char *aregm[]={"a0:", "a1:", "a2:", "a3:", "a4:", "a5:","a6:"};
    static char *fregm[]={"f0:", "f1:", "f2:", "f3:", "f4:", "f5:","f6:","f7:"};
    int dregs [8];	/* data registers */
    int aregs [7];	/* address registers */
    char *sp;		/* stack pointer */
    USHORT sr;		/* status register */
    INSTR *pc;		/* program counter */
    double fpregs [8];	/* floating point data registers */
    int fpcr;		/* floating point control register */
    int fpsr;		/* floating point status register */
    int fpiar;		/* floating point instruction address register */
    int ix;
    BOOL fppOk = FALSE;
    BOOL done  = FALSE;

    taskId = taskIdFigure (taskId);	/* translate superName to task id */

    if (taskId == ERROR)		/* couldn't figure out super name */
	return (ERROR);
    taskId = taskIdDefault (taskId);	/* set the default id */

    if (taskRegsGet (taskId, dregs, aregs, &sp, &sr, &pc) != OK)
	return (ERROR);

    /* data regs */
    for (ix = 0; ix < 8 && !done; ix++)
	done = (changeReg (dregm [ix], dregs[ix], &dregs[ix]) == ERROR);

    /* address regs */
    for (ix = 0; ix < 7 && !done; ix++)
	done = (changeReg (aregm [ix], aregs[ix], &aregs[ix]) == ERROR);

    if (!done)		/* stack pointer */
	done = (changeReg ("sp: ", (int)sp, &ix) == ERROR);
    if (!done)		/* status reg */
	{
	sp = (char *)ix;

	done = (changeReg ("sr: ", (int)sr, &ix) == ERROR);
	}
    if (!done)		/* program counter */
	{
	sr = (USHORT)ix;

	done = (changeReg ("pc: ", (int)pc, &ix) == ERROR);
	}
    if (!done)
	{
	pc = (INSTR *)ix;

	fppOk = (fppGetTaskRegs (taskId, fpregs, &fpcr, &fpsr, &fpiar) == OK);

	if (fppOk)
	    {
	    /* data regs */
	    for (ix = 0; ix < 8 && !done; ix++)
		done = (changeFpReg (fregm [ix], fpregs [ix], &fpregs[ix])
			== ERROR);

	    if (!done)		/* control register */
		done = (changeReg ("fpcr: ", fpcr, &ix) == ERROR);
	    if (!done)		/* status register */
		{
		fpcr = ix;

		done = (changeReg ("fpsr: ", fpsr, &ix) == ERROR);
		}
	    if (!done)		/* instruction address register */
		{
		fpsr = ix;

		done = (changeReg ("fpiar: ", fpiar, &ix) == ERROR);
		}
	    if (!done)
		fpiar = ix;
	    }
	}


    taskRegsSet (taskId, dregs, aregs, sp, sr, pc);

    if (fppOk)
	fppSetTaskRegs (taskId, fpregs, fpcr, fpsr, fpiar);


    return (OK);
    }
/*******************************************************************************
*
* changeReg - prompt the user for a hexadecimal register value
*
* ChangeReg reads a hexadecimal value from stdin.
* If a NULL string is read, the routine returns immediately.
* If an invalid hexadecimal value is read, ERROR is returned.
*
* RETURNS: OK, or ERROR.
*/

LOCAL STATUS changeReg (pPrompt, value, pValue)
    char *pPrompt;	/* prompt string */
    int value;		/* original value */
    int *pValue;	/* where to return new value */

    {
    char stringBuf [20];
    int valueBuf;
    char excess;

    printf ("%-6s %8x - ", pPrompt, value);

    if (fioRdString (STD_IN, stringBuf, sizeof (stringBuf)) == 1)
	{
	*pValue = value;
	return (OK);	/* just a CR */
	}
    
    if (sscanf (stringBuf, "%x%1s", &valueBuf, &excess) != 1)
	return (ERROR);

    *pValue = valueBuf;
    return (OK);
    }
/*******************************************************************************
*
* changeFpReg - prompt the user for a floating-point register value
*
* ChangeFpReg reads a floating-point value from stdin.
* If a NULL string is read, the routine returns immediately.
* If an invalid floating-point value is read, ERROR is returned.
*
* RETURNS: OK, or ERROR.
*/

LOCAL STATUS changeFpReg (pPrompt, value, pValue)
    char *pPrompt;	/* prompt string */
    double value;	/* original value */
    double *pValue;	/* where to return new value */

    {
    char stringBuf [20];
    double newValue;
    char excess;

    printf ("%-6s %8g - ", pPrompt, value);

    if (fioRdString (STD_IN, stringBuf, sizeof (stringBuf)) == 1)
	{
	*pValue = value;
	return (OK);		/* just a CR */
	}
    
    if (sscanf (stringBuf, "%F%1s", &newValue, &excess) != 1)
	return (ERROR);

    *pValue = newValue;
    return (OK);
    }
/*******************************************************************************
*
* printErrno - print the definition of specified error status value
*
* This routine prints the error status string from statSymTbl corresponding
* to the error status value passed as a parameter.  It is only useful if
* statSymTbl has been built and included in the system.
* If the errno argument is zero then the current task status is
* used by calling errnoGet (2).
* 
* This facility is described in errnoLib (1).
*
* SEE ALSO: errnoLib (1), errnoGet(2)
*/

VOID printErrno (errno)
    FAST int errno;	/* status code whose name is to be printed */

    {
    char statName [100];
    int pval;
    UTINY ptype;

    /* if no errno specified, get current tasks errno */

    if (errno == 0)
	{
	errno = errnoGet ();
	if (errno == OK)
	    {
	    printf ("OK.\n");
	    return;
	    }
	}

    if (statSymTbl == NULL)
	printf ("Error status symbol table not included.  errno = %#x.\n",
		errno);
    else
	{
	symValFind (statSymTbl, errno, statName, &pval, &ptype);

	if (pval != errno)
	    printf ("Unknown errno: 0x%x.\n", errno);
	else
	    printf ("%s\n", statName);
	}
    }
/*******************************************************************************
*
* printLogo - print UniWorks logo
*/

#include	"version.h"
#define	LOGO	"\n\
     **   **                   **     **\n\
    **   **   **    **   **   **     **   *******   ******    **  **   ******\n\
   **   **   ***   **   **   **     **   **   **   **   **   ** **    **\n\
  **   **   **  * **   **   **  *  **   **   **   ******    ****     *******\n\
 **   **   **   ***   **   **** ****   **   **   **  **    ** **         **\n\
*******   **    **   **   ***   ***   *******   **    **  **   **   ******\n\n"

VOID printLogo ()
{
	char	line[80];

	printf(LOGO);
	strcpy(line, "UniWorks Development System:  Version ");
	strcat(line, UniWorksVersion);
	center(line);
	strcpy(line,kernelVersion());
	center(line);
	center("copyright Integrated Solutions, Inc., 1987, 1988");
}

/*******************************************************************************
*
* iam - set remote user name and password
*
* The user name given here determines the access privileges available
* on the remote machine.  It should, of course, exist in /etc/passwd on
* that machine.
*
* SEE ALSO: whoami(2), remGetCurId(2), remCurIdSet(2)
*/

STATUS iam (newUser, newPasswd)
    char *newUser;	/* user name to use on remote */
    char *newPasswd;	/* password to use on remote (NULL = none) */

    {
    if (remCurIdSet (newUser, newPasswd) != OK)
	printErr ("User name or password too long\n");
    }
/*******************************************************************************
*
* whoami - print current remote identity
*
* Prints the user name being used on a remote machine, which was set with
* iam (2), on standard out.
*
* SEE ALSO: iam(2), remGetCurId(2), remCurIdSet(2)
*/

VOID whoami ()
    {
    char user [MAX_IDENTITY_LEN];

    remGetCurId (user, (char *) NULL);
    printf ("%s\n", user);
    }
/*******************************************************************************
* 
* logout - logout of UniWorks system
*
* This routine logs out from UniWorks.
* The client's socket is closed and standard I/O is restored to the console.
*
* SEE ALSO: rlogindTask (2), telnetdTask (2), shellLogout(2)
*/ 

VOID logout ()

    {
    shellLogout ();
    }
/*******************************************************************************
*
* h - display (or set) shell history
*/

VOID h (size)
    int size;	/* 0 = display, >0 = set history to new size */

    {
    shellHistory (size);
    }

center(s)
char	*s;
{
	int	n;

	n = 39 - strlen(s)/2;
	while(n-->0)
		printf(" ");
	printf("%s\n", s);
}

proNet(name,netaddr)
	char *name;
	char *netaddr;
{
	remAddHost(name,netaddr);
	ifconfig("pn0","inet",netaddr,0,0,0,0);
	return(OK);
}

tset (on)
int on;
{
    if (on) 
	tySetBackspace (0x7f);
    else
	tySetBackspace ('');
}
