/*    MIPS Computer Systems, Inc. Sunnyvale, CA. 
 *
 * $Author: rml $
 * $Source: /u2/src/graphics/X.V11R3/src/server/ddx/bwfb/bwfbdev/RCS/kbd.c,v $
 * $Revision: 1.1 $
 * $Date: 89/01/23 17:16:05 $
 *
 */
#ifndef lint
static char rcsid[] = "$Header: kbd.c,v 1.1 89/01/23 17:16:05 rml Exp $";
#endif
/*
 *   MIPS  */
/*
 * Keyboard input ( line discipline if LDISC is defined )
 * define TERMINAL for ascii terminals
 * handles conversion of up/down codes to ASCII
 */

#include "mkb.h"
#if NMKB > 0

#include "../h/param.h"
#include "../h/time.h"
#include "../h/kernel.h"
#include "../h/ioctl.h"
#include "../h/tty.h"
#include "../h/conf.h"
#include "../h/systm.h"
#include "../h/file.h"
#include "../h/uio.h"
#include "../h/errno.h"
#ifdef PROM
#include "../mon/mipsromvec.h"
#endif PROM
#ifdef CONS_KBD
#include "../bwfbdev/consdev.h"
#endif CONS_KBD
#include "../bwfbdev/kbd.h"
#include "../bwfbdev/kbio.h"
#include "../bwfbdev/kbdreg.h"
#include "../bwfbdev/kbstate.h"
#include "../bwfbdev/ev_queue.h"

#ifdef mips
#define spl5 spltty
#endif mips
/*
 * For now these are shared.
 */
extern struct keyboard	*mkbkeytables[];
extern char mkbkeystringtab[16][KTAB_STRLEN];

/*
 * Keyboard instance data.
 */
static int	kbd_downs_size = 15;		/* about 10 fingers */
typedef	struct	key_event {
	u_char	key_station;	/* Physical key station associated with event */
	vsEvent event;	/* Event that sent out on down */
} Key_event;
struct	kbddata {
	struct	keyboardstate kbdd_state;	/* State of keyboard & keyboard
					   						specific settings, e.g., tables */
	int	kbdflags;					/* contains Keyboard type and mode bits	*/
	int	kbd_translate;				/* is Keyboard doing ascii translation? */
#define YES	1
#define NOT 0
	struct	tty *kbdd_tp;				/* Tty connected to */
	int	(*kbdd_usecode)();				/* call out which calls back
											for translation */
	short	kbdd_sg_flags;		/* Tty mode flags */
	Event_queue kbdd_q;		/* kbd input q */
	caddr_t	kbdd_qdata;		/* address of kbdd_q data block */
	int	kbdd_qbytes;		/* # of bytes used for kbdd_qdata */
	struct	key_event *kbdd_downs;	/* Table of key stations currently down
					   that have vsEvents that need
					   to be matched with up transitions
					   when KB_EV_MODE flag is set */
	int	kbdd_downs_entries;	/* # of possible entries in kbdd_downs*/
	int	kbdd_downs_bytes;	/* # of bytes allocated for kbdd_downs*/
#ifdef LDISC
#define KBMAXREC 20			/* max input record size	*/
	char cbuf[KBMAXREC];	/* input buffer				*/
#endif LDISC
} kbddata[NMKB];

struct	kbddata *kbdtptokbdd();

/*
 * Constants setup during the first open of a kbd (so that hz is defined).
 */
int	kbd_repeatrate;
int	kbd_repeatdelay;

int	kbd_overflow_cnt;	/* Number of times kbd overflowed input q */
static int	kbd_overflow_msg = 1;	/* Whether to print message on q overflow */

#ifdef	KBD_DEBUG
int	kbd_debug = 0;
int	kbd_ra_debug = 0;
int	kbd_raw_debug = 0;
int	kbd_rpt_debug = 0;
int	kbd_input_debug = 0;
#endif	KBD_DEBUG

int	kbd_flush_fudge = 1;

#ifdef CONS_KBD
	int	kbduseconsole();
#else  CONS_KBD
	int kbdusetty();
#endif CONS_KBD
/*
 * Open a keyboard line discipline.
 * ttyopen sets line characteristics
 */
/* ARGSUSED */
mkbopen(dev, tp)
	dev_t dev;
	struct	tty *tp;
{
	register struct	kbddata *kbdd;
	struct cdevsw *dp;
	struct	sgttyb sg;
	register int err, i;

#ifdef LDISC
	if (tp->t_line == MKBLDISC )
		return (ENODEV);
	ttywflush(tp);
#endif LDISC

	/*
	 * See if tp is being used to drive kbd already.
	 */
	for (i = 0;i < NMKB; ++i)
		if (kbddata[i].kbdd_tp == tp)
			return (0);
	/*
	 * Get next free kbddata.
	 */
	for (i = 0;i < NMKB; ++i)
		if (kbddata[i].kbdd_tp == 0) {
			kbdd = &kbddata[i];
			goto found;
		}
	return (EBUSY);
found:
	/* Set these up only once so that they could be changed from adb */
	if (!kbd_repeatrate) {
		kbd_repeatrate = (hz+29)/30;
		kbd_repeatdelay = hz/2;
	}
#ifndef LDISC
	/*
	 * Open tty.
	 */
	if (err = ttyopen(dev, tp))
		goto error;
#else
	tp->t_cp = kbdd->cbuf;
	tp->t_inbuf = 0;
	tp->T_LINEP = (caddr_t)kbdd;
#endif LDISC
	/*
	 * Setup tty flags
	 */
	dp = &cdevsw[major(dev)];
	if (err = (*dp->d_ioctl) (dev, TIOCGETP, (caddr_t)&sg, 0))
			goto error;
	sg.sg_flags = RAW+ANYP;
	sg.sg_ispeed = sg.sg_ospeed = B9600;
	if (err = (*dp->d_ioctl) (dev, TIOCSETP, (caddr_t)&sg, 0))
		goto error;
	/*
	 * Setup private data.
	 */
#ifdef CONS_KBD
	kbdd->kbdd_usecode = (int (*)())kbduseconsole;
#else  
	kbdd->kbdd_usecode = (int (*)())kbdusetty;
#endif CONS_KBD
	kbdd->kbdd_tp = tp;
	kbdd->kbd_translate = YES;
	kbdd->kbdflags = KB_ASCII_MODE|KB_UNKNOWN;
	/* Allocate dynamic memory for q and downs table */
	kbdd->kbdd_qdata = (caddr_t) evq_malloc(&kbdd->kbdd_qbytes );
	kbdd->kbdd_downs_entries = kbd_downs_size;
	kbdd->kbdd_downs_bytes = kbd_downs_size * sizeof (Key_event);
	kbdd->kbdd_downs = (Key_event *) kmem_alloc((u_int)kbdd->kbdd_downs_bytes);
	if ((kbdd->kbdd_qdata == NULL) || (kbdd->kbdd_downs == NULL)) {
		printf("kbd: Couldn't allocate dynamic memory\n");
		err = ENOMEM;
		goto error;
	}
	evq_initialize(&kbdd->kbdd_q, kbdd->kbdd_qdata, kbdd->kbdd_qbytes);
	/*
	 * Reset kbd.
	 */
	kbdreset(tp);
	return (0);
error:
	bzero((caddr_t)kbdd, sizeof(struct kbddata));
	return (err);
}

/*
 * Close a keyboard.
 */
mkbclose(tp)
	struct	tty *tp;
{
	register int s;
	register struct kbddata *kbdd = kbdtptokbdd(&tp);

#ifdef CONS_KBD
	if (tp == (cdevsw[major(kbddev)].d_ttys + minor(kbddev)))
		return;
#endif CONS_KBD
	if (kbdd == 0)
		return;
	/*
	 * Close tty then zero kbddata structure.
	 */
/*	kmem_free(kbdd->kbdd_qdata, (u_int)kbdd->kbdd_qbytes); */
	evq_free( kbdd->kbdd_qbytes);
	kmem_free((caddr_t)kbdd->kbdd_downs, (u_int)kbdd->kbdd_downs_bytes);
#ifdef LDISC
	s = spl5();
	tp->t_cp = 0;
	tp->t_inbuf = 0;
	tp->t_rawq.c_cc = 0;		/* clear queues -- paranoid */
	tp->t_canq.c_cc = 0;
	tp->t_line = 0;			/* paranoid: avoid races */
	splx(s);
#else LDISC
	ttyclose(tp);
#endif LDISC
	bzero((caddr_t)kbdd, sizeof (*kbdd));
}

/*
 * Keyboard ioctl - only supports KIOC*
 */
mkbioctl(tp, cmd, data, flag)
	struct	tty *tp;
	int cmd;
	register caddr_t data;
	int flag;
{
	register int err;
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register short	new_flag;
	int	pri;

	if (kbdd == 0)
		return (EINVAL);
	err = 0;
	pri = spl5();
	switch (cmd) {

	case FIONREAD:
		if (kbdd->kbdflags & KB_EV_MODE) {
			*((int *) data) =
			sizeof (vsEvent) * evq_used(&kbdd->kbdd_q);
		}else if (kbdd->kbdflags & KB_ASCII_MODE) {
			*((int *) data) = evq_used(&kbdd->kbdd_q);
		} else 
			*((int *) data) = tp->t_rawq.c_cc;
		break;

	case KIOGMODE:
		*(int *)data = kbdd->kbdflags & KBMODE;
		break;

#ifdef TERMINAL
	case KIOCSTYPE:
		kbdd->kbdflags &= ~KBTYPE;
		kbdd->kbdflags |= *(int *)data & KBTYPE;
		kbdid(tp,kbdd->kbdflags & KBTYPE );
		/* fall thru... to set mode bits */
#endif TERMINAL

	case KIOSMODE:
		new_flag = *(int *)data & KBMODE;
		if (new_flag == kbdd->kbdflags & KBMODE)
			break;
		kbdd->kbdflags &= ~KBMODE;
		kbdd->kbdflags |= *(int *)data & KBMODE;
		if (new_flag & KB_ASCII_MODE) 
			kbdd->kbd_translate = YES;
		else
			kbdd->kbd_translate = NOT;
		kbdflush(tp);
#ifndef TERMINAL
		kbdreset(tp);
#endif  TERMINAL
		break;
		
	case KIOCGTYPE:
		*(int *)data = (kbdd->kbdd_state.k_idstate == KID_OK)?
		    kbdd->kbdd_state.k_id: -1;
		break;

	case KIOCCMD: {
		char cmd = *(int *)data;

		kbdcmd(tp, cmd);
		break;
		}

	case KIOCSETKEY:
		err = kbdsetkey(tp, (struct kiockey *)data);
		/*
		 * Since this only affects any subsequent key presses,
		 * don't do output_format_change.  One might want to
		 * toggle the keytable entries dynamically.
		 */
		break;

	case KIOCGETKEY:
		err = kbdgetkey(tp, (struct kiockey *)data);
		break;

	case KIOCSUSECODE:
		kbdd->kbdd_usecode = (int (*)())data;
		kbdflush(tp);
		kbdreset(tp);
		break;

	case KIOCGUSECODE:
		*(int *)data = (int)kbdd->kbdd_usecode;
		break;


#ifdef LDISC

	case TIOCSETD:
	case TIOCGETD:
#endif LDISC
	default:
		err = ttioctl(tp, cmd, data, flag);
#ifdef  KBD_DEBUG
		printf("mkbioctl: t_line= %d cmd= %d err= %d\n", tp->t_line, cmd, err);
#endif  KBD_DEBUG
		return (err);		/* pass thru... */

	}
	(void) splx(pri);
	return (err);
}
 
static
kbdflush(tp)
	struct  tty *tp;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	vsEvent fe;

#ifdef LDISC
		tp->t_cp = kbdd->cbuf;
		tp->t_inbuf = 0;
#else
	/* Flush pending data already sent to tty */
	ttyflush(tp, FREAD);
#endif LDISC
	/* Flush unread input */
	while (evq_get(&kbdd->kbdd_q, &fe) != EVENT_Q_EMPTY) {}
	/* Flush pending ups */
	bzero((caddr_t)(kbdd->kbdd_downs), kbdd->kbdd_downs_bytes);
	kbdcancelrpt(tp);
}
 
/*
 * Call consumer of keycode.
 */
static
kbduse(tp, keycode, k)
	struct	tty *tp;
	u_char	keycode;
	register struct keyboardstate *k;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);

	if (kbdd == 0 || kbdd->kbdd_usecode == 0)
		return;
	(*(kbdd)->kbdd_usecode)(tp, keycode, k);
}

/*
 * Match *tp to kbddata.
 */
struct	kbddata *
kbdtptokbdd(tp)
	struct	tty **tp;
{
	register i;
	extern	struct	tty cons;

#ifdef CONS_KBD
	/*
	 * If talking about the console then use kbddev.
	 */
	if (*tp == cdevsw[0/*console*/].d_ttys)
		*tp = cdevsw[major(kbddev)].d_ttys + minor(kbddev);
#endif CONS_KBD
			
	/*
	 * Get kbddata whose tp matches tp.
	 */
	for (i = 0; i < NMKB; i++)
		if (kbddata[i].kbdd_tp == *tp)
			return (&kbddata[i]);
	printf("kbd: kbdtptokbdd called with unknown tp %X\n", *tp);
	return (0);
}

/*
 * kbdclick is used to remember the current click value of the
 * Intrepid-3 keyboard.  This brain damaged keyboard will reset the
 * clicking to the "default" value after a reset command and
 * there is no way to read out the current click value.  We
 * cannot send a click command immediately after the reset
 * command or the keyboard gets screwed up.  So we wait until
 * we get the ID byte before we send back the click command.
 * Unfortunately, this means that there is a small window
 * where the keyboard can click when it really shouldn't be.
 * A value of -1 means that kbdclick has not been initialized yet.
 */
static int kbdclick = -1;

/*
 * Send command byte to keyboard
 */
static
kbdcmd(tp, cmd)
	struct	tty *tp;
	char cmd;
{
	int s;

	s = spl5();
	(void) ttyoutput(cmd, tp);
	ttstart(tp);
	(void) splx(s);
	if (cmd == KBD_CMD_NOCLICK)
		kbdclick = 0;
	else if (cmd == KBD_CMD_CLICK)
		kbdclick = 1;
}

/*
 * Reset the keyboard
 */
static
kbdreset(tp)
	struct	tty *tp;
{
	register struct keyboardstate *k;
	register struct kbddata *kbdd = kbdtptokbdd(&tp);

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;
	if (kbdd->kbd_translate) {
		k->k_idstate = KID_NONE;
#ifdef TERMINAL
		k->k_idstate = KID_OK;
#endif TERMINAL
		k->k_state = NORMAL;
		kbdcmd(tp, KBD_CMD_RESET);
	} else {
		bzero((caddr_t)k, sizeof(struct keyboardstate));
		k->k_id = KB_ASCII_MODE|KB_UNKNOWN;
		k->k_idstate = KID_OK;
	}
}

/*
 * Set individual keystation translation
 * TODO: Have each keyboard own own translation tables.
 */
static
kbdsetkey(tp, key)
	struct	tty *tp;
	struct	kiockey *key;
{
	int	strtabindex, i;
	struct	keymap *settable();
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	struct	keymap *km;

	if (kbdd == 0 || key->kio_station > 127)
		return (EINVAL);
	if (kbdd->kbdd_state.k_curkeyboard == NULL)
		return (EINVAL);
	if (key->kio_tablemask == KIOCABORT1) {
		kbdd->kbdd_state.k_curkeyboard->k_abort1 = key->kio_station;
		return (0);
	}
	if (key->kio_tablemask == KIOCABORT2) {
		kbdd->kbdd_state.k_curkeyboard->k_abort2 = key->kio_station;
		return (0);
	}
	if ((km = settable(tp, (u_int)key->kio_tablemask)) == NULL)
		return (EINVAL);
	if (key->kio_entry >= STRING && key->kio_entry <= STRING+15) {
		strtabindex = key->kio_entry-STRING;
		for (i = 0;i < KTAB_STRLEN;i++)
			mkbkeystringtab[strtabindex][i] = key->kio_string[i];
		mkbkeystringtab[strtabindex][KTAB_STRLEN-1] = '\0';
	}
	km->keymap[key->kio_station] = key->kio_entry;
	return (0);
}

/*
 * Get individual keystation translation
 */
static
kbdgetkey(tp, key)
	struct	tty *tp;
	struct	kiockey *key;
{
	int	strtabindex, i;
	struct	keymap *settable();
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	struct	keymap *km;

	if (kbdd == 0 || key->kio_station > 127)
		return (EINVAL);
	if (kbdd->kbdd_state.k_curkeyboard == NULL)
		return (EINVAL);
	if (key->kio_tablemask == KIOCABORT1) {
		key->kio_station = kbdd->kbdd_state.k_curkeyboard->k_abort1;
		return (0);
	}
	if (key->kio_tablemask == KIOCABORT2) {
		key->kio_station = kbdd->kbdd_state.k_curkeyboard->k_abort2;
		return (0);
	}
	if ((km = settable(tp, (u_int)key->kio_tablemask)) == NULL)
		return (EINVAL);
	key->kio_entry = km->keymap[key->kio_station];
	if (key->kio_entry >= STRING && key->kio_entry <= STRING+15) {
		strtabindex = key->kio_entry-STRING;
		for (i = 0; i < KTAB_STRLEN; i++)
			key->kio_string[i] = mkbkeystringtab[strtabindex][i];
	}
	return (0);
}

static
kbdidletimeout(tp)
        struct tty *tp;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;

	untimeout(kbdidletimeout, (caddr_t)tp);
	if (kbdd == 0)
		return;
	/*
	 * Double check that was waiting for idle timeout.
	 */
	k = &kbdd->kbdd_state;
	if (k->k_idstate == KID_IDLE)
		mkbinput(IDLEKEY, tp);
}

/*
 * A keypress received is processed
 * through the state machine to check for aborts.
 */
/*ARGSUSED*/
mkbinput(key, tp)
	register u_char key;
	struct tty *tp;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;
#ifdef	KBD_DEBUG
if (kbd_input_debug) printf("mkbinput key %X\n", key);
#endif  KBD_DEBUG
	switch (k->k_idstate) {

	case KID_NONE:
		if (key == IDLEKEY) {
			k->k_idstate = KID_IDLE;
			timeout(kbdidletimeout, (caddr_t)tp, hz/10);
		} else if (key == RESETKEY)
			k->k_idstate = KID_LKDT_2;
		return;

	case KID_IDLE:
		if (key == IDLEKEY)
			kbdid(tp, KB_KLUNK);
		else if (key == RESETKEY)
			k->k_idstate = KID_LKDT_2;
		else if (key & 0x80)
			kbdid(tp, (int)(KB_VT100 | (key&0x40)));
		else
			kbdreset(tp);
		return;

	case KID_LKDT_2:
		if (key == 0x02) {	    /* Intrepid-2 keyboard */
			kbdid(tp, KB_DT_2);
			return;
		}
		if (key == 0x03) {	    /* Intrepid-3 keyboard */
			kbdid(tp, KB_DT_3);
			/*
			 * We just did a reset command to a Intrepid-3 keyboard
			 * which sets the click back to the default
			 * (which is currently ON!).  We use the kbdclick
			 * variable to see if the keyboard should be
			 * turned on or off.  If it has not been set,
			 * then on we use the eeprom to determine
			 * if the default value is on or off.
			 */
			switch (kbdclick) {
			case 0:
				kbdcmd(tp, KBD_CMD_NOCLICK);
				break;
			case 1:
				kbdcmd(tp, KBD_CMD_CLICK);
				break;
			case -1:
			default:
				{
#ifdef PROM
#include "../mips/eeprom.h"

				if (EEPROM->ee_diag.eed_keyclick ==
				    EED_KEYCLICK)
					kbdcmd(tp, KBD_CMD_CLICK);
				else
#endif PROM
					kbdcmd(tp, KBD_CMD_NOCLICK);
				}
				break;
			}
			return;
		}
		kbdreset(tp);
		return;

	case KID_OK:
		if (key == 0 || key == 0xFF) {
			kbdreset(tp);
			return;
		}
		break;
	}
			
	switch (k->k_state) {

	normalstate:
		k->k_state = NORMAL;

	case NORMAL:
		if (k->k_curkeyboard && key == k->k_curkeyboard->k_abort1) {
			k->k_state = ABORT1;
			break;
		}
		kbduse(tp, key, k);
		if (key == IDLEKEY)
			k->k_state = IDLE1;
		break;

	case IDLE1:
		if (key & 0x80)	{	/* ID byte */
			if (k->k_id == KB_VT100)
				k->k_state = IDLE2;
			else 
				kbdreset(tp);
			break;
		}
		if (key != IDLEKEY) 
			goto normalstate;	/* real data */
		break;

	case IDLE2:
		if (key == IDLEKEY) k->k_state = IDLE1;
		else goto normalstate;
		break;

#ifdef PROM
	case ABORT1:
		if (k->k_curkeyboard) {
			if (key == k->k_curkeyboard->k_abort2) {
				DELAY(100000);
				montrap(*romp->v_abortent);
				k->k_state = NORMAL;
				kbduse(tp, (u_char)IDLEKEY, k);	/* fake */
				return;
			} else {
				kbduse(tp, k->k_curkeyboard->k_abort1, k);
				goto normalstate;
			}
		}
#endif PROM
	}
}

static
kbdid(tp, id)
	struct	tty *tp;
	int	id;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;

	k->k_id = id & 0xF;
	k->k_idstate = KID_OK;
	k->k_shiftmask = 0;
	if (id & 0x40)
		/* Not a transition so don't send event */
		k->k_shiftmask |= CAPSMASK;
	k->k_buckybits = 0;
	k->k_curkeyboard = mkbkeytables[k->k_id];
	k->k_rptkey = IDLEKEY;	/* Nothing happening now */
}

/*
 * This routine determines which table we should look in to decode
 * the current keycode.
 */
static struct keymap *
settable(tp, mask)
	struct	tty *tp;
	register u_int mask;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboard *kp;

	if (kbdd == 0)
		return (NULL);
	kp = kbdd->kbdd_state.k_curkeyboard;
	if (kp == NULL)
		return (NULL);
	if (mask & UPMASK)
		return (kp->k_up);
	if (mask & CTRLMASK)
		return (kp->k_control);
	if (mask & SHIFTMASK)
		return (kp->k_shifted);
	if (mask & CAPSMASK)
		return (kp->k_caps);
	return (kp->k_normal); 
}

static
kbdrpt(tp)
	struct	tty *tp;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;
	int	pri;

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;
#ifdef	KBD_DEBUG
if (kbd_rpt_debug) printf("kbdrpt key %X\n", k->k_rptkey);
#endif  KBD_DEBUG
	/*
	 * Since timeout is at low priority (interruptable),
	 * protect code with spl's.
	 */
	pri = spl5();
	kbdkeyreleased(tp, KEYOF(k->k_rptkey));
	kbduse(tp, k->k_rptkey, k);
	if (k->k_rptkey != IDLEKEY)
		timeout(kbdrpt, (caddr_t)tp, kbd_repeatrate);
	(void) splx(pri);
}

static
kbdcancelrpt(tp)
	struct	tty *tp;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;
	if (k->k_rptkey != IDLEKEY) {
		untimeout(kbdrpt, (caddr_t)tp);	/* Ignored if not found */
		k->k_rptkey = IDLEKEY;
	}
}

#ifdef CONS_KBD
/*ARGSUSED*/
static
kbduseconsole(tp, keycode, k)
	struct tty *tp;
	register u_char keycode;
	struct keyboardstate *k;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	extern int	cninput();
	if (!kbdd->kbd_translate )
		cninput((char)keycode, tp);
	else
		kbdtranslate(tp, keycode, cninput);
}
#endif CONS_KBD

#ifdef LDISC
/*ARGSUSED*/
static
kbdusetty(tp, keycode, k)
	struct tty *tp;
	register u_char keycode;
	struct keyboardstate *k;
{
	int kbdconsume();
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	if ( tp->t_inbuf == KBMAXREC ) {
		tp->t_cp = kbdd->cbuf;
		tp->t_inbuf = 0;
	}
	if (kbdd->kbdflags & KB_ASCII_MODE)
		kbdtranslate(tp, keycode, kbdconsume);
	else if (kbdd->kbdflags & KB_EV_MODE) {
		vsEvent fe;
		fe.vse_device = VSE_DKB;
		fe.vse_key = keycode;
		fe.vse_type = VSE_BUTTON;
		fe.vse_direction = VSE_KBTDOWN;
		kbdconsume((char)keycode, tp);
		kbdqueueevent(tp, &fe);
		fe.vse_direction = VSE_KBTUP;
		kbdconsume((char)keycode, tp);
		kbdqueueevent(tp, &fe);
	}
	else
		kbdconsume((char)keycode, tp);
	ttwakeup(tp);
}
static
kbdconsume(c, tp)
	struct tty *tp;
	register u_char c;
{
	*tp->t_cp++ = c;
	tp->t_rawq.c_cc++;
	tp->t_inbuf++;
}
#endif LDISC

static
kbdtranslate(tp, keycode, rint)
	struct	tty *tp;
	register u_char keycode;
	int	(*rint)();
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register u_char key, newstate, entry;
	register u_char enF0;
	register char *cp;
	register struct keyboardstate *k;
	struct keymap *km;
	vsEvent fe;

	if (kbdd == 0)
		return;
	if (!kbdd->kbdflags)
		rint(keycode, tp);
	k = &kbdd->kbdd_state;
	newstate = STATEOF(keycode);
	key = KEYOF(keycode);

	km = settable(tp, (u_int)(k->k_shiftmask | newstate));
	if (km == NULL) {		/* gross error */
		kbdcancelrpt(tp);
		return;
	}
	entry = km->keymap[key];
	enF0 = entry & 0xF0;
	/*
	 * Handle the state of toggle shifts specially.
	 * Toggle shifts should only come on downs.
	 */
	if (((entry >> 4) == (SHIFTKEYS >> 4)) &&
	    ((1 << (entry & 0x0F)) & k->k_curkeyboard->k_toggleshifts)) {
		if ((1 << (entry & 0x0F)) & k->k_togglemask) {
			newstate = RELEASED;
		} else {
			newstate = PRESSED;
		}
	}

/* We might need to add the RESET/IDLE/ERROR entries here.  FIXME? */
	if (newstate == PRESSED && entry != NOSCROLL &&
	    enF0 != SHIFTKEYS && enF0 != BUCKYBITS &&
	    !(entry >= LEFTFUNC && entry <= BOTTOMFUNC+15 &&
	    (kbdd->kbdflags & KB_EV_MODE ))) {
		if (k->k_rptkey != keycode) {
			kbdcancelrpt(tp);
			timeout(kbdrpt, (caddr_t)tp, kbd_repeatdelay);
			k->k_rptkey = keycode;
		}
	} else if (key == KEYOF(k->k_rptkey))		/* key going up */
		kbdcancelrpt(tp);
	if ((newstate == RELEASED) && (kbdd->kbdflags & KB_EV_MODE ))
		kbdkeyreleased(tp, key);

	switch (entry >> 4) {

	case 0: case 1: case 2: case 3:
	case 4: case 5: case 6: case 7:

		if (kbdd->kbdflags & KB_EV_MODE) {
			fe.vse_key = entry | k->k_buckybits;
			fe.vse_direction = VSE_KBTDOWN;
			/* Assume "up" table only generates shift changes */
			kbdkeypressed(tp, key, &fe);
		} else if (kbdd->kbdflags & KB_ASCII_MODE) {
			rint(entry | k->k_buckybits, tp);
		}
		break;

	case SHIFTKEYS >> 4: {
		u_int shiftbit = 1 << (entry & 0x0F);

		/* Modify toggle state (see toggle processing above) */
		if (shiftbit & k->k_curkeyboard->k_toggleshifts) {
			if (newstate == RELEASED) {
				k->k_togglemask &= ~shiftbit;
				k->k_shiftmask &= ~shiftbit;
			} else {
				k->k_togglemask |= shiftbit;
				k->k_shiftmask |= shiftbit;
			}
		} else
			k->k_shiftmask ^= shiftbit;
	    if ((kbdd->kbdflags & KB_EV_MODE ) && newstate == PRESSED){
			/*
			 * Relying on ordinal correspondence between
			 * vuid_event.h SHIFT_CAPSLOCK-SHIFT_RIGHTCTRL &
			 * kbd.h CAPSLOCK-RIGHTCTRL in order to
			 * correctly translate entry into fe.vse_key.
			 */
			fe.vse_key = SHIFT_CAPSLOCK + (entry & 0x0F);
			fe.vse_direction = VSE_KBTDOWN;
			kbdkeypressed(tp, key, &fe);
		}
		break;
		}

	case BUCKYBITS >> 4:
		k->k_buckybits ^= 1 << (7 + (entry & 0x0F));
	    if ((kbdd->kbdflags & KB_EV_MODE ) && newstate == PRESSED){
			/*
			 * Relying on ordinal correspondence between
			 * vuid_event.h SHIFT_META-SHIFT_TOP &
			 * kbd.h METABIT-SYSTEMBIT in order to
			 * correctly translate entry into fe.vse_key.
			 */
			fe.vse_key = SHIFT_META + (entry & 0x0F);
			fe.vse_direction = VSE_KBTDOWN;
			kbdkeypressed(tp, key, &fe);
		}
		break;

	case FUNNY >> 4:
		switch (entry) {
		case NOP:
			break;

		/*
		 * NOSCROLL/CTRLS/CTRLQ exist so that these keys, on keyboards
		 * with NOSCROLL, interact smoothly.  If a user changes
		 * his tty output control keys to be something other than those
		 * in keytables for CTRLS & CTRLQ then he effectively disables
		 * his NOSCROLL key.  One could imagine computing CTRLS & CTRLQ
		 * dynamically by watching TIOCSETC ioctl's go by in kbdioctl.
		 */
		case NOSCROLL:
			if (k->k_shiftmask & CTLSMASK)	goto sendcq;
			else				goto sendcs;

		case CTRLS:
		sendcs:
			/* A CTLSMASK change is not an event transition */
			k->k_shiftmask |= CTLSMASK;

			if (kbdd->kbdflags & KB_EV_MODE ) {
				fe.vse_key = ('S'-0x40) | k->k_buckybits;
				fe.vse_direction = VSE_KBTDOWN;
				kbdkeypressed(tp, key, &fe);
			} else if (kbdd->kbdflags & KB_ASCII_MODE ) {
				rint(('S'-0x40) | k->k_buckybits, tp);
			}
			break;

		case CTRLQ:
		sendcq:
			/* A CTLSMASK change is not event transition */

			if (kbdd->kbdflags & KB_EV_MODE ) {
				fe.vse_key = ('Q'-0x40) | k->k_buckybits;
				fe.vse_direction = VSE_KBTDOWN;
				kbdkeypressed(tp, key, &fe);
			}else if (kbdd->kbdflags & KB_ASCII_MODE ) {
				rint(('Q'-0x40) | k->k_buckybits, tp);
			}
			k->k_shiftmask &= ~CTLSMASK;
			break;

		case IDLE:
			/*
			 * Minor hack to prevent keyboards unplugged
			 * in caps lock from retaining their capslock
			 * state when replugged.  This should be
			 * solved by using the capslock info in the 
			 * KBDID byte.
			 */
			if (keycode == NOTPRESENT)
				k->k_shiftmask = 0;
			/* Fall thru into RESET code */

		case KBRESET:
		gotreset:
			k->k_shiftmask &= k->k_curkeyboard->k_idleshifts;
			k->k_shiftmask |= k->k_togglemask;
			k->k_buckybits &= k->k_curkeyboard->k_idlebuckys;
			kbdcancelrpt(tp);
			kbdreleaseall(tp);
			break;

		case ERROR:
			printf("kbd: Error detected\r\n");
			goto gotreset;

		/*
		 * Remember when adding new entries that,
		 * if they should NOT auto-repeat,
		 * they should be put into the IF statement
		 * just above this switch block.
		 */
		default:
			goto badentry;
		}
		break;

	case STRING >> 4:
		cp = &mkbkeystringtab[entry & 0x0F][0];
		while (*cp != '\0') {
			if (kbdd->kbdflags & KB_EV_MODE ) {
				fe.vse_key = *cp;
				fe.vse_direction = VSE_KBTDOWN;
				/*
				 * Pretend as if each cp pushed and released
				 * Calling kbdqueueevent avoids addr translation
				 * and pair base determination of kbdkeypressed.
				 */
				kbdqueueevent(tp, &fe);
				fe.vse_direction = VSE_KBTUP;
				kbdqueueevent(tp, &fe);
			} else if (kbdd->kbdflags & KB_ASCII_MODE ) {
				rint(*cp, tp);
			}
			cp++;
		}
		break;

	/*
	 * Remember when adding new entries that,
	 * if they should NOT auto-repeat,
	 * they should be put into the IF statement
	 * just above this switch block.
	 */
	default:
		if (entry >= LEFTFUNC && entry <= BOTTOMFUNC+15) {
			switch ((kbdd->kbdflags & KBMODE) >> 4) {

			case KB_ASCII_MODE >>4: {
				char	buf[10], *strsetwithdecimal();

				if (newstate == RELEASED)
					break;
				cp = strsetwithdecimal(&buf[0], (u_int)entry,
				    sizeof (buf) - 1);
				rint('\033', tp); /* Escape */
				rint('[', tp);
				while (*cp != '\0') {
					rint(*cp, tp);
					cp++;
				}
				rint('z', tp);
				break;
				}

			case KB_EV_MODE >>4:
				/*
				 * Take advantage of the similar ordering of
				 * kbd.h function keys and vuid_event.h function
				 * keys to do a simple translation to achieve
				 * a mapping between the 2 different address
				 * spaces.
				 */
				fe.vse_key = entry-LEFTFUNC+KEY_LEFTFIRST;
				fe.vse_direction = VSE_KBTDOWN;
				/*
				 * Assume "up" table only generates
				 * shift changes.
				 */
				kbdkeypressed(tp, key, &fe);
				/*
				 * Function key events can be expanded by
				 * terminal emulator software to produce the
				 * standard escape sequence generated by the
				 * KB_ASCII_MODE case above if a function key event
				 * is not used by terminal emulator software
				 * directly.
				 */
				 break;
			}
		}
	badentry:
		break;
	}
}

static
char *
strsetwithdecimal(buf, val, maxdigs)
	char	*buf;
	u_int	val, maxdigs;
{
	int	hradix = 5;
	char	*bp;
	int	lowbit;
	char	*tab = "0123456789abcdef";

	bp = buf + maxdigs;
	*(--bp) = '\0';
	while (val) {
		lowbit = val & 1;
		val = (val >> 1);
		*(--bp) = tab[val % hradix * 2 + lowbit];
		val /= hradix;
	}
	return (bp);
}

static
kbdkeypressed(tp, key_station, fe)
	struct	tty *tp;
	u_char	key_station;
	vsEvent *fe;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct keyboardstate *k;
	register struct keyboard *kp;
	register struct key_event *ke;
	register int i;
	register u_char base, pair;

	if (kbdd == 0)
		return;
	k = &kbdd->kbdd_state;
	kp = k->k_curkeyboard;
	if (kp == NULL)
		return;
#ifdef	KBD_DEBUG
	if (kbd_debug) printf("KBD PRESSED key=%D\n", key_station);
#endif  KBD_DEBUG
	/* Set key directions */
	if (fe->vse_key < VKEY_FIRST) {
		/* Strip buckybits */
		base = fe->vse_key & 0x7F;
		/* Find base that is the physical keyboard if CTRLed */
		if (k->k_shiftmask & (CTRLMASK | CTLSMASK)) {
			struct keymap *km;

			/* Order of these tests same as in settable */
			if (k->k_shiftmask & SHIFTMASK)
				km = kp->k_shifted;
			else if (k->k_shiftmask & CAPSMASK)
				km = kp->k_caps;
			else
				/* CTRLed keyboard maps into normal for base */
				km = kp->k_normal;
			if (km == NULL)
				/* Gross error */
				km = kp->k_normal;
			base = km->keymap[key_station];
		}
		if (base != fe->vse_key) {
			pair = base;
			goto send;
		}
	}
	pair = 0;
send:
	/* Adjust event id address for multiple keyboard/workstation support */
	/* Scan table of down key stations */
	for (i = 0, ke = kbdd->kbdd_downs;
	    i < kbdd->kbdd_downs_entries; i++, ke++) {
		/* Keycode already down? */
		if (ke->key_station == key_station) {
#ifdef	KBD_DEBUG
			printf("kbd: Double entry in downs table (%D,%D)!\n",
			    key_station, i);
#endif	KBD_DEBUG
			goto add_entry;
		}
		if (ke->key_station == 0) {
add_entry:
			ke->key_station = key_station;
			ke->event = *fe;
			kbdqueueevent(tp, fe);
			return;
		}
	}
	printf("kbd: Too many keys down!\n");
}

static
kbdkeyreleased(tp, key_station)
	struct	tty *tp;
	u_char key_station;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct key_event *ke;
	register int i;

	if (kbdd == 0)
		return;
	if (! (kbdd->kbdflags & KB_EV_MODE))
		return;
	if (key_station == IDLEKEY)
		return;
#ifdef	KBD_DEBUG
	if (kbd_debug) printf("KBD RELEASE key=%D\n", key_station);
#endif  KBD_DEBUG
	/* Scan table of down key stations */
	for (i = 0, ke = kbdd->kbdd_downs;
	    i < kbdd->kbdd_downs_entries; i++, ke++) {
		/* Found? */
		if (ke->key_station == key_station) {
			ke->key_station = 0;
			ke->event.vse_direction = VSE_KBTUP;
			kbdqueueevent(tp, &ke->event);
			return;
		}
	}
	/*
	 * Ignore if couldn't find because may be called twice for the
	 * same key station in the case of the kbdrpt routine being called
	 * unnecessarily.
	 */
	return;
}
	
static
kbdreleaseall(tp)
	struct	tty *tp;
{
	struct kbddata *kbdd = kbdtptokbdd(&tp);
	register struct key_event *ke;
	register int i;

	if (kbdd == 0)
		return;
#ifdef	KBD_DEBUG
	if (kbd_debug && kbd_ra_debug) printf("KBD RELEASE ALL\n");
#endif  KBD_DEBUG
	/* Scan table of down key stations */
	for (i = 0, ke = kbdd->kbdd_downs;
	    i < kbdd->kbdd_downs_entries; i++, ke++) {
		/* Key station not zero */
		if (ke->key_station)
			kbdkeyreleased(tp, ke->key_station);
			/* kbdkeyreleased resets kbdd_downs entry */
	}
}

static
kbdqueueevent(tp, fe)
	struct	tty *tp;
	vsEvent *fe;
{
	struct kbddata *kbdd = kbdtptokbdd(&tp);

	if (kbdd == 0)
		return;
	fe->vse_device = VSE_DKB;
	uniqtime(&fe->time);
	if (evq_put(&kbdd->kbdd_q, fe) == EVENT_Q_OVERFLOW) {
		if (kbd_overflow_msg)
			printf("kbd: Buffer flushed when overflowed.\n");
		kbdflush(tp);
		kbd_overflow_cnt++;
	}
#ifdef ndef
	if (evq_used(&kbdd->kbdd_q) == 1)
		/* trigger select, sigio etc...*/
		ttwakeup(tp);
#endif ndef
}

mkbread(tp, uio)
	struct	tty *tp;
	struct uio *uio;
{
	register struct kbddata *kbdd = kbdtptokbdd(&tp);
	int pri;
	char c;
	vsEvent fe;
	int error = 0;

	if (kbdd == 0)
		return (EINVAL);
#ifdef LDISC
	if ((tp->t_state&TS_CARR_ON) == 0)
		return (EIO);
#endif LDISC
	pri = spl5();
	/*
	 * Wait on tty raw queue if this queue is empty 
	 */
	while (tp->t_rawq.c_cc <= 0) {
		if (tp->t_state&TS_NBIO) {
			(void) splx(pri);
			return (EWOULDBLOCK);
		}
		sleep((caddr_t)&tp->t_rawq, TTIPRI);
	}
	if (kbdd->kbdflags & KB_EV_MODE ) {
		while ((evq_peek(&kbdd->kbdd_q, &fe) != EVENT_Q_EMPTY) &&
				(uio->uio_resid >= sizeof (fe))) {
			/* lower pri to avoid mouse droppings */
			(void) splx(pri);
			if (error = uiomove((caddr_t)&fe, 
				sizeof(fe), UIO_READ, uio)) 
				break;
			else
				(void)evq_get(&kbdd->kbdd_q, &fe);
			/* spl5 should return same priority as pri */
			pri = spl5();
		}
	} else if (kbdd->kbdflags & KB_ASCII_MODE ) {

		while ((evq_peek(&kbdd->kbdd_q, &fe) != EVENT_Q_EMPTY) &&
		(uio->uio_resid != 0)) {
			/* lower pri to avoid mouse droppings? */
			c = fe.vse_key;
			if (error = ureadc(c, uio))
				break;
			else
				(void)evq_get(&kbdd->kbdd_q, &fe);
		}
			
	}
#ifdef LDISC
	else {
		c = *--tp->t_cp;
		error = ureadc(c, uio);
	}
	tp->t_rawq.c_cc--;
#else
	/* Flush tty if no more to read */
	if (evq_is_empty(&kbdd->kbdd_q))
		ttyflush(tp, FREAD);
#endif  LDISC
	return (error);
}
#endif  NMKB
