/*
 * $Header: bwfbMouse.c,v 1.1 89/01/23 17:15:26 rml Exp $
 */
/*-
 * bwfbMouse.c --
 *
 */

#define NEED_EVENTS
#include    "bwfb.h"

typedef struct {
    int	    bmask;	    /* Current button state */
    Bool    mouseMoved;	    /* Mouse has moved */
} BwfbMsPrivRec, *BwfbMsPrivPtr;

static void 	  	bwfbMouseCtrl();
static int 	  	bwfbMouseGetMotionEvents();
static vsEvent 	*bwfbMouseGetEvents();
static void 	  	bwfbMouseProcessEvent();
void 	  	bwfbMouseDoneEvents();

static BwfbMsPrivRec	bwfbMousePriv;
static PtrPrivRec 	sysMousePriv = {
    -1,				/* Descriptor to device */
    bwfbMouseGetEvents,		/* Function to read events */
    bwfbMouseProcessEvent,	/* Function to process an event */
    bwfbMouseDoneEvents,	/* When all the events have been */
				/* handled, this function will be */
				/* called. */
    0,				/* Current X coordinate of pointer */
    0,				/* Current Y coordinate */
    NULL,			/* Screen pointer is on */
    (pointer)&bwfbMousePriv,	/* Field private to device */
};

#ifdef ISWINDOWS
extern	int indev;
#endif

/*-
 *-----------------------------------------------------------------------
 * bwfbMouseProc --
 *	Handle the initialization, etc. of a mouse
 *
 * Results:
 *	none.
 *
 * Side Effects:
 *
 * Note:
 *	When using bwfbwindows, all input comes off a single fd, stored in the
 *	global windowFd.  Therefore, only one device should be enabled and
 *	disabled, even though the application still sees both mouse and
 *	keyboard.  We have arbitrarily chosen to enable and disable windowFd
 *	in the keyboard routine bwfbKbdProc rather than in bwfbMouseProc.
 *
 *-----------------------------------------------------------------------
 */
int
bwfbMouseProc (pMouse, what)
    DevicePtr	  pMouse;   	/* Mouse to play with */
    int	    	  what;	    	/* What to do with it */
{
    register int  fd;
    int	    	  format;
    static int	  oformat;
    BYTE    	  map[4];

    switch (what) {
	case DEVICE_INIT:
	    if (pMouse != LookupPointerDevice()) {
		ErrorF ("Cannot open non-system mouse");	
		return (!Success);
	    }

		if (sysMousePriv.fd >= 0) {
		    fd = sysMousePriv.fd;
		} else {
#ifdef ISWINDOWS
		    fd = indev;
#else
		    fd = open ("/dev/mouse", O_RDWR, 0);
		    if (fd < 0) {
			Error ("Opening /dev/mouse");
			return (!Success);
		    }
			format = MMOUSELDISC;
			if (ioctl (fd, TIOCSETD , &format) < 0) {
			  Error ("TIOCSETD");
			  return(!Success);
		  	}
#endif
		    if (fcntl (fd, F_SETFL, (FNDELAY|FASYNC)) < 0
			|| fcntl(fd, F_SETOWN, getpid()) < 0) {
			    perror("bwfbMouseProc");
			    ErrorF("Can't set up mouse on fd %d\n", fd);
			}
		    
		    sysMousePriv.fd = fd;
		}

	    sysMousePriv.pScreen = &screenInfo.screen[0];
	    sysMousePriv.x = sysMousePriv.pScreen->width / 2;
	    sysMousePriv.y = sysMousePriv.pScreen->height / 2;

	    bwfbMousePriv.bmask = 0;
	    bwfbMousePriv.mouseMoved = FALSE;

	    pMouse->devicePrivate = (pointer) &sysMousePriv;
	    pMouse->on = FALSE;
	    map[1] = 1;
	    map[2] = 2;
	    map[3] = 3;
	    InitPointerDeviceStruct(
		pMouse, map, 3, bwfbMouseGetMotionEvents, bwfbMouseCtrl);
	    break;

	case DEVICE_ON:
#ifndef ISWINDOWS
		format = TBMOUSE | TB_EV_MODE | TB_DELTA;
		if (ioctl (((PtrPrivPtr)pMouse->devicePrivate)->fd,
			BIOSTYPE , &format) < 0) {
			Error ("BIOSTYPE");
			return(!Success);
		}
#endif
		AddEnabledDevice (((PtrPrivPtr)pMouse->devicePrivate)->fd);
	    pMouse->on = TRUE;
	    break;

	case DEVICE_CLOSE:
#ifndef ISWINDOWS
		oformat = TBMOUSE | TB_DELTA;
		if (ioctl (((PtrPrivPtr)pMouse->devicePrivate)->fd,
			BIOSTYPE, &oformat) < 0) {
			Error ("BIOSTYPE");
		}
#endif
	    break;

	case DEVICE_OFF:
	    pMouse->on = FALSE;
		RemoveEnabledDevice (((PtrPrivPtr)pMouse->devicePrivate)->fd);
	    break;
    }
    return (Success);
}
	    
/*-
 *-----------------------------------------------------------------------
 * bwfbMouseCtrl --
 *	Alter the control parameters for the mouse. Since acceleration
 *	etc. is done from the PtrCtrl record in the mouse's device record,
 *	there's nothing to do here.
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
static void
bwfbMouseCtrl (pMouse)
    DevicePtr	  pMouse;
{
}

/*-
 *-----------------------------------------------------------------------
 * bwfbMouseGetMotionEvents --
 *	Return the (number of) motion events in the "motion history
 *	buffer" (snicker) between the given times.
 *
 * Results:
 *	The number of events stuffed.
 *
 * Side Effects:
 *	The relevant xTimecoord's are stuffed in the passed memory.
 *
 *-----------------------------------------------------------------------
 */
static int
bwfbMouseGetMotionEvents (buff, start, stop)
    CARD32 start, stop;
    xTimecoord *buff;
{
    return 0;
}

/*-
 *-----------------------------------------------------------------------
 * bwfbMouseGetEvents --
 *	Return the events waiting in the wings for the given mouse.
 *
 * Results:
 *	A pointer to an array of vsEvents or (vsEvent *)0 if no events
 *	The number of events contained in the array.
 *
 * Side Effects:
 *	None.
 *-----------------------------------------------------------------------
 */
static vsEvent *
bwfbMouseGetEvents (pMouse, pNumEvents)
    DevicePtr	  pMouse;	    /* Mouse to read */
    int	    	  *pNumEvents;	    /* Place to return number of events */
{
    int	    	  nBytes;	    /* number of bytes of events available. */
    register PtrPrivPtr	  pPriv;
    static vsEvent	evBuf[MAXEVENTS];   /* Buffer for vsEvents */

    pPriv = (PtrPrivPtr) pMouse->devicePrivate;

    nBytes = read (pPriv->fd, evBuf, sizeof(evBuf));

    if (nBytes < 0) {
	if (errno == EWOULDBLOCK) {
	    *pNumEvents = 0;
	} else {
	    Error ("Reading mouse");
	    FatalError ("Could not read from mouse");
	}
    } else {
	*pNumEvents = nBytes / sizeof (vsEvent);
    }
    return (evBuf);
}

/*-
 *-----------------------------------------------------------------------
 * MouseAccelerate --
 *	Given a delta and a mouse, return the acceleration of the delta.
 *
 * Results:
 *	The corrected delta
 *
 * Side Effects:
 *	None.
 *
 *-----------------------------------------------------------------------
 */
static short
MouseAccelerate (pMouse, delta)
    DevicePtr	  pMouse;
    int	    	  delta;
{
    register int  sgn = sign(delta);
    register PtrCtrl *pCtrl;

    delta = abs(delta);
    pCtrl = &((DeviceIntPtr) pMouse)->u.ptr.ctrl;

    if (delta > pCtrl->threshold) {
	return ((short) (sgn * (pCtrl->threshold +
				((delta - pCtrl->threshold) * pCtrl->num) /
				pCtrl->den)));
    } else {
	return ((short) (sgn * delta));
    }
}
/*-
 *-----------------------------------------------------------------------
 * bwfbMouseProcessEvent --
 *	Given a vsEvent for a mouse, pass it off the the dix layer
 *	properly converted...
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	The cursor may be redrawn...? devPrivate/x/y will be altered.
 *
 *-----------------------------------------------------------------------
 */
static void
bwfbMouseProcessEvent (pMouse, fe)
    DevicePtr	  pMouse;   	/* Mouse from which the event came */
    vsEvent	  *fe;	    	/* Event to process */
{
    xEvent		xE;
    register PtrPrivPtr	pPriv;	/* Private data for pointer */
    register BwfbMsPrivPtr pBwfbPriv; /* Private data for mouse */
    register int  	bmask;	/* Temporary button mask */

    pPriv = (PtrPrivPtr)pMouse->devicePrivate;
    pBwfbPriv = (BwfbMsPrivPtr) pPriv->devPrivate;

    xE.u.keyButtonPointer.time = TVTOMILLI(fe->time);

	if ((fe->vse_type == VSE_BUTTON) &&
	    (fe->vse_device == VSE_DKB)) {					
		/* better be a button */
	    xE.u.u.detail = fe->vse_key;
		xE.u.u.type = ((fe->vse_direction == VSE_KBTUP) ? KeyRelease : KeyPress);
#ifdef TODO
	    switch (fe->vse_direction)
	    {
		case VSE_KBTDOWN: 
		    xE.u.u.type = KeyPress;
		    (qvKeyboard->processInputProc)(&xE, qvKeyboard);
		    break;
		case VSE_KBTUP: 
		    xE.u.u.type = KeyRelease;
		    (qvKeyboard->processInputProc)(&xE, qvKeyboard);
		    break;
		default: 	       /* hopefully BUTTON_RAW_TYPE */
		    ProcessLK201Input(&xE, qvKeyboard);
	    }
#endif TODO
	} else {
	    if (fe->vse_type == VSE_BUTTON) {
			if (fe->vse_direction == VSE_KBTDOWN)
				xE.u.u.type = ButtonPress;
			else
				xE.u.u.type = ButtonRelease;
			/* mouse buttons numbered from one */
			xE.u.u.detail = fe->vse_key + 1;
			/*
			 * If the mouse has moved, we must update any interested client
			 * as well as DIX before sending a button event along.
			 */
			if (pBwfbPriv->mouseMoved) {
				bwfbMouseDoneEvents (pMouse, FALSE);
			}
	    } else {
			xE.u.u.type = MotionNotify;
			/*
			 * When we detect a change in the mouse coordinates, we call
			 * the cursor module to move the cursor. It has the option of
			 * simply removing the cursor or just shifting it a bit.
			 * If it is removed, DIX will restore it before we goes to sleep...
			 *
			 * What should be done if it goes off the screen? Move to another
			 * screen? For now, we just force the pointer to stay on the
			 * screen...
			 */

			pPriv->x += MouseAccelerate (pMouse, fe->vse_x);
			/*
			 * For some reason, motion up generates a positive y delta
			 * and motion down a negative delta, so we must subtract
			 * here instead of add...
			 */
			pPriv->y -= MouseAccelerate (pMouse, fe->vse_y);

			if (!bwfbConstrainXY (&pPriv->x, &pPriv->y)) 
				return;
			((BwfbMsPrivPtr)pPriv->devPrivate)->mouseMoved = TRUE;
			return;
		}
	xE.u.keyButtonPointer.rootX = pPriv->x;
	xE.u.keyButtonPointer.rootY = pPriv->y;
	(* pMouse->processInputProc) (&xE, pMouse);
	}
}

/*-
 *-----------------------------------------------------------------------
 * bwfbMouseDoneEvents --
 *	Finish off any mouse motions we haven't done yet. (At the moment
 *	this code is unused since we never save mouse motions as I'm
 *	unsure of the effect of getting a keystroke at a given [x,y] w/o
 *	having gotten a motion event to that [x,y])
 *
 * Results:
 *	None.
 *
 * Side Effects:
 *	A MotionNotify event may be generated.
 *
 *-----------------------------------------------------------------------
 */
/*ARGSUSED*/
void
bwfbMouseDoneEvents (pMouse,final)
    DevicePtr	  pMouse;
    Bool	  final;
{
    PtrPrivPtr	  pPriv;
    BwfbMsPrivPtr  pBwfbPriv;
    xEvent	  xE;

    pPriv = (PtrPrivPtr) pMouse->devicePrivate;
    pBwfbPriv = (BwfbMsPrivPtr) pPriv->devPrivate;

    if (pBwfbPriv->mouseMoved) {
	bwfbMoveCursor (pPriv->pScreen, pPriv->x, pPriv->y);
	xE.u.keyButtonPointer.rootX = pPriv->x;
	xE.u.keyButtonPointer.rootY = pPriv->y;
	xE.u.keyButtonPointer.time = lastEventTime;
	xE.u.u.type = MotionNotify;
	(* pMouse->processInputProc) (&xE, pMouse);
	pBwfbPriv->mouseMoved = FALSE;
    }
}
