#include "../h/ashelp.h"
#include "../h/ucode.h"

/******************************************************************************
* SETUP
*
* This routine initializes the registers to the values needed by the raster
* operation routines.  On entry, the registers should contain:
*
*	D0 (W):		source rectangle X coordinate
*	D1 (W):		source rectangle Y coordinate
*	D2 (W):		destination rectangle X coordinate
*	D3 (W):		destination rectangle Y coordinate
*	D4 (W):		rectangle width
*	D5 (W):		rectangle height
*
*	A0 (W):		source raster width
*	A1 (W):		destination raster width
*	A2 (W):		source raster address (0 if no source)
*	A3 (W):		destination raster address
*
* On exit, the registers contain:
*
*	D0 (B):		?
*	D1 (W):		alignment rotation
*	D2 (W):		initial mask
*	D3 (W):		final mask
*	D4 (W):		adjusted width in words
*	D5 (W):		adjusted height in lines
*
*	A0 (W):		source vertical increment
*	A1 (W):		destination vertical increment
*	A2 (L):		source address
*	A3 (L):		destination address
*	A4 (W):		horizontal increment
*
* Also, on exit, the N condition code is set if the height of the rectangle
* is zero.
*
* This routine uses all registers except D7, A5, and A6.  Callers with values
* to preserve should use these registers.
*
******************************************************************************/

	.globl	setup
	.text
setup:
/*
 * Compensate for non-positive width or height by updating the X or Y
 * coordinates.
 */
	tstw	d4			/* test the width */
	bgt	1f			/* skip adjust if already positive */
	beq	zilch			/* return with N bit if width zero */

	addw	d4,d0			/* compute X before source left edge */
	addqw	#1,d0			/* X of source left edge */
	addw	d4,d2			/* compute X before destination left edge */
	addqw	#1,d2			/* X of destination left edge */
	negw	d4			/* make width positive */

    1:	tstw	d5			/* test the height */
	bgt	3f			/* skip adjust if already positive */
	beq	zilch			/* return with N bit if height zero */

	addw	d5,d1			/* compute Y before source top */
	addqw	#1,d1			/* Y of source top */
	addw	d5,d3			/* compute Y before destination top */
	addqw	#1,d3			/* Y of destination top */
	negw	d5			/* make height positive */

/*
 * Determine whether addresses should be incremented or decremented.  If the
 * source and destination overlap and the source precedes the destination,
 * addresses must be decremented to avoid overwriting portions of the source
 * before they are copied.  In all other cases, addresses are incremented.
 *
 * For aesthetic reasons, addresses are incremented whenever possible.  The
 * same raster and overlap tests can be eliminated if this is not desired.
 */
    3:	cmpl	a2,a3			/* same raster? */
	bne	isetup			/* increment addresses if not */

	cmpw	d1,d3			/* is destination before source? */
	blt	isetup			/* increment addresses if yes */
	bgt	4f			/* test for overlap addresses if no */
	cmpw	d0,d2			/* check x coordinates if maybe */
	blt	isetup			/* increment if destination is first */

	movw	d0,d6			/* get source x coordinate */
	addw	d4,d6			/* compute first x after source */
	cmpw	d2,d6			/* compare with left side of destination */
	bgt	dsetup			/* decrement if overlap */
	bra	isetup			/* otherwise increment */

    4:	movw	d1,d6			/* get source y coordinate */
	addw	d5,d6			/* compute first y after source */
	cmpw	d3,d6			/* compare with top of destination */
	bgt	dsetup			/* decrement if overlap */

/*
 * Compute source address and store in A2.  This is the address of the word
 * before the word that contains the source bit corresponding to the low order
 * bit of the left partial destination word.
 */
isetup:	movw	a0,d6			/* get source raster width */
	mulu	d6,d1			/* compute offset of row */
	addl	d1,a2			/* address of row */
	movw	d2,d1			/* get destination x */
	notw	d1			/*  */
	andw	#0xF,d1			/* size of left partial minus 1 */
	addw	d0,d1			/* position of last partial source bit */
	asrw	#4,d1			/* word offset */
	subqw	#1,d1			/* point to previous word */
	aslw	#1,d1			/* byte offset */
	addw	d1,a2			/* address of source */

/*
 * Compute destination address and store in A3.  The destination address is the
 * address of the word containing the first destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	mulu	d6,d3			/* compute offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* copy destination x */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of first word */

/*
 * Compute X of last destination bit and store in D3.  This is simply the
 * X of the first bit plus the width minus 1.
 */
	movw	d2,d3			/* copy destination x */
	addw	d4,d3			/* add width */
	subqw	#1,d3			/* and adjust */

/*
 * Compute the amount of right shift necessary to align the source with the
 * destination.  This is the difference of their X's taken modulo 16.
 */
	movw	d2,d1			/* get destination x */
	subw	d0,d1			/* compute difference */
	andw	#0xF,d1			/* take result modulo 16 */
	
/*
 * Compute width in words of middle section and store in D4.  This is the
 * difference between the first and last addresses minus 1.
 */
	movw	d3,d4			/* get x of last point */
	asrw	#4,d4			/* find its word address */
	movw	d2,d6			/* get x of first point */
	asrw	#4,d6			/* find its word address */
	subw	d6,d4			/* compute difference */
	subqw	#1,d4			/* and adjust */

/*
 * Compute the left mask and store in D2.  This mask is right justified and
 * contains one bit for each bit of the destination in the first word.
 */
	movw	#0xFFFF,d6		/* start with all ones */
	andw	#0xF,d2			/* take initial x modulo word width */
	lsrw	d2,d6			/* clear appropriate number of bits */
	movw	d6,d2			/* put the result back in d2 */

/*
 * Compute the right mask and store in D3.  This mask is left justified and
 * contains one bit for each bit of the destination in the final word.
 */
	andw	#0xF,d3			/* take final x modulo word width */
	movw	#15,d6			/* get index of high order bit */
	subw	d3,d6			/* compute bits to right of final bit */
	movw	#0xFFFF,d3		/* start with all ones */
	lslw	d6,d3			/* clear appropriate number of bits */

/*
 * Adjust for the case where the entire width is within a single word.  In
 * this case, use just the left mask.
 */
	tstw	d4			/* single word (width negative)? */
	bge	1f			/* branch if not */
	andw	d3,d2			/* otherwise combine masks */
	clrw	d3			/* zero right mask */
	clrw	d4			/* zero width */

/*
 * Compute source and destination row increments and store in A0 and A1.  These
 * are the raster widths minus the width of the left and middle sections.
 */
    1:	subw	d4,a0			/* adjust source raster width */
	subw	d4,a0			/* do it again for byte addresses */
	subqw	#2,a0			/* account for left word */

	subw	d4,a1			/* adjust destination raster width */
	subw	d4,a1			/* do it again for byte addresses */
	subqw	#2,a1			/* account for left word */

	movw	#2,a4			/* initialize horizontal increment */

	bra	adjust			/* adjust width and height and return */

/*
 * Compute source address and store in A1.  This is the address of the source
 * word containing the bit corresponding to the last bit in the next to last
 * destination word.
 */
dsetup:	movw	a0,d6			/* get source raster width */
	addw	d5,d1			/* add height to y */
	subqw	#1,d1			/* compute y of last row */
	mulu	d6,d1			/* compute offset of row */
	addl	d1,a2			/* address of row */
	movw	d2,d1			/* get destination x */
	addw	d4,d1			/* add width to it */
	subqw	#1,d1			/* compute x of last bit */
	andw	#0xFFF0,d1		/* compute x of last word */
	subqw	#1,d1			/* x of bit before */
	subw	d2,d1			/* offset from beginning of destination */
	addw	d0,d1			/* corresponding source bit */
	asrw	#4,d1			/* word offset from beginning of row */
	aslw	#1,d1			/* byte offset */
	addw	d1,a2			/* address of source */

/*
 * Compute destination address and store in A3.  The destination address is the
 * address of the word containing the last destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	addw	d5,d3			/* add height to y */
	subqw	#1,d3			/* compute y of last row */
	mulu	d6,d3			/* compute offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* copy destination x */
	addw	d4,d6			/* add width to x */
	subqw	#1,d6			/* compute x of last bit */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of destination */

/*
 * Compute X of last destination bit and store in D2.  This is simply the
 * X of the first bit plus the width minus 1.  Store the X of the first bit
 * in D3.
 */
	movw	d2,d3			/* copy destination x */
	addw	d4,d2			/* add width */
	subqw	#1,d2			/* and adjust */

/*
 * Compute the amount of right shift necessary to align the source with the
 * destination.  This is the difference of their X's taken modulo 16.
 */
	movw	d3,d1			/* get destination x */
	subw	d0,d1			/* compute difference */
	andw	#0xF,d1			/* take result modulo 16 */

/*
 * Compute width in words of middle section and store in D4.  This is the
 * difference between the first and last addresses minus 1.
 */
	movw	d2,d4			/* get x of last point */
	asrw	#4,d4			/* find its word address */
	movw	d3,d6			/* get x of first point */
	asrw	#4,d6			/* find its word address */
	subw	d6,d4			/* compute difference */
	subqw	#1,d4			/* and adjust */

/*
 * Compute the left mask and store in D3.  This mask is right justified and
 * contains one bit for each bit of the destination in the first word.
 */
	movw	#0xFFFF,d6		/* start with all ones */
	andw	#0xF,d3			/* take initial x modulo word width */
	lsrw	d3,d6			/* clear appropriate number of bits */
	movw	d6,d3			/* put the result back in d2 */

/*
 * Compute the right mask and store in D2.  This mask is left justified and
 * contains one bit for each bit of the destination in the final word.
 */
	andw	#0xF,d2			/* take final x modulo word width */
	movw	#15,d6			/* get index of high order bit */
	subw	d2,d6			/* compute bits to right of final bit */
	movw	#0xFFFF,d2		/* start with all ones */
	lslw	d6,d2			/* clear appropriate number of bits */

/*
 * Adjust for the case where the entire width is within a single word.  In
 * this case, use just the left mask.
 */
	tstw	d4			/* single word (width negative)? */
	bge	1f			/* branch if not */
	andw	d3,d2			/* otherwise combine masks */
	clrw	d3			/* zero right mask */
	clrw	d4			/* zero width */

/*
 * Compute source and destination row increments and store in A0 and A1.  These
 * are the raster widths minus the width of the left and middle sections.
 */
    1:	subw	d4,a0			/* adjust source raster width */
	subw	d4,a0			/* do it again for byte addresses */
	subqw	#2,a0			/* account for left word */
	movw	a0,d6			/* copy for negation */
	negw	d6			/* negate for going backward */
	movw	d6,a0			/* copy back */
	subw	d4,a1			/* adjust destination raster width */
	subw	d4,a1			/* do it again for byte addresses */
	subqw	#2,a1			/* account for left word */
	movw	a1,d6			/* copy for negation */
	negw	d6			/* negate for going backward */
	movw	d6,a1			/* copy back */
	movw	#-2,a4			/* initialize horizontal increment */

/*
 * Adjust width and height.  The raster routines use the dbf instruction in
 * their loops.  Since this instruction (inconveniently) loops until the
 * specified register reached -1, the counts must be adjusted.
 */
adjust:	subqw	#1,d4			/* adjust width */
	subqw	#1,d5			/* adjust height (and set N bit) */
	rts				/* return */

zilch:	moveq	#-1,d1			/* set N bit */
	rts				/* return */
