/*
 *	These are the various routines used to manipulate the display area of
 *	a window.  These include setting the window parameters, painting
 *	characters and strings.
 */

#include		<vt_hdrs.h>	    /* system wide defines */
#include		<rectangle.h>	    /* rectangle structure */
#include		<tile.h>	    /* tile structure */
#include		<cursor.h>	    /* cursor structure */
#include		<window.h>	    /* window structure */
#include 		<vt_fonts.h>	    /* font mgmt structures */

/*
 *	Routine to set up default attributes for the given window
 */
ResetAttributes(window)
register struct window *window;
{
    /* set current position */
    window->attributes.xposition = 0;
    window->attributes.yposition = 0;
    /* set color and font */
    window->attributes.color = BLACK;
    window->attributes.bcolor = WHITE;
    window->attributes.pattern = solid;
    window->attributes.bpattern = solid;
    window->attributes.font = DEFPROPFONT;
    window->attributes.ctype = 0;
    /* set string justification, addressing mode and phase */
    window->attributes.justification = FLUSHLEFT;
    window->attributes.xphase = 0;
    window->attributes.yphase = 0;
    window->attributes.addressing = RELATIVE;
    window->attributes.orientation = LEFT;
    window->attributes.direction = LEFT;
    window->attributes.count = 0;
    window->attributes.mask = 0;
    window->attributes.thickness = 2;
}

/*	Set Window Attribute Calls */

/*
 *	Routine to set the position to x,y in the given window
 */
SetPosition(window, x, y)
register struct window *window;
word x, y;
{
    window->attributes.xposition = x;
    window->attributes.yposition = y;
}

/*
 *	Routine to set a new x position in the given window
 */
SetXPosition(window, x)
register struct window *window;
word x;
{
    window->attributes.xposition = x;
}

/*
 *	Routine to set a new y position in the given window
 */
SetYPosition(window, y)
register struct window *window;
word y;
{
    window->attributes.yposition = y;
}

/*
 *	Routine to increment the current position in the given window by the
 *	given amounts
 */
BumpPosition(window, dx, dy)
register struct window *window;
word dx, dy;
{
    window->attributes.xposition += dx;
    window->attributes.yposition += dy;
}

/*
 *	Routine to increment the y postion in the given window by the
 *	given amount
 */
BumpXPosition(window, dx)
register struct window *window;
word dx;
{
    window->attributes.xposition += dx;
}

/*
 *	Routine to increment the x position in the given window by the given
 *	amount
 */
BumpYPosition(window, dy)
register struct window *window;
word dy;
{
    window->attributes.yposition += dy;
}

/*
 *	Routine to set a new addressing mode in the given window
 */
SetAddressing(window, addressing)
register struct window *window;
word addressing;
{
    window->attributes.addressing = addressing;
}

/*
 *	Routine to set a new foreground color in the given window
 */
SetColor(window, color)
register struct window *window;
COLORT color;
{
    window->attributes.color = color;
}

/*
 *	Routine to set a new background color in the given window
 */
SetBColor(window, color)
register struct window *window;
COLORT color;
{
    window->attributes.bcolor = color;
}

/*
 *	Routine to set a new foreground pattern in the given window
 */
SetPattern(window, pattern)
register struct window *window;
PATTERN pattern;
{
    window->attributes.pattern = pattern;
}

/*
 *	Routine to set a new background pattern in the given window
 */
SetBPattern(window, pattern)
register struct window *window;
PATTERN pattern;
{
    window->attributes.bpattern = pattern;
}

/*
 *	Routine to set a new phase in the given window
 */
SetPhase(window, xphase, yphase)
register struct window *window;
word xphase, yphase;
{
    window->attributes.xphase = xphase;
    window->attributes.yphase = yphase;
}

/*
 *	Set a new font in the given window
 */
SetFont(window, font)
register struct window *window;
register unsigned short	font;
{
    window->attributes.font = font;
}

/*
 *	Sets the current character attribute flags
 */
SetCharacterAttributes(window, ctype)
register struct window *window;
word ctype;
{
    window->attributes.ctype = ctype;
}

/*
 *	Set a new justification mode in the given window
 */
SetJustification(window, justification)
register struct window *window;
word justification;
{
    window->attributes.justification = justification;
}

/*
 *	Set a new thickness in the given window
 */
SetThickness(window, thickness)
register struct window *window;
word thickness;
{
    window->attributes.thickness = thickness;
}

/*
 *	Set a new orientation in the given window
 */
SetOrientation(window, orientation)
register struct window *window;
word orientation;
{
    window->attributes.orientation = orientation;
}

/*
 *	Set a new direction in the given window
 */
SetDirection(window, direction)
register struct window *window;
word direction;
{
    window->attributes.direction = direction;
}

/*
 *	Set line style in given window 
 */
SetLineStyle(window, mask, count)
register struct window * window;
word mask;
word count;
{
    window->attributes.count = count;
    window->attributes.mask = mask;
}

/* 	Get Window Attribute Calls */

/*
 *	Get visiblity of local cursor
 */
bool
GetCursorVisibility(window)
struct window * window;
{
    return((bool) window->cursor->visible);
}

/*
 *	Routine to get current window colors
 */
GetCurrentPatterns(window, fpat, bpat)
register struct window *window;
PATTERN *fpat;
PATTERN *bpat;
{
    *fpat = window->attributes.pattern;
    *bpat = window->attributes.bpattern;
}

/*
 *	Routine to get current window colors
 */
GetCurrentColors(window, fcolor, bcolor)
register struct window *window;
COLORT *fcolor;
COLORT *bcolor;
{
    *fcolor = window->attributes.color;
    *bcolor = window->attributes.bcolor;
}

/*
 *	Routine to get current window font
 */
GetCurrentFont(window, font)
register struct window *window;
unsigned short *font;
{
    *font = window->attributes.font;
}

/*
 * 	Routine to get the current windows x and y position 
 */
GetCurrentPosition(window, x, y)
register struct window * window;
short *x, *y;
{
	*x = window->attributes.xposition;
	*y = window->attributes.yposition;
}

/*
 *	Gets the current addressing mode 
 */
GetCurrentAddressing(window, addressing)
register struct window *window;
register word *addressing;
{
    *addressing = window->attributes.addressing;
}

/*
 *	Gets the current string justification 
 */
GetStringAttributes(window, justification, orientation, direction, ctype)
register struct window *window;
word *justification;
word *orientation;
word *direction;
word *ctype;
{
    *justification = window->attributes.justification;
    *orientation = window->attributes.orientation;
    *direction = window->attributes.direction;
    *ctype = window->attributes.ctype;
}

/*
 *	Gets the current line phase 
 */
GetCurrentPhase(window, xphase, yphase)
register struct window *window;
word *xphase;
word *yphase;
{
    *xphase = window->attributes.xphase;
    *yphase = window->attributes.yphase;
}

/*
 *	Gets the current line attributes 
 */
GetLineAttributes(window, thickness, mask, count)
register struct window *window;
word *thickness;
word *mask;
word *count;
{
    *thickness = window->attributes.thickness;
    *mask = window->attributes.mask;
    *count = window->attributes.count;
}

/*
 *	Routine to paint the string s in the given window at the current
 *	position
 *	If size is -2 then the current position is updated.
 *	If size  is -1 then the position is not updated.
 */
PaintString(window, size, s)
register struct window *window;
register char *s;
{
    register int 		x, y, width;
    int 			n;
    COLORT			fc, bc;
    PATTERN			pat;
    register char 		*sp;
    register struct tile 	*tile, *tiles;
    register struct rectangle 	*bp;
    struct rectangle 	cb, clipbounds;	    /* bounds for cursor removeal */
    struct font_plane		*charinfo;

    /* point bp to a recangle */
    bp = &clipbounds;

    /* get the screen coordinates for the current position */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    charinfo = GetFontPlane(window->attributes.font, DEFPLANE);
    /* calculate the length of the string */
    for (width = 0, sp = s, n = 0; *sp != 0 && n != size; sp++, n++)
	width += CharacterWidth( charinfo, *sp);

    /* recompute position using justification mode */
    if (window->attributes.justification != FLUSHLEFT)
	x -= (window->attributes.justification == CENTER) ? width/2 : width;

    /* remove any cursors in the given area of the window */
    bp->x = x;
    bp->w = width;
    bp->h = CharacterHeight( charinfo);
    bp->y = y - CharacterBaseline( charinfo);
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* set colors with respect to reverse video attribute */
    fc = ((window->attributes.ctype & RVIDEO) ? window->attributes.bcolor :
	  window->attributes.color);
    bc = ((window->attributes.ctype & RVIDEO) ? window->attributes.color :
	  window->attributes.bcolor);
    pat = ((window->attributes.ctype & RVIDEO) ? window->attributes.pattern :
	  window->attributes.bpattern);

    /* set n to size or -1 for DrawString below */
    n = ((size == -2) ? -1 : size);
    /* loop through all display tiles of window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (!Overlap(*bp, tile->bounds)) continue;
	Intersection(&cb, *bp, tile->bounds);
	LiftCursors(window,&cb);

	/* clip to that tile */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);

	/* does background need to be cleared ? */
	if (window->attributes.ctype & CLEARBACK) 
	    DrawRectangleInterior(bp->x, bp->y, bp->w, bp->h, bc, pat, 
			      window->attributes.xphase+window->ibounds.x,
			      window->attributes.yphase+window->ibounds.y);

	/* draw string on screen */
	DrawString(x, y, charinfo, fc, n, s);

	/* is this a bold string ? */
	if (window->attributes.ctype & BOLD) {
	    DrawString(x + 1, y + 1, charinfo, fc, n, s);
	}

	/* does and underline need to be ? */
	if (window->attributes.ctype & ULINE) 
	    DrawRectangleInterior(x, y + 2, bp->w, 1, fc, 0,0);
	UnsetClipping();
	DropCursors(window, &cb);
    }
    DoneWithPlane(window->attributes.font, DEFPLANE);
    /* set position if needed */
    if (size == -2) window->attributes.xposition = 
		(x - window->ibounds.x + width);
}

/*
 *	Routine to paint a rectangle interior of the given size in the
 *	given window of the current color
 */
PaintRectangleInterior(window, xx, yy)
register struct window *window;
register int xx, yy;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen coordinates of current position */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* adjust xx,yy to be relative to current position */
    if (window->attributes.addressing == ABSOLUTE) {
	xx -= window->attributes.xposition-1;
	yy -= window->attributes.yposition-1;
    }

    /* turn negative values positive and adjust x,y */
    if (xx < 0) {
	xx = -xx;
	x -= xx;
    }

    if (yy < 0) {
	yy = -yy;
	y -= yy;
    }

    /* find bounds for this rectangle and remove any cursors */
    bp->x = x;
    bp->y = y;
    bp->w = xx + 1;
    bp->h = yy + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles for window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping to this tile and draw part of rectangle */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);
	DrawRectangleInterior(x, y, xx, yy, window->attributes.color,
			      window->attributes.pattern,
			      window->attributes.xphase+window->ibounds.x,
			      window->attributes.yphase+window->ibounds.y);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}

/*
 *	Routine to paint a rectangle border in the given window of the given
 *	size at the current position of the current thickness
 */
PaintRectangleBorder(window, xx, yy)
register struct window *window;
register int xx, yy;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen coordinates of current position in window */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* adjust given points to be relative */
    if (window->attributes.addressing == ABSOLUTE) {
	xx -= window->attributes.xposition-1;
	yy -= window->attributes.yposition-1;
    }

    /* turn negative values positive and set current screen position */
    if (xx < 0) {
	xx = -xx;
	x -= xx;
    }

    if (yy < 0) {
	yy = -yy;
	y -= yy;
    }

    /* find bounds for this rectangle and remove any cursors */
    bp->x = x - window->attributes.thickness;
    bp->y = y - window->attributes.thickness;
    bp->w = xx + (window->attributes.thickness << 1) + 1;
    bp->h = yy + (window->attributes.thickness << 1) + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop thru all display tiles of window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* clip to this tile and draw that part of border in tile */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);
	DrawRectangleBorder(x, y, xx, yy, window->attributes.thickness,
			    window->attributes.color, 
			    window->attributes.pattern,
			    window->attributes.xphase+window->ibounds.x,
			    window->attributes.yphase+window->ibounds.y);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}

/*
 *	Routine to call one of the user rasterops, for use in a window
 */
DoRasterOp(window, op, uraster, ux, uy, x, y, w, h, color)
register struct window *window;
int	(*op)();
RASTER	*uraster;
short	ux, uy, x, y, w, h;
int	color;
{
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    struct rectangle clipbounds;	    /* bounds for cursor removal */
    struct rectangle tbounds;		    /* bounds for tiles */
    extern int	ToggleURaster(), CopyURaster(), PaintURaster();
    int	(*dummy)();

    /* force ld to link in the URaster routines */
    dummy = ToggleURaster;
    dummy = CopyURaster;
    dummy = PaintURaster;

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* find bounds for the destination rectangle and remove any cursors */
    bp->x = x + window->ibounds.x;
    bp->y = y + window->ibounds.y;
    bp->w = w;
    bp->h = h;

    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles for window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (Overlap(*bp, tile->bounds)){
	    Intersection(&tbounds, *bp, tile->bounds);
	    LiftCursors(window,&tbounds);

	    /* do the rasterop */
	    (*op)(uraster, ux + (tbounds.x - bp->x), uy + (tbounds.y - bp->y),
		  &screen, tbounds.x, tbounds.y, tbounds.w, tbounds.h, color);

	    DropCursors(window,&tbounds);
	}
    }
}

/*
 *	Routine to paint a line from the current position, to the given
 *	position in the given window of the current thickness
 */
PaintLine(window, xx, yy)
register struct window *window;
register int xx, yy;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* make sure this is not zero thick line */
    if (window->attributes.thickness <= 0) return;

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen coordinates of current position */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* change points for relative addressing */
    if (window->attributes.addressing == ABSOLUTE) {
	xx -= window->attributes.xposition;
	yy -= window->attributes.yposition;
    }

    /* find bounds for this line and remove any cursors-make sure w,h > 0 */
    if (xx > 0) {
        bp->x = x;
        bp->w = xx;
    } else if (xx < 0) {
	bp->x = x + xx;
	bp->w = -xx;
    } else {
	bp->x = x - window->attributes.thickness;
	bp->w = window->attributes.thickness << 1;
    }
    if (yy > 0) {
        bp->y = y;
        bp->h = yy;
    } else if (yy < 0) {
	bp->y = y + yy;
	bp->h = -yy;
    } else {
	bp->y = y - window->attributes.thickness;
	bp->h = window->attributes.thickness << 1;
    }
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles in window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping to this tile and draw that part of vector */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);
#ifdef notdef
printf("DrawVector(0x%x 0x%x 0x%x 0x%x 0x%x 0x%x\n",
x, y, xx, yy, window->attributes.thickness, window->attributes.color);
#endif notdef
	DrawVector(x, y, xx, yy, window->attributes.thickness,
	           window->attributes.color);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }

    /* set new position to end of vector */
    window->attributes.xposition += xx;
    window->attributes.yposition += yy;
}

/*
 *	Routine to paint a polygon interior in the given window of the given
 *	color
 */
PaintPolygonInterior(window, n, xx, yy)
register struct window *window;
register short n, *xx, *yy;	    /* array of vertices, n is number valid */
{
    register int i;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */
    short px, py;			    /* previous x, y for relative */

    short xv[256], yv[256];	    /* array for vertices in polygon */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* set initial bounds for rectangle here to find min/max x/y below */
    bp->x = 1024;
    bp->y = 1024;
    bp->w = 0;		/* use w and h to find max x and y respect */
    bp->h = 0;

    /* copy screen vertices into array with proper mode */
    if (window->attributes.addressing == RELATIVE) {
	/* get screen coordinates of current position */
	px = window->attributes.xposition + window->ibounds.x;
	py = window->attributes.yposition + window->ibounds.y;
	for (i = 0; i < n; i++) {
	    xv[i] = px + xx[i]; px = xv[i];
	    yv[i] = py + yy[i]; py = yv[i];
	    if (bp->x > xv[i]) bp->x = xv[i];
	    if (bp->y > yv[i]) bp->y = yv[i];
	    if (bp->w < xv[i]) bp->w = xv[i];
	    if (bp->h < yv[i]) bp->h = yv[i];
	}
    } else {
	for (i = 0; i < n; i++) {
	    xv[i] = xx[i]+window->ibounds.x;
	    yv[i] = yy[i]+window->ibounds.y;
	    if (bp->x > xv[i]) bp->x = xv[i];
	    if (bp->y > yv[i]) bp->y = yv[i];
	    if (bp->w < xv[i]) bp->w = xv[i];
	    if (bp->h < yv[i]) bp->h = yv[i];
	}
   }

   /* turn bounds into x,y,width,height and remove cursors */
   bp->w -= (bp->x - 1);
   bp->h -= (bp->y - 1);
   /* return if not visible */
   if (!Overlap(*bp, window->ibounds)) return;

   /* loop thru all display tiles in window */
   tiles = window->tiles;
   for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping to this tile and draw that part of polygon */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);
	DrawPolygonInterior(n, xv, yv, window->attributes.color,
			window->attributes.pattern,
			window->attributes.xphase, window->attributes.yphase);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
   }
}

/*
 *	Routine to paint the interior of a circle centered at the current
 *	position in the current color of radius r
 */
PaintCircleInterior(window, r)
register struct window *window;
register int r;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen position of center */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* set bounds for this circle and remove cursors */
    bp->x = x - r;
    bp->y = y - r;
    bp->w = bp->h = (r << 1) + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through display tiles for this window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping bounds to bounds of this tile */
	SetClipping(tile->bounds.x, tile->bounds.y,
		    tile->bounds.w, tile->bounds.h);
	/* draw an arc centered at x,y of radius 0, thickness r, from
	 * x+r,y to x+r,y on arc in color of current phase */
	DrawArc(x, y, 0, r, x+r, y, x+r, y,
	           window->attributes.color, window->attributes.pattern,
		   window->attributes.xphase, window->attributes.yphase);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}

/*
 *	Routine to paint the border of a circle centered at the current
 *	position of current thickness and color and radius r
 */
PaintCircleBorder(window, r)
register struct window *window;
register int r;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen coors for current position */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* set bounds for this circle and remove cursors */
    bp->x = x - r - window->attributes.thickness;
    bp->y = y - r - window->attributes.thickness;
    bp->w = bp->h = ((r + window->attributes.thickness) << 1) + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles for this window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping bounds to bounds of this tile */
	SetClipping(tile->bounds.x, tile->bounds.y,
		    tile->bounds.w, tile->bounds.h);
	/* draw an arc centered at x,y of radius r, current thickness , from
	 * x+r,y to x+r,y on arc in current color and phase */
	DrawArc(x, y, r, window->attributes.thickness, x+r, y, x+r, y,
	           window->attributes.color, window->attributes.pattern,
		   window->attributes.xphase, window->attributes.yphase);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}

/*
 *	Routine to paint the interior of an arc centered at the current
 *	position from xs,ys to xe,ye on the arc, of radius r, curremt color
 *	and phase
 */
PaintArcInterior(window, r, xs, ys, xe, ye)
register struct window *window;
int r, xs, ys, xe, ye;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* get screen coorordinates for center */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* adjust arc points with respect to addressing mode */
    if (window->attributes.addressing == ABSOLUTE) {
	xs -= window->attributes.xposition;
	ys -= window->attributes.yposition;
	xe -= window->attributes.xposition;
	ye -= window->attributes.yposition;
    }

    /* set bounds for whole circle and remove cursors */
    bp->x = x - r;
    bp->y = y - r;
    bp->w = bp->h = (r << 1) + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles for this window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping bounds to bounds of this tile */
	SetClipping(tile->bounds.x, tile->bounds.y,
		    tile->bounds.w, tile->bounds.h);
	/* draw an arc centered at x,y of radius 0, thickness r , from/to
	 * the given points on the arc in current color and phase */
	DrawArc(x, y, 0, r, x+xs, y+ys, x+xe, y+ye,
	           window->attributes.color, window->attributes.pattern,
		   window->attributes.xphase, window->attributes.yphase);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}


/*
 *	Routine to paint the border of an arc centered at the current
 *	position, of radius r from xs,ys to xe,ye on the arc in the current
 *	color and phase
 */
PaintArcBorder(window, r, xs, ys, xe, ye)
register struct window *window;
int r, xs, ys, xe, ye;
{
    register int x, y;
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* find screen coordinates for center */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* adjust arc points with respect to addressing mode */
    if (window->attributes.addressing == ABSOLUTE) {
	xs -= window->attributes.xposition;
	ys -= window->attributes.yposition;
	xe -= window->attributes.xposition;
	ye -= window->attributes.yposition;
    }

    /* set bounds for whole circle and remove cursors */
    bp->x = x - r - window->attributes.thickness;
    bp->y = y - r - window->attributes.thickness;
    bp->w = bp->h = ((r + window->attributes.thickness) << 1) + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop through all display tiles for this window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set clipping bounds to the bounds of this tile */
	SetClipping(tile->bounds.x, tile->bounds.y,
		    tile->bounds.w, tile->bounds.h);
	/* draw an arc centered at x,y of radius r, current thickness ,from/to
	 * given points on arc in current color and phase */
	DrawArc(x, y, r, window->attributes.thickness, x+xs, y+ys, x+xe, y+ye,
	           window->attributes.color, window->attributes.pattern,
		   window->attributes.xphase, window->attributes.yphase);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}


/*
 *	Routine to paint the given pattern in the given window starting at
 *	the current position with a width w and a height h in current color
 */
PaintIcon(window, icon, w, h)
register struct window *window;
register struct word *icon;
register w, h;
{
    register struct tile *tile, *tiles;
    register x,y;
    register struct rectangle *bp;
    bool	lifted;
    struct rectangle clipbounds;	    /* bounds for cursor removeal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* find screen coordinates for center */
    x = window->attributes.xposition + window->ibounds.x;
    y = window->attributes.yposition + window->ibounds.y;

    /* find bounds for this icon and lift any cursors */
    bp->x = x;
    bp->y = y;
    bp->w = w + 1;
    bp->h = h + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop thru display tiles for this window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (lifted = Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	}
	/* set cliping to this tile and paint that part of pattern */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);
	PaintArea(x, y, w, h, icon, window->attributes.color);
	UnsetClipping();
	if (lifted) DropCursors(window,&liftbounds);
    }
}

/*
 *	Routine to invert the rectangle in the given window starting at the
 *	given position x,y of width w and height h
 */
InvertRegion(window, x, y, w, h)
register struct window *window;
register int x, y, w, h;
{
    register struct tile *tile, *tiles;
    register struct rectangle *bp;
    struct rectangle clipbounds;	    /* bounds for cursor removal */
    struct rectangle liftbounds;	    /* bounds for cursor removal */

    /* point bp to a rectangle */
    bp = &clipbounds;

    /* turn x,y into screen coordinates */
    x += window->ibounds.x;
    y += window->ibounds.y;

    /* find bounds for this region and remove cursors */
    bp->x = x;
    bp->y = y;
    bp->w = w + 1;
    bp->h = h + 1;
    /* return if not visible */
    if (!Overlap(*bp, window->ibounds)) return;

    /* loop thru all display tiles for window */
    tiles = window->tiles;
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	if (Overlap(*bp, tile->bounds)){
	    Intersection(&liftbounds, *bp, tile->bounds);
	    LiftCursors(window,&liftbounds);
	    /* set clipping to this tile and paint invert area there */
	    SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
			tile->bounds.h);
	    InvertArea(x, y, w, h);
	    UnsetClipping();
	    DropCursors(window,&liftbounds);
	}
    }
}

/*
 *	Routine to paint the given string at the given x,y screen coordinates
 *	in the given font and color
 */
CenterString(x, y, font, color, string)
int x, y;
unsigned word font;
COLORT color;
char *string;
{
    char *sp;
    int width;
    struct font_plane	*charinfo;

    width = 0;
    sp = string;
    charinfo = GetFontPlane(font, DEFPLANE);
    /* find width of string */
    while (*sp != 0)
	width += CharacterWidth( charinfo, *sp++);
    /* adjust given position and draw it */
    DrawString(x-width/2, y, charinfo, color, -1, string);
    DoneWithPlane(font, DEFPLANE);
}

