/*
 * 	Routines used to draw a random arc in a window and on the screen
 *	clipped appropriately.  This algorithm, and the circle algorithm
 *	is based on a paper "A Linear Algorithm for Incremental Digital 
 *	Display of Circular Arcs" by Jack Bresenham of IBM.  This paper
 *	can be found in Communications of the ACM of Feb 1977, Vol 20 number
 *	2.  This code was originally wirtten by bob watkins, and modified 
 *	by jeff mason.
 */

#include <vt_hdrs.h>
#include <graphics.h>
#include <gvector.h>
#include <gcircle.h>
#include <rectangle.h>


/*
 *	Routine to draw a single scanline of an arc on the screen given a
 *	direction to draw (positive), a scanline (yline), a right and left point
 *	for the sides (lvec & rvec), a left and right interior point (lxi, rxi -
 *	these are for an arc of a given thickness ) a left and right 
 *	exterior point (lxo, rxo) and a color/phase 
 */
static AScanline(positive, yline, lvec, rvec, lxi, rxi, lxo, rxo, color, pat, xphase, yphase)
bool positive;
int yline;
register int lvec, rvec, lxi, rxi, lxo, rxo;
COLORT color;
PATTERN pat;
int xphase, yphase;
{
    int lx, rx;
    
    /* check if drawing left to right (TRUE) or right to left */
    if (positive) {

	/* check if drawing an arc of a given thickness - lxi != rxi */
	if (lxi == rxi) {
	    /* check if drawing from either edge thru circle */
	    if (rxo >= lvec && lxo <= rvec) {
		/* calculate end points and draw it */
		lx = (lxo < lvec) ? lvec : lxo;
		rx = (rxo > rvec) ? rvec : rxo;
		Scanline(lx, rx, yline, color, pat, xphase, yphase);
	    }
	} else {
	    /* check if left part of arc needs to be drawn */
	    if (lxo <= rvec && lxi >= lvec) {
		/* calculate end points and draw it */
		lx = (lxo < lvec) ? lvec : lxo;
		rx = (lxi < rvec) ? lxi : rvec;
		Scanline(lx, rx, yline, color, pat, xphase, yphase);
	    }
	    /* check if right part of arc needs to be drawn */
	    if (rxo >= lvec && rxi <= rvec) {
		/* calculate end points and draw it */
		lx = (rxi > lvec) ? rxi : lvec;
		rx = (rxo > rvec) ? rvec : rxo;
		Scanline(lx, rx, yline, color, pat, xphase, yphase);
	    }
	}
    /* drawing in right to left direction */
    } else {
	/* check if left exterior side is less than right vector */
	if (lxo <= rvec) {
	    /* check if circle (lxi == rxi) or arc */
	    if (lxi == rxi) { 
		/* calcualte right point and draw in line */
		rx = (rvec <= rxo) ? rvec : rxo;
		Scanline(lxo, rx, yline, color, pat, xphase, yphase);
	    } else {
		/* calcualte left point and draw in line */
		rx = (rvec <= lxi) ? rvec : lxi;
		Scanline(lxo, rx, yline, color, pat, xphase, yphase);
	        /* check if a right arc piece needs to be drawen */
		if (rxi <= rvec) {
		    /* calcualte right point and draw in line */
		    rx = (rvec <= rxo) ? rvec: rxo;
		    Scanline(rxi, rx, yline, color, pat, xphase, yphase);
		}
	    }
	}
	    
	/* check if right exterior is greater than left vector */
	if (rxo >= lvec) {
	    /* check if circle (lxi == rxi) or arc */
	    if (lxi == rxi) {
		/* calculate left side point and draw in line */
		lx = (lvec >= lxo) ? lvec : lxo;
		Scanline(lx, rxo, yline, color, pat, xphase, yphase);
	    } else {
		/* calculate left side point and draw in line */
		lx = (lvec >= rxi) ? lvec : rxi;
		Scanline(lx, rxo, yline, color, pat, xphase, yphase);
		/* check if right side needs to be done */
		if (lxi >= lvec) {
		    lx = (lvec >= lxo) ? lvec: lxo;
		    Scanline(lx, lxi, yline, color, pat, xphase, yphase);
		}
	    }
	}
    }
}



/*
 *	Routine to return TRUE if (xs,ys) is before (xe,ye) when travelling
 *	around a square center at (xc,yc) in a clockwise direction
 *	starting from the immediate right of the (xc,yc).  True is returned
 *	if this is the case, otherwise false is returned 
 */
bool AngleCompare(xc, yc, xs, ys, xe, ye)
int xc, yc, xs, ys, xe, ye;
{
    /* divide square into eight octants as follows
     *	  		5    6
     *		    4            7
     *		    3            0
     *			2    1 	
     *  use this array to map from number generated below to correct octant 
     */
    static octants[] = {4, 5, 3, 2, 7, 6, 0, 1};

    int os, oe;
    int nxs, nys, nxe, nye;
    int axs, ays, axe, aye;

     /* compute distance and length to start point from center point when
      * projected onto x/y axis */
    nxs = xs-xc;
    nys = ys-yc;
    axs = (nxs > 0) ? nxs : -nxs;
    ays = (nys > 0) ? nys : -nys;

     /* compute distance and length to end point from center point when
      * projected onto x/y axis */
    nxe = xe-xc;
    nye = ye-yc;
    axe = (nxe > 0) ? nxe : -nxe;
    aye = (nye > 0) ? nye : -nye;
    
    /* figure which octant each point lies in */
    os = octants[(nxs >= 0 ? 4 : 0) + (nys >= 0 ? 2 : 0) + (ays > axs ? 1 : 0)];
    oe = octants[(nxe >= 0 ? 4 : 0) + (nye >= 0 ? 2 : 0) + (aye > axe ? 1 : 0)];

    /* if different octants then return true if start is smaller octant */
    if (os != oe)
	return(os < oe);

    /* points are in same octant, compute which has smaller angle */
    if (nys > nxs)
	return((ays > axs) ? (nxs > nxe) : (nys > nye));
    else
	return((ays > axs) ? (nxs < nxe) : (nys < nye));
}

    



/*
 *	Routine used to draw in the needed portions of an arc in the upper
 *	half of the enclosing circle given a center (xc,yc), a radius r,
 * 	a border thickness t, a start point (xs,ys), an end point (xe,ye)
 *	a color and a pattern phase.
 */
DrawUpperSemiArc(xc, yc, r, t, xs, ys, xe, ye, color, pat, xphase, yphase)
int xc, yc, r, t, xs, ys, xe, ye;
COLORT color;
PATTERN pat;
int xphase, yphase;
{
    struct circle oc, ic;
    struct vector sv, ev;
    int y;
    register int lxi, rxi, lxo, rxo, lxr, rxr;
    bool positive;

    /* check which direction to draw in, TRUE = left to right */
    positive = AngleCompare(xc, yc, xs, ys, xe, ye) || (ye == yc && xe > xc);
    
    /* make a circle structure for the border (oc) and the interior (ic) */
    SetupCircle(&oc, lxo = rxo = xc, yc, r+t-1);
    SetupCircle(&ic, lxi = rxi = xc, yc, r);
    /* set up a vector structure from center to start (sv) and end (ev)points */
    SetupVector(&sv, lxr = xs, xc-xs, yc-ys);
    SetupVector(&ev, rxr = xe, xc-xe, yc-ye);

    /* loop thru all scan lines of circle starting from top */
    for (y = yc-r-t+1; y <= yc; y++) {

	/* find a the right x (rxo) and left x (lxo) points to draw to */ 
	rxo = GenerateUpperCircle(&oc);
	lxo = 2*xc-rxo;

	/* if we are drawing border then get right and left border points */
	if (y > yc-r) {
	    rxi = GenerateUpperCircle(&ic);
	    lxi = 2*xc-rxi;
	}

	/* if arc intersects left border line then get intersection point */
	if (y > ys || (xs > xc && y == ys))
	    lxr = GenerateVector(&sv);

	/* if arc intersects right border line then get intersection point */
	if (y > ye || (xe < xc && y == ye))
	    rxr = GenerateVector(&ev);

	/* put this line ono screen */
	AScanline(positive, y, lxr, rxr, lxi, rxi, lxo, rxo, 
						color, pat, xphase, yphase);
    }
}



/*
 *	Routine used to draw in the needed portions of an arc in the lower
 *	half of the enclosing circle given a center (xc,yc), a radius r,
 * 	a border thickness t, a start point (xs,ys), an end point (xe,ye)
 *	a color and a pattern phase.
 */
DrawLowerSemiArc(xc, yc, r, t, xs, ys, xe, ye, color, pat, xphase, yphase)
int xc, yc, r, t, xs, ys, xe, ye;
COLORT color;
PATTERN pat;
int xphase, yphase;
{
    struct circle oc, ic;
    struct vector sv, ev;
    int y;
    register int lxi, rxi, lxo, rxo, lxr, rxr;
    bool positive;

    positive = AngleCompare(xc, yc, xs, ys, xe, ye);

    /* make a circle structure for the border (oc) and the interior (ic) */
    SetupCircle(&oc, xc, yc, r+t-1);
    SetupCircle(&ic, xc, yc, r);
    /* set up a vector structure from center to start (sv) and end (ev)points */
    SetupVector(&sv, lxr = xc, xe-xc, ye-yc);
    SetupVector(&ev, rxr = xc, xs-xc, ys-yc);

    /* do first step of circle generation to set up */
    GenerateLowerCircle(&ic);

    /* loop thru all scan lines of circle starting from top */
    for (y = yc; y <= yc+r+t-1; y++)
    {
	/* find a the right x (rxo) and left x (lxo) points to draw to */ 
	rxo = GenerateLowerCircle(&oc);
	lxo = 2*xc-rxo;

	/* if we are drawing border then get right and left border points */
	if (y <= yc+r)
	{
	    rxi = GenerateLowerCircle(&ic);
	    lxi = 2*xc-rxi;
	}

	/* if arc intersects left border line then get intersection point */
	if (y > yc || (xs > xc && y == yc))
	    lxr = GenerateVector(&sv);

	/* if arc intersects right border line then get intersection point */
	if (y > yc || (xe < xc && y == yc))
	    rxr = GenerateVector(&ev);

	/* put this line ono screen */
	AScanline(positive, y, lxr, rxr, lxi, rxi, lxo, rxo, 
						color, pat, xphase, yphase);
    }
}



/*
 *	Routine to find point of intersection of a square centered around
 *	(xc,yc) of width r and a line from (xc,yc) to (xr,yr).  The found
 *	point is returned in as (xr,yr).  Note this routine will return the
 *	given point if (xc,yc) = (xr,yr).  This routine is called from
 *	DrawArc below to determine the correct starting and finishing points
 *	for the draw. 
 */
static RoundOut(xc, yc, r, xr, yr)
int xc, yc, r;			/* center point and side width */
int *xr, *yr;			/* pointer to given point */
{
    register int sdx, udx, sdy, udy, max;

    /* return same point if center equals point */
    if ((xc == *xr) && (yc == *yr)) return;

    /* find length and distance of projections of points on x axis */
    sdx = *xr-xc;
    udx = (sdx >= 0) ? sdx : -sdx;

    /* find length and distance of projections of points on y axis */
    sdy = *yr-yc;
    udy = (sdy >= 0) ? sdy : -sdy;

    /* find the greater of the two */
    max = (udx > udy) ? udx : udy;

    /* compute the new points which lie on the boundry of the square */
    *xr = xc + ((r+1)*sdx) / max;
    *yr = yc + ((r+1)*sdy) / max;
}




/*
 *	Actual routine to draw an arc on the screen clipped to the current
 * 	clipping bounds.
 */

DrawArc(xc, yc, r, t, xs, ys, xe, ye, color, pat, xphase, yphase)
int xc, yc, r;					/* coors of center and radius */
int t;						/* thickness */
int xs, ys, xe, ye;				/* start and end points */
int xphase, yphase;				/* phase for color */
COLORT color;
PATTERN pat;
{
    extern short cliplx, cliply, clipux, clipuy;
    
    /* exit if this is an empty circle */
    if (((xc == xs) && (yc == ys)) || ((xc == xe) && (yc == ye))) return;

    /* find points on containing square which lie on line between center
     * and the start/end points */
    RoundOut(xc, yc, r+t, &xs, &ys);
    RoundOut(xc, yc, r+t, &xe, &ye);
    
    /* check if figure lies in both upper and lower semi circles or is
     * wholly contained in one or both */
    if (AngleCompare(xc, yc, xs, ys, xe, ye) || (ye == yc && xe > xc)) {
	/* figure is wholly in upper, lower or both (circle) */
	/* check if upper region needs to be drawn */
	if ((ye < yc) || ((ye == yc) && (xe > xc)))
	    DrawUpperSemiArc(xc, yc, r, t,
			     (ys < yc) ? xs : xc-r-t, (ys < yc) ? ys : yc,
			     xe, ye,
			     color,pat,xphase,yphase);
	/* check if lower region needs to be drawn */
	if ((ys > yc) || ((ys == yc) && (xs > xc)))
	    DrawLowerSemiArc(xc, yc, r, t,
			     xs, ys,
			     (ye > yc) ? xe : xc-r-t, (ye > yc) ? ye : yc, 
			     color,pat,xphase,yphase);
    } else {
	/* figure lies in both upper and lower regions, paint both parts */
	DrawUpperSemiArc(xc, yc, r, t,
			 (ys <= yc) ? xs : xc-r-t, (ys < yc) ? ys : yc,
			 (ye <= yc) ? xe : xc+r+t, (ye < yc) ? ye : yc,
			 color,pat,xphase,yphase);
	DrawLowerSemiArc(xc, yc, r, t,
			 (ys >= yc) ? xs : xc+r+t, (ys > yc) ? ys : yc,
			 (ye >= yc) ? xe : xc-r-t, (ye > yc) ? ye : yc,
			 color,pat,xphase,yphase);
    }
}
