/*-----------------------------------------------------------------------------
 *	vt_term.c
 *
 *	terminal emulator w/ marking support
 *
 *	This emulator understands the same escape sequences as a vt100.  In
 *	addition several new escape sequences control the behavior of this
 *	emulator that is unique to working in a window environment.  See
 *	the document "Terminal Emulator User's Guide" for complete details.
 *
 *	This software is designed to be compiled into the kernel (if KERNEL is
 *	defined) or compiled to function standalone with the appropriate driver
 *	program.
 *
 *	Written by: H. David Scarbro
 *				Integrated Solutions, Inc.
 *
 *	Dates:
 *
 *	9 Nov 85	First version complete.
 *	6 Jan 86	Completed CR tracking, line style support and marking routines.
 * 30 Jan 86	Completed dynamic sizing and addition of escape feedback.
 *-----------------------------------------------------------------------------
 */

/*
 *	INCLUDES
 */

#ifdef KERNEL

#include "../machine/reg.h"
#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../vt/vt_hdrs.h"
#include "../vt/vt_output.h"
#include "../vt/vt_kernel.h"
#include "../vt/vt_key.h"
#include "../vt/vt_fonts.h"

#else

#define VT100_DEBUG

#include <vt.h> 

#define CreateBlock(x)			malloc(x)
#define DestroyBlock(x)			free(x)

#endif

#include <ctype.h>

/*
 *	DEFINES
 */

#define pad_charp(d,l,c)		((unsigned char *)pad_linep(d,l)+c)
#define min(a,b)				((a>b)?b:a)
#define max(a,b)				((a>b)?a:b)

#define DEF_PARMVAL		(-1)	/* initial parm_list entry value		*/
#define QUES_PARMVAL	(-2)	/* question mark maps to this			*/

#define NLINES80		24		/* lines on 80 character display		*/
#define NLINES132		14		/* lines on 132 character display		*/
#define NCOLS80			80		/* columns on 80 character display		*/
#define NCOLS132		132		/* columns on 132 character display		*/
#define MAX_NLINES		128		/* maximum number of lines allowed		*/
#define MAX_NCOLS		132		/* maximum number of columns allowed	*/

#define NSPECIALCOLS	2		/* column positions containing special	*/
								/* stuff.								*/

#define CR_COL 			(dp->cursor.ncols)
								/* position of first special column, it	*/
								/* contains the column where "logically" */
								/* the carriage return is in the line	*/

#define NO_CR			0xff	/* special value put in CR_COL to		*/
								/* indicate wrap occured in line		*/

#define LINESTYLE_COL 	(dp->cursor.ncols+1)
								/* position of second special column, it */
								/* contains the style of characters in	*/
								/* this line							*/

#define DHEIGHT_TOP		3		/* line styles							*/
#define DHEIGHT_BOT		4
#define SINGLEWIDTH		5
#define DOUBLEWIDTH		6

								/* number of tab bytes required for		*/
								/* packed tabs							*/
#define MAX_NTBYTES		(MAX_NCOLS/8 + ((MAX_NCOLS%8)?1:0))


#define PARMLIST_SIZE	8		/* number of entries in parm.list		*/
#define BUFFER_SIZE		NCOLS80	/* number of entries in buffer.data		*/

#ifndef MARK_CANCEL
#define	MARK_CANCEL		0		/* command codes for term_mark			*/
#define	MARK_REGION		1
#define MARK_WORD		2
#define MARK_LINE		3
#endif

#define	LED_ISI_IDOT	0x01
#define LED_OFFLINE		0x02
#define	LED_KBD_LOCKED	0x04
#define	LED_L1			0x08
#define	LED_L2			0x10
#define	LED_L3			0x20
#define LED_L4			0x40
#define	LED_HIDDEN		0x80
#define LED_RESET		LED_ISI_IDOT


/*
 *	DECLARATIONS
 */

struct pad_location {
	short line, col;
};

struct pad_bounds {
	struct pad_location top, bot;
};

struct term_data {

	struct tty *vttyp;				/* vtty for this emulator				*/

#ifdef KERNEL
	int *winp;						/* window for this emulator				*/
	COLORT fcolor, bcolor;			/* foreground and background color		*/
#else
	int winp;
	VT_COLOR fcolor, bcolor;
#endif

	short yoff,						/* offset to keep display anchored at 	*/
									/* bottom of window						*/
		cwidth,						/* width, height and baseline of a		*/
		cheight,					/* character in the current font		*/
		cbaseline;

	short width,					/* current width and height of window	*/
		height;

	struct {
		bool call_hilight;			/* flag to stop infinite recursion		*/
		bool ing;					/* marking is on						*/
		struct pad_bounds region;	/* bounds of marked region				*/
	} mark;

	struct {
		bool visible;				/* is cursor currently visible?			*/
		short line, col;			/* cursor position						*/
		short sav_line, sav_col;	/* saved cursor position				*/
		short nlines, ncols;		/* number of lines and columns on		*/
									/* screen (bounds for cursor)			*/
	} cursor;

	short sr_top, sr_bot;			/* scrolling region top and bottom		*/

	unsigned char keyb_leds;		/* keyboard LED bit mask				*/

	struct {
		bool size_fixed;			/* the number of lines and columns are	*/
									/* fixed, if unfixed the pad is auto-
									/* matically sized to match the window	*/
		bool application_cursorkey;
		bool reverse_screen;
		bool relative_origin;
		bool wraparound;
		bool wrap_on_next_char;		/* used internally when in wraparound	*/
		bool application_keypad;
#ifdef NOT_IMPLEMENTED
		bool smooth_scroll;
		bool auto_repeat;
		bool interlace;
#endif
	} mode;

	struct {
		bool reverse;
#ifdef NOT_IMPLEMENTED
		bool blink;
		bool bold;
		bool underscore;
#endif
	} char_attr;

	struct {
		enum {
			SAW_NEWJERSEY, SAW_ESC, SAW_LBRACKET, SAW_POUND,
			SAW_LPAREN, SAW_RPAREN
		} state;
		short power;
		short *listp;
		short list[PARMLIST_SIZE];
	} parm;

	struct {
		short cache_cnt;			/* cached character count				*/
		short cache_col;			/* column on pad of first cached char	*/
		unsigned char *p;			/* pointer to pad						*/
		unsigned char *first_linep; /* pointer to first pad line 			*/
		unsigned char *limitp;		/* pointer to first character off pad	*/
	} pad;

	struct {
		bool active;
		bool flushing;
		unsigned char *datap;
		unsigned char data[BUFFER_SIZE];
	} buffer;

	unsigned char tabs[ MAX_NTBYTES ];
};

unsigned char *pad_linep();
unsigned char *CreateBlock();
unsigned char *DestroyBlock();
static bool adjust();

/*
 *	my_sprintf - sprintf used internally
 */
static void
my_sprintf( dp, fmt, a0, a1, a2, a3, a4, a5 )
register struct term_data *dp;
register char *fmt;
register a0, a1, a2, a3, a4, a5;
{
	char buf[80];
	register char *bufp;
	register short n, arg = 0;

	bufp = buf;
	for( bufp = buf; *fmt; ++bufp, ++fmt ) {
#ifndef KERNEL
		if( *fmt == '\033' ) {
			*bufp = 'E';
			++bufp;
			*bufp = 'S';
			++bufp;
			*bufp = 'C';
		}
		else
#endif	
		if( *fmt == '%' ) {
			++fmt; /* skip the d in %d */
			switch( arg++ ) {
			case 0:
				n = a0;
				break;
			case 1:
				n = a1;
				break;
			case 2:
				n = a2;
				break;
			case 3:
				n = a3;
				break;
			case 4:
				n = a4;
				break;
			case 5:
				n = a5;
				break;
			}
			if( n/100 > 0 ) {
				*bufp = n/100 + '0';
				*(++bufp) = (n/10)%10 + '0';
				*(++bufp) = (n%10) + '0';
			}
			else if( n/10 > 0 ) {
				*bufp = (n/10) + '0';
				*(++bufp) = (n%10) + '0';
			}
			else {
				*bufp = (n%10) + '0';
			}
		}
		else {
			*bufp = *fmt;
		}
	}
	*bufp = '\0';

#ifdef KERNEL
	for( bufp = buf; *bufp; ++bufp ) {
		/* make them characters put out (or should it be output) */
		(*keybsw[dp->vttyp->v_te].k_keyin)(dp->vttyp->v_keyb, *bufp);
	}
#else
	DisplayStatus( 1, buf );
#endif
}

/*
 *	adjust_pad - adjust pad to have the passed number of lines and columns
 *
 *	TRUE is returned if the adjust was successful.  FALSE is returned if the
 *	adjust failed due to malloc.
 *
 *	When called a new pad is malloced.  Then if it is not the first time
 *	called the the contents of the old pad are copied to the new one.
 */
static
bool
adjust_pad( dp, nlines, ncols )
register struct term_data *dp;
register short nlines, ncols;
{
	int pad_size			= nlines*(ncols+NSPECIALCOLS);

	/* create new pad and tab set to sized to nlines and ncols */
	register unsigned char *new_padp = CreateBlock( pad_size );

	if( new_padp == NULL ) {
		/* the malloc failed, continue using old pad */
		return( FALSE );
	}

	/* initialize new pad */
	{
		register unsigned char *p;
		register unsigned char *pad_limitp = &new_padp[ pad_size ];
		for( p = new_padp; p < pad_limitp; p+=2 ) {
			register unsigned char *line_limitp = &p[ ncols ];
			for( ; p < line_limitp; ++p ) {
				*p = ' ';
			}
			*p = 0;
			*(p+1) = SINGLEWIDTH;
		}

	}

	if( dp->pad.p ) {

		/* copy old pad to new one */

		short old_line = dp->cursor.nlines - 1;
		short new_line = nlines;
		short hold_new_line;

		short nchars, hold_nchars, lines_needed;

		register unsigned char *n_linep;
		short o_line;
		register unsigned char *new_charp, *old_charp;
		short new_nchars, old_nchars;

		bool cursor_adjusted = FALSE;

#ifdef ADJUST_PAD
		debug_printf( "*new* nlines %d, ncols %d\n", nlines, ncols );
		debug_printf( "*old* dp->cursor.nlines %d, dp->cursor.ncols %d\n",
			dp->cursor.nlines, dp->cursor.ncols );
#endif ADJUST_PAD

		while( new_line > 0 && old_line >= 0 ) {

			/* count characters to copy to new line */
			nchars = *(pad_charp( dp, old_line, CR_COL ));	
#ifdef ADJUST_PAD
			debug_printf( "old_line %d, nchars %d, ", old_line, nchars );
#endif ADJUST_PAD

			while( TRUE ) {
			
				old_line--;
				if( old_line < 0 ) {
					/* finished all old lines */
					break;
				}

				if( *(pad_charp( dp, old_line, CR_COL )) == NO_CR ) {
					nchars += dp->cursor.ncols;
				}
				else {
					/* counted characters until an unwrapped line */
					break;
				}
			}
#ifdef ADJUST_PAD
			debug_printf( "nchars %d\n", nchars );
#endif ADJUST_PAD

			/* number of new lines required for nchars */
			lines_needed = nchars / ncols;
			if( (nchars % ncols) > 0 ) {
				++lines_needed;
			}
			else if( nchars == 0 ) {
				lines_needed = 1;
			}
#ifdef ADJUST_PAD
			debug_printf( "new lines_needed %d, ", lines_needed );
#endif ADJUST_PAD

			/* count down the appropriate number of new lines */
			new_nchars = 0;
			old_nchars = 0;
			while( new_line < lines_needed ) {
				nchars -= ncols;
				lines_needed--;
				old_nchars += ncols;		
			}
			old_line += old_nchars / dp->cursor.ncols;
			old_nchars = old_nchars % dp->cursor.ncols;

			new_line -= lines_needed;
			hold_new_line = new_line;

#ifdef ADJUST_PAD
			debug_printf( "new_line %d\n", new_line );
#endif ADJUST_PAD

			/* copy characters from old lines to new lines */
			n_linep = new_padp + new_line*(ncols+NSPECIALCOLS); 
			o_line = old_line+1;
			new_charp = n_linep;
			old_charp = pad_charp( dp, o_line, old_nchars );

			hold_nchars = nchars;
			for( ; nchars; nchars-- ) {

				*(new_charp++) = *(old_charp++);

				if( !cursor_adjusted &&
					o_line == dp->cursor.line &&
					old_nchars == dp->cursor.col ) {
					cursor_adjusted = TRUE;
					dp->cursor.line = new_line;
					dp->cursor.col  = new_nchars;
				}

				++new_nchars;
				++old_nchars;
				if( nchars-1 == 0 ) {
					break;
				}

				if( new_nchars >= ncols ) {
					/* next line from new pad */

					/* characters are left, wrap occured */
					*new_charp = NO_CR;

#ifdef ADJUST_PAD
					debug_printf( "wrap on new_line %d\n", new_line );
#endif ADJUST_PAD
					new_nchars = 0;
					++new_line;
					n_linep = new_charp+=2;
				}

				if( old_nchars >= dp->cursor.ncols ) {
					/* next line from old pad */
#ifdef ADJUST_PAD
					debug_printf( "completed o_line %d\n", o_line );
#endif ADJUST_PAD
					old_nchars = 0;
					old_charp = pad_linep( dp, ++o_line );
				}

			}

			if( ncols == dp->cursor.ncols ) {
				/* both pads have the same number of columns, use
				 * character count from old line.  Must do this to
				 * preserve possible NO_CR in old line */
				n_linep[ ncols ] = *(pad_charp( dp, o_line, CR_COL ) );
			}
			else {
				n_linep[ ncols ] = new_nchars;
			}

			n_linep[ ncols+1 ] = *(pad_charp( dp, o_line, LINESTYLE_COL ) );

			if( !cursor_adjusted &&
				o_line == dp->cursor.line ) {
				cursor_adjusted = TRUE;
				dp->cursor.line = new_line;
				dp->cursor.col  += (new_nchars - old_nchars);
			}

			new_line = hold_new_line;
		}
		DestroyBlock( dp->pad.p );

	} /* if( dp->pad.p ) */

	/* adjust cursor and related cursor stuff */

	if( dp->cursor.line >= nlines ) {
		dp->cursor.line = nlines-1;
	}
	if( dp->cursor.col >= ncols ) {
		dp->cursor.col = ncols-1;
	}
	if( dp->cursor.sav_line >= nlines ) {
		dp->cursor.sav_line = nlines-1;
	}
	if( dp->cursor.sav_col >= ncols ) {
		dp->cursor.sav_col = ncols-1;
	}

	dp->cursor.sav_line		= 0;
	dp->cursor.sav_col		= 0;
	dp->cursor.nlines		= nlines;
	dp->cursor.ncols		= ncols;
	dp->sr_top				= 0;
	dp->sr_bot				= nlines-1;

	dp->pad.p				= new_padp;
	dp->pad.cache_cnt		= 0;
	dp->pad.first_linep		= new_padp;
	dp->pad.limitp			= &dp->pad.p[ pad_size ];

	adjust( dp, dp->width, dp->height );

	return( TRUE );
}

/*
 *	cub - Move cursor backward
 */
static
cub( dp, n )
register struct term_data *dp;
register short n;
{
	if( n == DEF_PARMVAL || n == 0 ) {
		/* set n to 1 if it is zero or DEF_PARMVAL */
		n = 1;
	}
	if( ( dp->cursor.col -= n ) < 0 ) {
		dp->cursor.col = 0;
	}
}

/*
 *	put_char - Put displayable character on pad
 */
static
put_char( dp, c )
register struct term_data *dp;
unsigned char c;
{
	register short cur_col = dp->cursor.col;
	register unsigned char *cp = pad_linep( dp, dp->cursor.line );


	/* put character on pad and on screen */

	if( (cur_col+1) == dp->cursor.ncols ) {
		if( dp->mode.wrap_on_next_char ) {
			dp->mode.wrap_on_next_char = FALSE;
			if( dp->pad.cache_cnt ) {
				dp->pad.cache_cnt--;
				flush_cache( dp );
				dp->pad.cache_cnt = 1;
				dp->pad.cache_col = 0;
			}
			cur_col = 0;
			lf( dp );
			cp[CR_COL] = NO_CR;
			cp = pad_linep( dp, dp->cursor.line );
		}
		else if( dp->mode.wraparound ) {
			dp->mode.wrap_on_next_char = TRUE; 
		}
	}

	/* CR is always assumed to be past right most character */

	if( cp[CR_COL] < (cur_col+1) /*|| cp[CR_COL] == NO_CR*/  ) {
		cp[CR_COL] = (cur_col+1);
	}

	cp += cur_col;
	*cp = c | ( dp->char_attr.reverse ? 0x80 : 0x00 );
	if( !dp->buffer.flushing ) {
		/* not buffering display character now */
		SetPosition( dp->winp, cur_col*dp->cwidth,
			dp->cursor.line*dp->cheight + dp->yoff + dp->cbaseline );
		PaintString( dp->winp, 1, &c );
	}

	/* adjust cursor position */

	if( cur_col+1 < dp->cursor.ncols ) {
		++cur_col;
		dp->mode.wrap_on_next_char = FALSE;
	}
	dp->cursor.col = cur_col;
}

/*
 *	ed - Erase in display 
 */
static
ed( dp, n )
register struct term_data *dp;
register short n;
{
	if( dp->mode.wrap_on_next_char ) {
		dp->mode.wrap_on_next_char = FALSE;
		*(pad_charp( dp, dp->cursor.line, CR_COL )) = NO_CR;
		dp->cursor.col = 0;
		lf( dp );
	}
	switch( n ) {
	case DEF_PARMVAL:
	case 0:
		/* erase from the active position to the end of the screen */
		pad_fill( dp, ' ',
			dp->cursor.line, dp->cursor.col,
			dp->cursor.nlines-1, dp->cursor.ncols-1 ); 

		screen_erase( dp,
			dp->cursor.line, dp->cursor.col,
			dp->cursor.line, dp->cursor.ncols-1 );
		if( dp->cursor.line < dp->cursor.nlines-1 ) {
			screen_erase( dp,
				dp->cursor.line+1, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );
		}
		break;
	case 1:
		/* erase from the start of the screen to the active position */
		pad_fill( dp, ' ',
			0, 0, dp->cursor.line, dp->cursor.col ); 

		if( dp->cursor.line > 0 ) {
			screen_erase( dp,
				0, 0, dp->cursor.line-1, dp->cursor.ncols-1 );
		}
		screen_erase( dp,
			dp->cursor.line, 0, dp->cursor.line, dp->cursor.col );
		break;
	case 2:
		/* erase all of the display */
		pad_fill( dp, ' ',
			0, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );
		screen_erase( dp, 0, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );

		/* erasing the whole display resets the CR col and line style for all
		lines */
		{
			register short line;

			for( line = 0; line < dp->cursor.nlines; ++line ) {
				*(pad_charp( dp, line, CR_COL )) = 0;
				*(pad_charp( dp, line, LINESTYLE_COL )) = SINGLEWIDTH;
			}
		}
		break;
	default:
		/* invalid parameter value */
		break;
	}
}

/*
 *	el - Erase in line 
 */
static
el( dp, n )
register struct term_data *dp;
register short n;
{

	if( dp->mode.wrap_on_next_char ) {
		dp->mode.wrap_on_next_char = FALSE;
		*(pad_charp( dp, dp->cursor.line, CR_COL )) = NO_CR;
		dp->cursor.col = 0;
		lf( dp );
	}
	switch( n ) {
	case DEF_PARMVAL:
	case 0:
		/* erase from the active position to the end of the line */
		pad_fill( dp, ' ',
			dp->cursor.line,dp->cursor.col,dp->cursor.line,dp->cursor.ncols-1 );
		screen_erase( dp,
			dp->cursor.line,dp->cursor.col,dp->cursor.line,dp->cursor.ncols-1 );
		break;
	case 1:
		/* erase from the start of the line to the active position */
		pad_fill( dp, ' ',
			dp->cursor.line, 0, dp->cursor.line, dp->cursor.col );
		screen_erase( dp,
			dp->cursor.line, 0, dp->cursor.line, dp->cursor.col );
		break;
	case 2:
		/* erase all of the line */
		pad_fill( dp, ' ',
			dp->cursor.line, 0, dp->cursor.line, dp->cursor.ncols-1 );
		screen_erase( dp,
			dp->cursor.line, 0, dp->cursor.line, dp->cursor.ncols-1 );
		break;
	default:
		/* invalid parameter value */
		break;
	}
}

/*
 *	lf - Line feed
 */
static
lf( dp )
register struct term_data *dp;
{
	if( ++(dp->cursor.line) > dp->sr_bot ) {
		--(dp->cursor.line);
		/* scroll */
		pad_scroll_up( dp );
		screen_scroll_up( dp );
	}
}

/*
 *	TAB ROUTINES
 *
 *	For vt100 emulation the tab set is always 132.  The window emulator
 *	tab set always matches the number of columns.
 */

/*
 *	tab - move cursor to next tab position
 */
static
tab( dp )
register struct term_data *dp;
{
	register short curr_bit = (dp->cursor.col+1)%8;
	register short curr_byte = (dp->cursor.col+1)/8;
	register short last_bit = dp->cursor.ncols%8;
	register short last_byte = dp->cursor.ncols/8;
	register short offset = 1;

	while( TRUE ) {
		if( offset + dp->cursor.col >= dp->cursor.ncols-1 ) {
			dp->cursor.col = dp->cursor.ncols-1;
			return;
		}
		if( dp->tabs[ curr_byte ] & (0x01 << curr_bit) ) {
			dp->cursor.col += offset;
			return;
		}
		if( ++curr_bit > 7 ) {
			curr_bit = 0;
			++curr_byte;
		}
		++offset;
	}
}

/*
 *  clear_tab - clear tab at "col"
 */
static 
clear_tab( dp )
register struct term_data *dp;
{
	register short col = dp->cursor.col;
	dp->tabs[ col/8 ] &= ~(0x01<< col%8);
}

/*
 *	set_tab - clear tab at "col"
 */
static
set_tab( dp )
register struct term_data *dp;
{
	register short col = dp->cursor.col;
	dp->tabs[ col/8 ] |= (0x01<< col%8);
}

/*
 *	clear_tabs - clear all tabulations
 */
static
clear_tabs( dp )
register struct term_data *dp;
{
	register unsigned char *p;
	register unsigned char *limitp = &dp->tabs[ MAX_NTBYTES ];

	for( p = dp->tabs; p < limitp; ++p ) {
		*p = FALSE;
	}
}

/*
 *	set_tabs - set all tabulations
 */
static
set_tabs( dp )
register struct term_data *dp;
{
	register unsigned char *p;
	register unsigned char *limitp = &dp->tabs[ MAX_NTBYTES ];

	for( p = dp->tabs; p < limitp; ++p ) {
		*p = 0x01;
	}
}

/*
 *	pad_linep - Return address of line indexed by "line"
 */
static
unsigned char *
pad_linep( dp, line )
register struct term_data *dp;
register short line;
{
	register unsigned char *linep =
		&dp->pad.first_linep[line*(dp->cursor.ncols+NSPECIALCOLS)];

	if( linep >= dp->pad.limitp ) {
		linep = dp->pad.p + (linep - dp->pad.limitp );
	}
	return( linep );
}
	
/*
 *	pad_fill - Fill pad with 'c'.
 */
static
pad_fill( dp, c, f_line, f_col, l_line, l_col )
register struct term_data *dp;
register unsigned char c;
short f_line, f_col, l_line, l_col; 
{
	register short line;
	register unsigned char *charp, *limitp;

	for( line = f_line; line <= l_line; ++line ) {
		/* first character in line */
		charp		= pad_linep( dp, line );
		/* last character in line */
		limitp		= charp + ( (line==l_line)?l_col:(dp->cursor.ncols-1) );
		/* set charp to the appropriate starting character.  Set CR byte
		 * for line */
		if( line == f_line ) {
			/* move CR to position of first character erased in first line */
			charp[ CR_COL ] = f_col;
			charp += f_col;
		}
		else if( line == l_line && (l_col != (dp->cursor.ncols-1))  ) {
			/* move CR to position past last character erased */
			charp[CR_COL] = l_col+1;
		}
		else {
			/* move CR to first character in line */ 
			charp[ CR_COL ] = 0;
		}
		for( ; charp <= limitp; ++charp ) {
			*charp = c;
		}
	}
}

/*
 *	pad_scroll_up - scroll the specified scrolling region up
 *	one line.
 */
static
pad_scroll_up( dp )
register struct term_data *dp;
{
	register unsigned char *charp;
	register short sr_top = dp->sr_top;
	register short sr_bot = dp->sr_bot;
	register short ncols = dp->cursor.ncols+NSPECIALCOLS;

	if( sr_top == 0 && sr_bot == dp->cursor.nlines-1 ) {

		/* scroll entire pad */

		/* clear the first line, it is about to become the line at the
		 * bottom of the screen */
		charp = dp->pad.first_linep;

		/* increment first line pointer, and wrap if necessary */
		if( (dp->pad.first_linep+=ncols) == dp->pad.limitp ) {
			dp->pad.first_linep = dp->pad.p;
		}
	}

	else {

		/* scroll a region within the pad */

		for( ; sr_top < sr_bot; ++sr_top ) {
			bcopy(	pad_linep( dp, sr_top+1 ),
					pad_linep( dp, sr_top ), ncols );
		}

		charp = pad_linep( dp, sr_bot );
	}
	
	{
		/* clear the appropriate line, the one adressed by charp.			*/
		/* this includes the CR_COL and LINESTYLE_COL						*/

		register unsigned char *limitp = charp + dp->cursor.ncols;

		for(; charp < limitp; ++charp ) {
			*charp = ' ';
		}
		*charp = 0;
		*(++charp) = SINGLEWIDTH;
	}
}

/*
 *	pad_scroll_dn - scroll the specified scrolling region down
 *	one line.
 */
static
pad_scroll_dn( dp )
register struct term_data *dp;
{
	register unsigned char *charp;
	register short sr_top = dp->sr_top;
	register short sr_bot = dp->sr_bot;
	register short ncols = dp->cursor.ncols+NSPECIALCOLS;

	if( sr_top == 0 && sr_bot == dp->cursor.nlines-1 ) {

		/* scrolling entire pad, decrement line pointer and wrap if
		 * necessary */

		if( (dp->pad.first_linep-=ncols) < dp->pad.p ) {
			dp->pad.first_linep = dp->pad.limitp - ncols;
		}

		/* clear the first line */
		charp = dp->pad.first_linep;
	}

	else {

		/* scroll a region within the pad */

		for( ; sr_top < sr_bot; sr_bot-- ) {
			bcopy(	pad_linep( dp, sr_bot-1 ),
					pad_linep( dp, sr_bot ), ncols );
		}

		charp = pad_linep( dp, sr_top );
	}
	
	{
		/* clear the appropriate line, the one adressed by charp.			*/
		/* this includes the CR_COL and LINESTYLE_COL						*/

		register unsigned char *limitp = charp + dp->cursor.ncols;

		for(; charp < limitp; ++charp ) {
			*charp = ' ';
		}
		*charp = 0;
		*(++charp) = SINGLEWIDTH;
	}
}

/*
 *	screen_erase - Erase screen region
 */
static
screen_erase( dp, f_line, f_col, l_line, l_col )
register struct term_data *dp;
short f_line, f_col, l_line, l_col; 
{
	SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
	SetPosition( dp->winp, f_col*dp->cwidth, f_line*dp->cheight+dp->yoff );
	PaintRectangleInterior( dp->winp,
		(l_col-f_col+1)*dp->cwidth, (l_line-f_line+1)*dp->cheight );
	SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );
}

/*
 *	screen_scroll_up - scroll the scrolling region up one line.
 */
static
screen_scroll_up( dp )
register struct term_data *dp;
{
	register short sr_top = dp->sr_top;
	register short sr_bot = dp->sr_bot;
	register short cheight = dp->cheight;
	register short w = dp->cursor.ncols*dp->cwidth;
	register short yoff = dp->yoff;
	register short sy = (sr_top+1)*cheight+yoff;

	if( sy< 0 ) {
		CopyRegion( dp->winp, 0,0, 0,-cheight,
			w, (sr_bot+1)*cheight+ yoff ); 
	}
	else {
		CopyRegion( dp->winp, 0, sy, 0, sy-cheight, 
			w, (sr_bot-sr_top)*cheight );
	}
	SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
	SetPosition( dp->winp, 0, sr_bot*cheight+yoff );
	PaintRectangleInterior( dp->winp, w, cheight );
	SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );
}

/*
 *	screen_scroll_dn - scroll the scrolling region down one line.
 *
 *	NOTE: The pad must be scrolled first or the refresh call that
 *	happens on small windows will be illbehaved.
 */
static
screen_scroll_dn( dp )
register struct term_data *dp;
{
	register short sr_top = dp->sr_top;
	register short sr_bot = dp->sr_bot;
	register short cheight = dp->cheight;
	register short w = dp->cursor.ncols*dp->cwidth;
	register short yoff = dp->yoff;
	register short sy = sr_top*cheight+yoff;

	if( sy < 0 ) {
		CopyRegion( dp->winp, 0,0, 0,cheight, w, sr_bot*cheight+yoff ); 
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
		SetPosition( dp->winp, 0, 0 );
		PaintRectangleInterior( dp->winp, w, cheight );
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );
		refresh( dp, 0,0, w, cheight );
	}
	else {
		CopyRegion( dp->winp, 0, sy, 0, sy+cheight,
			w, (sr_bot-sr_top)*cheight );
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
		SetPosition( dp->winp, 0, sy );
		PaintRectangleInterior( dp->winp, w, cheight );
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );
	}
}

/*
 *	hide_cursor
 */
static
hide_cursor( dp )
register struct term_data *dp;
{
	if( dp->cursor.visible ) {
		dp->cursor.visible = FALSE;
		InvertRegion( dp->winp, 
			dp->cursor.col*dp->cwidth, dp->cursor.line*dp->cheight + dp->yoff,
			dp->cwidth, dp->cheight );
	}
}

/*
 *	show_cursor
 */
static
show_cursor( dp )
register struct term_data *dp;
{
	if( !dp->cursor.visible ) {
		dp->cursor.visible = TRUE;
		InvertRegion( dp->winp, 
			dp->cursor.col*dp->cwidth, dp->cursor.line*dp->cheight + dp->yoff,
			dp->cwidth, dp->cheight );
	}
}

/*
 *	term_init - Initialize terminal emulator
 */
struct term_data *
term_init( vttyp )
register struct tty *vttyp;
{
	register struct term_data *dp;

	dp = (struct term_data *) CreateBlock ( sizeof( struct term_data ) );

	if( dp == NULL ) {
		/* malloc failed, quit now */
		return( NULL );
	}

#ifdef KERNEL
	{
		unsigned short font;
		register struct font_plane *font_planep;
		dp->vttyp		= vttyp;
		dp->winp		= vttyp->v_win;
		GetCurrentFont( dp->winp, &font );
		font_planep		= GetFontPlane( font, DEFPLANE );
		SetAddressing( dp->winp, RELATIVE );
		SetJustification( dp->winp, FLUSHLEFT );
		SetCharacterAttributes( dp->winp, CLEARBACK );
		dp->cwidth		= CharacterWidth( font_planep, 'W' );
		dp->cheight		= CharacterHeight( font_planep );
		dp->cbaseline	= CharacterBaseline( font_planep );
		DoneWithPlane( font, DEFPLANE );
		GetCurrentColors( dp->winp, &dp->fcolor, &dp->bcolor );
		GetWindowSize( dp->winp, &dp->width, &dp->height );
		/* ignore mouse strokes */
		(*keybsw[dp->vttyp->v_te].k_ctl)(dp->vttyp->v_keyb, KMMASK, 0 );
	}
#else
	{
		struct wstate window_state;
		dp->vttyp		= vttyp;
		dp->winp		= 1; 
		SetLineDisc( 1, TWSDISC );
		GetWindowState( 1, &window_state );
		SetAddressing( dp->winp, VT_RELATIVE );
		SetJustification( dp->winp, VT_FLUSHLEFT );
		SetCharAttributes( dp->winp, VT_CLEARBACK );
		dp->cwidth		= CharacterWidth( window_state.font, 'W' );
		dp->cheight		= CharacterHeight( window_state.font );
		dp->cbaseline	= CharacterBaseline( window_state.font );
		dp->fcolor		= window_state.fcolor;
		dp->bcolor		= window_state.bcolor;
		dp->width		= window_state.width;
		dp->height		= window_state.height;
	}
#endif

	dp->mark.call_hilight		= TRUE;
	dp->mark.ing				= FALSE;

	dp->cursor.visible 			= FALSE;
	dp->cursor.line				= 0;
	dp->cursor.col				= 0;

	dp->char_attr.reverse		= FALSE;
#ifdef NOT_IMPLEMENTED
	dp->char_attr.blink			= FALSE;
	dp->char_attr.bold			= FALSE;
	dp->char_attr.underscore	= FALSE;
#endif

	dp->mode.size_fixed			= TRUE;
	dp->mode.reverse_screen		= FALSE;
	dp->mode.relative_origin	= FALSE;
	dp->mode.wraparound			= TRUE;
	dp->mode.application_keypad	= FALSE;
#ifdef NOT_IMPLEMENTED
	dp->mode.smooth_scroll		= FALSE;
	dp->mode.auto_repeat		= TRUE;
	dp->mode.interlace			= FALSE;
#endif

	dp->parm.state				= SAW_NEWJERSEY;

	dp->pad.p					= NULL;

	dp->buffer.active			= FALSE;
	dp->buffer.flushing			= FALSE;

	if( dp->mode.size_fixed ) {
		if( adjust_pad( dp, NLINES80, NCOLS80 ) == NULL ) {
			DestroyBlock( dp );
			return( NULL );
		}
	}
	else if( adjust( dp, dp->width, dp->height ) == FALSE ) {
		/* mallocing the pad failed, clean up and fail */
		DestroyBlock( dp );
		return( NULL );
	}

	set_tabs( dp );

	SetRefresh( dp->winp, dp, refresh );
	SetAdjust( dp->winp, dp, adjust );

	return( dp );
}

/*
 *	term_buffer_start
 */
term_buffer_start( dp )
register struct term_data *dp;
{
	if( !dp->buffer.active ) {

		dp->buffer.active = TRUE;
		dp->buffer.datap = &dp->buffer.data[0];
		hide_cursor( dp );
	}
}

/*
 *	term_buffer_stop
 */
term_buffer_stop( dp )
register struct term_data *dp;
{
	register unsigned char *datap, *limitp;

	if( dp->buffer.active ) {

		dp->buffer.active = FALSE;

		datap = &dp->buffer.data[0];
		limitp = dp->buffer.datap;
		dp->buffer.flushing = TRUE;
		for( ; datap < limitp; ++datap ) {
			term_disp( dp, *datap );
		}
		dp->buffer.flushing = FALSE;

		if( !dp->mode.wrap_on_next_char  ) {
			flush_cache( dp );
		}
		show_cursor( dp );
	}
}

/*
 *	flush_cache
 */
flush_cache( dp )
register struct term_data *dp;
{
#ifdef notdef
	debug_printf(" cache_cnt %d, cache_col %d \n",
	dp->pad.cache_cnt,dp->pad.cache_col);
#endif notdef
	if( dp->pad.cache_cnt ) {
		SetPosition( dp->winp, dp->pad.cache_col*dp->cwidth,
			dp->cursor.line*dp->cheight+dp->cbaseline+dp->yoff );
		PaintString( dp->winp, dp->pad.cache_cnt,
			pad_charp( dp, dp->cursor.line, dp->pad.cache_col ) );
		dp->pad.cache_cnt = 0;
	}
}

/*
 *	term_quit
 */
term_quit( dp )
register struct term_data *dp;
{
	term_buffer_stop( dp );
	DestroyBlock( dp->pad.p );
	DestroyBlock( dp );
#ifndef KERNEL
	SetLineDisc( 1, NTTYDISC );
#endif
}

#ifdef KERNEL
/*
 *	term_change_font
 */
term_change_font( dp, font )
register struct term_data *dp;
register short font;
{
	register struct font_plane *font_planep;

	SetFont( dp->winp, font );
	font_planep		= GetFontPlane( font, DEFPLANE );
	dp->cwidth		= CharacterWidth( font_planep, 'W' );
	dp->cheight		= CharacterHeight( font_planep );
	dp->cbaseline	= CharacterBaseline( font_planep );
	DoneWithPlane( font, DEFPLANE );

	adjust( dp, dp->width, dp->height );
	if( dp->mode.size_fixed ) {
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
		SetPosition( dp->winp, 0, 0 );
		PaintRectangleInterior( dp->winp, dp->width, dp->height );
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );

		refresh( dp, 0, 0, dp->width, dp->height );
	}
}
#endif

/*
 *	term_disp - display character
 *
 *	This is the routine used to pass all characters are to the emulator.
 *	If buffering is active, characters recieved are saved and no further
 *	action is taken until the buffer fills, buffering is turned off or
 *	a non-printable character is recieved.
 */

term_disp( dp, c )
register struct term_data *dp;
register unsigned char c;
{
	register short parm_cnt, parm_index, p0, p1;

	if( dp->buffer.active ) {
		if( dp->buffer.datap == &dp->buffer.data[BUFFER_SIZE] ) {
			/* buffer is full, flush before adding a character to it */
			term_buffer_stop( dp );
			term_buffer_start( dp );
		}
		*dp->buffer.datap = c;
		dp->buffer.datap++;
		return;
	}
	else if( !dp->buffer.flushing ) {
		/* hide cursor before continuing */
		hide_cursor( dp );
	}

	if( dp->buffer.flushing && dp->parm.state == SAW_NEWJERSEY ) {
		/* while flushing buffer, the pad is used to cache characters before 
		 * displaying them. */
		if( isprint( c ) ) {
			/* displayable characters can be cached */
			if(	dp->pad.cache_cnt == 0 ) {
				/* begin caching characters */
				dp->pad.cache_col = dp->cursor.col;
			}
			dp->pad.cache_cnt++;
		}
		else {
			/* Empty cache on encountering a non-displayable chacacter */
			flush_cache( dp );
		}
	}

	/* handle control characters first */
	switch( c ) {
	case 007: /*BEL*/
		/* sound bell tone from keyboard */
#ifdef KERNEL
		(*keybsw[dp->vttyp->v_te].k_ctl) (dp->vttyp->v_keyb, KBELL, 1);
#endif
		goto done;
	case 010: /*BS*/
		/* move cursor left */
		if( dp->cursor.col == 0 && dp->cursor.line > 0 ) {
			if( *pad_charp( dp, dp->cursor.line-1, CR_COL ) == NO_CR ) {
				dp->cursor.line--;
				dp->cursor.col = dp->cursor.ncols - 1;
			}
		}
		else {
			cub( dp, 1 );
		}
		goto done;
	case 011: /*HT*/
		/* move cursor to next tab */
		tab( dp );
		goto done;
	case 012: /*LF*/
	case 013: /*VT*/
	case 014: /*FF*/
		/* line feed */
		lf( dp );
		goto done;
	case 015: /*CR*/
		dp->cursor.col = 0;
		goto done;
	case 016: /*SO*/
		/* select G1 character set */
		goto done;
	case 017: /*SI*/
		/* select G0 character set */
		goto done;
	case 030: /*CAN*/
	case 032: /*SUB*/
		dp->parm.state = SAW_NEWJERSEY;
		goto done;
	case 033: /*ESC*/
		/* introduction of escape sequence */
		dp->parm.state = SAW_ESC;
		return;
	default:
		if( iscntrl( c ) ) {
			/* eat all other control codes */
			goto done;
		}
		break;
	} /*switch*/

	switch( dp->parm.state ) {

	case SAW_NEWJERSEY:
		/* do plain character */
		put_char( dp, c );
		break;

	case SAW_ESC:
		switch( c ) {
		case '[':
			/* saw control sequence introducer */
			dp->parm.state = SAW_LBRACKET;
			{
				/* reset parameter array, counter and digit power */
				register short i;
				register short *p;

				for( i=0, p = &dp->parm.list[0]; i<PARMLIST_SIZE; i++, p++ ) {
					*p = DEF_PARMVAL;
				}
				dp->parm.listp = &dp->parm.list[0];
				dp->parm.power = 1;
			}
			return;
		case '#':
			/* saw intermediate pound */
			dp->parm.state = SAW_POUND;
			return;
		case '(':
			/* saw intermediate left paren */
			dp->parm.state = SAW_LPAREN;
			return;
		case ')':
			/* saw intermediate right paren */
			dp->parm.state = SAW_RPAREN;
			return;

		case 'Z':
			/* identify terminal */
			break;
		case '=':
			/* keypad application mode */
			dp->mode.application_keypad = TRUE;
#ifdef KERNEL
			(*keybsw[dp->vttyp->v_te].k_ctl)
					(dp->vttyp->v_keyb, KNUMERIC, 0);
#endif
			break;
		case '>':
			/* keypad numeric mode */
			dp->mode.application_keypad = FALSE;
#ifdef KERNEL
			(*keybsw[dp->vttyp->v_te].k_ctl)
					(dp->vttyp->v_keyb, KNUMERIC, 1);
#endif
			break;

		case '8':
			/* restore cursor */
			dp->cursor.line	= dp->cursor.sav_line;
			dp->cursor.col	= dp->cursor.sav_col;
			break;
		case '7':
			/* save cursor */
			dp->cursor.sav_line	= dp->cursor.line;
			dp->cursor.sav_col	= dp->cursor.col;
			break;
		case 'H':
			/* set horizontal tab */
			set_tab( dp );
			break;
		case 'D':
			/* index */
			lf( dp );
			break;
		case 'E':
			/* next line */
			dp->cursor.col = 0;
			lf( dp );
			break;
		case 'M':
			/* reverse index */
			if( --(dp->cursor.line) < dp->sr_top ) {
				(dp->cursor.line)++;
				/* scroll */
				pad_scroll_dn( dp );
				screen_scroll_dn( dp );
			}
			break;
		case 'c':
			/* reset to initial state */
			dp->cursor.line = 0;
			dp->cursor.col = 0;
			dp->cursor.sav_line	= dp->cursor.line;
			dp->cursor.sav_col	= dp->cursor.col;
			dp->sr_top = 0;
			dp->sr_bot = dp->cursor.nlines-1;
			set_tabs( dp );
			ed( dp, 2 );
			break;
		default:
			/* strange character following escape, eat it */
			break;
		}
		break; /*case SAW_ESC*/

	case SAW_LBRACKET:
		/* parse parameter string */
		if( c == '?' ) {
			*dp->parm.listp = QUES_PARMVAL;
			/* force execution through next if.  '?' is treated as
			 * a parameter */
			c = ';';
		}
		if( c == ';' ) {
			/* parameter separater */
			if( dp->parm.listp < &dp->parm.list[PARMLIST_SIZE-1] ) {
				/* dp->parm.list not full */
				dp->parm.listp++;
			}
			return;
		}
		else if( isdigit( c ) ) {
			/* parameter digit */
			if( *dp->parm.listp == DEF_PARMVAL ) {
				/* zero entry before first use */
				*dp->parm.listp = 0;
			}
			*dp->parm.listp = (*dp->parm.listp * 10) + (c - '0');
			return;
		}

		/* final character.  figure number of parameters for use below. */
		parm_cnt = (dp->parm.listp - dp->parm.list) + 1;
		p0 = dp->parm.list[0];
		p1 = dp->parm.list[1];
		switch( c ) {
		case 'A':
			/* cursor up */
			{
				if( p0 == DEF_PARMVAL || p0 == 0 ) {
					p0 = 1;
				}
				if( ( dp->cursor.line -= p0 ) < dp->sr_top ) {
					dp->cursor.line = dp->sr_top;
				}
				/* resetting this flag fixes the bug that exists in vt100's
				 * i.e incorrect pre-cursor positioning following cursor
				 * movement. */
				dp->mode.wrap_on_next_char = FALSE;
			}
			break;
		case 'B':
			/* cursor down */
			{
				if( p0 == DEF_PARMVAL || p0 == 0 ) {
					p0 = 1;
				}
				if( ( dp->cursor.line += p0 ) > dp->sr_bot ) {
					dp->cursor.line = dp->sr_bot;
				}
				/* resetting this flag fixes the bug that exists in vt100's
				 * i.e incorrect pre-cursor positioning following cursor
				 * movement. */
				dp->mode.wrap_on_next_char = FALSE;
			}
			break;
		case 'C':
			/* cursor forward */
			{
				if( p0 == DEF_PARMVAL || p0 == 0 ) {
					p0 = 1;
				}
				if( ( dp->cursor.col += p0 ) >= dp->cursor.ncols ) {
					dp->cursor.col = dp->cursor.ncols-1;
				}
				/* resetting this flag fixes the bug that exists in vt100's
				 * i.e incorrect pre-cursor positioning following cursor
				 * movement. */
				dp->mode.wrap_on_next_char = FALSE;
			}
			break;
		case 'D':
			/* cursor backward */
			cub( dp, p0 );
			break;
		case 'f':
		case 'H':
			/* cursor position */
			{
				if( p0 == DEF_PARMVAL || p0 == 0 ) {
					p0 = 1;
				}
				else if( p0 > dp->cursor.nlines ) {
					p0 = dp->cursor.nlines;
				}
				if( p1 == DEF_PARMVAL || p1 == 0 ) {
					p1 = 1;
				}
				else if( p1 > dp->cursor.ncols ) {
					p1 = dp->cursor.ncols;
				}

				p0--;
				dp->cursor.col  = p1 - 1;

				if( dp->mode.relative_origin ) {
					p0 += dp->sr_top;
					if( p0 > dp->sr_bot ) {
						p0 = dp->sr_bot;
					}
				}

				dp->cursor.line = p0;

				/* resetting this flag fixes the bug that exists in vt100's
				 * i.e incorrect pre-cursor positioning following cursor
				 * movement. */
				dp->mode.wrap_on_next_char = FALSE;
			}
			break;
		case 'c':
			/* what am i - base vt100, no options */
			my_sprintf( dp, "\033[?1;0c" );
			break;
		case 'q':
			/* load LEDS */
			{
				register unsigned char keyb_leds = dp->keyb_leds;
				for( parm_index = 0; parm_index < parm_cnt; ++parm_index ) {
					switch( dp->parm.list[parm_index] ) {
					case DEF_PARMVAL:
					case 0:
						keyb_leds = LED_RESET;
						break;
					case 1:
						keyb_leds |= LED_L1;
						break;
					case 2:
						keyb_leds |= LED_L2;
						break;
					case 3:
						keyb_leds |= LED_L3;
						break;
					case 4:
						keyb_leds |= LED_L4;
						break;
					default:
						break;
					}
				}
#ifdef KERNEL
				(*keybsw[dp->vttyp->v_te].k_ctl)
					(dp->vttyp->v_keyb, KSETLED, keyb_leds );
#endif
				dp->keyb_leds = keyb_leds;
			}
			break;
		case 'x':
			/* request terminal parameters */
			break;
		case 'r':
			/* set top and bottom margins */
			{
				if( p0 == DEF_PARMVAL || p0 == 0 ) {
					p0 = 1;
				}
				if( p1 == DEF_PARMVAL || p1 == 0 ) {
					p1 = dp->cursor.nlines;
				}
				if( p1 > dp->cursor.nlines || p0 > p1 ) {
					break;
				}
				dp->sr_top		= --p0;
				dp->sr_bot		= p1 - 1;
				dp->cursor.line	= (dp->mode.relative_origin)?p0:0;
				dp->cursor.col	= 0;
			}
			break;
		case 't':
			switch( p0 ) {
			case 7:
				/* set window state */
				if( p1 == 0 || p1 == 1 ) {
					dp->mode.size_fixed = p1;
				}
				break;
			case 8:
			case 9:
				/* set pad size */
				{
					register ncols = dp->parm.list[2];
					if( p1 == DEF_PARMVAL || p1 == 0 ) {
						p1 = dp->cursor.nlines;
					}
					else if( p1 > MAX_NLINES ) {
						p1 = MAX_NLINES;
					}
					if( ncols == DEF_PARMVAL || ncols == 0 ) {
						ncols = dp->cursor.ncols;
					}
					else if( ncols > MAX_NCOLS ) {
						ncols = MAX_NCOLS;
					}
					dp->mode.size_fixed = TRUE;
					adjust_pad( dp, p1, ncols );
				}
				break;
			case 17:
				/* request window fixedness*/
				my_sprintf( dp, "\033[7;%dt", dp->mode.size_fixed );
				break;
			case 18:
				/* request window size */
				{
#ifdef KERNEL
					short width, height;
					GetWindowSize( dp->winp, &width, &height );
					my_sprintf( dp, "\033[8;%d;%dt",
						height / dp->cheight, width / dp->cwidth );
#else
					struct wstate ws;
					GetWindowState( dp->winp, &ws );
					my_sprintf( dp, "\033[8;%d;%dt",
						ws.height / dp->cheight, ws.width / dp->cwidth );
#endif
				}
				break;
			case 19:
				/* request pad size */
				my_sprintf( dp, "\033[9;%d;%dt",
					dp->cursor.nlines, dp->cursor.ncols );
				break;
			default:
				/* invalid command */
				break;
			}
		case 'y':
			/* invoke confidence test */
			if( p0 == 2 ) {
				/* do it */
			}
			break;
		case 'n':
			/* device status report */
			switch( p0 ) {
			case 5:
				/* status report, always reply 0=okay never 3=!okay */
				my_sprintf( dp, "\033[0n" );
				break;
			case 6:
				/* cursor position report */
				my_sprintf( dp, "\033[%d;%dR", dp->cursor.line+1,dp->cursor.col+1 );
				break;
			default:
				break;
			}
			break;
		case 'J':
			/* erase in display */
			ed( dp, p0 );
			break;
		case 'K':
			/* erase in line */
			el( dp, p0 );
			break;
		case 'l':
			/* reset mode */
			if( p0 == 20 ) {
				/*line feed*/
			}
			else if( p0 == QUES_PARMVAL ) {
				switch( p1 ) {
				case 1:
					/*cursor key application mode*/
					dp->mode.application_cursorkey = FALSE;
#ifdef KERNEL
					(*keybsw[dp->vttyp->v_te].k_ctl)
							(dp->vttyp->v_keyb, KCURMOD, 0);
#endif
					break;
				case 3:
					/*80 column mode*/
					dp->mode.size_fixed = TRUE;
					dp->cursor.line		= 0;
					dp->cursor.col		= 0;
					pad_fill( dp, ' ',
						0, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );
					adjust_pad( dp, NLINES80, NCOLS80 );
					break;
#ifdef NOT_IMPLEMENTED
				case 4:
					/*smooth scroll*/
					dp->mode.smooth_scroll = FALSE;
					break;
#endif
				case 5:
					/*screen mode reverse*/
					if( dp->mode.reverse_screen ) {
						/* unreverse the screen */
						dp->mode.reverse_screen = FALSE;
						InvertRegion( dp->winp, 0,0, 2000,2000 );
						SetColor( dp->winp, dp->fcolor );
						SetBColor( dp->winp, dp->bcolor );
					}
					break;
				case 6:
					/*origin relative*/
					dp->mode.relative_origin = FALSE;
					dp->cursor.line = 0;
					dp->cursor.col = 0;
					break;
				case 7:
					/*wraparound*/
					dp->mode.wraparound = FALSE;
					break;
#ifdef NOT_IMPLEMENTED
				case 8:
					/*auto repeat*/
					dp->mode.auto_repeat = FALSE;
					break;
				case 9:
					/*interlace*/
					dp->mode.interlace = FALSE;
					break;
#endif
				default:
					/*unknown mode*/
					break;
				}
			}
			break;
		case 'm':
			/* select graphics rendition */
			for( parm_index = 0; parm_index < parm_cnt; ++parm_index ) {
				switch( dp->parm.list[parm_index] ) {
				case DEF_PARMVAL:
				case 0:
					/* all atributes off */
					dp->char_attr.reverse = FALSE;
#ifdef KERNEL
					SetCharacterAttributes( dp->winp, CLEARBACK );
#else
					SetCharAttributes( dp->winp, VT_CLEARBACK );
#endif
					break;
				case 4:
				case 7:
					/* underscore on */
					/* reverse video on */
					dp->char_attr.reverse = TRUE;
#ifdef KERNEL
					SetCharacterAttributes( dp->winp, CLEARBACK | RVIDEO);
#else
					SetCharAttributes( dp->winp, VT_CLEARBACK | VT_RVIDEO);
#endif
					break;
#ifdef NOT_IMPLEMENTED
				case 1:
					/* bold on */
					break;
				case 5:
					/* blink on */
					break;
#endif
				default:
					/* ignore invalid value */
					break;
				}
			}
			break;
		case 'h':
			/* set mode */
			if( p0 == 20 ) {
				/*line feed/new line*/
			}
			else if( p0 == QUES_PARMVAL ) {
				switch( p1 ) {
				case 1:
					/*cursor key application mode*/
					dp->mode.application_cursorkey = TRUE;
#ifdef KERNEL
					(*keybsw[dp->vttyp->v_te].k_ctl)
							(dp->vttyp->v_keyb, KCURMOD, 1);
#endif
					break;
				case 3:
					/*132 column mode*/
					dp->mode.size_fixed = TRUE;
					dp->cursor.line		= 0;
					dp->cursor.col		= 0;
					pad_fill( dp, ' ',
						0, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );
					adjust_pad( dp, NLINES132, NCOLS132 );
					break;
#ifdef NOT_IMPLEMENTED
				case 4:
					/*smooth scroll*/
					dp->mode.smooth_scroll = TRUE;
					break;
#endif
				case 5:
					/*screen mode reverse*/
					if( !dp->mode.reverse_screen ) {
						/* unreverse the screen */
						dp->mode.reverse_screen = TRUE;
						InvertRegion( dp->winp, 0,0, 2000,2000 );
						SetColor( dp->winp, dp->bcolor );
						SetBColor( dp->winp, dp->fcolor );
					}
					break;
				case 6:
					/*origin relative*/
					dp->mode.relative_origin = TRUE;
					dp->cursor.line = dp->sr_top;
					dp->cursor.col = 0;
					break;
				case 7:
					/*wraparound on*/
					dp->mode.wraparound = TRUE;
					break;
#ifdef NOT_IMPLEMENTED
				case 8:
					/*auto repeat on*/
					dp->mode.auto_repeat = TRUE;
					break;
				case 9:
					/*interlace on*/
					dp->mode.interlace = TRUE;
					break;
#endif
				default:
					/*unknown mode*/
					break;
				}
			}
			break;
		case 'g':
			/* tabulation clear */
			switch( p0 ) {
			case DEF_PARMVAL:
			case 0:
				clear_tab( dp );
				break;
			case 3:
				clear_tabs( dp );
				break;
			default:
				break;
			}
			break;
		default:
			/* unknown escape sequence, eat it */
			break;
		}

		break; /*case SAW_LBRACKET*/

	case SAW_POUND:
		switch( c ) {
		case '3': /* double height line - top */
		case '4': /* double height line  - bottom */
		case '5': /* single-width line */
		case '6': /* double width line */
			*(pad_charp( dp, dp->cursor.line, LINESTYLE_COL )) = c - '0';
			/* add refresh line here */
			break;
		case '8': /* screen alignment display */
			dp->cursor.line	= 0;
			dp->cursor.col	= 0;
			dp->sr_top		= 0;
			dp->sr_bot		= dp->cursor.nlines-1;
			pad_fill( dp, 'E',
				0, 0, dp->cursor.nlines-1, dp->cursor.ncols-1 );
			refresh( dp, 0, 0, 2000, 2000 );
			break;
		default:
			/* unknown command, eat escape sequence */
			break;
		} /*switch*/
		break;

	case SAW_LPAREN:
		/* select g0 character set */
		break;

	case SAW_RPAREN:
		/* select g1 character set*/
		break;

	} /*switch*/

	/* IMPORTANT: Must return to maintain state.  Otherwise break to
	 * here state will be reset and cursor will be turned back on. */

	dp->parm.state = SAW_NEWJERSEY;

done:
	if( !dp->buffer.flushing ) {
		/* plop cursor back down */
		show_cursor( dp );
	}

	return;
}

/*
 *	BEGINNING OF MARK CODE
 */

/*
 *	STATIC FUNCTIONS
 */

/*
 *	mark_hilight - hilight from top(line,col) to bot(line,col).
 *
 */
static
mark_hilight( dp, top, bot )
register struct term_data *dp;
struct pad_location top, bot;
{
	register short line, first_col, last_col;

#ifdef KERNEL
#ifdef COLOR
	SetBColor( dp->winp, GRAY50 );
#else COLOR
	SetBPattern( dp->winp, gray87 );
#endif COLOR
#else KERNEL
	SetBColor( dp->winp, VT_Gray87 );
#endif KERNEL

	for( line = top.line; line <= bot.line; ++line ) {
		first_col	= (line==top.line) ? top.col : 0;
		last_col	= (line==bot.line  ) ? bot.col   :
			*(pad_charp(dp,line,CR_COL));
		if( last_col == NO_CR ) {
			last_col = dp->cursor.ncols-1;
		}
		dp->mark.call_hilight = FALSE;
		refresh( dp,
			first_col*dp->cwidth, line*dp->cheight + dp->yoff,
			((last_col-first_col) + 1)*dp->cwidth, dp->cheight );
		dp->mark.call_hilight = TRUE;
	}

#ifdef KERNEL
#ifndef COLOR
	SetBPattern( dp->winp, solid );
#endif COLOR
#endif KERNEL
	SetBColor( dp->winp, dp->bcolor );
}

/*
 *	mark_unlight - unlight from top(line,col) to bot(line,col).
 */
static
mark_unlight( dp, top, bot )
register struct term_data *dp;
struct pad_location top, bot;
{
	register short line, first_col, last_col;

#ifdef KERNEL
	SetBColor( dp->winp, WHITE );
#else KERNEL
	SetBColor( dp->winp, VT_White );
#endif KERNEL

	for( line = top.line; line <= bot.line; ++line ) {
		first_col	= (line==top.line) ? top.col : 0;
		last_col	= (line==bot.line  ) ? bot.col   :
			*(pad_charp(dp,line,CR_COL));
		if( last_col == NO_CR ) {
			last_col = dp->cursor.ncols-1;
		}
		dp->mark.call_hilight = FALSE;
		refresh( dp,
			first_col*dp->cwidth, line*dp->cheight + dp->yoff,
			((last_col-first_col) + 1)*dp->cwidth, dp->cheight );
		dp->mark.call_hilight = TRUE;
	}
}

/*
 *	mark_normalize - return normalized mark region
 */
static struct pad_bounds
mark_normalize( dp )
register struct term_data *dp;
{
	struct pad_bounds region;

	if( dp->mark.region.top.line > dp->mark.region.bot.line || 
		( dp->mark.region.top.line == dp->mark.region.bot.line &&
		  dp->mark.region.top.col > dp->mark.region.bot.col ) ) {
		/* top is past bot */
		region.top = dp->mark.region.bot;
		region.bot   = dp->mark.region.top;
		return( region );
	}
	return( dp->mark.region );
}

/*
 *	within - answers that all important question: Is "loc" within "region".
 */
static bool
within( region, loc )
struct pad_bounds region;
struct pad_location loc;
{
	if( loc.line >= region.top.line && loc.line <= region.bot.line ) {
		if( loc.line == region.top.line && loc.col < region.top.col ) {
			return( FALSE );
		}
		if( loc.line == region.bot.line && loc.col > region.bot.col ) {
			return( FALSE );
		}
		return( TRUE );
	}

	return( FALSE );
}

/*
 *	location_lt - Answers that all important question: Is first < sec?
 */
static bool
location_lt( first, sec )
struct pad_location first, sec;
{
	return( ( first.line < sec.line ) ||
			( first.line == sec.line && first.col < sec.col ) );
}

/*
 *	term_mark
 */
term_mark( dp, cmd, x, y )
register struct term_data *dp;
register short cmd, x, y;
{
	register unsigned char *linep;
	register unsigned char *charp;
	register unsigned char *last_colp;

	struct pad_location new_bot, mark_bot;
	struct pad_bounds norm;
	register short last_col;

	/* set line and col */

	new_bot.line = (y-dp->yoff)/dp->cheight;
	new_bot.col = x/dp->cwidth;

	/* locate last column, can't mark past it */
	linep = pad_linep( dp, new_bot.line );
	last_col = linep[ CR_COL ];
	if( last_col == NO_CR ) {
		last_col = dp->cursor.ncols-1;
	}
	if( new_bot.col > last_col ) {
		new_bot.col = last_col;
	}

	switch( cmd ) {

	case MARK_CANCEL:
		if( dp->mark.ing ) {
			/* start marking */
			dp->mark.ing = FALSE;
			norm = mark_normalize( dp );
			mark_unlight( dp, norm.top, norm.bot );
		}
		break;

	case MARK_REGION:
		/* mark to x, y */
		if( !dp->mark.ing ) {
			/* start marking */
			dp->mark.ing = TRUE;
			dp->mark.region.top = new_bot;
			dp->mark.region.bot = new_bot;
			mark_hilight( dp, new_bot, new_bot );
		}
		else if( dp->mark.region.bot.line != new_bot.line ||
				 dp->mark.region.bot.col  != new_bot.col   ) {

			/* the new bottom mark is different, move mark to new_bot */
			norm = mark_normalize( dp );

			if( (location_lt( dp->mark.region.top, dp->mark.region.bot ) &&
				 location_lt( new_bot, dp->mark.region.top ) ) ||
				(location_lt( dp->mark.region.bot, dp->mark.region.top ) &&
				 location_lt( dp->mark.region.top, new_bot ) ) )  {
				/* new_bot crossed top, erase existing mark then
				 * draw new mark */
				mark_unlight( dp, norm.top, norm.bot );
				dp->mark.region.bot = new_bot;
				norm = mark_normalize( dp );
				mark_hilight( dp, norm.top, norm.bot );
			}
			else {
				mark_bot = new_bot;
				if( within( norm, new_bot ) ) {
					/* new bot is within mark region, unlight */
					if( location_lt( new_bot, dp->mark.region.bot ) ) {
						/* unlight from column past new_bot to
						 * mark.region.bot */
						if( ++mark_bot.col > last_col ) {
							/* unlight from first column of next line */
							++mark_bot.line;
							mark_bot.col = 0;
						}
						mark_unlight( dp, mark_bot, dp->mark.region.bot );
					}
					else {
						/* unlight from column before new_bot to
						 * mark.region.bot */
						if( --mark_bot.col < 0 ) {
							/* unlight to last column of previous line */
							mark_bot.line--;
							mark_bot.col = 
								*(pad_charp(dp, mark_bot.line, CR_COL )); 
							if( mark_bot.col == NO_CR ) {
								mark_bot.col = dp->cursor.ncols-1;
							}
						}
						mark_unlight( dp, dp->mark.region.bot, mark_bot );
					}
				}
				else {
					/* new bot is outside mark region, hilight */
					if( location_lt( new_bot, dp->mark.region.bot ) ) {
						/* hilight from new_bot to column before
						 * mark.region.bot */
						if( --(dp->mark.region.bot.col) < 0 ) {
							/* unlight to last column of previous line */
							(dp->mark.region.bot.line)--;
							dp->mark.region.bot.col = 
								*(pad_charp( dp,dp->mark.region.bot.line,
												   CR_COL )); 
							if( dp->mark.region.bot.col == NO_CR ) {
								dp->mark.region.bot.col = dp->cursor.ncols-1;
							}
						}
						mark_hilight( dp, new_bot, dp->mark.region.bot );
					}
					else {
						/* hilight from new_bot to column after
						 * mark.region.bot */
						last_col = 
							*(pad_charp( dp,dp->mark.region.bot.line,
												   CR_COL )); 
						if( last_col == NO_CR ) {
							last_col = dp->cursor.ncols-1;
						}
						if( ++(dp->mark.region.bot.col) > last_col ) {
							/* hilight from first column of next line */
							++(dp->mark.region.bot.line);
							dp->mark.region.bot.col = 0;
						}
						mark_hilight( dp, dp->mark.region.bot, new_bot );
					}
				}
				dp->mark.region.bot = new_bot;
			}
		}
		break;

	case MARK_WORD:
		/* mark the word at x, y */
		if( dp->mark.ing ) {
			norm = mark_normalize( dp );
			/* unmark currently marked region */
			mark_unlight( dp, norm.top, norm.bot );
		}
		dp->mark.ing = TRUE;

		/* locate start of word to be marked */
		for( charp = &linep[ new_bot.col ];
			charp > linep && (*charp & 0x7f) != ' '; charp-- ) {
			/* null */
		}
		if( *charp = ' ' ) {
			/* don't include leading blank */
			++charp;
		}
		dp->mark.region.top.col = charp - linep;
		dp->mark.region.top.line = new_bot.line;

		/* locate end of word to be marked */
		last_colp = &linep[ last_col ];
		for( charp = &linep[ new_bot.col ];
			charp < last_colp && (*charp & 0x7f) != ' '; ++charp ) {
			/* null */
		}
		if( charp == last_colp && linep[ CR_COL ] != NO_CR ) {
			/* don't include trailing CR */
			charp--;
		}
		dp->mark.region.bot.col = charp - linep;
		dp->mark.region.bot.line = new_bot.line;

		/* hilight it */
		mark_hilight( dp, dp->mark.region.top, dp->mark.region.bot );
		break;

	case MARK_LINE:
		/* mark the line at x, y */
		if( dp->mark.ing ) {
			/* unmark currently marked region */
			norm = mark_normalize( dp );
			mark_unlight( dp, norm.top, norm.bot );
		}
		dp->mark.ing = TRUE;
		dp->mark.region.top.line = new_bot.line;
		dp->mark.region.top.col = 0;
		dp->mark.region.bot.line = new_bot.line;
		dp->mark.region.bot.col = last_col;
		mark_hilight( dp, dp->mark.region.top, dp->mark.region.bot );
		break;

	default:
		/* invalid marking command */
		break;

	}
}

/*
 *	term_mark_stop
 */
unsigned char *
term_mark_stop( dp )
register struct term_data *dp;
{
	struct pad_bounds norm;

	register bool add_cr;
	register short line, first_col, last_col, cr_col;
	register short nchars;

	register unsigned char *linep, *blockp, *p;

	norm = mark_normalize( dp );

	if( !dp->mark.ing ) {
		/* nothing marked */
		return( NULL );
	}

	/* count characters in marked region */
	nchars = 0;
	for( line = norm.top.line; line <= norm.bot.line; ++line ) {
		cr_col = *(pad_charp(dp,line,CR_COL));
		first_col	= (line==norm.top.line) ? norm.top.col : 0;
		last_col	= (line==norm.bot.line  ) ? norm.bot.col : cr_col;
		if( last_col == NO_CR ) {
			last_col = dp->cursor.ncols-1;
		}
		nchars += (last_col - first_col)+1;
	}

	blockp = (unsigned char *) CreateBlock( nchars+1 );
	if( blockp == NULL ) {
		/* couldn't malloc block for mark buffer */
		return( NULL );
	}

	p = blockp;
	for( line = norm.top.line; line <= norm.bot.line; ++line ) {
		linep		= pad_linep( dp, line );
		cr_col		= linep[CR_COL];
		first_col	= (line==norm.top.line) ? norm.top.col : 0;
		last_col	= (line==norm.bot.line  ) ? norm.bot.col : cr_col;

		if( last_col == NO_CR ) {
			last_col = dp->cursor.ncols-1;
		}

		nchars = (last_col - first_col)+1;
		strncpy( p, &linep[ first_col ], nchars );

		for( ; nchars; nchars--, ++p ) {
			*p &= 0x7f;
		}

		if( last_col == cr_col ) {
			*(p-1) = '\r';
		}
	}
	*p = '\0';

	/* unmark region */
	term_mark( dp, MARK_CANCEL );

	return( blockp );
}

/*
 *	adjust
 */
static bool
adjust( dp, w, h )
register struct term_data *dp;
register short w, h;
{
	SetPermanentClipping( dp->winp, 0, 0, w, h);

	if( dp->mode.size_fixed ) {
		register short delta_h = h - dp->height;
		register short delta_w = w - dp->width;
		dp->yoff = h - ( dp->cheight * dp->cursor.nlines );

		if( delta_h > 0 ) {
			/* window height increased, copy contents down and refresh
			 * strip at top */

			register short yoff = ( dp->yoff>0 ) ? dp->yoff : 0;

			CopyRegion( dp->winp, 0,0, 0,delta_h,
				dp->width, dp->height );

			SetColor( dp->winp,
				dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
			SetPosition( dp->winp, 0, 0 );
			PaintRectangleInterior( dp->winp, dp->width, yoff+delta_h );
			SetColor( dp->winp,
				dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );

			refresh( dp, 0, 0, dp->width, yoff+delta_h );
		}
		else if( delta_h < 0 ) {
			/* window height decreased, copy contents up and refresh
			 * strip at bottom */
			CopyRegion( dp->winp, 0,-delta_h, 0,0,
				dp->width, dp->height+delta_h );
			refresh( dp, 0, dp->height+delta_h, w, -delta_h );
		}
		if( delta_w > 0 ) {
			/* refresh right edge */
			refresh( dp, dp->width,0, delta_w, dp->height );
		}
		dp->width = w;
		dp->height = h;
	}
	else {
		register short nlines = h/dp->cheight;
		register short ncols = w/dp->cwidth;
		dp->yoff = h % dp->cheight;
		if( nlines == 0 || ncols == 0 ||
			( nlines == dp->cursor.nlines && ncols == dp->cursor.ncols ) ) {
			return;
		}
		if( ncols > MAX_NCOLS ) {
			ncols = MAX_NCOLS;
		}

		dp->width = w;
		dp->height = h;

		if( !adjust_pad( dp, nlines, ncols ) ) {
			return( FALSE);
		}

		SetColor( dp->winp, dp->mode.reverse_screen ? dp->fcolor : dp->bcolor );
		SetPosition( dp->winp, 0, 0 );
		PaintRectangleInterior( dp->winp, w, h );
		SetColor( dp->winp, dp->mode.reverse_screen ? dp->bcolor : dp->fcolor );

		refresh( dp, 0, 0, w, h );
	}

	return( TRUE );
}

/*
 *	refresh
 */
static
refresh( dp, x, y, w, h )
register struct term_data *dp;
short x, y, w, h;
{
	register unsigned char *charp;
	register unsigned char *char_limitp;
	register unsigned char *stringp;
	register unsigned char hi_bit;
	register short line, last_line;
	register short yoff = dp->yoff;
	register short cheight = dp->cheight;
	register short cwidth = dp->cwidth;
	short char_start, char_limit, limit_col;

#ifdef KERNEL
	short xc, yc, wc, hc;
	GetPermanentClipping( dp->winp, &xc, &yc, &wc, &hc );
#else
	struct wstate window_state;
	GetWindowState( 1, &window_state );
#endif

	SetPermanentClipping( dp->winp, x, y, w, h );


	line=(y-yoff)/cheight;
	if( line < 0 ) {
		line = 0;
	}
	last_line=((y-yoff)+h)/cheight;
	if( last_line < 0 ) {
		last_line = 0;
	}
	else if( last_line > dp->cursor.nlines ) {
		last_line = dp->cursor.nlines-1;
	}

	char_start = x/cwidth;
	char_limit = (x+w)/cwidth + 1;
	if( char_limit > dp->cursor.ncols ) {
		/* limit is off pad, set to end of pad */
		char_limit = dp->cursor.ncols;
	}

	for( ; line<=last_line; line++ ) {

		SetPosition( dp->winp,
			char_start*cwidth, line*cheight+dp->cbaseline+yoff );

		charp = pad_linep( dp, line );

		limit_col = charp[ CR_COL ];
		if( limit_col == NO_CR ) {
			limit_col = dp->cursor.ncols;
		}
		else {
			++limit_col;
		}
		char_limitp = &charp[ (char_limit<limit_col)?char_limit:limit_col ];

		charp += char_start;
		stringp = charp;
		hi_bit = *charp & 0x80;
		for( ; charp < char_limitp; ++charp ) {

			/* look at all characters in a line */

			if( hi_bit != (*charp & 0x80 ) ) {

				/* display string each time hi bit changes */

#ifdef KERNEL
				SetCharacterAttributes( dp->winp,
					(CLEARBACK | (hi_bit ? RVIDEO : 0 )) );
#else
				SetCharAttributes( dp->winp,
					(VT_CLEARBACK | (hi_bit ? VT_RVIDEO : 0 )) );
#endif
				PaintString( dp->winp, charp-stringp, stringp );
				BumpXPosition( dp->winp, (charp-stringp)*cwidth );
				hi_bit = *charp & 0x80;
				stringp = charp;

			}
		}
		/* display characters left at end of line */
#ifdef KERNEL
		SetCharacterAttributes( dp->winp,
			(CLEARBACK | (hi_bit ? RVIDEO : 0 )) );
#else
		SetCharAttributes( dp->winp,
			(VT_CLEARBACK | (hi_bit ? VT_RVIDEO : 0 )) );
#endif
		PaintString( dp->winp, charp-stringp, stringp );

	}

	if( dp->mark.ing && dp->mark.call_hilight ) {
		struct pad_bounds norm;
		norm = mark_normalize( dp );
		mark_hilight( dp, norm.top, norm.bot );
	}

	if( dp->cursor.visible ) {
		InvertRegion( dp->winp, 
			dp->cursor.col*cwidth, dp->cursor.line*cheight + yoff,
			cwidth, dp->cheight );
	}

#ifdef KERNEL
	SetCharacterAttributes( dp->winp,
		(CLEARBACK | (dp->char_attr.reverse ? RVIDEO : 0 )) );
	SetPermanentClipping( dp->winp, xc, yc, wc, hc );
#else
	SetWindowState( 1, &window_state );
#endif

}
