/*#define DEBUG	/* */
/*
 *	Terminal emulator for graphic mode windows
 *
 *	Escape sequence parser by Jeff Mason, NBI; 1983
 */
#include "../h/param.h"		/* system parameters */
#include "../h/systm.h"
#include "../h/errno.h"
#include "../h/buf.h"		/* defines for useracc */
#include "../vt/vt_hdrs.h"	/* virtual terminal parameters */
#include "../vt/vt_output.h"
#include "../vt/vt_kernel.h"	/* kernel only vt defines */
#include "../vt/vt_fonts.h"

/* 
 * defines used for escape state
 */
#define		NSTATE  5	/* number of states as above */

#define		DULL	0	/* simple output char state */
#define		SESC	1	/* start esc sequence read */
#define		GETFUN	2	/* escape read, get function name */
#define		VARGS   3	/* varaiable number of args to call */
#define		GETARG  4	/* function read, get arguments now */

#define		FIRSTBYTE 0x40  /* only one byte has been read */
#define		NSTRS   128     /* length of a string of chars */

#define	NEARGS		130	/* max number args for esc sequence */

/* 
 * 	Structure used in each virtual terminal to hold arguments and function
 *	names for outputing escape sequences. 
 */
struct	escbuf	{
    unsigned char   indx;	/* index of function for call in esctab*/
	     char   numa;	/* number of arguments still needed  */
    unsigned char   vptr;	/* index into val for next code */
	     char   dummy;	/* used for alignment problems */
    unsigned char   val[NEARGS];/* actual arguments to this sequence */
};

struct buffer {
	struct tty	*vp;
	struct escbuf	esc;	    /* escape sequence buffer */
	unsigned char	estate;   /* current output state for vt */
	short		index;
	char		strbuf[NSTRS+1];
	struct { short	x, y, w, h} refresh;
#ifdef COLOR
	struct { short val, type} locallut[LUTSIZE];	/* local LUT */
#endif COLOR
};

/* 	macro used to decide if a key is displayable (return = 0), local control
 * 	(return > 0) or nonlocal token (return = -1) 
 */
#define vthash(c) ((c) < 0x20 ? ((c) + 1) : (((c) >= 0x80 && (c) <= 0xa0) ? \
		(-1) : (0)))


extern PATTERN		v_color[];  /* different colors for system */
extern short		globallut[];
extern struct trackcursor trackcursor; /* cursor tracking box */
extern int	mousestate;
extern short	minx, miny, maxx, maxy, mousex, mousey;

	/* output states */
unsigned char		esctab[NSTATE] = {
		DULL,	GETFUN, DULL/*INVALID*/, VARGS, GETARG
};
	
struct	 seqcall 	etab[] = {	/* table of output escape sequeunces*/
    SetPosition_n,		SetPosition,		/* 0 */
    SetXPosition_n,		SetXPosition,		/* 1 */
    SetYPosition_n,		SetYPosition,		/* 2 */
    BumpPosition_n,		BumpPosition,		/* 3 */
    BumpXPosition_n,		BumpXPosition,		/* 4 */
    BumpYPosition_n,		BumpYPosition,		/* 5 */
    PaintRectangleInterior_n,	PaintRectangleInterior,	/* 6 */
    PaintRectangleBorder_n,	PaintRectangleBorder,	/* 7 */
    CopyRegion_n,		CopyRegion,		/* 8 */
    MoveLocalCursor_n,		MoveLocalCursor,	/* 9 */
    HideLocalCursor_n,		HideLocalCursor,	/* 10 */
    ShowLocalCursor_n,		ShowLocalCursor,	/* 11 */
    SendNote_n,			Sendtones,		/* 12 */
    SetFont_n,			SetFont,		/* 13 */
    SetPattern_n,		SetPattern,		/* 14 */
    SetJustification_n,		SetJustification,	/* 15 */
    SetThickness_n,		SetThickness,    	/* 16 */
    SetPhase_n,			SetPhase,    		/* 17 */
    SetAddressing_n,		SetAddressing,		/* 18 */
    InvertRegion_n,		InvertRegion,		/* 19 */
    PaintPolygonInterior_n,	PaintPolygonInterior,	/* 20 */
    PaintLine_n,		PaintLine,		/* 21 */
    PaintCircleBorder_n,	PaintCircleBorder,	/* 22 */
    PaintCircleInterior_n,	PaintCircleInterior,	/* 23 */
    PaintArcBorder_n,		PaintArcBorder,		/* 24 */
    PaintArcInterior_n,		PaintArcInterior,	/* 25 */
    PaintString_n,		PaintString,		/* 26 */
    SetCharAttributes_n,	SetCharacterAttributes,	/* 27 */
    PaintIcon_n,		PaintIcon,		/* 28 */
    SetBPattern_n,		SetBPattern,		/* 29 */
    SetColor_n,			SetColor,		/* 30 */
    SetBColor_n,		SetBColor		/* 31 */
};

char *
gr_init(vp)
register struct tty	*vp;
{
	register struct buffer	*bp;
	int	refreshsig(), adjustsig();
	register short	i;

#ifdef DEBUG
cprintf("gr_init\n");
#endif DEBUG
	bp = (struct buffer *)CreateBlock(sizeof(struct buffer));
	if(bp){
	    bp->vp = vp;
	    vp->t_flags |= RAW;
	    vp->t_flags &= ~ECHO;
	    (*keybsw[vp->v_te].k_ctl)(vp->v_keyb, KMMASK, VT_MOUSE_DOWN);
	    bp->index = 0;
	    bp->estate = DULL;
#ifdef COLOR
	    for (i=0; i<LUTSIZE; i++){
		bp->locallut[i].val = i;
		bp->locallut[i].type = VT_FREE;
	    }
#endif COLOR
	    if( vp->v_win != (int *)DesktopWindow()){
		SetRefresh( vp->v_win, bp, refreshsig);
		SetAdjust(  vp->v_win, bp, adjustsig);
	    }
	}
	return( (char *)bp);
}

gr_exit(bp)
register struct buffer	*bp;
{
	register short	i;

#ifdef DEBUG
cprintf("gr_exit\n");
#endif DEBUG
    if(bp){
	bp->vp->t_flags |= ECHO;
	bp->vp->t_flags &= ~RAW;
	(*keybsw[bp->vp->v_te].k_ctl)(bp->vp->v_keyb, KMMASK, 0);
#ifdef COLOR
	for (i=0; i<LUTSIZE; i++){
	    switch(bp->locallut[i].type){
	    case VT_OWN:
		globallut[bp->locallut[i].val] = VT_FREE;
		break;
	    case VT_SHARE:
		globallut[bp->locallut[i].val]--;
		break;
	    }
	}
#endif COLOR

	DestroyBlock(bp);
    }
}

gr_disp(bp, c)
register struct buffer	*bp;
register unsigned char	c;
{

    if((c == CS_DISPLAY_TOKEN) && (bp->estate == DULL)) {
	bp->estate = SESC;	/* set state to start escape sequence */
	if(bp->index){
#ifdef DEBUG
cprintf("finishing pstring\n");
#endif DEBUG
	    bp->strbuf[bp->index] = '\0';
	    PaintString(bp->vp->v_win, -1, bp->strbuf);
	    bp->index = 0;
	}
	return;
    }

    switch(esctab[bp->estate & 0x7]) {
    case DULL :				/* regular char display     */
	    if (vthash(c) == 0) {	/* displayable character ? */
		bp->strbuf[bp->index++] = c;	/* save in string */
		/* output string if need be */
		if (bp->index == NSTRS) {
		    bp->strbuf[bp->index] = '\0';
		    PaintString(bp->vp->v_win, -1, bp->strbuf);
		    bp->index = 0;
		}
	    }
#ifdef DEBUG
	     else{
cprintf("character (0x%x) in not displayable\n", c);
	    }
#endif DEBUG
	break;			/* end of dull old char display     */

    case GETFUN :			/* read function index              */
	if (c > VT_NSEQ) {		/* valid function ?		    */
	    bp->estate = DULL;	/* nope, forget it all              */
	} else {
	    bp->esc.indx = c;	/* save index into etab of function */
	    bp->esc.vptr = 0;	/* set pointer into val array       */
	    bp->esc.numa = (etab[c].numa << 1);
	    /* set state for read args or read number of args */
	    if (bp->esc.numa >= 0) {
		bp->estate = GETARG;	/* set state for read of args */
		if (bp->esc.numa == 0) outesc(bp);  /* output if enough */
	    } else {
		bp->estate = VARGS;	/* state to get # args */
	    }
	}
	break;

    /* get the number of arguments for this function */
    case VARGS :
	    /* save argument in buffer */
	    bp->esc.val[bp->esc.vptr++] = c;
	    /* have both bytes for # args been read */
	    if (bp->estate & FIRSTBYTE) {
		/* sure is honey, get # in args counter and clear index */
		bp->estate &= ~FIRSTBYTE;
		bp->esc.numa = *((short *) bp->esc.val);
		bp->esc.vptr = 0;
		/* adjust for specific function */
		if (bp->esc.indx == PaintPolygonInterior_f)
		    bp->esc.numa <<= 2;
		else 
		    bp->esc.numa += 2;
		/* check to avoid too many args */
		if (bp->esc.numa > NEARGS-1) bp->esc.numa = NEARGS - 1;
		bp->estate = GETARG;
	    } else {
		bp->estate |= FIRSTBYTE;
	    }
	    break;

    /* read next esc arg (a byte) and display if done */	
    case GETARG:
	bp->esc.val[bp->esc.vptr++] = c;     /* save next arg	    */
	if (--bp->esc.numa == 0) outesc(bp);   /* output if enough    */
	break;
    }	/* end of display action switch     */
}

gr_ioctl(bp, cmd, data, flag)
register struct buffer * bp;
caddr_t	data;
{
    register struct tty *vp = bp->vp;
    register short	*shp = (short *)data;
    unsigned short	font;	/* actual font */
    COLORT		fcolor, bcolor; /* colors of window */

    switch(cmd){
    /*
     * Set current mouse mode 
     */
    case TIOVSETMM: {

	(*keybsw[vp->v_te].k_ctl)(vp->v_keyb, KMMASK, *shp);
	break;
    }


     /*
     * Set or Restrict the clipping bounds
     */
    case TIOVSETC : {
	register struct clipbnd * bnds = (struct clipbnd *) data;
#ifdef DEBUG
cprintf("touching clipping\n");
#endif DEBUG
    	switch(bnds->type) {
	    case SetPerm :
		SetPermanentClipping(vp->v_win, bnds->x, bnds->y,
		    bnds->w, bnds->h);
		break;
	    case ResPerm :
		RestrictPermanentClipping(vp->v_win, bnds->x, bnds->y,
		    bnds->w, bnds->h);
		break;
	    case SetTemp :
		SetTemporaryClipping(vp->v_win, bnds->x, bnds->y,
		    bnds->w, bnds->h);
		break;
	    case ResTemp :
		RestrictTemporaryClipping(vp->v_win, bnds->x, bnds->y,
		    bnds->w, bnds->h);
		break;
	    default :
		break;
	}
        break;
    }
	
    /*
     * Get the clipping bounds
     */
    case TIOVGETC : {
	register struct clipbnd * bnds = (struct clipbnd *) data;
    	switch(bnds->type) {
	    case GetPerm :
		GetPermanentClipping(vp->v_win, &(bnds->x), &(bnds->y),
		    &(bnds->w), &(bnds->h));
		break;
	    case GetTemp :
		GetTemporaryClipping(vp->v_win, &(bnds->x), &(bnds->y),
		    &(bnds->w), &(bnds->h));
		break;
	    case GetRefr :
		bnds->x = bp->refresh.x;
		bnds->y = bp->refresh.y;
		bnds->w = bp->refresh.w;
		bnds->h = bp->refresh.h;
		bp->refresh.w = bp->refresh.h = 0;  /* set so new sig can go */
		break;
	   default :
	       break;
	}
	break;
    }


    /*
     * Get full window state
     */
    case TIOVGSTATE: {
	register struct wstate * wp = (struct wstate *) data;
	register i;
	PATTERN	fpat, bpat;
	
#ifdef DEBUG
cprintf("getting window state\n");
#endif DEBUG
	GetCurrentColors(vp->v_win, &fcolor, &bcolor);
	GetCurrentPatterns(vp->v_win, &fpat, &bpat);
	wp->fcolor = wp->bcolor = 0;
	for (i = 0; i < NCOLOR; i++) {
	    if (v_color[i] == fpat) wp->fcolor = -i;
	    if (v_color[i] == bpat) wp->bcolor = -i;
        }
#ifdef COLOR
	if(wp->fcolor >= -1) 
	    for (i = 0; i < LUTSIZE; i++){
		if (bp->locallut[i].val == fcolor){
		    wp->fcolor = i;
		    break;
		}
	    }
	if(wp->bcolor >= -1)
	    for (i = 0; i < LUTSIZE; i++){
		if (bp->locallut[i].val == bcolor){
		    wp->bcolor = i;
		    break;
		}
	    }
#else COLOR
	if(wp->fcolor >= -1) wp->fcolor = fcolor;
	if(wp->bcolor >= -1) wp->bcolor = bcolor;
#endif COLOR

	GetCurrentFont(vp->v_win, &font);
	wp->font = font;

	GetCurrentPosition(vp->v_win, &(wp->xpos), &(wp->ypos));
	GetCurrentAddressing(vp->v_win, &(wp->addressing));
	GetStringAttributes(vp->v_win,&(wp->justification), &(wp->orientation),
	     &(wp->direction), &(wp->ctype));
	GetCurrentPhase(vp->v_win, &(wp->xphase), &(wp->yphase));
	GetLineAttributes(vp->v_win,&(wp->thickness),&(wp->mask),&(wp->count));
	GetWindowSize(vp->v_win, &(wp->width), &(wp->height));
	GetTemporaryClipping(vp->v_win, &(wp->tx), &(wp->ty), &(wp->tw), 
	    &(wp->th));
	GetPermanentClipping(vp->v_win, &(wp->px), &(wp->py), &(wp->pw), 
	    &(wp->ph));
	wp->mmode = (*keybsw[vp->v_te].k_ctl)(vp->v_keyb, KGMASK, 0);
	wp->ldisc = vp->t_line;
	wp->cvisible = GetCursorVisibility(vp->v_win);
        break;
    }

#ifdef COLOR
    case TIOVSLLUT: {
	register struct colorlut *cp = (struct colorlut *) data;
	int	c;

	if (cp->color >= 0 && cp->color < LUTSIZE){
		c = bp->locallut[cp->color].val;
		if (cp->type != VT_FREE){
		    if (bp->locallut[cp->color].type < 0){
			/* we already own this color, just load it in */
			if (cp->type == VT_SHARE){
			    bp->locallut[cp->color].type = VT_SHARE;
			    globallut[c] = VT_SHARE;
			}
			LoadGlobalLUT(c, cp->red, cp->green, cp->blue);
		    }else if (bp->locallut[cp->color].type == VT_FREE){
			/* this slot is free, so find a hardware entry */
			c = findcolor(cp->type,cp->red,cp->green, cp->blue);
			if (c>=0){
			    bp->locallut[cp->color].type = cp->type;
			    bp->locallut[cp->color].val = c;
			    LoadGlobalLUT(c, cp->red, cp->green, cp->blue);
			    if (cp->type == VT_SHARE)
				globallut[c]++;
			    else
				globallut[c] = VT_OWN;
			}else{
			    return(ENXIO);
			}
		    }else{
			/* this color is already shared */
			if (globallut[c] == 1){
			    c = findcolor(cp->type,cp->red,cp->green, cp->blue);
			    if ( c >= 0){
				globallut[bp->locallut[cp->color].val]--;
				bp->locallut[cp->color].val = c;
				bp->locallut[cp->color].type = cp->type;
				LoadGlobalLUT(c, cp->red, cp->green, cp->blue);
				if(cp->type == VT_SHARE)
				    globallut[c]++;
				else
				    globallut[c] = VT_OWN;
			    }else{
				/* we have the only refrence, so change it */
				bp->locallut[cp->color].type = cp->type;
				globallut[bp->locallut[cp->color].val] 
							= cp->type;
				LoadGlobalLUT(c, cp->red, cp->green, cp->blue);
			    }
			}else{
			    c = findcolor(cp->type,cp->red,cp->green, cp->blue);
			    if ( c >= 0){
				globallut[bp->locallut[cp->color].val]--;
				bp->locallut[cp->color].val = c;
				bp->locallut[cp->color].type = cp->type;
				LoadGlobalLUT(c, cp->red, cp->green, cp->blue);
				if(cp->type == VT_SHARE)
				    globallut[c]++;
				else
				    globallut[c] = VT_OWN;
			    }else{
				return(ENXIO);
			    }
			}
		    }
		}else{
		    /* new type is free */
		    if(bp->locallut[cp->color].type != VT_FREE){
			if (globallut[c] > 0){
			    /* we were sharing this */
			    globallut[c]--;
			}else{
			    /* we owned this color, no more */
			    globallut[c] = VT_FREE;
			}
			bp->locallut[cp->color].val = cp->color;
			bp->locallut[cp->color].type = VT_FREE;
		    }
		}
	}else{
	    return(ENXIO);
	}
	break;
    }

    case TIOVGLLUT: {
	register struct colorlut *cp = (struct colorlut *) data;
	if (cp->color >= 0 && cp->color < LUTSIZE){
	    GetGlobalLUT(bp->locallut[cp->color].val, 
			    &cp->red, &cp->green, &cp->blue);
	    cp->type = bp->locallut[cp->color].type;
	}
	break;
    }
#endif COLOR

    default:
	return(-1);
    }
    return(0);
}

outesc(bp)		/* routine to output valid escape sequence to screen*/
register struct	buffer	* bp;
{

    struct yep {
	short arg1;
	short arg2;
	short arg3;
	short arg4;
	short arg5;
	short arg6;
    };
    register struct yep * huhp;    /* pointer into escape buffer */
    register unsigned char indx;   /* function number */

    huhp = (struct yep *) bp->esc.val;
    indx = bp->esc.indx;

    if ((indx == SetPattern_f) || (indx == SetBPattern_f)) {
	if(huhp->arg1 <2){		/* use real colors instead */
	    (*etab[indx].func)(bp->vp->v_win,solid);
	    if( indx == SetPattern_f)
		indx = SetColor_f;
	    else
		indx = SetBColor_f;
	    if(huhp->arg1 <2)		/* black and white are reversed */
		    huhp->arg1 = huhp->arg1 ? 0 : 1;
	    (*etab[indx].func)(bp->vp->v_win,huhp->arg1);
	}else{
	    (*etab[indx].func)(bp->vp->v_win,v_color[huhp->arg1]);
	}
    }else if ((indx == SetColor_f) || (indx == SetBColor_f)) {
#ifdef COLOR
	(*etab[indx].func)(bp->vp->v_win, bp->locallut[huhp->arg1].val);
#else 
	(*etab[indx].func)(bp->vp->v_win, huhp->arg1);
#endif COLOR
	if( indx == SetColor_f)
	    indx = SetPattern_f;
	else
	    indx = SetBPattern_f;
	if(huhp->arg1 < 0){		/* use simulated colors (patterns) */
	    (*etab[indx].func)(bp->vp->v_win,v_color[-huhp->arg1]);
	}else{
	    (*etab[indx].func)(bp->vp->v_win, solid);
	}
    } else  if (indx == PaintPolygonInterior_f) {
	/* get number of shorts in x verticies list */
	indx = (bp->esc.vptr >> 1);
        PaintPolygonInterior(bp->vp->v_win, indx >> 1, &huhp->arg1, 
		&bp->esc.val[indx]);
    } else if (indx == PaintString_f) {
	/* get number of chars in string */
	indx = (bp->esc.vptr - 2);
	bp->esc.val[indx + 1] = 0;
	PaintString(bp->vp->v_win, huhp->arg1, &huhp->arg2);
    } else {
	(*etab[indx].func)(bp->vp->v_win,huhp->arg1, huhp->arg2,
			huhp->arg3, huhp->arg4, huhp->arg5, huhp->arg6);
    }			/* end of switch for # of args needed*/
    bp->estate = DULL;	/* reset for read of next simple char*/
}				/* end of outputting escape sequeunce*/

/*
 *	Routine to send a refresh signal to the process group of the process
 *	which needs a window refresh
 */
refreshsig(bp, x, y, w, h)
register struct buffer * bp;
int x,y,w,h;				/* area to refresh */
{
	register short x2, y2;

	/* if no bounds set then set them and send signal */
	if (bp->refresh.w == 0) {
	    bp->refresh.x = x;
	    bp->refresh.y = y;
	    bp->refresh.w = w;
	    bp->refresh.h = h;
	    gsignal(bp->vp->t_pgrp, SIGREF);
	/* otherwise find smallest bounding rectangle and save as bounds */ 
	} else {
	    x2 = bp->refresh.x + bp->refresh.w;
	    y2 = bp->refresh.y + bp->refresh.h;
	    if (x < bp->refresh.x) bp->refresh.x = x;
	    if (y < bp->refresh.y) bp->refresh.y = y;
	    if (x + w > x2) bp->refresh.w = x + w - bp->refresh.x;
	    else	    bp->refresh.w = x2 - bp->refresh.x;
	    if (y + h > y2) bp->refresh.h = y + h - bp->refresh.y;
	    else	    bp->refresh.h = y2 - bp->refresh.y;
	}
}

/*
 *	send an adjust signal to the process of this vt 
 */
adjustsig(bp, w, h)
register struct buffer * bp;
{
    gsignal(bp->vp->t_pgrp, SIGADJ);
}

