/*#define DEBUG /* */
/*#define XDEBUG /* */
#include "vty.h"
#if NVTY > 0 || Nvty > 0
/* 
 * Virtual terminal driver for virtual terminals called directly
 * from cdevsw.  The device argument to each of the calls is parsed
 * to give the following information - physical terminal (for now
 * this is hard coded to cons) and the window number (minor macro
 * used to get this).  These numbers are then used to write to and/or
 * manipulate windows.
 */


#include "../h/param.h"		/* system parameters */
#include "../h/systm.h"		/* system sturcts et al */
#include "../h/dir.h"		/* directories and files */
#include "../h/user.h"		/* user block */
#include "../h/file.h"		/* file structure */
#include "../h/proc.h"		/* process strucutures */
#include "../h/inode.h"		/* inodes */
#include "../h/conf.h"		/* configuration */
#include "../h/buf.h"		/* buffers */
#include "../h/dk.h"		/* disk stuff */
#include "../h/uio.h"		/* user input/output struct */
#include "../h/kernel.h"	/* need lbolt and hz */
#include "../h/signal.h"	/* signal defines */
#include "../h/errno.h"		/* error defines */
#include "../vt/vt_hdrs.h"	/* virtual terminal parameters */
#include "../vt/vt_fonts.h"	/* font defines and structures */
#include "../vt/vt_output.h"	/* vt output defines */
#include "../vt/vt_kernel.h"	/* kernel only vt defines */
#include "../vt/vt_key.h"	/* external virtual key names */
#include "../vt/vt_keyin.h"	/* internal virtual key names */
#include "../is68kdev/pdma.h"	/* pseudo dma defs */
#include "../is68kdev/sioreg.h"	/* serial I/O register defs */
#include "../machine/board.h"
#include "../h/ttychars.h"
extern struct pdma sio_pdma[];

#define PADLED		3
#define MOUSELIMIT	16
extern unsigned char	mousetab[];    /* Mouse multiplication factor */
static short 	lastmousex, lastmousey;
extern short	minx, miny, maxx, maxy, mousex, mousey;
extern int	mousestate;
extern int	gpaddr;
extern short	consx, consy, consw, consh;

extern struct tty	vt_tty[];   /* system wide virtual terms */
extern short		globallut[];/* global lookup table status */
extern int  iconf[][128];	    /* icon faces */
extern int  iconm[][128]; 	    /* icon masks */

char flushvtok = 0;
extern int flushvtecho();
extern struct trans_tab gws_table;
extern struct padtrans_tab npad, fpad;

extern int *		v_color[];  /* different colors for system */
extern struct itable	v_itable[]; /* system defined icons */
extern struct statline v_status;    /* status line */
extern struct popquiz popquiz;      /* popup menu */
extern struct trackcursor trackcursor; /* cursor tracking box */

extern	int	mactive, kactive;

#ifdef XDEBUG
/* ***** DEBUGGING STUFF */
unsigned char	*pad[NVT+1], *padlimit[NVT+1];
#define NCOLS80			80		/* columns on 80 character display		*/
#define MAX_NCOLS		132		/* maximum number of columns allowed	*/
#define MAX_NTBYTES		(MAX_NCOLS/8 + ((MAX_NCOLS%8)?1:0))


#define PARMLIST_SIZE	8		/* number of entries in parm.list		*/
#define BUFFER_SIZE		NCOLS80	/* number of entries in buffer.data		*/

struct pad_location {
	short line, col;
};

struct pad_bounds {
	struct pad_location top, bot;
};

struct term_data {

	struct tty *vttyp;				/* vtty for this emulator				*/

#ifdef KERNEL
	int *winp;						/* window for this emulator				*/
	COLORT fcolor, bcolor;			/* foreground and background color		*/
#else
	int winp;
	VT_COLOR fcolor, bcolor;
#endif

	short yoff,						/* offset to keep display anchored at 	*/
									/* bottom of window						*/
		cwidth,						/* width, height and baseline of a		*/
		cheight,					/* character in the current font		*/
		cbaseline;

	short width,					/* current width and height of window	*/
		height;

	struct {
		bool call_hilight;			/* flag to stop infinite recursion		*/
		bool ing;					/* marking is on						*/
		struct pad_bounds region;	/* bounds of marked region				*/
	} mark;

	struct {
		bool visible;				/* is cursor currently visible?			*/
		short line, col;			/* cursor position						*/
		short sav_line, sav_col;	/* saved cursor position				*/
		short nlines, ncols;		/* number of lines and columns on		*/
									/* screen (bounds for cursor)			*/
	} cursor;

	short sr_top, sr_bot;			/* scrolling region top and bottom		*/

	unsigned char keyb_leds;		/* keyboard LED bit mask				*/

	struct {
		bool size_fixed;			/* the number of lines and columns are	*/
									/* fixed, if unfixed the pad is auto-
									/* matically sized to match the window	*/
		bool application_cursorkey;
		bool reverse_screen;
		bool relative_origin;
		bool wraparound;
		bool wrap_on_next_char;		/* used internally when in wraparound	*/
		bool application_keypad;
#ifdef NOT_IMPLEMENTED
		bool smooth_scroll;
		bool auto_repeat;
		bool interlace;
#endif
	} mode;

	struct {
		bool reverse;
#ifdef NOT_IMPLEMENTED
		bool blink;
		bool bold;
		bool underscore;
#endif
	} char_attr;

	struct {
		enum {
			SAW_NEWJERSEY, SAW_ESC, SAW_LBRACKET, SAW_POUND,
			SAW_LPAREN, SAW_RPAREN
		} state;
		short power;
		short *listp;
		short list[PARMLIST_SIZE];
	} parm;

	struct {
		short cache_cnt;			/* cached character count				*/
		short cache_col;			/* column on pad of first cached char	*/
		unsigned char *p;			/* pointer to pad						*/
		unsigned char *first_linep; /* pointer to first pad line 			*/
		unsigned char *limitp;		/* pointer to first character off pad	*/
	} pad;

	struct {
		bool active;
		bool flushing;
		unsigned char *datap;
		unsigned char data[BUFFER_SIZE];
	} buffer;

	unsigned char tabs[ MAX_NTBYTES ];
};
#endif XDEBUG

#define ISPEED 	B9600
#define IFLAGS	(EVENP|ODDP|ECHO|CRMOD|CRTBS|CRTERA|CRTKIL|CTLECH)

vtopen(dev, flag)
register dev_t	dev;			/* device number */
int		flag;
{
	register struct tty * vp;	/* pointer to a vt */
	register	       wnum, i;	/* window number and counter */
	int	vtstart();
	int	error;

#ifdef DEBUG /* */
cprintf("vtopen(%x, %x) pid %d pgrp %d\n",dev,flag,
			u.u_procp->p_pid,u.u_procp->p_pgrp);
#endif /* */

	flushvtok = 1;			/* enable type ahead demon */
	/* find window number and return if not valid */
	wnum = minor(dev);
	if (wnum > NVT - 1)
	    return(ENOVT);
	if(!gpaddr)
	    return(ENXIO);
	vp = &vt_tty[wnum];

	/* check to see if this vt is already open */
	if (flag&FEXCL && vp->t_state&(TS_WOPEN|TS_ISOPEN))
		return(EEXIST);

	vp->t_oproc = vtstart;
	vp->t_state |= TS_WOPEN;

	if((vp->t_state&TS_ISOPEN)==0){
	    kactive = wnum;
	    ttychars(vp);
	    vp->t_ispeed = vp->t_ospeed = ISPEED;
	    vp->t_flags = IFLAGS;
	    /* if this is the user agent vt, then init screen et al */
	    if (wnum == 0) {
		/* zero out all other vts */
		/*for (vp = &vt_tty[1]; vp <= &vt_tty[NVT]; vp++)
			vp->t_state = 0;*/
		/*vp = &vt_tty[0];	/* back to the desktop!!! */
		for(i=1; i<=NVT; i++)
		    if(vt_tty[i].t_state & TS_ISOPEN)
			vtclose(i);

		/* init screen, give desktop as window and set a mouse cursor */
		setupvt(vp);
		SetCharacterAttributes(vp->v_win, CLEARBACK);	/* for icons */

		/* set up default terminal emulator */
		/* WARNING, start keyboard before output terminal! */
		vp->v_keyb = (*keybsw[vp->v_te=deskem].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
	    /* define regular default window */
	    } else {
		vp->v_win = (int *)CreateWindow(2000,2000,10,10,0,NORMAL);
		gsignal(vt_tty[0].t_pgrp, SIGREF);	/* tell dm about it */
		vtopen1(vp);
	    } 
	}else if (vp->t_state&TS_XCLUDE && u.u_uid != 0)
		return (EBUSY);

	vp->t_state |= TS_CARR_ON;
	
	error = (*linesw[vp->t_line].l_open)(dev, vp);
	if(error){
		return(error);
	}else{
		if(vp->t_pgrp == 0)
			vp->t_pgrp = u.u_procp->p_pgrp;
	}
	return(0);
}

vtopen1(vp)
register struct tty * vp;	/* pointer to a vt */
{
	extern int 	localmask[],globalmask[], localface[],globalface[];

#ifdef DEBUG
cprintf("vtopen1\n");
#endif DEBUG
	/* set up default cursors */
	DefineGlobalCursor(vp->v_win, globalmask, globalface, 5, 3, 0);
	DefineLocalCursor(vp->v_win, localmask, localface, 8, 20, 0);  

	/* set up default fonts, colors and position */
	SetFont(vp->v_win,DEFWINDOWFONT);
	SetColor(vp->v_win,BLACK);
	SetBColor(vp->v_win,WHITE);
	SetPosition(vp->v_win, 0, 0); 

	/* set up default terminal emulator */
	/* WARNING, start keyboard before output terminal! */
#ifdef DEBUG
cprintf("initialized: ");
#endif DEBUG
	vp->v_keyb = (*keybsw[vp->v_te=defterm].k_init)(vp);
#ifdef DEBUG
cprintf("keyboard, ");
#endif DEBUG
	vp->v_term = (*termsw[vp->v_te].t_init)(vp);
#ifdef XDEBUG
	pad[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.p;
	padlimit[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.limitp;
#endif XDEBUG
#ifdef DEBUG
cprintf("emulator; done.\n");
#endif DEBUG

}	/* end of vtopen1 */

vtclose(dev, flag)
register dev_t dev;		/* device */
{
        register struct tty * vp;		/* virtual termnial */
        register	       wnum,i;		/* window number and counter */
	extern struct mark	vt_marks[];

#ifdef DEBUG
cprintf("vtclose(0x%x, 0x%x) pid %d\n", dev, flag, u.u_procp->p_pid);
#endif

	/* find window number and return if not valid */
	wnum = minor(dev);
	if (wnum > NVT)  
	    return(ENOVT);

	vp = &vt_tty[wnum];
	if ((vp->t_state & TS_ISOPEN) == 0)
	    return(ENXIO);

	/* clean up if this window is in the middle of a mark */
	if (vt_marks[wnum].req&MARKING){
	    vt_marks[wnum].req = 0;
	    vp->t_state &= ~TS_BUSY;
	    (*termsw[vp->v_te].t_mark) (vp->v_term, MARK_CANCEL, 0, 0);
	}
	(*linesw[vp->t_line].l_close)(vp);
	ttyclose(vp);
	/* if this is the desktop then clean up screen and die */
    	if (wnum == 0) {
	    unsetup(vp);
	/* otherwise make sure desktop manager is active */
	} else { 
	    /* get rid of any emulators */
	    (*termsw[vp->v_te].t_exit)(vp->v_term);
	    (*keybsw[vp->v_te].k_exit)(vp->v_keyb);
	    /* remove window from screen */
	    if (wnum == mactive)
		mactive = 0; 
	    if (wnum == kactive)
		kactive = 0; 
	    DestroyWindow(vp->v_win);
	    gsignal(vt_tty[0].t_pgrp, SIGREF);	/* tell dm about it */
	}
	return(0);
}

vtread(dev, uio)
dev_t dev;			/* calling device */
register struct uio * uio;	/* user io structure */
{
	register struct tty    *vp = &vt_tty[minor(dev)];

#ifdef notdef
	/* more debugging stuff */
	int	x;
	cprintf("vtread(0x%x) t_line = %d\n", dev, vp->t_line);
	cprintf("vp = 0x%x, u.u_ttyp = 0x%x\n", vp, u.u_ttyp);
	cprintf("u.pgrp = %d, t.pgrp = %d\n", u.u_procp->p_pgrp, vp->t_pgrp);
	x = (*linesw[vp->t_line].l_read)(vp, uio);
	cprintf("returning(%d)\n", x);
	return(x);
#endif

	return ((*linesw[vp->t_line].l_read)(vp, uio));
}

vtwrite(dev, uio)
dev_t dev;			/* calling device */
struct uio *uio;		/* user io structure */
{

	register struct tty    *vp = &vt_tty[minor(dev)];
	
	return ((*linesw[vp->t_line].l_write)(vp, uio));
}

vtioctl(dev, cmd, data, flag)
caddr_t	data;
{
	register struct tty *vp = &vt_tty[minor(dev)];
	register short	*shp = (short *)data;
	int	error;
	
	if((termsw[vp->v_te].t_ioctl) &&
	((error = (*termsw[vp->v_te].t_ioctl)(vp->v_term, cmd, data, flag)) 
	    >=0))
		return(error);
	switch(cmd){
	case TIOVSETE : 
		if((*data < 0) || (*data >= nterms))
			return(ENOVT);
		(*termsw[vp->v_te].t_exit)(vp->v_term);
		(*keybsw[vp->v_te].k_exit)(vp->v_keyb);
		vp->v_keyb = (*keybsw[vp->v_te=(*data)].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
		break;
	case TIOCSETD : 
	    if(vp->t_line == *(int*)data) break;
	    /* soon this will be the old way to change emulators */
	    if ((vp->t_line != TWSDISC) && (*(int *) data == TWSDISC)) {
		(*termsw[vp->v_te].t_exit)(vp->v_term);
		(*keybsw[vp->v_te].k_exit)(vp->v_keyb);
		vp->v_keyb = (*keybsw[vp->v_te=1].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
	    } else if ((vp->t_line == TWSDISC) && (*(int *)data != TWSDISC)) {
		(*termsw[vp->v_te].t_exit)(vp->v_term);
		(*keybsw[vp->v_te].k_exit)(vp->v_keyb);
		vp->v_keyb = (*keybsw[vp->v_te=defterm].k_init)(vp);
		vp->v_term = (*termsw[vp->v_te].t_init)(vp);
#ifdef XDEBUG
pad[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.p;
padlimit[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.limitp;
#endif XDEBUG
	    }
	    return(ttioctl(vp, cmd, data, flag));
	/*
	 * Get some of the window state
	 */
	case TIOVGSTATE: {
	    register struct wstate * wp = (struct wstate *) data;
	    register i;
	    unsigned short	font;
	    COLORT	fcolor, bcolor;
	    PATTERN	fpat, bpat;
	    
	    GetCurrentColors(vp->v_win, &fcolor, &bcolor);
	    GetCurrentPatterns(vp->v_win, &fpat, &bpat);
	    wp->fcolor = wp->bcolor = 0;
	    for (i = 0; i < NCOLOR; i++) {
		if (v_color[i] == fpat) wp->fcolor = -i;
		if (v_color[i] == bpat) wp->bcolor = -i;
	    }
	    if(wp->fcolor >= -1) wp->fcolor = fcolor;
	    if(wp->bcolor >= -1) wp->bcolor = bcolor;

	    GetCurrentFont(vp->v_win, &font);
	    wp->font = font;

	    GetCurrentPosition(vp->v_win, &(wp->xpos), &(wp->ypos));
	    GetCurrentAddressing(vp->v_win, &(wp->addressing));
	    GetStringAttributes(vp->v_win,&(wp->justification), 
		&(wp->orientation), &(wp->direction), &(wp->ctype));
	    GetCurrentPhase(vp->v_win, &(wp->xphase), &(wp->yphase));
	    GetLineAttributes(vp->v_win,&(wp->thickness),&(wp->mask),
		&(wp->count));
	    GetWindowSize(vp->v_win, &(wp->width), &(wp->height));
	    GetTemporaryClipping(vp->v_win, &(wp->tx), &(wp->ty), &(wp->tw), 
		&(wp->th));
	    GetPermanentClipping(vp->v_win, &(wp->px), &(wp->py), &(wp->pw), 
		&(wp->ph));
	    wp->mmode = (*keybsw[vp->v_te].k_ctl)(vp->v_keyb, KGMASK, 0);
	    wp->ldisc = vp->t_line;
	    wp->cvisible = GetCursorVisibility(vp->v_win);
	    break;
	}

	case TIOVGETHW: {
	    register struct gconfig *cp = (struct gconfig *) data;
	    cp->w = SCREENW; cp->h = SCREENH;
	    cp->x_res = 90; cp->y_res = 90;
#ifdef COLOR
	    cp->realcolors = LUTSIZE;
	    cp->lutsize = LUTSIZE;
	    cp->nred    = 16;
	    cp->ngreen  = 16;
	    cp->nblue   = 16;
#else
	    cp->realcolors = 2;
	    cp->lutsize = 0;
	    cp->nred    = 0;
	    cp->ngreen  = 0;
	    cp->nblue   = 0;
#endif
	    cp->nwindows = NVT;
	    cp->nistyles = NISTYLES;
	    cp->nicons   = NICON;
	    break;
	}

	/*
	 * Get Character Attributes
	 */
	 case TIOVGCHR : {
	    COLORT	fcolor;
	    register struct charatt * cp = (struct charatt *) data;
	    switch (cp->type) {
		case Cbaseline :
		    cp->val=CharacterBaseline(GetFontPlane(cp->font,DEFPLANE));
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;
		case Cheight :
		    cp->val=CharacterHeight(GetFontPlane(cp->font, DEFPLANE));
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;
		case Cwidth :
		    cp->val=CharacterWidth(GetFontPlane( cp->font, DEFPLANE), 
					     cp->val);
		    DoneWithPlane( cp->font, DEFPLANE);
		    break;
		case Ctype :
		    GetStringAttributes(vp->v_win, &fcolor, &fcolor, &fcolor,
			&cp->val); 
		    break;
		default :
		    break;
	    }
	    break;
	}
	/*
	 * Put a title on the window
	 */
	case TIOVSETT :
	    SetWindowTitle(vp->v_win,data);	    /* set title */
	    break;

	/*
  	 * Change the emulator font
	 */
	case TIOVSETEF: {
	    register short *sp = (short *) data;
	    if(termsw[vp->v_te].t_font)
		    (*termsw[vp->v_te].t_font)(vp->v_term, *sp);
	    break;
	}

	/* 
	 * Display a pop-up menu, and return the selected item
	 */
	case TIOVPOPQ : 
	    *data = DisplayPopQuiz(vp, data);
	    break;

		
	case TIOVTRCKM : {
	    register struct trackmouse * t = (struct trackmouse *) data;
	    short	x, y;
	    short	min_x, min_y, max_x, max_y;

	    GetWindowPosition( vp->v_win, &x, &y);
	    min_x = minx; min_y = miny;
	    max_x = maxx; max_y = maxy;
	    while( trackcursor.type)	/* somebody else is tracking cursor */
		/* don't interrupt me */
		sleep( (caddr_t)&trackcursor.type, PZERO);
	    trackcursor.x = x + t->x;
	    trackcursor.y = y + t->y;
	    trackcursor.w = t->w;
	    trackcursor.h = t->h;
	    trackcursor.thickness = t->thickness;

	    switch(t->type){
	    case TrackFixed:
		trackcursor.type = TrackFixed;
		trackcursor.xoffset = (x + t->x) - mousex;
		trackcursor.yoffset = (y + t->y) - mousey;
		minx = (x + t->bx) - trackcursor.xoffset;
		miny = (y + t->by) - trackcursor.yoffset;
		maxx = (x + t->bx + t->bw) - t->w - trackcursor.xoffset;
		maxy = (y + t->by + t->bh) - t->h - trackcursor.yoffset;
		if( (mousestate&7) != 7){
		    sleep( (caddr_t)&trackcursor.type, PZERO);
		}
		t->x = trackcursor.x - x;
		t->y = trackcursor.y - y;
		break;
	    case TrackRubber:
		trackcursor.type = TrackRubber;
		trackcursor.xoffset = (x + t->x + t->w) - mousex;
		trackcursor.yoffset = (y + t->y + t->h) - mousey;
		minx = (x + t->bx) - trackcursor.xoffset;
		miny = (y + t->by) - trackcursor.yoffset;
		maxx = (x + t->bx + t->bw) - trackcursor.xoffset;
		maxy = (y + t->by + t->bh) - trackcursor.yoffset;
		if( (mousestate&7) != 7){
		    sleep( (caddr_t)&trackcursor.type, PZERO);
		}
		t->w = trackcursor.w;
		t->h = trackcursor.h;
		break;
	    }
	    trackcursor.type = 0;
	    RemoveGlobalRubberBox();
	    minx = min_x; miny = min_y;
	    maxx = max_x; maxy = max_y;
	    break;
	}
	    
	/*
	 * Define Cursors
	 */
	case TIOVDEFC : {
	    register struct curpat * cp = (struct curpat *) data;
	    int	mask[32], face[32];

	    if( copyin(cp->face, face, 128) || copyin(cp->mask, mask, 128)){
		return(EFAULT);
	    }
	    if (cp->type == LocalCur) {
		DefineLocalCursor(vp->v_win, mask, face, cp->xref,
		    cp->yref, cp->blink);
	    } else if (cp->type == GlobalCur) {
		DefineGlobalCursor(vp->v_win, mask, face, cp->xref,
		    cp->yref, cp->blink); 
	    }
	    break;
	}

	case TIOVRASTOP:{
	    register struct rasterop *rp = (struct rasterop *) data;
	    RASTER uraster;
	    register short	*first;
	    register int	count;
	    short jnk;
	    extern int	ToggleURaster(), CopyURaster(), PaintURaster();

	    first = (short *)((char *)rp->ras + (rp->sy * rp->width) + (rp->sx>>3));
	    count = ((rp->h - 1) * rp->width) + (rp->w>>3);

	    if (!useracc(first, count, B_READ)){
		    return(EFAULT);
	    }
    /* this code should lock n at a time ...... CTH */
	    vslock(first, count);

	    uraster.address = rp->ras;
	    uraster.width = rp->width;

	    switch( rp->type){
	    case COPYRASTER:
		DoRasterOp(vp->v_win, CopyURaster, &uraster, rp->sx, rp->sy, 
			   rp->x, rp->y, rp->w, rp->h);
		break;
	    case TOGGLERASTER:
		DoRasterOp(vp->v_win, ToggleURaster, &uraster, rp->sx, rp->sy, 
			   rp->x, rp->y, rp->w, rp->h);
		break;
	    case PAINTRASTER:
		DoRasterOp(vp->v_win, PaintURaster, &uraster, rp->sx, rp->sy, 
			   rp->x, rp->y, rp->w, rp->h, rp->color);
		break;
	    }
	    vsunlock(first, count, B_WRITE);	/* device write, memory read */
	    break;
	}
	/*
	 *	Set the console open size/position
	 */
	case TIOUSETCP : {
	    consx = *shp++;
	    consy = *shp++;
	    consw = *shp++;
	    consh = *shp;
	    break;
	}
	/*
	 *	Display/delete status line
	 */
	case TIOUSSTAT : {
	    register unsigned char * sp;
	    register unsigned char *sp1;
	    register i;

		/* point to characters and copy in */
		sp = (unsigned char *) shp;
		sp1 = v_status.message;
		for (i = 0; i < 128; i++)
		    *sp1++ = *sp++;
		v_status.visible = 1;
		DisplayStatusLine();
		break;
	}

	case TIOUDSTAT :
		EraseStatusLine();
		break;
	    
	/*
	 * Turn on/off rubber baby band boxes
         */
	case TIOUMVBOX:
		if (((struct clipbnd *) data)->type == BabyBox) {
		    HideGlobalCursor();		/* hide cursor for ua */
		    DisplayGlobalRubberBox(((struct clipbnd *) data)->x,
				       ((struct clipbnd *) data)->y,
				       ((struct clipbnd *) data)->w,
				       ((struct clipbnd *) data)->h,2);
		}
		break;

	case TIOUNOBOX:
		RemoveGlobalRubberBox();	/* kill baby box of rubber */
		ShowGlobalCursor();
		break;

	/*
	 * Get the internal vt number for a window
         */
	case TIOUGETWN:
		*(short *)data = vp - vt_tty;
		break;

	/*
	 * Get the pgrp of a window
         */
	 case TIOUWPGRP:{
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return -1 if trying to get bad window pgrp */
		wp->val1 = -1;
		if ((wp->win < 0) || (wp->win > NVT)) break;
		if(wp->win == NVT){	/* is this how we say its the console */
		    wp->val1 = -2;
		    break;
		}
		if (vt_tty[wp->win].t_state & TS_ISOPEN) 
		    wp->val1 = vt_tty[wp->win].t_pgrp;
		break;
	    }

	/*
	 * Set Pane Colors
	 */
         case TIOUSETPC :
		/* return if trying to change bad window panes */
		if ((*shp < 0) || (*shp > NVT)) break;
		vp = &vt_tty[*shp++];
		if ((vp->t_state & TS_ISOPEN) == 0) break;
		SetPaneColors(vp->v_win, *shp, *(shp +1), *(shp + 2));
		break;
	/*
	 * Set a new active vt
	 */
	case TIOUNEWVT:
		/* return if trying to change bad window panes */
		if ((*shp < 0) || (*shp > NVT)) break;
		kactive = *shp;
		mactive = *shp;
		break;

	/*
 	 * Set mouse clipping boundries
	 */
	case TIOUMBND :
		/* just copy the little buggers in */
		minx = *shp++;
		maxx = *shp++;
		miny = *shp++;
		maxy = *shp;

		/* clip the mouse if needed */
		if (mousex < minx) mousex = minx;
		if (mousey < miny) mousey = miny;
		if (mousex > maxx) mousex = maxx;
		if (mousey > maxy) mousey = maxy;
		break;

	/*
	 * Get/Set a windows size
	 */
       case TIOUGETWS: {
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return if trying to change bad window panes */
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) return(ESRCH);

		/* no quiz window */
		GetWindowSize(vp->v_win, &(wp->val1), &(wp->val2));
		break;
	}

	case TIOUSETWS:  {
	    register struct uwindow * wp = (struct uwindow *) data;
	    struct font_plane	*charinfo;

		/* return if trying to change bad window panes */
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) break;
		DoneWithPlane( DEFNONPROPFONT, DEFPLANE);
		ChangeWindowSize(vp->v_win, wp->val1, wp->val2);
		break;
	}
	    
	/*
	 * Get/set a windows position
	*/
	case TIOUGETWP: {
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return if trying to change bad window panes */
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) return(ESRCH);

		GetWindowPosition(vp->v_win, &(wp->val1), &(wp->val2));
		break;
	}

	case TIOUSETWP: {
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return if trying to change bad window panes */
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) break;

		ChangeWindowPosition(vp->v_win, wp->val1, wp->val2);
		break;
	}

	/*
  	 *  Find the top window
         */
	case TIOUTOPW: {
	    register	w;

#ifdef DEBUG
cprintf("TopWindow[");
#endif DEBUG
	    for (w=0; w<=NVT; w++){
		if (vt_tty[w].t_state & TS_ISOPEN){
#ifdef DEBUG
cprintf("%d ", w);
#endif DEBUG
		    GetWindowDepth(vt_tty[w].v_win, shp); /* check its depth */
		    if (!*shp) break;
		}
	    }
#ifdef DEBUG
cprintf("]\n");
#endif DEBUG
	    *shp = w;
	    break;
	}

	/*
	 * Get/set a windows depth
	 */
	case TIOUGETWD: {
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return -1 if trying to change bad window panes */
		wp->val1 = -1;
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) return(ESRCH);

		GetWindowDepth(vp->v_win, &(wp->val1));
		break;
	}

        case TIOUSETWD: {
	    register struct uwindow * wp = (struct uwindow *) data;
		/* return if trying to change bad window panes */
		if ((wp->win < 0) || (wp->win > NVT)) break;
		vp = &vt_tty[wp->win];
		if ((vp->t_state & TS_ISOPEN) == 0) break;
		ChangeWindowDepth(vp->v_win,wp->val1);
		gsignal(vt_tty[0].t_pgrp, SIGREF);
		break;
	}
	
	/*
	 *	Configuration ioctl's
	 */

	case TIOUGMC:	{
	    register struct mconfig *cp = (struct mconfig *) data;
	    register int	i;
	    extern unsigned char Mdelta, mousetab[];
	    
	    cp->mc_delta = Mdelta;
	    for( i=0; i<=VT_MTABSIZE; i++){
		cp->mc_table[i] = mousetab[i];
	    }
	    break;
	}

	case TIOUSMC:	{
	    register struct mconfig *cp = (struct mconfig *) data;
	    register int	i;
	    extern unsigned char Mdelta, mousetab[];

	    Mdelta = cp->mc_delta;
	    for( i=0; i<=VT_MTABSIZE; i++){
		mousetab[i] = cp->mc_table[i];
	    }
	    break;
	}

#ifdef COLOR
	case TIOUSGLUT: {
	    register struct colorlut *cp = (struct colorlut *) data;
	    LoadGlobalLUT(cp->color, cp->red, cp->green, cp->blue);
	    break;
	}

	case TIOUGGLUT: {
	    register struct colorlut *cp = (struct colorlut *) data;
	    short	r, g, b;
	    GetGlobalLUT(cp->color, &r, &g, &b);
	    cp->red = r;
	    cp->green = g;
	    cp->blue = b;
	    break;
	}
#endif COLOR

#ifdef notdef
	case TIOUOCON:	{
	    /* make this window the console window */
	    if (!vt_cnpop) { /* there is no console already */
		vt_cnpop = minor(dev); /* mark this as it */
		vt_cnvp = vp;
	    } else { 
		/* already have one, error return */
		return(EEXIST);
	    }
	    break;
	}
#endif notdef
	case TIOUCCON:
	    /* close console window */
	    if (vt_tty[NVT].t_state & TS_ISOPEN) {
		return(vtclose(NVT, 0));
	    }else{
		return(ESRCH);
	    }

	case TIOUSETWF: {
	    register struct uwindow * wp = (struct uwindow *) data;

	    if ((wp->win < 0) || (wp->win > NVT)) break;
	    vp = &vt_tty[wp->win];
	    if(termsw[vp->v_te].t_font)
		    (*termsw[vp->v_te].t_font)(vp->v_term, wp->val1);
	    break;
	}

	    
	default:
	    if((error = (*linesw[vp->t_line].l_ioctl)(vp, cmd, data, flag)) >=0)
		    return(error);
	    if((error = ttioctl(vp, cmd, data, flag)) >=0)
		    return(error);
	    return(ENOTTY);
	}
	return(0);
}


/*
 *	This routine is called when the user agent vt is opened for the
 *	first time.  It sets up and initializes all the little things
 *	needed to deal with the screen 
 */
setupvt(tp)
register struct tty * tp;	/* data structures on the first vt open   */
{				
	register int    i,j;		/* counters */		
	WINDOW	    DesktopWindow();	/* window of desktop */
	extern int globalmask[], globalface[];
	extern struct tty sio_tty[];
	struct tty	*stp;
#ifdef DEBUG
cprintf("setupvt");
#endif

	/* initialize icon table */
	for (i = 0; i < NICON; i++) 
	    v_itable[i].iaddr = 0;

#ifdef COLOR
	/* initialize the global lookup table */
	globallut[0] = 1;		/* claim black */
	LoadGlobalLUT(0, 0, 0, 0);
	globallut[1] = 1;		/* and white */
	LoadGlobalLUT(1, 15, 15, 15);
	globallut[2] = 1;		/* also gray50 */
	LoadGlobalLUT(2, 11, 11, 11);
	for (i=3; i<LUTSIZE; i++){
	    globallut[i] = VT_FREE;
	}
#endif COLOR

	/* give desktop as tty window, clear background & set up refresh */
	tp->v_win = (int *) DesktopWindow();
	v_status.visible = FALSE;
	consx = (SCREENW - CONSW)>>1;
	consy = (SCREENH - CONSH)>>1;
	consw = CONSW;
	consh = CONSH;

	/* set up the mouse and keyboard (console and port 1) to work */
	/* with the desktop */
	/* kludge kludge kludge kludge (city) !!!!! */
	sioinit();
	for( i=0; i<NSIO; i++){
		int	siostart();

		stp = &sio_tty[i];
		stp->t_addr = (caddr_t)&sio_pdma[i];
		/* stp->t_oproc = siostart;*/
		if ((tp->t_state&TS_ISOPEN) == 0) {
			stp->t_state |= TS_ISOPEN;
			stp->t_flags = (EVENP|ODDP|ECHO|XTABS|CRMOD);
		} 
		/* for 0 = kb, 1 = ms */
		if(i==0){
			switch (*BSR & BSR_BAUD) {
	    		case BSR_BAUD_9600:
				stp->t_ispeed = B9600;
				break;
	    		case BSR_BAUD_19200:
				stp->t_ispeed = EXTA;
				break;
	    		case BSR_BAUD_1200:
				stp->t_ispeed = B1200;
				break;
	    		case BSR_BAUD_300:
				stp->t_ispeed = B300;
				break;
			}
			stp->t_ospeed = stp->t_ispeed;
			stp->t_line = KYBDDISC;
		} else {
			stp->t_ospeed = stp->t_ispeed = B1200;
			stp->t_line = MOUSDISC;
		}
		sioparam(i);
	}

	/* show cursor, set up mouse bounds and keyboard state */
	minx = miny = 0;
	maxx = SCREENW;
	maxy = SCREENH;
	DefineGlobalCursor(tp->v_win, globalmask, globalface, 5, 3, 0);
	MoveGlobalCursor(mousex, mousey);
	ShowGlobalCursor();
	SetActiveCursor(tp->v_win);

#ifdef DEBUG
cprintf("\n");
#endif
}

/*
 *	Routine used when the last vt (user agent vt ) is closed to shut
 * 	down screen 
 */
unsetup(tp)			/* routine to close out everything  */
register struct tty * tp;
{
    register i;			/* counter */
    register struct tty *stp;
#ifdef DEBUG
cprintf("unsetup\n");
#endif

    /* zero out icon table */
    for (i = 0; i < NICON; i++) 
        v_itable[i].iaddr = 0;

    /* clear the whole screen */
#ifdef COLOR
    SetColor(tp->v_win, GRAY50);
#else COLOR
    SetPattern(tp->v_win, gray50);
#endif COLOR
    SetPosition(tp->v_win, 0, 0);
    PaintRectangleInterior(tp->v_win, SCREENW, SCREENH);
#ifndef COLOR
    SetPattern(tp->v_win, solid);
#endif 
    consx = (SCREENW - CONSW)>>1;
    consy = (SCREENH - CONSH)>>1;
    consw = CONSW;
    consh = CONSH;
}

/*
 *	Routine to do any type-ahead echo.  Called from switch.
 */
flushvtecho()
{
    register struct tty *vp;


    /* do the mouse first */
    updatemouse(1);

    /* and then echo for each of the other windows. */
    for( vp = &vt_tty[0]; vp <= &vt_tty[NVT]; vp++){
	if( (vp->t_state & TS_ISOPEN) && !(vp->t_state & TS_TTSTOP))
		(*vp->t_oproc)(vp); /* really vtstart(vp);*/
    }
}

/* 
 * update the mouse position
 */

updatemouse(flag)
int	flag;	/* update popups or trackboxes too! */
{
    if (lastmousex != mousex || lastmousey != mousey){
	
	if (flag && ((int)popquiz.win > 1)){
	    short absx = mousex;
	    short absy = mousey;
	    short new;
	    int relx, rely;

	    /* find the new choice */
	    if (WithinPane(popquiz.win,DISPLAY,absx,absy)){ 
		MapCoordinates(popquiz.win,DISPLAY,absx,absy,&relx,&rely);
		new = (rely / popquiz.cheight) + 1;
	    } else 
		new = 0;
	    if ((new != popquiz.current)/* && !cflag*/){
		if (popquiz.current){ /* unhighlight current */
		    InvertRegion(popquiz.win,0,popquiz.cheight*(popquiz.current-1),
				 1000, popquiz.cheight);
		}
		if (new){ /* hightlight new current */
		    InvertRegion(popquiz.win,0,popquiz.cheight*(new-1),
				 2000, popquiz.cheight);
		}
		popquiz.current = new;
	    }
	}else if (flag && trackcursor.type){
	    switch(trackcursor.type){
	    case 1 /* TrackFixed */:
		DisplayGlobalRubberBox(
			trackcursor.x = mousex + trackcursor.xoffset,
			trackcursor.y = mousey + trackcursor.yoffset,
			trackcursor.w, trackcursor.h, trackcursor.thickness);
		break;
	    case 2 /* TrackRubber */:
		DisplayGlobalRubberBox( trackcursor.x, trackcursor.y,
			trackcursor.w = (mousex + trackcursor.xoffset)
					- trackcursor.x,
			trackcursor.h = (mousey + trackcursor.yoffset)
					- trackcursor.y, trackcursor.thickness);
		break;
	    }
	}
	MoveGlobalCursor(mousex, mousey);
	lastmousex = mousex; lastmousey = mousey;
    }
}

/*
 * Routine used to start output from a virtual terminal to the screen. 
 * this routine will tell the correct terminal emulator when to buffer its 
 * output.  It will give up the processor if the output takes too long.
 */

vtstart(vp)
register struct tty *	vp;	/* this virtual terminal */
{
    register unsigned char c;
    int		s;

startout:
    if (vp->t_state&(TS_TIMEOUT|TS_BUSY|TS_TTSTOP))
	return;

    /* check to see that we aren't already running */	
    if (vp->v_state & VT_STARTED)
	return;

    /* if we are at interrupt priority return  */
    /* after waking up the type-ahead daemon */
    if(s = ((spl()>>8)&7)){
	wakeup((caddr_t)flushvtecho);
	return;
    }

    vp->v_state |= VT_STARTED;

    /* wakeup any sleepers */
    if (vp->t_outq.c_cc <= TTLOWAT(vp)) {
	    if (vp->t_state&TS_ASLEEP) {
		    vp->t_state &= ~TS_ASLEEP;
		    wakeup((caddr_t)&vp->t_outq);
	    }
	    if (vp->t_wsel) {
		    selwakeup(vp->t_wsel, vp->t_state & TS_WCOLL);
		    vp->t_wsel = 0;
		    vp->t_state &= ~TS_WCOLL;
	    }
    }

    if (vp->t_outq.c_cc == 0)
	goto out;

    /* let terminal emulator know its ok to buffer */
    if(termsw[vp->v_te].t_bufon)
	(*termsw[vp->v_te].t_bufon)(vp->v_term);

    while (vp->t_outq.c_cc > 0) {	/* display all of outq to screen    */
	/* quit if output has become stopped */
        /* hang here if ouptput is stopped */
    	if (vp->t_state & TS_TTSTOP){
	    /* tell the emulator to stop buffering */
	    if(termsw[vp->v_te].t_bufoff)
		(*termsw[vp->v_te].t_bufoff)(vp->v_term);
	    vp->v_state &= ~VT_STARTED;
	    return;
/*	    sleep((caddr_t)&vp->t_outq, TTOPRI);
	    goto startout;*/
	}

	updatemouse(0);
#ifdef notdef /* */
	if (runrun) {	/* give-up the processor */
		if(termsw[vp->v_te].t_bufoff)
		    (*termsw[vp->v_te].t_bufoff)(vp->v_term);
		vp->v_state &= ~VT_STARTED;
		return;
/*		goto out;*/
	}
#endif notdef /* */

	(*termsw[vp->v_te].t_disp)(vp->v_term, getc(&vp->t_outq));
    }

    if(termsw[vp->v_te].t_bufoff)
	(*termsw[vp->v_te].t_bufoff)(vp->v_term);
#ifdef XDEBUG
if ((vp->v_te == defterm) && 
    ((pad[vp - vt_tty] != ((struct term_data *)vp->v_term)->pad.p) ||
    (padlimit[vp-vt_tty] != ((struct term_data *)vp->v_term)->pad.limitp))){
	printf("pad[%d] changed: old(0x%x, 0x%x) now(0x%x, 0x%x)\n", vp-vt_tty,
	pad[vp-vt_tty], padlimit[vp-vt_tty], 
	((struct term_data *)vp->v_term)->pad.p, 
	((struct term_data *)vp->v_term)->pad.limitp);
	pad[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.p;
	padlimit[vp-vt_tty] = ((struct term_data *)vp->v_term)->pad.limitp;
}
#endif XDEBUG

out:
    wakeup((caddr_t)&vp->t_outq);
    vp->v_state &= ~VT_STARTED;
}
#endif
