#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/kernel.h"
#include "../h/file.h"
#include "../h/stat.h"
#include "../h/inode.h"
#include "../h/fs.h"
#include "../h/buf.h"
#include "../h/proc.h"
#include "../h/quota.h"
#include "../h/uio.h"
#include "../h/socket.h"
#include "../h/socketvar.h"
#include "../h/nami.h"
#include "../h/mount.h"
#include "../h/mbuf.h"

#include "../net/if.h"
#include "../netinet/in.h"
#include "../netinet/if_ether.h"

#include "../h/Wipc.h"

struct mbuf *WCatToMbuf();

int BarfPid;
unsigned long Mid;
unsigned long GetMid();
int rcode_seg[3] = {
	RCODE_USER_SEGMENT, RCODE_KERN_SEGMENT, RCODE_PHYS_SEGMENT
};

#define	MYPID	(Mid+(u.u_procp->p_pid == BarfPid ? 0xFFFF : u.u_procp->p_pid))

/*
 * Find an unreceived packet on the current process' receive queue
 *  from the process specified by rpid (zero means from anybody).
 *  If there is no such packet, wait for one.  Zero the PacketType
 *  field so that we don't try to receive it again.
 */
struct Wrproc *
FindNewRequest(rpid)
  register unsigned long rpid;
  {
	register struct Wrproc *wp, **wpp = &u.u_procp->p_Wrecq;
	int s = spl5();

	while (1) {
		for (wp = *wpp;  wp;  wp = wp->Wp_next)
			if (wp->Wp_ipc.ipcPacketType == IPC_SEND  &&
			    (rpid == 0  ||  wp->Wp_ipc.ipcSrcPid == rpid)) {
				wp->Wp_ipc.ipcPacketType = 0;
				splx(s);
				return wp;
			}
		sleep(wpp, PREC_WIPC);
	}
}

/*
 * Search the reply queue for a packet from the specified process.
 *  Any other packets encountered are discarded.
 *  If an appropriate packet is not found, wait a while
 *  until timeout occurs.  If an appropriate packet is found,
 *  it is removed from the queue.
 */
struct Wrproc *
GetReply(rpid, tm)
  unsigned long rpid;
  int tm;
  {
	register struct Wrproc *wp, **wpp = &u.u_procp->p_Wrepq;
	int s = spl5();
	int wakeup();

	if (rpid == 0xFFFFFFFF)
		rpid = 0;
	do {
		while (wp = *wpp) {
			*wpp = wp->Wp_next;
			if (wp->Wp_ipc.ipcSrcPid == rpid  ||
			    (rpid == 0  &&
			     wp->Wp_ipc.ipcPacketType == IPC_REPLY)) {
				splx(s);
				return wp;
			}
			Wrelse(wp);	/* ignore unexpected crap */
		}
		timeout(wakeup, (caddr_t)wpp, tm);
		sleep(wpp, PREP_WIPC);
		untimeout(wakeup, (caddr_t)wpp);
	} while (*wpp);
	splx(s);
	return NULL;
}

/*
 * Find the packet from the indicated process on our receive queue.
 *  Remove it from the receive queue.
 *  Caller is responsible for actually freeing the packet.
 */
struct Wrproc *
GetRequest(rpid)
  register unsigned long rpid;
  {
	register struct Wrproc *wp, *prev_wp =
					(struct Wrproc *)&u.u_procp->p_Wrecq;
	int s = spl5();

	while ((wp = prev_wp->Wp_next)  &&  wp->Wp_ipc.ipcSrcPid != rpid)
		prev_wp = wp;
	if (wp)
		prev_wp->Wp_next = wp->Wp_next;
	splx(s);
	return wp;
}

/*
 * Find the packet from the indicated process on our receive queue.
 */
struct Wrproc *
FindRequest(rpid)
  register unsigned long rpid;
  {
	register struct Wrproc *wp;
	int s = spl5();

	for (wp = u.u_procp->p_Wrecq;  wp;  wp = wp->Wp_next)
		if (wp->Wp_ipc.ipcSrcPid == rpid)
			break;
	splx(s);
	return wp;
}

/*
 * Returns pid of replying process, else 0.
 */
unsigned long
Send(urp, rpid)
  struct request *urp;
  register unsigned long rpid;
  {
	register struct request *rp;
	register unsigned short rcode, retry;
	register char *data;
	register int append, seg;
	int rval;
	struct ipc ipc;
	struct request msg_save;
#ifdef	ROLANE
	int s = spl7();
#ifndef	PSL_IPL
#define	PSL_IPL		0x00000700	/* interrupt priority level */
#endif	not PSL_IPL
	if ((s & PSL_IPL) >= 0x0600) {
		stacktrace();
		splx(s);
		panic("Send: hipri");
	}
	splx(s);
#endif	ROLANE

	if (Mid == 0  &&  (Mid = GetMid()) == 0)
		return u.u_error = EIO;
	if (rpid == 0xFFFF)
		rpid = Mid + BarfPid;
	if (rp = urp) {
		if (copyin(rp, &msg_save, sizeof u.u_Rmsg)  ||
		    !useracc(rp, sizeof *rp, B_WRITE)) {
			u.u_error = EFAULT;
			return 0;
		}
	} else
		msg_save = *(struct request *)u.u_Rmsg;	/* structure assign */
	rp = &msg_save;

  tryagain:
	rcode = rp->reqRequestCode;
	if (urp)
		FORCE_SEG_USER(rcode);
	data = rp->reqSegPtr;
	if ((append = rp->reqSegLength)  &&
	    (SEG_USER_READ(rcode) && !useracc(data, append, B_READ)) ||
	    (SEG_USER_WRITE(rcode) && !useracc(data, append, B_WRITE))) {
		u.u_error = EFAULT;
		return 0;
	}

	if (append  &&  SEG_READ(rcode)) {
		if (append > MAX_APPENDED_SEGMENT) {
			int maxlen = MaxPacket(rpid);
			if (append > maxlen)
				append = maxlen;
		}
		seg = rcode & RCODE_SEGMENT;
	} else {
		append = 0;
		data = (char *)0;
		seg = 0;
	}

	ipc.ipcPacketType = IPC_SEND;
	ipc.ipcSequenceNo = ++SeqNo;
	ipc.ipcSrcPid = MYPID;
	ipc.ipcDestPid = rpid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcDataLength = append;
	ipc.ipcSrcAddress = NULL;
	ipc.ipcDestAddress = NULL;

	retry = WRETRY;
	do {
		register struct Wrproc *wp;
		static int rtsec[WRETRY] = { 4, 5, 3, 4, 5 };

		SendKernelPacket(&ipc, rp, data, append, seg);
		if (rpid == 0xFFFF0000) {	/* broadcast; no reply */
			u.u_error = 0;
			return 0;
		}
		while (wp = GetReply(rpid, rtsec[WRETRY-retry] * (hz - 3))) {
			switch (wp->Wp_ipc.ipcPacketType) {
			  case IPC_BREATH_OF_LIFE:
				retry = WRETRY;	/* reset retry counter */
				break;
			  case IPC_COPY_TO_REQ:
				retry = WRETRY;	/* reset retry counter */
				reqCopyTo(wp, rp, rcode);
				break;
			  case IPC_COPY_FROM_REQ:
				retry = WRETRY;	/* reset retry counter */
				reqCopyFrom(wp, rp, rcode);
				break;
			  case IPC_REPLY:
				if (wp->Wp_ipc.ipcSequenceNo != ipc.ipcSequenceNo)
					break;
				if (reqReply(wp, rp, rcode))
					break;
				if (urp) {
					if (copyout(rp, urp, sizeof *urp)) {
						printf("Send: bad copyout\n");
						break;
					}
				} else
					*(struct request *)u.u_Rmsg = *rp;
				rval = wp->Wp_ipc.ipcSrcPid;
				Wrelse(wp);
				return rval;
			  case IPC_NACK:
				rval = reqNack(wp);
				Wrelse(wp);
				if (rval == ESRCH  &&
				    (rcode & RCODE_ANY_COUSIN)) {
					if (rpid = ReRpid(rpid))
						goto tryagain;
				}
				u.u_error = rval;
				return 0;
			}
			Wrelse(wp);
		}
	} while (--retry);
	u.u_error = EIO;
	return 0;
}

reqReply(wp, rp, rcode)
  register struct Wrproc *wp;
  register struct request *rp;
  register unsigned short rcode;
  {
	register char *destptr = wp->Wp_ipc.ipcDestAddress;
	register int len = wp->Wp_ipc.ipcDataLength;

	if (len) {
		if (!SEG_WRITE(rcode)  ||
		    destptr < rp->reqSegPtr  ||
		    destptr + len > rp->reqSegPtr + rp->reqSegLength) {
			Nack(wp, EFAULT);	/* remote invoker gets this */
			return EFAULT;
		}
		if (WCopyFromMbuf(wp, sizeof(struct Wrproc),  destptr, len,
					rcode & RCODE_SEGMENT) == 0) {
			printf("reqReply: bad copyout\n");
			Nack(wp, EFAULT);	/* tough luck, invoker */
			return EFAULT;
		}
	}
	*rp = wp->Wp_msg;
	return 0;
}

reqCopyTo(wp, rp, rcode)
  register struct Wrproc *wp;
  register struct request *rp;
  register unsigned short rcode;
  {
	register char *destptr = wp->Wp_ipc.ipcDestAddress;
	register int len = wp->Wp_ipc.ipcDataLength;
	struct ipc ipc;

/*	if (u.u_WSequenceNo != 0  &&
/*	    (short)wp->Wp_ipc.ipcSequenceNo - (short)u.u_WSequenceNo < 0)
/*		return;			/* ignore obsolete packets */
/* */
	if (!SEG_WRITE(rcode)  ||
	    destptr < rp->reqSegPtr  ||
	    destptr + len > rp->reqSegPtr + rp->reqSegLength) {
		Nack(wp, EFAULT);	/* remote CopyTo invoker gets this */
		return;
	}

	if (u.u_WDataExpected == NULL  ||
	    u.u_WSequenceNo != wp->Wp_ipc.ipcSequenceNo) {
		u.u_WSequenceNo = wp->Wp_ipc.ipcSequenceNo;
		u.u_WDataStart = u.u_WDataExpected = destptr;
		u.u_WDataDone = destptr + wp->Wp_ipc.ipcCopyLength;
	}
	if (u.u_WDataExpected == destptr) {
		if (WCopyFromMbuf(wp, sizeof(struct Wrproc),  destptr, len,
				rcode & RCODE_SEGMENT) == 0  &&  len) {
			Nack(wp, EFAULT);	/* tough luck, invoker */
			return;
		}
		if ((u.u_WDataExpected += len) < u.u_WDataDone)
			return;
	} else
		return;	/* for now, if we miss anything, redo everything */
	ipc.ipcPacketType = IPC_COPY_TO_REP;
	ipc.ipcSequenceNo = wp->Wp_ipc.ipcSequenceNo;
	u.u_WSequenceNo = 0;
	ipc.ipcSrcPid = MYPID;
	ipc.ipcDestPid = wp->Wp_ipc.ipcSrcPid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcDataLength = u.u_WDataExpected - u.u_WDataStart;
	ipc.ipcSrcAddress = u.u_WDataStart;
	ipc.ipcDestAddress = NULL;
	u.u_WDataExpected = NULL;
	u.u_WDataStart = NULL;
	u.u_WDataDone = NULL;
	SendKernelPacket(&ipc, (char *)0, (char *)0, 0, 0);
}

reqCopyFrom(wp, rp, rcode)
  register struct Wrproc *wp;
  register struct request *rp;
  unsigned short rcode;
  {
	register char *srcptr = wp->Wp_ipc.ipcDestAddress;
	register char *destptr = wp->Wp_ipc.ipcSrcAddress;
	register unsigned long copylen = wp->Wp_ipc.ipcDataLength, len, maxlen;
	register int seg = rcode & RCODE_SEGMENT;
	struct ipc ipc;

	if (!SEG_READ(rcode)  ||
	    srcptr < rp->reqSegPtr  ||
	    srcptr + copylen > rp->reqSegPtr + rp->reqSegLength) {
		Nack(wp, EFAULT);	/* remote CopyFrom invoker gets this */
		return;
	}
	ipc.ipcPacketType = IPC_COPY_FROM_REP;
	ipc.ipcSequenceNo = wp->Wp_ipc.ipcSequenceNo;
	ipc.ipcSrcPid = MYPID;
	ipc.ipcDestPid = wp->Wp_ipc.ipcSrcPid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcCopyLength = copylen;
	maxlen = MAX_APPENDED_SEGMENT;
	if (copylen > MAX_APPENDED_SEGMENT)
		maxlen = MaxPacket(ipc.ipcDestPid);
	do {	/* this loop may cause the destination machine to gag */
		if ((len = copylen) > maxlen)
			len = maxlen;
		ipc.ipcDataLength = len;
		ipc.ipcDestAddress = destptr;
		SendKernelPacket(&ipc, (char *)0, srcptr, len, seg);
		srcptr += len;
		destptr += len;
	} while (copylen -= len);
}

reqNack(wp)
  register struct Wrproc *wp;
  {
	return u.u_error = wp->Wp_ipc.ipcDataLength;
}

/*
 * Returns pid of sending process, else 0.
 */
unsigned long
ReceiveSpecific(rp, rpid, segptr, segsize)
  struct request *rp;
  unsigned long rpid;
  char *segptr;
  int segsize;
  {
	register struct proc *p = u.u_procp;
	register struct Wrproc *wp;
	int seg;

	if (rp)
		if (!useracc(rp, sizeof *rp, B_WRITE)  ||
		    (segsize  &&  !useracc(segptr, segsize, B_WRITE))) {
			u.u_error = EFAULT;
			return 0;
		}

	wp = FindNewRequest(rpid);
	wp->Wp_ipc.ipcDataOff = sizeof(struct Wrproc);
	wp->Wp_ipc.ipcSrcAddress = wp->Wp_msg.reqSegPtr;
	if (wp->Wp_ipc.ipcDataLength < segsize)
		segsize = wp->Wp_ipc.ipcDataLength;
	if (rp) {
		if (copyout(&wp->Wp_msg, rp, sizeof *rp)) {
			printf("Receive: bad copyout\n");
			return 0;
		}
		seg = RCODE_USER_SEGMENT;
	} else {
		rp = (struct request *)u.u_Rmsg;
		*rp = wp->Wp_msg;	/* structure assignment */
		seg = RCODE_KERN_SEGMENT;
	}
	if (segsize) {
		if (WCopyFromMbuf(wp, sizeof(struct Wrproc), segptr, segsize,
							seg) != segsize) {
			return 0;
		}
		wp->Wp_ipc.ipcDataOff += segsize;
		wp->Wp_ipc.ipcSrcAddress += segsize;
		if ((wp->Wp_ipc.ipcDataLength -= segsize) == 0)
			WFreeData(wp);
	}
	u.u_r.r_val2 = segsize;
	return wp->Wp_ipc.ipcSrcPid;
}

unsigned long
ReceiveWithSegment(rp, segptr, segsize)
  struct request *rp;
  char *segptr;
  int segsize;
  {
	return ReceiveSpecific(rp, (unsigned long)0, segptr, segsize);
}

unsigned long
Receive(rp)
  struct request *rp;
  {
	return ReceiveSpecific(rp, (unsigned long)0, (char *)0, 0);
}

/*
 * Returns pid of replied-to process, else 0.
 */
unsigned long
ReplyWithSegment(rp, rpid, srcptr, destptr, segsize)
  struct reply *rp;
  unsigned long rpid;
  char *srcptr, *destptr;
  int segsize;
  {
	register struct Wrproc *wp;
	struct ipc ipc;
	int seg = RCODE_KERN_SEGMENT;
	struct reply msg_save;

	if (rp) {
		if (copyin(rp, &msg_save, sizeof msg_save)  ||
		    (segsize  &&  !useracc(srcptr, segsize, B_READ))) {
			u.u_error = EFAULT;
			return 0;
		}
		seg = RCODE_USER_SEGMENT;
	} else
		msg_save = *(struct reply *)u.u_Rmsg;
	rp = &msg_save;

	if ((wp = GetRequest(rpid)) == NULL) {
		u.u_error = ESRCH;
		return 0;
	}
	ipc.ipcPacketType = IPC_REPLY;
	ipc.ipcSequenceNo = wp->Wp_ipc.ipcSequenceNo;
	ipc.ipcSrcPid = MYPID;
	ipc.ipcDestPid = rpid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcDataLength = segsize;
	ipc.ipcSrcAddress = NULL;
	ipc.ipcDestAddress = destptr;
	SendKernelPacket(&ipc, rp, srcptr, segsize, seg);
	Wrelse(wp);
	return rpid;
}

unsigned long
Reply(rp, rpid)
  struct reply *rp;
  unsigned long rpid;
  {
	ReplyWithSegment(rp, rpid, (char *)0, (char *)0, 0);
}

FlushReply(rpid)
  unsigned long rpid;
  {
	struct Wrproc *wp;

	if (wp = GetRequest(rpid))
		Wrelse(wp);
}

/*
 * Returns 0, else error code.
 */
CopyTo(rpid, dptr, sptr, cplen, xseg)
  unsigned long rpid;
  char *dptr, *sptr;
  int cplen, xseg;
  {
	register struct Wrproc *wp;
	register char *destptr, *srcptr;
	register int copylen;
	register int len, maxlen;
	register unsigned short retry;
	struct ipc ipc;
	int seg = rcode_seg[xseg];

  more:
	destptr = dptr;
	srcptr = sptr;
	copylen = cplen;

	if (seg == RCODE_USER_SEGMENT  &&  !useracc(srcptr, copylen, B_READ))
		return u.u_error = EFAULT;
	if ((wp = FindRequest(rpid)) == NULL)
		return u.u_error = ESRCH;

	ipc.ipcPacketType = IPC_COPY_TO_REQ;
	ipc.ipcSrcPid = MYPID;
	ipc.ipcDestPid = rpid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcCopyLength = copylen;

	retry = WRETRY;
	do {
		ipc.ipcSequenceNo = ++SeqNo;
		destptr = dptr;
		srcptr = sptr;
		copylen = cplen;
		maxlen = MAX_APPENDED_SEGMENT;
		if (copylen > MAX_APPENDED_SEGMENT)
			maxlen = MaxPacket(ipc.ipcDestPid);
		do {	/* this loop could be a real choker for the dest */
			if ((len = copylen) > maxlen)
				len = maxlen;
			ipc.ipcDataLength = len;
			ipc.ipcDestAddress = destptr;
			SendKernelPacket(&ipc, (char *)0, srcptr, len, seg);
			srcptr += len;
			destptr += len;
		} while (copylen -= len);
		while (wp = GetReply(rpid, hz << 2)) {
			if (wp->Wp_ipc.ipcSequenceNo != ipc.ipcSequenceNo) {
				Wrelse(wp);
				continue;
			}
			switch (wp->Wp_ipc.ipcPacketType) {
			  case IPC_BREATH_OF_LIFE:
				retry = WRETRY;	/* reset retry cntr */
				break;
			  case IPC_COPY_TO_REP:
				destptr = wp->Wp_ipc.ipcSrcAddress;
				copylen = wp->Wp_ipc.ipcDataLength;
				Wrelse(wp);
				if (len = destptr - dptr) {
					if (len < 0  ||  len > cplen)
						return u.u_error = EIO;
					if (CopyTo(rpid, dptr, sptr, len, xseg))
						return u.u_error;
				}
				if (cplen -= (len += copylen)) {
					dptr += len;
					sptr += len;
					goto more;
				}
				return 0;
			  case IPC_NACK:
				u.u_error = reqNack(wp);
				Wrelse(wp);
				return u.u_error;
			}
			Wrelse(wp);
		}
	} while (--retry > 0);
	return u.u_error = EIO;
}

/*
 * Returns 0, else error code.
 */
CopyFrom(rpid, destptr, srcptr, copylen, xseg)
  unsigned long rpid;
  register char *destptr, *srcptr;
  register int copylen, xseg;
  {
	register struct Wrproc *wp;
	register unsigned short retry;
	struct ipc ipc;
	register int seg = rcode_seg[xseg];

	if (seg == RCODE_USER_SEGMENT  &&  !useracc(destptr, copylen, B_WRITE))
		return u.u_error = EFAULT;
	if ((wp = FindRequest(rpid)) == NULL)
		return u.u_error = ESRCH;

	if ((int)wp->Wp_ipc.ipcDataLength > 0) {
		register int len = srcptr - wp->Wp_ipc.ipcSrcAddress;

		if (len >= 0) {
			wp->Wp_ipc.ipcSrcAddress += len;
			wp->Wp_ipc.ipcDataOff += len;
			wp->Wp_ipc.ipcDataLength -= len;
			if ((int)wp->Wp_ipc.ipcDataLength > 0) {
				len = copylen;
				if (wp->Wp_ipc.ipcDataLength < len)
					len = wp->Wp_ipc.ipcDataLength;
				len = WCopyFromMbuf(wp, wp->Wp_ipc.ipcDataOff,
							destptr, len, seg);
				wp->Wp_ipc.ipcSrcAddress += len;
				wp->Wp_ipc.ipcDataOff += len;
				wp->Wp_ipc.ipcDataLength -= len;
				if ((int)wp->Wp_ipc.ipcDataLength <= 0)
					WFreeData(wp);
				if ((copylen -= len) == 0)
					return 0;
				destptr += len;
				srcptr += len;
			} else
				WFreeData(wp);
		} else
			WFreeData(wp);
	}

	ipc.ipcPacketType = IPC_COPY_FROM_REQ;
	ipc.ipcSrcPid = MYPID;			/* i am src of this cmnd */
	ipc.ipcDestPid = rpid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	retry = WRETRY;
	do {
		ipc.ipcDataLength = copylen;
		ipc.ipcSrcAddress = destptr;	/* data dest in src's space */
		ipc.ipcDestAddress = srcptr;	/* data src in dest's space */
		ipc.ipcSequenceNo = ++SeqNo;
		u.u_WDataExpected = NULL;
		u.u_WDataStart = destptr;
		u.u_WDataDone = destptr + copylen;
		SendKernelPacket(&ipc, (char *)0, (char *)0, 0, 0);
		while (wp = GetReply(rpid, hz << 2)) {
			if (wp->Wp_ipc.ipcSequenceNo != ipc.ipcSequenceNo) {
				Wrelse(wp);
				continue;
			}
			switch (wp->Wp_ipc.ipcPacketType) {
			  case IPC_BREATH_OF_LIFE:
				retry = WRETRY;	/* reset retry counter */
				break;
			  case IPC_COPY_FROM_REP:
				retry = WRETRY;	/* reset retry counter */
				if (repCopyFrom(wp, seg) == 0)
					break;
				if (u.u_error) {
					Wrelse(wp);
					goto done;
				}
				if (u.u_WDataStart != destptr  ||
				    u.u_WDataExpected != u.u_WDataDone)
					goto more;
				Wrelse(wp);
				goto done;
			  case IPC_NACK:
				u.u_error = reqNack(wp);
				Wrelse(wp);
				goto done;
			}
			Wrelse(wp);
		}
  more:
		if (u.u_WDataExpected) {
			register llen = u.u_WDataStart - destptr;
			register tlen = copylen - (u.u_WDataExpected - destptr);
			u.u_WDataExpected = NULL;
			retry = WRETRY;
			if (tlen == 0) {
				if (llen == 0)
					return u.u_error;
				copylen = llen;
				continue;
			}
			if (llen) {
				CopyFrom(rpid, destptr, srcptr, llen, xseg);
				if (u.u_error)
					return u.u_error;
			}
			destptr += copylen - tlen;
			srcptr += copylen - tlen;
			copylen = tlen;
		}
	} while (--retry > 0);
	u.u_error = EIO;
  done:
	u.u_WDataExpected = NULL;
	u.u_WDataStart = NULL;
	u.u_WDataDone = NULL;
	return u.u_error;
}

repCopyFrom(wp, seg)
  register struct Wrproc *wp;
  int seg;
  {
	register char *destptr = wp->Wp_ipc.ipcDestAddress;
	register unsigned long len = wp->Wp_ipc.ipcDataLength;

	if (u.u_WDataExpected == NULL  &&
	    destptr >= u.u_WDataStart  &&
	    destptr + len <= u.u_WDataDone)
		u.u_WDataStart = u.u_WDataExpected = destptr;
	if (u.u_WDataExpected == destptr) {
		if (WCopyFromMbuf(wp, sizeof(struct Wrproc),  destptr, len,
							seg) == 0  &&  len) {
			printf("repCopyFrom: bad copyout\n");
			return u.u_error = EFAULT;
		}
		if ((u.u_WDataExpected += len) < u.u_WDataDone)
			return 0;
	}
	return 1;
}

WFreeData(wp)
  register struct Wrproc *wp;
  {
	register struct mbuf *m0 = dtom(wp), *m;

	if (m0->m_len >= sizeof(struct Wrproc)) {
		m0->m_len = sizeof(struct Wrproc);
		if (m = m0->m_next) {
			m0->m_next = NULL;
			m_freem(m);
		}
		wp->Wp_ipc.ipcDataLength = 0;
	}
}

Nack(wp, error)
  register struct Wrproc *wp;
  int error;
  {
	struct ipc ipc;

	if (wp->Wp_ipc.ipcPacketType == IPC_NACK)
		return;		/* prevent NACK, NACK, NACK, ... */
	ipc.ipcPacketType = IPC_NACK;
	ipc.ipcSequenceNo = wp->Wp_ipc.ipcSequenceNo;
	ipc.ipcSrcPid = wp->Wp_ipc.ipcDestPid;	/* not necessarily! */
	ipc.ipcDestPid = wp->Wp_ipc.ipcSrcPid;
	ipc.ipcForwarder = 0;
	ipc.ipcUserNo = 0;
	ipc.ipcDataLength = error;
	ipc.ipcSrcAddress = ipc.ipcDestAddress = (char *)0;
	SendKernelPacket(&ipc, (char *)0, (char *)0, 0, 0);
}

Forward(rp, fromrpid, torpid)
  struct request *rp;
  unsigned long fromrpid, torpid;
  {
}

#define	NMACH	32

struct NetAddr {
	unsigned short	na_mid;		/* BARF machine id */
	struct ifnet	*na_ifp;	/* net-specific interface */
	struct sockaddr	na_addr;	/* net-dependent machine address */
} NetAddr[NMACH];

#define	MID(rpid)	((unsigned long)(rpid) >> 16)

struct ifnet *
Eaddr(sapp, mid)
  struct sockaddr **sapp;
  register unsigned short mid;
  {
	register unsigned short i = NMACH;
	register struct NetAddr *nap = NetAddr;
	static struct {
		unsigned short	sa_family;
		unsigned char	sa_ether_dest[6];
		unsigned char	sa_ether_source[6];
		unsigned short	sa_ether_type;
	} bsa = {
		AF_UNSPEC,
		{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
		{ 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF },
		ETHERPUP_WIPCTYPE
	};

	if (mid != 0xFFFF) {
		while (--i >= 0  &&  nap->na_mid) {
			if (nap->na_mid == mid) {
				if (sapp)
					*sapp = &nap->na_addr;
				return nap->na_ifp;
			}
			nap += 1;
		}
	}
	if (sapp)
		*sapp = (struct sockaddr *)&bsa;
	return NULL;
}

InvalidateNaddr(mid)
  unsigned short mid;
  {
	register struct NetAddr *nap;

	for (nap = NetAddr;  nap < &NetAddr[NMACH] && nap->na_mid;  nap += 1) {
		if (nap->na_mid == mid) {
			nap->na_mid = 0xFFFF;
			nap->na_ifp = NULL;
			return 1;
		}
	}
	return 0;
}

int
MaxPacket(rpid)
  unsigned long rpid;
  {
	register struct ifnet *ifp;
	unsigned short mid = MID(rpid);

	if (mid != 0  &&  mid != 0xFFFF) {
		ifp = Eaddr((struct sockaddr*)0, mid);
		if (ifp  &&  ifp->if_name  &&  strcmp(ifp->if_name, "vb") == 0)
			return 8 * 1024;
	}
	return MAX_APPENDED_SEGMENT;
}

Winput(m, ifp, sap)
  register struct mbuf *m;
  register struct ifnet *ifp;
  register struct sockaddr *sap;
  {
	struct Wrproc *wp;
	register unsigned short mid;
	register struct NetAddr *nap;
	struct NetAddr *xnap = NULL;

	if (Mid == 0  &&  (Mid = GetMid()) == 0) {
		m_freem(m);
		return;
	}

	if (m->m_len < sizeof(struct Wrproc))
		if ((m = m_pullup(m, sizeof(struct Wrproc))) == NULL)
			return;
	wp = mtod(m, struct Wrproc *);
	mid = MID(wp->Wp_ipc.ipcSrcPid);
	for (nap = NetAddr;  nap < &NetAddr[NMACH];  nap += 1) {
		if (nap->na_mid == 0xFFFF) {
			if (xnap == NULL)
				xnap = nap;
			continue;
		}
		if (nap->na_mid == 0) {
			if (xnap)
				nap = xnap;
			nap->na_mid = mid;
			nap->na_ifp = ifp;
			nap->na_addr = *sap;
			RcvPacket(wp, ifp, sap);
			return;
		}
		if (nap->na_ifp == ifp  &&
		    bcmp(&nap->na_addr, sap, sizeof(*sap)) == 0) {
			if (nap->na_mid == mid) {
				RcvPacket(wp, ifp, sap);
				return;
			}
			/* Invalidate previous use of mid */
			InvalidateMachInfo(nap->na_mid);
			break;
		}
		if (nap->na_mid == mid) {
			register unsigned long temppid;
			/* turn packet around */
			temppid = wp->Wp_ipc.ipcDestPid;
			wp->Wp_ipc.ipcDestPid = wp->Wp_ipc.ipcSrcPid;
			if ((temppid & 0xFFFF0000) == 0xFFFF0000)
				temppid = Mid + (temppid & 0xFFFF);
			wp->Wp_ipc.ipcSrcPid = temppid;
			/* (*ifp->if_output)(ifp, m, sap); */ m_freem(m);
			return;
		}
	}
	m_freem(m);
}

RetToSender(wp, ifp, sap, type)
  struct Wrproc *wp;
  struct ifnet *ifp;
  struct sockaddr *sap;
  unsigned short type;
  {
	if (ifp  &&  sap  &&  Mid) {
		register unsigned long temp_rpid = wp->Wp_ipc.ipcDestPid;

		wp->Wp_ipc.ipcDestPid = wp->Wp_ipc.ipcSrcPid;
		if ((temp_rpid & 0xFFFF0000) == 0xFFFF0000)
			temp_rpid = Mid + (temp_rpid & 0xFFFF);
		wp->Wp_ipc.ipcSrcPid = temp_rpid;
		if (type)
			wp->Wp_ipc.ipcPacketType = type;
		(*ifp->if_output)(ifp, dtom(wp), sap);
		return;
	}
	Wrelse(wp);
}

RcvPacket(wp, ifp, sap)
  register struct Wrproc *wp;
  register struct ifnet *ifp;
  register struct sockaddr *sap;
  {
	register struct proc *p;
	register struct Wrproc *xp;
	register unsigned long rpid = wp->Wp_ipc.ipcDestPid;
	int s;

	if (Mid == 0  &&  (Mid = GetMid()) == 0) {
		Wrelse(wp);
		return;
	}

	/*
	 * Ignore packets that are not destined for a process on this machine.
	 */
	if ((rpid & 0xFFFF0000) != 0xFFFF0000  &&
	    (rpid & 0xFFFF0000) != Mid) {
		Wrelse(wp);
		return;
	}

	/*
	 * If destined for the generic barf server, determine the true pid.
	 */
	if (rpid == 0xFFFF0000  ||  (rpid & 0xFFFF) == 0xFFFF) {
		rpid = Mid + BarfPid;
	}

	if ((rpid & 0xFFFF) == 0)	/* pfind is too stupid to find pid #0 */
		p = &proc[0];
	else
		p = NULL;
	if ((p == NULL  &&  (p = pfind(rpid & 0xFFFF)) == NULL)  ||
	    p->p_stat == SZOMB) {
		if ((rpid & 0xFFFF0000) != 0xFFFF0000)
			Nack(wp, ESRCH);
		Wrelse(wp);
		return;
	}
	wp->Wp_next = (struct Wrproc *)0;
	switch(wp->Wp_ipc.ipcPacketType) {
	  case IPC_SEND:
		xp = (struct Wrproc *)&p->p_Wrecq;
		wakeup(xp);
		s = spl7();
		while (xp->Wp_next) {
			xp = xp->Wp_next;
			if (xp->Wp_ipc.ipcSrcPid == wp->Wp_ipc.ipcSrcPid  &&
			    xp->Wp_ipc.ipcSequenceNo == wp->Wp_ipc.ipcSequenceNo) {
				splx(s);
				/* reply with IPC_BREATH_OF_LIFE packet? */
				RetToSender(wp, ifp, sap, IPC_BREATH_OF_LIFE);
				return;
			}
		}
		xp->Wp_next = wp;
		splx(s);
		break;
	  case IPC_REPLY:
	  case IPC_FORWARD:
	  case IPC_RECEIVE_SPECIFIC:
	  case IPC_BREATH_OF_LIFE:
	  case IPC_GET_PID:
	  case IPC_GET_PID_REPLY:
	  case IPC_COPY_FROM_REQ:
	  case IPC_COPY_FROM_REP:
	  case IPC_COPY_TO_REQ:
	  case IPC_COPY_TO_REP:
	  case IPC_NACK:
		xp = (struct Wrproc *)&p->p_Wrepq;
		wakeup(xp);
		s = spl7();
		while (xp->Wp_next)
			xp = xp->Wp_next;
		xp->Wp_next = wp;
		splx(s);
		break;
	  default:
		printf("RcvPacket: bad PacketType\n");
		Wrelse(wp);
		return;
	}
}

SendKernelPacket(ip, mp, p, len, seg)
  register struct ipc *ip;
  struct request *mp;
  char *p;
  int len, seg;
  {
	register struct mbuf *m;
	register struct Wrproc *wp;
	struct sockaddr *sap;
	register struct ifnet *ifp = Eaddr(&sap, MID(ip->ipcDestPid));

#ifdef	ENP
	{
		extern enpoutput();

		if ((ip->ipcDestPid & 0xFFFF0000) != Mid  &&
		    ifp  &&  ifp->if_output == enpoutput) {
			enp_trfs_xmit(ifp, sap, ip, mp, p, len, seg);
			return;
		}
	}
#endif	ENP

#ifdef	VB
	{
		extern vboutput();

		if ((ip->ipcDestPid & 0xFFFF0000) != Mid  &&
		    ifp  &&  ifp->if_output == vboutput) {
			vb_trfs_xmit(ifp, sap, ip, mp, p, len, seg);
			return;
		}
	}
#endif	VB

	MGET(m, M_DONTWAIT, MT_DATA);
	if (m == 0)
		return;
	m->m_len = sizeof(struct Wrproc);
	m->m_off += sizeof(struct ether_header);
	wp = mtod(m, struct Wrproc *);
	wp->Wp_next = (struct Wrproc *)0;
	wp->Wp_ipc = *ip;	/* structure assignment */
	if (mp)
		wp->Wp_msg = *mp;	/* structure assignment */
	if (len) {
		if (WCatToMbuf(wp, p, len, seg) == (struct mbuf *)0)
			return;
	}
	if ((wp->Wp_ipc.ipcDestPid & 0xFFFF0000) == Mid)
		RcvPacket(wp, NULL, NULL);
	else {
		if (ifp)
			return ((*ifp->if_output)(ifp, m, sap));

		for (ifp = ifnet; ifp; ifp = ifp->if_next) {
			if (ifp  &&  ifp->if_output) {
				struct mbuf *m0;
				extern looutput();
				if (ifp->if_output == looutput)
					continue;	/* KLUDGE */
				if (m0 = m_copy(m, 0, M_COPYALL))
					(*ifp->if_output)(ifp, m0, sap);
			}
		}
		m_freem(dtom(wp));
	}
}

Wrelse(wp)
  struct Wrproc *wp;
  {
	m_freem(dtom(wp));
}

sysSend()
  {
	register struct a {
		struct request	*rp;
		unsigned long	rpid;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = Send(uap->rp, uap->rpid);
}

sysReceive()
  {
	register struct a {
		struct request	*rp;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = Receive(uap->rp);
}

sysReply()
  {
	register struct a {
		struct reply	*rp;
		unsigned long	rpid;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = Reply(uap->rp, uap->rpid);
}

sysCopyFrom()
  {
	register struct a {
		unsigned long	rpid;
		char		*dest;
		char		*src;
		unsigned	len;
	} *uap = (struct a *)u.u_ap;

	CopyFrom(uap->rpid, uap->dest, uap->src, uap->len, 0);
}

sysCopyTo()
  {
	register struct a {
		unsigned long	rpid;
		char		*dest;
		char		*src;
		unsigned	len;
	} *uap = (struct a *)u.u_ap;

	CopyTo(uap->rpid, uap->dest, uap->src, uap->len, 0);
}

sysForward()
  {
	register struct a {
		struct request	*rp;
		unsigned long	fromrpid;
		unsigned long	torpid;
	} *uap = (struct a *)u.u_ap;

	Forward(uap->rp, uap->fromrpid, uap->torpid);
}

sysReceiveSpecific()
  {
	register struct a {
		struct request	*rp;
		unsigned long	rpid;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = ReceiveSpecific(uap->rp, uap->rpid, (char *)0, 0);
}

sysReceiveWithSegment()
  {
	register struct a {
		struct request	*rp;
		char		*ptr;
		unsigned	len;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = ReceiveWithSegment(uap->rp, uap->ptr, uap->len);
}

sysReplyWithSegment()
  {
	register struct a {
		struct request	*rp;
		unsigned long	rpid;
		char		*srcptr;
		char		*destptr;
		unsigned long	len;
	} *uap = (struct a *)u.u_ap;

	u.u_r.r_val1 = ReplyWithSegment(uap->rp, uap->rpid,
					uap->srcptr, uap->destptr, uap->len);
}

struct mbuf *
WCatToMbuf(wp, cp, len, seg)
  struct Wrproc *wp;
  register unsigned char *cp;
  register int len;
  register short seg;
  {
	register int n;
	register struct mbuf *m = dtom(wp);
	struct mbuf **mp;

	if (len == 0)
		return 0;
	while (m->m_next)
		m = m->m_next;

#ifdef	MT_SEG_XMIT
	if (len > 120  &&  seg != RCODE_USER_SEGMENT) {
		struct mbuf *m0 = m;

		MGET(m, M_DONTWAIT, MT_SEG);
		if (m) {
			register struct mb_seg *msp;

			m0->m_next = m;
			m->m_len = sizeof(struct mb_seg);
			msp = mtod(m, struct mb_seg *);
			msp->ms_type = MS_PSEG;
			msp->ms_len = len;
			switch (seg) {
			  case RCODE_KERN_SEGMENT:
				msp->ms_addr = (caddr_t)svtop(cp);
				break;
			  case RCODE_PHYS_SEGMENT:
				msp->ms_addr = (caddr_t)cp;
				break;
			  default:
				panic("WCatToMbuf: bad seg");
			}
			return (dtom(wp));
		}
		m = m0;
	}
#endif	MT_SEG_XMIT
	if ((n = MMAXOFF - m->m_off - m->m_len) > 0) {
		if (n > len)
			n = len;
		switch(seg) {
		  case RCODE_USER_SEGMENT:
			if(copyin(cp, mtod(m, unsigned char *) + m->m_len, n))
				goto bad;
			break;
		  case RCODE_KERN_SEGMENT:
			bcopy(cp, mtod(m, unsigned char *) + m->m_len, n);
			break;
		  case RCODE_PHYS_SEGMENT:
			pcopyin(cp, mtod(m, unsigned char *) + m->m_len, n);
			break;
		}
		m->m_len += n;
		cp += n;
	} else
		n = 0;
	mp = &m->m_next;
	while ((len -= n) > 0) {
		MGET(m, M_DONTWAIT, MT_DATA);
		if (m == 0)
			goto bad;
		m->m_len = n = MIN(MLEN, len);
		*mp = m;
		mp = &m->m_next;
		switch(seg) {
		  case RCODE_USER_SEGMENT:
			if(copyin(cp, mtod(m, unsigned char *), n))
				goto bad;
			break;
		  case RCODE_KERN_SEGMENT:
			bcopy(cp, mtod(m, unsigned char *), n);
			break;
		  case RCODE_PHYS_SEGMENT:
			pcopyin(cp, mtod(m, unsigned char *), n);
			break;
		}
		cp += n;
	}
	return (dtom(wp));
bad:
	m_freem(dtom(wp));
	return (struct mbuf *)0;
}

WCopyFromMbuf(wp, off, cp, len, seg)
  register struct Wrproc *wp;
  register int off;
  register unsigned char *cp;
  register int len;
  register short seg;
  {
	register int n;
	register struct mbuf *m = dtom(wp);
	unsigned char *start = cp;

	while (m  &&  off > 0) {
		if ((n = m->m_len) > off) {
			n -= off;
			if (n > len)
				n = len;
			switch(seg) {
			  case RCODE_USER_SEGMENT:
				if(copyout(mtod(m, unsigned char *) + off, cp, n))
					return 0;
				break;
			  case RCODE_KERN_SEGMENT:
				bcopy(mtod(m, unsigned char *) + off, cp, n);
				break;
			  case RCODE_PHYS_SEGMENT:
				pcopyout(mtod(m, unsigned char *) + off, cp, n);
				break;
			}
			cp += n;
			len -= n;
			m = m->m_next;
			break;
		}
		off -= n;
		m = m->m_next;
	}
	while (m  &&  len > 0) {
		if ((n = m->m_len) > len)
			n = len;
		switch(seg) {
		  case RCODE_USER_SEGMENT:
			if(copyout(mtod(m, unsigned char *), cp, n))
				return 0;
			break;
		  case RCODE_KERN_SEGMENT:
			bcopy(mtod(m, unsigned char *), cp, n);
			break;
		  case RCODE_PHYS_SEGMENT:
			pcopyout(mtod(m, unsigned char *), cp, n);
			break;
		}
		cp += n;
		len -= n;
		m = m->m_next;
	}
	return (cp - start);
}
