#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/kernel.h"
#include "../h/file.h"
#include "../h/stat.h"
#include "../h/inode.h"
#include "../h/fs.h"
#include "../h/buf.h"
#include "../h/proc.h"
#include "../h/quota.h"
#include "../h/uio.h"
#include "../h/nami.h"

#define	NOHANGFORNAME

#define	NMACH	32
#define	MNLEN	11

struct MachInfo {
	unsigned long	mi_rpid;	/* rpid of remote Barf Server */
	short		mi_pgrp;
	short		mi_ref;
	struct inode	*mi_ip;		/* inode of swap file */
	char		mi_name[MNLEN+1];
} MachInfo[NMACH];

#define	MI_MID(mp)	(*((unsigned short *)(&(mp)->mi_rpid)))
#define	MID(rpid)	(((unsigned long)(rpid)) >> 16)
#define	REFCOUNT	5

/*
 * Enter a new machine into the table.
 *  Both the machine name and the rpid of the machine's barf server
 *  must be specified.
 */
UpdateMachInfo(name, rpid)
  char *name;
  unsigned long rpid;
  {
	register struct MachInfo *mp = MachInfo;
	register unsigned short mid = MID(rpid);
	register int s;

	if (rpid == 0  ||  rpid == 0xFFFFFFFF  ||
	    name == (char *)0  ||  *name == '\0')
		return -1;
	s = splnet();
	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mp->mi_rpid == 0xFFFFFFFF)
			continue;
		if (mp->mi_rpid == 0) {		/* found an empty slot */
			strncpy(mp->mi_name, name, MNLEN);
			mp->mi_name[MNLEN] = '\0';
			mp->mi_pgrp = 0;
			mp->mi_ref = REFCOUNT;
			mp->mi_rpid = rpid;
			splx(s);
			return 0;
		}

		if (MI_MID(mp) == mid  &&
		    strncmp(mp->mi_name, name, MNLEN) == 0) {
			mp->mi_ref = REFCOUNT;
			splx(s);
			return 0;		/* already known */
		}

		/* check for collision */
		if (strncmp(mp->mi_name, name, MNLEN) == 0) {
			splx(s);
			printf("UpdateMachInfo: %s collision\n", name);
			return -1;
		}
		if (MI_MID(mp) == mid) {
			splx(s);
			printf("UpdateMachInfo: 0x%X\n", mp->mi_rpid);
			return -1;
		}
	}
	splx(s);
	tablefull("MachInfo");
	return -1;
}

CheckOtherMachs()
  {
	register struct MachInfo *mp;
	int s = splnet();

	for (mp = MachInfo;  mp < &MachInfo[NMACH]  &&  mp->mi_rpid;  mp += 1) {
		while (mp->mi_rpid == 0xFFFFFFFF  ||  --mp->mi_ref <= 0) {
			struct MachInfo *mpsave = mp;

			if (mp->mi_rpid != 0xFFFFFFFF)
				InvalidateNaddr(MI_MID(mp));
			if (mp->mi_pgrp) {
				gsignal(mp->mi_pgrp, SIGKILL);
				mp->mi_pgrp = 0;
			}
			if (mp->mi_ip) {
				irele(mp->mi_ip);
				mp->mi_ip = NULL;
			}

			while (++mp < &MachInfo[NMACH])
				if (mp->mi_rpid == 0)
					break;	/* gone past last entry */
			if (--mp != mpsave)
				*mpsave = *mp;	/* copy last entry into hole */
			mp->mi_rpid = 0;	/* invalidate old last entry */
			mp = mpsave;
			if (mp->mi_rpid == 0)
				goto done;
		}
	}
  done:
	splx(s);
}

SetMachPgrp(rpid)
  unsigned long rpid;
  {
	register struct MachInfo *mp;
	register unsigned short mid = MID(rpid);

	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (MI_MID(mp) == mid) {
			if (mp->mi_pgrp == 0)
				mp->mi_pgrp = u.u_procp->p_pid;
			u.u_procp->p_pgrp = mp->mi_pgrp;
			return;
		}
	}
}

/*
 * Invalidate the specified machine entry.
 */
InvalidateMachInfo(mid)
  register unsigned short mid;
  {
	register struct MachInfo *mp = MachInfo;

	if (mid == 0)
		return -1;
	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mid == MI_MID(mp)) {
			mp->mi_rpid = 0xFFFFFFFF;
			InvalidateNaddr(mid);
			if (mp->mi_pgrp) {
				gsignal(mp->mi_pgrp, SIGKILL);
				mp->mi_pgrp = 0;
			}
			if (mp->mi_ip) {
				irele(mp->mi_ip);
				mp->mi_ip = NULL;
			}
			/* probably ought to remove the swap file, too */
			return 0;
		}
	}
	return -1;
}

/*
 * Given a machine name, return the corresponding machine id.
 */
unsigned short
MnameToMid(name)
  register char *name;
  {
	register struct MachInfo *mp = MachInfo;

	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mp->mi_rpid == 0xFFFFFFFF)
			continue;
		if (mp->mi_rpid == 0)
			break;
		if (strncmp(mp->mi_name, name, MNLEN) == 0)
			return MI_MID(mp);
	}
	return 0;
}

/*
 * Given a machine id, returns machine name.
 */
char *
MidToMname(mid)
  register unsigned short mid;
  {
	register struct MachInfo *mp;

	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mp->mi_rpid == 0xFFFFFFFF)
			continue;
		if (mp->mi_rpid == 0)
			break;
		if (mid == MI_MID(mp))
			return mp->mi_name;
	}
	return (char *)0;
}

char *
RpidToMname(rpid)
  unsigned long rpid;
  {
	return MidToMname(MID(rpid));
}


/*
 * Enter a cousin's rpid in the current user's u_Rrpid table.
 *  It is an error for there to already be an entry in the table
 *  corresponding to the cousin's machine id.
 */
NewRpid(rpid)
  unsigned long rpid;
  {
	register struct Rrpid *p, *q;
	register unsigned short mid = MID(rpid);

	q = &u.u_Rrpid[sizeof u.u_Rrpid / sizeof u.u_Rrpid[0]];
	for (p = u.u_Rrpid;  p < q;  p += 1) {
		if (p->u_m_rpid == 0) {	/* look for an empty slot */
			p->u_m_rpid = rpid;
			return 0;
		}
		if (p->u_m_mid == mid)
			return -1;	/* we already have a cousin on mid! */
	}
	return -1;
}

/*
 * Disassociate ourself from the specified cousin.
 *  It is assumed that he is dead.
 */
InvalidateRpid(rpid)
  register unsigned long rpid;
  {
	register struct Rrpid *p, *q;

	q = &u.u_Rrpid[sizeof u.u_Rrpid / sizeof u.u_Rrpid[0]];
	for (p = u.u_Rrpid;  p < q;  p += 1) {
		if (p->u_m_rpid == rpid) {
			struct Rrpid *psave = p;

			/*
			 * We have found the appropriate entry.
			 *  Now find the last entry, and move
			 *  it into this slot, thus keeping the
			 *  table tightly packed.
			 */
			for (p += 1;  p < q;  p += 1)
				if (p->u_m_rpid == 0)
					break;
			p -= 1;
			psave->u_m_rpid = p->u_m_rpid;
			p->u_m_rpid = 0;
			return 0;
		}
	}
	return -1;
}

/*
 * Determine the rpid of our cousin on the specified machine.
 *  If none return 0.
 */
unsigned long
MidToRpid(mid)
  register unsigned short mid;
  {
	if (mid) {
		register struct Rrpid *p, *q;
		static short first = 1;

		q = &u.u_Rrpid[sizeof u.u_Rrpid / sizeof u.u_Rrpid[0]];
		for (p = u.u_Rrpid;  p < q;  p += 1)
			if (p->u_m_mid == mid)
				return p->u_m_rpid;
		if (first) {
			unsigned long rpid;

			if (REMCreateCousin(mid, &rpid) == 0  &&
			    NewRpid(rpid) == 0) {
				first = 0;
				return rpid;
			}
			if (u.u_error)
				return 0;
		}
	}
	u.u_error = EIO;
	return 0;
}

/*
 * Returns the rpid of the cousin associated with this process
 *  on the specified machine.  If the machine is not known then
 *  a broadcast is sent and the machine name is noted for future
 *  reference.  If a cousin does not currently exist then a
 *  message is sent causing one to be created, and the new rpid
 *  is noted for future reference.
 */
unsigned long
MnameToRpid(name)
  char *name;
  {
	unsigned short mid;
	unsigned long rpid;

	/*
	 * Make sure the machine name is not trash.
	 */
	if (name == (char *)0  ||  name[0] == '\0')
		return 0;

	/*
	 * First determine the machine id
	 *  corresponding to the specified machine name.
	 *  The rpid returned by REMGetMid is not the rpid
	 *  of our cousin (if we need to call REMGetMid then
	 *  we don't even have a cousin yet), but rather the
	 *  rpid of the remote machine's barf server.
	 */
	if ((mid = MnameToMid(name)) == 0) {	/* look up name */
#ifdef	NOHANGFORNAME
		return 0;			/* rely on AM_ALIVE broadcasts*/
#endif	NOHANGFORNAME
		if (REMGetMid(name, &rpid)	/* if unknown, send inquiry, */
		 || UpdateMachInfo(name, rpid))	/*  and remember name/rpid */
			return 0;		/*   no such machine! */
		mid = MID(rpid);		/*  extract mid from rpid */
	}

	/*
	 * Next, determine the rpid of this process' cousin.
	 *  If none, create one.
	 */
	if ((rpid = MidToRpid(mid)) == 0) {	/* look up rpid */
		u.u_error = 0;			/* ok if not found */
		if (REMCreateCousin(mid, &rpid)	/* if none, create a new one, */
		 || NewRpid(rpid))		/*  and remember it */
			return 0;		/*   something failed! */
	}
	return rpid;
}

SetupSwap(rpid)
  unsigned long rpid;
  {
	unsigned short mid = MID(rpid);
	register struct MachInfo *mp = MachInfo;
	register struct inode *ip;
	char swappath[60];

	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mp->mi_rpid == 0xFFFFFFFF)
			continue;
		if (mp->mi_rpid == 0)
			break;
		if (mid == MI_MID(mp))
			break;
	}
	if (mp >= &MachInfo[NMACH])
		return ENXIO;

	if (ip = mp->mi_ip) {
		mp->mi_ip = NULL;
		irele(ip);
	}

	if (mp->mi_name == NULL  ||  mp->mi_name[0] == '\0')
		return ENXIO;
	strcpy(swappath, "/usr/spool/diskless/");
	strcat(swappath, mp->mi_name);
	strcat(swappath, ".swap");
	u.u_dirp = swappath;
	if ((ip = namei(schar, CREATE, 1)) == NULL)
		if (u.u_error  ||  (ip = maknode(0600)) == NULL)
			return u.u_error;
	itrunc(ip, (u_long)0);
	mp->mi_ip = ip;
	iunlock(ip);
	return 0;
}

struct inode *
GetSwapInode(rpid)
  unsigned long rpid;
  {
	unsigned short mid = MID(rpid);
	register struct MachInfo *mp = MachInfo;

	for (mp = MachInfo;  mp < &MachInfo[NMACH];  mp += 1) {
		if (mp->mi_rpid == 0xFFFFFFFF)
			continue;
		if (mp->mi_rpid == 0)
			break;
		if (mid == MI_MID(mp))
			return mp->mi_ip;
	}
	return (struct inode *)0;
}

unsigned long
ReRpid(rpid)
  unsigned long rpid;
  {
	unsigned short mid = MID(rpid);

	InvalidateRpid(rpid);
	if ((rpid = MidToRpid(mid)) == 0) {
		u.u_error = 0;
		if (REMCreateCousin(mid, &rpid)  ||  NewRpid(rpid))
			return 0;
		return rpid;
	}
	return 0;
}

sysMname()
  {
	register struct a {
		int	rpid;
		char	*mname;
		int	len;
	} *uap = (struct a *)u.u_ap;
	char *name = MidToMname(MID(uap->rpid));
	int len;

	if (name == (char *)0) {
		u.u_error = ESRCH;
		return;
	}
	len = min(uap->len, strlen(name) + 1);
	u.u_error = copyout((caddr_t)name, (caddr_t)uap->mname, len);
}
