/*
 * Standalone driver for the Interphase VME_SMD disk controller.
 */
#include "../h/param.h"
#include "../h/inode.h"
#include "../h/fs.h"

#include "../is68kdev/smreg.h"

#include "saio.h"
#include "sais68k.h"

#ifdef	M68020
#define	SMADDR0	  	((struct smdevice *) 0xFD0000) /* 1st addr */
#define	SMADDR1	  	((struct smdevice *) 0xFD0200) /* 2nd addr */
#define	SMADDR2	  	((struct smdevice *) 0xFD0400) /* 3rd addr */
#define	SMADDR3	  	((struct smdevice *) 0xFD0600) /* 4th addr */
#else	M68020
#define	SMADDR0	  	((struct smdevice *) 0x7D0000) /* 1st addr */
#define	SMADDR1	  	((struct smdevice *) 0x7D0200) /* 2nd addr */
#define	SMADDR2	  	((struct smdevice *) 0x7D0400) /* 3rd addr */
#define	SMADDR3	  	((struct smdevice *) 0x7D0600) /* 4th addr */
#endif	M68020

#define	SM_CN		4			/* number of controllers */
#define	SM_DR		4			/* drives per controller */
#define	SM_MAXCNT	0x1FE00			/* must be multiple of nbsec */
#define	SM_RETRYCNT	1

static struct smdevice *smstd[SM_CN] = { SMADDR0, SMADDR1, SMADDR2, SMADDR3 };

int forbb, xbad144, smstrat();

smopen(io)
	register struct iob *io;
{
	register int		cn = UTOCN(io->i_unit); 
	register int 		dr = UTODR(io->i_unit);
	register struct smdevice *smaddr;
	register struct sm_iopb	*iopb;
	struct smuib		*uib = (struct smuib *)0x400; /* KLUDGE */

	if (dr >= SM_DR || cn >= SM_CN) {
		printf("sm: bad unit\n");
		return (-1);
	}
	smaddr = smstd[cn];
	if (!probe(&smaddr->sm_cstatus,&smaddr->sm_cstatus)) {
		printf("sm%d: controller not present\n", dr);
		return (-1);
	}
	smaddr->sm_cstatus |= CS_BDCLR;
	smaddr->sm_cstatus &= ~CS_BDCLR;
	if (smbusy(smaddr)) {
		printf("sm%d: fail diagnostics 0\n", dr);
		return(-1);
	}
	if ((smaddr->sm_cstatus & CS_BOK) == 0) {
		printf("sm%d: fail diagnostics 1\n", dr);
		return (-1);
	}
	if (smdrdy(smaddr, (dr&0x2)>>1)) {
		printf("sm%d: drive not ready\n", dr);
		return (-1);
	}
	smaddr->sm_cstatus &= ~CS_SLED;
	iopb = &smaddr->sm_iopb[0];
	iopb->iopb_cmd = CMD_READ;
	iopb->iopb_coptions = ((dr&0x2) << 6) | COP_ECC_EN;
	iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
	iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
	iopb->iopb_blk_count = 1;
	iopb->iopb_buf_addrh = hiword(uib);
	iopb->iopb_buf_addrl = loword(uib);
	iopb->iopb_mem_type = MEMT_16BIT;
	iopb->iopb_adr_mod = ADRM_STD_N_D;
	iopb->iopb_dma_burst = DMA_BURST;
	iopb->iopb_skew_offset = 0;
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error reading UIB %x %x\n", dr,
		    smaddr->sm_DSTATUS((dr&0x2)>>1), iopb->iopb_error_code);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return (-1);
	}
	smaddr->sm_cstatus &= ~CS_DONE;
	if (uib->uib_magic != UIB_MAGIC) {
		printf("sm%d: not formatted\n", dr);
		return (-1);
	}
	uib->uib_magic = 0;
	iopb->iopb_cmd = CMD_INIT;
	iopb->iopb_coptions = (dr&0x2) << 6;
	if (xbad144) {
		uib->uib_retries = 4;
		uib->uib_attrib &= ~(UIB_AT_CE | UIB_AT_RSK);
	}
	iopb->iopb_buf_addrh = hiword(uib);
	iopb->iopb_buf_addrl = loword(uib);
	iopb->iopb_mem_type = MEMT_16BIT;
	iopb->iopb_adr_mod = ADRM_STD_N_D;
	iopb->iopb_dma_burst = DMA_BURST;
	iopb->iopb_skew_offset = 0;
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR) ) {
		printf("sm%d: error INIT\n", dr);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return (-1);
	}
	smaddr->sm_cstatus &= ~CS_DONE;
	if (uib->uib_vol_head[dr&1].num_heads == 0) {
		printf("sm%d: null volume\n", dr);
		return (-1);
	}
	io->i_st.nsect = uib->uib_sec_trk;
	io->i_st.ntpc = uib->uib_vol_head[dr&1].num_heads;
	io->i_st.nspc = io->i_st.nsect * io->i_st.ntpc;
	io->i_st.nbsec = 512;
	io->i_st.ncyl = uib->uib_num_cyl - 1;	/* reserve 1 cylinder for UIB */
	io->i_st.off = NULL;
/*	printf("(%dx%dx%d)\n",io->i_st.nsect,io->i_st.ntpc,io->i_st.ncyl); /**/
	if ((io->i_boff = diskpart(io)) < 0) {
		printf("sm%d: bad offset\n", dr);
		return (-1);
	}
	return(0);
}

smstrategy(io, func)
	register struct iob *io;
{
	register int 			bn = io->i_bn;
	register int 			dr = UTODR(io->i_unit);

	switch (func) {
	  case READ:
		func = CMD_READ; 
		break;
	  case WRITE:
		func = CMD_WRITE; 
		break;
	  default:
		printf("sm%d: func %x\n", func, dr);
		return -1;
	}
	if (forbb) {
		if (smstrat(io,bn,io->i_cc,func) == -1)
			return -1;
		io->i_ma += io->i_cc;
	} else if (badstrat(io,bn,io->i_cc,func,smstrat,CMD_READ) == -1)
		return -1;
	return io->i_cc;
}

smstrat(io,bn,cc,func)
	register struct iob *io;
	register int bn;
	int func;
{
	register int 			cn = UTOCN(io->i_unit);
	register int 			dr = UTODR(io->i_unit);
	register struct smdevice 	*smaddr = smstd[cn];
	register struct sm_iopb		*iopb = &smaddr->sm_iopb[0];
	static int 			errcnt = 0;
	int				i;

	while (1) {
		iopb->iopb_cmd = func;
		iopb->iopb_coptions = ((dr&0x3) << 6) | COP_LOG_TRAN;
		if (!xbad144)
			iopb->iopb_coptions |= COP_ECC_EN;
		i = bn + (io->i_st.nsect * io->i_st.ntpc);
		iopb->iopb_dsk_addr.log.logh = hiword(i);
		iopb->iopb_dsk_addr.log.logl = loword(i);
		iopb->iopb_blk_count = (cc + 511) >> 9;
		iopb->iopb_buf_addrh = hiword(io->i_ma);
		iopb->iopb_buf_addrl = loword(io->i_ma);
		iopb->iopb_mem_type = MEMT_16BIT;
		iopb->iopb_adr_mod = ADRM_STD_N_D;
		iopb->iopb_dma_burst = DMA_BURST;
		iopb->iopb_skew_offset = 0;
		smaddr->sm_cstatus |= CS_GO;
		if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			printf("sm%d: sn%d HARD %s %x\n",
				dr, i - io->i_st.nspc,
				(func == CMD_READ) ? "READ" : "WRITE", 
				iopb->iopb_error_code);
			smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			if (++errcnt < SM_RETRYCNT)
				continue;
			errcnt = 0;
			return -1;
		} else if (iopb->iopb_status == STA_COMP_EX) {
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    iopb->iopb_dsk_addr.log.logl;
			printf("sm%d: sn%d SOFT %s %b\n",
				dr, i - io->i_st.nspc,
				(func == CMD_READ) ? "READ" : "WRITE", 
				iopb->iopb_error_code, ERR_COR_BITS);
		}
		smaddr->sm_cstatus &= ~CS_DONE;
		return 0;
	}
}

static
smbusy(smaddr)
	register struct smdevice *smaddr;
{ 	
	register int l = 0;

	DELAY(400);
	while (smaddr->sm_cstatus & CS_BUSY)  {
#ifdef	M68020
		if (++l == 200000)
#else	M68020
		if (++l == 50000)
#endif	M68020
			return (-1);
		DELAY(400);
	}
	return (0);
}

static 
smwait(smaddr)
	register struct smdevice *smaddr;
{ 	
	register int l = 0;

	DELAY(400);
	while ((smaddr->sm_cstatus & CS_DONE) == 0)  {
#ifdef	M68020
		if (++l == 200000) {
#else	M68020
		if (++l == 50000) {
#endif	M68020
			printf("sm: timeout\n");
			return (-1);
		}
		DELAY(400);
	}
	return (0);
}

static 
smdrdy(smaddr, dr)
	register struct smdevice *smaddr;
{ 	
	register int l = 0;

	DELAY(400);
	while ((smaddr->sm_DSTATUS(dr)&DS_DREADY)==0) {
#ifdef	M68020
		if (++l == 200000)
#else	M68020
		if (++l == 50000)
#endif	M68020
			return (-1);
		DELAY(400);
	}
	return (0);
}
