/*
 *   Integrated Solutions SIO I/O Routines
 */
#define XONXOFF

#include "../h/param.h"
#include "../is68kdev/sioreg.h"

int	putlocal(), wputlocal(), putremote(), wputremote();
int	getloc(), getlocal(), wgetlocal(), getrem(), getremote(), wgetremote();
int	getchar(),  putchar();

int (*putch)() = putchar;
int (*getch)() = getchar;
int putslow = 0;
int autoboot = 0; 

#ifdef XONXOFF
#define CTLQ 	 0x11
#define CTLS 	 0x13
static char lochar = 0, rmchar = 0;
#endif XONXOFF

#define SPLOCAL ((struct s2681device *)S2681_ADDR)
#define SPREMOTE ((struct s2681device *)(S2681_ADDR+0x10))

getchar ()
{
	register int c;

	c = wgetlocal();
	if (c == '\r')
		wputlocal('\n');
	putchar(c);
	return c;
}

putchar (c)
register int c;
{
	if (autoboot)
		return;
	wputlocal(c);
	if (c == '\n')
		wputlocal('\r');
}

/* wait for and return a character from the remote */
wgetremote()
{
	register int c;

	while ((c = getremote()) == 0) ;
	return(c);
}

/* wait for and return a character from the local */
wgetlocal()
{
	register int c;

	while ((c = getlocal()) == 0) ;
	return(c);
}

/* wait for local to be ready and output a charater */
wputlocal(c)
{
#ifdef XONXOFF
	register int ic;

	if ((ic = getloc()) != 0) {
		if (ic == CTLS)
			while (getloc() != CTLQ) ;
		else
			lochar = ic;
	}
#endif XONXOFF

	while (putlocal(c) == 0) ;
}

/* wait for remote to be ready and output a charater */
wputremote(c)
{
#ifdef XONXOFF
	register int ic;

	if ((ic = getrem()) != 0) {
		if (ic == CTLS)
			while (getrem() != CTLQ) ;
		else
			rmchar = ic;
	}
#endif XONXOFF

	while (putremote(c) == 0) ;
}

/*
 * getremote()
 *	get a character from the remote port
 *	do not wait! return 0 if there is none
 */
getremote()
{
	register char c;

#ifdef XONXOFF
	if (rmchar) {
		c = rmchar;
		rmchar = 0;
		return(c);
	} else
#endif XONXOFF
		return(getrem());
}

/*
 * getlocal()
 *	get a character from the local port
 *	do not wait! return 0 if there is none
 */
getlocal()
{
	register char c;
#ifdef XONXOFF
	if (lochar) {
		c = lochar;
		lochar = 0;
		return(c);
	} else
#endif XONXOFF
		return(getloc());
}

getloc(x)
{
	if ((SPLOCAL->s2681_r_sr & SR_RXRDY) == 0)
		return (0);
	return(SPLOCAL->s2681_r_rhr & 0177);
}

/*
 * putlocal()
 *	put a character to the local port
 *	do not wait! return 0 if it is not ready
 */
putlocal(c)
char c;
{
	putslow = 0;
	if ((SPLOCAL->s2681_r_sr & SR_TXRDY) == 0)
		return (0);
	SPLOCAL->s2681_w_thr = c;
	g_putc(c);		/* graphics putc */
	return(1);
}

getrem()
{
	if ((SPREMOTE->s2681_r_sr & SR_RXRDY) == 0)
		return (0);
	return(SPREMOTE->s2681_r_rhr & 0177);
}

/*
 * putremote()
 *	put a character to the remote port
 *	do not wait! return 0 if it is not ready
 */
putremote(c)
char c;
{
	register int i;

	if ((SPREMOTE->s2681_r_sr & SR_TXRDY) == 0)
		return (0);
	SPREMOTE->s2681_w_thr = c;
	if (putslow)
		DELAY(4000);
	return(1);
}

