/*#define DEBUG /* */

/*
 * Calculate standard disk partition sizes.
 */
#include "../h/param.h"
#include "../h/inode.h"
#include "../h/fs.h"
#include "saio.h"

#define	NPARTITIONS	8
#define	PART(x)		(x - 'a')

/*
 * Default partition sizes, where they exist.
 */
#define	NDEFAULTS	5
#define	SMALLDISK	3
static int	defpart[NDEFAULTS][NPARTITIONS] = {
   { 15884, 66880, 0, 15884, 307200, 0, 0, 291346 },	/* ~ 356+ Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 291346 },	/* ~ 206-355 Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 0 },		/* ~ 61-205 Mbytes */
   { 15884, 10032, 0, 15884, 0, 0, 0, 0 },		/* ~ 20-60 Mbytes */
   { 0, 0, 0, 0, 0, 0, 0, 0 },				/* < 20 Mbytes */
};

/*
 * Each array defines a layout for a disk; that is, the collection of partitions
 * totally covers the physical space on a disk.
 */
#define	NLAYOUTS	3
static char	layouts[NLAYOUTS][NPARTITIONS] = {
   { 'a', 'b', 'h', 'g' },
   { 'a', 'b', 'h', 'd', 'e', 'f' },
   { 'c' },
};

/*
 * Default disk block and disk block fragment sizes for each file system.  
 * Those file systems with zero block and frag sizes are special cases
 * (e.g. swap areas or for access to the entire device).
 */
static struct	defparam {
	int	p_bsize;	/* block size */
	int	p_fsize;	/* frag size */
} defparam[NPARTITIONS] = {
	{ 8192, 1024 },		/* a */
	{ 0 },			/* b */
	{ 0 },			/* c */
	{ 8192, 1024 },		/* d */
	{ 4096, 512 },		/* e */
	{ 4096, 1024 },		/* f */
	{ 4096, 1024 },		/* g */
	{ 4096, 512 }		/* h */
};

/*
 * Each disk has some space reserved for a bad sector forwarding table.  DEC 
 * standard 144 uses the first 5 even numbered sectors in the last track of the
 * last cylinder for replicated storage of the bad sector table; another 126 
 * sectors past this is needed as a pool of replacement sectors.
 */
#define BADSECTTABLE 126	/* # sectors */

diskpart(io)
register struct iob	*io;
{
	register int	curcyl, def, part;
	register int	layout;
	int		threshhold, 
			nblks[NPARTITIONS], 
			numcyls[NPARTITIONS], 
			startcyl[NPARTITIONS];
	register int	badsecttable = BADSECTTABLE;
	register char	*lp;

	if (io->i_flgs & F_BLKOFF)
		return (io->i_boff);
	io->i_boff = 0;
	if (io->i_fpart < 0 || io->i_fpart > 7 )
		return (-1);
	else if (io->i_st.off) {
#ifdef DEBUG
		printf("diskpart: table %d -> %d\n",
			io->i_fpart,io->i_st.off[io->i_fpart] * io->i_st.nspc);
#endif
		return ( io->i_st.off[io->i_fpart] * io->i_st.nspc );
	}

#ifdef DEBUG
	printf("#sectors/track=%d, #tracks/cylinder=%d #cylinders=%d\n",
		io->i_st.nsect, io->i_st.ntpc, io->i_st.ncyl);
#endif
	/*
	 * Bad sector table contains one track for the replicated copies of the
	 * table and enough full tracks preceding the last track to hold the 
	 * pool of free blocks to which bad sectors are mapped.
	 */
	badsecttable = io->i_st.nsect + roundup(badsecttable, io->i_st.nsect);
	threshhold = howmany(io->i_st.nspc, badsecttable);

	/* 
	 * Figure out if disk is large enough for expanded swap area and 'd', 
	 * 'e', and 'f' partitions.  Otherwise, use smaller defaults based on 
	 * RK07.
	 */
	for (def = 0; def < NDEFAULTS; def++) {
		curcyl = 0;
		for (part = PART('a'); part < NPARTITIONS; part++) {
			nblks[part] = defpart[def][part];
			curcyl += howmany(nblks[part], io->i_st.nspc);
		}
		if (curcyl < io->i_st.ncyl - threshhold)
			break;
	}
	if (def >= NDEFAULTS)
		return (-1);

	/*
	 * Calculate number of cylinders allocated to each disk partition. We 
	 * may waste a bit of space here, but it's in the interest of 
	 * compatibility (for mixed disk systems).
	 */
	for (curcyl = 0, part = PART('a'); part < NPARTITIONS; part++) {
		numcyls[part] = 0;
		if (nblks[part] != 0) {
			numcyls[part] = howmany(nblks[part], io->i_st.nspc);
			curcyl += numcyls[part];
		}
	}
	numcyls[PART('f')] = io->i_st.ncyl - curcyl;
	numcyls[PART('g')] =
		numcyls[PART('d')] + numcyls[PART('e')] + numcyls[PART('f')];
	numcyls[PART('c')] = io->i_st.ncyl;
	nblks[PART('f')] = numcyls[PART('f')] * io->i_st.nspc - badsecttable;
	nblks[PART('g')] = numcyls[PART('g')] * io->i_st.nspc - badsecttable;
	nblks[PART('c')] = numcyls[PART('c')] * io->i_st.nspc;

	/*
	 * Calculate starting cylinder number for each partition. Note the 'h' 
	 * partition is physically located before the 'g' or 'd' partition.  
	 * This is reflected in the layout arrays defined above.
	 */
	for (layout = 0; layout < NLAYOUTS; layout++) {
		curcyl = 0;
		for (lp = layouts[layout]; *lp != 0; lp++) {
			startcyl[PART(*lp)] = curcyl;
			curcyl += numcyls[PART(*lp)];
		}
	}

	if (defpart[def][PART('a')] == 0) {
	    register int xxx = nblks[PART('f')];

	    part = PART('a');
	    if (xxx >= defpart[SMALLDISK][part]) {
		startcyl[part] = 0;
		numcyls[part]=howmany( defpart[SMALLDISK][part], io->i_st.nspc);
		nblks[PART('a')] = defpart[SMALLDISK][part];
		xxx -= numcyls[part] * io->i_st.nspc;
		part = PART('b');
		if (xxx >= defpart[SMALLDISK][part]) {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], io->i_st.nspc);
		    nblks[part] = defpart[SMALLDISK][part];
		    xxx -= numcyls[part] * io->i_st.nspc;
		    part = PART('g');
		    if (xxx >= 0) {
		    	startcyl[part] = startcyl[PART('b')]+numcyls[PART('b')];
		    	numcyls[part] = howmany( xxx, io->i_st.nspc);
		    	nblks[part] = xxx;
		    }
	        } else {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], io->i_st.nspc);
		    nblks[part] = defpart[SMALLDISK][part];
		}
	    } else {
		startcyl[part] = 0;
		numcyls[part] = howmany( defpart[SMALLDISK][part], io->i_st.nspc);
		nblks[part] = defpart[SMALLDISK][part];
	    }
	}

#ifdef DOZO
	if (dflag) {
		int nparts;

		/*
		 * In case the disk is in the ``in-between'' range where the 'g'
		 * partition is smaller than the 'h' partition, reverse the frag
		 * sizes so the /usr partition is always set up with a frag size
		 * larger than the user's partition.
		 */
		if (nblks[PART('g')] < nblks[PART('h')]) {
			int temp;

			temp = defparam[PART('h')].p_fsize;
			defparam[PART('h')].p_fsize =
				defparam[PART('g')].p_fsize;
			defparam[PART('g')].p_fsize = temp;
		}
		for (nparts = 0, part = PART('a'); part < NPARTITIONS; part++)
			if (nblks[part] != 0)
				nparts++;
#ifdef DEBUG
		for (part = PART('a'); part < NPARTITIONS; part++) {
			if (nblks[part] == 0)
				continue;
			printf("\t:p%c#%d:", 'a' + part, nblks[part]);
			if (defparam[part].p_bsize != 0) {
				printf("b%c#%d:f%c#%d:",
				  'a' + part, defparam[part].p_bsize,
				  'a' + part, defparam[part].p_fsize);
			}
			nparts--;
			printf("%s\n", nparts > 0 ? "\\" : "");
		}
#endif
	}
#endif
#ifdef DEBUG
	printf("\n    Partition\t   Size\t   Range\n");
	for (part = PART('a'); part < NPARTITIONS; part++) {
		printf("\t%c\t", 'a' + part);
		if (numcyls[part] == 0) {
			printf(" unused\n");
			continue;
		}
		printf("%d\t%d - %d\n", nblks[part], startcyl[part],
			startcyl[part] + numcyls[part] - 1);
	}
	printf("file system index %d -> sector offset %d\n",io->i_fpart,
			startcyl[io->i_fpart]*io->i_st.nspc);
#endif
	return (startcyl[io->i_fpart]*io->i_st.nspc);
}
