/*	ts.c	6.1	83/07/29	*/

/* # define DOUBLEBUF /* */

#include "ts.h"
#if NTS > 0 || Nts > 0
/*
 * TS11 tape driver
 *
 * TODO:
 *	write dump code
 */
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/dir.h"
#include "../h/conf.h"
#include "../h/user.h"
#include "../h/file.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/ioctl.h"
#include "../h/mtio.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/tsreg.h"

u_short	*tsstd[] = { 
#ifdef	QBUS
		(u_short *)0x3FF550, 
		(u_short *)0x3ff520,
#else	QBUS
#ifdef	M68020
		(u_short *)0xFFF550,
		(u_short *)0xFff520,
#else	M68020
		(u_short *)0x7FF550,
		(u_short *)0x7ff520,
#endif	M68020
#endif	QBUS
		0 };

/*
 * There is a ctsbuf per tape controller. It is used as the token to pass to 
 * the internal routines to execute tape ioctls. In particular, when the tape 
 * is rewinding on close we release the user process but any further attempts 
 * to use the tape drive before the rewind completes will hang waiting for 
 * ctsbuf.
 */
struct	buf	ctsbuf[NTS];

/*
 * Raw tape operations use a rtsbuf.  The driver notices when a rtsbuf is being
 * used and allows the user program to continue after errors and read records
 * not of the standard length (BSIZE).
 *
 * There are two raw buffer headers per drive, used alternately in an
 * effort to allow the unit to stream.
 */
struct buf rtsbuf[NTS][2];
int rtsblkno[NTS];	/* physical tape record index */
#define	MAXLEN	(20*512)
char tsbuffer[NTS][2][MAXLEN];

/*
 * Driver unibus interface routines and variables.
 */
int	tsprobe(), tsslave(), tsattach(), tsintr();
struct	qb_ctlr		*tscinfo[NTS];
struct	qb_device	*tsdinfo[Nts];
struct	buf		tsutab[NTS];

#ifndef	QBUS
extern u_short		*battery_clock;
static short		ts_battery_clock;
#endif	QBUS

struct	qb_driver 	TSdriver =
	{ tsprobe, tsslave, tsattach, tsstd, "ts", tsdinfo, "TS", tscinfo };

/* bits in minor device */
#define	TSUNIT(dev)	(minor(dev)&03)
#define	T_NOREWIND	04
#define	T_NOSWAP	040

#define	INF	(daddr_t)1000000L
int	tstimer();

/*
 * Software state per tape transport.
 * Also contains hardware state in message packets.
 *
 * 1. A tape drive is a unique-open device; we refuse opens when it is already.
 * 2. We keep track of the current position on a block tape and seek
 *    before operations by forward/back spacing if necessary.
 * 3. We remember if the last operation was a write on a tape, so if a tape
 *    is open read write and the last thing done is a write we can
 *    write a standard end of tape mark (two eofs).
 * 4. We remember the status registers after the last command, using
 *    then internally and returning them to the SENSE ioctl.
 */
struct	ts_softc {
	char	sc_openf;	/* lock against multiple opens */
	char	sc_lastiow;	/* last op was a write */
	char	sc_waiting;
	char	sc_busy;
	short	sc_resid;	/* copy of last bc */
	daddr_t	sc_blkno;	/* block number, for block device tape */
	daddr_t	sc_nxrec;	/* position of end of tape, if known */
	short	sc_tact;	/* timeout is active */
	daddr_t	sc_timo;	/* time until timeout expires in seconds */
	struct	ts_cmd *sc_cmd;	/* the command packet */
	struct	ts_sts sc_sts;	/* status packet, for returned status */
	struct	ts_char sc_char; /* characteristics packet */
#ifdef	QBUS
	u_short	sc_qb;		/* Qbus addr of cmd pkt for tsdb */
# define SETADDR(x)	sc->sc_qb = x | (hiword(x) & 0x03);
# define PGMADDR(x)	(x)->tsdb = sc->sc_qb;
#else	QBUS
# define SETADDR(x)
# define PGMADDR(x)	{ (x)->tssr = (((int)sc->sc_cmd >> 16) & 0xFF);\
			(x)->tsdb = (short)sc->sc_cmd; }
#endif	QBUS
} ts_softc[NTS];


/*
 * States for qm->qm_tab.b_active, the per controller state flag.
 * This is used to sequence control in the driver.
 */
#define	SSEEK	1		/* seeking */
#define	SIO	2		/* doing seq i/o */
#define	SCOM	3		/* sending control command */
#define	SREW	4		/* sending a drive rewind */

/*
 * Determine if there is a controller for a ts at address reg.  Our goal is to 
 * make the device interrupt.
 */
tsprobe(tsaddr)
register struct tsdevice *tsaddr;
{
	extern int cvec;

	tsaddr->tssr = TSSR_RESET;
	DELAY(100);
	if ((tsaddr->tssr & TS_NBA) == 0)
		return(0);
	/* IT'S TOO HARD TO MAKE THIS THING INTERRUPT JUST TO FIND ITS VECTOR */
	cvec = ((((unsigned)tsaddr & 0xf0) == 0x50) ? 0224 : 0230);
#ifndef	QBUS
	if (!badaddr(((u_short *)tsaddr)+2, 2) && battery_clock == 0  && 
	   ((*(((u_short *)tsaddr)+2))&0x1) == 0 ) {
		battery_clock = ((u_short *)tsaddr)+2;
		ts_battery_clock = 1;
	} else
		ts_battery_clock = 0;
#endif	QBUS
	return (sizeof (struct tsdevice));
}

/*
 * TS11 only supports one drive per controller; check for qi_slave == 0.
 */
tsslave(qi, reg)
struct qb_device *qi;
caddr_t reg;
{
	if (qi->qi_slave)	/* non-zero slave not allowed */
		return(0);
	return (1);
}

/*
 * Record attachment of the unit to the controller.
 */
tsattach(qi)
struct qb_device *qi;
{
#ifndef	QBUS
	struct tsdevice *tsaddr = (struct tsdevice *)qi->qi_mi->qm_addr;

	if (badaddr(((u_short *)tsaddr)+3, 2))
		printf("	1/4\" QIC2 tape drive");
	else
		printf("	1/2\" tape drive");
	if (ts_battery_clock)
		printf(" with battery clock");
#endif	QBUS
	return (1);
}

/*
 * Open the device.  Tapes are unique open devices, so we refuse if it is 
 * already open. We also check that a tape is available, and don't block waiting
 * here; if you want to wait for a tape you should timeout in user code.
 */

static inopen = 0; 	/* KLUDGE - sleep in tswait on open, nowhere else */

tsopen(dev, flag)
dev_t dev;
int flag;
{
	register int tsunit;
	register struct qb_device *qi;
	register struct ts_softc *sc;
	int s;

	tsunit = TSUNIT(dev);
	if (tsunit>=Nts || (sc = &ts_softc[tsunit])->sc_openf ||
	    (qi = tsdinfo[tsunit]) == 0 || qi->qi_alive == 0) {
		return (ENXIO);
	}
	inopen = 1;
	if (tsinit(tsunit)) {
		return (ENXIO);
	}
	inopen = 0;
	sc->sc_waiting = 0;
	sc->sc_busy = 0;
	tscommand(dev, TS_SENSE, 1);
	if ((sc->sc_sts.s_xs0&TS_ONL) == 0) {
		uprintf("ts%d: not online\n", tsunit);
		return (EIO);
	}
	if ((flag&(FREAD|FWRITE)) == FWRITE && (sc->sc_sts.s_xs0&TS_WLK)) {
		uprintf("ts%d: no write ring\n", tsunit);
		return (EIO);
	}
	sc->sc_openf = 1;
	sc->sc_blkno = (daddr_t)0;
	sc->sc_nxrec = INF;
	sc->sc_lastiow = 0;
	s = spl6();
	if (sc->sc_tact == 0) {
		sc->sc_timo = INF;
		sc->sc_tact = 1;
		timeout(tstimer, (caddr_t)dev, 10*hz);	/* every 10 secs */
	}
	splx(s);

	rtsblkno[tsunit] = 0;
	rtsbuf[tsunit][0].b_un.b_addr = tsbuffer[tsunit][0];
	rtsbuf[tsunit][0].b_bufsize = sizeof tsbuffer[tsunit][0];
	rtsbuf[tsunit][0].b_blkno = -1;
	rtsbuf[tsunit][1].b_un.b_addr = tsbuffer[tsunit][1];
	rtsbuf[tsunit][1].b_bufsize = sizeof tsbuffer[tsunit][1];
	rtsbuf[tsunit][1].b_blkno = -1;

	return (0);
}

/*
 * Close tape device. If tape was open for writing or last operation was a 
 * write, then write two EOF's and backspace over the last one. Unless this 
 * is a non-rewinding special file, rewind the tape. Make the tape available 
 * to others.
 */
tsclose(dev, flag)
register dev_t dev;
register flag;
{
	register struct ts_softc *sc = &ts_softc[TSUNIT(dev)];

	if (flag != FWRITE && (flag&FWRITE)) {
		int s = spl5();		/* wait for sc_lastiow to be valid */
		while (sc->sc_busy) {
			sc->sc_waiting = 1;
			sleep((caddr_t)&sc->sc_busy, PRIBIO);
		}
		splx(s);
	}
	if (flag == FWRITE || (flag&FWRITE) && sc->sc_lastiow) {
		tscommand(dev, TS_WEOF, 1);
		tscommand(dev, TS_WEOF, 1);
		tscommand(dev, TS_SREV, 1);
	}
	/*
	 * 0 count means don't hang waiting for rewind complete rather ctsbuf 
	 * stays busy until the operation completes preventing further opens 
	 * from completing by preventing a TS_SENSE from completing.
	 */
	if ((minor(dev)&T_NOREWIND) == 0)
		tscommand(dev, TS_REW, 0);
	sc->sc_openf = 0;
}

/*
 * Initialize the TS11.  Set up device characteristics.
 */
tsinit(unit)
register int unit;
{
	register struct ts_softc *sc = &ts_softc[unit];
	register struct qb_ctlr *qm = tscinfo[unit];
	register struct tsdevice *tsaddr = (struct tsdevice *)qm->qm_addr;
	int i;

	/*
	 * Now initialize the TS11 controller. Set the characteristics.
	 */
	if (tsaddr->tssr & (TS_NBA|TS_OFL)) {
		tsaddr->tssr = TSSR_RESET;	/* subsystem initialize */
		if (tswait(tsaddr,unit)) {
			return(1);
		}
		sc->sc_cmd = (struct ts_cmd *)
			(0x400 + (sizeof (struct ts_cmd) * unit)); /* KLUDGE */
		i = (int)sc->sc_cmd;
		if (hiword(i)&~0x03)
			panic("ts: command packet not in 18 bit land\n"); 
		if (i & 0x03)
			panic("ts: command packet not long aligned\n"); 
 		i = (i | (hiword(i) & 0x03));
 		SETADDR(i);

		i = svtop(&sc->sc_sts);
		sc->sc_char.char_ladr = loword(i);
		sc->sc_char.char_hadr = hiword(i);
		sc->sc_char.char_size = sizeof(struct ts_sts);
		sc->sc_char.char_mode = TS_ESS;

		sc->sc_cmd->c_cmd = TS_ACK | TS_CVC | TS_SETCHR;
		i = svtop(&sc->sc_char);
		sc->sc_cmd->c_loba = loword(i);
		sc->sc_cmd->c_hiba = hiword(i);
		sc->sc_cmd->c_size = sizeof(struct ts_char);
/*
printf("\n %x:	cmd	%x %x %x %x\n",sc->sc_cmd,sc->sc_cmd->c_cmd,sc->sc_cmd->c_loba,sc->sc_cmd->c_hiba,sc->sc_cmd->c_size);
printf(" %x	char	%x %x %x %x %x\n",&sc->sc_char,sc->sc_char.char_ladr,sc->sc_char.char_hadr,sc->sc_char.char_size,sc->sc_char.char_mode,sc->sc_char.ext_char_mode );
printf(" %x	sts	%x %x %x %x %x %x %x ",&sc->sc_sts,sc->sc_sts.s_sts,sc->sc_sts.s_len,sc->sc_sts.s_rbpcr,sc->sc_sts.s_xs0,sc->sc_sts.s_xs1,sc->sc_sts.s_xs2,sc->sc_sts.s_xs3);
if (sc->sc_sts.s_len >= (sizeof(struct ts_sts)-4))
	printf("%x",sc->sc_sts.s_xs4);
printf("\n");
printf("	scr	%x\n",tsaddr->tsdb);
printf("sr=%x ", ((int)sc->sc_cmd >> 16) & 0xFF);
printf("db=%x ", (short)sc->sc_cmd);
/**/
		PGMADDR(tsaddr);
		if (tswait(tsaddr,unit)) {
			return(1);
		}
		if (tsaddr->tssr & TS_NBA) {
			return(1);
		}
	}
	return(0);
}

/*
 * Execute a command on the tape drive a specified number of times.
 */
tscommand(dev, com, count)
dev_t dev;
int com, count;
{
	register struct buf *bp;
	register int s;

	bp = &ctsbuf[TSUNIT(dev)];
	s = spl5();
	while (bp->b_flags&B_BUSY) {
		/*
		 * This special check is because B_BUSY never gets cleared in 
		 * the non-waiting rewind case.
		 */
		if (bp->b_repcnt == 0 && (bp->b_flags&B_DONE))
			break;
		bp->b_flags |= B_WANTED;
		sleep((caddr_t)bp, PRIBIO);
	}
	bp->b_flags = B_BUSY|B_READ;
	splx(s);
	bp->b_dev = dev;
	bp->b_repcnt = count;
	bp->b_command = com;
	bp->b_blkno = 0;
	tsstrategy(bp);
	/*
	 * In case of rewind from close, don't wait. This is the only case 
	 * where count can be 0.
	 */
	if (count == 0)
		return;
	iowait(bp);
	if (bp->b_flags&B_WANTED)
		wakeup((caddr_t)bp);
	bp->b_flags &= B_ERROR;
}

/*
 * Queue a tape operation.
 */
tsstrategy(bp)
register struct buf *bp;
{
	int tsunit = TSUNIT(bp->b_dev);
	register struct qb_ctlr *qm;
	register struct buf *dp;
	register int s;

	/*
	 * Put transfer at end of controller queue
	 */
	bp->av_forw = NULL;
	qm = tsdinfo[tsunit]->qi_mi;
	s = spl5();
	dp = &tsutab[tsunit];
	if (dp->b_actf == NULL)
		dp->b_actf = bp;
	else
		dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	qm->qm_tab.b_actf = qm->qm_tab.b_actl = dp;
	/*
	 * If the controller is not busy, get it going.
	 */
	if (qm->qm_tab.b_active == 0)
		tsstart(qm, tsunit);
	splx(s);
}

/*
 * Start activity on a ts controller.
 */
tsstart(qm, tsunit)
register struct qb_ctlr *qm;
int tsunit;
{
	register struct buf *bp;
	register struct tsdevice *tsaddr = (struct tsdevice *)qm->qm_addr;
	register struct ts_softc *sc;
	register struct ts_cmd *tc;
	register struct qb_device *qi;
	int cmd;
	int ioaddr;
	daddr_t blkno;

	/*
	 * Start the controller if there is something for it to do.
	 */
loop:
	
	qi = tsdinfo[tsunit];
	sc = &ts_softc[tsunit];
	tc = sc->sc_cmd;
	sc->sc_busy = 0;
	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL) {
		if (sc->sc_waiting)
			wakeup((caddr_t)&sc->sc_busy);
		sc->sc_waiting = 0;
		return;
	}
	sc->sc_busy = 1;
	/*
	 * Default is that last command was NOT a write command; if we do a 
	 * write command we will notice this in tsintr().
	 */
	sc->sc_lastiow = 0;
	if (sc->sc_openf < 0 || (tsaddr->tssr&TS_OFL)) {
		/*
		 * Have had a hard error on a non-raw tape or the tape unit is 
		 * now unavailable (e.g. taken off line).
		 */
		bp->b_flags |= B_ERROR;
		goto next;
	}
	if (bp == &ctsbuf[TSUNIT(bp->b_dev)]) {
		/*
		 * Execute control operation with the specified count.
		 */
		sc->sc_timo = 15*60;		/* 15 mins */
		qm->qm_tab.b_active = (bp->b_command == TS_REW) ? SREW : SCOM;
		tc->c_repcnt = bp->b_repcnt;
		goto dobpcmd;
	}

	if (bp >= &rtsbuf[0][0]  &&  bp < &rtsbuf[NTS][0])
		blkno = bp->b_blkno;
	else
		blkno = bdbtofsb(bp->b_blkno);

	/*
	 * The following checks handle boundary cases for operation on non-raw 
	 * tapes. On raw tapes the initialization of sc->sc_nxrec by tsphys 
	 * causes them to be skipped normally (except in the case of retries).
	 */
	if (blkno > sc->sc_nxrec) {
		/*
		 * Can't read past known end-of-file.
		 */
		bp->b_flags |= B_ERROR;
		bp->b_error = ENXIO;
		goto next;
	}
	/*
	 * Reading at end of file returns 0 bytes.
	 */
	if (blkno == sc->sc_nxrec && bp->b_flags&B_READ) {
		clrbuf(bp);
		bp->b_resid = bp->b_bcount;
		goto next;
	}
	/*
	 * Writing sets EOF
	 */
	if ((bp->b_flags&B_READ) == 0)
		sc->sc_nxrec = blkno + 1;
	/*
	 * If the data transfer command is in the correct place, set up all the
	 * registers except the csr, and give control over to the UNIBUS 
	 * adapter routines, to wait for resources to start the i/o.
	 */
	if ((blkno = sc->sc_blkno) == blkno) {
		if ((bp->b_flags&B_READ) == 0)
			cmd = TS_WCOM;
		else
			cmd = TS_RCOM;
		if (qm->qm_tab.b_errcnt)
			cmd |= TS_RETRY;
		qm->qm_tab.b_active = SIO;
		sc->sc_timo = 1*60;		/* 1 min */
		tc->c_cmd = TS_ACK | TS_CVC | TS_IE | cmd;
		if (bp->b_dev & T_NOSWAP)
			tc->c_cmd |= TS_SWB;
		ioaddr = qbaddr(bp);
		tc->c_loba = loword(ioaddr);
		tc->c_hiba = hiword(ioaddr);
		tc->c_size = bp->b_bcount + (bp->b_bcount & 1);
/*printf("ts: start command %x, %x bytes to %x\n",cmd,bp->b_bcount,ioaddr);/**/
		PGMADDR(tsaddr);
		return;
	}
	/*
	 * Tape positioned incorrectly; set to seek forwards or backwards to 
	 * the correct spot. This happens for raw tapes only on error retries.
	 */
	sc->sc_timo = 15*60;			/* 15 mins */
	qm->qm_tab.b_active = SSEEK;
	if (sc->sc_blkno < blkno) {
		bp->b_command = TS_SFORW;
		tc->c_repcnt = blkno - sc->sc_blkno;
	} else {
		bp->b_command = TS_SREV;
		tc->c_repcnt = sc->sc_blkno - blkno;
	}
dobpcmd:
	/*
	 * Do the command in bp.
	 */
	tc->c_cmd = TS_ACK | TS_CVC | TS_IE | bp->b_command;
/*printf("ts: start cmd %x\n",bp->b_command);/**/
	PGMADDR(tsaddr);
	return;

next:
	/*
	 * Done with this operation due to error or the fact that it doesn't 
	 * do anything. Dequeue the transfer 
	 * and continue processing this slave.
	 */
	qm->qm_tab.b_errcnt = 0;
	qm->qm_tab.b_actf->b_actf = bp->av_forw;
	iodone(bp);
	goto loop;
}

/*
 * Ts interrupt routine.
 */
tsintr(ctlr)
int ctlr;
{
	register struct buf *bp;
	register struct qb_ctlr *qm = tscinfo[ctlr];
	register struct tsdevice *tsaddr;
	register struct ts_softc *sc;
	int tsunit;
	register state;

	if ((bp = qm->qm_tab.b_actf->b_actf) == NULL) {
		printf("TS%d: unexpected interrupt\n", ctlr);
		return;
	}
	tsunit = TSUNIT(bp->b_dev);
	tsaddr = (struct tsdevice *)qm->qm_addr;
	/*
	 * If last command was a rewind, and tape is still rewinding, wait for 
	 * the rewind complete interrupt.
	 *
	 * SHOULD NEVER GET AN INTERRUPT IN THIS STATE.
	 */
	if (qm->qm_tab.b_active == SREW) {
		qm->qm_tab.b_active = SCOM;
		if ((tsaddr->tssr&TS_SSR) == 0)
			return;
	}
	/*
	 * An operation completed... record status
	 */
	sc = &ts_softc[tsunit];
	sc->sc_timo = INF;
	if ((bp->b_flags & B_READ) == 0)
		sc->sc_lastiow = 1;
	state = qm->qm_tab.b_active;
	qm->qm_tab.b_active = 0;
/*
printf("ts%d: interrupt bn%d xs0=%b", TSUNIT(bp->b_dev),
	bp->b_blkno, sc->sc_sts.s_xs0, TSXS0_BITS);
if (sc->sc_sts.s_xs1)
	printf(" xs1=%b", sc->sc_sts.s_xs1, TSXS1_BITS);
if (sc->sc_sts.s_xs2)
	printf(" xs2=%b", sc->sc_sts.s_xs2, TSXS2_BITS);
if (sc->sc_sts.s_xs3)
	printf(" xs3=%b", sc->sc_sts.s_xs3, TSXS3_BITS);
if ((sc->sc_sts.s_len >= (sizeof(struct ts_sts)-4)) && (sc->sc_sts.s_xs4))
	printf(" xs4=%b", sc->sc_sts.s_xs4, TSXS4_BITS);
if (sc->sc_sts.s_rbpcr)
	printf(" rbpcr=%x", sc->sc_sts.s_rbpcr);
printf("\n");
printf("tssr= %b \n", tsaddr->tssr, TSSR_BITS);
/**/
	/*
	 * Check for errors.
	 */
	if (tsaddr->tssr&TS_SC) {
		switch (tsaddr->tssr & TS_TC) {
		case TS_UNREC:		/* unrecoverable */
		case TS_FATAL:		/* fatal error */
		case TS_ATTN:		/* attention (shouldn't happen) */
		case TS_RECNM:		/* recoverable, no motion */
			break;

		case TS_SUCC:		/* success termination */
			printf("ts%d: success\n", TSUNIT(minor(bp->b_dev)));
			goto ignoreerr;

		case TS_ALERT:		/* tape status alert */
			/*
			 * If we hit the end of the tape file, update position.
			 */
			if (sc->sc_sts.s_xs0 & (TS_TMK|TS_EOT)) {
				tsseteof(bp);		/* set blkno, nxrec */
				state = SCOM;		/* force completion */
				/*
				 * Stuff bc so it will be unstuffed correctly
				 * later to get resid.
				 */
				sc->sc_sts.s_rbpcr = bp->b_bcount;
				goto opdone;
			}
			/*
			 * If reading raw tape and the record was too long
			 * or too short, then we don't consider this an error.
			 */
			if (bp >= &rtsbuf[0][0]  &&  bp < &rtsbuf[NTS][0] && 
			    (bp->b_flags&B_READ) &&
			    sc->sc_sts.s_xs0&(TS_RLS|TS_RLL))
				goto ignoreerr;
		case TS_RECOV:		/* recoverable, tape moved */
			/*
			 * If this was an i/o operation retry up to 8 times.
			 */
			if (state==SIO)
				if (++qm->qm_tab.b_errcnt < 7)
					goto opcont;
				else
					sc->sc_blkno++;
			else if (sc->sc_openf>0 && 
			    !(bp >= &rtsbuf[0][0]  &&  bp < &rtsbuf[NTS][0]))
				/*
				 * Non-i/o errors on non-raw tape: close.
				 */
				sc->sc_openf = -1;
			break;

		case TS_REJECT:		/* function reject */
			if (state == SIO && sc->sc_sts.s_xs0 & TS_WLE)
				printf("ts%d: write locked\n", TSUNIT(bp->b_dev));
			if ((sc->sc_sts.s_xs0 & TS_ONL) == 0)
				printf("ts%d: offline\n", TSUNIT(bp->b_dev));
			break;
		}
		/*
		 * Couldn't recover error
		 */
		printf("ts%d: hard error bn%d xs0=%b", TSUNIT(bp->b_dev),
		    bp->b_blkno, sc->sc_sts.s_xs0, TSXS0_BITS);
		if (sc->sc_sts.s_xs1)
			printf(" xs1=%b", sc->sc_sts.s_xs1, TSXS1_BITS);
		if (sc->sc_sts.s_xs2)
			printf(" xs2=%b", sc->sc_sts.s_xs2, TSXS2_BITS);
		if (sc->sc_sts.s_xs3)
			printf(" xs3=%b", sc->sc_sts.s_xs3, TSXS3_BITS);
		if ((sc->sc_sts.s_len >= (sizeof(struct ts_sts)-4)) && 
		    (sc->sc_sts.s_xs4))
			printf(" xs4=%b", sc->sc_sts.s_xs4, TSXS4_BITS);
		printf("\n");
		printf("tssr= %b \n", tsaddr->tssr, TSSR_BITS);
		bp->b_flags |= B_ERROR;
		goto opdone;
	}
	/*
	 * Advance tape control FSM.
	 */
ignoreerr:
	switch (state) {

	case SIO:
		/*
		 * Read/write increments tape block number
		 */
		sc->sc_blkno++;
		goto opdone;

	case SCOM:
		/*
		 * For forward/backward space record update current position.
		 */
		if (bp == &ctsbuf[TSUNIT(bp->b_dev)])
			switch (bp->b_command) {
			case TS_SFORW:
				sc->sc_blkno += bp->b_repcnt; break;

			case TS_SREV:
				sc->sc_blkno -= bp->b_repcnt; break;
			}
		goto opdone;

	case SSEEK:
		sc->sc_blkno = bdbtofsb(bp->b_blkno);
		goto opcont;

	default:
		panic("tsintr");
	}
opdone:

/*
 *	KLUDGE - next 3 lines necessary since
 *	currently QIC II hangs on physical end of tape
 */
	if (sc->sc_sts.s_xs0 & TS_EOT) {
		printf("TS%d: unexpected physical EOT\n",tsunit);
		tsreset(-1);
		return;
	}

	/*
	 * Reset error count and remove from device queue.
	 */
	qm->qm_tab.b_errcnt = 0;
	qm->qm_tab.b_actf->b_actf = bp->av_forw;
	bp->b_resid = sc->sc_sts.s_rbpcr;
	iodone(bp);
/*	if (qm->qm_tab.b_actf->b_actf == 0)
/*		return;
/* */
opcont:
	tsstart(qm, tsunit);
}

tsseteof(bp)
register struct buf *bp;
{
	register int tsunit = TSUNIT(bp->b_dev);
	register struct ts_softc *sc = &ts_softc[tsunit];

	if (bp == &ctsbuf[TSUNIT(bp->b_dev)]) {
		if (sc->sc_blkno > bdbtofsb(bp->b_blkno)) {
			/* reversing */
			sc->sc_nxrec = bdbtofsb(bp->b_blkno) - sc->sc_sts.s_rbpcr;
			sc->sc_blkno = sc->sc_nxrec;
		} else {
			/* spacing forward */
			sc->sc_blkno = bdbtofsb(bp->b_blkno) + sc->sc_sts.s_rbpcr;
			sc->sc_nxrec = sc->sc_blkno - 1;
		}
		return;
	} 
	/* eof on read */
	if (bp >= &rtsbuf[0][0]  &&  bp < &rtsbuf[NTS][0])
		sc->sc_nxrec = bp->b_blkno + 1;
	else
		sc->sc_nxrec = bdbtofsb(bp->b_blkno);

}

tsread(dev, uio)
dev_t dev;
struct uio *uio;
{
	register struct iovec *iov = uio->uio_iov;
	register struct buf *bp, *nxtbp;
	register int blkno;
	register caddr_t base;
	register int len, unit, error, s;

	unit = TSUNIT(dev);
nextiov:
	if (uio->uio_iovcnt == 0)
		return (0);
	if (useracc(iov->iov_base, (u_int)iov->iov_len, B_WRITE) == NULL)
		return (EFAULT);
	base = iov->iov_base;
	while (iov->iov_len > 0) {
		blkno = rtsblkno[unit]++;
# ifdef DOUBLEBUF
		bp = &rtsbuf[unit][blkno&1];
		nxtbp = &rtsbuf[unit][(blkno+1)&1];
# else DOUBLEBUF
		bp = &rtsbuf[unit][0];
# endif DOUBLEBUF
		if (blkno != bp->b_blkno) {
		    if (error = tsphys(dev,blkno))
			return error;
		    tsrawio(dev,tsstrategy,bp,blkno,base,MAXLEN,B_READ);
		}
# ifdef DOUBLEBUF
		if (blkno+1 != nxtbp->b_blkno) {
		    if (error = tsphys(dev,blkno+1))
			return error;
		    tsrawio(dev,tsstrategy,nxtbp,blkno+1,base,MAXLEN,B_READ);
		}
# endif DOUBLEBUF
		s = spl5();
		while ((bp->b_flags & B_DONE) == 0)
			sleep((caddr_t)bp, PRIBIO);
		splx(s);
		len = MIN(MAXLEN - bp->b_resid, iov->iov_len);
		copyout(bp->b_un.b_addr, base, len);  
		s = spl6();
		if (bp->b_flags&B_WANTED)
			wakeup((caddr_t)bp);
		splx(s);
		iov->iov_len -= len;
		uio->uio_resid -= len;
		uio->uio_offset += len;
		base += len;
# ifdef NOTDEF
		if (bp->b_resid || (bp->b_flags&B_ERROR))
			break;
# else NOTDEF
/*
 *	it seems to me that a read should always return after
 *	one record has been read; this is precluded by the conditional
 *	break above, hence the change here
 */
		break;
# endif NOTDEF
	}
	bp->b_flags &= ~(B_BUSY|B_WANTED);
	error = geterror(bp);
	if (bp->b_resid || error)
		return (error);
	uio->uio_iov++;
	uio->uio_iovcnt--;
	goto nextiov;
}

tswrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register struct iovec *iov = uio->uio_iov;
	register struct buf *bp;
	register int blkno;
	register caddr_t base;
	register int len, unit, s, error;

	unit = TSUNIT(dev);
nextiov:
	if (uio->uio_iovcnt == 0)
		return (0);
	if (useracc(iov->iov_base, (u_int)iov->iov_len, B_READ) == NULL)
		return (EFAULT);
	base = iov->iov_base;
	while (iov->iov_len > 0) {
		blkno = rtsblkno[unit]++;
# ifdef DOUBLEBUF
		bp = &rtsbuf[unit][blkno&1];
# else DOUBLEBUF
		bp = &rtsbuf[unit][0];
# endif DOUBLEBUF
		if (bp->b_flags & B_BUSY) {
			s = spl6();
			while ((bp->b_flags & B_DONE) == 0)
				sleep((caddr_t)bp, PRIBIO);
			splx(s);
			if (bp->b_flags&B_WANTED)
				wakeup((caddr_t)bp);
			if ((bp->b_flags & B_READ) == 0) {
				error = geterror(bp);
				if (bp->b_resid || error) {
					bp->b_flags &= ~(B_BUSY|B_WANTED);
					return (error);
				}
			}
			bp->b_flags &= ~(B_BUSY|B_WANTED);
		}
		len = MIN(MAXLEN, iov->iov_len);
		if (error = tsphys(dev,blkno))
			return error;
		tsrawio(dev, tsstrategy, bp, blkno, base, len, B_WRITE);
		iov->iov_len -= len;
		uio->uio_resid -= len;
		uio->uio_offset += len;
		base += len;
	}
	uio->uio_iov++;
	uio->uio_iovcnt--;
	goto nextiov;
}

/*
 * Check that a raw device exists. If it does, set up sc_blkno and sc_nxrec
 * so that the tape will appear positioned correctly.
 */
tsphys(dev, blkno)
dev_t dev;
daddr_t blkno;
{
	register int tsunit = TSUNIT(dev);
	register struct ts_softc *sc;
	register struct qb_device *qi;

	if (tsunit >= Nts || (qi=tsdinfo[tsunit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	sc = &ts_softc[tsunit];
	sc->sc_nxrec = blkno + 1;
	return (0);
}


tsrawio(dev, strat, bp, blkno, base, len, rw)
int (*strat)(); 
register struct buf *bp;
daddr_t blkno;
{
	register int s;

	s = spl6();
	if (bp->b_flags & B_BUSY)
		while ((bp->b_flags & B_DONE) == 0)
			sleep((caddr_t)bp, PRIBIO);
	bp->b_flags &= B_QUALIFY;
	bp->b_flags |= B_BUSY | rw;
	splx(s);
	bp->b_error = 0;
	bp->b_proc = u.u_procp;
/*	bp->b_un.b_addr = XXXXX; */
	if ((bp->b_flags & B_READ) == 0)
		copyin(base, bp->b_un.b_addr, len);  
	bp->b_dev = dev;
	bp->b_blkno = blkno;
	bp->b_bcount = len;
	(*strat)(bp);
}

tsreset(which)
int	which ;
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register struct buf *dp;
	register struct tsdevice *tsaddr;
	register ctlr;

	for (ctlr = 0; ctlr < NTS; ctlr++) {
		if ( which >= 0  &&  ctlr != which )
			continue ;
		if ((qm = tscinfo[ctlr]) == 0 || qm->qm_alive == 0 )
			continue;
		tsaddr = (struct tsdevice *)qm->qm_addr;
		tsaddr->tssr = TSSR_RESET;
		qm->qm_tab.b_active = 0;
		qm->qm_tab.b_actf = qm->qm_tab.b_actl = 0;
		if (ts_softc[ctlr].sc_openf > 0)
			ts_softc[ctlr].sc_openf = -1;
		if ((qi = tsdinfo[ctlr]) && qi->qi_mi == qm && qi->qi_alive) {
			dp = &tsutab[ctlr];
			dp->b_active = 0;
			dp->b_forw = 0;
			if (qm->qm_tab.b_actf == NULL)
				qm->qm_tab.b_actf = dp;
			else
				qm->qm_tab.b_actl->b_forw = dp;
			qm->qm_tab.b_actl = dp;
		}
		(void) tsinit(ctlr);
		tsstart(qm, ctlr);
	}
}

tsioctl(dev, cmd, data, flag)
caddr_t data;
dev_t dev;
{
	int tsunit = TSUNIT(dev);
	register struct ts_softc *sc = &ts_softc[tsunit];
	register struct buf *bp = &ctsbuf[TSUNIT(dev)];
	register callcount;
	int fcount;
	struct mtop *mtop;
	struct mtget *mtget;
	/* we depend of the values and order of the MT codes here */
	static tsops[] =
	 {TS_WEOF,TS_SFORWF,TS_SREVF,TS_SFORW,TS_SREV,TS_REW,TS_OFFL,
		TS_SENSE,TS_RET};

	switch (cmd) {

	case MTIOCTOP:	/* tape operation */
		mtop = (struct mtop *)data;
		switch (mtop->mt_op) {

		case MTWEOF:
			callcount = mtop->mt_count;
			fcount = 1;
			break;

		case MTFSF: case MTBSF:
		case MTFSR: case MTBSR:
			callcount = 1;
			fcount = mtop->mt_count;
			break;

		case MTREW: case MTOFFL: case MTNOP: case MTRET:
			callcount = 1;
			fcount = 1;
			break;

		case MTRST:
			inopen = 1 ;
			tsreset(tsunit) ;
			inopen = 0 ;
			return(0) ;
		default:
			return (ENXIO);
		}
		if (callcount <= 0 || fcount <= 0)
			return (EINVAL);
		while (--callcount >= 0) {
			tscommand(dev, tsops[mtop->mt_op], fcount);
			if ((mtop->mt_op == MTFSR || mtop->mt_op == MTBSR) &&
			    bp->b_resid)
				return (EIO);
			if ((bp->b_flags&B_ERROR) || sc->sc_sts.s_xs0&TS_BOT)
				break;
		}
		return (geterror(bp));

	case MTIOCGET:
		mtget = (struct mtget *)data;
		mtget->mt_dsreg = 0;
		mtget->mt_erreg = sc->sc_sts.s_xs0;
		mtget->mt_resid = sc->sc_resid;
		mtget->mt_type = MT_ISTS;
		break;

	default:
		return (ENXIO);
	}
	return (0);
}

tstimer(dev)
	int dev;
{
	register struct ts_softc *sc = &ts_softc[TSUNIT(dev)];

	if (sc->sc_openf)
		timeout(tstimer, (caddr_t)dev, 10*hz);	/* every 10 secs */
	else {
		sc->sc_tact = 0;
		return;
	}

	if (sc->sc_timo != INF) {
		sc->sc_timo -= 10;
		if (sc->sc_timo <= 0) {
			printf("TS%d: lost interrupt\n", TSUNIT(dev));
			sc->sc_timo = INF;
			tsreset(-1);
		}
	}
}

#define	DBSIZE	20

tsdump()
{
#ifdef BOZO
	register struct qb_device *qi;
	register struct qb_regs *up;
	register struct tsdevice *tsaddr;
	int blk, num;
	int start;

	start = 0;
	num = maxfree;
#define	phys(a,b)	((b)((int)(a)&0x7fffffff))
	if (tsdinfo[0] == 0)
		return (ENXIO);
	qi = phys(tsdinfo[0], struct qb_device *);
	up = phys(qi->qi_hd, struct qb_hd *)->uh_physqb;
	qbinit(up);
	DELAY(1000000);
	tsaddr = (struct tsdevice *)qi->qi_physaddr;
	tsaddr->tssr = TSSR_RESET;
	tswait(tsaddr,0);  /*	MUST GIVE UNIT HERE */
	while (num > 0) {
		blk = num > DBSIZE ? DBSIZE : num;
		tsdwrite(start, blk, tsaddr, up);
		start += blk;
		num -= blk;
	}
	tseof(tsaddr);
	tseof(tsaddr);
	tswait(tsaddr,0);  /*	MUST GIVE UNIT HERE */
	if (tsaddr->tssr&TS_SC)
		return (EIO);
	tsaddr->tssr = TSSR_RESET;
	tswait(tsaddr,0);  /*	MUST GIVE UNIT HERE */
	return (0);
#else  BOZO
printf("ts: dump not implemented\n");
#endif BOZO
}

#ifdef BOZO
tsdwrite(dbuf, num, tsaddr, up)
register int dbuf, num;
register struct tsdevice *tsaddr;
struct qb_regs *up;
{
	register struct pte *io;
	register int npf;

	tswait(tsaddr,0);  /*	MUST GIVE UNIT HERE */
	io = up->qb_map;
	npf = num+1;
	while (--npf != 0)
		 *(int *)io++ = (dbuf++ | (1<<UBAMR_DPSHIFT) | UBAMR_MRV);
	*(int *)io = 0;
#ifdef notyet
	tsaddr->tsbc = -(num*NBPG);
	tsaddr->tsba = 0;
	tsaddr->tscs = TS_WCOM | TM_GO;
#endif
}
#endif BOZO

int waitloc[Nts];

# define MAXTIME 0x100

tswtimer(unit)
int unit;
{
	wakeup((caddr_t)&(waitloc[unit]));
}

tswait(tsaddr,unit)
register struct tsdevice *tsaddr;
int unit;
{
	register int s,i;
	register u_short addr;

	if (inopen) {
		for ( i = 0 ; i < MAXTIME ; i++) {
			if ((tsaddr->tssr & TS_SSR) != 0)
				return (0);
			s = spl6();
			timeout(tswtimer, (caddr_t)unit, 2*hz); /* 2 sec */
			sleep((caddr_t)&(waitloc[unit]), PRIBIO);
			splx(s);
		}
		return (1);
	}

	do {
		addr = tsaddr->tssr;
	}
	while ((addr & TS_SSR) == 0);
}

tseof(tsaddr)
struct tsdevice *tsaddr;
{

	tswait(tsaddr,0);
#ifdef notyet
	tsaddr->tscs = TS_WEOF | TM_GO;
#endif
}
#endif
