/*	sm.c	6.1	84/07/10	*/

#include "sm.h"

#if NSM > 0 || Nsm > 0
/*
 * Interphase VME-SMD disk (SM controler, sm disk) driver  --CTH--
 *
 * TODO:   Add linked commands: Remove sorting from strategy, allocate list of
 *	   iopb/bp pairs to be dispached in start routine, and freed in 
 *	   interrupt routine.
 */
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/dk.h"
#include "../h/dkbad.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/smreg.h"

#define Nsm_SM		4	/* max number of sm drives on SM controler */
#define SM_MAXCNT	0x1FF00	/* multiple of block size */
#define	SM_TIMEOUT	hz*5
#define SM_RETRYCNT	5

struct	size {
	daddr_t	nblocks;
	int	cyloff;
};

/* 
 * State of controller from last transfer. Since one transfer is done 
 * at a time per controller, only allocate one for each controller.
 */
struct	SM_stat {
	short		sm_ndrive;	/* Number of drives on controller */
	short		sm_wticks;	/* Monitor time for function */
	u_short		sm_cmd;		/* command to issue */
	u_int		sm_bn;		/* block number */
	u_int		sm_addr;	/* physical address of transfer */
	int		sm_nblks;	/* number of blocks to transfer */
	int		sm_drive;	/* drive/volume to issue */
} 			SM_stat[NSM];

int	smprobe(),	smslave(),	smattach(),	smintr();
int	smstrat(),	smminphys();
struct	qb_ctlr		*smcinfo[NSM];
struct	qb_device	*smdinfo[Nsm];

/* 
 * SM controler addresses 
 */
#ifdef	M68020
u_short *SMstd[] = { (u_short *)0xFD0000, 
		     (u_short *)0xFD0200, 
		     (u_short *)0xFD0400,
		     (u_short *)0xFD0600,
		     0};
#else	M68020
extern u_char shortio[];		/* virtual address of short io memory */
u_short *SMstd[] = { (u_short *)&shortio[0x0000], 
		     (u_short *)&shortio[0x0200], 
		     (u_short *)&shortio[0x0400],
		     (u_short *)&shortio[0x0600],
		     0};
#endif	M68020

/*
 * SM/sm controler/device driver structure
 */
struct	qb_driver SMdriver =
	{ smprobe, smslave, smattach, SMstd, "sm", smdinfo, "SM", smcinfo };

/* 
 * sm drive structure: each drive can have a different geometry.
 */
struct smst {
	ushort		st_ncpd;	/* Number of cylinders per drive */
	ushort		st_nspt;	/* Number of sectors per track */
	ushort		st_ntpc;	/* Number of tracks per cylinder */
	ushort		st_nspc;	/* Number of sectors per cylinder */
	struct size	st_size[8];
} smst[Nsm];

struct	buf	rsmbuf[Nsm];
struct	dkbad	smbad[Nsm];

#define	b_bn		b_resid	

int	smwstart, smwatch();		/* Have started guardian */
static int vector;
static int smpwrup = 0;
struct smuib smuib_dflt = { 0,10,0,0,100,0,512,16,32,1,3,1024,5,0,1,0xFF};

/* 
 * Check that controller exists
 */
smprobe(smaddr)
register struct smdevice	*smaddr;
{
	extern int cvec;

	smaddr->sm_cstatus |= CS_BDCLR;
	smaddr->sm_cstatus &= ~CS_BDCLR;
	if (smbusy(smaddr))
		return(0);
	cvec = vector = freevec();
	smpwrup = 0;
	return (sizeof (struct smdevice));
}

/*
 * Check that a given drive/volume exists.  
 */
smslave(qi, smaddr)
register struct qb_device	*qi;
register struct smdevice	*smaddr;
{
	register struct sm_iopb *iopb = &smaddr->sm_iopb[0];
	struct smuib *uib = (struct smuib *)0x400;	/* KLUDGE */
	int retry_cnt = 0;

	if ((smaddr->sm_cstatus & CS_BOK) == 0) {
		if (smpwrup++ == 0)
		    printf("	CONTROLLER FAILED POWER UP DIAGNOSTICS\n");
		return (0);
	}

 	/* if second volume, UIB already read, and the initialize done. */
	if (qi->qi_slave&0x1)
		goto ok;

	if ((smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1)&DS_DREADY)==0)
		return(0);
	smaddr->sm_cstatus &= ~CS_SLED;

	/* initialize portion of iopb that does not change */
	iopb->iopb_mem_type = MEMT_16BIT; 
	iopb->iopb_adr_mod = ADRM_STD_N_D;
	iopb->iopb_int_level = 3; 
	iopb->iopb_n_vector = iopb->iopb_e_vector = vector;
	iopb->iopb_dma_burst = DMA_BURST; 
	iopb->iopb_skew_offset = 0;

	/* clear any faults on the drive */
	iopb->iopb_cmd = CMD_RECALB;
	iopb->iopb_coptions = (qi->qi_slave&0x2)<<6;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
		printf("	sm%d	at SM%d	slave %d	", 
		    qi->qi_unit, qi->qi_ctlr, qi->qi_slave);
		printf("	*** DRIVE FAILED RECALIBRATE %x %x\n",
		    smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1),
		    iopb->iopb_error_code);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return (0);
	}
	smaddr->sm_cstatus &= ~CS_DONE;

	/* supply a good "DEFAULT" UIB */
	bcopy(&smuib_dflt, uib, sizeof(struct smuib));
	iopb->iopb_cmd = CMD_INIT;
	iopb->iopb_coptions =  (qi->qi_slave&0x2)<<6;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_buf_addrh = hiword(uib);
	iopb->iopb_buf_addrl = loword(uib);
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
		printf("	sm%d	at SM%d	slave %d	",
			qi->qi_unit, qi->qi_ctlr, qi->qi_slave);
		printf("	*** ERROR ON INIT %x %x\n",
			smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1),
			iopb->iopb_error_code);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return (0);
	}
	smaddr->sm_cstatus &= ~CS_DONE;

	/* read the UIB writen to the drive during format */
retry:	iopb->iopb_cmd = CMD_READ;
	iopb->iopb_coptions = (qi->qi_slave&0x2)<<6;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_dsk_addr.log.logh = UIB_DSK_ADDR >> 16;
	iopb->iopb_dsk_addr.log.logl = UIB_DSK_ADDR & 0xFFFF;
	iopb->iopb_blk_count = 1;
	iopb->iopb_buf_addrh = hiword(uib);
	iopb->iopb_buf_addrl = loword(uib);
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
	    if (retry_cnt++ > SM_RETRYCNT) {
		printf("	sm%d	at SM%d	slave %d	",
		    qi->qi_unit, qi->qi_ctlr, qi->qi_slave);
		printf("	*** ERROR READING UIB %x %x\n",
		    smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1),
		    iopb->iopb_error_code);
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return (0);
	    }
	    smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
	    goto retry;
	}
	smaddr->sm_cstatus &= ~CS_DONE;

	if (uib->uib_magic != UIB_MAGIC ) {
		printf("	sm%d	at SM%d	slave %d	",
		    qi->qi_unit, qi->qi_ctlr, qi->qi_slave);
		printf("	*** UNFORMATTED DRIVE\n");
		return (0);
	}
	uib->uib_magic = 0;
    ok:	if (uib->uib_vol_head[qi->qi_slave&1].num_heads == 0)
		return (0);
	return (1);
}

/*
 * Logicaly attach a unit to a drive/controler, and set up unit info
 */
smattach(qi)
register struct qb_device	*qi;
{
	register struct smdevice *smaddr = (struct smdevice *)qi->qi_mi->qm_addr;
	register struct sm_iopb *iopb = &smaddr->sm_iopb[0];
	struct smuib *uib = (struct smuib *)0x400;	/* KLUDGE */
	register struct smst *st = &smst[qi->qi_unit];

	/* Start watchdog for lost interupts from drive */
	if (smwstart == 0) {
		smwstart++;
		timeout(smwatch, (caddr_t)0, SM_TIMEOUT);
	}

	/* Initialize iostat values */
	if (qi->qi_dk >= 0)
		dk_mspw[qi->qi_dk] = .000008333;   /* 16bit transfer time? */
	SM_stat[qi->qi_ctlr].sm_ndrive++;

 	/* if second volume, initialize already done. */
	if (qi->qi_slave&0x1)
		goto ok;

	/* initialize with the UIB read from the drive */
	iopb->iopb_cmd = CMD_INIT;
	iopb->iopb_coptions =  (qi->qi_slave&0x2)<<6;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_buf_addrh = hiword(uib);
	iopb->iopb_buf_addrl = loword(uib);
	smaddr->sm_cstatus |= CS_GO;
	if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
		printf("	ERROR ON INIT ");
		smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
		return;
	}
	smaddr->sm_cstatus &= ~CS_DONE;

    ok:	st->st_ntpc = uib->uib_vol_head[qi->qi_slave&1].num_heads;
	uib->uib_vol_head[qi->qi_slave&1].num_heads == 0;	/* done */
	st->st_ncpd = uib->uib_num_cyl - 1;
	st->st_nspt = uib->uib_sec_trk;
	st->st_nspc = st->st_nspt * st->st_ntpc;
	printf("	%mM (%d x %d x %d) ", 
		st->st_nspt * st->st_ntpc * st->st_ncpd * 512,
		st->st_nspt, st->st_ntpc, st->st_ncpd);
	diskpart(st->st_size, st->st_nspt, st->st_ntpc, st->st_ncpd);
}

smopen(dev)
dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;

	if (unit >= Nsm || (qi = smdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	return (0);
}

smstrategy(bp)
register struct buf *bp;
{
	register struct qb_device *qi;
	register struct smst *st;
	register struct buf *bbp;
	register int unit;
	int partition = minor(bp->b_dev) & 07, s;
	long bn, sz;

	unit = dkunit(bp);
	if (unit >= Nsm)
		goto bad;
	qi = smdinfo[unit];
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	st = &smst[qi->qi_unit];
	sz = (bp->b_bcount+511) >> 9;
	if (bp->b_blkno < 0 ||
	    (bn = dkblock(bp))+sz > st->st_size[partition].nblocks)
		goto bad;
	/* 
	 * The controller can only do sector transfers. On a read we must
	 * do the transfer into a kernel buffer and then out to memory so
	 * we do not overwrite memory beyond the requested location.
	 */
	if ((bp->b_bcount & 511) && (bp->b_flags&B_READ)) {
		bbp = (struct buf *)getqeblk((bp->b_bcount+511) & ~511, 
			BQUALIFY(bp->b_dev));
		bbp->b_dev = bp->b_dev;
		bbp->b_blkno = bp->b_blkno;
		bbp->b_flags |= (bp->b_flags & (B_READ|B_WRITE));
		badstrat(qi, bbp, &smbad[qi->qi_unit], smstrat, st->st_nspt, 
			st->st_ntpc, st->st_ncpd,st->st_size[partition].cyloff);
		biowait(bbp);
		pcopyout(bbp->b_un.b_addr, qbaddr(bp), bp->b_bcount);
		bp->b_flags |= (bbp->b_flags & B_ERROR);
		bp->b_resid = bbp->b_resid;
		bbp->b_dev = NODEV;
		bfree(bbp);
		brelse(bbp);
		iodone(bp);
		return;
	}
	badstrat(qi, bp, &smbad[qi->qi_unit], smstrat, st->st_nspt, 
		st->st_ntpc, st->st_ncpd,st->st_size[partition].cyloff);
	return;

bad:	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

smstrat(qi, bp)
register struct qb_device *qi;
register struct buf *bp;
{
	register struct smst *st = &smst[qi->qi_unit];
	int partition = minor(bp->b_dev) & 07, s;
	register struct buf *dp;

	bp->b_bn = dkblock(bp) + (st->st_size[partition].cyloff * st->st_nspc);
	s = splx(qi->qi_mi->qm_psl);
	dp = &qi->qi_mi->qm_tab;
	disksort(dp, bp);
	if (dp->b_active == 0)
		smstart(qi->qi_mi);
	splx(s);
}

/*
 * Start up a transfer on a drive.
 */
smstart(qm)
register struct qb_ctlr *qm;
{
	register struct buf *bp;
	register struct qb_device *qi;
	register struct SM_stat *stat;

	if ((bp = qm->qm_tab.b_actf) == NULL)
		return;

	/* Mark controller busy, and determine destination.  */
	qm->qm_tab.b_active++;
	qi = smdinfo[dkunit(bp)];		/* Controller */
	stat = &SM_stat[qm->qm_ctlr];

	/* save away current data transfer drive info */
	stat->sm_bn = bp->b_bn + smst[qi->qi_unit].st_nspc;
	stat->sm_addr = qbaddr(bp);
	stat->sm_nblks = (bp->b_bcount+511) >> 9;
	stat->sm_drive = qi->qi_slave & 0x3;
	stat->sm_cmd = (bp->b_flags & B_READ) ? CMD_READ : CMD_WRITE;
	smio((struct smdevice *)qm->qm_addr, stat);

	if (qi->qi_dk >= 0) {
		dk_busy |= 1 << qi->qi_dk;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount >> 6;
	}
}

smio(smaddr, stat)
register struct smdevice *smaddr;
register struct SM_stat	*stat;
{
	register struct sm_iopb *iopb = &smaddr->sm_iopb[0];

	iopb->iopb_cmd = stat->sm_cmd;
	iopb->iopb_coptions = (stat->sm_drive<<6) | COP_LOG_TRAN | 
				COP_ECC_EN | COP_INT_EN;
	iopb->iopb_error_code = iopb->iopb_status = 0;
	iopb->iopb_dsk_addr.log.logh = hiword(stat->sm_bn);
	iopb->iopb_dsk_addr.log.logl = loword(stat->sm_bn);
	iopb->iopb_blk_count = stat->sm_nblks;
	iopb->iopb_buf_addrh = hiword(stat->sm_addr);
	iopb->iopb_buf_addrl = loword(stat->sm_addr);
	smaddr->sm_cstatus |= CS_GO;
}

smintr(ctlr)
register ctlr;
{
	register struct qb_ctlr *qm = smcinfo[ctlr];
	struct SM_stat *stat = &SM_stat[ctlr];
	register struct buf *bp = qm->qm_tab.b_actf;
	register struct smdevice *smaddr = (struct smdevice *)qm->qm_addr;
	register struct sm_iopb *iopb = &smaddr->sm_iopb[0];
	register struct qb_device *qi= smdinfo[dkunit(bp)];
	register unsigned int i;

	stat->sm_wticks = 0;
	dk_busy &= ~(1 << qi->qi_dk);

	/* Process errors on either the drive or the controller. */
	if (smaddr->sm_cstatus & CS_ERR ) {
		/* After 10 retries give up. */
		if (++qm->qm_tab.b_errcnt > SM_RETRYCNT) {
		    harderr(bp, "sm");
		    i = (iopb->iopb_dsk_addr.log.logh << 16) |
			(iopb->iopb_dsk_addr.log.logl&0xFFFF);
		    printf("esn=%d HARD %s %x\n", i-smst[qi->qi_unit].st_nspc, 
			    (iopb->iopb_cmd == CMD_READ) ? "READ" : "WRITE",
			    iopb->iopb_error_code);
		    bp->b_flags |= B_ERROR;
		} else
		    qm->qm_tab.b_active = 0;	/* force retry */

		/* If drive faulted, clear fault */
		if (smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1) & DS_FAULT) {
			i = (iopb->iopb_dsk_addr.log.logh << 16) |
			    (iopb->iopb_dsk_addr.log.logl&0xFFFF);
			printf("sm%d: sn%d RECALIBRATE %s %b\n", dkunit(bp),
			    i - smst[qi->qi_unit].st_nspc,
			    (iopb->iopb_cmd == CMD_READ) ? "READ" : "WRITE",
			    iopb->iopb_error_code, ERR_COR_BITS);
			smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);
			iopb->iopb_cmd = CMD_RECALB;
			iopb->iopb_coptions = (qi->qi_slave&0x2)<<6;
			iopb->iopb_error_code = iopb->iopb_status = 0;
			smaddr->sm_cstatus |= CS_GO;
			if (smwait(smaddr) || (smaddr->sm_cstatus & CS_ERR)) {
				harderr(bp, "sm");
				printf("failed recalibrate %x %x\n",
				    smaddr->sm_DSTATUS((qi->qi_slave&0x2)>>1),
			    	    iopb->iopb_error_code);
				bp->b_flags |= B_ERROR;
			}
		}
	} else if (iopb->iopb_status == STA_COMP_EX) {
		i = (iopb->iopb_dsk_addr.log.logh << 16) |
		    (iopb->iopb_dsk_addr.log.logl&0xFFFF);
		printf("sm%d: sn%d SOFT %s %b\n", dkunit(bp),
			i - smst[qi->qi_unit].st_nspc,
			(iopb->iopb_cmd == CMD_READ) ? "READ" : "WRITE",
			iopb->iopb_error_code, ERR_COR_BITS);
	}
	smaddr->sm_cstatus &= ~(CS_DONE | CS_ERR_LC);

	/* If still ``active'', then don't need any more retries. */
	if (qm->qm_tab.b_active) {
		dk_busy &= ~(1 << qi->qi_dk);
		qm->qm_tab.b_active = qm->qm_tab.b_errcnt = 0;
		bp->b_resid = 0;
		qm->qm_tab.b_actf = bp->av_forw;
		iodone(bp);
	}

	/* If controller not active and have request, start controller. */
	if (qm->qm_tab.b_actf && qm->qm_tab.b_active == 0)
		smstart(qm);
}

smread(dev, uio)
dev_t	dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nsm)
		return (ENXIO);
	return (physio(smstrategy, &rsmbuf[unit], dev, B_READ, smminphys, uio));
}

smwrite(dev, uio)
dev_t dev;
struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= Nsm)
		return (ENXIO);
	return (physio(smstrategy, &rsmbuf[unit], dev, B_WRITE, smminphys, uio));
}

/*
 * Reset driver after lost interupt. Cancsm software state of all pending 
 * transfers and restart all units and the controller.
 */
smreset(ctlr)
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register struct smdevice *smaddr;
	register struct SM_stat *stat;
	register int unit;

	printf(" SM%d", ctlr);
	smaddr = (struct smdevice *)qm->qm_addr;
	stat = &SM_stat[ctlr];
	qm->qm_tab.b_active = 0;
	smwait(smaddr);
	smstart(qm);
}

/*
 * Wake up every second and if an interrupt is pending but nothing has happened
 * increment a counter. If nothing happens for 20 seconds, reset the controler
 * and begin anew.
 */
smwatch()
{
	register struct qb_ctlr *qm;
	register ctlr, unit;
	register struct SM_stat *stat;

	timeout(smwatch, (caddr_t)0, SM_TIMEOUT);
	for (ctlr = 0; ctlr < NSM; ctlr++) {
		qm = smcinfo[ctlr];
		if (qm == 0 || qm->qm_alive == 0 || qm->qm_tab.b_active == 0)
			continue;
		stat = &SM_stat[ctlr];
		stat->sm_wticks++;
		if (stat->sm_wticks >= 20) {
			stat->sm_wticks = 0;
			printf("SM%d: lost interrupt\n", ctlr);
			smreset(ctlr);
		}
	}
}

smdump(smaddr)
{
}

smsize(dev)
dev_t dev;
{
	register int unit = minor(dev) >> 3;
	register struct qb_device *qi;
	register struct smst *st;

	if (unit >= Nsm || (qi = smdinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	st = &smst[qi->qi_unit];
	return (st->st_size[minor(dev)&0x7].nblocks);
}

smbusy(smaddr)
struct smdevice *smaddr;
{ 	
	long l = 0;

	DELAY(400);
	while (smaddr->sm_cstatus & CS_BUSY)  {
		if (++l == 50000)
			return (-1);
		DELAY(400);
	}
	return (0);
}

smwait(smaddr)
struct smdevice *smaddr;
{ 	
	long l = 0;

	DELAY(400);
	while ((smaddr->sm_cstatus & CS_DONE) == 0)  {
		if (++l == 50000)
			return (-1);
		DELAY(400);
	}
	return (0);
}

smminphys(bp)
struct buf *bp;
{
	if (bp->b_bcount < 512)
		return;
	bp->b_bcount = (min(bp->b_bcount, SM_MAXCNT)) & ~511;
}
#endif
