# undef RTRACE
# undef WTRACE
# undef TRACE

/*
 *	RAMPAGE DMA device driver
 *
 *	Mike Gottlieb - April 24, 1984
 *	Copyright 1984 Fluency International Software
 *
 *	hacked up at ISI to run under 4.2 - Mar 11, 1985 (dgw)
 */

#include "rp.h"
#if NRP > 0 || Nrp > 0

#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/ioctl.h"
#include "../h/tty.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/cmap.h"
#include "../h/uio.h"
#include "../h/kernel.h"

#include "../is68kdev/qbvar.h"
#include "../is68kdev/rpreg.h"

# define min(a,b) ((a) < (b) ? (a) : (b))

# define waitaddr(reg) { while (badaddr((caddr_t)(reg), 2)) ;}

/*
 * Driver information for auto-configuration stuff.
 */
int	rpprobe(), rpslave(), rpattach(), rprint();

struct	qb_ctlr *rpcinfo[NRP];
struct	qb_device *rpdinfo[NRP];

u_short	*rpstd[] = { (u_short *)0x3FF0C0, (u_short *)0x3FF0C8,
		     (u_short *)0x3FF0D0, (u_short *)0x3FF0D8,
		     0 };

struct	qb_driver RPdriver =
	{ rpprobe, rpslave, rpattach, rpstd, "rp", rpdinfo, "RP", rpcinfo };

/* Rampage i/o buffer structure */
struct rpbuf
{
	char		xmt[RPGBUFSIZ];	/* transmit buffer */
	char		rcv[RPGBUFSIZ];	/* receive buffer */
	unsigned short	xmtcnt;		/* space left in transmit buffer */
	char		state;		/* Rampage DMA buffer state */
	char		dir;		/* current direction (RCV,XMT) */
	char		open;		/* true if device is open, else false */
	char		autoflush;	/* true if buffer is to be flushed
					/*    after each write, else false */
} rpbuf[NRP];

rpprobe(rpaddr)		 /* make controller at reg interrupt */
	register struct rpdevice *rpaddr;
{
	rpaddr->csr = CS_RST;		/* reset the sucker */
	rpaddr->csr = CS_IE;		/* enable interrupts (and interrupt) */
	DELAY(120);
	return (sizeof (struct rpdevice));
}

rpslave(qi, rpaddr)
	struct qb_device *qi;
	struct rpdevice *rpaddr;
{
	return (1);
}

rpattach(qi)		 /* Record attachment of the unit to the controller. */
	struct qb_device *qi;
{
	printf("	Parallax Rampage Graphics Processor ");
	return (1);
}

rpopen(dev, flag)
	dev_t dev;
{
	register struct rpbuf *buf;
	register short unit;
	register struct rpdevice *rp_addr;

	unit = minor(dev);
	rp_addr = (struct rpdevice *)rpcinfo[unit]->qm_addr; 
	buf = rpbuf + unit;

	if ( unit > NRP )		/* check for valid minor */
		return ENXIO;

	if ( buf->open )		/* make sure buffer isn't open */
		return EMFILE;

	buf->open = TRUE;		/* set rampage to open state */
	waitaddr(rp_addr);
	rp_addr->csr = CS_RST;		/* reset rampage */
	DELAY(120);

	buf->xmtcnt = RPGBUFSIZ;	/* empty xmt buffer */
	buf->state = RPGINIT;		/* set state to initialize DMA */
	buf->dir = XMT;

	waitaddr(rp_addr);
	rp_addr->csr = CS_IE;		/* enable interrupts (and interrupt) */

	return 0;
}

rpclose(dev,flag)
	dev_t dev;
{
	register struct rpbuf *buf;
	register short unit;
	register int s;

	unit = minor(dev);
	buf = rpbuf + unit;

	rpioctl(dev,FLUSHBUF,0,0);	/* flush xmt buffer */

	s = spl5();
	if ( buf->state != RPGRDY )	/* wait for rampage to be inactive */
		sleep(buf,RPGPRI);
	splx(s);

	buf->open = FALSE;		/* set buffer to indicate not open */
	return 0;
}

rpioctl(dev, request, arg, mode)
	dev_t dev;
{
	register struct rpbuf *buf;
	register short unit;
	register int s;

	unit = minor(dev);
	buf = rpbuf + unit;

	switch ( request ) {
	    case FLUSHBUF:		/* immediate xmt buffer flush */
		if ( (buf->state == RPGRDY) && (buf->xmtcnt !=  RPGBUFSIZ) )
			rpstx(unit);
		break;

	    case AUTOFLUSH_ON:		/* flush xmt buffer after every write */
		buf->autoflush = TRUE;	
		break;

	    case AUTOFLUSH_OFF:		/* DONT flush xmt buffer after write */
		buf->autoflush = FALSE;
		break;

	default:
		uprintf("rp: ioctl error: invalid request\n");
		return EINVAL;
	}
	return 0;
}
	
rpwrite(dev,uio)
	dev_t dev;
	struct uio *uio;
{
	register struct rpbuf *buf;
	register short xfer,unit;
	register int error;
	register char *bufptr;
	register int s;

	unit = minor(dev);
	buf = rpbuf + unit;

	s = spl5();
	if ( buf->state != RPGRDY)
		sleep(buf,RPGPRI);
	splx(s);

	while (xfer = min(buf->xmtcnt, (unsigned)uio->uio_resid )) {
		s = spl5();
		if ( buf->state != RPGRDY) {	/* wait if buffer in use */
			sleep(buf,RPGPRI);
			xfer = min(buf->xmtcnt, (unsigned)uio->uio_resid);
		}
		splx(s);

		/* put user data into xmt buffer */
		bufptr = buf->xmt+RPGBUFSIZ-buf->xmtcnt;
		if (error = uiomove(bufptr, (int)xfer, UIO_WRITE, uio))
			return error;
# ifdef WTRACE
		{
			register int i; 

			printf("bufptr is %x, xfer is %x: ",
					bufptr,xfer);
			for (i=0; i < xfer; i+=2) {
			    printf("%x ",*(short*)bufptr);
			    bufptr += 2;
			}
			printf("\n");
		}
# endif TRACE

		/* flush buffer if in autoflush mode */
		if ( ((buf->xmtcnt -= xfer) == 0) || buf->autoflush )
			rpstx(unit);
	}
	return 0;
}

rpread(dev,uio)
	dev_t dev;
	struct uio *uio;
{
	register struct rpbuf *buf;
	register short unit,xfer;
	register int error;
	register int s;
	
	unit = minor(dev);
	buf = rpbuf + unit;

	/*
	 * WHEN READ IS CALLED XMT BUFFER MUST HAVE AS ITS LAST COMMAND A 
	 * RAMPAGE COMMAND WHICH SETS THE DIRECTION OF DATA TRANSFER from THE 
	 * RAMPAGE to THE HOST
	 */
	if (error  = rpioctl(dev,FLUSHBUF,0,0))	/* flush xmt buffer */
		return error;

	while (xfer = min(RPGBUFSIZ, (unsigned)uio->uio_resid )) {
		s = spl5();
		if ( buf->state != RPGRDY )	/* wait if buffer in use */
			sleep(buf,RPGPRI);
		splx(s);
# ifdef RTRACE
		printf("bufptr is %x, xfer is %x: ",buf->rcv,xfer);
# endif RTRACE
		if (error = uiomove(buf->rcv, (int)xfer, UIO_READ, uio))
			return error;

		/* if there's more data to come, restart rcv interrupts */
		if (uio->uio_resid)
			rpstr(unit);
	}
	return 0;
}

int rpstx(unit)			/* start xmt interrupts */
{
	register struct rpbuf *buf;

	buf = rpbuf + unit;
	buf->state = RPGBLO;
	rpintx(unit);
}

int rpstr(unit)			/* start rcv interrupts */
{
	register struct rpbuf *buf;

	buf = rpbuf + unit;
	buf->state = RPGBLO;
	rpintr(unit);
}


rprint(unit)			/* Rampage interrupt handler */
unsigned short unit;
{
	register struct buf *bp;
	register struct qb_ctlr *qm;
	register struct rpbuf *buf;
	register unsigned short csr;
	register struct rpdevice *rp_addr; 
	register int s;

# ifdef TRACE
putchar('I',0);
# endif TRACE
	s = spl5();
	qm = rpcinfo[unit];
	rp_addr = (struct rpdevice *)qm->qm_addr; 
	buf = rpbuf + unit;
	waitaddr(rp_addr);
	csr = rp_addr->csr; 

	if ( csr & CS_ER ) {	/* check if interrupt is due to Rampage error */
		uprintf("rp: error CSR=%x\n",csr);
		printf("rp: error CSR=%x\n",csr);
		buf->xmtcnt = RPGBUFSIZ;	/* empty xmt buffer */
		buf->state = RPGINIT;		/* state is initialize DMA */
		wakeup(buf);
		bp = qm->qm_tab.b_actf;
		bp->b_flags |= B_ERROR;
	}

	if ( csr & CS_DAV ) {		/* if Rampage has data available */
		if( buf->dir == XMT ) {	/* ... and buffer is in xmt direction */
			buf->dir = RCV;	/* ... change buffer to rcv direction */
			buf->state = RPGBLO;
		}
		rpintr(unit);		/* ... and start receive interrupts */
	} else {			/* if Rampage wants data */
		if( buf->dir == RCV ) {	/* ... and buffer is in rcv direction */
			buf->dir = XMT;	/* ... change buffer to xmt direction */
			buf->state = RPGDONE;
		}
		rpintx(unit);		/* ... and start xmt interrupts */
	}
	splx(s);
}

rpintx(unit)			/* transmit interrupt state machine */
{
	register struct rpbuf *buf;
	register struct rpdevice *rp_addr = 
				(struct rpdevice *)rpcinfo[unit]->qm_addr; 
	register int s;

	s = spl5();
	buf = rpbuf + unit;
	switch(buf->state) {
	    case RPGINIT:
		buf->state = RPGBLO;
		waitaddr(rp_addr);
		rp_addr->cdr = JUMPB;
		break;

	    case RPGBLO:
		buf->state = RPGBHI;
		waitaddr(rp_addr);
# ifdef WTRACE
printf("%x\n", (unsigned int) buf->xmt & LOWORD);
# endif WTRACE
		rp_addr->cdr =  (unsigned int) buf->xmt & LOWORD;
		break;
	    case RPGBHI:
		buf->state = RPGWCNT;
		waitaddr(rp_addr);
# ifdef WTRACE
printf("%x\n", (unsigned int) buf->xmt >> WORDBITS);
# endif WTRACE
		rp_addr->cdr = (unsigned int) buf->xmt >> WORDBITS;
		break;

	    case RPGWCNT:
		buf->state = RPGDONE;
		waitaddr(rp_addr);
# ifdef WTRACE
printf("%x\n", (RPGBUFSIZ - buf->xmtcnt) >> 1);
# endif WTRACE
		rp_addr->cdr = (RPGBUFSIZ - buf->xmtcnt) >> 1;
		break;

	    case RPGDONE:
		buf->xmtcnt = RPGBUFSIZ;
		buf->state = RPGRDY;
		wakeup(buf);
		break;
	}
	splx(s);
}

rpintr(unit)			/* receive interrupt state machine */
{
	register struct rpbuf *buf;
	register struct rpdevice *rp_addr = 
				(struct rpdevice *)rpcinfo[unit]->qm_addr; 
	register int s;

	s = spl5();
	buf = rpbuf + unit;

	switch(buf->state) {
	    case RPGBLO:
		buf->state = RPGBHI;
		waitaddr(rp_addr);
# ifdef RTRACE
printf("%x\n", (unsigned int) buf->rcv & LOWORD);
# endif RTRACE
		rp_addr->cdr =  (unsigned int) buf->rcv & LOWORD;
		break;

	    case RPGBHI:
		buf->state = RPGWCNT;
		waitaddr(rp_addr);
# ifdef RTRACE
printf("%x\n", (unsigned int) buf->rcv >> WORDBITS);
# endif RTRACE
		rp_addr->cdr = (unsigned int) buf->rcv >> WORDBITS;
		break;

	    case RPGWCNT:
		buf->state = RPGDONE;
		waitaddr(rp_addr);
# ifdef RTRACE
printf("%x\n", RPGBUFSIZ >> 1);
# endif RTRACE
		rp_addr->cdr = RPGBUFSIZ >> 1;
		break;

	    case RPGDONE:
		buf->state = RPGRDY;
		wakeup(buf);
		break;
	}
	splx(s);
}
#endif 
