/*	ra.c	6.1	83/07/29	*/

#include "ra.h"
#if NRA > 0 || Nra > 0
/*
 * UDA50/RAxx driver
 *
 * Restrictions:
 *	Unit numbers must be less than 8.
 *
 * TO DO:
 *	write dump code
 */
#include "../machine/pte.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/buf.h"
#include "../h/conf.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/map.h"
#include "../h/vm.h"
#include "../h/dk.h"
#include "../h/cmap.h"
#include "../h/uio.h"

#include "../is68kdev/qbvar.h"

#define	NRSPL2	3		/* log2 number of response packets */
#define	NCMDL2	3		/* log2 number of command packets */
#define	NRSP	(1<<NRSPL2)
#define	NCMD	(1<<NCMDL2)

#include "../is68kdev/rareg.h"
#include "../machine/mscp.h"

struct ra_softc {
	short	sc_state;	/* state of controller */
	int	sc_ivec;	/* interrupt vector address */
	short	sc_credits;	/* transfer credits */
	short	sc_lastcmd;	/* pointer into command ring */
	short	sc_lastrsp;	/* pointer into response ring */
} ra_softc[NRA];

/*
 * Controller states
 */
#define	S_IDLE	0		/* hasn't been initialized */
#define	S_STEP1	1		/* doing step 1 init */
#define	S_STEP2	2		/* doing step 2 init */
#define	S_STEP3	3		/* doing step 3 init */
#define	S_SCHAR	4		/* doing "set controller characteristics" */
#define	S_RUN	5		/* running */

struct ra {
	struct raca	ra_ca;		/* communications area */
	struct mscp	ra_rsp[NRSP];	/* response packets */
	struct mscp	ra_cmd[NCMD];	/* command packets */
} ra[NRA];

/* THIS SHOULD BE READ OFF THE PACK, PER DRIVE */
struct size {
	daddr_t	nblocks;
	daddr_t	blkoff;
} ra_sizes[8] ={
	15884,	0,		/* A=blk 0 thru 15883 */
	33440,	15884,		/* B=blk 15884 thru 49323 */
	-1,	0,		/* C=blk 0 thru end */
	15884,	340670,		/* D=blk 340670 thru 356553 */
	55936,	356554,		/* E=blk 356554 thru 412489 */
	-1,	412490,		/* F=blk 412490 thru end */
	82080,	49324,		/* G=blk 49324 thru 131403 */
	-1,	131404,		/* H=blk 131404 thru end */
};
/* END OF STUFF WHICH SHOULD BE READ IN PER DISK */

int	ra_error = 0;			/* causes hex dump of packets */
int	ra_debug = 0;
#define	printd	if (ra_debug) printf

daddr_t	radsize[Nra];			/* disk size, from ONLINE end packet */

int	raprobe(), raslave(), raattach(), raintr(), raminphys();
struct	mscp *ragetcp();

struct	qb_ctlr *raminfo[NRA];
struct	qb_device *radinfo[Nra];
struct	qb_device *raip[NRA][8];	/* 8 == max number of drives */

u_short	*rastd[] = { (u_short *)0x3FF468, (u_short *)0x3FF568, 
		     (u_short *)0x3FFF68, 0 };
struct	qb_driver RAdriver =
 { raprobe, raslave, raattach, rastd, "ra", radinfo, "RA", raminfo};

struct	buf rrabuf[NRA];
struct	buf rautab[NRA];
struct	buf rawtab[NRA];		/* I/O wait queue, per controller */

#define	b_qsize		b_resid		/* queue size per drive, in rautab */

/*
 * The following macros perform word swaping for accessing longs in mscp packets
 */
#define hiloval(x)	(((u_short)x/**/_h << 16) | (u_short)x/**/_l)
#define hiloass(x,v)	x/**/_h = ((u_long)v >> 16); x/**/_l = (u_short)v;

raprobe(reg, ctlr)		/* no other probe routine has this ctlr arg  */
	caddr_t reg;
	int ctlr;
{
	extern int cvec;
	register struct ra_softc *sc = &ra_softc[ctlr];

	cvec = sc->sc_ivec = freevec();
	return(sizeof (struct radevice));
}

raslave(qi, reg)
	struct qb_device *qi;
	caddr_t reg;
{
	return(1);
}

raattach(qi)
	register struct qb_device *qi;
{

	if (qi->qi_dk >= 0)
		dk_mspw[qi->qi_dk] = 1.0 / (60 * 31 * 256);	/* approx */
	qi->qi_flags = 0;
	raip[qi->qi_ctlr][qi->qi_slave] = qi;
	radsize[qi->qi_unit] = (daddr_t)0xffffff;	/* max possible size */
}

/*
 * Open a RA.  Initialize the device and set the unit online.
 */
raopen(dev, flag)
	dev_t dev;
	int flag;
{
	register int unit;
	register struct qb_device *qi;
	register struct ra_softc *sc;
	int s;

	unit = minor(dev) >> 3;
	if (unit >= NRA || (qi = radinfo[unit]) == 0 || qi->qi_alive == 0)
		return (ENXIO);
	sc = &ra_softc[qi->qi_ctlr];
	s = splx(qi->qi_mi->qm_psl);
	if (sc->sc_state != S_RUN) {
		if (sc->sc_state == S_IDLE)
			rainit(qi->qi_ctlr);
		/* wait for initialization to complete */
		sleep((caddr_t)qi->qi_mi, 0);
		if (sc->sc_state != S_RUN)
			return (EIO);
	}
	splx(s);
	return (0);
}

/*
 * Initialize a RA.  Set up UBA mapping registers, initialize data structures,
 * and start hardware initialization sequence.
 */
rainit(d)
	int d;
{
	register struct ra_softc *sc;
	register struct ra *ra;
	struct radevice *raaddr;
	struct qb_ctlr *qm;

	sc = &ra_softc[d];
	qm = raminfo[d];
	qm->qm_tab.b_active++;
	ra = &ra[d];
	raaddr = (struct radevice *)qm->qm_addr;

	/*
	 * Start the hardware initialization sequence. Initialization continues
	 * in interrupt routine.
	 */
	raaddr->ra_ip = 0;
	while ((raaddr->ra_sa & RA_STEP1) == 0)
		;
	raaddr->ra_sa = RA_ERR|(NCMDL2<<11)|(NRSPL2<<8)|RA_IE|(sc->sc_ivec/4);
	sc->sc_state = S_STEP1;
	sc->sc_credits = 0;
}

rastrategy(bp)
	register struct buf *bp;
{
	register struct qb_device *qi;
	register struct qb_ctlr *qm;
	register struct buf *dp;
	register int unit;
	int xunit = minor(bp->b_dev) & 07;
	daddr_t sz, maxsz;
	int s;

	sz = (bp->b_bcount+511) >> 9;
	unit = dkunit(bp);
	if (unit >= NRA)
		goto bad;
	qi = radinfo[unit];
	qm = qi->qi_mi;
	if (qi == 0 || qi->qi_alive == 0)
		goto bad;
	if ((maxsz = ra_sizes[xunit].nblocks) < 0)
		maxsz = radsize[unit] - ra_sizes[xunit].blkoff;
	if (bp->b_blkno < 0 || bp->b_blkno+sz > maxsz ||
	    ra_sizes[xunit].blkoff >= radsize[unit])
		goto bad;

	/*
	 * Link the buffer onto the drive queue
	 */
	s = splx(qi->qi_mi->qm_psl);
	dp = &rautab[qi->qi_unit];
	if (dp->b_actf == 0)
		dp->b_actf = bp;
	else
		dp->b_actl->av_forw = bp;
	dp->b_actl = bp;
	bp->av_forw = NULL;

	/*
	 * Link the drive onto the controller queue
	 */
	if (dp->b_active == 0) {
		dp->b_forw = NULL;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
		dp->b_active = 1;
	}
	if (qm->qm_tab.b_active == 0)
		rastart(qm);
	splx(s);

	return;

bad:
	bp->b_flags |= B_ERROR;
	iodone(bp);
	return;
}

rastart(qm)
	register struct qb_ctlr *qm;
{
	register struct buf *bp, *dp;
	register struct mscp *mp;
	register struct ra_softc *sc;
	register struct qb_device *qi;
	struct radevice *raaddr;
	int i, ioaddr;

	sc = &ra_softc[qm->qm_ctlr];
	
loop:
	if ((dp = qm->qm_tab.b_actf) == NULL) {
		/*
		 * Release uneeded UBA resources and return
		 */
		qm->qm_tab.b_active = 0;
		return (0);
	}
	if ((bp = dp->b_actf) == NULL) {
		/*
		 * No more requests for this drive, remove from controller queue
		 * and look at next drive. We know we're at the head of the 
		 * controller queue.
		 */
		dp->b_active = 0;
		qm->qm_tab.b_actf = dp->b_forw;
		goto loop;
	}
	qm->qm_tab.b_active++;
	raaddr = (struct radevice *)qm->qm_addr;
	if ((raaddr->ra_sa&RA_ERR) || sc->sc_state != S_RUN) {
		harderr(bp, "ra");
		printf("ra_sa %x, state %d\n", raaddr->ra_sa&0xffff, sc->sc_state);
		rainit(qm->qm_ctlr);
		/* SHOULD REQUEUE OUTSTANDING REQUESTS, LIKE UDRESET */
		return (0);
	}
	qi = radinfo[dkunit(bp)];
	/*
	 * If no credits, can't issue commands until other commands complete.
	 */
	if (sc->sc_credits < 2)
		return (0);
	if ((mp = ragetcp(qm)) == NULL)
		return (0);
	sc->sc_credits--;		/* committed to issuing a command */
	if (qi->qi_flags == 0) {	/* not online */
		mp->mscp_opcode = M_OP_ONLIN;
		mp->mscp_unit = qi->qi_slave;
		dp->b_active = 2;
		qm->qm_tab.b_actf = dp->b_forw;	/* remove from controller q */
		printd("ra: bring unit %d online\n", qi->qi_slave);
		*((long *)mp->mscp_dscptr + 1) |= RA_OWN|RA_INT;
		i = raaddr->ra_ip;
		goto loop;
	}

	hiloass(mp->mscp_cmdref, bp);	/* pointer to get back */
	mp->mscp_opcode = bp->b_flags&B_READ ? M_OP_READ : M_OP_WRITE;
	mp->mscp_unit = qi->qi_slave;
	hiloass(mp->mscp_lbn,bp->b_blkno + ra_sizes[minor(bp->b_dev)&7].blkoff);
	hiloass(mp->mscp_bytecnt, bp->b_bcount);
	ioaddr = qbaddr(bp);
	hiloass(mp->mscp_buffer, ioaddr);
	*((long *)mp->mscp_dscptr + 1) |= RA_OWN|RA_INT;
	i = raaddr->ra_ip;		/* initiate polling */
	if (qi->qi_dk >= 0) {
		dk_busy |= 1<<qi->qi_dk;
		dp->b_qsize++;
		dk_xfer[qi->qi_dk]++;
		dk_wds[qi->qi_dk] += bp->b_bcount>>6;
	}

	/*
	 * Move drive to the end of the controller queue
	 */
	if (dp->b_forw != NULL) {
		qm->qm_tab.b_actf = dp->b_forw;
		qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
		dp->b_forw = NULL;
	}
	/*
	 * Move buffer to I/O wait queue
	 */
	dp->b_actf = bp->av_forw;
	dp = &rawtab[qm->qm_ctlr];
	bp->av_forw = dp;
	bp->av_back = dp->av_back;
	dp->av_back->av_forw = bp;
	dp->av_back = bp;
	goto loop;
}

/*
 * RA interrupt routine.
 */
raintr(d)
	int d;
{
	register struct qb_ctlr *qm = raminfo[d];
	register struct radevice *raaddr = (struct radevice *)qm->qm_addr;
	struct buf *bp;
	int ioaddr;
	register int i;
	register struct ra_softc *sc = &ra_softc[d];
	register struct ra *rap = &ra[d];
	struct mscp *mp;

	printd("raintr: state %d, ra_sa %o\n", sc->sc_state, raaddr->ra_sa);
	switch (sc->sc_state) {
	case S_IDLE:
		printf("ra%d: random interrupt ignored\n", d);
		return;

	case S_STEP1:
#define	STEP1MASK	0174377
#define	STEP1GOOD	(RA_STEP2|RA_IE|(NCMDL2<<3)|NRSPL2)
		if ((raaddr->ra_sa&STEP1MASK) != STEP1GOOD) {
			sc->sc_state = S_IDLE;
			wakeup((caddr_t)qm);
			return;
		}
		raaddr->ra_sa = ((int)&rap->ra_ca.ca_ringbase);
		sc->sc_state = S_STEP2;
		return;

	case S_STEP2:
#define	STEP2MASK	0174377
#define	STEP2GOOD	(RA_STEP3|RA_IE|(sc->sc_ivec/4))
		if ((raaddr->ra_sa&STEP2MASK) != STEP2GOOD) {
			sc->sc_state = S_IDLE;
			wakeup((caddr_t)qm);
			return;
		}
		raaddr->ra_sa = ((int)&rap->ra_ca.ca_ringbase)>>16;
		sc->sc_state = S_STEP3;
		return;

	case S_STEP3:
#define	STEP3MASK	0174000
#define	STEP3GOOD	RA_STEP4
		if ((raaddr->ra_sa&STEP3MASK) != STEP3GOOD) {
			sc->sc_state = S_IDLE;
			wakeup((caddr_t)qm);
			return;
		}
		raaddr->ra_sa = RA_GO;
		sc->sc_state = S_SCHAR;

		/*
		 * Initialize the data structures.
		 */
		for (i = 0; i < NRSP; i++) {
			ioaddr = (int)(&(rap->ra_rsp[i].mscp_cmdref_l));
			rap->ra_ca.ca_rspdsc[i].rh = RA_OWN|RA_INT|hiword(ioaddr);
			rap->ra_ca.ca_rspdsc[i].rl = loword(ioaddr);
			rap->ra_rsp[i].mscp_dscptr = &rap->ra_ca.ca_rspdsc[i].rl;
			rap->ra_rsp[i].mscp_header.mscp_msglen = sizeof (struct mscp);
		}
		for (i = 0; i < NCMD; i++) {
			ioaddr = (int)(&(rap->ra_rsp[i].mscp_cmdref_l));
			rap->ra_ca.ca_cmddsc[i].ch = RA_INT|hiword(ioaddr);
			rap->ra_ca.ca_cmddsc[i].cl = loword(ioaddr);
			rap->ra_cmd[i].mscp_dscptr = &rap->ra_ca.ca_cmddsc[i].cl;
			rap->ra_cmd[i].mscp_header.mscp_msglen = sizeof (struct mscp);
		}
		bp = &rawtab[d];
		bp->av_forw = bp->av_back = bp;
		sc->sc_lastcmd = 0;
		sc->sc_lastrsp = 0;
		if ((mp = ragetcp(qm)) == NULL) {
			sc->sc_state = S_IDLE;
			wakeup((caddr_t)qm);
			return;
		}
		mp->mscp_opcode = M_OP_STCON;
		mp->mscp_cntflgs = M_CF_ATTN|M_CF_MISC|M_CF_THIS;
		*((long *)mp->mscp_dscptr + 1) |= RA_OWN|RA_INT;
		i = raaddr->ra_ip;	/* initiate polling */
		return;

	case S_SCHAR:
	case S_RUN:
		break;

	default:
		printf("ra%d: interrupt in unknown state %d ignored\n",
			d, sc->sc_state);
		return;
	}

	if (raaddr->ra_sa&RA_ERR) {
		printf("ra%d: fatal error (%o)\n", d, raaddr->ra_sa&0xffff);
		raaddr->ra_ip = 0;
		wakeup((caddr_t)qm);
	}

	/*
	 * Check for response ring transition.
	 */
	if (rap->ra_ca.ca_rspint) {
		rap->ra_ca.ca_rspint = 0;
		for (i = sc->sc_lastrsp;; i++) {
			i %= NRSP;
			if (rap->ra_ca.ca_rspdsc[i].rh&RA_OWN)
				break;
			rarsp(qm, ra, sc, i);
			rap->ra_ca.ca_rspdsc[i].rh |= RA_OWN;
		}
		sc->sc_lastrsp = i;
	}

	/*
	 * Check for command ring transition.
	 */
	if (rap->ra_ca.ca_cmdint) {
		printd("ra: command ring transition\n");
		rap->ra_ca.ca_cmdint = 0;
	}
	(void) rastart(qm);
}

/*
 * Process a response packet
 */
rarsp(qm, rap, sc, i)
	register struct qb_ctlr *qm;
	register struct ra *rap;
	register struct ra_softc *sc;
	int i;
{
	register struct mscp *mp;
	struct qb_device *qi;
	struct buf *dp, *bp;
	int st;

	mp = &rap->ra_rsp[i];
	mp->mscp_header.mscp_msglen = sizeof (struct mscp);
	sc->sc_credits += mp->mscp_header.mscp_credits & 0xf;
	if ((mp->mscp_header.mscp_credits & 0xf0) > 0x10)
		return;
	/*
	 * If it's an error log message (datagram), pass it on for more 
	 * extensive processing.
	 */
	if ((mp->mscp_header.mscp_credits & 0xf0) == 0x10) {
		raerror(qm, (struct mslg *)mp);
		return;
	}
	if (mp->mscp_unit >= 8)
		return;
	if ((qi = raip[qm->qm_ctlr][mp->mscp_unit]) == 0)
		return;
	st = mp->mscp_status&M_ST_MASK;
	switch (mp->mscp_opcode) {
	case M_OP_STCON|M_OP_END:
		if (st == M_ST_SUCC)
			sc->sc_state = S_RUN;
		else
			sc->sc_state = S_IDLE;
		qm->qm_tab.b_active = 0;
		wakeup((caddr_t)qm);
		break;

	case M_OP_ONLIN|M_OP_END:
		/*
		 * Link the drive onto the controller queue
		 */
		dp = &rautab[qi->qi_unit];
		dp->b_forw = NULL;
		if (qm->qm_tab.b_actf == NULL)
			qm->qm_tab.b_actf = dp;
		else
			qm->qm_tab.b_actl->b_forw = dp;
		qm->qm_tab.b_actl = dp;
		if (st == M_ST_SUCC) {
			qi->qi_flags = 1;	/* mark it online */
			radsize[qi->qi_unit] = (daddr_t)mp->mscp_untsize;
			printd("ra: unit %d online\n", mp->mscp_unit);
#ifdef notdef
			printf("ra%d: online, size=%d\n",
			      mp->mscp_unit, (daddr_t)mp->mscp_untsize);
#endif
		} else {
			harderr(dp->b_actf, "ra");
			printf("OFFLINE\n");
			while (bp = dp->b_actf) {
				dp->b_actf = bp->av_forw;
				bp->b_flags |= B_ERROR;
				iodone(bp);
			}
		}
		dp->b_active = 1;
		break;

	case M_OP_AVATN:
		printd("ra: unit %d attention\n", mp->mscp_unit);
		qi->qi_flags = 0;	/* it went offline and we didn't notice */
		break;

	case M_OP_READ|M_OP_END:
	case M_OP_WRITE|M_OP_END:
		bp = (struct buf *)hiloval(mp->mscp_cmdref);
		/*
		 * Unlink buffer from I/O wait queue.
		 */
		bp->av_back->av_forw = bp->av_forw;
		bp->av_forw->av_back = bp->av_back;
		dp = &rautab[qi->qi_unit];
		if (qi->qi_dk >= 0)
			if (--dp->b_qsize == 0)
				dk_busy &= ~(1<<qi->qi_dk);
		if (st == M_ST_OFFLN || st == M_ST_AVLBL) {
			qi->qi_flags = 0;	/* mark unit offline */
			/*
			 * Link the buffer onto the front of the drive queue
			 */
			if ((bp->av_forw = dp->b_actf) == 0)
				dp->b_actl = bp;
			dp->b_actf = bp;
			/*
			 * Link the drive onto the controller queue
			 */
			if (dp->b_active == 0) {
				dp->b_forw = NULL;
				if (qm->qm_tab.b_actf == NULL)
					qm->qm_tab.b_actf = dp;
				else
					qm->qm_tab.b_actl->b_forw = dp;
				qm->qm_tab.b_actl = dp;
				dp->b_active = 1;
			}
			return;
		}
		if (st != M_ST_SUCC) {
			harderr(bp, "ra");
			printf("status %o\n", mp->mscp_status);
			bp->b_flags |= B_ERROR;
		}
		bp->b_resid = bp->b_bcount - hiloval(mp->mscp_bytecnt);
		iodone(bp);
		break;

	case M_OP_GTUNT|M_OP_END:
		break;

	default:
		printf("ra: unknown packet\n");
	}
}


/*
 * Process an error log message
 *
 * For now, just log the error on the console. Only minimal decoding is done, 
 * only "useful" information is printed.  Eventually should send message to an 
 * error logger.
 */
raerror(qm, mp)
	register struct qb_ctlr *qm;
	register struct mslg *mp;
{
	printf("ra%d: %s error, ", qm->qm_ctlr,
		mp->mslg_flags&M_LF_SUCC ? "soft" : "hard");
	switch (mp->mslg_format) {
	case M_FM_CNTERR:
		printf("controller error, event 0%o\n", mp->mslg_event);
		break;

	case M_FM_BUSADDR:
		printf("host memory access error, event 0%o, addr 0%o\n",
			mp->mslg_event, mp->mslg_busaddr);
		break;

	case M_FM_DISKTRN:
		printf("disk transfer error, unit %d, grp 0x%x, hdr 0x%x\n",
			mp->mslg_unit, mp->mslg_group, hiloval(mp->mslg_hdr));
		break;

	case M_FM_SDI:
		printf("SDI error, unit %d, event 0%o, hdr 0x%x\n",
			mp->mslg_unit, mp->mslg_event, hiloval(mp->mslg_hdr));
		break;

	case M_FM_SMLDSK:
		printf("small disk error, unit %d, event 0%o, cyl %d\n",
			mp->mslg_unit, mp->mslg_event, mp->mslg_sdecyl);
		break;

	default:
		printf("unknown error, unit %d, format 0%o, event 0%o\n",
			mp->mslg_unit, mp->mslg_format, mp->mslg_event);
	}

	if (ra_error) {
		register long *p = (long *)mp;
		register int i;

		for (i = 0; i < mp->mslg_header.mscp_msglen; i += sizeof(*p))
			printf("%x ", *p++);
		printf("\n");
	}
}


/*
 * Find an unused command packet
 */
struct mscp *
ragetcp(qm)
	struct qb_ctlr *qm;
{
	register struct mscp *mp;
	register struct raca *cp;
	register struct ra_softc *sc;
	register int i;

	cp = &ra[qm->qm_ctlr].ra_ca;
	sc = &ra_softc[qm->qm_ctlr];
	i = sc->sc_lastcmd;
	if ((cp->ca_cmddsc[i].ch & (RA_OWN|RA_INT)) == RA_INT) {
		cp->ca_cmddsc[i].ch &= ~RA_INT;
		mp = &ra[qm->qm_ctlr].ra_cmd[i];
		mp->mscp_unit = mp->mscp_modifier = 0;
		mp->mscp_opcode = mp->mscp_flags = 0;
		hiloass(mp->mscp_bytecnt, 0);
		hiloass(mp->mscp_buffer, 0);
		hiloass(mp->mscp_errlgfl, 0);
		mp->mscp_copyspd = 0;
		sc->sc_lastcmd = (i + 1) % NCMD;
		return(mp);
	}
	return(NULL);
}

raread(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= NRA)
		return (ENXIO);
	return (physio(rastrategy, &rrabuf[unit], dev, B_READ, raminphys, uio));
}

rawrite(dev, uio)
	dev_t dev;
	struct uio *uio;
{
	register int unit = minor(dev) >> 3;

	if (unit >= NRA)
		return (ENXIO);
	return (physio(rastrategy, &rrabuf[unit], dev, B_WRITE, raminphys, uio));
}

rareset(qbn)
	int qbn;
{
	register struct qb_ctlr *qm;
	register struct qb_device *qi;
	register struct buf *bp, *dp;
	register int unit;
	struct buf *nbp;
	int d;

	for (d = 0; d < NRA; d++) {
		if ((qm = raminfo[d]) == 0 || qm->qm_alive == 0)
			continue;
		printf(" ra%d", d);
		qm->qm_tab.b_active = 0;
		qm->qm_tab.b_actf = qm->qm_tab.b_actl = 0;
		ra_softc[d].sc_state = S_IDLE;
		for (unit = 0; unit < NRA; unit++) {
			if ((qi = radinfo[unit]) == 0)
				continue;
			if (qi->qi_alive == 0 || qi->qi_mi != qm)
				continue;
			rautab[unit].b_active = 0;
			rautab[unit].b_qsize = 0;
		}
		for (bp = rawtab[d].av_forw; bp != &rawtab[d]; bp = nbp) {
			nbp = bp->av_forw;
			/*
			 * Link the buffer onto the drive queue
			 */
			dp = &rautab[dkunit(bp)];
			if (dp->b_actf == 0)
				dp->b_actf = bp;
			else
				dp->b_actl->av_forw = bp;
			dp->b_actl = bp;
			bp->av_forw = 0;
			/*
			 * Link the drive onto the controller queue
			 */
			if (dp->b_active == 0) {
				dp->b_forw = NULL;
				if (qm->qm_tab.b_actf == NULL)
					qm->qm_tab.b_actf = dp;
				else
					qm->qm_tab.b_actl->b_forw = dp;
				qm->qm_tab.b_actl = dp;
				dp->b_active = 1;
			}
		}
		rainit(d);
	}
}

radump()
{
	return(ENXIO);
}
 
rasiz(dev)
	dev_t dev;
{
	int unit = minor(dev) >> 3;
	struct qb_device *qi;

	if (unit >= NRA || (qi = radinfo[unit]) == 0 || qi->qi_alive == 0)
		return (-1);
	return (ra_sizes[minor(dev) & 07].nblocks);	/* XXX */
}

raminphys(bp)
	struct buf *bp;
{
	bp->b_bcount = min(bp->b_bcount, 0xf000); /* KLUDGE */
}
#endif
