/*
 * IBIC- Interface Bus Interactive Control program.  Interactively
 *	allows the user to send commands and receive output from 
 *	the GPIB.  Particularly useful for debugging newly written
 *	drivers, and troubleshooting complete GPIB systems.
 *
 * Copyright 1980 National Instruments
 * REV C:  01/01/82
 *
 * 16mar82- Source received and installed. 
 *	EOF set to -1 for v7 compatibility. -M. Nurmi
 *
 * 27sep82- Added introductory message.  Would be nice to 
 *	add a 'help' command some day. -M. Nurmi
 *
 * 27oct82- Removed 'errno==EINTR' check on EOF (getline()).  Errno is 
 *	not cleared on successive system calls.  This caused the user
 *	headaches in that he couldn't exit if an SRQ interrupt had 
 *	occurred. -M. Nurmi
 *
 * 27jun83- Cleaned up the code to make it more readable. -M.Nurmi
 */
extern int ibup(), gpib();
extern int errno;

#include <stdio.h>
#define EINTR   4

#define SIGSRQ 2
#define MAXLIN 256
#define MAXBUF 2048
#define MAXINT 0100000

char	*fns[] = { "ibup", "gpib" };
int	(*fctn[])() = { ibup, gpib };
int	maxcode[] = { 10, 14 };
struct	argdef { char *name, *args, *ans; };
struct	argdef	ibupargs[] = {
	{ "write",	"ISL",		"I" 	} ,
	{ "read",	"IBN",		"IB" 	} ,
	{ "clear",	"I",		"I" 	} ,
	{ "trigger",	"I",		"I" 	} ,
	{ "remote",	"I",		"I" 	} ,
	{ "local",	"I",		"I" 	} ,
	{ "poll",	"I",		"O" 	} ,
	{ "configure",	"III",		"I" 	} ,
	{ "passcontrol", "I",		"I" 	} ,
	{ "define",	"IIIIIII",	"I" 	} ,
	{ "finish",	"",		"I" 	}  
};
struct	argdef	gpibargs[] = {
	{ "command",		"SL",	"I" 	} ,
	{ "write",		"SLI",	"I" 	} ,
	{ "read",		"BNII",	"IB" 	} ,
	{ "transfer",		"",	"I" 	} ,
	{ "clear",		"",	"I" 	} ,
	{ "remote",		"",	"I" 	} ,
	{ "local",		"",	"I" 	} ,
	{ "parallelpoll",	"",	"O" 	} ,
	{ "passcontrol",	"",	"I" 	} ,
	{ "setstatus",		"I",	"I" 	} ,
	{ "monitor",		"I",	"I" 	} ,
	{ "readcommand",	"",	"O" 	} ,
	{ "setparameters",	"I",	"I" 	} ,
	{ "testsrq",		"",	"I" 	} ,
	{ "finish",		"",	"I" 	}  
};
struct	argdef	*fnargs[] = { ibupargs, gpibargs };
char	lbuf[MAXLIN], ansbuf[MAXBUF], *cp;
int	strlen, fn;
int	argc;
int	argv[9];
int	code;
extern	char	*dev_file;

main(acnt,aval)
int acnt;
char *aval[];
{ 
char *sccs_id = "%Z%  Module: %M%    Version: %I%    Date: %G%";
	int	n; 
	char	*atp, c;
	int	srqint();

	signal(SIGSRQ, srqint);

	/* print introductory message */

	if (acnt == 1) dev_file = "/dev/ib0";
	else switch (*aval[1])
		{
		case '0': dev_file = "/dev/ib0";
		     break;
		
		case '1': dev_file = "/dev/ib1";
		     break;
		
		case '2': dev_file = "/dev/ib2";
		     break;
		
		case '3': dev_file = "/dev/ib3";
		     break;
		
		default:
		     printf("Bad bus number, use (0-3)\n");
		     exit(1);
		}
	printf("\n\nUsing %s\n",dev_file);

	printf("\n Instrument Bus Interactive Control program\n");

loop:   
	while ((n=getline()) >= 0)
	{
		if (n == 0) continue;
		cp= lbuf;
		if ((fn=getfn()) < 0) continue;
		if ((code=getcode(fn)) < 0) continue;
		for (argc=0, atp=fnargs[fn][code].args; c= *atp++; )
			switch(c) {
			case 'N':
				if ((n=getint()) == MAXINT) goto loop;
				if (n > MAXBUF) n= MAXBUF;
				argv[argc++]= n;
				break;
			case 'I':
				if ((argv[argc++]= getint()) == MAXINT)
					goto loop;
				break;
			case 'S':
				if ((argv[argc++]= getstr()) == 0)
					goto loop;
				break;
			case 'L':
				argv[argc++]= strlen;
				break;
			case 'B':
				argv[argc++]= (int)ansbuf;
				break;
			}
		n= (*fctn[fn])(code,argv[0],argv[1],argv[2],argv[3],argv[4],argv[5]);
		for (atp= fnargs[fn][code].ans; c= *atp++; )
			switch(c) {
			case 'I': 
				putd(n); 
				putchar(' '); 
				break;
			case 'O': 
				if (n < 0) putd(n);
				else puto(n);
				putchar(' '); 
				break;
			case 'B': 
				put(ansbuf,n); 
				putchar(' '); 
				break;
			}
		putchar('\n');
	}       
}

/*****************************************************************/
srqint(){
	puts("SRQ interrupt\n");
	signal(SIGSRQ, srqint); 
}

/*****************************************************************/
getfn()
{ 
	int	fn;
	char	**a;

	skpspc();
	for (fn=0, a=fns; fn<2; fn++, a++)
		if(match(cp,*a)) break;
	if (fn >= 2){ 
		putl("unrecognized function"); 
		return -1; 
	}
	skpnspc();
	return fn; 
}

/*****************************************************************/
getcode(fn)
{ 
	int	f; 
	struct	argdef *a;

	skpspc();
	if (*cp>='0' && *cp<='9') f= atoi(cp);
	else for(f=0, a=fnargs[fn]; f<=maxcode[fn]; a++, f++)
		if(match(cp,a->name)) break;
	if (f<0 || f>maxcode[fn]) { 
		putl("illegal code"); 
		return -1; 
	}
	skpnspc();
	return f; 
}

/*****************************************************************/
getint()
{ 
	register int i;

	skpspc();
	if (*cp>='0' && *cp<='9' || *cp=='-') {
		i= atoi(cp);
		skpnspc();
		return i; 
	}
	putl("illegal integer argument");
	return MAXINT; 
}

/*****************************************************************/
getstr()
{ 
	char	*s;

	skpspc();
	if (*cp++=='"'){
		for(strlen=0, s=ansbuf; *cp && *cp!='"'; strlen++)
			*s++= cscan();
		skpnspc();
		return (int)ansbuf; 
	}
	putl("illegal string argument");
	return 0; 
}

/*****************************************************************/
skpspc()
{
	while(*cp && *cp<=' ') cp++; 
}

/*****************************************************************/
skpnspc()
{
	while(*cp>' ') cp++; 
}

/*****************************************************************/
putl(s) 
char *s;
{
	puts(s); 
	putchar('\n'); 
}

/*****************************************************************/
puts(s) 
char *s;
{
	while(*s) putchar(*s++); 
}

/*****************************************************************/
put(s,i) 
char *s;
{
	register int c;

	while(i-->0)
		if ((c= *s++&0377)>=' ' && c<0177)
			putchar(c);
		else {  
			putchar('\\');
			switch(c){
			case '\r': 
				putchar('r'); 
				break;
			case '\n': 
				putchar('n'); 
				break;
			case '\t': 
				putchar('t'); 
				break;
			case '\b': 
				putchar('b'); 
				break;
			case '\014': 
				putchar('p'); 
				break;
			default: 
				putchar(((c>>6)&7) +'0');
				putchar(((c>>3)&7) +'0');
				putchar((c&7) +'0');
				break;
			}               
		}       
}

/*****************************************************************/
putd(n)
{
	if (n<0){ 
		putchar('-'); 
		n= -n; 
	}
	putu(n); 
}

/*****************************************************************/
putu(u) 
unsigned u;
{
	if (u>=10) putu(u/10);
	putchar(u%10 + '0'); 
}

/*****************************************************************/
puto(o)
{
	if (o&0177770) puto((o>>3)&017777);
	putchar((o&7) +'0'); 
}

/*****************************************************************/
atoi(s) 
char *s;
{ 
	int sign, base, n;
	sign= 0;
	base= 10;
	if (*s=='-'){ 
		sign++; 
		s++; 
	}
	else if (*s=='0'){ 
		base= 8; 
		s++; 
	}
	for(n=0; *s>='0' && *s<='9'; ) n= n*base + *s++ - '0';
	return sign? -n:n; 
}

/*****************************************************************/
getline()
{ 
	char *s;
	putchar(':');
	for (s= lbuf; (*s= getchar()) != '\n'; ) {
		if (*s == EOF) {
			printf("\n");
			return s>lbuf? s-lbuf: -1;
		}
		else s++;
	}
	*s= 0;
	return s-lbuf; 
}

/*****************************************************************/
match(st,s) 
char *st, *s;
{ 
	register char *t;
	for(t=st; *t==*s || *t==(*s&~040); t++)
		if (*s++==0) return 1;
	if (t>st && *t<=' ') return 1;
	return 0; 
}

/*****************************************************************/
cscan()
{ 
	register int n, i;

	if (*cp!='\\') return *cp++;
	cp++;
	if (*cp>='0' && *cp<='7'){
		for(i=3, n=0; i-->0 && *cp>='0' && *cp<='7'; )
			n= (n<<3) + *cp++ -'0';
		return n; 
	}
	switch(*cp){
	case 0: 
		return 0;
	case 'n': 
		cp++; 
		return '\n';
	case 't': 
		cp++; 
		return '\t';
	case 'r': 
		cp++; 
		return '\r';
	case 'b': 
		cp++; 
		return '\b';
	case 'p': 
		cp++; 
		return '\014';
	default: 
		return *cp++;
	}       
}
