/*	kern_realtime.c	1.0	85/04/24	*/

/*
 * System calls for real time processes.
 */
#ifdef	REALTIME

#include "../machine/reg.h"
#include "../machine/pte.h"
#include "../machine/psl.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/map.h"
#include "../h/dir.h"
#include "../h/user.h"
#include "../h/kernel.h"
#include "../h/proc.h"
#include "../h/buf.h"
#include "../h/inode.h"
#include "../h/seg.h"
#include "../h/acct.h"
#include "../h/wait.h"
#include "../h/vm.h"
#include "../h/text.h"
#include "../h/file.h"
#include "../h/quota.h"
#include "../h/uio.h"
#include "../h/mbuf.h"
#include "../h/nami.h"

#ifdef	DEBUG
#define DEBUGPRINT(arg)	printf arg
#else	DEBUG
#define DEBUGPRINT(arg)	/* printf arg */
#endif	DEBUG

#define DEBUGENTRY(x)	DEBUGPRINT(("x(%d): entry\n", u.u_procp->p_pid))
#define DEBUGEXIT(x)	DEBUGPRINT(("x(%d): exit\n", u.u_procp->p_pid))

/*
 * Lock the process in core.
 * Bring in all pages and mark the process locked.
 */
plock()
{
	register struct proc * p = u.u_procp;
	register struct text * xp = p->p_textp;
	register int size;

	DEBUGENTRY(plock);
	/*
	 * Only a super user can issue this call
	 */
	if (!suser())
		goto ret;

	/*
	 * Compute the total space to be used by
	 * the process.  If it exceeds lockable
	 * memory limit, return with ENOMEM.
	 */
	size = p->p_tsize + p->p_dsize + p->p_ssize;
	if (xp && !(xp->x_flag & XPLOCK))
		size += xp->x_size;

	if (size > lockablemem) {
		u.u_error = ENOMEM;
		goto ret;
	}
	lockablemem -= size;

	/*
	 * Mark the process locked.
	 */
	p->p_flag |= SPLOCK;

	if (xp) {
		/*
		 * If no one has PLOCKED the text,
		 * bring in all the text pages.
		 */
		xlock(xp);
		if (!(xp->x_flag & XPLOCK)) {
			xp->x_flag |= XPLOCK;
			allpagesin(p, tptov(p, 0), xp->x_size);
		}
		xunlock(xp);
	}

	/*
	 * Bring data and stack pages in
	 */
	allpagesin(p, dptov(p, 0), p->p_dsize);
	allpagesin(p, sptov(p, p->p_ssize - 1), p->p_ssize);

ret:
	DEBUGEXIT(plock);
}

/*
 * Unlock the process from core
 * and make it swappable / pageable.
 */
punlock()
{
	register struct proc * p = u.u_procp;
	register struct text * xp = p->p_textp;

	DEBUGENTRY(punlock);

	p->p_flag &= ~SPLOCK;
	lockablemem += p->p_dsize + p->p_ssize + p->p_tsize;
	if (xp) {
		/*
		 * If no processes sharing this text
		 * are plocked, the text need not be
		 * locked anymore.  Clear XPLOCK bit.
		 */
		for (p = xp->x_caddr; p; p = p->p_xlink)
			if (p->p_flag&SPLOCK)
				goto ret;
		xp->x_flag &= ~XPLOCK;
		lockablemem += xp->x_size;
	}
ret:
	DEBUGEXIT(punlock);
}

/*
 * Mark the process to be a high priority process.
 */
highpri()
{
	DEBUGENTRY(highpri);
	/*
	 * Only a super user can issue this call
	 */
	if (!suser())
		goto ret;

	u.u_procp->p_flag |= SPHIGH;
ret:
	DEBUGEXIT(highpri);
}

/*
 * Make the process a normal priority process.
 */
normalpri()
{
	DEBUGENTRY(normalpri);
	u.u_procp->p_flag &= ~SPHIGH;
	DEBUGEXIT(normalpri);
}

/*
 * Bring pages v, v+1 ... v+npf-1 in.
 * If a page is not valid, bring it in.
 */
allpagesin(p, v, npf)
	struct proc * p;
	register unsigned v;
	register int npf;
{
	register struct pte *pte;

	DEBUGENTRY(allpagesin);
	pte = vtopte(p, v);
	while (npf > 0) {
		if (!pte->pg_v)
			pagein(ctob(v), 0);
		pte += CLSIZE;
		v += CLSIZE;
		npf -= CLSIZE;
	}
	DEBUGEXIT(allpagesin);
}
#endif	REALTIME
