/*	autoconf.c	6.3	83/08/11	*/

/*
 * Setup the system to run on the current machine: configure() is called at boot
 * time and initializes the bus devices. Available devices are determined (from
 * possibilities mentioned in ioconf.c), and the drivers are initialized.
 */

#include "../machine/pte.h"
#include "../machine/psl.h"

#include "../h/param.h"
#include "../h/systm.h"
#include "../h/map.h"
#include "../h/buf.h"
#include "../h/dk.h"
#include "../h/vm.h"
#include "../h/conf.h"
#include "../h/dmap.h"

#include "../is68k/scb.h"
#include "../is68kdev/qbvar.h"

/*
 * The following several variables are related to the configuration process, and
 * are used in initializing the machine.
 */
#ifndef	QBUS
#ifndef	M68020
u_char	shortio[];	/* virtual address of short io space */
#endif	M68020
#endif	QBUS
int	cvec;		/* vector used in interrupt */
int	clev;		/* level  used in interrupt */
int	dkn;		/* number of iostat dk numbers assigned so far */
int	howto;
int	devtype;

/*
 * The following structure is indexed by the major number, and should return
 * the two letter name for the block device.  NOTE: This should be part of the 
 * "bdevsw" block device switch structure.
 */
struct devname {
	char	*name;
}	devname[] = {
#ifdef	QBUS
	0, "el", "hp", "rk", "ra", "rl", "rd", "rx", "tm", "ts", 0
#else	QBUS
	0, "sd", "sm", "??", "??", "??", "rd", "??", "??", "ts", 0
#endif	QBUS
	};

char *clu_name_list[] = {
	"vb",
	"gp",
	"sk",
	(char *)0
};

/*
 * Determine controller, device, and memory configuration for a machine
 */
configure()
{
	printf("\n");
	sioinit();
	qbconfigure();
	/*
	 * If the root device was not specified in the configuration process,
	 * use the "devtype" paramater from boot to determine where the root and
	 * swap are located.
	 */
	if (rootdev == 0)
		rootdev = devtype;
	if (dumpdev == 0)
		if (minor(rootdev) & 0x7)
			dumpdev = rootdev;
		else
			dumpdev = makedev(major(rootdev), minor(rootdev)+1);
	if (argdev == 0)
		argdev = dumpdev;
	if (swdevt[0].sw_dev == 0)
		swdevt[0].sw_dev = dumpdev;
	printf("\n");
	printf("   root on %s%d%c", devname[major(rootdev)], 
		minor(rootdev)>>3, 'a'+(minor(rootdev)&7));
	printf(", dump on %s%d%c", devname[major(dumpdev)], 
		minor(dumpdev)>>3, 'a'+(minor(dumpdev)&7));
	printf(", args on %s%d%c", devname[major(argdev)], 
		minor(argdev)>>3, 'a'+(minor(argdev)&7));
	/*
	 * Configure swap area and related system parameter based on device(s) 
	 * used.
	 */
	swapconf();
	printf("\n\n");
#ifndef	M68020
	prcpuspeed();
	memenable();
	trapcheck();
#endif	M68020
}

/*
 * Configure controllers and devices on the bus.
 */
qbconfigure()
{
	register struct qb_device 	*qi;
	register struct qb_ctlr 	*qm;
	register struct qb_driver 	*udp;
	register struct ivecvec 	*iv;
	u_short 			**ap;
	u_short				*reg,
					*addr;
	int 				i;
	int 				s;
	int				*ip;
	extern int			vbnum;
	extern char			*clu_name_list[];

	/*
	 * Allow any pending interrupts to be serviced as STRAY
	 */
	s = spl0();
	DELAY(4000);
	splx(s);

	/*
	 * Check each bus controller. For each one which is potentially there,
	 * see if it is really there, and if it is record it and then go looking
	 * for slaves.
	 */
	for (qm = qbcinit; udp = qm->qm_driver; qm++) {
	    if ((int)udp == -1)
		continue;
	    if (vbnum > 0  &&  !name_in_list(udp->ud_dname, clu_name_list))
		continue;
	    addr = qm->qm_addr;
	    /*
	     * Use the particular address specified first, or if it is given as
	     * "0", or there is no device at that address, try all the standard
	     * addresses in the driver until we find it.
	     */
	    for (ap = udp->ud_addr; addr || (addr = *ap++); addr = NULL) {
		reg = addr;
		/*
		 * if the address is already in use, or we get a violation 
		 * trying to touch registers then continue.
		 */
		if (!iosavail((caddr_t)reg, 1) || badaddr((caddr_t)reg, 2))
			continue;
		cvec = NSCBVEC;
		clev = -1;
		/*
		 * Cause an interrupt on the device, asynchronously redefining 
		 * 'cvec' to be the controllers interrupt vector. Probe returns 
		 * the size of the controllers registers. Interupts must be 
		 * enabled during the probe.
		 */
		s = spl0();
		i = (*udp->ud_probe)(reg, qm->qm_ctlr);
		splx(s);
		if (i == 0  ||  !iosavail((caddr_t)reg, i))
			continue;
		printf("   %s%d	at ",udp->ud_mname,qm->qm_ctlr);
#ifdef	VBUS
#ifdef	M68020
		if ((addr >= (u_short *)0xFD0000) && 
		    (addr <= (u_short *)0xFDFFFE) )
			printf("shortio 0x%x/0%o		",
				(u_char *)addr - (u_char *)0xFD0000, 
				(u_char *)addr - (u_char *)0xFD0000);
		else
#else	M68020
		if ((addr >= (u_short *)shortio) && 
		    (addr <= (u_short *)&shortio[0xFFFE]) )
			printf("shortio 0x%x/0%o		",
				(u_char *)addr - shortio, 
				(u_char *)addr - shortio);
		else
#endif	M68020
#endif	VBUS
		    printf("address 0x%x/0%o	", svtop(addr), svtop(addr));

		switch (cvec) {
		    case 0:
			printf("*** no vector\n");
			qm->qm_psl = PSL_IPL6;
			goto novec;

		    case NSCBVEC:
			printf("*** didn't interrupt\n");
			continue;
		}
		iosused((caddr_t)reg, i);
		printf("vector 0x%x/0%o  ", cvec, cvec);
		if (cvec < SCBDEV) {
			printf("*** interrupt vector too low\n");
			continue;
		}
		/* fill in the interrupt mask psl for the controller */
		if (clev != -1) {
#ifdef	QBUS
			printf("level %d  ",clev + 3);
#else	QBUS
			printf("level %d  ",clev);
#endif	QBUS
			qm->qm_psl = PSL_CURMOD | (clev << PSL_IPL_SHIFT);
		} else
			qm->qm_psl = PSL_IPL6;
		/*
		 * fill in the interrupt vector slots for the controller,
		 * checking for devices interrupting at slots already used.
		 */
		for (iv = qm->qm_intr; iv->iv_intr; iv++) {
			if (scb.scb_vec.scb_blk[cvec + iv->iv_offs] != Xstray)
				break;
			scb.scb_vec.scb_blk[cvec + iv->iv_offs] = iv->iv_intr;
		}
		if (iv->iv_intr) {
			printf("*** vector in use\n");
			continue;
		}
		printf("\n");
		/*
		 * the controller is there at addr 'reg'. If indicated, fill in 
		 * the drivers back pointer to the qb_ctrl structure.
		 */
    novec:	qm->qm_alive = 1;
		qm->qm_addr = reg;
		if (udp->ud_minfo)
			udp->ud_minfo[qm->qm_ctlr] = qm;

		/*
		 * look through all the devices on the machine for ones that are
		 * controlled by the current controller, and are not yet alive.
		 */
		for (qi = qbdinit; qi->qi_driver; qi++) {
		    if (qi->qi_driver != udp || qi->qi_alive ||
		    	qi->qi_ctlr != qm->qm_ctlr && qi->qi_ctlr != '?' )
			    continue;
		    /*
		     * If the slave device is present then logically attach the 
		     * controller to the device.
		     */
		    if ((*udp->ud_slave)(qi, reg)) {
			qi->qi_alive = 1;
			qi->qi_ctlr = qm->qm_ctlr;
			if (qi->qi_dk && dkn < DK_NDRIVE)
			    qi->qi_dk = dkn++;
			else
			    qi->qi_dk = -1;
			qi->qi_mi = qm;
			udp->ud_dinfo[qi->qi_unit] = qi;
			printf("	%s%d	at %s%d	slave %d	",
				udp->ud_dname,qi->qi_unit,udp->ud_mname,
				qm->qm_ctlr,qi->qi_slave);
			(*udp->ud_attach)(qi);
			printf("\n");
		    }
		}
		break;
	    }
	}

	/*
	 * From now on all stray vectors will produce 'stray' messages in trap.
	 * Move the configured scb into the vector table area.
	 */
	cvec = 0;
	s = spl7();
	((int *)0)[0] = 0;		/* loc 0 == 0 for derefrence of NULL */
	((int *)0)[1] = 0;		/* not used ... */
	for (i = 2 ; i < NSCBVEC ; i++)
		((int *)0)[i] = (int)scb.scb_vec.scb_blk[i];
	splx(s);
}

freevec()
{
	int i;

	for (i = SCBDEV ; i < NSCBVEC ; i++)
		if ((i%4) == 0 && scb.scb_vec.scb_blk[i] == Xstray)
			return (i);
}

/*
 * The following structure is to detect overlapping io address space registers
 */
#define	NCTRLRS	20
struct iospace {
	char	*start;		/* address of first register of device */
	char	*end;		/* address just past last register of device */
} iosp[NCTRLRS];

iosused(p, n)
	char *p;
	int n;
{
	register struct iospace *sp;

	for (sp = iosp;  sp < &iosp[NCTRLRS];  sp += 1)
		if (sp->start == 0) {
			sp->start = p;
			sp->end = p + n;
			return;
		}
	panic("autoconf: too many controllers");	
}

iosavail(p, n)
	register char *p;
	int n;
{
	register struct iospace *sp;
	register char *q = p + n;

	for (sp = iosp;  sp < &iosp[NCTRLRS];  sp += 1)
		if (sp->start  &&  p < sp->end  &&  q > sp->start)
			return 0;
	return 1;
}

int
name_in_list(name, namelist)
	register char *name;
	register char *namelist[];
{
	if (namelist)
		while (*namelist)
			if (strcmp(name, *namelist++) == 0)
				return 1;
	return 0;
}

#define	DMMIN	8
#ifdef	QBUS
#define	DMMAX	1024
#define	DMTEXT	1024
#else	QBUS
#define	DMMAX	2048
#define	DMTEXT	2048
#endif	QBUS
#define	MAXDUMP	(10*2048)
/*
 * Configure swap space and related parameters.
 */
swapconf()
{
	register struct swdevt *swp;
	register int nblks;

	printf(", swap on");
	for (swp = swdevt; swp->sw_dev; swp++) {
		if (bdevsw[major(swp->sw_dev)].d_psize) {
			nblks =
			  (*bdevsw[major(swp->sw_dev)].d_psize)(swp->sw_dev);
			if (swp->sw_nblks == 0 || swp->sw_nblks > nblks)
			    swp->sw_nblks = nblks;
		}
		printf(" %s%d%c", devname[major(swp->sw_dev)], 
			minor(swp->sw_dev)>>3, 'a'+(minor(swp->sw_dev)&7));
	}
	if (dumplo == 0)
		dumplo = swdevt[0].sw_nblks - MAXDUMP;
	if (dumplo < 0)
		dumplo = 0;
	if (dmmin == 0)
		dmmin = DMMIN;
	if (dmmax == 0)
		dmmax = DMMAX;
	if (dmtext == 0)
		dmtext = DMTEXT;
	if (dmtext > dmmax)
		dmtext = dmmax;
}

/*
 *******************************************************************************
 * Calculate standard disk partition sizes.
 */
#include "../h/inode.h"
#include "../h/fs.h"

#define	NPARTITIONS	8
#define	PART(x)		(x - 'a')

/*
 * Default partition sizes, where they exist.
 */
#define	NDEFAULTS	5
#define	SMALLDISK	3
static int	defpart[NDEFAULTS][NPARTITIONS] = {
   { 15884, 66880, 0, 15884, 307200, 0, 0, 291346 },	/* ~ 356+ Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 291346 },	/* ~ 206-355 Mbytes */
   { 15884, 33440, 0, 15884, 55936, 0, 0, 0 },		/* ~ 61-205 Mbytes */
   { 15884, 10032, 0, 15884, 0, 0, 0, 0 },		/* ~ 20-60 Mbytes */
   { 0, 0, 0, 0, 0, 0, 0, 0 },				/* < 20 Mbytes */
};

/*
 * Each array defines a layout for a disk; that is, the collection of partitions
 * totally covers the physical space on a disk.
 */
#define	NLAYOUTS	3
static char	layouts[NLAYOUTS][NPARTITIONS] = {
   { 'a', 'b', 'h', 'g' },
   { 'a', 'b', 'h', 'd', 'e', 'f' },
   { 'c' },
};

/*
 * Default disk block and disk block fragment sizes for each file system.  
 * Those file systems with zero block and frag sizes are special cases
 * (e.g. swap areas or for access to the entire device).
 */
static struct	defparam {
	int	p_bsize;	/* block size */
	int	p_fsize;	/* frag size */
} defparam[NPARTITIONS] = {
	{ 8192, 1024 },		/* a */
	{ 0 },			/* b */
	{ 0 },			/* c */
	{ 8192, 1024 },		/* d */
	{ 4096, 512 },		/* e */
	{ 4096, 1024 },		/* f */
	{ 4096, 1024 },		/* g */
	{ 4096, 512 }		/* h */
};

/*
 * Each disk has some space reserved for a bad sector forwarding table.  DEC 
 * standard 144 uses the first 5 even numbered sectors in the last track of the
 * last cylinder for replicated storage of the bad sector table; another 126 
 * sectors past this is needed as a pool of replacement sectors.
 */
#define BADSECTTABLE	126

struct size {
	daddr_t	nblocks;
	int	cyloff;
};

diskpart(size, nspt, ntpc, ncpd)
	register struct size	*size;
{
	register int curcyl, def, part, layout;
	int 	threshhold, 
		numcyls[NPARTITIONS], 
		numblks[NPARTITIONS], 
		startcyl[NPARTITIONS];
	char 	*lp;
	int	badsecttable = BADSECTTABLE;	/* # sectors */
	int	xxx;
	register struct size	*ssize = size;
	int	nspc = nspt * ntpc;

	/*
	 * Bad sector table contains one track for the replicated copies of the
	 * table and enough full tracks preceding the last track to hold the 
	 * pool of free blocks to which bad sectors are mapped.
	 */
	badsecttable = nspt + roundup(badsecttable, nspt);
	threshhold = howmany(nspc, badsecttable);

	/* 
	 * Figure out if disk is large enough for expanded swap area and 'd', 
	 * 'e', and 'f' partitions.
	 */
	for (def = 0; def < NDEFAULTS; def++) {
		curcyl = 0;
		for (part = PART('a'); part < NPARTITIONS; part++)
			curcyl += howmany(defpart[def][part], nspc);
		if (curcyl < ncpd - threshhold)
			break;
	}

	if (def >= NDEFAULTS)
		return (-1);

	/*
	 * Calculate number of cylinders allocated to each disk partition. We 
	 * may waste a bit of space here, but it's in the interest of 
	 * compatibility (for mixed disk systems).
	 */
	for (curcyl = 0, part = PART('a'); part < NPARTITIONS; part++) {
		numcyls[part] = 0;
		numblks[part] = 0;
		if (defpart[def][part] != 0) {
			numcyls[part] = howmany(defpart[def][part], nspc);
			numblks[part] = defpart[def][part];
			curcyl += numcyls[part];
		}
	}
	numcyls[PART('f')] = ncpd - curcyl;
	numblks[PART('f')] = numcyls[PART('f')] * nspc - badsecttable;
	numcyls[PART('g')] =
		numcyls[PART('d')] + numcyls[PART('e')] + numcyls[PART('f')];
	numblks[PART('g')] = numcyls[PART('g')] * nspc - badsecttable;
	numcyls[PART('c')] = ncpd;
	numblks[PART('c')] = numcyls[PART('c')] * nspc;

	/*
	 * Calculate starting cylinder number for each partition. Note the 'h' 
	 * partition is physically located before the 'g' or 'd' partition.  
	 * This is reflected in the layout arrays defined above.
	 */
	for (layout = 0; layout < NLAYOUTS; layout++) {
		curcyl = 0;
		for (lp = layouts[layout]; *lp != 0; lp++) {
			if (numcyls[PART(*lp)])
				startcyl[PART(*lp)] = curcyl;
			else
				startcyl[PART(*lp)] = 0;
			curcyl += numcyls[PART(*lp)];
		}
	}

	if (defpart[def][PART('a')] == 0) {
	    xxx = numblks[PART('f')];
	    part = PART('a');

	    if (xxx >= defpart[SMALLDISK][part]) {
		startcyl[part] = 0;
		numcyls[part]=howmany( defpart[SMALLDISK][part], nspc);
		numblks[PART('a')] = defpart[SMALLDISK][part];
		xxx -= numcyls[part] * nspc;
		part = PART('b');
		if (xxx >= defpart[SMALLDISK][part]) {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		    xxx -= numcyls[part] * nspc;
		    part = PART('g');
		    if (xxx >= 0) {
		    	startcyl[part] = startcyl[PART('b')]+numcyls[PART('b')];
		    	numcyls[part] = howmany( xxx, nspc);
		    	numblks[part] = xxx;
		    }
	        } else {
		    startcyl[part] = startcyl[PART('a')]+numcyls[PART('a')];
		    numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		    numblks[part] = defpart[SMALLDISK][part];
		}
	    } else {
		startcyl[part] = 0;
		numcyls[part] = howmany( defpart[SMALLDISK][part], nspc);
		numblks[part] = defpart[SMALLDISK][part];
	    }
	}

#ifdef DFLAG_DISKPART
	if (dflag) {
		/*
		 * In case the disk is in the ``in-between'' range where the 'g'
		 * partition is smaller than the 'h' partition, reverse the frag
		 * sizes so the /usr partition is always set up with a frag size
		 * larger than the user's partition.
		 */
		if (defpart[def][PART('g')] < defpart[def][PART('h')]) {
			int temp;

			temp = defparam[PART('h')].p_fsize;
			defparam[PART('h')].p_fsize =
				defparam[PART('g')].p_fsize;
			defparam[PART('g')].p_fsize = temp;
		}
	}
#endif	DFLAG_DISKPART
	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		size->nblocks = numblks[part];
		size->cyloff = startcyl[part];
	}
	printpart(ssize, nspc);
}

printpart(isize, nspc)
	register struct size	*isize;
{
	int part, ppart;
	int ip, overlap = 0;
	register struct size	*size = isize;
	register struct size	*tsize;

	ppart = 0;
	for (part = PART('a'); part < NPARTITIONS; part++, size++) {
		if (size->nblocks == 0)
			continue;
		if ((ppart % 4) == 0)
			printf("\n		");
		printf("%c",'a'+part);
		ppart++;
		overlap = 0;

		for (tsize=isize, ip=PART('a'); ip < NPARTITIONS; ip++, tsize++)
			if (ip != part && tsize->nblocks != 0 &&
			    tsize->cyloff >= size->cyloff && 
			    tsize->cyloff*nspc+tsize->nblocks <=
			    size->cyloff*nspc+size->nblocks) {
				if (!overlap) printf(":");
				printf("%c",'a'+ip);
				overlap++;
			}
		printf(":%d", size->nblocks);
		if ((ppart)%4 != 0) {
			printf("	");
			if (size->nblocks < 100000)
				printf("	");
		}
	}
}
