/*#define	VERBOSE /**/
/*
 * Kernel Profiler Analizer
 */

/* THE FOLLOWING MUST AGREE WITH KERNEL DEFINITIONS IN trap.c */
#define KPROF		0x10000		/* size of profile buffer */
#define	KPROF_SHIFT	4		/* granularity */
#define	KPROF_MASK	0xF

#define MIN_CNT		1		/* min reference count threshold */

#include <sys/param.h>
#include <sys/file.h>
#include <stdio.h>
#include <ctype.h>
#include <a.out.h>

char	*kernelf = "/vmunix";
char	*kmemf = "/dev/kmem";

#define	N_KPROF		0		/* index of namelist items to lookup */
#define	N_KPROF_SPSL	1
#define	N_KPROF_UPSL	2
#define	N_KPROF_SU	3
#define	N_KPROF_ENABLE	4
#define	N_KPROF_MAGIC	5
#define	N_KPROF_SWTCH	6
#define	N_KPROF_BUSE	7
#define	N_KPROF_SYSC	8
#define	N_KPROF_PID	9
#define	N_KPROF_PID_CNT	10
#define	N_KPROF_PID_NAM	11
#define	N_ETEXT		12
#define N_NLST		13
struct nlist nlst[N_NLST];

unsigned int	kprof[KPROF];		/* copy of kernels kprof structures */
unsigned int	kprof_spsl[8];
unsigned int	kprof_upsl[8];
unsigned int	kprof_su[2];
unsigned int	kprof_enable;
unsigned int	kprof_magic;
unsigned int	kprof_swtch;
unsigned int	kprof_buse;
unsigned int	kprof_sysc;
unsigned int	kprof_pid[20+32];
unsigned int	kprof_pid_cnt[20+32];
unsigned char	kprof_pid_nam[20+32][8];
unsigned int	kprof_stillon;

struct prof {				/* massaged kprof structure */
	int	addr;
	int	cnt;
	char	*name;
}	prof[KPROF];
int	prof_valcmp();

struct npe {				/* massaged symbol table info */
	int	value;
	char	*name;
}	*npe,	*nl; 
int	npe_valcmp();

int	kmem;				/* fd for kmemf */
int	nsamples;			/* number of samples in system mode */
int	ssiz;				/* size of symbol table */
char	*strtab;			/* pointer to string table */
int	nname;				/* number of names in kernel namelist */
int	nsnap;				/* snapshot number */
int	min_cnt = MIN_CNT;		/* min percent system mode to print */

int	kprof_signal(), kprof_exit();	/* signal handler to start/stop, exit */

#define min(a, b)	( (a) < (b) : (a) ? (b))
#define max(a, b)	( (a) > (b) : (a) ? (b))

main(argc, argv)
	int argc;
	char *argv[];
{
	register int i;

	if (argc >= 2)
		min_cnt = atoi(argv[1]);
	nlst[N_KPROF].n_un.n_name =		"_kprof";
	nlst[N_KPROF_SPSL].n_un.n_name =	"_kprof_spsl";
	nlst[N_KPROF_UPSL].n_un.n_name =	"_kprof_upsl";
	nlst[N_KPROF_SU].n_un.n_name =		"_kprof_su";
	nlst[N_KPROF_ENABLE].n_un.n_name =	"_kprof_enable";
	nlst[N_KPROF_MAGIC].n_un.n_name =	"_kprof_magic";
	nlst[N_KPROF_SWTCH].n_un.n_name =	"_kprof_swtch";
	nlst[N_KPROF_BUSE].n_un.n_name =	"_kprof_buse";
	nlst[N_KPROF_SYSC].n_un.n_name =	"_kprof_sysc";
	nlst[N_KPROF_PID].n_un.n_name =		"_kprof_pid";
	nlst[N_KPROF_PID_CNT].n_un.n_name =	"_kprof_pid_cnt";
	nlst[N_KPROF_PID_NAM].n_un.n_name =	"_kprof_pid_nam";
	nlst[N_ETEXT].n_un.n_name =		"_etext";

	/* read kernels namelist, finding locations of kprof information */
	nlist(kernelf, nlst);
	for (i = 0 ; i < N_NLST ; i++) {
		if (nlst[0].n_type == 0) {
			fprintf(stderr, "kprof: %s not found in %s namelist\n",
				nlst[i].n_un.n_name,kernelf);
			exit(1);
		}
		nlst[i].n_value &= 0x0FFFFFFF;
	}
	if ((nlst[N_ETEXT].n_value >> KPROF_SHIFT) >= KPROF) {
		fprintf(stderr, "kprof: %s too big to profile!\n", kernelf);
		exit(1);
	}

	/* open memory mapping file to read/write kernel kprof information */
	kmem = open(kmemf, O_RDWR);
	if (kmem < 0) {
		fprintf(stderr, "kprof: can't open %s\n",kmemf);
		exit(2);
	}
	getnfile();

	if (lseek(kmem, nlst[N_KPROF_MAGIC].n_value, 0) < 0) {
		fprintf(stderr, "kprof: can't seek %s\n",kmemf);
		exit(3);
	}
	if (read(kmem,&kprof_magic,sizeof(kprof_magic))<sizeof(kprof_magic)) {
		fprintf(stderr, "kprof: can't read %s\n",kmemf);
		exit(3);
	}
	if (kprof_magic != 0x12345678) {
	    fprintf(stderr, "kprof: or not running %s\n", kernelf);
		exit(3);
	}
	kprof_enable = 0;		/* turn off profiling */
	lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
	if (write(kmem,&kprof_enable,sizeof(kprof_enable))<sizeof(kprof_enable)) {
		fprintf(stderr, "kprof: can't write %s\n",kmemf);
		exit(3);
	}
	resetstate();

	getpid();
	/* toggle state of profile on hangup signal */
#define	KPROF_SIG_SS	SIGINT			/* ^C */
#define	KPROF_SIG_EXIT	SIGQUIT			/* ^\ */
	signal(KPROF_SIG_SS, kprof_signal);
	signal(KPROF_SIG_EXIT, kprof_exit);
	fprintf(stderr, "\n  Kernel Profiler:\n");
	fprintf(stderr, "	kill -%d %d	Start/Stop Snapshots\n",
			KPROF_SIG_SS, getpid());
	fprintf(stderr, "	kill -%d %d	Stop Kernel Profiler\n",
			KPROF_SIG_EXIT, getpid());

	/* all action resluts from signals ... */
	while (1)
		pause();
}

kprof_exit()
{

	if (kprof_enable) {
		lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
		read(kmem, &kprof_stillon, sizeof (kprof_stillon));
		kprof_enable = 0;		/* turn off profiling */
		lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
		write(kmem, &kprof_enable, sizeof (kprof_enable));
		fprintf(stderr,"*** Stop      Snapshot %d ***\n", nsnap);
		fprintf(stderr,"*** Analyze   Snapshot %d ***\n", nsnap);
		getstate();			/* get profile information */
		asgnsamples();			/* match it to symbol table */
		dumpstate();			/* print out information */
		resetstate();			/* reset profile information */
		fprintf(stderr,"*** Done      Snapshot %d ***\n", nsnap);
	}
	fprintf(stderr,"*** KPROF     STOPPED  %d ***\n", nsnap);
	exit (0);
}

kprof_signal()
{
	if (kprof_enable == 0) {
		kprof_enable = -1;		/* turn on profiling */
		nsnap++;
		fprintf(stderr,"*** Start   Snapshot %d ***\n", nsnap);
		lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
		write(kmem, &kprof_enable, sizeof (kprof_enable));
	} else {
		lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
		read(kmem, &kprof_stillon, sizeof (kprof_stillon));
		kprof_enable = 0;		/* turn off profiling */
		lseek(kmem, nlst[N_KPROF_ENABLE].n_value, 0);
		write(kmem, &kprof_enable, sizeof (kprof_enable));
		fprintf(stderr,"*** Stop    Snapshot %d ***\n", nsnap);
		fprintf(stderr,"*** Analyze   Snapshot %d ***\n", nsnap);
		getstate();			/* get profile information */
		asgnsamples();			/* match it to symbol table */
		dumpstate();			/* print out information */
		resetstate();			/* reset profile information */
		fprintf(stderr,"*** Done      Snapshot %d ***\n", nsnap);
	}
}

getstate()
{
	lseek(kmem, nlst[N_KPROF].n_value, 0);
	read(kmem, kprof, sizeof(kprof));
	lseek(kmem, nlst[N_KPROF_SPSL].n_value, 0);
	read(kmem, kprof_spsl, sizeof(kprof_spsl));
	lseek(kmem, nlst[N_KPROF_UPSL].n_value, 0);
	read(kmem, kprof_upsl, sizeof(kprof_upsl));
	lseek(kmem, nlst[N_KPROF_SU].n_value, 0);
	read(kmem, kprof_su, sizeof(kprof_su));
	lseek(kmem, nlst[N_KPROF_SWTCH].n_value, 0);
	read(kmem, &kprof_swtch, sizeof(kprof_swtch));
	lseek(kmem, nlst[N_KPROF_BUSE].n_value, 0);
	read(kmem, &kprof_buse, sizeof(kprof_buse));
	lseek(kmem, nlst[N_KPROF_SYSC].n_value, 0);
	read(kmem, &kprof_sysc, sizeof(kprof_sysc));
	lseek(kmem, nlst[N_KPROF_PID].n_value, 0);
	read(kmem, kprof_pid, sizeof(kprof_pid));
	lseek(kmem, nlst[N_KPROF_PID_CNT].n_value, 0);
	read(kmem, kprof_pid_cnt, sizeof(kprof_pid_cnt));
	lseek(kmem, nlst[N_KPROF_PID_NAM].n_value, 0);
	read(kmem, kprof_pid_nam, sizeof(kprof_pid_nam));
}

resetstate()
{
	register int i, j;

	for (i = 0 ; i < KPROF ; i++)
		kprof[i] = 0;
	for (i = 0 ; i < 8 ; i++)
		kprof_upsl[i] = kprof_spsl[i] = 0;
	kprof_su[0] = kprof_su[1] = 0;
	kprof_sysc = kprof_buse = kprof_swtch = 0;
	for (i = 0 ; i < (20+32) ; i++) {
		kprof_pid_cnt[i] = kprof_pid[i] = 0;
		for (j = 0 ; j < 8 ; j++)
			kprof_pid_nam[i][j] = 0;
	}
	lseek(kmem, nlst[N_KPROF].n_value, 0);
	write(kmem, kprof, sizeof(kprof));
	lseek(kmem, nlst[N_KPROF_SPSL].n_value, 0);
	write(kmem, kprof_spsl, sizeof(kprof_spsl));
	lseek(kmem, nlst[N_KPROF_UPSL].n_value, 0);
	write(kmem, kprof_upsl, sizeof(kprof_upsl));
	lseek(kmem, nlst[N_KPROF_SU].n_value, 0);
	write(kmem, kprof_su, sizeof(kprof_su));
	lseek(kmem, nlst[N_KPROF_SWTCH].n_value, 0);
	write(kmem, &kprof_swtch, sizeof(kprof_swtch));
	lseek(kmem, nlst[N_KPROF_BUSE].n_value, 0);
	write(kmem, &kprof_buse, sizeof(kprof_buse));
	lseek(kmem, nlst[N_KPROF_SYSC].n_value, 0);
	write(kmem, &kprof_sysc, sizeof(kprof_sysc));
	lseek(kmem, nlst[N_KPROF_PID].n_value, 0);
	write(kmem, kprof_pid, sizeof(kprof_pid));
	lseek(kmem, nlst[N_KPROF_PID_CNT].n_value, 0);
	write(kmem, kprof_pid_cnt, sizeof(kprof_pid_cnt));
	lseek(kmem, nlst[N_KPROF_PID_NAM].n_value, 0);
	write(kmem, kprof_pid_nam, sizeof(kprof_pid_nam));
}

prof_valcmp(p1, p2)
	register struct prof *p1, *p2;
{
	if ( p1->cnt < p2->cnt )
		return 1;
	if ( p1->cnt > p2->cnt )
		return -1;
	return 0;
}

npe_valcmp(p1, p2)
	register struct npe *p1, *p2;
{
	if ( p1->value < p2->value )
		return -1;
	if ( p1->value > p2->value )
		return 1;
	return 0;
}

dumpstate()
{
	register int i, j, hits;

	printf("\n\nKprof snapshot #%d: ", nsnap);
	hits = kprof_su[1]+kprof_su[0];
	if (hits == 0) {
		printf("No samples\n");
		return;
	} else
		printf(" %d samples @ _____ Hz, ", hits);
	if (kprof_stillon == 0)
		printf(", Auto-Shutoff");
	printf("\n\n  %d context switches, %d bus errors, %d system calls\n", 
		kprof_swtch, kprof_buse, kprof_sysc);
	printf("\n		   System		   User\n");
	printf("  Hits		%d	%d%%		%d	%d%%\n\n", 
	    	kprof_su[1], (kprof_su[1]*100)/hits , 
		kprof_su[0], (kprof_su[0]*100)/hits);
	printf("  Per level:\n");
	for (i = 0 ; i < 8 ; i++) {
		printf("	%d:",i);
		printf("	%d	%d%%	", kprof_spsl[i], 
			kprof_su[1] ? (kprof_spsl[i]*100)/kprof_su[1] : 0);
		printf("	%d	%d%%", kprof_upsl[i], 
			kprof_su[0] ? (kprof_upsl[i]*100)/kprof_su[0] : 0);
		printf("\n");
	}
	printf("\n  Per PID:(20-32)\n");
	printf("	(i)	PID	Name		Hits	%% total\n");
	for (i = 0; i < (20+32) ; i++)
		if (kprof_pid_cnt[i]) {
			printf("	%d	%d	", 
				i, kprof_pid[i]);
			for (j = 0; j < 8; j++) {
			    if (kprof_pid_nam[i][j] == 0)
				break;
			    printf("%c",kprof_pid_nam[i][j]);
			}
			if ( j < 8 )
				printf("	");
			printf("	%d	%d\n",
			    kprof_pid_cnt[i], (kprof_pid_cnt[i]*100)/hits);
		}

	if (nsamples) {
#ifdef	VERBOSE
	    	printf("\n  Histogram: (threshold of %d%%, granularity of 16)\n", min_cnt);
		printf("	Address	Count	%% Sys	Name\n");
	    	for (i = 0 ; i < nsamples ; i++)
		    if ((prof[i].cnt * 100)/kprof_su[1] >= min_cnt)
	    	    	printf("	%x	%d	%d	%s\n", 
			    prof[i].addr, prof[i].cnt, 
			    (prof[i].cnt * 100)/kprof_su[1], prof[i].name);
#endif	VERBOSE
	    	printf("\n  Per Routine: (threshold of %d%%, granularity of 16)\n", min_cnt);
		printf("		Count	%% Sys	Name\n");
		for (i = 0 ; i < nsamples ; i++) 
		    for (j = i+1 ; j < nsamples ; j++)
			if (prof[j].name == prof[i].name) {
			    prof[i].cnt += prof[j].cnt;
			    prof[j].cnt = 0;
			}
    		qsort(prof, nsamples, sizeof(struct prof), prof_valcmp);
	    	for (i = 0 ; i < nsamples ; i++)
		    if ((prof[i].cnt * 100)/kprof_su[1] >= min_cnt)
	    		printf("		%d	%d	%s\n",
			    prof[i].cnt, (prof[i].cnt*100)/kprof_su[1], 
			    prof[i].name);
	} else
		printf("  No system samples ???\n"); 
	printf("\n");
}

/*
 * Set up string and symbol tables from kernelf. On return symbol table is 
 * sorted by value.
 */
getnfile()
{
	FILE		*nfile;
	register int	i;
	int		askfor;
	struct nlist	nbuf;
	struct exec	xbuf;

	nfile = fopen( kernelf,"r");
	if (nfile == NULL) {
		fprintf(stderr, "kprof: can't open %s\n", kernelf);
		exit(5);
	}
	fread(&xbuf, 1, sizeof(xbuf), nfile);
	if (N_BADMAG(xbuf)) {
		fprintf(stderr, "kprof: bad format %s\n", kernelf);
		exit(6);
	}
	fseek(nfile, (int)(N_SYMOFF(xbuf) + xbuf.a_syms), 0);
	if (fread(&ssiz, sizeof (ssiz), 1, nfile) == 0) {
		fprintf(stderr, "kprof: %s no symbol table (old format?)\n", 
			kernelf);
		exit(7);
	}
	strtab = (char *)calloc(ssiz, 1);
	if (strtab == NULL) {
		fprintf(stderr, "kprof: no room for bytes of string table\n");
		exit(8);
	}
	if (fread(strtab+sizeof(ssiz), ssiz-sizeof(ssiz), 1, nfile) != 1) {
		fprintf(stderr, "kprof: error reading string table\n");
		exit(9);
	}

	/* pass1 - count symbols */
	fseek(nfile, (int)N_SYMOFF(xbuf), 0);
	nname = 0;
	for (i = xbuf.a_syms; i > 0; i -= sizeof(struct nlist)) {
		fread(&nbuf, sizeof(nbuf), 1, nfile);
		if ( ! funcsymbol( &nbuf ) )
		    continue;
		nname++;
	}
	if (nname == 0) {
		fprintf(stderr, "kprof: %s no symbols\n", kernelf);
		exit(10);
	}
	askfor = nname + 1;
	nl = (struct npe *) calloc( askfor, sizeof(struct npe) );
	if (nl == 0) {
		fprintf(stderr, "kprof: No room for bytes of symbol table\n");
		exit(11);
	}

	/* pass2 - read symbols */
	fseek(nfile, (int)N_SYMOFF(xbuf), 0);
	npe = nl;
	nname = 0;
	for (i = xbuf.a_syms; i > 0; i -= sizeof(struct nlist)) {
		fread(&nbuf, sizeof(nbuf), 1, nfile);
		if ( ! funcsymbol( &nbuf ) )
		    continue;
		npe->value = nbuf.n_value & 0x0FFFFFFF;
		npe->name = strtab+nbuf.n_un.n_strx;
		npe++;
		nname++;
	}
	npe->value = -1;
	qsort(nl, nname, sizeof(struct npe), npe_valcmp);
	fclose(nfile);
}

asgnsamples()
{
	register int		i, j;
	register unsigned int	pcl, pch;
	register int		s0, s1;

	for (i = 0, nsamples = 0 ; i < KPROF ; i++) {
		if (kprof[i]) {
			prof[nsamples].cnt = kprof[i];
			prof[nsamples].addr = i << KPROF_SHIFT;
			prof[nsamples].name = 0;
			nsamples++;
		}
	}
	if (nsamples == 0)
		return;

    	qsort(prof, nsamples, sizeof(struct prof), prof_valcmp);

	/* read samples and assign to namelist symbols */
	for (i = 0; i < nsamples; i++) {
		pcl = prof[i].addr;
		pch = (pcl + KPROF_MASK);
		for (j = 0; j < nname; j++) {
		    s0 = nl[j].value;
		    s1 = nl[j+1].value;
		    if (pch < s0)
			    break;
		    if (pcl >= s1)
			    continue;
		}
		prof[i].name = nl[j-1].name;
	}
}

funcsymbol( nlistp )
	register struct nlist	*nlistp;
{
	register char	*name;

	if ( !((nlistp->n_type == (N_TEXT|N_EXT))) )
		return 0;
	for ( name = strtab + nlistp->n_un.n_strx ; *name ; name += 1 )
		if ( *name == '.' || *name == '$' )
		    return 0;
	return 1;
}
