#define ALLOW_WRITES	/* comment this line to prevent writing files */
#define SERVER
#include <stdio.h>
#include <errno.h>
#include <sgtty.h>
#include <signal.h>
#include <sys/param.h>
#include "/V/is68kif/enet.h"
#include "boot.h"

extern int errno;
struct eniocb EnetControlBlk;
struct BootPacket	bp;
NetAddress hostaddr;
char hostname[32];

#define BOOT_PACKET	0x807

struct enfilter PacketFilter =   /* Receive only boot packets */
{	127, /* enf_Priority  */
    	3,   /* enf_FilterLen */
    	ENF_PUSHWORD+6,ENF_PUSHLIT|ENF_EQ,BOOT_PACKET };

/*
 * 	This bootd runs on a host with an "enet" filterd connection to 
 * ethernet.  It is capable of providing access to files on the host from 
 * other machines on the network, called remote machines.  It is intended 
 * to allow other machines on the network to boot from files on hosts that are 
 * running the bootd.  
 * 	The protocol used to request services of the bootd is specific 
 * to the ethernet.  All operations are initiated by the remote machine by 
 * transmiting packets. All packets are the same length.  The host machine 
 * will transmit packets back (possibly with added information) to the remote
 * machine as an acknoldgement of its request.  The remote machine initiates 
 * a connection with a bootd by broadcasting a "connect" packet, with the 
 * hostname of the server in the data portion of the packet.  After this 
 * connection is acknoldged all transmits from the remote machine are to the
 * host that acknoldged the connect request.  A state diagram of the protocol 
 * is shown below:
 *
 *   +-------+             +----------+           +-------+--read|write-|
 *   | CLOSED|             | CONNECTED|           | OPENED|             |
 *   +-------+---connect-->+----------+---open--->+-------+<------------|
 *       ^                                          |
 *       |--------------close-----------------------|
 *
 * 	In order for the bootd to support requests from more than one 
 * machine at once a "connection block" list is used.  When a packet comes
 * in we check through all active connections blocks to see if there is one with
 * a network address field the same as the packets source address.  If so, we 
 * use that connection state to service the packet.  If no connection was found
 * we get a new connection block from the idle pool.  When the connection is 
 * closed the connection block is returned to the idle pool.
 */
#define MAXCONECTIONS 10

struct connection {
	struct connection	*c_next;
	int			c_state;
	NetAddress		c_addr;
}	cnct[MAXCONECTIONS];

struct connection *activep;
struct connection *idlep;
int	showactive(), inactive(), traceonoff();

trace = 0;

main()
{
	int net;
	FILE *fid;
	NetAddress tmp;
	struct connection *ap, *tp;
	int n, i;

	/* print out active connections on hangup signal */
	signal(SIGHUP, showactive);

	/* make all connections inactive */
	signal(SIGTERM, inactive);

	/* toggle trace option */
	signal(SIGINT, traceonoff);

	/* open up the enet packet filter file for the boot server */
	net = NetOpen();

	/* initialize the connection lists */
	for (i = 0 ; i < (MAXCONECTIONS - 1) ; i++)
		cnct[i].c_next = &cnct[i+1];
	idlep = &cnct[0];
	activep = 0;
		
	/* service requests */
	for (;;) {
		/* pick up a packet */
		NetRead(net, &bp, sizeof(struct BootPacket));

		/* try to find an active connection at that source address */
		for (ap = activep; ap; ap = ap->c_next)
			if (ap->c_addr.addrhigh == bp.bp_eh.srchost.addrhigh &&
			    ap->c_addr.addrmid  == bp.bp_eh.srchost.addrmid  &&
			    ap->c_addr.addrlow  == bp.bp_eh.srchost.addrlow    )
				break;
		/* if no active connection then create a new connection */
		if (ap == 0) {
			if (idlep == 0) {
				printf("bootd: out of connections\n");
				continue;
			}
			ap = idlep;
			idlep = ap->c_next;
			ap->c_next = activep;
			activep = ap;
			ap->c_state = BOOT_CLOSE;
			ap->c_addr = bp.bp_eh.srchost;
			if (trace)
				printf("ACTIVATE connection %x\n",ap);
		}
			
		if (trace)
			printf("%x: STATE %s -> CMD %s\n", ap, 
				bootstate[ap->c_state], bootstate[bp.bp_cmd]);
		/* switch on the state of the connection and service request */
		switch (ap->c_state) {
		    case BOOT_CLOSE:
			/* if not a connect packet then throw away */
			if (bp.bp_cmd != BOOT_CONNECT)
				continue;
		reconnect:
			/* if not for this host then throw away */
			gethostname(hostname,sizeof hostname);
			if (trace)
				printf("	HOST %s==%s\n", 
					bp.bp_data, hostname);
			if ((strcmp(bp.bp_data, hostname) != 0) && 
				(strcmp(bp.b_data, "PUBLIC") != 0))
					continue;
			bp.bp_eh.dsthost = hostaddr;
			break;

		    case BOOT_CONNECT:
			/* if not an open packet then throw away */
			if (bp.bp_cmd != BOOT_OPEN)
				continue;
			if (trace)
				printf("	OPEN %s\n", bp.bp_data);
			if ((fid = fopen(bp.bp_data,"rw")) < 0) {
				if (trace)
					printf(":file not found ");
			}
			break;

		    case BOOT_OPEN:
		    case BOOT_READ:
#ifdef	ALLOW_WRITES
		    case BOOT_WRITE:
#endif	ALLOW_WRITES
			switch (bp.bp_cmd) {
		    	    case BOOT_CONNECT:
				/* connect packet out of sequence, reconnect */
				goto reconnect;

			    case BOOT_READ:
			    case BOOT_WRITE:
				if ((n = fseek(fid, bp.bp_offset, 0)) != -1)
				    if (bp.bp_cmd == BOOT_READ)
					n = fread(bp.bp_data,1,bp.bp_len,fid);
				    else
					n = fwrite(bp.bp_data,1,bp.bp_len,fid);
				if (n <= 0)
					bp.bp_len = 0;
				else
					bp.bp_len = n;
				break;

			    case BOOT_CLOSE:
				/* delete the connection from active list */
				if (ap == activep)
					activep = ap->c_next;
				else {
					for (tp = activep; tp; tp = tp->c_next)
						if (tp->c_next == ap)
							break;
					if (tp == 0) {
					    	printf("bootd:lost\n");
						exit();
					}
					tp->c_next = ap->c_next;
				}
				ap->c_next = idlep;
				idlep = ap;
				if (trace)
					printf("DELETE connection %x\n",ap);
				fclose(fid);
				break;
			}
			break;

		    default:
			printf("bad packet\n");
		}

		/* send back reply packet */
		tmp = bp.bp_eh.dsthost;
		bp.bp_eh.dsthost = bp.bp_eh.srchost;
		bp.bp_eh.srchost = tmp;
		NetWrite(net, &bp, sizeof(struct BootPacket));
		ap->c_state = bp.bp_cmd;
	}
}

/*
 * Open an ethernet connection returning fileid on connection
 */
NetOpen()
{
	register int i, net;
	struct eniocb *buffer = &EnetControlBlk;
	u_int maxwaiting;
	struct endevp endevp;
	register NetAddress *p;

	if( (net = open("/dev/enet0", 2)) >= 0)
		ioctl( net, FIOCLEX, 0 ); /* Close the file on a UNIX exec */
	else {
		if (errno == EBUSY)
			printf("bootd: already active\n");
		else
			perror("bootd:");
		exit();
	}

    	ioctl( net, EIOCDEVP, &endevp );
    	bcopy((caddr_t)endevp.end_addr, (caddr_t)&hostaddr, sizeof(hostaddr));

    	/* Ask for maximum incoming packet buffering */
    	maxwaiting = buffer->en_maxwaiting;
    	ioctl( net, EIOCSETW, &maxwaiting );

    	/* Flush packets in the input queue */
    	ioctl( net, EIOCFLUSH, 0 );

    	/* Set the kernel packet filter */
    	ioctl( net, EIOCSETF, &PacketFilter ); /* default is to ignore all */

    	return( net );
}

/* 
 * Receive a packet from the ethernet specified by net and place it in packet.
 */
NetRead( net, packet, packetlength)
int	net;			/* File id of ethernet */
struct	BootPacket *packet;	/* Packet to read */
int	packetlength;		/* Actual packet length */
{
    packetlength = read( net, packet, packetlength );

    if( packetlength < sizeof(struct BootPacket) ) 
        return( -1 );

    return( packetlength );
}

/* 
 * Send a boot packet over the ethernet specified by net
 */
NetWrite( net, packet, packetlength)
int	net;				/* file id to write to */
register struct	BootPacket *packet;	/* Packet to write */
{
	return( write(net, packet, packetlength));
}

showactive()
{
	struct connection *ap;

	printf("bootd: connection status: trace %s\n", 
		trace ? "on" : "off");
	for (ap = activep; ap; ap = ap->c_next)
		printf("%s%x %x %x\n", bootstate[ap->c_state],
			ap->c_addr.addrhigh, ap->c_addr.addrmid, 
			ap->c_addr.addrlow);
}

inactive()
{
	struct connection *ap;

	for (ap = activep; ap->c_next; ap = ap->c_next)
		;
	if (ap) {
		ap->c_next = idlep;
		idlep = activep;
		activep = 0;
	}
}

traceonoff()
{
	if (trace) {
		printf("bootd: trace off\n");
		trace = 0;
	} else {
		printf("bootd: trace on\n");
		trace = 1;
	}
}
