/* floating point routines for 68000/68010 */

#include "DEFS.h"

				/* offsets in internal float structure */
SIGN	= 0				/* sign */
EXPT	= 2				/* exponent (-127/+127) */
MANH	= 4				/* high mantissa */
MANL	= 8				/* low mantissa */

	.text
etoi:
	clrw	a0@			/* clear sign */
	tstl	d0			/* test sign of external */
	jge	1f			/* set sign 0(+), 1(-) */
	movw	#1,a0@

1:	movl	d1,a0@(MANL)		/* save low 32 bits of mantissa */
	movl	d0,d1
	andl	#0x7FFFFF,d1
	orl	#0x800000,d1		/* add hidden high order bit */
	movl	d1,a0@(MANH)		/* save high 1+23 bits of mantissa */
	swap	d0
	asrl	#7,d0
	andw	#0xFF,d0		/* isolate exponent */
	jne	2f
	clrl	a0@			/* zero sign, exponent, */
	clrl	a0@(MANH)		/*   high mantissa, and */
	clrl	a0@(MANL)		/*   low mantissa */
	rts

2:	subw	#128,d0			/* convert from excess 0200 */
	movw	d0,a0@(EXPT)		/* store converted value */
	rts				/* done */

/*  */
/* convert internal format to external float */
/* a0 points to afloat or bfloat */
/* external float returned in d0,d1 */
/*  */
	.text
itoe:
	moveq	#0,d0
	movw	a0@(EXPT),d0		/* get exponent */
	addw	#128,d0			/* convert to excess 0200 */
	jne	1f			/* if exponent is zero */
	moveq	#0,d1			/*   clear d0,d1 */
	rts				/*   and return */

1:	tstw	a0@			/* set sign */
	jeq	2f			/* sign bit 0(+), 1(-) */
	orw	#0x100,d0

2:	swap	d0			/* align sign and exponent */
	asll	#7,d0			/*   in high part of d0 */
	movl	a0@(MANH),d1		/* get high part of mantissa */
	jne	3f			/* check for zero mantissa */
	moveq	#0,d0			/* if zero - clear sign and */
	rts				/*   exponent and return */

3:	andl	#0x7FFFFF,d1		/* delete high order hidden bit */
	orl	d1,d0			/* put high 23 bits of mantissa */
	movl	a0@(MANL),d1		/* put low 32 bits of mantissa */
	rts				/* done */

/*  */
/* normalize internal float by adjusting exponent and */
/* shifting  mantissa appropriately so 1/2 <= mnt < 1 */
/* a0 points to afloat or bfloat */
/*  */
	.text
normal:
	jbsr	offset			/* determine amount to shift */
	addw	d0,a0@(EXPT)		/* adjust exponent */
	jbsr	shift			/* shift mantissa */
	rts

/*  */
/* determine position of most significant bit of */
/* mantissa in relation to normalized decimal point */
/* a0 points to afloat or bfloat */
/* d0 returns offset of msb from decimal point */
/*  */
	.text
offset:
	moveq	#1,d0
	movl	a0@(MANH),d1		/* check for high order bits */
	jne	2f
	movl	a0@(MANL),d1		/* check low order bits */
	jne	1f
	clrw	d0			/* zero shift count */
	rts

1:	subw	#32,d0			/* need to shift at least 23 */
2:	subqw	#1,d0			/* find most significant bit */
	asll	#1,d1
	jcc	2b
	addqw	#8,d0			/* d0 contains exponent correction */
	rts

/*  */
/* shift mantissa according to offset in d0 */
/* a0 points to afloat or bfloat */
/* d0 contains shift count, <0 -> left shift, >0 -> right shift */
/* on return, d1 = 0, d2,d3 have shifted mantissa */
/*  */
	.text
shift:
	moveq	#0,d1
	movl	a0@(MANH),d2		/* d2 = high part of mantissa */
	movl	a0@(MANL),d3		/* d3 = low part of mantissa */
	movw	d0,d1			/* examine exponent correction */
	jmi	2f			/* shift left */
	jne	1f			/* shift right */
	rts				/* no shift - return */

1:	asrl	#1,d2			/* shift entire mantissa */
	roxrl	#1,d3			/*   right by one bit */
	subqw	#1,d1			/* repeat until count */
	jne	1b			/*   is zero */
	andl	#0xFFFFFF,d2		/* zero top byte */
	jra	shifte			/* return */

2:	asll	#1,d3			/* shift entire mantissa */
	roxll	#1,d2			/*   left by one bit */
	addqw	#1,d1			/* repeat until count */
	jne	2b			/*   is zero */
	
shifte:	movl	d2,a0@(MANH)		/* store high part of mantissa */
	movl	d3,a0@(MANL)		/* store low part of mantissa */
	rts				/* done */

/*  */
/* fetch floating arguments off stack */
/* convert to internal format in afloat and bfloat */
/* on return, a0 points to afloat, a1 points to bfloat */
/*  */
getargs:
	lea	a6@(-12),a1		/* a1 points to bfloat */
/*	movl	#bfloat,a1		/* a1 points to bfloat */
	tstw	d0			/* branch to 1f for */
	jne	1f			/*   indirect argument fetch */
	movl	a6@(16),d0		/* b-high */
	movl	a6@(20),d1		/* b-low */
	movl	a1,a0			/* setup a0 for conversion */
	jbsr	etoi			/* convert b-arg to internal form */
	movl	a6@(8),d0		/* a-high */
	movl	a6@(12),d1		/* a-low */
	lea	a6@(-24),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	jbsr	etoi			/* convert a-arg to internal form */
	rts
1:	movl	a6@(12),d0		/* b-high */
	movl	a6@(16),d1		/* b-low */
	movl	a1,a0			/* setup a0 for conversion */
	jbsr	etoi			/* convert b-arg to internal form */
	movl	a6@(8),a0		/* a0 points to a-arg */
	movl	a0@+,d0			/* a-high */
	movl	a0@,d1			/* a-low */
	lea	a6@(-24),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	jbsr	etoi			/* convert a-arg to internal form */
	rts	

/*	.globl	fltused			/* */
/*	.text				/* */
/*fltused:				/* */
/*	rts				/* simply define fltused */

/*  */
/* free exponent returning fractional value */
/*  */
ENTRY(frexp)
	link	a6,#-12
	lea	a6@(-12),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	movl	a6@(8),d0		/* a-high */
	movl	a6@(12),d1		/* a-low */
	movl	a6@(16),a1		/* place to return exponent */
	jbsr	etoi			/* convert to internal form */
	movw	a0@(EXPT),d0		/* get unbiased exponent */
	extl	d0			/*   convert to long and */
	movl	d0,a1@			/*   return value */
	clrw	a0@(EXPT)		/* set exponent for fractional */
	jbsr	itoe			/*   value, convert for return */
	unlk	a6
	rts

/*  */
/* add/load exponent of float */
/*  */
ENTRY(ldexp)
	link	a6,#-12
	lea	a6@(-12),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	movl	a6@(8),d0		/* a-high */
	movl	a6@(12),d1		/* a-low */
	jbsr	etoi			/* convert to internal form */
	movl	a6@(16),d0		/* add argument */
	addw	d0,a0@(EXPT)		/*   to exponent */
	jbsr	itoe			/* convert and return */
	unlk	a6
	rts

/*  */
/* separate integer/fractional parts of float */
/*  */
ENTRY(modf)
	link	a6,#-24
	moveml	#0x3800,sp@-		/* save d2,d3,d4 */
	movl	a6@(8),d0		/* a-high */
	movl	a6@(12),d1		/* a-low */
	lea	a6@(-24),a0		/* a0 -> afloat = fractional */
/*	movl	#afloat,a0		/* a0 -> afloat = fractional */
	jbsr	etoi			/*   part on return */
	lea	a6@(-12),a1		/* a1 -> bfloat = integer part on ret */
/*	movl	#bfloat,a1		/* a1 -> bfloat = integer part on ret */
	movw	a0@,a1@			/* copy signs */
	movw	a0@(EXPT),d4		/* if exponent > 0 */
	jgt	1f			/*   separate integer/fractional */
	movw	#-128,a1@(EXPT)		/* else integer part = 0 */
	movl	a1,a0			/*   convert integer part first */
	jra	modfe			/*   no need to separate */

1:	movw	d4,a1@(EXPT)		/* set integer exponent */
	clrw	a0@(EXPT)		/* set fractional exponent */
	cmpw	#56,d4			/* if shift count is < 56 */
	jlt	2f			/*   shift mantissa */
	movl	a0@(MANL),a1@(MANL)	/* else move mantissa to integer */
	movl	a0@(MANH),a1@(MANH)	/*   part and set fractional */
	movw	#-128,a0@(EXPT)		/*   part = 0 */
	movl	a1,a0			/* convert decimal part first */
	jra	modfe			/*   on exit from modf */

2:	moveq	#-8,d0			/* shift mantissa left */
	jbsr	shift			/*   by 8 for alignment */
	moveq	#0,d0
3:	asll	#1,d3			/* rotate d0<--d1<--d2<--d3 */
	roxll	#1,d2			/*   registers until shift */
	roxll	#1,d1			/*   count = 0 */
	roxll	#1,d0
	subqw	#1,d4
	jne	3b
	movl	d3,a0@(MANL)		/* save fractional components */
	movl	d2,a0@(MANH)		/*   of mantissa */
	movl	d1,a1@(MANL)		/* save integer components */
	movl	d0,a1@(MANH)		/*   of mantissa */
	jbsr	normal			/* align fractional part */
	subqw	#8,a0@(EXPT)		/*   and adjust exponent */
	movl	a1,a0			/* align integer part at */
	jbsr	offset			/*   decimal point without */
	jbsr	shift			/*   altering exponent */

modfe:	movl	a6@(16),a1		/* get pointer argument */
	jbsr	itoe			/* convert integer part */
	movl	d0,a1@+			/*   store in location */
	movl	d1,a1@			/*   given in argument */
	lea	a6@(-24),a0		/* convert fractional */
/*	movl	#afloat,a0		/* convert fractional */
	jbsr	itoe			/*   part and return */
	moveml	sp@+,#0x1C
	unlk	a6
	rts

/*  */
/* convert floating value to fixed 32-bit integer */
/*  */
ASENTRY(fix)
	link	a6,#-12
	moveml	#0x7080,sp@-		/* save d2,d3 */
	movl	a6@(8),d0		/* d0 = high part of float */
	movl	a6@(12),d1		/* d1 = low order part */
	lea	a6@(-12),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	jbsr	etoi			/* convert to internal format */
	tstw	a0@(EXPT)		/* test exponent */
	jgt	1f			/* if exponent is less */
	moveq	#0,d0			/*   than or equal to zero */
	jra	fixe			/*   return zero */

1:	moveq	#-8,d0			/* shift mantissa left */
	jbsr	shift			/*   by 8 for alignment */
	movw	a0@(EXPT),d1		/* (note: after shift d1 = 0) */
	moveq	#0,d0			/* clear d0 */
2:	asll	#1,d3			/* rotate d0<--d2<--d3 */
	roxll	#1,d2			/*   registers until exponent */
	roxll	#1,d0			/*   count is exhausted */
	subqw	#1,d1			/* resultant fixed 32-bit */
	jne	2b			/*   value is in d0 */

	tstw	a0@			/* check sign of float */
	jeq	fixe			/* positive - d0 is ok */
	negl	d0			/* negative - negate d0 */
fixe:	moveml	sp@+,#0x010E		/* pop d2,d3 */
	unlk	a6
	rts

/*  */
/* convert fixed 32-bit integer to floating */
/*  */
ASENTRY(float)
	link	a6,#-12
	moveml	#0x3080,sp@-		/* save d2,d3 */
	lea	a6@(-12),a0		/* a0 points to afloat */
/*	movl	#afloat,a0		/* a0 points to afloat */
	clrl	a0@(MANH)		/* clear junk from mantissa */
	clrw	a0@			/* clear sign */
	movl	a6@(8),d0		/* d0 = 32-bit long */
	jmi	1f			/* negative */
	jpl	2f			/* positive */
	movw	#-128,a0@(EXPT)		/* floating zero */
	jra	floate			/* return */

1:	movw	#1,a0@			/* negative sign */
	negl	d0			/* convert to positive */
2:	movl	d0,a0@(MANL)		/* move d0 to lower mantissa */
	jbsr	offset			/* determine amount to shift */
	jbsr	shift			/* shift mantissa */
	addw	#56,d0			/* calculate exponent */
	movw	d0,a0@(EXPT)		/* set exponent */

floate:	jbsr	itoe			/* convert to external float */
	moveml	sp@+,#0x010C		/* pop d2,d3 */
	unlk	a6
	rts

/*  */
/* add, subtract, compare two floating point numbers */
/* d0,d1 return result of fadd,fsub operations */
/* result of afadd,afaddf,afsub,afsubf stored */
/* fcmp sets condition codes upon return */
/*  */
ASENTRY(fadd)
	link	a6,#-24
	moveml	#0x30C0,sp@-		/* save d2,d3,a0,a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	add			/* perform addition */
	moveml	sp@+,#0x030C		/* pop d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(fsub)
	link	a6,#-24
	moveml	#0x30C0,sp@-		/* save d2,d3,a0,a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	eorw	#1,a1@			/* reverse sign of b-arg */
	jbsr	add			/* perform addition */
	moveml	sp@+,#0x030C		/* pop d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(fcmp)
	link	a6,#-24
	moveml	#0xF0C0,sp@-		/* save d0-d3,a0,a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
/**/
/* Unfortunately, the add that is done by this routine can overflow, */
/* giving incorrect results.  So....  */
/**/
	movw	a0@,d0			/* get a-arg */
	movw	a1@,d1			/* get b-arg */
	eorw	d0,d1
	andw	#1,d1
	jeq	1f			/* if signs same, do add*/
	rorl	#1,d0			/* get d0 sign bit in place */
	orl	#1,d0			/* make sure d0 not zero */
	jbr	2f			/* then set cc from d0 sign */
1:
/**/
/* End of bug fix -- Wendy T. */
/**/
	eorw	#1,a1@			/* reverse sign of b-arg */
	jbsr	add			/* perform compare */
2:
	movl	d0,sp@(40+24)		/* stuff value up the stack for later */
	moveml	sp@+,#0x030F		/* restore registers */
	unlk	a6			/* restore a6 */
	movl	sp@+,sp@(12)		/* stuff return address up the stack */
	addl	#8,sp			/* flush rest of input parameters */
	tstl	sp@+			/* set condition codes based on value */
	rts				/* return */
ASENTRY(faddf)
	link	a6,#-24
	moveml	#0x70C0,sp@-		/* save d1,d2,d3,a0,a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	clrl	a0@(MANL)		/* clear lower part of a-arg */
	jbsr	add			/* perform addition */
	moveml	sp@+,#0x030E		/* pop d1,d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(fsubf)
	link	a6,#-24
	moveml	#0x70C0,sp@-		/* save d1,d2,d3,a0,a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	clrl	a0@(MANL)		/* clear lower part of a-arg */
	eorw	#1,a1@			/* reverse sign of b-arg */
	jbsr	add			/* perform addition */
	moveml	sp@+,#0x030E		/* pop d1,d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(afadd)
	link	a6,#-24
	moveml	#0x30C0,sp@-		/* save d2,d3,a0,a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	add			/* perform addition */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@+			/*   to store result */
	movl	d1,a0@			/*   of operation */
	moveml	sp@+,#0x030C		/* pop d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(afsub)
	link	a6,#-24
	moveml	#0x30C0,sp@-		/* save d2,d3,a0,a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	eorw	#1,a1@			/* reverse sign of b-arg */
	jbsr	add			/* perform addition */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@+			/*   to store result */
	movl	d1,a0@			/*   of operation */
	moveml	sp@+,#0x030C		/* pop d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(afaddf)
	link	a6,#-24
	moveml	#0x70C0,sp@-		/* save d1,d2,d3,a0,a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	clrl	a0@(MANL)		/* clear lower part of a-arg */
	jbsr	add			/* perform addition */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@			/*   to store result */
	moveml	sp@+,#0x030E		/* pop d1,d2,d3,a0,a1 */
	unlk	a6
	rts
ASENTRY(afsubf)
	link	a6,#-24
	moveml	#0x70C0,sp@-		/* save d1,d2,d3,a0,a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	clrl	a0@(MANL)		/* clear lower part of a-arg */
	eorw	#1,a1@			/* reverse sign of b-arg */
	jbsr	add			/* perform addition */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@			/*   to store result */
	moveml	sp@+,#0x030E		/* pop d1,d2,d3,a0,a1 */
	unlk	a6
	rts
add:
	movl	a0@(MANH),d0		/* check for zero first arg */
	jne	3f
	movl	a0@(MANL),d1
	jne	3f
	movl	a1,a0
	jbsr	normal
	jbsr	itoe
	rts
3:	movl	a1@(MANH),d0		/* check for zero second arg */
	jne	4f
	movl	a1@(MANL),d1
	jne	4f
	jbsr	normal			/* normalize result */
	jbsr	itoe			/* convert to external form */
	rts
4:	movw	a0@(EXPT),d0		/* compare */
	movw	a1@(EXPT),d1		/*   exponents */
	subw	d1,d0			/*   of a and b */
	jmi	1f			/* a_expt < b_expt */
	movl	a1,a0			/* switch a0 to point to bfloat */
	jbsr	shift			/* shift mantissa of bfloat */
	lea	a6@(-24),a1		/* switch a1 to point to afloat */
/*	movl	#afloat,a1		/* switch a1 to point to afloat */
	addw	d0,a0@(EXPT)		/* adjust b_expt accordingly */
	jra	2f
1:	negw	d0			/* make the shift count positive */
	jbsr	shift			/* shift mantissa of afloat */
	addw	d0,a0@(EXPT)		/* adjust a_expt accordingly */
2:	tstw	a0@
	jeq	3f			/* a-arg is negative so */
	negl	d3			/*   negate mantissa */
	negxl	d2			/*   for addition */
3:	movl	a1@(MANH),d0
	movl	a1@(MANL),d1
	tstw	a1@
	jeq	4f			/* b-arg is negative so */
	negl	d1			/*   negate mantissa */
	negxl	d0			/*   for addition */
4:	addl	d1,d3			/* perform addition of */
	addxl	d0,d2			/*   mantissas */
	tstl	d2			/* check sign of result */
	jge	5f
	negl	d3			/* result is negative so */
	negxl	d2			/*   negate mantissa */
	orw	#1,a0@			/*   and set sign */
	jra	adde
5:	andw	#0,a0@			/* result positive */

adde:	movl	d2,a0@(MANH)		/* store result */
	movl	d3,a0@(MANL)		/*   of computation */
	jbsr	normal			/* normalize result */
	jbsr	itoe			/* convert to external form */
	rts

/*  */
/* negate a floating number */
/* argument on stack */
/* d0,d1 return result */
/*  */
ASENTRY(fneg)
	link	a6,#0
	movl	a6@(12),d1		/* d1 = low part of float */
	movl	a6@(8),d0		/* d0 = high part of float */
	jmi	1f			
	orl	#0x80000000,d0		/* turn high order bit on */
	jra	2f
1:	andl	#0x7FFFFFFF,d0		/* turn high order bit off */
2:	unlk	a6
	rts	

/*  */
/* multiply two floating numbers */
/* d0,d1 return result for fmul */
/* result for afmul,afmulf stored */
/*  */
ASENTRY(fmul)
	link	a6,#-36
	moveml	#0x3CE0,sp@-		/* save d2-d5,a0-a2 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	mult			/* multiply arguments */
	moveml	sp@+,#0x073C		/* pop */
	unlk	a6
	rts
ASENTRY(fmulf)
	link	a6,#-36
	moveml	#0x7CE0,sp@-		/* save d1-d5,a0-a2 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	mult			/* perform multiply */
	moveml	sp@+,#0x073E		/* pop */
	unlk	a6
	rts
ASENTRY(afmul)
	link	a6,#-36
	moveml	#0x3CE0,sp@-		/* save d2-d5,a0-a2 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	mult			/* perform multiply */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@+			/*   to store result */
	movl	d1,a0@			/*   of operation */
	moveml	sp@+,#0x073C		/* pop */
	unlk	a6
	rts
ASENTRY(afmulf)
	link	a6,#-36
	moveml	#0x7CE0,sp@-		/* save d1-d5,a0-a2 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	mult			/* perform multiply */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@			/*   to store result */
	moveml	sp@+,#0x073E		/* pop */
	unlk	a6
	rts
mult:
	movw	a1@+,d0			/* d0 = sign of b-arg */
	eorw	d0,a0@+			/* a_sign gets resultant sign */
	movw	a1@+,d0 		/* d0 = exponent of b-arg */
	addw	d0,a0@+			/* a_expt gets sum of exponents */

	moveq	#0,d2			/* clear */
	moveq	#0,d3			/*   summation registers */
	moveq	#0,d5			/*   for multiply */
	moveq	#4,d4			/* loop count */
	addql	#8,a1			/* adjust a1 pointer */
	lea	a6@(-36),a2		/* adjust a2 pointer */
/*	movl	#lsum,a2		/* adjust a2 pointer */
1:	movw	a0@+,d0			/* high to low words of afloat */
	movw	a1@-,d1			/* low to high words of bfloat */
	mulu	d0,d1			/* perform multiply */
	addl	d1,d3
	addxl	d5,d2	
	subqw	#1,d4
	jne	1b
	movl	d4,a2@+
	movl	d2,a2@+
	movl	d3,a2@

	moveq	#0,d2
	moveq	#0,d3
	moveq	#3,d4			/* loop count */
	subql	#2,a0			/* adjust a0 pointer */
	addql	#2,a2			/* adjust a2 pointer */
2:	movw	a0@-,d0			/* low to high of afloat */
	movw	a1@+,d1			/* high to low of bfloat */
	mulu	d0,d1			/* perform multiply */
	addl	d1,d3
	addxl	d5,d2
	subqw	#1,d4
	jne	2b
	movl	a2@-,d1
	movl	a2@-,d0
	addl	d1,d3
	addxl	d0,d2
	movl	d2,a2@+
	movl	d3,a2@	
	
	moveq	#0,d2
	moveq	#0,d3
	moveq	#2,d4			/* loop count */
	subql	#2,a1			/* adjust a1 pointer */
	addql	#2,a2			/* adjust a2 pointer */
3:	movw	a0@+,d0			/* high to low of afloat */
	movw	a1@-,d1			/* low to high of bfloat */
	mulu	d0,d1
	addl	d1,d3
	addxl	d5,d2
	subqw	#1,d4
	jne	3b
	movl	a2@-,d1
	movl	a2@-,d0
	addl	d1,d3
	addxl	d0,d2
	movl	d2,a2@+
	movl	d3,a2@

	subql	#2,a0			/* adjust a0 pointer */
	addql	#2,a2			/* adjust a2 pointer */
	movw	a0@-,d0
	movw	a1@+,d1
	mulu	d0,d1
	addl	d1,a2@-

multe:	addql	#1,a2			/* adjust a2 pointer */
	movb	a2@+,a0@+		/* move result */
	movb	a2@+,a0@+		/*   to afloat */
	movb	a2@+,a0@+
	movb	a2@+,a0@+
	movb	a2@+,a0@+
	movb	a2@+,a0@+
	movb	a2@+,a0@+
	movb	a2@+,a0@+
	subql	#8,a0			/* align a0 to point */
	subql	#4,a0			/*   to afloat */
	jbsr	normal			/* normalize result */
	jbsr	itoe			/* convert to external form */
	rts

/*  */
/* divide two floating numbers */
/*  */
ASENTRY(fdiv)
	link	a6,#-24
	moveml	#0x3EC0,sp@-		/* save d2-d6,a0-a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	div			/* divide arguments */
	moveml	sp@+,#0x037C		/* pop */
	unlk	a6
	rts
ASENTRY(fdivf)
	link	a6,#-24
	moveml	#0x7EC0,sp@-		/* save d1-d6,a0-a1 */
	clrw	d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	div			/* divide arguments */
	moveml	sp@+,#0x037E		/* pop */
	unlk	a6
	rts
ASENTRY(afdiv)
	link	a6,#-24
	moveml	#0x3EC0,sp@-		/* save d2-d6,a0-a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	div			/* divide arguments */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@+			/*   to store result */
	movl	d1,a0@			/*   of operation */
	moveml	sp@+,#0x037C		/* pop */
	unlk	a6
	rts
ASENTRY(afdivf)
	link	a6,#-24
	moveml	#0x7EC0,sp@-		/* save d1-d6,a0-a1 */
	moveq	#1,d0			/* flag to getargs */
	jbsr	getargs			/* get arguments */
	jbsr	div			/* divide arguments */
	movl	a6@(8),a0		/* a0 points to where */
	movl	d0,a0@			/*   to store result */
	moveml	sp@+,#0x037E		/* pop */
	unlk	a6
	rts
div:
	movw	a1@+,d0			/* d0 = sign of b-arg */
	eorw	d0,a0@+			/* a-sign gets resultant sign */
	movw	a1@+,d0			/* d0 = exponent of b-arg */
	subw	d0,a0@+			/* a-expt gets diff of exponents */
	movl	a1@+,d0			/* d0 = divisor high */
	jne	ok			/* if divisor = 0 */
	divu	d0,d1			/*   cause trap and core dump */

ok:	movl	a1@,d1			/* d1 = divisor low */
	movl	a0@+,d2 		/* d2 = dividend high */
	movl	a0@,d3			/* d3 = dividend low */
	moveq	#0,d4			/* clear quotient */
	moveq	#0,d5			/*   register set */
	moveq	#58,d6			/* setup shift count */
	jra	2f

1:	subqw	#1,d6			/* exit computation when */
	jeq	dive			/*   loop count = 0 */
	movw	#0x10,cc		/* set x-bit in ccr */
	roxll	#1,d5			/* shift 1 into quotient */
	roxll	#1,d4			/*   accumulator registers */
	roxll	#1,d3			/* continue shift into */
	roxll	#1,d2			/*   into remainder registers */
2:	subl	d1,d3			/* subtract divisor from */
	subxl	d0,d2			/*   remainder */
	jge	1b

3:	subqw	#1,d6			/* exit computation when */
	jeq	dive			/*   loop count = 0 */
	asll	#1,d5			/* shift 0 into quotient */
	roxll	#1,d4			/*   accumulator registers */
	roxll	#1,d3			/* continue shift into */
	roxll	#1,d2			/*   remainder registers */
	addl	d1,d3			/* add divisor to */
	addxl	d0,d2			/*   remainder */
	jge	1b			/* positive -> shift 1 */
	jra	3b			/* negative -> shift 0 */

dive:	subql	#4,a0			/* align a0 to mantissa */
	movl	d4,a0@+			/* store quotient */
	movl	d5,a0@			/*   into afloat mantissa */
	subql	#8,a0			/* align a0 to afloat */
	jbsr	normal			/* normalize mantissa */
	jbsr	itoe			/* convert to external float */
	rts
