#include <vt.h>
#include <tools.h>


#define MENUFONT	"small"
#define MENUBUFSIZE	1024
#define MENUX		200
#define MENUY		200

#define PADDING		3
#define THICKNESS	2
#define SHADOW		6

#define	SHSPACING	-THICKNESS
#define THSPACING	10
#define VSPACING	20

#define VMARGIN		10
#define HMARGIN		10

#define	FPOPUP		"Menu\0Select and go\0Select\0Go\0"  /* field popup */
#define	BPOPUP		"Menu\0Go\0"		/* background popup */


static VT_WINDOW menuwindow;
static VT_FONTID menufont;

static bool cursor;
static int cursorq, cursorc;
static int  bcolor;

static bool editing;

static struct menu	*refreshmenu;
static int		**refreshresults;

static MenuRefresh(id, x, y, w, h)
int	id;
short	x;
short	y;
short	w;
short	h;
{
	struct wstate	ws;		/* window state */

	GetWindowState(menuwindow, &ws);
	SetTemporaryClipping(menuwindow, x, y, w, h);
	DrawMenu(refreshmenu, refreshresults);
	SetWindowState(menuwindow, &ws);
}

static int StringWidth(font, string)
VT_FONTID font;
register char *string;
{
    register int width;

    for (width = 0; *string != '\0'; string++)
	width += CharacterWidth(font, *string);

    return(width);
}

static int ChoiceWidth(choice)
struct choice *choice;
{
    switch (choice->type) {
    case  LABEL:
	    return(StringWidth(menufont, choice->label)+2*(PADDING+THICKNESS));
	    break;
    case  SHADE:
	    return(SHADEWIDTH);
	    break;
    case  GLYPH:
	    /* not implemented */
	    return(-1);
	    break;
    default:
	    return(-1);
	    break;
    }
}

static int ChoiceHeight(choice)
struct choice *choice;
{
    switch (choice->type) {
    case  LABEL:
	    return(CharacterHeight(menufont)+2*(PADDING+THICKNESS));
	    break;
    case  SHADE:
	    return(SHADEHEIGHT);
	    break;
    case  GLYPH:
	    /* not implemented */
	    return(-1);
	    break;
    default:
	    return(-1);
	    break;
    }
}

static int QuestionWidth(question)
register struct question *question;
{
    register struct choice *choice;
    register int i, sumw;

    if (question->type == STRING)
	return(question->size*CharacterWidth(menufont, 'M')
	       + 2*(PADDING+THICKNESS));
	/*
	 * M is chosen as a middle ground; there are pathological cases,
	 * such as all W's that screw this up; it is probably a bug we
	 * can live with and will not be seen much; if not, use W instead.
	 */

    sumw = 0;

    choice = question->choices;
    for (i = 0; i < question->size; i++)
	sumw += ChoiceWidth(choice++);

    sumw += (question->size-1)*
	    ((question->type == SELECT) ? SHSPACING : THSPACING);

    return(sumw);
}

static int QuestionHeight(question)
struct question *question;
{
    return(CharacterHeight(menufont)+2*(THICKNESS+PADDING));
}

static int maxwidth;
static int MenuTabPosition(menu)
register struct menu *menu;
{
    register struct question *question;
    register int i, width;

    if(maxwidth > 0){
	return(maxwidth+20);
    }
    maxwidth = 0;
    question = menu->questions;
    for (i = 0; i < menu->size; i++)
    {
	width = StringWidth(menufont, (question++)->label);
	if (width > maxwidth)
	    maxwidth = width;
    }

    return(maxwidth+20);
}

static int MenuWidth(menu)
register struct menu *menu;
{
    register struct question *question;
    register int i, qw, sumw;

    sumw = 0;
    question = menu->questions;
    for (i = 0; i < menu->size; i++)
    {
	if ((qw = QuestionWidth(question++)) > sumw)
	    sumw = qw;
    }

    sumw += MenuTabPosition(menu);
    sumw += 2*HMARGIN;

    return(sumw);
}

static int MenuHeight(menu)
register struct menu *menu;
{
    register struct question *question;
    register int i, sumqh;

    sumqh = 0;

    question = menu->questions;
    for (i = 0; i < menu->size; i++)
	sumqh += QuestionHeight(question++);

    sumqh += (menu->size-1)*VSPACING;
    sumqh += 2*VMARGIN;

    return(sumqh);
}

static bool CellCoordinates(menu, results, x, y, question, choice)
register struct menu *menu;
int **results;
register int x, y, *question, *choice;
{
    register int q, c;
    register int cellx, celly, cellw, cellh;

    celly = VMARGIN;
    for (q = 0; q < menu->size; q++)
    {
	cellh = QuestionHeight(&menu->questions[q]);
	if (y >= celly && y < celly+cellh)
	    break;
	celly += cellh+VSPACING;
    }

    if (q == menu->size)
	return(FALSE);

    if (menu->questions[q].type == STRING)
    {
	cellx = HMARGIN+MenuTabPosition(menu);
	cellw = QuestionWidth(&menu->questions[q]);
	if (x < cellx || x >= cellx+cellw)
	    return(FALSE);

	cellx += PADDING;
	for (c = 0; ((char*)results[q])[c] != '\0'; c++)
	{
	    cellx += CharacterWidth(menufont, ((char*)results[q])[c]);
	    if (cellx > x)
		break;
	}
    }
    else
    {
	cellx = HMARGIN+MenuTabPosition(menu);
	for (c = 0; c < menu->questions[q].size; c++)
	{
	    cellw = ChoiceWidth(&menu->questions[q].choices[c]);
	    if (x >= cellx && x < cellx+cellw)
		break;
	    cellx += cellw+((menu->questions[q].type == SELECT) ? SHSPACING : THSPACING);
	}

	if (c == menu->questions[q].size)
	    return(FALSE);
    }

    *question = q;
    *choice = c;

    return(TRUE);
}

static CellBounds(menu, question, choice, x, y, w, h)
register struct menu *menu;
register int question, choice;
int *x, *y, *w, *h;
{
    register int q, c;

    *y = VMARGIN;
    for (q = 0; q < question; q++)
	*y += QuestionHeight(&menu->questions[q]) + VSPACING;
    *h = QuestionHeight(&menu->questions[question]);

    if (menu->questions[question].type == STRING)
    {
	*x = HMARGIN+MenuTabPosition(menu);
	*w = QuestionWidth(&menu->questions[q]);
    }
    else
    {
	*x = HMARGIN+MenuTabPosition(menu);
	for (c = 0; c < choice; c++)
	    *x += ChoiceWidth(&menu->questions[question].choices[c]) +
		  ((menu->questions[question].type == SELECT) ? SHSPACING : THSPACING);
	*w = ChoiceWidth(&menu->questions[question].choices[choice]);
    }
}

static bool Selected(menu, results, question, choice)
struct menu *menu;
int **results;
int question, choice;
{
    if (menu->questions[question].type == SELECT)
	return(*results[question] == choice);
    else
	return(((bool*)results[question])[choice]);
}

static InvertCell(menu, question, choice)
struct menu *menu;
int question, choice;
{
    int xc, yc, wc, hc;

    CellBounds(menu, question, choice, &xc, &yc, &wc, &hc);
    InvertRegion(menuwindow, xc+THICKNESS, yc+THICKNESS,
		 wc-2*THICKNESS, hc-2*THICKNESS);
}

static InitializeCellCursor()
{
    cursor = FALSE;
}

static DrawCellCursor(menu, question, choice, color)
struct menu *menu;
int question, choice;
VT_COLOR color;
{
    int xc, yc, wc, hc;

    CellBounds(menu, question, choice, &xc, &yc, &wc, &hc);
    SetColor(menuwindow, color);
    SetThickness(menuwindow, PADDING);
    SetPosition(menuwindow, xc+THICKNESS+PADDING, yc+THICKNESS+PADDING);
    PaintRectangleBorder(menuwindow, wc-2*(THICKNESS+PADDING), hc-2*(THICKNESS+PADDING));
}

static PaintCellCursor(menu, results, question, choice)
struct menu *menu;
int **results, question, choice;
{
    int xc, yc, wc, hc;

    editing = menu->questions[question].type == STRING;

    if (editing)
    {
	CellBounds(menu, question, choice, &xc, &yc, &wc, &hc);
	StartEdit(menuwindow, menufont,
		  (char*)results[question],
		  xc+THICKNESS+PADDING,
		  yc+THICKNESS+PADDING+CharacterBaseline(menufont),
		  choice,
		  menu->questions[question].size,
		  wc);
    }
    else
	DrawCellCursor(menu, question, choice,
		   Selected(menu, results, question, choice) ? VT_White : VT_Black);
}

static EraseCellCursor(menu, results, question, choice)
struct menu *menu;
int **results, question, choice;
{
    if (editing)
	StopEdit();
    else
	DrawCellCursor(menu, cursorq, cursorc,
		       Selected(menu, results, cursorq, cursorc) ? VT_Black : VT_White);
}

static MoveCellCursor(menu, results, question, choice)
struct menu *menu;
int **results;
int question, choice;
{
    EraseCellCursor(menu, results, cursorq, cursorc);

    cursorq = question;
    cursorc = choice;

    PaintCellCursor(menu, results, cursorq, cursorc);
}

static HideCellCursor(menu, results)
struct menu *menu;
int **results;
{
    if (cursor)
    {
        cursor = FALSE;
	if (editing)
	    StopEdit();
	else
	    DrawCellCursor(menu, cursorq, cursorc,
			   Selected(menu, results, cursorq, cursorc) ? VT_Black : VT_White);
    }
}

static ShowCellCursor(menu, results)
struct menu *menu;
int **results;
{
    if (!cursor)
    {
	cursor = TRUE;
	cursorq = cursorc = 0;
	PaintCellCursor(menu, results, cursorq, cursorc);
    }
}

static CursorLeft(menu, results)
struct menu *menu;
int **results;
{
    if (!cursor)
	ShowCellCursor(menu, results);
    else
	if (cursorc > 0)
	    MoveCellCursor(menu, results, cursorq, cursorc-1);
}

static CursorRight(menu, results)
struct menu *menu;
int **results;
{
    if (!cursor)
	ShowCellCursor(menu, results);
    else
	if (cursorc < menu->questions[cursorq].size-1)
	    MoveCellCursor(menu, results, cursorq, cursorc+1);
}

static CursorUp(menu, results)
struct menu *menu;
int **results;
{
    if (!cursor)
	ShowCellCursor(menu, results);
    else
	if (cursorq > 0)
	    MoveCellCursor(menu, results, cursorq-1, 0);
}

static CursorDown(menu, results)
struct menu *menu;
int **results;
{
    if (!cursor)
	ShowCellCursor(menu, results);
    else
	if (cursorq < menu->size-1)
	    MoveCellCursor(menu, results, cursorq+1, 0);
}

static SelectCursorCell(menu, results)
struct menu *menu;
int results;
{
    ShowCellCursor(menu, results);
    SelectCell(menu, results, cursorq, cursorc);
}

static SelectCell(menu, results, question, choice)
register struct menu *menu;
register int **results;
int question, choice;
{
    int xc, yc, wc, hc;

    switch (menu->questions[question].type)
    {
	case SELECT:
	    if (editing)
	    {
		StopEdit();
		editing = FALSE;
	    }
	    if (*results[question] != choice)
	    {
		InvertCell(menu, question, *results[question]);
		InvertCell(menu, question, choice);
		*results[question] = choice;
	    }
	    break;

	case TOGGLE:
	    if (editing)
	    {
		StopEdit();
		editing = FALSE;
	    }
	    InvertCell(menu, question, choice);
	    ((bool*)results[question])[choice] = !((bool*)results[question])[choice];
	    break;

	case STRING:
	    CellBounds(menu, question, choice, &xc, &yc, &wc, &hc);
	    StartEdit(menuwindow, menufont,
		      (char*)results[question],
		      xc+THICKNESS+PADDING,
		      yc+THICKNESS+PADDING+CharacterBaseline(menufont),
		      choice,
		      menu->questions[question].size,
		      wc);
	    editing = TRUE;
	    break;
    }
}

static DrawField(x, y, w, h, invert, choice)
register int x, y, w, h;
bool invert;
register struct choice	 *choice;
{
    register int base;

    base = CharacterBaseline(menufont);

    /* paint field shadow */
    SetColor(menuwindow, VT_Black);
    SetPosition(menuwindow, x+THICKNESS+SHADOW, y+THICKNESS+SHADOW);
    PaintRectangleInterior(menuwindow, w, h);
    /* paint field border */
    SetColor(menuwindow, VT_Black);
    SetPosition(menuwindow, x+THICKNESS, y+THICKNESS);
    SetThickness(menuwindow, THICKNESS);
    PaintRectangleBorder(menuwindow, w, h);

    switch (choice->type) {
    case  LABEL:
	/* paint field interior */
	SetColor(menuwindow, choice->shade);
	PaintRectangleInterior(menuwindow, w, h);
	/* paint string in field */
	SetColor(menuwindow, VT_Black);
	SetPosition(menuwindow, x+THICKNESS+PADDING, y+base+THICKNESS+PADDING);
	PaintString(menuwindow, -1, choice->label);
	if (invert) {
		InvertRegion(menuwindow, x+THICKNESS, y+THICKNESS, w, h);
	}
	break;
    case  SHADE:
	/* paint field interior */
	SetColor(menuwindow, choice->shade);
	PaintRectangleInterior(menuwindow, w, h);
	if (invert) {
	}
	break;
    case  GLYPH:
	/* not implemented */
	break;
    default:
	break;
    }
}

static DrawSelectQuestion(menu, results, qn, x, y)
struct menu *menu;
int **results;
register int qn;
register int x, y;
{
    register int cn, w, h;
    register struct question *q;
    register struct choice *c;

    q = &menu->questions[qn];
    for (cn = 0, c = q->choices; cn < q->size; cn++, c++)
    {
	w = ChoiceWidth(c)-2*THICKNESS;
	h = ChoiceHeight(c)-2*THICKNESS;
	DrawField(x, y, w, h, Selected(menu, results, qn, cn), c);
	x += ChoiceWidth(c) + SHSPACING;
    }
}

static DrawToggleQuestion(menu, results, qn, x, y)
struct menu *menu;
int **results;
register int qn;
register int x, y;
{
    register int cn, w, h;
    register struct question *q;
    register struct choice *c;

    q = &menu->questions[qn];
    for (cn = 0, c = q->choices; cn < q->size; cn++, c++)
    {
	w = ChoiceWidth(c)-2*THICKNESS;
	h = ChoiceHeight(c)-2*THICKNESS;
	DrawField(x, y, w, h, Selected(menu, results, qn, cn), c);
	x += ChoiceWidth(c) + THSPACING;
    }
}

static DrawStringQuestion(menu, results, qn, x, y)
struct menu *menu;
int **results;
int qn;
int x, y;
{
    int w, h;
    struct question *q;
    struct choice   c;

    q = &menu->questions[qn];
    c.type = LABEL;
    c.label = (char*)results[qn];
    c.shade = VT_White;
    w = QuestionWidth(q)-2*THICKNESS;
    h = QuestionHeight(q)-2*THICKNESS;
    DrawField(x, y, w, h, FALSE, &c);
}

static DrawMenu(menu, results)
register struct menu *menu;
register int **results;
{
    int xo, yo;
    register int y, qn;
    register struct question *q;

    SetColor(menuwindow, bcolor);
    SetPosition(menuwindow, 0, 0);
    PaintRectangleInterior(menuwindow, 10000, 10000);

    SetFont(menuwindow, menufont);

    xo = MenuTabPosition(menu);
    yo = CharacterBaseline(menufont);

    y = VMARGIN;
    for (qn = 0, q = menu->questions; qn < menu->size; qn++, q++)
    {
	SetPosition(menuwindow, HMARGIN, y+yo);
	SetColor(menuwindow, VT_Black);
	SetJustification(menuwindow, VT_FLUSHLEFT);
	PaintString(menuwindow, -1, q->label);
	switch (q->type)
	{
	    case SELECT:
		DrawSelectQuestion(
			menu, results, qn, HMARGIN+MenuTabPosition(menu), y);
		break;
	    case TOGGLE:
		DrawToggleQuestion(
			menu, results, qn, HMARGIN+MenuTabPosition(menu), y);
		break;
	    case STRING:
		DrawStringQuestion(
			menu, results, qn, HMARGIN+MenuTabPosition(menu), y);
		break;
	}
	y += QuestionHeight(q)+VSPACING;
    }
}

int PresentMenu(menu, results)
register struct menu *menu;
register int **results;
{
    int qw, qh;
    int c, s, p, x, y;
    int qn, cn;
    bool done;
    int	 blocked;

    refreshmenu = menu;		/* set global values to these for refresh */
    refreshresults = results;	/* kind of kludgey */

    menufont = GetFontId(MENUFONT);
    maxwidth = 0; /* Initialize this for MenuTabPosition (for efficiency) */
    qw = MenuWidth(menu);
    qh = MenuHeight(menu);

    if ((menuwindow = OpenWindow(MENUX, MENUY, qw, qh, menu->label)) == -1) {
	return(-1);
    }
    SetLineDisc(menuwindow, TWSDISC);
    blocked = BlockRefreshAdjust(1);
    if (SetLocalLUT(menuwindow, 2, 12, 12, 12, VT_SHARE) == -1) {
	/* cannot get real gray; just use pattern */
	bcolor = VT_Gray75;
    } else {
	/* use the real gray as local color 2 */
	bcolor = 2;
    }
    SetBColor(menuwindow, bcolor);
    SetMouseMode(menuwindow, VT_MOUSE_DOWN);
    SetCharAttributes(menuwindow, 0);
    SetBuf(menuwindow, MENUBUFSIZE);
    SetRefresh(menuwindow, 0, MenuRefresh);
    HideLocalCursor(menuwindow);

    DrawMenu(menu, results);

    InitializeCellCursor();
    editing = FALSE;

    for (done = FALSE; !done;)
    {
	c = getvtchar(menuwindow);

	switch (c)
	{
	    case VT_HARDKEY:
		(void) getvtchar(menuwindow);	/* ignore augmentation */
		c = getvtchar(menuwindow);
		switch (c)
		{
		case VT_LEFTARROW:
		    if (editing)
		        MoveLeft();
		    else
			CursorLeft(menu, results);
		    break;
		case VT_RIGHTARROW:
		    if (editing)
			MoveRight();
		    else
			CursorRight(menu, results);
		    break;
		case VT_UPARROW:
		    CursorUp(menu, results);
		    break;
		case VT_DOWNARROW:
		    CursorDown(menu, results);
		    break;
		}
		break;
	    case VT_NAK:	/* control U */
		if (editing) {
		    while (DeleteLeft());
		}
		break;
	    case VT_BS:
	    case VT_DEL:
		if (editing)
		    DeleteLeft();
		break;
	    case VT_CR:
		CursorDown(menu, results);
		break;
	    case VT_SPACE:
		if (editing)
		    InsertCharacter(c);
		else
		    SelectCursorCell(menu, results);
		break;
	    case VT_EOT:
		done = TRUE;
		break;

	    case VT_MOUSE:
		/* get mouse button state */
		s = getvtchar(menuwindow);
		/* ignore window number */
		getvtchar(menuwindow);
		/* get pane number */
		p = getvtchar(menuwindow);
		/* get x and y postion */
		x = getvtchar(menuwindow) << 8;  x |= getvtchar(menuwindow);
		y = getvtchar(menuwindow) << 8;  y |= getvtchar(menuwindow);

		/* if not in the display pane, ignore it */
		if (p != VT_DISPLAY)
		    done = TRUE;
		else if ((s&VT_MOUSE_DOWN) &&
			    ((s&VT_MOUSE_LEFT)||(s&VT_MOUSE_RIGHT))) {
			if (CellCoordinates(menu, results, x, y, &qn, &cn))
			{
			    HideCellCursor(menu, results);
			    SelectCell(menu, results, qn, cn);
			}
			if (s&VT_MOUSE_RIGHT)
			    /* a right click terminates the menu */
			    done = TRUE;
		}
		else if ((s&VT_MOUSE_DOWN) && (s&VT_MOUSE_MIDDLE)) {
			SetPosition(menuwindow, x, y);
			if (CellCoordinates(menu, results, x, y, &qn, &cn))
			{
			    HideCellCursor(menu, results);
			    switch (DisplayPopUp(menuwindow, FPOPUP)) {
			    case 1:	/* Select and go */
			        SelectCell(menu, results, qn, cn);
			        done = TRUE;
				break;
			    case 2:	/* Select */
			        SelectCell(menu, results, qn, cn);
				break;
			    case 3:	/* Go */
			        done = TRUE;
				break;
			    }
			} else {
			    switch (DisplayPopUp(menuwindow, BPOPUP)) {
			    case 1:	/* Go */
			        done = TRUE;
				break;
			    }
			}
		}
		break;

	    default:
		if (editing)
		    InsertCharacter(c);
	}
    }

    Flush(menuwindow);
    SetBuf(menuwindow, 0);
    SetRefresh(menuwindow, 0, (int (*)())0);
    close(menuwindow);
    BlockRefreshAdjust(0);		/* allow refresh/adjust */
    BlockRefreshAdjust(blocked);	/* set it back to original state */

    return(0);
}

static struct
{
    VT_WINDOW	window;
    short	*font;
    char	*field;
    int		x, y;
    int		curx;
    int		endx;
    int		maxx;
    int		curi;
    int		endi;
    int		maxi;
    int		topy;
    int		height;
} state;


static unsigned int mask[] =
{
    0x04000000, 0x0e000000, 0x1f000000, 0x3f800000,
    0x7fc00000, 0xffe00000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
};

static unsigned int face[] =
{
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000,
    0x00000000, 0x00000000, 0x00000000, 0x00000000
};

static TrackCursor()
{
    MoveLocalCursor(state.window, state.curx, state.y);
}

static StartEdit(window, font, field, x, y, index, size, width)
VT_WINDOW window;
short *font;
char *field;
int x, y, index, size, width;
{
    register int i;

    state.window = window;
    state.font = font;
    state.field = field;
    state.x = x;
    state.y = y;
    state.maxi = size-1;
    state.maxx = x+width;

    state.curx = x;
    for (i = 0; i < index; i++)
	state.curx += CharacterWidth(font, field[i]);
    state.curi = i;

    state.endx = state.curx;
    for (i = index; field[i] != '\0'; i++)
	state.endx += CharacterWidth(font, field[i]);
    state.endi = i;

    state.topy = y-CharacterBaseline(font);
    state.height = CharacterHeight(font);

    DefineLocalCursor(state.window, mask, face, 5, 0, 0);
    MoveLocalCursor(state.window, state.curx, state.y);
    ShowLocalCursor(state.window);
}

static StopEdit()
{
    HideLocalCursor(state.window);
}

static InsertCharacter(c)
char c;
{
    register int i, width;

    if (state.endi == state.maxi)
        return;

    for (i = state.endi; i >= state.curi; i--)
	state.field[i+1] = state.field[i];
    state.field[state.curi] = c;

    width = CharacterWidth(state.font, c);

    CopyRegion(state.window, state.curx, state.topy,
	       state.curx+width, state.topy,
	       state.endx-state.curx, state.height);

    SetPosition(state.window, state.curx, state.topy);
    SetColor(state.window, VT_White);
    PaintRectangleInterior(state.window, width, state.height);

    SetPosition(state.window, state.curx, state.y);
    SetColor(state.window, VT_Black);
    SetFont(state.window, state.font);
    PaintCharacter(state.window, c);

    state.curi++;  state.curx += width;
    state.endi++;  state.endx += width;

    TrackCursor();
}

static MoveLeft()
{
    if (state.curi == 0)
	return;

    state.curi--;
    state.curx -= CharacterWidth(state.font, state.field[state.curi]);

    TrackCursor();
}

static MoveRight()
{
    if (state.curi == state.endi)
	return;

    state.curx += CharacterWidth(state.font, state.field[state.curi]);
    state.curi++;

    TrackCursor();
}

static int DeleteLeft()
{
    register int i, width;

    if (state.curi == 0)
	return(0);

    width = CharacterWidth(state.font, state.field[state.curi-1]);

    for (i = state.curi; i <= state.endi; i++)
	state.field[i-1] = state.field[i];

    CopyRegion(state.window, state.curx, state.topy,
		state.curx-width, state.topy,
		state.endx-state.curx, state.height);

    SetPosition(state.window, state.endx-width, state.topy);
    SetColor(state.window, VT_White);
    PaintRectangleInterior(state.window, width, state.height);

    state.curi--;  state.curx -= width;
    state.endi--;  state.endx -= width;

    TrackCursor();
    return(1);
}

static DeleteRight()
{
    register int i, width;

    if (state.curi == state.endi)
	return;

    width = CharacterWidth(state.font, state.field[state.curi]);

    for (i = state.curi; i <= state.endi; i++)
	state.field[i] = state.field[i+1];

    CopyRegion(state.window, state.curx+width, state.topy,
		state.curx, state.topy,
		state.endx-state.curx-width, state.height);

    SetPosition(state.window, state.endx-width, state.topy);
    SetColor(state.window, VT_White);
    PaintRectangleInterior(state.window, width, state.height);

    state.endi--;  state.endx -= width;

    TrackCursor();
}
