/*
 * create 
 *
 * make a new {file,directory,symbolic_link} with specified owner, group,
 * permissions, and (for a file) contents.  Examples: 
 *
 * create -rw-rw-r-- mark staff foo 
 * create -rw-rw-r-- mark staff newfile foo/bar 
 * create drwxr-xr-x root staff spool 
 * create lrwxrwxrwx mark staff foo -> bar 
 *
 * The first example creates an empty file foo, the second creates a file
 * newfile and writes into it the contents of foo/bar.
 * The placement of arguments is chosen to match output of "ls -l".
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>

static char     pname[256];

main(ac, av)
	int             ac;
	char           *av[];
{
	int	mode, owner, group, type, gotsource=0;
	static char	name[256], target[256], source[256];

	strcpy(pname, *av);
	if (ac <= 5 && ac >= 7)
		usage();

	++av;
	--ac;
	switch ( *av[0] ) {
		case '-' :	type = S_IFREG;		break;
		case 'd' :	type = S_IFDIR;		break;
		case 'l' :	type = S_IFLNK;		break;
		default :	usage();		break;
	}
	mode = parse_mode(*av);
	++av;
	--ac;
	owner = get_owner(*av);
	++av;
	--ac;
	group = get_group(*av);
	++av;
	--ac;
	strcpy(name, *av);
	++av;
	--ac;
	if (ac > 0) {
		if (strcmp(*av, "->") == 0) {
			if ( type != S_IFLNK )
				usage();
			++av;
			--ac;
			if (ac > 0)
				strcpy(target, *av);
			else
				usage();
		}
		else {
			strcpy(source, *av);
			gotsource++;
		}
		++av;
		--ac;
	}
	if (ac > 0) {
		printf("extra=%s\n", *av);
		usage();
	}

	switch (type) {
		case S_IFREG :
			make_reg(name, mode, owner, group, gotsource, source);
			break;
		case S_IFDIR :
			make_dir(name, mode, owner, group);
			break;
		case S_IFLNK :
			make_lnk(name, mode, owner, group, target);
			break;
		default :
			printf("type=%o\n", type);
			usage();
			break;
	}
	exit(0);
}

#include <sys/file.h>
#define	BSZ	8192

make_reg(name, mode, owner, group, gotsource, source)
char	*name;
int	mode, owner, group, gotsource;
char	*source;
{
	int	fd, src, n, n2;
	struct stat	stat;
	static char	buf[BSZ];

	mode &= 07777;
	umask(0);
	if ((fd = creat(name, 0600)) < 0)
		fatal( "can't create %s\n", name);
	if (gotsource) {
		src = open(source, O_RDONLY, 0);
		do {
			n = read(src, buf, BSZ);
			if (n < 0)
				fatal("read error in %s\n", source);
			n2 = write(fd, buf, n);
			if ( n != n2 )
				fatal( "write error in %s\n", name );
		} while (n > 0);
	}
	if (fchown(fd, owner, group))
		fatal("can't chown %s\n", name);
	if (fchmod(fd, mode))
		fatal("can't chmod %s\n", name);
	close(fd);
}

make_dir(name, mode, owner, group)
char	*name;
int	mode, owner, group;
{
	mode &= 07777;
	umask(0);
	mkdir(name, 0700);
	if (chown(name, owner, group))
		fatal("can't chmod %s\n", name);
	if (chmod(name, mode))
		fatal("can't chmod %s\n", name);
}

make_lnk(name, mode, owner, group, target)
char	*name;
int	mode, owner, group;
char	*target;
{
	mode &= 07777;
	symlink(target, name);
	if (chown(name, owner, group))
		fatal("can't chmod %s\n", name);
}

int m0[] = {3, '-', 0, 'd', S_IFDIR, 'l', S_IFLNK};

int m1[] = {2, 'r', S_IREAD, '-', 0};
int m2[] = {2, 'w', S_IWRITE, '-', 0};
int m3[] = {3, 's', S_ISUID|S_IEXEC, 'x', S_IEXEC, '-', 0};

int m4[] = {2, 'r', (S_IREAD>>3), '-', 0};
int m5[] = {2, 'w', (S_IWRITE>>3), '-', 0};
int m6[] = {3, 's', (S_ISGID|(S_IEXEC>>3)), 'x', (S_IEXEC>>3), '-', 0};

int m7[] = {2, 'r', (S_IREAD>>6), '-', 0};
int m8[] = {2, 'w', (S_IWRITE>>6), '-', 0};
int m9[] = {3, 't', (S_ISVTX), 'x', (S_IEXEC>>6), '-', 0};

int *m[] = {m0, m1, m2, m3, m4, m5, m6, m7, m8, m9};

parse_mode(s)
	char           *s;
{
	int             mode, *mp, n, i;

	mode = 0;
	for (i = 0; i <= 9; ++i) {
		mp = m[i];
		n = *mp++;
		while (n--) {
/*			printf("testing %c against %c\n", s[i], *mp);	/**/
			if (s[i] == *mp) {
				mode |= *(mp+1);
/*				printf("    match: mode=%06o\n", mode);	/**/
				break;
			}
			mp += 2;
		}
		if (n < 0)
			fatal("illegal mode string [%s]\n", s);
	}
	return(mode);
}

fatal(a, b, c, d, e, f, g, h)
{
	fprintf("%s: ", pname);
	fprintf(stderr, a, b, c, d, e, f, g, h);
	fflush(stderr);
	exit(1);
}

usage()
{
	fatal("usage: <modes> <owner> <group> ...\n");
}

#include <pwd.h>

get_owner(s)
char	*s;
{
	struct passwd *p;

	p = getpwnam(s);
	if ( ! p )
		fatal("no such user [%s]\n", s);
	return( p->pw_uid);
}

#include <grp.h>

get_group(s)
char	*s;
{
	struct group *g;

	g = getgrnam(s);
	if ( ! g )
		fatal("no such group [%s]\n", s);
	return(g->gr_gid);
}
