/*
 *	These routines are used for manipulation of windows on the screen.  A
 *	window is a set of parameters, rectangles and tiles.  They are moved
 *	and updated as below.
 */

#define	VTINTERNAL

#include <vt_hdrs.h>		    /* system wide defines */
#include <vt_fonts.h>		    /* font mgmt structures */
#include <rectangle.h>		    /* rectangle structure */
#include <tile.h>		    /* tile structure */
#include <window.h>		    /* window structure */
#include <cursor.h>		    /* cursor structure */


extern struct tile * Visible(),* RestrictTiles(), * CreateTileList();
extern struct tile * InsertTile(), * TileIntersection(), *RVisible();
extern struct tile * freetiles, * Covered(), * TileTranslation();
extern struct tile * TileDifference();
extern int * v_icon[];		    /* system icon table */
extern struct cursor cursor;        /* global cursor */
struct window *desk;		    /* the desktop */

/*
 *	Routine to insert newwindow into a window list immediatily after
 *	oldwindow.  The window list is ordered in terms of depth.
 */
InsertWindow(oldwindow, newwindow)
register struct window *oldwindow, *newwindow;
{
    newwindow->next = oldwindow->next;
    newwindow->prev = oldwindow;
    newwindow->next->prev = newwindow;
    oldwindow->next = newwindow;
}


/*
 *	Routine to remove given window from the window list
 */
DeleteWindow(oldwindow)
register struct window *oldwindow;
{
    oldwindow->next->prev = oldwindow->prev;
    oldwindow->prev->next = oldwindow->next;
}

/*
 *	Routine to recreate new tile lists for all windows whenever window
 *	organization changes 
 */
Shuffle()
{
    register struct window *w;
    register struct tile * tiles;

    /* loop through all windows on desktop */
    for (w = desk->next; w != desk; w = w->next)
    {
	DestroyTileList(w->itiles);
	DestroyTileList(w->tiles);
	/* remove all invisible tiles */
	w->itiles = Visible(desk->next, w, w->ibounds);
	/* make all tiles either ?????  */
	w->tiles = RestrictTiles(w->itiles, w->tcbounds);

    }
    /* fix tile list for desktop as above */
    DestroyTileList(desk->itiles);
    DestroyTileList(desk->tiles);
    desk->itiles = Visible(desk->next, desk, desk->ibounds);
    desk->tiles = RestrictTiles(w->tiles, w->tcbounds);
}

/*
 *	Routine to return a rectangle in the given window for the named pane
 */
PaneBounds(window, pane, bounds)
register struct window *window;
register int pane;
register struct rectangle *bounds;
{
    if (window->type == QUIZ) {
	switch (pane) {		    /* switch to correct pane and set bounds*/
	    case DISPLAY:		    /* regular display surface */
		*bounds = window->ibounds;
		break;
	    case TITLE:		    /* title bar */
		bounds->x = window->ibounds.x;
		bounds->y = window->ibounds.y -RT -TT;
		bounds->w = window->ibounds.w;
		bounds->h = TT;
		break;
	    default:
		bounds->x = window->ibounds.x;
		bounds->y = window->ibounds.y;
		bounds->w = 0;
		bounds->h = 0;
		break;
	}
    } else {
	switch (pane) {		    /* switch to correct pane and set bounds*/
	    case TITLE:		    /* title bar */
		bounds->x = window->ibounds.x - RT+TT+RT;
		bounds->y = window->ibounds.y - RT-BT-TT;
		bounds->w = window->ibounds.w - 2*(RT+TT) + 2*RT;
		bounds->h = TT;
		break;
	    case BLCORNER:		    /* bottom left corner */
		bounds->x = window->ibounds.x - RT;
		bounds->y = window->ibounds.y - RT-BT-TT;
		bounds->w = bounds->h = TT;
		break;
	    case BRCORNER:		    /* bottom right corner (now size) */
		bounds->x = window->ibounds.x + window->ibounds.w +RT -TT;
		bounds->y = window->ibounds.y - RT-BT-TT;
		bounds->w = bounds->h = TT;
		break;
	    case DISPLAY:		    /* regular display surface */
		*bounds = window->ibounds;
		break;
				/* The following glyphs are not used anymore */
	    case TEDGE:		 	    /* top scroll bar */
	    case BEDGE:		    	    /* bottom scroll bar */
	    case LEDGE:		    	    /* left scroll bar */
	    case REDGE:		    	    /* rigth scroll bar */
	    case TRCORNER:		    /* upper right corner */
	    case TLCORNER:		    /* upper left corner */
		bounds->x = window->ibounds.x;
		bounds->y = window->ibounds.y;
		bounds->w = 0;
		bounds->h = 0;
		break;
	}
    }
}

/*
 *	Routine to draw full window frame on screen 
 */
CDrawBorder(tiles, window)
register struct tile *tiles;
register struct window *window;
{
    register int x, y, w, h;
    register struct tile *tile;
    int *fcolor, *bcolor;
    struct rectangle bounds;
    int i;
    extern int	 *v_color[];

    if (window->type == NOSHOW)		/* dont do anything if invisible */
	return;

    /* get size of window */
    x = window->ibounds.x;
    y = window->ibounds.y;
    w = window->ibounds.w;
    h = window->ibounds.h;

    /* loop through all tiles in tile list */
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	/* clip to edges of this tile */
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w,
		    tile->bounds.h);

	/* loop through all tiles and draw contents */
	for (i = TITLE; i < NTILES; i++) {
		/* get bounds of this window */
	        PaneBounds(window, i, &bounds);
		/* clear pane to given color */
		if(window->bcolor[i] < 0){
		    /* psuedo color used for bcolor */
		    DrawRectangleInterior(bounds.x, bounds.y, bounds.w,
				    bounds.h, 0, v_color[-(window->bcolor[i])],
				    window->ibounds.x, window->ibounds.y);
		}else{
		    DrawRectangleInterior(bounds.x, bounds.y, bounds.w,
				    bounds.h, window->bcolor[i], solid,
				    window->ibounds.x, window->ibounds.y);
		}
		if (i == TITLE)		/* draw title */
		    CenterString(x-RT+w/2, window->obounds.y+25, DEFTITLEFONT,
			     window->fcolor[TITLE], window->title);
	        else
		    if ((window->glyph[i] != 0) && 
			(bounds.w>=16) && (bounds.h>=16))
		    PaintArea(bounds.x+bounds.w/2-8, bounds.y+bounds.h/2-8,
			      16, 16, window->glyph[i], window->fcolor[i]);
	    }		    /* end loop through all panes */

	/* draw full window border */
	if (window->type == NORMAL) {
	    DrawRectangleBorder(x, y, w, h, RT, WHITE, solid, 0, 0);
	    DrawRectangleBorder(x-RT, y-BT-RT-TT, w+2*RT, 
				h+2*RT+BT+TT, BT, BLACK, solid, 0, 0);
	    /* draw inside borders for normal window */
	    DrawRectangleInterior(x-RT, y-RT-BT,w+2*RT, BT, BLACK, solid, 0, 0);

	    DrawRectangleInterior(x-RT+TT,y-RT-BT-TT, RT, TT, BLACK,solid,0,0);
	    DrawRectangleInterior(x+w+RT-TT-RT,y-RT-BT-TT,RT,TT,
				  BLACK,solid,0,0);
	} else {		    /* quiz window */
	    DrawRectangleBorder(x, y-RT-TT, w, h+RT+TT, BT, BLACK, solid, 0, 0);
	    /* draw inside borders for quiz window */
	    DrawRectangleInterior(x-RT, y-RT, w + 2*RT, BT, BLACK, solid, 0, 0);
    	}
        UnsetClipping();
    }			    /* end loop through all tiles */
}			    /* end of CDrawBorder */

/*
 *	Routine to refresh the given window with respect to given tile list
 */
CDrawInside(tiles, window)
register struct tile *tiles;
register struct window *window;
{
    register struct tile *tile, *wtiles, *dtile;

    /* loop through all tiles in tile list */
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
#ifdef POPUPBUG
	LiftCursors(window, &tile->bounds);
#endif POPUPBUG
	SetClipping(tile->bounds.x, tile->bounds.y,
		    tile->bounds.w, tile->bounds.h);
	DrawRectangleInterior(tile->bounds.x, tile->bounds.y,
			      tile->bounds.w, tile->bounds.h,
			      window->attributes.bcolor, 
			      window->attributes.bpattern, 
			      window->attributes.xphase+window->ibounds.x,
			      window->attributes.yphase+window->ibounds.y);
        UnsetClipping();
#ifdef POPUPBUG
	DropCursors(window, &tile->bounds);
#endif POPUPBUG
	if (window->refresh != 0) {  /* call refresh routine if not null */
	    /* call the refresh routine */
	    (*window->refresh)(window->id,
			       tile->bounds.x-window->ibounds.x,
			       tile->bounds.y-window->ibounds.y,
			       tile->bounds.w, tile->bounds.h);
	}
    }
}

/*
 *	Routine to clear and/or refresh the new interior of a window
 *	when its size has increased.
 */
CreateStrip(window, strip)
register struct window *window;
struct rectangle strip;
{
    register struct tile *tiles, *tile;

    /* get a tile list of visable tiles to be refreshed */
    tiles = Visible(desk->next, window, strip);

    /* refresh these areas of the screen */
    for (tile = tiles->next; tile != tiles; tile = tile->next) {
	SetClipping(tile->bounds.x, tile->bounds.y, tile->bounds.w, 
		tile->bounds.h);
	DrawRectangleInterior(tile->bounds.x, tile->bounds.y, tile->bounds.w, 
		tile->bounds.h, window->bcolor[DISPLAY], solid, 0, 0);
        UnsetClipping();
    }
    /* paint back inside of window */
    CDrawInside(tile, window);

    /* return tile list to the wild */
    DestroyTileList(tiles);
}

/*
 *	Routine to refresh screen after the size of a window has decreased
 */
EraseStrip(window, strip)
register struct window *window;
struct rectangle strip;
{
    register struct window *w;
    register struct tile *tiles, *dtiles, *htiles, *itiles;

    /* get a tile list for tiles to be reset */
    dtiles = Visible(desk->next, window, strip);

    /* loop through all windows below given window */
    for (w = window->next; w != desk->next; w = w->next)
    {
	/* get tile list for current window and redraw border */
	htiles = Visible(window->next, w, w->obounds);
	itiles = TileIntersection(dtiles, htiles);
	CDrawBorder(itiles, w);
	DestroyTileList(htiles);
	DestroyTileList(itiles);

	/* get tile list for current window and redraw interior */
	htiles = Visible(window->next, w, w->ibounds);
	itiles = TileIntersection(dtiles, htiles);
	CDrawInside(itiles, w);
	DestroyTileList(htiles);
	DestroyTileList(itiles);
    }

    /* return tile list to freedom */
    DestroyTileList(dtiles);

}

/*
 *	Routine update all windows on screen between hi and lowindow
 */
Uncover(hiwindow, lowindow, tiles)
register struct window *hiwindow, *lowindow;
register struct tile *tiles;
{
    struct window *w;
    register struct tile *htiles, *itiles;

    /* loop backwards to draw top windows first */
    w = hiwindow;
    do {
	/* redraw window borders clipped appropriately */
	htiles = Visible(hiwindow, w, w->obounds);
	itiles = TileIntersection(tiles, htiles);
	CDrawBorder(itiles, w);
	DestroyTileList(htiles);
	DestroyTileList(itiles);

	/* refresh window interior clipped appropriately */
	htiles = Visible(hiwindow, w, w->ibounds);
	itiles = TileIntersection(tiles, htiles);
	CDrawInside(itiles, w);
	DestroyTileList(htiles);
	DestroyTileList(itiles);

	w = w->next;
    } while (w != lowindow->next);
}

/*
 *	Routine to copy region on window at sx,sy of width w and height h to
 *	dx,dy.
 */
CopyRegion(window, sx, sy, dx, dy, w, h)
register struct window *window;
int sx, sy, dx, dy;
register w,h;
{
    register struct tile *tile, *tile2, *rtile, *ttile;
    register cx, cy;	
    struct tile *stile, *dtile;
    struct rectangle bounds, cbounds;
    int		lessthan;		/* flag for sorting direction */

    /* compute relative change in region to copy */
    cx = dx - sx;
    cy = dy - sy;

    /* check if whole window is in one tile */
    tile = window->tiles->next;
    if ((tile->bounds.w == window->ibounds.w) &&
	(tile->bounds.h == window->ibounds.h)) {

	/* do something only if whole tile is visible */
	if (!tile->hidden) {
	    /* remove any cursors in the way */
	    cbounds.x = (sx < dx ? sx : dx) + window->ibounds.x;
	    cbounds.y = (sy < dy ? sy : dy) + window->ibounds.y;
	    cbounds.w = w + (cx < 0 ? -cx : cx);
	    cbounds.h = h + (cy < 0 ? -cy : cy);
	    Intersection(&cbounds, cbounds, tile->bounds);
	    LiftCursors(window, &cbounds);

	    /* set clipping to tile to avoid grief */
    	    SetClipping(tile->bounds.x, tile->bounds.y, 
		tile->bounds.w, tile->bounds.h);
	    /* clip source rectangle since this is not done in CopyArea */
	    bounds = window->ibounds;
	    bounds.x += sx;
	    bounds.y += sy;
	    bounds.w = w;
	    bounds.h = h;
	    Intersection(&bounds, bounds, window->ibounds);
            CopyArea(bounds.x, bounds.y, window->ibounds.x + dx, 
		window->ibounds.y + dy, bounds.w, bounds.h);
            UnsetClipping();
	    DropCursors(window, &cbounds);
	}
	return;
    }

    /* get a tile list for source and destination blocks */
    stile = CreateTileList();
    tile = InsertTile(stile);
    tile->bounds.x = sx + window->ibounds.x;
    tile->bounds.y = sy + window->ibounds.y;
    tile->bounds.w = w;
    tile->bounds.h = h;
    tile->hidden = FALSE;

    dtile = CreateTileList();
    tile = InsertTile(dtile);
    tile->bounds.x = dx + window->ibounds.x;
    tile->bounds.y = dy + window->ibounds.y;
    tile->bounds.w = w;
    tile->bounds.h = h;
    tile->hidden = FALSE;

    /* Intersect each with tiles of window to get visibility */
    for (tile = window->tiles->next; tile != window->tiles; tile = tile->next) {
	Partition(stile, tile->bounds);
	Partition(dtile, tile->bounds);
    }

    /* create a tile list to hold any hidden tiles which need to be refreshed */
    rtile = CreateTileList();

    /* sort the source and destination tiles, so we can find the overlaps */
    lessthan =  (cy < 0) || ((cy == 0) && (cx < 0));
    SortTileList( stile, lessthan);
    SortTileList( dtile, lessthan);

    /* Intersect translated source tile list (so that it overlays dest tiles)
     * and destination tile list, copying any visible tiles */
    for (tile = stile->next; tile != stile; tile = tile->next) {
	tile->bounds.x += cx;
	tile->bounds.y += cy;
	/* look at all dest tiles to get visibility */
	for (tile2 = dtile->next; tile2 != dtile; tile2 = tile2->next) {
	    /* do nothing if these two tiles do not overlap */
	    if (Overlap(tile->bounds, tile2->bounds)) {
		/* get area of commonality */
		Intersection(&bounds, tile->bounds, tile2->bounds);
		/* do nothing if destination is hidden */ 
	        if (tile2->hidden == TRUE) { 
            	    /* if source tile is visible then copy it, else refresh */
	    	    if (tile->hidden == TRUE) {
			/* remove any cursors in the way */
			cbounds = bounds;
			cbounds.x += (cx < 0 ? cx : -cx);
			cbounds.y += (cy < 0 ? cy : -cy);
			cbounds.w += 2*(cx < 0 ? -cx : cx);
			cbounds.h += 2*(cy < 0 ? -cy : cy);
			Intersection(&cbounds, cbounds, window->ibounds);
			LiftCursors(window, &cbounds);

    			/* set clipping to tile to avoid grief */
    			SetClipping(bounds.x, bounds.y, bounds.w, bounds.h);
		        CopyArea(bounds.x - cx, bounds.y - cy, 
			    bounds.x, bounds.y, bounds.w, bounds.h);
			UnsetClipping();
			DropCursors(window, &cbounds);
	            } else { 
			/* add this tile to be refreshed */
			ttile = InsertTile(rtile);
			ttile->bounds = bounds;
		    }
		}
	    }
	}
    }

    /* send a call to refresh if needed */
    if (rtile->next != rtile){
	CDrawInside(rtile, window);
    }

    /* return tile lists to freedom */
    DestroyTileList(dtile);
    DestroyTileList(rtile);
    DestroyTileList(stile);

}

/*
 *	Routine to create desktop
 */
InitializeWindow()
{
    freetiles = NULL;		/*  initialize free tile list to empty */

    /* get a window structure */
    desk = (struct window *)CreateBlock(sizeof(struct window));

    /* link window to self for there are no others */
    desk->next = desk;
    desk->prev = desk;

    /* set bounds to screen size */
    desk->ibounds.x = SCREENX;
    desk->ibounds.y = SCREENY;
    desk->ibounds.w = SCREENW;
    desk->ibounds.h = SCREENH;

    /* inside and frame bounds are the same */
    desk->obounds = desk->ibounds;

    /* clipping bounds are the world */
    desk->pcbounds = desk->ibounds;
    desk->tcbounds = desk->ibounds;

    /* set default attributes */
    ResetAttributes(desk);
    desk->type = NOSHOW;

    /* set a default cursor */
    desk->cursor = CreateCursor();
    desk->gcursor = CreateCursor();
    desk->cursor->visible = desk->cursor->blanked = 0;

    /* set up an empty tile list */
    desk->itiles = CreateTileList();
    desk->tiles = CreateTileList();

    /* set no adjust or refresh routine */
    desk->adjust = 0;
    desk->refresh = 0;

    /* set up window list and paint it all */
    Shuffle();
}

/*
 *	Routine to create a window at xs,ys on the screen with width w,
 *	height h and depth d.
 */
struct window * CreateWindow(xs, ys, w, h, d, type)
register xs, ys, w, h, d, type;
{
    int i;
    register struct window *window, *nindow;
    register struct tile *tiles;

    /* find spot in window list for window of this depth */
    window = desk;
    for (i = 0; i < d && window->next != desk; i++)
	window = window->next;

    /* get a new window struct and insert in window list */
    nindow = (struct window *)CreateBlock(sizeof(struct window));
    InsertWindow(window, nindow);

    /* set interior bounds of this window to requested size,position */
    nindow->ibounds.x = xs;
    nindow->ibounds.y = ys;
    nindow->ibounds.w = w;
    nindow->ibounds.h = h;

    /* set outside bounds for this window to include borders et al */
    if (type == QUIZ) {
	nindow->obounds.x = xs - BT;
	nindow->obounds.y = ys - RT - TT - BT;
	nindow->obounds.w = w + (2 * BT);
	nindow->obounds.h = h + RT + TT + (2 * BT);
    } else {
	nindow->obounds.x = xs -RT-BT;
	nindow->obounds.y = ys -RT-BT-TT-BT;
	nindow->obounds.w = w + 2*(BT+RT);
	nindow->obounds.h = h + 2*(BT+RT)+BT+TT;
    }

    /* set permanent clipping bounds */
    nindow->pcbounds.x = xs;
    nindow->pcbounds.y = ys;
    nindow->pcbounds.w = w;
    nindow->pcbounds.h = h;

    /* set temporary clipping bounds as above */
    nindow->tcbounds = nindow->pcbounds;

    /* null out title and set colors for glyphs */
    nindow->title[0] = 0;
    for (i = 0; i < NTILES; i++) {
	nindow->fcolor[i] = BLACK;
	nindow->bcolor[i] = WHITE;
	nindow->glyph[i] = v_icon[i];
    }

    /* set up default attributes, and defined type */
    ResetAttributes(nindow);
    nindow->type = type;    

    /* create a cursor for this window */
    nindow->cursor = CreateCursor();
    nindow->gcursor = CreateCursor();
    nindow->cursor->visible = nindow->cursor->blanked = 0;
    nindow->gcursor->visible = nindow->gcursor->blanked = 0;

    /* set a null adjust and refresh routine */
    nindow->adjust = 0;
    nindow->refresh = 0;

    /* create an empty clipped and unclipped tile list */
    nindow->itiles = CreateTileList();
    nindow->tiles = CreateTileList();

    /* reorganize tile lists for new window */
    Shuffle();

    /* create a tile list for painting this window on screen */

    tiles = Visible(desk->next, nindow, nindow->obounds);
    /* paint this window on screen and return tile list */
    if (type != NOSHOW) {
        LiftCursors(desk, &nindow->obounds);
	Uncover(nindow, nindow, tiles);
        DropCursors(desk, &nindow->obounds);
	MoveGlobalCursor(cursor.bounds.x + cursor.xref,
		     cursor.bounds.y + cursor.yref);
    }
    DestroyTileList(tiles);

    /* return window pointer */
    return(nindow);
}

/*
 *	Routine to delete a window structure from the window list
 */
DestroyWindow(window)
register struct window *window;
{
    register struct tile *dtiles;
    register struct window *neighbor;
    register short xpos, ypos, flag = FALSE; /* current global cursor poition */
    struct rectangle bounds;

    dtiles = Visible(desk->next, window, window->obounds);
    neighbor = window->next;

    /* set semephore and remove cursors */
    bounds = window->obounds;
    /* save current position of global cursor and move to bee on desk */
    if (cursor.visible) {
	HideGlobalCursor();
	flag = TRUE;
    }
    LiftCursors(desk, &bounds);

    /* remove this window from linked list, reorginze list and repaint */
    DeleteWindow(window);
    Shuffle();
    Uncover(neighbor, desk, dtiles);

    /* drop global cursor and restore to window */
    DropCursors(desk, &bounds);
    if (flag == TRUE) ShowGlobalCursor();

    /* return cursor, tile lists and window structs to free lists */
    DestroyCursor(window->cursor);
    DestroyCursor(window->gcursor);
    DestroyTileList(window->itiles);
    DestroyTileList(window->tiles);
    DestroyBlock(window);
    DestroyTileList(dtiles);
}

/*
 *	Routine to change the depth of the given window
 */
ChangeWindowDepth(window, depth)
register struct window *window;
register int depth;
{
    register int i, currentdepth;
    register struct window *w, *nindow, *neighbor;
    struct tile *tile, *ctiles, *vtiles, *itiles;
    register bool gflag = FALSE, lflag = FALSE;

    /* change type to display */
    if (window->type == NOSHOW) window->type = NORMAL;

    /* find window of next greater depth */
    neighbor = window->next;

    /* find window of previous depth */
    nindow = desk->next;
    for (i = 0; i < depth && nindow->next != desk; i++)
	nindow = nindow->next;

    /* set depth to true value as calculated by i */
    depth = i;

    /* find current depth of window */
    currentdepth = 0;
    for (w = desk->next; w != window; w = w->next)
	currentdepth++;

    /* hide any cursors in the way */
    if (cursor.visible) {
	gflag = TRUE;
	HideGlobalCursor();
    }
    if (window->cursor->visible) {
	lflag = TRUE;
	HideLocalCursor(window);
    }
    LiftCursors(window, &window->obounds);

    if (depth < currentdepth) {	    /* new depth is less than old */
	/* find tile list of tiles that need to be repainted */
	ctiles = Covered(desk->next, window, window->obounds);
	vtiles = Visible(desk->next, nindow, window->obounds);
	itiles = TileIntersection(ctiles, vtiles);

	/* remove window from old slot in tile list and insert at new */
	DeleteWindow(window);
	InsertWindow(nindow->prev, window);

	/* reorganize tile lists on all windows */
	Shuffle();

	/* repaint window at new depth */
	Uncover(window, window, itiles);

    } else if (depth > currentdepth) {	   /* new depth is greater than old */
	/* get list of tiles that need to be repainted at new depth */
	ctiles = Covered(window->next, nindow->next, window->obounds);
	vtiles = Visible(desk->next, window, window->obounds);
	itiles = TileIntersection(ctiles, vtiles);

	/* remove window from old slot in window list and insert at new */
	DeleteWindow(window);
	InsertWindow(nindow, window);

	/* recompute tiles for all windows */
	Shuffle();

	/* paint this window on screen and free captive tile lists */
	Uncover(neighbor, nindow, itiles);

    }
    /* replace those cursors, make sure local is still visible */
    DropCursors(window, &window->obounds);
    if (gflag) ShowGlobalCursor();
    if (lflag) ShowLocalCursor(window);

    /* free up temp tile lists */
    if (depth != currentdepth) {
	DestroyTileList(ctiles);
	DestroyTileList(vtiles);
	DestroyTileList(itiles);
    }
}

/*
 *	Routine to find the depth of the given window
 */
GetWindowDepth(window, d)
register struct window *window;
register short *d;
{
    register word depth;
    register struct window *w;

    depth = 0;
    for (w = desk->next; w != window; w = w->next)
	depth++;

    *d = depth;
}

/*
 *	Routine to change the size of the given window to w X h
 */
ChangeWindowSize(window, w, h)
register struct window *window;
register int w, h;
{
    register struct tile *tile, *tiles, *newtiles, *oldtiles;
    struct rectangle newbounds, newobounds, cbounds;
    register bool gflag = FALSE;

    /* do the easy one first */
    if( w == window->ibounds.w && h == window->ibounds.h)
	return;
	
    /* if there is an adjust routine then use it */
    if (window->adjust != 0)
	(*window->adjust)(window->aid, w, h);


    /* get interior rectangle for new size */
    newbounds.x = window->ibounds.x;
    newbounds.y = window->ibounds.y;
    newbounds.w = w;
    newbounds.h = h;

    /* get frame rectangle for new size */
    newobounds.x = window->obounds.x;
    newobounds.y = window->obounds.y;
    if (window->type == NORMAL) {
	newobounds.w = w + 2*(BT+RT);
	newobounds.h = h + 2*(BT+RT)+BT+TT;
    } else {
	newobounds.w = w + 2*(RT+BT);
	newobounds.h = h + TT+BT+2*(RT+BT);
    }

    /* get union of area to move and Hide all cursors */
    Union(&cbounds, newobounds, window->obounds);
    if (cursor.visible) {
	gflag = TRUE;
	HideGlobalCursor();
    }
    LiftCursors(window, &cbounds);

    /* find a tiling for the move */
    newtiles = RVisible(window->ibounds, newbounds);
    oldtiles = RVisible(newobounds, window->obounds);

    /* set new bounds in window */
    window->ibounds = newbounds;
    window->obounds = newobounds;
    /* reset clipping bounds to new window size */
    window->tcbounds = newbounds;
    window->pcbounds = newbounds;

    /* recreate tile lists for all windows */
    Shuffle();

    /* draw border at new location */
    tiles = Visible(desk->next, window, window->obounds);
    CDrawBorder(tiles, window);
    DestroyTileList(tiles);

    /* draw new tiles in window if it has grown bigger */
    for (tile = newtiles->next; tile != newtiles; tile = tile->next)
	CreateStrip(window, tile->bounds);

    /* erase old tiles on screen if window has shrunk */
    for (tile = oldtiles->next; tile != oldtiles; tile = tile->next)
	EraseStrip(window, tile->bounds);

    /* replace cursors on the screen */
    DropCursors(window, &cbounds);
    if (gflag) ShowGlobalCursor();

    /* return tile lists to freedom */
    DestroyTileList(newtiles);
    DestroyTileList(oldtiles);
}

/*
 *	Routine to get current window size
 */
GetWindowSize(window, w, h)
register struct window *window;
register short *w, *h;
{
    *w = window->ibounds.w;
    *h = window->ibounds.h;
}

/*
 *	Routine to move given window to new position x.y
 */
ChangeWindowPosition(window, x, y)
register struct window *window;
register int x, y;
{
    struct tile *tile;
    struct tile *vnew, *vold, *vnewx, *voldx;
    struct tile *vslide, *vpaint, *verase;
    struct rectangle bounds, ubounds;
    register int dx, dy;
    register bool gflag = FALSE;

    /* find bounds of current window */
    bounds = window->obounds;
    if (window->type == NORMAL) {
	bounds.x = x -RT-BT;
	bounds.y = y -RT-BT-TT-BT;
    } else {
	bounds.x = x - RT-BT;
	bounds.y = y - TT-RT-2*BT;
    }

    /* get area on screen to change and hide all cursors */
    Union(&ubounds, bounds, window->obounds);
    if (cursor.visible) {
	gflag = TRUE;
	HideGlobalCursor();
    }
    LiftCursors(window, &ubounds);

    /* find translation vector and move local cursor */
    dx = x - window->ibounds.x;
    dy = y - window->ibounds.y;
    window->cursor->bounds.x += dx;
    window->cursor->bounds.y += dy;


    /* find tiles that need to be repainted at new position */
    vold = Visible(desk->next, window, window->obounds);
    voldx = TileTranslation(vold, dx, dy);

    /* reset interior and frame bounds */
    window->ibounds.x = x;
    window->ibounds.y = y;
    window->obounds = bounds;

    /* reset clipping bounds */
    window->pcbounds.x += dx;
    window->pcbounds.y += dy;
    window->tcbounds.x += dx;
    window->tcbounds.y += dy;

    /* remake tile lists for all windows */
    Shuffle();

    /* find a tiling for new location */
    vnew = Visible(desk->next, window, window->obounds);
    vnewx = TileTranslation(vnew, -dx, -dy);

    /* find tiles which can be slide and do it */
    vslide = TileIntersection(vold, vnewx);
    SlideTiles(vslide, dx, dy);
    DestroyTileList(vslide);

    /* repaint tiles not done above */
    vpaint = TileDifference(vnew, voldx);
    Uncover(window, window, vpaint);
    DestroyTileList(vpaint);

    /* erase old tiles */
    verase = TileDifference(vold, vnew);
    Uncover(window->next, desk, verase);

    /* replace cursors */
    DropCursors(window, &ubounds);    
    if (gflag) ShowGlobalCursor();

    /* return tile lists to freedom */
    DestroyTileList(verase);
    DestroyTileList(vnew);
    DestroyTileList(vnewx);
    DestroyTileList(vold);
    DestroyTileList(voldx);
}

/*
 *	Routine to get given windows position
 */
GetWindowPosition(window, x, y)
register struct window *window;
register short *x, *y;
{
    *x = window->ibounds.x;
    *y = window->ibounds.y;
}

/*
 *	Routine to get and display a new window title
 */
SetWindowTitle(window, title)
register struct window *window;
register char *title;
{
    register int i;
    register struct tile *tiles;
    struct rectangle bounds;

    /* copy title to window structure , don't forget to null terminate */
    for (i = 0; i < TITLELENGTH-1; i++)
	window->title[i] = title[i];
    window->title[TITLELENGTH-1] = 0;

    /* get bounds for title to remove cursors */
    PaneBounds(window, TITLE, &bounds);
    LiftCursors(window, &bounds);

    /* get tiles need to refresh title bar and draw it, free tile list */
    tiles = Visible(desk->next, window, bounds);
    CDrawBorder(tiles, window);
    DropCursors(window, &bounds);
    DestroyTileList(tiles);
}

/*
 *	Routine to set new colors in named pane of given window
 */
SetPaneColors(window, pane, fcolor, bcolor)
register struct window *window;
register int pane;
COLORT fcolor, bcolor;
{
    register struct tile *tiles;
    struct rectangle bounds;

    /* copy in colors */
    window->fcolor[pane] = fcolor;
    window->bcolor[pane] = bcolor;

    /* get bounds for this pane and remove cursors */
    PaneBounds(window, pane, &bounds);
    LiftCursors(window, &bounds);

    /* compute tiling for pane and redraw */
    tiles = Visible(desk->next, window, bounds);
    CDrawBorder(tiles, window);
    DestroyTileList(tiles);

    /* return cursors */
    DropCursors(window, &bounds);
}

/*
 *	Routine to set a new glyph in the named pane of the given window
 */
SetPaneGlyph(window, pane, glyph)
register struct window *window;
register int pane;
register int *glyph;
{
    register struct tile *tiles;
    struct rectangle bounds;

    /* copy in pointer to glyph */
    window->glyph[pane] = glyph;

    /* get bounds for this pane and remove cursors */
    PaneBounds(window, pane, &bounds);
    LiftCursors(window, &bounds);

    /* compute a tiling for named pane and redraw */
    tiles = Visible(desk->next, window, bounds);
    CDrawBorder(tiles, window);
    DestroyTileList(tiles);

    /* return cursors */
    DropCursors(window, &bounds);
}

/*
 *	Routine to set a new adjust routine for the window
 */
SetAdjust(window, id, routine)
register struct window *window;
register int id;
register int  (*routine)();
{
    window->aid = id;
    window->adjust = routine;
}


/*
 *	Routine to set a new refresh routine for the named window
 */
SetRefresh(window, id, routine)
register struct window *window;
register int id;
register int  (*routine)();
{
    window->id = id;
    window->refresh = routine;
}


/*
 *	routine to get window of destop
 */
struct window *DesktopWindow()
{
    return(desk);
}
