#include "../h/ucode.h"
/******************************************************************************
* LSETUP
*
* This routine initializes the registers to the values needed by the raster
* operation routines. On entry the registers should contain:
*
*	D0 (W):		source rectangle X coordinate
*	D1 (W):		source rectangle Y coordinate
*	D2 (W):		destination rectangle X coordinate
*	D3 (W):		destination rectangle Y coordinate
*	D4 (W):		rectangle width
*	D5 (W):		rectangle height
*
*	A0 (W):		source raster width
*	A1 (W):		destination raster width
*	A2 (W):		source raster address (0 if no source)
*	A3 (W):		destination raster address
*
* On exit, the registers contain:
*
*	D0 (B):		?
*	D1 (B):		alignment rotation value (0 and 80)
*	D2 (W):		initial mask
*	D3 (W):		final mask
*	D4 (W):		adjusted width in longs
*	D5 (W):		adjusted height in lines
*
*	A0 (W):		source vertical increment
*	A1 (W):		destination vertical increment
*	A2 (L):		source address
*	A3 (L):		destination address
*	A4 (W):		horizontal increment
*
* Also, on exit, the N condition code is set if the height of the rectangle
* is zero.
*
* This routine uses all registers except D7, A5, and A6.  Callers with values
* to preserve should use these registers.
*
******************************************************************************/
	.globl	lsetup
	.text
lsetup:
/*
 * Compensate for non-positive width or height by updating the X or Y
 * coordinates.
 */
	tstw	d4			/* test the width */
	bgt	1f			/* skip adjust if already positive */
	addw	d4,d0			/* compute X before source left edge */
	addqw	#1,d0			/* X of source left edge */
	addw	d4,d2			/* compute X before destination left edge */
	addqw	#1,d2			/* X of destination left edge */
	negw	d4			/* make width positive */

    1:	tstw	d5			/* test the height */
	bgt	3f			/* skip adjust if already positive */
	addw	d5,d1			/* compute Y before source top */
	addqw	#1,d1			/* Y of source top */
	addw	d5,d3			/* compute Y before destination top */
	addqw	#1,d1			/* Y of destination top */
	negw	d5			/* make height positive */

/*
 * Compute source address and store in A2.  This is the address of the word
 * containing the first source bit.
 */
    3:	movw	a0,d6			/* get source raster width */
	mulu	d6,d1			/* compute byte offset of row */
	addl	d1,a2			/* address of row */
	movw	d0,d6			/* get source X coordinate */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a2			/* address of word */

/*
 * Compute destination address and store in A3.  This is the address of the
 * word containing the first destination bit.
 */
	movw	a1,d6			/* get destination raster width */
	mulu	d6,d3			/* compute byte offset of row */
	addl	d3,a3			/* address of row */
	movw	d2,d6			/* get destination X coordinate */
	asrw	#4,d6			/* compute word offset */
	aslw	#1,d6			/* byte offset */
	addw	d6,a3			/* address of word */

/*
 * Copy the destination X coordinate to D0 and adjust it so that the first bit
 * lies in the high word of a long.  This is used in the following mask and
 * width computations to ensure that the left mask contains as many bits as
 * possible.  The modulo 16 address computations above require this.
 */
	movw	d2,d0			/* copy destination x to d0 */
	bclr	#4,d0			/* force to high word */

/*
 * Compute the left mask and store in D2.  This mask has one bit for each bit
 * in the left partial words.
 */
	movw	d0,d6			/* get source x */
	andw	#0x1F,d6		/* take modulo 32 (size of long) */
	moveq	#-1,d2			/* get all ones */
	lsrl	d6,d2			/* shift in the proper number of zeros */

/*
 * Compute the right mask and store in D3.  This mask has one bit for each bit
 * in the right partial words.  A side effect of this computation is that D4
 * contains the X coordinate of the last source bit on the first row.
 */
	addw	d0,d4			/* add source x to width */
	subqw	#1,d4			/* adjust to get x of last bit */
	movw	d4,d6			/* copy it */
	notw	d6			/* reverse bit numbering scheme */
	andw	#0x1F,d6		/* take modulo 32 */
	moveq	#-1,d3			/* get all ones */
	lsll	d6,d3			/* shift in the proper number of zeros */

/*
 * Compute the width in longs of the middle section and store in D4.  This is
 * the difference between the first and last long offsets, minus one.
 */
	asrw	#5,d0			/* compute long offset of first bit */
	asrw	#5,d4			/* compute long offset of last bit */
	subw	d0,d4			/* take their difference */
	subqw	#1,d4			/* and adjust */

/*
 * Adjust for the case where the entire row lies within a single long.  This
 * is indicated by a negative width.  In this case, only the left mask is used
 * and the right mask is zeroed.
 */
	bge	4f			/* branch if width is non-negative */
	andl	d3,d2			/* combine masks if width negative */
	clrl	d3			/* clear left mask */
	clrw	d4			/* and zero width */

/*
 * Compute the source and destination vertical increments and store in A0 
 * and A1.  These are the amounts by which the pointers have to be incremented
 * at the end of one row to get to the beginning of the next.  Also store
 * 4 as the horizontal increment in A4.
 */
    4:	movw	d4,d6			/* get width of middle section */
	aslw	#2,d6			/* convert to a byte width */
	subw	d6,a0			/* subtract middle from source width */
	subqw	#4,a0			/* subtract left side */
	subw	d6,a1			/* subtract middle from destination */
	subqw	#4,a1			/* subtract left side */
	movw	#4,a4			/* initialize horizontal increment */

/*
 * Finally, adjust the width and height for the dbf instruction used by the
 * raster routines.  Also set the N bit if the height is zero.
 */
	subqw	#1,d4			/* adjust width */
	subqw	#1,d5			/* adjust height (and set N) */
	rts				/* return */
