/*
 *	These routines are used to manipulate the various cursors of all
 *	windows on the screen.  Currently there are three main types of
 *	cursors available.  The top most cursor (meaning that it slides
 *	over every other cursor ) is the global rubber box cursor, refered
 *	as box.  The next level cursor, which can travel over the whole
 *	screen, yet under the box cursor is the global cursor, refered to
 *	as cursor.  Each window has a definition for a global cursor which
 *	will define the global cursor when it is in that window.  When the
 *	global cursor is not in any window, it is the cursor as defined for
 *	the desktop.  The third type of cursor is a local cursor, one per
 *	window.  These cursor are clipped to the window bounds and lie
 *	both of the above cursors.  The following module manipulates all
 *	of these cursors in an ordered manner.
 */

#include	<vt_hdrs.h>	    /* system wide defines */
#include	<rectangle.h>	    /* rectangle structure */
#include	<block.h>	    /* storage allocation */
#include	<tile.h>	    /* tile structure */
#include	<cursor.h>	    /* cursor structure */
#include	<window.h>	    /* window structure */
#include	<ucode.h>


extern int		offscreen_off;
struct cursor     	*activecursor;	    /* pointer to the global cursor */
struct cursor     	*loadedcursor;	    /* pointer to the global cursor */
struct cursor     	  cursor;	    /* the global cursor */
struct rubbercursor	     box;	    /* global rubber baby box */
char	vidbuf[];

Sendtones(){}

/*
 *	Routine to draw the cursor to the raster clipped to the given
 *	rectangle.
 */
CursorToRaster(raster,c,wbounds)
RASTER *raster;
register struct cursor * c;
struct rectangle wbounds;
{
    struct rectangle common;

    /* check if cursor intersects given area */
    if (Overlap(wbounds, c->bounds))
    {
	/* get rectangle of intersction */
	Intersection(&common, wbounds, c->bounds);

	/* paint the cursor mask and rectangle to the screen */
	PaintRaster(&c->mask, common.x - c->bounds.x, common.y - c->bounds.y,
		    raster, common.x, common.y, common.w, common.h, BLACK);


	/* paint the cursor face and rectangle to the screen */
	if (gptype & GPCOLOR)
	  PaintRaster(&c->face, common.x - c->bounds.x, common.y - c->bounds.y,
		    raster, common.x, common.y, common.w, common.h, WHITE);
	else
	  ToggleRaster(&c->face, common.x - c->bounds.x, common.y - c->bounds.y,
		    raster, common.x, common.y, common.w, common.h);
    }
}


/*
 *	Routine to a create a cursor structure and initialize
 */
struct cursor *CreateCursor()
{
    register struct cursor *c;

    /* get space for a new cursor */
    c = (struct cursor *)CreateBlock(sizeof(struct cursor));

    /* set width of face and get space for it, size is in bytes */
    c->face.width = (CURSORW+15)/16*2;
    c->face.address = (word *)CreateBlock((CURSORW+15)/16*2*CURSORH);

    /* set RASTER width for mask and get space for it */
    c->mask.width = (CURSORW+15)/16*2;
    c->mask.address = (word *)CreateBlock((CURSORW+15)/16*2*CURSORH);

    /* set RASTER width for background and get space for it */
    c->back.width = (CURSORW+15)/16*2;
    c->back.address = (word *)CreateBlock((CURSORW+15)/16*2*CURSORH);

    /* set a default cursor size and position within window */
    c->bounds.x = 0;
    c->bounds.y = 0;
    c->bounds.w = CURSORW;
    c->bounds.h = CURSORH;

    /* clear out mask and face areas of cursor */
    ClearRaster(&c->mask, 0, 0, CURSORW, CURSORH, WHITE);
    ClearRaster(&c->face, 0, 0, CURSORW, CURSORH, BLACK);

    /* make cursor invisible and not blanked */
    c->visible = FALSE;
    c->blanked = 0;

    /* set default reference points */
    c->xref = 0;
    c->yref = 0;

    /* set blink rate to noblink */
    c->blink = 0;

    /* return cursor */
    return(c);
}


/*
 *	Routine to return all the space for a cursor to freedom
 */
DestroyCursor(c)
register struct cursor *c;
{
    /* return mask, face and backgound, the cursor struct itself */
    DestroyBlock(c->face.address);
    DestroyBlock(c->mask.address);
    DestroyBlock(c->back.address);
    DestroyBlock(c);
}

/*
 *	Routine to make box cursor invisible and set up semephore flag
 */
InitializeCursor()
{
    /* make box cursor not there */
    box.visible = FALSE;
    box.blanked = 0;

    /* initialize hardware cursor rasters */
    cursor.face.width = 4;
    cursor.face.address = (short *)(vidbuf + offscreen_off + 0x200 + 0x000);
    cursor.mask.width = 4;
    cursor.mask.address = (short *)(vidbuf + offscreen_off + 0x200 + 0x080);
    cursor.back.width = 4;
    cursor.back.address = (short *)(vidbuf + offscreen_off + 0x200 + 0x100);
}

/*
 *	Routine to completely create the given cursor with the given mask,
 *	face, referance points and blink rate. 
 */
DefineCursor(c, mask, face, xref, yref, blink)
register struct cursor *c;
int *mask, *face;
byte xref, yref;
byte blink;
{
    register int i;
    register int *sp, *dp;

    /* set new referance position for cursor */
    c->bounds.x += (c->xref - xref);
    c->bounds.y += (c->yref - yref);

    /* copy in face - NOTE implied hard coded width of 32 bits */
    sp = face;
    dp = (int *) c->face.address;
    for (i = 0; i < CURSORH; i++)
	*dp++ = *sp++;		    /* copy in 32 bits */

    /* copy in mask - NOTE implied hard coded width of 32 bits */
    sp = mask;
    dp = (int *) c->mask.address;
    for (i = 0; i < CURSORH; i++)
	*dp++ = *sp++;		    /* copy in 32 bits */

    /* set new referance point of cursor */
    c->xref = xref;
    c->yref = yref;

    /* set new blink rate of cursor */
    c->blink = blink;

}

/*
 *	Routine to move the given windows local cursor to the given
 *	position x,y within the window.
 */
MoveLocalCursor(window, x, y)
register struct window *window;
register int x, y;
{
    struct rectangle gbounds, oldbounds, newbounds, ubounds;
    register struct tile * tiles, *tile;
    unsigned char IntersectBox();
    /* visibility flags */
    bool boxflag = 0, globalflag = FALSE, oldflag = FALSE, newflag = FALSE;

    
    /* adjust location for referance point, x,y in screen coordinates */
    x = (window->ibounds.x + x - window->cursor->xref);
    y = (window->ibounds.y + y - window->cursor->yref);

    /* find visible area of cursor to remove */
    oldbounds = window->cursor->bounds;
    newbounds = window->cursor->bounds;
    newbounds.x = x;
    newbounds.y = y;

    /* do nothing if not visible */
    if (window->cursor->visible) {

	/* loop through tiles of window and restrict bounds for window */
	tiles = window->itiles;
	for (tile = tiles->next; tile != tiles; tile = tile->next) {
	    if (Overlap(oldbounds, tile->bounds)) {
		Intersection(&oldbounds, oldbounds, tile->bounds);
		oldflag = TRUE;
	    }
	    if (Overlap(newbounds, tile->bounds)) {
		Intersection(&newbounds, newbounds, tile->bounds);
		newflag = TRUE;
	    }
	}

	/* make sure cursor is visible in either new or old position */
	if ((newflag) || (oldflag)) {

	    /* get union of old and new position to remove other cursors */
	    Union(&gbounds, oldbounds, newbounds);

	    /* restore area behind box cursor if visible */
	    if ((box.visible) && ((boxflag = IntersectBox(gbounds)))) 
		RestoreBoxBackground(boxflag);

	    /* restore area behind global cursor to screen if visible */
	    if (oldflag && cursor.visible && 
		Overlap(oldbounds,cursor.bounds)) {

		Intersection(&gbounds, oldbounds, cursor.bounds);
		CopyRaster(&cursor.back, gbounds.x - cursor.bounds.x , 
		    gbounds.y - cursor.bounds.y, &screen, gbounds.x, 
		    gbounds.y, gbounds.w, gbounds.h);
		globalflag = TRUE;
	    }

	    /* remove old cursor if visible */
	    if (oldflag) 
		CopyRaster(&window->cursor->back,oldbounds.x - 
		    window->cursor->bounds.x, oldbounds.y - 
		    window->cursor->bounds.y, &screen, oldbounds.x, oldbounds.y,
		    oldbounds.w, oldbounds.h);

	    /* replace global cursor in old position if removed above */
	    if (globalflag) {
		CopyRaster(&screen, gbounds.x, gbounds.y, &cursor.back, 
		    gbounds.x - cursor.bounds.x, gbounds.y - cursor.bounds.y,
		    gbounds.w, gbounds.h);
		CursorToRaster(&screen, &cursor, gbounds);
		globalflag = FALSE;
	    }

	    /* save new bounds for this cursor */
	    window->cursor->bounds.x = x;
	    window->cursor->bounds.y = y;

	    /* restore area behind global cursor if visible */
	    if (newflag && cursor.visible && 
		Overlap(newbounds,cursor.bounds)) {

		Intersection(&gbounds, newbounds, cursor.bounds);
		CopyRaster(&cursor.back, gbounds.x - cursor.bounds.x , 
		    gbounds.y - cursor.bounds.y, &screen, gbounds.x, 
		    gbounds.y, gbounds.w, gbounds.h);
		globalflag = TRUE;
	    }

	    /* save area behind local cursor and put it on screen if visible */
	    /*
	     * Danger of bug here, can local cursor be moved to area which 
	     * expects to have no local cursor there in the async world ????
	     */
	    if (newflag) {
		CopyRaster(&screen, newbounds.x, newbounds.y, 
		  &window->cursor->back, newbounds.x - window->cursor->bounds.x,
		  newbounds.y - window->cursor->bounds.y, newbounds.w, 
		  newbounds.h);
		CursorToRaster(&screen, window->cursor, newbounds);
	    }

	    /* save area behind global cursor if it needs it */
	    if (globalflag) {
		Intersection(&gbounds, newbounds, cursor.bounds);
		CopyRaster(&screen, gbounds.x, gbounds.y, &cursor.back, 
		  gbounds.x - cursor.bounds.x, gbounds.y - cursor.bounds.y,
		  gbounds.w, gbounds.h);
		CursorToRaster(&screen, &cursor, gbounds);
	    }

	    /* save area behind box cursor and put on screen */
	    if (boxflag) {
		SaveBoxBackground(boxflag);
		DrawBox(boxflag);
	    }
	}
    } else {
	/* save new bounds for this cursor */
	window->cursor->bounds.x = x;
	window->cursor->bounds.y = y;
    }

}

/*
 *	Routine to move local cursor of window to window position
 */
TrackLocalCursor(window)
register struct window *window;
{
    MoveLocalCursor(window, window->attributes.xposition,
		    window->attributes.yposition);
}

/*
 *	Set the global cursor to be the gcursor of the given window
 */
SetActiveCursor(window)
WINDOW	window;
{
	activecursor = window->gcursor;
}

/*
 *	Routine to move the global cursor to the given position x,y
 *	given in screen coordinates.
 */
MoveGlobalCursor(x, y)
register int x, y;
{
    RASTER src, dest;
    struct rectangle cbounds;		/* bounds of new cursor position */
    RASTER back, newback;		/* raster for the backgrounds */
    int	data[32];
    unsigned char IntersectBox(), boxflag;

    /* do nothing if cursor is not visible */
    if (cursor.visible) {
	struct rectangle boxibounds;

	cbounds = cursor.bounds;
	back = cursor.back;
	
	/* set flag on old cursor, make new cursor and set flag */
	if (loadedcursor != activecursor) {
	    cursor.visible = FALSE;
	    activecursor->blanked = cursor.blanked;
	    LoadGlobalCursor(activecursor);
	    cursor.visible = TRUE;
	}
	
	if (gptype & GPCOLOR)
	    if ((y - cursor.yref) <= 0)
	    	y = cursor.yref+1;

	/* set new cursor position */
	cursor.bounds.x = x - cursor.xref;
	cursor.bounds.y = y - cursor.yref;
	cursor.bounds.w = CURSORW;
	cursor.bounds.h = CURSORH;

	/* restore area behind box cursor if visible */
	Union(&boxibounds, cursor.bounds, cbounds);
	if (box.visible && (boxflag = IntersectBox(boxibounds))) {
	    box.blanked++;
	    RestoreBoxBackground(boxflag);
	}

	/* lets keep the global cursor on the desktop */
	if (Overlap(cursor.bounds, desk->ibounds))
	    Intersection(&cursor.bounds, cursor.bounds, desk->ibounds);
	else {			/* try to salvage the damage */
	    if (Overlap(cbounds, desk->ibounds))
		CopyRaster(&back, 0, 0, &screen, cbounds.x, cbounds.y, 
		    cbounds.w, cbounds.h);
	    goto exit;
	}

	if (Overlap(cbounds, cursor.bounds) && cbounds.x < cursor.bounds.x){
	    /* do the fancy update */
	    struct rectangle ibounds, tbounds, bbounds;

	    /* Get the background of the new cursor */
	    newback.width = 4;
	    newback.address= (word *)(vidbuf + offscreen_off + 0x200 + 0x180);
	    CopyRaster(&screen, cursor.bounds.x, cursor.bounds.y, 
		&newback, 0, 0, cursor.bounds.w, cursor.bounds.h);

	    Intersection(&ibounds, cbounds, cursor.bounds);
	    /* update the intersected area from the old back to the new */
	    CopyRaster(&back, ibounds.x - cbounds.x, 
		ibounds.y - cbounds.y, &newback, 
		ibounds.x - cursor.bounds.x, ibounds.y - cursor.bounds.y,
		ibounds.w, ibounds.h);

	    /* draw new cursor into the intersection with the old background*/
	    tbounds = cursor.bounds;
	    cursor.bounds.x -= cbounds.x;  /* cursor to raster needs this */
	    cursor.bounds.y -= cbounds.y;
	    bbounds = ibounds;
	    bbounds.x -= cbounds.x;
	    bbounds.y -= cbounds.y;
	    CursorToRaster(&back, &cursor, bbounds);
	    /* and put it on the screen */
	    CopyRaster(&back, 0, 0, &screen, cbounds.x, cbounds.y, 
		cbounds.w, cbounds.h);
	    CopyRaster(&newback, 0, 0, &back, 0, 0, CURSORW, CURSORH);

	    /* and finally finish drawing the new cursor */
	    cbounds = tbounds;
	    cbounds.x = cbounds.y = 0;
	    cursor.bounds.x = x - cursor.xref - tbounds.x;
	    cursor.bounds.y = y - cursor.yref - tbounds.y;
	    cursor.bounds.w = CURSORW;
	    cursor.bounds.h = CURSORH;
	    CursorToRaster(&newback, &cursor, cbounds);
	    cursor.bounds = tbounds;
	    CopyRaster(&newback, 0, 0, &screen, cursor.bounds.x, 
		cursor.bounds.y, cursor.bounds.w, cursor.bounds.h);
	} else {
	    CopyRaster(&back, 0, 0, &screen, cbounds.x, cbounds.y, 
		cbounds.w, cbounds.h);
	    /* save area behind global cursor and put on screen */
	    CopyRaster(&screen, cursor.bounds.x, cursor.bounds.y, 
		&cursor.back, 0, 0, cursor.bounds.w, cursor.bounds.h);
	    CursorToRaster(&screen, &cursor, cursor.bounds);
	}
    exit:
	/* replaced removed portion of box if need be */
	if (box.blanked) {
	    SaveBoxBackground(boxflag);
	    DrawBox(boxflag);
	    box.blanked--;
	}
    } else {
	/* set flag on old cursor, make new cursor and set flag */
	if (loadedcursor != activecursor) {
	    cursor.visible = FALSE;
	    activecursor->blanked = cursor.blanked;
	    LoadGlobalCursor(activecursor);
	    cursor.visible = TRUE;
	}

	/* set new cursor postion */
	cursor.bounds.x = x - cursor.xref;
	cursor.bounds.y = y - cursor.yref;
    }
}

/*
 *	Routine to turn visibility of global cursor off
 */
HideGlobalCursor()
{
    unsigned char IntersectBox(), boxflag;

    /* do nothing if not there */
    if (cursor.visible) {
	/* restore area behind global cursor */
	CopyRaster(&cursor.back, 0, 0, &screen, cursor.bounds.x, 
		cursor.bounds.y, cursor.bounds.w, cursor.bounds.h);

	/* get anything new that is in rubber cursor backgound */
	if ((box.visible) && (boxflag = IntersectBox(cursor.bounds))) {
	    SaveBoxBackground(boxflag);
	    DrawBox(boxflag);
	}
	/* set flags and decrement semephore */
	cursor.visible = FALSE;
	cursor.blanked = 0;
    }

}

/*
 *	Routine to turn visibility of global cursor on
 */
ShowGlobalCursor()
{
    RASTER	dest;
    unsigned char IntersectBox(), boxflag;
    WINDOW	window;

    /* do nothing if already visible */
    if (!cursor.visible) {
	if ((box.visible) && (boxflag = IntersectBox(cursor.bounds))) {
	    box.blanked++;
	    RestoreBoxBackground(boxflag);
	}

	/* find which window cursor is in */
	ContainingWindow(cursor.bounds.x , cursor.bounds.y , &window);
	if ((window != NULL) && (loadedcursor != activecursor)) {
/*	    cursor.visible = FALSE; /**/
	    activecursor->blanked = cursor.blanked;
	    activecursor->bounds.x = cursor.bounds.x;
	    activecursor->bounds.y = cursor.bounds.y;
	    LoadGlobalCursor (activecursor);
/*	    cursor.visible = TRUE; /**/
	}

	/* save background of global cursor and draw on screen */
	CopyRaster(&screen, cursor.bounds.x, cursor.bounds.y, 
	    &cursor.back, 0, 0, cursor.bounds.w, cursor.bounds.h);
	CursorToRaster(&screen, &cursor, cursor.bounds);

	/* save area behind box cursor and put on screen */
	if (box.blanked) {
	    SaveBoxBackground(boxflag);
	    DrawBox(boxflag);
	    box.blanked--;
	}
	/* set flags and decrement semephore */
	cursor.visible = TRUE;
	cursor.blanked = 0;
    }
}

/*
 *	Routine to turn visibility of local cursor in given window off
 */
HideLocalCursor(window)
register struct window *window;
{
    unsigned char IntersectBox(), boxflag = 0;
    struct rectangle cbounds, gbounds;
    register struct rectangle *bp;
    register struct tile *tiles, *tile;

    /* do nothing if not visible */
    if (window->cursor->visible) {
	/* set cbounds to cursor */
	bp = &window->cursor->bounds;
	cbounds = *bp;

	/* loop through tiles of window and restrict bounds for window */
	tiles = window->itiles;
	for (tile = tiles->next; tile != tiles; tile = tile->next) {
	    if (Overlap(cbounds, tile->bounds)) {
		Intersection(&cbounds, cbounds, tile->bounds);
		boxflag++;
	    }
	}

	/* make sure cursor is not covered, else flee while thee can */
	if (boxflag != 0) {
	    /* check if box cursor needs to be removed */
	    if ((box.visible) && (boxflag = IntersectBox(cbounds))) {
		RestoreBoxBackground(boxflag);
		box.blanked++;
	    }

	    /* check if global cursor needs to be removed */
	    if ((cursor.visible) && (Overlap(cbounds,cursor.bounds))) {
		Intersection(&gbounds, cbounds,cursor.bounds);
		CopyRaster(&cursor.back, gbounds.x - cursor.bounds.x, 
			gbounds.y - cursor.bounds.y, &screen, gbounds.x, 
			gbounds.y, gbounds.w, gbounds.h);
		cursor.blanked++;
	    }
		
	    /* restore area behind local cursor if in window */
	    CopyRaster(&window->cursor->back, cbounds.x - bp->x, cbounds.y -
	    bp->y, &screen, cbounds.x, cbounds.y, cbounds.w, cbounds.h);

	    /* check if global cursor needs to be repainted */
	    if (cursor.blanked) {
		CopyRaster( &screen, gbounds.x, gbounds.y, &cursor.back, 
		    gbounds.x - cursor.bounds.x, gbounds.y - cursor.bounds.y, 
		    gbounds.w, gbounds.h);
		CursorToRaster(&screen, &cursor, gbounds);
		cursor.blanked--;
	    }
		
	    /* check if box cursor needs to be removed */
	    if (box.blanked) {
		    SaveBoxBackground(boxflag);
		    DrawBox(boxflag);
		    box.blanked--;
	    }
	}
	/* set visibility flag */
	window->cursor->visible = FALSE;
	window->cursor->blanked = 0;
    }
}

/*
 *	Routine to turn visibility of local cursor for given window on
 */
ShowLocalCursor(window)
register struct window *window;
{
    unsigned char IntersectBox(), boxflag = 0;
    struct rectangle cbounds, gbounds;
    register struct tile *tiles, *tile;

    /* do nothing if visible */
    if (!window->cursor->visible) {
	/* set cbounds to cursor */
	cbounds = window->cursor->bounds;

	/* loop through tiles of window and restrict bounds for window */
	tiles = window->itiles;
	for (tile = tiles->next; tile != tiles; tile = tile->next) {
	    if (Overlap(cbounds, tile->bounds)) {
		Intersection(&cbounds, cbounds, tile->bounds);
		boxflag++;
	    }
	}

	/* make sure cursor is not obsured */
	if (boxflag) { 
	    /* check if box cursor needs to be removed */
	    if ((box.visible) && (boxflag = IntersectBox(cbounds))) {
		RestoreBoxBackground(boxflag);
		box.blanked++;
	    }

	    /* check if global cursor needs to be removed */
	    if ((cursor.visible) && (Overlap(cbounds,cursor.bounds))) {
		Intersection(&gbounds, cbounds,cursor.bounds);
		CopyRaster(&cursor.back, gbounds.x - cursor.bounds.x, 
		    gbounds.y - cursor.bounds.y, &screen, gbounds.x, 
		    gbounds.y, gbounds.w, gbounds.h);
		cursor.blanked++;
	    }
	
	    /* save background behind cursor and put it on screen */
	    CopyRaster(&screen, cbounds.x, cbounds.y, &window->cursor->back,
		cbounds.x - window->cursor->bounds.x, cbounds.y -
		window->cursor->bounds.y, cbounds.w, cbounds.h);
	    CursorToRaster(&screen, window->cursor, cbounds);

	    /* check if global cursor needs to be repainted */
	    if (cursor.blanked) {
		CopyRaster( &screen, gbounds.x, gbounds.y, &cursor.back, 
		    gbounds.x - cursor.bounds.x, gbounds.y - cursor.bounds.y, 
		    gbounds.w, gbounds.h);
		CursorToRaster(&screen, &cursor, gbounds);
		cursor.blanked--;
	    }
	
	    /* check if box cursor needs to be removed */
	    if (box.blanked) {
		SaveBoxBackground(boxflag);
		DrawBox(boxflag);
		box.blanked--;
	    }
	}

	/* reset flags and decrement semephore */
	window->cursor->visible = TRUE;
	window->cursor->blanked = 0;
    }
}

/*
 *	Routine to load the cursor pointed to by cur into the hardware 
 *	cursor area. (whatever that me be).
 */
LoadGlobalCursor(cur)
register struct cursor	*cur;
{
    RASTER	dest;

    loadedcursor = cur;

    cursor.xref = cur->xref;
    cursor.yref = cur->yref;
    cursor.visible = cur->visible;
    cursor.blanked = cur->blanked;
    cursor.blink = cur->blink;
    cursor.bounds = cur->bounds;

    CopyRaster(&cur->face, 0, 0, &cursor.face, 0, 0, CURSORW, CURSORH);
    CopyRaster(&cur->mask, 0, 0, &cursor.mask, 0, 0, CURSORW, CURSORH);
}

/*
 *	Routine to define global cursor, removes old global cursor
 *	from screen if it is in this window and repalces it on
 *	the screen with the new one
 */
DefineGlobalCursor(window, mask, face, xref, yref, blink)
register struct window *window;
register word *mask, *face;
register int xref, yref;
int blink;
{
    /* remove old cursor if needed */
    if (cursor.visible && (loadedcursor == window->gcursor)) {
	HideGlobalCursor();

	/* read in new cursor parameters */
	DefineCursor(window->gcursor, mask, face, xref, yref, blink);
	window->gcursor->blanked = cursor.blanked;
	window->gcursor->bounds.x = cursor.bounds.x;
	window->gcursor->bounds.y = cursor.bounds.y;
	LoadGlobalCursor(window->gcursor);

	/* redisplay cursor */
	ShowGlobalCursor();
    } else		/* read in new cursor parameters */
	DefineCursor(window->gcursor, mask, face, xref, yref, blink);
}

/*
 *	Routine to define local cursor for given window
 */
DefineLocalCursor(window, mask, face, xref, yref, blink)
register struct window *window;
register word *mask, *face;
register byte xref, yref;
register byte blink;
{
    /* if cursor is visible remove it, redfine it, and redisplay it */
    if (window->cursor->visible) {
	HideLocalCursor(window);

	/* read in new cursor parameters */
	DefineCursor(window->cursor, mask, face, xref, yref, blink);

	/* redisplay cursor */
	ShowLocalCursor(window);
    } else		/* read in new cursor parameters */
	DefineCursor(window->cursor, mask, face, xref, yref, blink);
}


/*
 *	Routine to remove all cursors from screen within given window
 * 	which intersect given rectangle and save background.  Note that
 *	bounds are already in screen coordinates upon entry.  This call is
 *	used in conjunction with DropCursors and can be nested
 */
LiftCursors(window, bounds)
register struct window    * window;
register struct rectangle * bounds;
{

    unsigned char IntersectBox(), boxflag;
    register bool vflag = FALSE;	/* flag for cursor visiblity */
    struct rectangle tbounds;
    register struct tile * tile, * tiles;

    /* check if box cursor needs to be removed */
    if (box.visible) {
	if (boxflag = IntersectBox(*bounds))
	    RestoreBoxBackground(boxflag);
	box.blanked++;
    }

    /* check if global cursor needs to be removed */
    if (cursor.visible) {
	if (Overlap(cursor.bounds,*bounds)) {
	    Intersection(&tbounds, *bounds, cursor.bounds);
	    CopyRaster(&cursor.back, tbounds.x - cursor.bounds.x,
	     tbounds.y - cursor.bounds.y, &screen, 
	     tbounds.x, tbounds.y, tbounds.w, tbounds.h);
	}
	cursor.blanked++;
    }

    /* check if local cursor is in the way */
    if (window->cursor->visible) { 
	if (Overlap(window->cursor->bounds,*bounds)) {
	    Intersection(&tbounds, *bounds, window->cursor->bounds);

	    tiles = window->itiles;
	    /* loop thru window tiles to find visbile portion of cursor */
	    for (tile = tiles->next; tile != tiles; tile = tile->next) {
		if (Overlap(tbounds, tile->bounds))  {
		    Intersection(&tbounds, tbounds, tile->bounds);
		    vflag = TRUE;
		}
	    }
	    if (vflag) CopyRaster(&window->cursor->back, tbounds.x - 
		    	window->cursor->bounds.x, tbounds.y - 
		    	window->cursor->bounds.y, &screen, 
		    	tbounds.x, tbounds.y, tbounds.w, tbounds.h);
	} 
	window->cursor->blanked++;
    }
}

/*
 *	Routine to restore all cursors blanked to screen within the given
 *	bounds
 */
DropCursors(window, bounds)
register struct window    * window;
register struct rectangle * bounds;
{
    
    struct rectangle tbounds;
    register struct tile * tile, * tiles;
    register bool vflag = FALSE;		/* local cursor visiblity */

    /* do nothing with local if not blanked */
    if (window->cursor->blanked) {
	if (Overlap(window->cursor->bounds,*bounds)) {
	    /* find portion to restore */
	    Intersection(&tbounds, *bounds, window->cursor->bounds);
	    tiles = window->itiles;
	    /* loop thru window tiles to find visbile portion of cursor */
	    for (tile = tiles->next; tile != tiles; tile = tile->next) {
		if (Overlap(tbounds, tile->bounds)) {
		    Intersection(&tbounds, tbounds, tile->bounds);
		    vflag = TRUE;
		}
	    }
	    /* repaint if at all visible */
	    if (vflag) {
		CopyRaster(&screen, tbounds.x, tbounds.y, &window->cursor->back,
		    tbounds.x - window->cursor->bounds.x, tbounds.y - 
		    window->cursor->bounds.y, tbounds.w, tbounds.h);
		CursorToRaster(&screen, window->cursor, tbounds);
	    }
	}
	window->cursor->blanked--;
    }
    
    /* do nothing with global if not blanked */
    if (cursor.blanked) {
	/* find porition to restore */
	if (Overlap(*bounds, cursor.bounds)) {
	    Intersection(&tbounds, *bounds, cursor.bounds);
	    CopyRaster(&screen, tbounds.x, tbounds.y, &cursor.back, 
		tbounds.x - cursor.bounds.x, tbounds.y - cursor.bounds.y, 
		tbounds.w, tbounds.h);
	    CursorToRaster(&screen, &cursor, tbounds);
	}
	cursor.blanked--;
    }

    /* do nothing with box if not blanked */
    if (box.blanked) {
	/* restore only on last call */
	if (--box.blanked == 0) {
	    SaveBoxBackground(box.flag);
	    DrawBox(box.flag);
	}
    }
}

/*
 *	Routine to allocate a raster structure of given width and height,
 *	used only in this module.
 */
CreateRaster(raster,w, h)
register RASTER *raster;
register w, h;
{
    raster->width = (w+15)/16*2;	/* set width and get space for it */
    raster->address = (word *)CreateBlock(h*raster->width);
}


/*
 *	Routine to return a given raster to the free pool - used in this
 *	module only
 */
DestroyRaster(raster)
RASTER raster;
{
    DestroyBlock(raster.address);
}

/*
 *	Routine to save the backgroundf behind the box cursor in the save
 *	area for the box cursor depending on the value of flag.
 */
SaveBoxBackground(flag)
register int flag;
{
    register int tempx, tempw, plane;

    /*  save top and bottom of frame if needed */
    if (flag & BLANKT) {
	CopyRaster(&screen, box.bounds.x, box.bounds.y,
		   &box.horz, 0, 0, box.bounds.w, (BT-1));
        if (gptype & GPCOLOR) {
	    CopyRaster(&screen1, box.bounds.x, box.bounds.y,
		   &box.horz, 0, 2*(BT-1), box.bounds.w, (BT-1));
	    CopyRaster(&screen2, box.bounds.x, box.bounds.y,
		   &box.horz, 0, 4*(BT-1), box.bounds.w, (BT-1));
	    CopyRaster(&screen3, box.bounds.x, box.bounds.y,
		   &box.horz, 0, 6*(BT-1), box.bounds.w, (BT-1));
        }
    }
    if (flag & BLANKB) {
	CopyRaster(&screen, box.bounds.x, box.bounds.y + box.bounds.h - BT+1,
		   &box.horz, 0, (BT-1), box.bounds.w, (BT-1));
        if (gptype & GPCOLOR) {
	    CopyRaster(&screen1, box.bounds.x, box.bounds.y+box.bounds.h - BT+1,
		   &box.horz, 0, 3*(BT-1), box.bounds.w, (BT-1));
	    CopyRaster(&screen2, box.bounds.x, box.bounds.y+box.bounds.h - BT+1,
		   &box.horz, 0, 5*(BT-1), box.bounds.w, (BT-1));
	    CopyRaster(&screen3, box.bounds.x, box.bounds.y+box.bounds.h - BT+1,
		   &box.horz, 0, 7*(BT-1), box.bounds.w, (BT-1));
        }
    }

    /* save left and right side of frame if needed ,
     * align x on a word boundry if possible for speed, else go ahead */

    if (flag & BLANKL) {
	tempx = box.bounds.x;
	tempw = BT;
	if ((tempx & 0xf) != 0xf) {
	    tempx &= ~0xf;
	    tempw = 16;
	}
	CopyRaster(&screen, tempx, box.bounds.y + BT - 1,
		   &box.lvert, 0, 0, tempw, box.bounds.h - ((BT-1) << 1));
        if (gptype & GPCOLOR) {
	    CopyRaster(&screen1, tempx, box.bounds.y + BT - 1,
		   &box.lvert, tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
	    CopyRaster(&screen2, tempx, box.bounds.y + BT - 1,
		   &box.lvert, 2*tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
	    CopyRaster(&screen3, tempx, box.bounds.y + BT - 1,
		   &box.lvert, 3*tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
        }
    }

    if (flag & BLANKR) {
	tempx = box.bounds.w + box.bounds.x - (BT - 1);
	tempw = BT;
	if ((tempx & 0xf) != 0xf) {
	    tempx &= ~0xf;
	    tempw = 16;
	}
	CopyRaster(&screen, tempx, box.bounds.y + BT - 1,
		   &box.rvert, 0, 0, tempw, box.bounds.h - ((BT-1) << 1));
	if (gptype & GPCOLOR) {
	    CopyRaster(&screen1, tempx, box.bounds.y + BT - 1,
		   &box.rvert, tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
	    CopyRaster(&screen2, tempx, box.bounds.y + BT - 1,
		   &box.rvert, 2*tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
	    CopyRaster(&screen3, tempx, box.bounds.y + BT - 1,
		   &box.rvert, 3*tempw, 0, tempw, box.bounds.h - ((BT-1) << 1));
        }
    }
}


/*
 *	Routine to restore the background under the rubber baby bandy box
 *	flag gives the sides to be saved.
 */
RestoreBoxBackground(flag)
register int flag;
{
    register int tempx, tempw;

    box.flag |= flag;
    /* copy top and bottom back to screen if needed */
    if (flag & BLANKT) {
	CopyRaster(&box.horz, 0, 0,
		   &screen, box.bounds.x, box.bounds.y, box.bounds.w, (BT-1));
        if (gptype & GPCOLOR) {
	    CopyRaster(&box.horz, 0, 2*(BT-1),
		   &screen1, box.bounds.x, box.bounds.y, box.bounds.w, (BT-1));
	    CopyRaster(&box.horz, 0, 4*(BT-1),
		   &screen2, box.bounds.x, box.bounds.y, box.bounds.w, (BT-1));
	    CopyRaster(&box.horz, 0, 6*(BT-1),
		   &screen3, box.bounds.x, box.bounds.y, box.bounds.w, (BT-1));
        }
    }
    if (flag & BLANKB) {
	CopyRaster(&box.horz, 0, (BT-1), &screen, box.bounds.x,
	    box.bounds.y + box.bounds.h - BT + 1, box.bounds.w, (BT-1));
        if (gptype & GPCOLOR) {
	    CopyRaster(&box.horz, 0, 3*(BT-1), &screen1, box.bounds.x,
		box.bounds.y + box.bounds.h - BT + 1, box.bounds.w, (BT-1));
	    CopyRaster(&box.horz, 0, 5*(BT-1), &screen2, box.bounds.x,
		box.bounds.y + box.bounds.h - BT + 1, box.bounds.w, (BT-1));
	    CopyRaster(&box.horz, 0, 7*(BT-1), &screen3, box.bounds.x,
		box.bounds.y + box.bounds.h - BT + 1, box.bounds.w, (BT-1));
	}
    }

    /* restore left and right side of frame if needed 
     * align x on a word boundry if possible for speed, else go ahead */
    if (flag & BLANKL)  {
	tempx = box.bounds.x;
	tempw = BT;
	if ((tempx & 0xf) != 0xf) {
	    tempx &= ~0xf;
	    tempw = 16;
	};
	CopyRaster(&box.lvert, 0, 0,
		   &screen, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
        if (gptype & GPCOLOR) {
	    CopyRaster(&box.lvert, tempw, 0,
		   &screen1, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
	    CopyRaster(&box.lvert, 2*tempw, 0,
		   &screen2, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
	    CopyRaster(&box.lvert, 3*tempw, 0,
		   &screen3, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
        }
    }

    if (flag & BLANKR)  {
	tempx = box.bounds.w + box.bounds.x - (BT - 1);
	tempw = BT;
	if ((tempx & 0xf) != 0xf) {
	    tempx &= ~0xf;
	    tempw = 16;
	}
	CopyRaster(&box.rvert, 0, 0,
		   &screen, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
        if (gptype & GPCOLOR) {
	    CopyRaster(&box.rvert, tempw, 0,
		   &screen1, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
	    CopyRaster(&box.rvert, 2*tempw, 0,
		   &screen2, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
	    CopyRaster(&box.rvert, 3*tempw, 0,
		   &screen3, tempx, box.bounds.y + BT - 1, tempw, 
		   box.bounds.h - ((BT-1) << 1)); 
        }
    }

}


/*
 *	routine to draw the box cursor on the screen at its current
 *	location, flag tells which edges to draw.
 */
DrawBox(flag)
register int flag;
{
    box.flag &= (flag ^ 0xf);
    /* paint box pieces onto screen as needed */
    if (flag & BLANKT)
    	ClearRaster(&screen, box.bounds.x, box.bounds.y,
		    box.bounds.w, BT - 1, BLACK);
    if (flag & BLANKB)
    	ClearRaster(&screen, box.bounds.x, box.bounds.y + box.bounds.h - (BT-1),
		    box.bounds.w, BT - 1, BLACK);
    if (flag & BLANKL)
    	ClearRaster(&screen, box.bounds.x, box.bounds.y + BT - 1,
		    BT - 1, box.bounds.h - ((BT - 1) << 1), BLACK);
    if (flag & BLANKR)
    	ClearRaster(&screen, box.bounds.x + box.bounds.w - (BT-1),
		box.bounds.y + BT - 1, BT - 1, box.bounds.h - ((BT-1) << 1), 
		BLACK);

}

/*
 *	Routine to display the global rubber baby bandy box on the screen at
 *	the given position x,y with size w,h - t is ignored
 */
DisplayGlobalRubberBox(x, y, w, h)
register int x, y, w, h;
{
    /* exit if box not on screen or too small */
    if( w < 0){
 	w = -w; x -= w;
    }
    if( h < 0){
 	h = -h; y -= h;
    }
    if ((x < 0) || (y < 0) || (x+w > SCREENW) || (y+h > SCREENH) ||
	(w < ((BT-1)<<1))  || (h < ((BT-1)<<1))) return;

    /* if the box is already visible then remove it and restore background */
    if (box.visible) {
	box.visible = FALSE;

	/* restore the old background behind the box, i.e. erase old box */
	RestoreBoxBackground(BLANKT|BLANKB|BLANKR|BLANKL);

    } else { 			/* other wise get space for moving it */
      if (gptype & GPCOLOR) {
	CreateRaster(&box.horz, SCREENW, 4*(3*(BT-1)+2*RT));
	CreateRaster(&box.lvert, 4*16, SCREENH);
	CreateRaster(&box.rvert, 4*16, SCREENH);
      } else {
	CreateRaster(&box.horz, SCREENW, 3*(BT-1)+2*RT);
	CreateRaster(&box.lvert, 16, SCREENH);
	CreateRaster(&box.rvert, 16, SCREENH);
      }
    }

    /* set position of box and hide it for now */

    box.bounds.x = x;
    box.bounds.y = y;
    box.bounds.w = w;
    box.bounds.h = h;

    /* save the background under the current box and draw it */
    SaveBoxBackground(BLANKT|BLANKB|BLANKR|BLANKL);
    DrawBox(BLANKT|BLANKB|BLANKR|BLANKL);

    /* set visibility */
    box.visible = TRUE;
}

/*
 *	Routine to remove the rubber baby bandy box from existance
 */
RemoveGlobalRubberBox()
{
    /* return if box not currently on screen, otherwise do it to it */
    if (box.visible) {

	box.visible = FALSE;		/* set to avoid circular calls */

	/* restore the old background behind the box */
	RestoreBoxBackground(BLANKT|BLANKB|BLANKR|BLANKL);

	/* remember the flags */
	box.visible = FALSE;
	box.blanked = 0;

	/* return the raster to freedom */
	DestroyRaster(box.horz);
	DestroyRaster(box.lvert);
	DestroyRaster(box.rvert);

    } 

}

/*
 *	Routine to determine which sides of the global rubber box
 *	intersect with the given rectanlge.
 */
unsigned char 
IntersectBox(bounds)
struct rectangle bounds;
{
    register unsigned char 	flag = FALSE;
    register struct rectangle 	*rp;
    struct 	    rectangle 	tbox;

    /* check if the box is visible */
    if (box.visible) {

	/* set pointer to rectangle */
	rp = &tbox;

	/* Compute a containing rectangle for each edge and pickup that part
 	 * of the rubber box if it overlaps given bounds */
	*rp = box.bounds;
	rp->h = BT-1;
	if (Overlap(bounds, *rp)) 
	    flag |= BLANKT;

	rp->y += (box.bounds.h - (BT-1));
	if (Overlap(bounds, *rp)) 
	    flag |= BLANKB;

	*rp = box.bounds;
	rp->h -= ((BT-1)<<1);
	rp->w = BT-1;
	if ((rp->x & 0xf) != 0xf) {
	    rp->x &= ~0xf;
	    rp->w = 16;
	};
	if (Overlap(bounds, *rp)) 
	    flag |= BLANKL;

	rp->x = box.bounds.w + box.bounds.x - (BT - 1);
	rp->w = BT-1;
	if ((rp->x & 0xf) != 0xf) {
	    rp->x &= ~0xf;
	    rp->w = 16;
	}
	if (Overlap(bounds, *rp)) 
	    flag |= BLANKR;

    }
    return(flag);
}
