/*
 *	These routines are used to query or modify any of the clipping bounds
 *	associated with a window.  The pcbounds are the permanant clipping
 *	bounds and the tcbounds are the temporary clipping bounds.  The
 *	former are used to divide a window into panes whereas the latter
 *	are used to access specific areas within an application pane.  Note
 *	that clipping bounds are stored in the window structure as absolute
 *	screen coordinatates, and thus must be converted to local coordinates
 *	before return.
 */

#include	<vt_hdrs.h>		    /* system wide defines */
#include	<rectangle.h>		    /* rectangle structure */
#include	<cursor.h>		    /* cursor structure */
#include	<window.h>		    /* window structure */

extern struct tile *RestrictTiles();	    /* routine to make new restricted
					     * tile list */

/*
 *	This routine will set both the permenent and temporary clipping
 *	bounds of the named window to a rectangle based at x,y of width w and
 *	height h.  It will also make a new display tile list for the window
 */
SetPermanentClipping(window, x, y, w, h)
register struct window *window;
int x, y, w, h;
{
    /* set the permanant bounds to screen coordinates */
    window->pcbounds.x = x + window->ibounds.x;
    window->pcbounds.y = y + window->ibounds.y;
    window->pcbounds.w = w;
    window->pcbounds.h = h;

    /* set the temporary bounds similarly */
    window->tcbounds = window->pcbounds;

    /* make new diplay tile list for window of full window tile list and
     * and new clipping bounds */
    DestroyTileList(window->tiles);
    window->tiles = RestrictTiles(window->itiles, window->tcbounds);
}

/*
 *	This routine will set the temporary clipping bounds for the window
 *	and make a new display tile list from the full tile list clipped
 *	by tcbounds
 */
SetTemporaryClipping(window, x, y, w, h)
register struct window *window;
int x, y, w, h;
{
    /* copy new rectangle to tcbounds */
    window->tcbounds.x = x + window->ibounds.x;
    window->tcbounds.y = y + window->ibounds.y;
    window->tcbounds.w = w;
    window->tcbounds.h = h;

    /* make a new display tile list from full tile list and tcbounds */
    DestroyTileList(window->tiles);
    window->tiles = RestrictTiles(window->itiles, window->tcbounds);
}


/*
 *	Routine to make a new pcbounds from old pcbounds intersected with the
 *	given rectangle and create a new display tile list
 */
RestrictPermanentClipping(window, x, y, w, h)
register struct window *window;
int x, y, w, h;
{
    struct rectangle bounds;

    /* get screen rectangle for given x,y,w,h */
    bounds.x = window->ibounds.x + x;
    bounds.y = window->ibounds.y + y;
    bounds.w = w;
    bounds.h = h;

    /* get new pcbounds as insersection of new bounds and old pcbounds */
    Intersection(&window->pcbounds, bounds, window->pcbounds);
    /* set tcbounds as well */
    window->tcbounds = window->pcbounds;

    /* make a new display tile list from full tile list & tcbounds */
    DestroyTileList(window->tiles);
    window->tiles = RestrictTiles(window->itiles, window->tcbounds);
}


/*
 *	Routine to make a new tcbounds from old tcbounds intersected with the
 *	given rectangle and create a new display tile list
 */
RestrictTemporaryClipping(window, x, y, w, h)
register struct window *window;
int x, y, w, h;
{
    struct rectangle bounds;

    /* get screen rectangle of given coordinates */
    bounds.x = window->ibounds.x + x;
    bounds.y = window->ibounds.y + y;
    bounds.w = w;
    bounds.h = h;

    /* get new tcbounds of old pcbounds and new rectangle */
    Intersection(&window->tcbounds, bounds, window->pcbounds);

    /* make new display tile list of full tile list and tcbounds */
    DestroyTileList(window->tiles);
    window->tiles = RestrictTiles(window->itiles, window->tcbounds);
}


/*
 *	Routine to return current permanant clipping bounds
 */
GetPermanentClipping(window, x, y, w, h)
register struct window *window;
short *x, *y, *w, *h;
{
    *x = window->pcbounds.x - window->ibounds.x;
    *y = window->pcbounds.y - window->ibounds.y;
    *w = window->pcbounds.w;
    *h = window->pcbounds.h;
}

/*
 *	Routine to get current temporary clipping bounds
 */
GetTemporaryClipping(window, x, y, w, h)
register struct window *window;
short *x, *y, *w, *h;
{
    *x = window->tcbounds.x - window->ibounds.x;
    *y = window->tcbounds.y - window->ibounds.y;
    *w = window->tcbounds.w;
    *h = window->tcbounds.h;
}
