/*
 *	Keyboard driver for the MICRO SWITCH 83ST13-5 
 *		(vt100 style keyboard)
 *	
 *	Requires jumper J10 be installed
 *
 */

#include "param.h"		/* system parameters */
#include "types.h"		/* system parameters */
#include "systm.h"
#include "conf.h"
#include "user.h"
#include "proc.h"
#include "../vt/vt_hdrs.h"	/* virtual terminal parameters */
#include "../vt/vt_output.h"
#include "../vt/vt_kernel.h"	/* kernel only vt defines */

extern struct trans_tab		gws_table;
extern struct padtrans_tab 	fpad, npad;
extern int	   		mousestate;

#define  CURMOD		1
#define  NUMERIC 	2
#define  CLICK		4
#define  SENTSTP	8

struct kbuffer	{
	struct tty	*vp;
	short	mode;
	short	mmask;
	short	buttons;
};

#ifdef	SYSV
#define sendsimple(vp, c) \
		if (vp->t_universe == UNIVERSE_SYSV) \
			(*sv_linesw[vp->svt_line].l_input)(c, vp, 0); \
		else \
			((*linesw[vp->t_line].l_rint)(c, vp))
#else	SYSV
#define sendsimple(vp, c)	((*linesw[vp->t_line].l_rint)(c, vp))
#endif	SYSV

char	*
kb_init(tty)
    register struct tty	*tty;
{
    register struct kbuffer	*kb;	/* pointer to keyboard buffer */

    kb = (struct kbuffer *)CreateBlock(sizeof(struct kbuffer));
    if (kb) {
	kb->vp = tty;
	kb->mode = NUMERIC;
	kb->mmask = 0;
	kb->buttons = kb->vp->v_mmask = 
				VT_MOUSE_RIGHT|VT_MOUSE_LEFT|VT_MOUSE_MIDDLE;
    }
    return((char *)kb);
}

kb_control(kb, f, data)
    register struct kbuffer	*kb;
    register int 		data;
{
    if (kb)
	switch (f) {
	    case KGMASK:
		return(kb->mmask);
		break;

	    case KMMASK:
		kb->mmask = data;
		kb->vp->v_mmask = VT_MOUSE_RIGHT|VT_MOUSE_LEFT|VT_MOUSE_MIDDLE;
		if (data&VT_MOUSE_CONTINUOUS)
			kb->vp->v_mmask |= VT_MOUSE_NOBUTTON;
		break;

	    case KSETLED:
		break;

	    case KNUMERIC:
		if (data)
		    kb->mode |= NUMERIC;
		else
		    kb->mode &= ~NUMERIC;
		break;

	    case KCURMOD:
		if (data)
		    kb->mode |= CURMOD;
		else
		    kb->mode &= ~CURMOD;
		break;

	    case KBELL:
#ifdef	REAL
		if (kb->mode & click)
		    cprintf("%c", 2);
		else
		    cprintf("%c", 8|2);
#else	REAL
		cprintf("%c", VT_BEL);
#endif	REAL
		break;

	    case KCLICK:
		if (data)
		    kb->mode |= CLICK;
		else
		    kb->mode &= ~CLICK;
		break;

	    default:
		return(-1);
	}
    return(0);
}

kb_exit(kb)
    register struct kbuffer	*kb;	/* pointer to keyboard buffer */
{
    if (kb)
	DestroyBlock(kb);
}

kb_keyin(kb,c)
    register struct kbuffer	*kb;	/* pointer to keyboard buffer */
    register unsigned char	c;	/* incoming character */
{
    register struct tty		*vp;   	/* pointer to active vt */
    register unsigned char	ctype;  /* translated character code */

    vp = kb->vp;
    if ( c & 0x80) {
	c &= 0x7f;			/* and out high bit for function keys */
	ctype = gws_table.val[c].c1;/* get low byte for key   */
	c    = gws_table.val[c].c2;	/* get high byte for key  */
    } else
	ctype = VT_INPUT_SIMPLE;
	    
decode_table:
    switch (ctype) {
	case VT_INPUT_SIMPLE:	/* Simple input key */
	    sendsimple(vp, c);
	    if (c == VT_DC1) 	/* REAL vt100's do this! */
		kb->mode &= ~SENTSTP;
	    else if (c == VT_DC3)
		kb->mode |= SENTSTP;
	    break;

	case VT_ARROWKEYS:	/* for silly cursor keys mode */
	    sendsimple(vp, VT_ESC);
	    if ((kb->mode & CURMOD)&& !(kb->mode & NUMERIC))
		sendsimple(vp, VT_SHIFTED);
	    else
		sendsimple(vp, VT_NORMAL);
	    sendsimple(vp, c);
	    break;

	case VT_NORMAL:	/* Hard input key */
	case VT_SHIFTED:
	case VT_CONTROL:
	case VT_SHFTCTRL:
	    if (c<NPADKEYS ) {
		if (c == VT_PADNOSCRL) {
		    ctype = VT_INPUT_SIMPLE;
		    if (kb->mode & SENTSTP) {
			kb->mode &= ~SENTSTP;
			c = VT_DC1;	/* control-Q */
		    } else {
			kb->mode |= SENTSTP;
			c = VT_DC3;	/* control-S */
		    }
		} else if (kb->mode & NUMERIC) {
		    /* use the numeric translation tables */
		    ctype = npad.val[c].c1;
		    c = npad.val[c].c2;
		} else {
		    /* use function translation tables */
		    ctype = fpad.val[c].c1;
		    c = fpad.val[c].c2;
		}
		goto decode_table;
	    } else {
		sendsimple(vp, VT_ESC);
		sendsimple(vp, ctype);
		sendsimple(vp, c);
	    }
	    break;

	case BADCODE:	/* Shouldn't really happen */
	    break;

	default:
	    panic("Key mapping table bad");
    }
}

kb_mousein(kb, buttons, pane, x, y)
    register struct kbuffer	*kb;
    register short		buttons;
    register short		pane, x, y;
{
    register short change;

#ifdef notdef
    /* 
     * this sounded like a good idea, but it is a problem when pointing to a 
     * non-active window.  The window gets the upstroke, and tries to simulate 
     * the downstroke, causing a mark to be set in emulator windows.  too bad.
     */
    /* if something has happened outside our window, let's catch up! */
    if (((kb->buttons&7) != (mousestate&7)) && 	/* old state is different */
	((buttons&7) != (mousestate&7)) &&	/* this state isn't old state */
	((kb->buttons&7)!=(buttons&7)))		/* and isn't the new state */
	kb_mousein(kb, mousestate, pane, x, y);
#endif notdef

    /* if something has happened outside our window, let's catch up! */
    if (((kb->buttons&7) != (mousestate&7))&& ((buttons&7) != (mousestate&7)))
	kb->buttons = mousestate;
	
    if (buttons & VT_MOUSE_NOBUTTON) {
	kbsendmouse(kb,VT_MOUSE_NOBUTTON|VT_MOUSE_CONTINUOUS, pane, x, y);
	return;
    }

    /*
     * Determine which buttons have changed since last report and
     * send the necessary escape sequences
     */
    change = kb->buttons ^ buttons;

    if (change & VT_MOUSE_LEFT)
	if (buttons & VT_MOUSE_LEFT) {	/* left mouse button gone up */
	    /* if its not already up, then send it on */
	    if (!(kb->buttons & VT_MOUSE_LEFT)) {
		kbsendmouse(kb,VT_MOUSE_LEFT|VT_MOUSE_UP,pane,x,y);
		kb->buttons |= VT_MOUSE_LEFT;
	    }
	} else { 			/* left mouse button gone down */
	    if ((kb->buttons&7) != 7)
		kbsendmouse(kb,(kb->buttons^7)|VT_MOUSE_UP, pane, x, y);
	    kb->buttons = 7 & ~VT_MOUSE_LEFT;
	    kbsendmouse(kb,VT_MOUSE_LEFT | VT_MOUSE_DOWN, pane, x, y);
	}

    if (change & VT_MOUSE_MIDDLE)
	if (buttons & VT_MOUSE_MIDDLE) {	/* middle mouse key released */
	    /* if its not already up, then send it on */
	    if (!(kb->buttons & VT_MOUSE_MIDDLE)) {
		kbsendmouse(kb,VT_MOUSE_MIDDLE|VT_MOUSE_UP, pane,x,y);
		kb->buttons |= VT_MOUSE_MIDDLE;
	    }
	} else { 				/* middle mouse key pressed */
	    if ((kb->buttons&7) != 7)
		    kbsendmouse(kb,(kb->buttons^7)|VT_MOUSE_UP, pane, x, y);
	    kb->buttons = 7 & ~VT_MOUSE_MIDDLE;
	    kbsendmouse(kb,VT_MOUSE_MIDDLE | VT_MOUSE_DOWN, pane, x, y);
	}
    
    if (change & VT_MOUSE_RIGHT)
	if (buttons & VT_MOUSE_RIGHT) { 	/* right mouse key released */
	    /* if its not already up, then send it on */
	    if (!(kb->buttons & VT_MOUSE_RIGHT)) {
		kbsendmouse(kb,VT_MOUSE_RIGHT|VT_MOUSE_UP, pane,x,y);
		kb->buttons |= VT_MOUSE_RIGHT;
	    }
	} else { 				/* right mouse key pressed */
	    if ((kb->buttons&7) != 7)
		kbsendmouse(kb,(kb->buttons^7)|VT_MOUSE_UP, pane, x, y);
	    kb->buttons = 7 & ~VT_MOUSE_RIGHT;
	    kbsendmouse(kb,VT_MOUSE_RIGHT | VT_MOUSE_DOWN, pane, x, y);
	}
}

#define MM	7	/* maximum mouse ahead in cont. mode */

kbsendmouse(kb, buttons, pane, x, y)
    register struct kbuffer	*kb;
    register short		buttons, pane, x, y;
{
    if (kb->mmask & buttons) {
	if ((buttons&VT_MOUSE_CONTINUOUS)&&(kb->vp->t_rawq.c_cc > MM))
		return;
	sendsimple(kb->vp, VT_MOUSE);
	sendsimple(kb->vp, buttons /* & ~VT_MOUSE_CONTINUOUS */);
	sendsimple(kb->vp, (pane>>8) & 0xff);
	sendsimple(kb->vp, pane & 0xff);
	sendsimple(kb->vp, (x>>8) & 0xff);
	sendsimple(kb->vp, x & 0xff);
	sendsimple(kb->vp, (y>>8) & 0xff);
	sendsimple(kb->vp, y & 0xff);
    } else if ((pane&0xff) == DISPLAY)
	mark_request(kb->vp, buttons, x, y);
}

struct  trans_tab gws_table = {   /* Creator vt100 keyboard mappings */
	VT_INPUT_SIMPLE,VT_NUL,		/* 00, ascii null */
	VT_INPUT_SIMPLE,VT_SOH,		/* 01, control-shift-a */
	VT_INPUT_SIMPLE,VT_STX,		/* 02, control-shift-b */
	VT_INPUT_SIMPLE,VT_ETX,		/* 03, control-shift-c */
	VT_INPUT_SIMPLE,VT_EOT,		/* 04, control-shift-d */
	VT_INPUT_SIMPLE,VT_ENQ,		/* 05, control-shift-e */
	VT_INPUT_SIMPLE,VT_ACK,		/* 06, control-shift-f */
	VT_INPUT_SIMPLE,VT_BEL,		/* 07, control-shift-g */
	VT_INPUT_SIMPLE,VT_BS,		/* 08, control-shift-h */
	VT_INPUT_SIMPLE,VT_HT,		/* 09, control-shift-i */
	VT_INPUT_SIMPLE,VT_LF,		/* 0a, control-shift-j */
	VT_INPUT_SIMPLE,VT_VT,		/* 0b, control-shift-k */
	VT_INPUT_SIMPLE,VT_FF,		/* 0c, control-shift-l */
	VT_INPUT_SIMPLE,VT_CR,		/* 0d, control-shift-m */
	VT_INPUT_SIMPLE,VT_SO,		/* 0e, control-shift-n */
	VT_INPUT_SIMPLE,VT_SI,		/* 0f, control-shift-o */
	VT_INPUT_SIMPLE,VT_DLE,		/* 10, control-shift-p */
	VT_NORMAL,VT_PADNOSCRL,		/* 11, control-shift-q, and no-scrl */
	VT_INPUT_SIMPLE,VT_DC2,		/* 12, control-shift-r */
	VT_NORMAL,VT_PADNOSCRL,		/* 13, control-shift-s, and no-scrl */
	VT_INPUT_SIMPLE,VT_DC4,		/* 14, control-shift-t */
	VT_INPUT_SIMPLE,VT_NAK,		/* 15, control-shift-u */
	VT_INPUT_SIMPLE,VT_SYN,		/* 16, control-shift-v */
	VT_INPUT_SIMPLE,VT_ETB,		/* 17, control-shift-w */
	VT_INPUT_SIMPLE,VT_CAN,		/* 18, control-shift-x */
	VT_INPUT_SIMPLE,VT_EM,		/* 19, control-shift-y */
	VT_INPUT_SIMPLE,VT_SUB,		/* 1a, control-shift-z */
	VT_INPUT_SIMPLE,VT_ESC,		/* 1b, control-shift-[ */
	VT_INPUT_SIMPLE,VT_FS,		/* 1c, control-shift-\ */
	VT_INPUT_SIMPLE,VT_GS,		/* 1d, control-shift-] */
	VT_INPUT_SIMPLE,VT_RS,		/* 1e, control-shift-6 */
	VT_INPUT_SIMPLE,VT_US,		/* 1f, control-shift-_ */
	BADCODE,VT_NUL,		/* 20, ascii null */
	BADCODE,VT_NUL,		/* 21, ascii null */
	BADCODE,VT_NUL,		/* 22, ascii null */
	BADCODE,VT_NUL,		/* 23, ascii null */
	BADCODE,BADCODE,		/* 24, setup key */
	BADCODE,BADCODE,		/* 25, setup key */
	BADCODE,BADCODE,		/* 26, setup key */
	BADCODE,BADCODE,		/* 27, setup key */
	BADCODE,VT_NUL,		/* 28, ascii null */
	BADCODE,VT_NUL,		/* 29, ascii null */
	BADCODE,VT_NUL,		/* 2a, ascii null */
	BADCODE,VT_NUL,		/* 2b, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 2c, break key */
	VT_INPUT_SIMPLE,VT_NUL,		/* 2d, break key */
	VT_INPUT_SIMPLE,VT_NUL,		/* 2e, break key */
	VT_INPUT_SIMPLE,VT_NUL,		/* 2f, break key */
	BADCODE,VT_NUL,		/* 30, ascii null */
	BADCODE,VT_NUL,		/* 31, ascii null */
	BADCODE,VT_NUL,		/* 32, ascii null */
	BADCODE,VT_NUL,		/* 33, ascii null */
	BADCODE,VT_NUL,		/* 34, ascii null */
	BADCODE,VT_NUL,		/* 35, ascii null */
	BADCODE,VT_NUL,		/* 36, ascii null */
	BADCODE,VT_NUL,		/* 37, ascii null */
	BADCODE,VT_NUL,		/* 38, ascii null */
	BADCODE,VT_NUL,		/* 39, ascii null */
	BADCODE,VT_NUL,		/* 3a, ascii null */
	BADCODE,VT_NUL,		/* 3b, ascii null */
	BADCODE,VT_NUL,		/* 3c, ascii null */
	BADCODE,VT_NUL,		/* 3d, ascii null */
	BADCODE,VT_NUL,		/* 3e, ascii null */
	BADCODE,VT_NUL,		/* 3f, ascii null */
	BADCODE,VT_NUL,		/* 40, ascii null */
	VT_ARROWKEYS,VT_UPARROW,	/* 41, up arrow key */
	VT_ARROWKEYS,VT_DOWNARROW,	/* 42, down arrow key */
	VT_ARROWKEYS,VT_RIGHTARROW,	/* 43, right arrow key */
	VT_ARROWKEYS,VT_LEFTARROW,	/* 44, left arrow key */
	BADCODE,VT_NUL,		/* 45, ascii null */
	BADCODE,VT_NUL,		/* 46, ascii null */
	BADCODE,VT_NUL,		/* 47, ascii null */
	BADCODE,VT_NUL,		/* 48, ascii null */
	BADCODE,VT_NUL,		/* 49, ascii null */
	BADCODE,VT_NUL,		/* 4a, ascii null */
	BADCODE,VT_NUL,		/* 4b, ascii null */
	BADCODE,VT_NUL,		/* 4c, ascii null */
	VT_NORMAL,VT_PAD15,		/* 4d, numeric pad enter key */
	BADCODE,VT_NUL,		/* 4e, ascii null */
	BADCODE,VT_NUL,		/* 4f, ascii null */
	VT_NORMAL,VT_PAD0,		/* 50, numeric pad pf1 */
	VT_NORMAL,VT_PAD1,		/* 51, numeric pad pf2 */
	VT_NORMAL,VT_PAD2,		/* 52, numeric pad pf3 */
	VT_NORMAL,VT_PAD3,		/* 53, numeric pad pf4 */
	VT_INPUT_SIMPLE,VT_NUL,		/* 54, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 55, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 56, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 57, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 58, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 59, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5a, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5b, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5c, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5d, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5e, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 5f, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 60, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 61, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 62, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 63, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 64, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 65, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 66, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 67, ascii null */
	VT_INPUT_SIMPLE,VT_BS,		/* 68, control-shift-backspace */
	VT_INPUT_SIMPLE,VT_HT,		/* 69, control-shift-tab */
	VT_INPUT_SIMPLE,VT_LF,		/* 6a, control-shift-linefeed */
	VT_INPUT_SIMPLE,VT_NUL,		/* 6b, ascii null */
	VT_NORMAL,VT_PAD11,		/* 6c, numeric pad , */
	VT_NORMAL,VT_PAD7,		/* 6d, numeric pad - */
	VT_NORMAL,VT_PAD17,		/* 6e, numeric pad . */
	VT_INPUT_SIMPLE,VT_NUL,		/* 6f, ascii null */
	VT_NORMAL,VT_PAD16,		/* 70, numeric pad 0 key */
	VT_NORMAL,VT_PAD12,		/* 71, numeric pad 1 key */
	VT_NORMAL,VT_PAD13,		/* 72, numeric pad 2 key */
	VT_NORMAL,VT_PAD14,		/* 73, numeric pad 3 key */
	VT_NORMAL,VT_PAD8,		/* 74, numeric pad 4 key */
	VT_NORMAL,VT_PAD9,		/* 75, numeric pad 5 key */
	VT_NORMAL,VT_PAD10,		/* 76, numeric pad 6 key */
	VT_NORMAL,VT_PAD4,		/* 77, numeric pad 7 key */
	VT_NORMAL,VT_PAD5,		/* 78, numeric pad 8 key */
	VT_NORMAL,VT_PAD6,		/* 79, numeric pad 9 key */
	VT_INPUT_SIMPLE,VT_NUL,		/* 7a, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 7b, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 7c, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 7d, ascii null */
	VT_INPUT_SIMPLE,VT_NUL,		/* 7e, ascii null */
	VT_INPUT_SIMPLE,VT_DEL};	/* 7f, control-shift-del */

struct padtrans_tab npad = { /* numeric pad in function mode */
	VT_SHIFTED, VT_UPPER_P,		/* PAD0 */
	VT_SHIFTED, VT_UPPER_Q, 	/* PAD1 */
	VT_SHIFTED, VT_UPPER_R,		/* PAD2 */
	VT_SHIFTED, VT_UPPER_S,		/* PAD3 */
	VT_INPUT_SIMPLE, VT_SEVEN,	/* PAD4 */
	VT_INPUT_SIMPLE, VT_EIGHT,	/* PAD5 */
	VT_INPUT_SIMPLE, VT_NINE,	/* PAD6 */
	VT_INPUT_SIMPLE, VT_MINUS,	/* PAD7 */
	VT_INPUT_SIMPLE, VT_FOUR,	/* PAD8 */
	VT_INPUT_SIMPLE, VT_FIVE,	/* PAD9 */
	VT_INPUT_SIMPLE, VT_SIX,	/* PAD10 */
	VT_INPUT_SIMPLE, VT_COMMA,	/* PAD11 */
	VT_INPUT_SIMPLE, VT_ONE,	/* PAD12 */
	VT_INPUT_SIMPLE, VT_TWO,	/* PAD13 */
	VT_INPUT_SIMPLE, VT_THREE,	/* PAD14 */
	VT_INPUT_SIMPLE, VT_CR,		/* PAD15 */
	VT_INPUT_SIMPLE, VT_ZERO,	/* PAD16 */
	VT_INPUT_SIMPLE, VT_PERIOD};	/* PAD17 */

struct padtrans_tab fpad = { /* numeric pad in function mode */
	VT_SHIFTED, VT_UPPER_P,		/* PAD0 */
	VT_SHIFTED, VT_UPPER_Q, 	/* PAD1 */
	VT_SHIFTED, VT_UPPER_R,		/* PAD2 */
	VT_SHIFTED, VT_UPPER_S,		/* PAD3 */
	VT_SHIFTED, VT_LOWER_W,		/* PAD4 */
	VT_SHIFTED, VT_LOWER_X,		/* PAD5 */
	VT_SHIFTED, VT_LOWER_Y,		/* PAD6 */
	VT_SHIFTED, VT_LOWER_M,		/* PAD7 */
	VT_SHIFTED, VT_LOWER_T,		/* PAD8 */
	VT_SHIFTED, VT_LOWER_U,		/* PAD9 */
	VT_SHIFTED, VT_LOWER_V,		/* PAD10 */
	VT_SHIFTED, VT_LOWER_L,		/* PAD11 */
	VT_SHIFTED, VT_LOWER_Q,		/* PAD12 */
	VT_SHIFTED, VT_LOWER_R,		/* PAD13 */
	VT_SHIFTED, VT_LOWER_S,		/* PAD14 */
	VT_SHIFTED, VT_UPPER_M,		/* PAD15 */
	VT_SHIFTED, VT_LOWER_P,		/* PAD16 */
	VT_SHIFTED, VT_LOWER_N};	/* PAD17 */
