#define FONTS_LONGWORD
#include "param.h"		/* system parameters */
#include "types.h"		/* system types */
#include "systm.h"		/* system sturcts et al */
#include "user.h"		/* user block */
#include "file.h"		/* file structure */
#include "proc.h"		/* process strucutures */
#include "uio.h"		/* user input/output struct */
#include "kernel.h"		/* need lbolt and hz */
#undef MAIN
#include "../vt/vt_hdrs.h"
#include "../vt/vt_fonts.h"
/*#include "../vt/vt_types.h" /**/
#include "errno.h"

extern char vidbuf[];
#define LASTVA	((int)(vidbuf+offscreen_off+0x1000))	/* KLUDGE */

int	vt_fontssetup = 0;		/* FontInit only once */
int	vt_loadfonts = 0;		/* On if /dev/font open */
int	vt_font_table_interlock = 0;	/* Font table access interlock */
int	vt_plane_count = 1;		/* number of currently loaded planes */
					/* default font/plane is preloaded */

/* Request structure to pass to font daemon.  */
struct vt_daemon_request	vt_font_request;

/* Return code from latest font daemon request.  */
int	vt_daemon_return;

/* Index of location in descriptor array of loaded plane. */
int	vt_load_index;

/* Table of currently loaded planes */
struct vt_font_table	loaded_planes = {
	{DEFNONPROPFONT, DEFPLANE, 1, 0, 0, 0}	/* Default plane */
};

/* Character descriptors for planes */
struct vt_plane_descriptors	plane_desc;

/* List of free memory blocks for video memory storage management.  */
struct freelistentry		vt_vmemseg[MAXPLANES * 2 + 2] = {
	{0, 0},
	{0, 0},
	{PLANESPACE, 0}
};

/* Number of free video memory segments. */
int	vt_freevsegcnt = 3;	/* Two dummy free segs and the rest of vmem */

/*
 * This routine initializes font plane management data structures and variables
 * for the font management code.  It should be called during kernel 
 * initialization.  For efficiency it only initializes things that cannot be 
 * initialized statically.
 */
FontInit()
{
    register int	i;	/* Loop index */
    register int	rlen;	/* length of default plane rasters */
    register short	*p1;	/* pointer to video memory */
    register short	*p2;	/* pointer to rasters */

    if (vt_fontssetup)
	return;
    loaded_planes.ftbftplanes[0].ftbrloc = LASTVA;
    vt_vmemseg[0].startaddr += LASTVA;
    vt_vmemseg[2].startaddr += LASTVA;
#ifdef	FONTS_LONGWORD
    rlen = AdjustFont(plane_desc.pldchardescripts,
	 (char *)&plane_desc.pldchardescripts[1], (char *)LASTVA);
#else	FONTS_LONGWORD
    p1 = (short *)LASTVA;
    p2 = (short *)&plane_desc.pldchardescripts[1];
    rlen = plane_desc.pldchardescripts[0].rast_size;
    for (i = 0; i < (rlen/2); i++)
	*p1++ = *p2++;
#endif	FONTS_LONGWORD
    plane_desc.pldchardescripts[0].rast_ptr = (u_word *)LASTVA;
    loaded_planes.ftbftplanes[0].ftbrlen = rlen;
    loaded_planes.ftbftplanes[0].ftbinuse = 1;
    loaded_planes.ftbftplanes[0].ftblastuse = time.tv_sec;
    vt_vmemseg[1].startaddr = LASTVA + rlen;
    vt_vmemseg[1].length = PLANESPACE - rlen;
    for (i = 1; i < MAXPLANES; i++) {
	loaded_planes.ftbftplanes[i].ftbfontid = INVALIDFONT;
	loaded_planes.ftbftplanes[i].ftbinuse = 0;
    }
    vt_fontssetup = 1;
}


#ifdef FONTS_LONGWORD
/* 
 * Adjust the raster portion of a 16 bit word aligned font to a 32 bit aligned
 */
AdjustFont(fp, srp, drp)
    register struct font_plane *fp; /* font header pointer */
    char *srp;			/* src raster ptr (actually short) */
    char *drp;			/* dst raster ptr (actually long) */
{
    register int	i;
    static RASTER	sraster, draster;
    register int	rw, rh;
    register char	*cdrp = drp;

    CPUgetdisp();
    for (i = 0; i < NUM_PLANE_CHARS; i++) {
	rw = fp->ch_desc[i].rast_width; 		/* in pixels */
	rh = fp->ch_desc[i].rast_height; 		/* in pixels */
	sraster.width = (((rw + 15) >> 4) << 1); 	/* in bytes */
	sraster.address = (short *)(srp + fp->ch_desc[i].offset);
	draster.width = (((rw + 31) >> 5) << 2); 	/* in bytes */
	draster.address = (short *)cdrp;
	CopyRaster(&sraster, 0, 0, &draster, 0, 0, rw, rh);
	fp->ch_desc[i].offset = cdrp - drp;
	cdrp += (draster.width * rh);
    }
    return (cdrp - drp); 
}

/*
 * Return the size of font raster after adjustment for 32 bit word
 */
AdjustFontSize(fp)
    register struct font_plane *fp;
{
    register int	i;
    register int	rw, rh;
    int			width;
    register int	rlen;

    rlen = 0;
    for (i = 0; i < NUM_PLANE_CHARS; i++) {
	rw = fp->ch_desc[i].rast_width; 	/* in pixels */
	rh = fp->ch_desc[i].rast_height; 	/* in pixels */
	width = (((rw + 31) >> 5) << 2); 	/* in bytes */
	rlen += width * rh;
    }
    return(rlen);
}
#endif FONTS_LONGWORD

/*
 * This routine adds the space occupied by the rasters for a removed font plane
 * back to the free video memory list.
 */
FreeRasterSpace(startaddr, length)
    register int	startaddr;	/* Starting address of freed space */
    register int	length;		/* Length of freed space */
{
    register int	i, j;		/* Loop indexes */
    register int	prev;		/* Previous vmemseg entry */

    prev = 0;
    for (i = 1; i < vt_freevsegcnt; i++) {
	/* Skip if not proper slot. */
	if (startaddr < vt_vmemseg[prev].startaddr ||
	    startaddr > vt_vmemseg[i].startaddr) {
	    prev = i;
	    continue;
	}
	/* See if can combine with previous free slot.  */
	if (vt_vmemseg[prev].startaddr + vt_vmemseg[prev].length == startaddr) {
	    vt_vmemseg[prev].length += length;
	    /* 
	     * See if extended previous can be combined with 
	     * next free slot.
	     */
	    if (vt_vmemseg[prev].startaddr+vt_vmemseg[prev].length ==
		vt_vmemseg[i].startaddr) {
		vt_vmemseg[prev].length += vt_vmemseg[i].length;
		vt_freevsegcnt--;
		for (j = i; j < vt_freevsegcnt; j++)
		    vt_vmemseg[j] = vt_vmemseg[j + 1];
	    }
	    return;
	}

	/* See if can combine with next free slot. */
	if ((i < vt_freevsegcnt - 1) && 
	    (startaddr + length == vt_vmemseg[i].startaddr)) {
	    vt_vmemseg[i].startaddr = startaddr;
	    vt_vmemseg[i].length += length;
	    return;
	}

	/* Can't combine, make a new free slot.  */
	for (j = vt_freevsegcnt++; j > i; j--)
	    vt_vmemseg[j] = vt_vmemseg[j - 1];
	vt_vmemseg[i].startaddr = startaddr;
	vt_vmemseg[i].length = length;
	return;
    }
}

/*
 * This routine allocates space in video memory from the freelist
 * for the rasters of a newly requested font plane.  It will return
 * zero if enough space cannot be found, else it will return the
 * location to put the rasters.
 */
int
FindRasterSpace(length)
    register int	length;		/* No. of contiguous bytes needed */
{
    register int	i, j;		/* Loop indexes */
    register int	where = 0;	/* Where to write rasters */

    if (length == 0)
	panic("FindRasterSpace 0 length!\n");
    for (i = 0; i < vt_freevsegcnt; i++)
	if (length <= vt_vmemseg[i].length) {
	    where = vt_vmemseg[i].startaddr;
	    vt_vmemseg[i].startaddr += length;
	    vt_vmemseg[i].length -= length;
	    if (vt_vmemseg[i].length == 0) {
		for (j = i; j < vt_freevsegcnt; j++)
		    vt_vmemseg[j] = vt_vmemseg[j + 1];
		vt_freevsegcnt--;
	    }
	    break;
	}
    return where;
}

/*
 * This routine is called by anyone doing an ioctl to /dev/font
 */
fntioctl(dev, request, datap, flag)
    dev_t		dev;		/* device entry */
    register int	request;	/* command */
    caddr_t		datap;		/* pointer to ioctl data */
    int		flag;			/* ioctl flag */
{
    register int	returnval;	/* Guess what */
    register int	i;		/* Font/plane index */
    struct vt_inv_plane	*sp;		/* pointer to ioctl args */

    returnval = 0;

    switch (request) {
	case FDIOINVP:	/* Force font plane out of cache */
	    sp = (struct vt_inv_plane *)datap;
	    if (sp->fontid == DEFNONPROPFONT && sp->plane == DEFPLANE)
		return EBUSY;
	    GetFontTableInterlock(PZERO-1);
	    i = GetPlaneTableIndex(sp->fontid, sp->plane);
	    if (i < 0){
		returnval = ENOSPC;
		goto done;
	    }
	    if (loaded_planes.ftbftplanes[i].ftbinuse > 0){
		returnval = EBUSY;
		goto done;
	    }
	    RemovePlane(i);
    done:   vt_font_table_interlock = 0;
	    wakeup(&vt_font_table_interlock);
	    break;

	case FDIORDCT:	/* Have fontdaemon re-read font catalog */
	    GetFontTableInterlock(PZERO-1);
	    vt_font_request.reqhdr.sdrreqtype = REREADCAT;
	    vt_font_request.reqhdr.sdrlength = sizeof(struct font_request);

	    wakeup(&vt_font_request);
	    sleep(&vt_daemon_return, PZERO-1);

	    vt_font_table_interlock = 0;
	    wakeup(&vt_font_table_interlock);
	    break;

	default:	/* Invalid ioctl specified */
	    returnval = ENOTTY;
    }
    return( returnval);
}

/*
 * This routine is called by someone opening the font device.
 * It will set the flag that allows fonts other than the default
 * one to be used.
 */
int
fntopen(dev, flag)
{
    if (vt_loadfonts)
	return(ENXIO);			/* already open */
    else
	vt_loadfonts = 1;
    /* change our process group so we dont get unwanted signals */
    u.u_procp->p_pgrp = u.u_procp->p_pid;
    return (0);
}

/*
 * This routine is called by someone closing the font device.
 * It will reset the flag that allows fonts other than the default
 * one to be used.
 */
int
fntclose(dev, flag)
{
	return(vt_loadfonts = 0);
}

/*
 * This routine is the routine called by read after being looked
 * up in the cdevsw table.  It does the actual read into the users
 * buffer from kernel space.
 */
int
fntread(dev, uio)
    struct uio	*uio;		/* I/O parameters structure */
{
    int	error;

    wakeup(&vt_daemon_return);		/* wakeup the write */
    sleep(&vt_font_request, PZERO+1);	/* wait for the next request */

    error = (uiomove((caddr_t)&vt_font_request, 
		    sizeof(struct vt_daemon_request), UIO_READ, uio));
    return(error);
}

/*
 * This routine is the routine called by write after being looked 
 * up in the cdevsw table.  It does the actual write into kernel
 * space from the users buffer.
 */
int
fntwrite(dev, uio)
    struct uio	*uio;	/* I/O parameters structure */
{
    register int	error;	/* Error indication returned from uiomove */
    register int	i;	/* Loop index */
    register int	where;	/* Where in video memory to put rasters */
    register int	index;	/* Loaded planes table index */
    struct vt_daemon_reply_header	vt_font_reply_hdr; /* Daemon reply */
    register int	rsize;
    static char	no_space = 0;
#ifdef	FONTS_LONGWORD
    extern char	*CreateBlock();
    char		*blkp;
#endif	FONTS_LONGWORD

    error = uiomove((caddr_t)&vt_font_reply_hdr,
		    sizeof(struct vt_daemon_reply_header), UIO_WRITE, uio);
    if (error)
	return(error);
    vt_daemon_return = vt_font_reply_hdr.sdrreptype;

    switch (vt_daemon_return) {
	case NOSUCHPLANE:
	case NOSUCHFONT:
	case NOSPACE:
	case RASTTOOBIG:
	case CATREREAD:
	    break;

	case PLANECHARDATA:
	    for (i = 1; i < MAXPLANES; i++)
		if (loaded_planes.ftbftplanes[i].ftbfontid == INVALIDFONT)
		    break;
	    error = uiomove((caddr_t)&plane_desc.pldchardescripts[i], 
			    sizeof(struct font_plane), UIO_WRITE, uio);
	    if (error) {
		vt_daemon_return = NOSUCHPLANE;
		break;	
	    }
#ifdef FONTS_LONGWORD
	    rsize = AdjustFontSize(&plane_desc.pldchardescripts[i]);
#else	FONTS_LONGWORD
	    rsise = plane_desc.pldchardescripts[i].rast_size;
#endif	FONTS_LONGWORD

	    while ((where = FindRasterSpace(rsize)) == 0) {
		index = FindOldestUnusedPlane();
		if (index < 0) {
		    if (no_space == 0)
			printf("font: no raster space\n");
		    no_space++;
		    vt_daemon_return = NOSPACE;
		    return(ENOMEM);
		} else
		    RemovePlane(index);
	    }
#ifdef	FONTS_LONGWORD
	
	    if (HaveBlock(plane_desc.pldchardescripts[i].rast_size) == 0) {
		vt_daemon_return = NOSPACE;
		FreeRasterSpace(where, rsize);
		return(ENOMEM);
	    }
	    blkp = CreateBlock(plane_desc.pldchardescripts[i].rast_size);
	    error = uiomove(blkp, plane_desc.pldchardescripts[i].rast_size,
			    UIO_WRITE, uio);
#else	FONTS_LONGWORD
	    error = uiomove((caddr_t)where, 
		plane_desc.pldchardescripts[i].rast_size, UIO_WRITE, uio);
#endif	FONTS_LONGWORD

	    if (error) {
		vt_daemon_return = NOSUCHPLANE;
#ifdef	FONTS_LONGWORD
		DestroyBlock(blkp);
#endif	FONTS_LONGWORD
		FreeRasterSpace(where, rsize);
	    } else {
#ifdef	FONTS_LONGWORD
		AdjustFont(&plane_desc.pldchardescripts[i], blkp, where);
		DestroyBlock(blkp);	/* release internal buffer */
#endif	FONTS_LONGWORD
		vt_load_index = i;
		loaded_planes.ftbftplanes[i].ftbrloc = where;
		loaded_planes.ftbftplanes[i].ftbrlen = rsize;
		plane_desc.pldchardescripts[i].rast_ptr = (u_word *)where;
	    }
	    break;

	default:
	    break;
    }
    return error;
}

/*
 * This routine gets the font table interlock for the caller.  When the caller
 * is done with the interlock he should set the interlock variable to zero
 * and do a wakeup on the address of the interlock.  While this interlock
 * is set no other PaintString to the screen can proceed as they will all
 * hang on the interlock.  This is necessary because we may be shuffling
 * the font tables around.  We still have a problem in that we cannot shuffle
 * rasters around because someone may already have a pointer to them.
 * The only time we can shuffle rasters is if all planes except the default
 * have a zero use count.
 */
GetFontTableInterlock(pri)
{
    register int	s = spltty();

    while (vt_font_table_interlock)
	sleep(&vt_font_table_interlock, pri);
    vt_font_table_interlock = 1;
    splx(s);
}

/*
 * This routine returns the index in the loaded planes table of the
 * oldest unused plane in the table whose rasters are still in video
 * memory, if there are no unused planes then -1 will be returned.
 */
int
FindOldestUnusedPlane()
{
    register int	i;	/* Font/plane index */
    register int	index;	/* Index of plane to remove */
    register long	oldest;	/* Time of oldest entry */

    index = -1;
    oldest = MAXINT;
    for (i = 1; i < MAXPLANES; i++)
	if (loaded_planes.ftbftplanes[i].ftbfontid != INVALIDFONT &&
	    loaded_planes.ftbftplanes[i].ftbinuse == 0 &&
	    loaded_planes.ftbftplanes[i].ftblastuse <= oldest) {
		oldest = loaded_planes.ftbftplanes[i].ftblastuse;
		index = i;
	}
    return index;
}

/*
 * This routine returns the index in the loaded planes table of the 
 * requested font and plane.  If the font/plane is not in the table
 * then a -1 is returned. 
 */
int
GetPlaneTableIndex(fontid, plane)
    register unsigned short fontid;	/* Font ID number */
    register char		plane;	/* Plane number */
{
    register int	i;		/* Loop index */
    register int	index;		/* Font/plane index */

    index = -1;
    for (i = 0; i < MAXPLANES; i++)
	if ( loaded_planes.ftbftplanes[i].ftbfontid == fontid && 
	     loaded_planes.ftbftplanes[i].ftbplane == plane) {
		index = i;
		break;
	}
    return index;
}

/*
 * This routine removes the specified font plane from the table of currently
 * loaded font planes.  This is done by marking the font ID as invalid and
 * freeing up the space occupied by the planes rasters in video memory.
 */
RemovePlane(index)
    register int	index;		/* Table index of plane to remove */
{
    FreeRasterSpace(loaded_planes.ftbftplanes[index].ftbrloc,
	    loaded_planes.ftbftplanes[index].ftbrlen);
    vt_plane_count--;
    loaded_planes.ftbftplanes[index].ftbfontid = INVALIDFONT;
}

/*
 * This routine calls the storage daemon to load a font plane.
 * The daemon will return the font plane character descriptors
 * and the rasters for the plane.  The mechanism used by the
 * daemon to communicate with the kernel is reads and writes
 * to /dev/font.  The index of the loaded plane is returned.
 */
int
LoadFontPlane(fontid, plane)
    register unsigned short	fontid;	/* Font ID number */
    register char		plane;	/* Plane number */
{
    register int	index;		/* Index of plane to remove */
    static char		no_space = 0;

    if (vt_plane_count == MAXPLANES) {
	index = FindOldestUnusedPlane();
	if (index < 0) {
	    if (no_space == 0)
		printf("font: no planes\n");
	    no_space++;
	    return 0;
	}
	RemovePlane(index);
    }
    vt_font_request.reqhdr.sdrreqtype = LOADFONT;
    vt_font_request.reqhdr.sdrlength = sizeof(struct font_request);
    vt_font_request.font_reqdata.frfontid = fontid;
    vt_font_request.font_reqdata.frplane = plane;

    wakeup(&vt_font_request);			/* wakeup the read */
    sleep(&vt_daemon_return, PZERO-1);		/* wait for the write */

    switch (vt_daemon_return) {
	case PLANECHARDATA:
	    loaded_planes.ftbftplanes[vt_load_index].ftbplane = plane;
	    loaded_planes.ftbftplanes[vt_load_index].ftbfontid = fontid;
	    loaded_planes.ftbftplanes[vt_load_index].ftbinuse = 0;
	    vt_plane_count++;
	    return vt_load_index;
    }
    return 0;
}

/*
 * This routine returns a pointer to a font plane descriptor array. If the font
 * plane is not one of the currently loaded font planes then the plane will be 
 * loaded before returning.  This routine sets the inuse flag on the font plane
 * entry, the caller should call DoneWithPlane(fontid, plane) after he writes 
 * out his string of chars.
 */
struct font_plane *
GetFontPlane(fontid, plane)
    register unsigned short	fontid;	/* Font ID number */
    register char		plane;	/* Plane number */
{
    register int	i;		/* Font/plane index */

    plane &= PLANEMSK;

    /* If font loading not enabled, return default plane. */
    if (vt_loadfonts == 0) {
	loaded_planes.ftbftplanes[0].ftbinuse++;
	return &plane_desc.pldchardescripts[0];
    }

    /* Check if using default plane, it is always available so return it. */
    if (fontid == DEFNONPROPFONT && plane == DEFPLANE) {
	loaded_planes.ftbftplanes[0].ftbinuse++;
	return &plane_desc.pldchardescripts[0];
    }

    GetFontTableInterlock(PZERO-1);
    i = GetPlaneTableIndex(fontid, plane);
    if (i < 0)
	i = LoadFontPlane(fontid, plane);
    loaded_planes.ftbftplanes[i].ftbinuse++;
    vt_font_table_interlock = 0;
    wakeup(&vt_font_table_interlock);
    loaded_planes.ftbftplanes[i].ftblastuse = time.tv_sec;
    return (&plane_desc.pldchardescripts[i]);
}

/*
 * This routine clears the in use flag on a font plane.  It should be called
 * after the characters in a particular font and plane have been referenced.
 */
DoneWithPlane(fontid, plane)
    register unsigned short	fontid;	/* Font ID number */
    register char		plane;	/* plane number */
{
    register int	i;		/* Font/plane index */

    if (vt_loadfonts) {
	plane &= PLANEMSK;
	if ( ((i = GetPlaneTableIndex(fontid, plane)) > 0) &&
	     (loaded_planes.ftbftplanes[i].ftbinuse   > 0)    )
	      loaded_planes.ftbftplanes[i].ftbinuse--;
    }
}

#ifdef	DEBUG
PrintPlaneUsage()
{
    register int	i;	/* Font/plane index */

    cprintf("indx	fontid	plane	inuse	lastuse		loc	len\n");
    for (i = 0 ; i < MAXPLANES ; i++) {
	cprintf(" %d	 %d	%d	%d	%x	%x	%x\n",
	    i, loaded_planes.ftbftplanes[i].ftbfontid,
	    loaded_planes.ftbftplanes[i].ftbplane,
	    loaded_planes.ftbftplanes[i].ftbinuse,
	    loaded_planes.ftbftplanes[i].ftblastuse,
	    loaded_planes.ftbftplanes[i].ftbrloc,
	    loaded_planes.ftbftplanes[i].ftbrlen);
    }
}
#endif	DEBUG

/*
 * This routine returns the baseline in pixels of "fontid" "plane".  This is the
 * distance from the top of the character cell to the baseline.
 */
int
CharacterBaseline( charinfo)
    register struct font_plane	*charinfo;	/* Pointer to plane */
{
    return( (int) -charinfo->baseline);
}

/*
 * This routine returns the height in pixels of all characters in 
 * "fontid" "plane".
 */
int
CharacterHeight( charinfo)
    register struct font_plane	*charinfo;	/* Pointer to plane */
{
    return (int) charinfo->height;
}

/*
 * This routine returns the width in pixels of "character" in "fontid" "plane".
 */
int
CharacterWidth( charinfo, character)
    register struct font_plane	*charinfo;	/* Pointer to plane */
    register char		character;	/* Character in plane */
{
    character = (character & CHARMSK) - PLANEBASE;
    if (character < 0)
	    return 0;
    return((int) charinfo->ch_desc[character].char_width);
}

/*
 * This routine returns the height in pixels (eventually motes) of the raster 
 * box for "character" in "font"/"plane".
 */
int
CharacterRHeight(fontid, plane, character)
    register unsigned short	fontid;		/* Font ID */
    register char		plane;		/* Plane number */
    register unsigned char	character;	/* Character */
{
    register struct font_plane	*charinfo;	/* Pointer to plane */
    register int		rheight;	/* Raster box height */

    character = (character & CHARMSK) - PLANEBASE;
    if (character < 0)
	return 0;
    charinfo = GetFontPlane(fontid, plane);
    rheight = (int) charinfo->ch_desc[character].rast_height;
    DoneWithPlane(fontid, plane);
    return rheight;
}

/*
 * This routine returns the width in pixels (eventually motes) of the raster box
 * for "character" in "fontid" "plane".
 */
int
CharacterRWidth(fontid, plane, character)
    register unsigned short	fontid;		/* Font ID */
    register char		plane;		/* Plane number */
    register char		character;	/* Character in plane */
{
    register struct font_plane	*charinfo;	/* Pointer to plane */
    register int			rwidth;		/* Raster box width */

    character = (character & CHARMSK) - PLANEBASE;
    if (character < 0)
	return 0;
    charinfo = GetFontPlane(fontid, plane);
    rwidth = (int) charinfo->ch_desc[character].rast_width;
    DoneWithPlane(fontid, plane);
    return rwidth;
}
