/* the NBI font structure has the following generalized form:
 *
 *	font external identification
 *	array [17] of ptr to font plane
 *	font plane #0
 *	font plane #1
 *	.
 *	.
 *	.
 *	font plane #15
 *	font plane #16 (the ASCII half plane)
 *
 * A NULL (== 0) entry in the font plane table, implies that that plane does
 * not exist and that there is no information stored for that plane.
 * (The entry is the (byte) offset into the file for the plane information
 * when the font is stored on disk.)
 */

/* NOTE: all dimensions for the char/raster boxes refer to pixels! */

/* miscellaneous internal typedefs */
typedef char		s_byte;
typedef short		s_word;
typedef unsigned char	u_byte;
typedef unsigned short	u_word;


/* the definition of the font plane */
/* if a particular character does not have a representation in the plane,
 * its char_width field will be set to zero (0).
 */
#define NUM_PLANE_CHARS	96
struct font_plane {
	struct plane_char_desc {
		u_byte	char_width;	/* char box width */
		s_byte	char_xoff;	/* char box left to char left margin */
		u_byte	rast_width;	/* raster box width */
		u_byte	rast_height;	/* raster box height */
		s_byte	rast_xoff;	/* char box left to raster lt margin */
		s_byte	rast_yoff;	/* char box top to raster top margin */
		u_word	offset;		/* offset (BYTES!) to actual pixels, */
					/* relative to raster pointer */
	} ch_desc[NUM_PLANE_CHARS];
	u_byte	height;		/* height of the font */
	s_byte	baseline;	/* offset from char box top to baseline */
	union {
		u_word		*ptr;	/* in memory, pointer to pixel info */
		unsigned long	size;	/* in file, # of bytes of pixel info */
	} raster;
#define rast_ptr raster.ptr
#define rast_size raster.size
	/* u_word pixels[];	/* the actual bit maps for the chars */
	/* While the actual pixels are conceptually stored here, they may
	 * really be elsewhere, since they are variable size information.
	 * The ptr field of the raster union locates them exactly.
	 * In a file, they will be stored immediately following this structure,
	 * and the size field of the raster union will contain their length
	 * (in bytes).
	 */
};


/* the definition of the font header */
/* NOTE: this definition follows that of the font plane so debuggers can
 * 	 access all of the information about fonts. (Them require definition
 *	 before use!)
 */
#define CTRL_PLANE	15
#define ASCII_PLANE	16
#define NUM_PLANES	17
struct font_header {
	short	ident;		/* double check that this is right font */
	char	name[32];	/* the descriptive name of the font */
	short	version;	/* the current release number */
	union {
		struct font_plane	*ptr[NUM_PLANES]; /* in memory usage */
		unsigned long		locn[NUM_PLANES]; /* in file usage */
	} plane_table;
#define plane_ptr	plane_table.ptr
#define plane_locn	plane_table.locn
};
