#include "param.h"
#include "systm.h"
#include "user.h"
#include "kernel.h"
#include "proc.h"
#include "file.h"
#include "vnode.h"
#include "conf.h"
#include "../ufs/inode.h"
#include "../ufs/mount.h"
#include "../specfs/snode.h"
#include "../wipc/wipc.h"
#include "../trfs/trfs.h"

extern unsigned long *Client, *ClientNPROC;
extern uid_t trfs_rootuid;		/* root uid transformation over net */

/* 
 * trfs specific symbolic link macro expansion
 */
char *
trfs_slink(p, len, mname, client)
	char *p;
	int *len;
	char *mname;
{
	static char *remote = "remote";
	char *RpidToName();
	register char *result;

	if (mname  &&  mname[0] == '\0')
                mname = (char *)0;

	if (strncmp(p, "$HOST", 5) == 0) {
		*len =5;
		if (mname)
			result = mname;
		else if (client  &&  (u.u_tdir & TRDIR))
			result = RpidToName(client);
		else
			result = hostname;
	} else if (strncmp(p, "$RHOST", 6) == 0) {
		*len = 6;
		if (mname) {
			if (IsMyMName(mname) == 0)
				result = mname;
			else
				result = "";
		} else if (client  &&  (u.u_tdir & TRDIR))
			result = RpidToName(client);
		else
			result = "";
	} else if (strncmp(p, "$REMOTE", 7) == 0) {
		*len = 7;
		if (mname) {
			if (IsMyMName(mname) == 0)
				result = remote;
			else
				result = "";
		} else if (client  &&  (u.u_tdir & TRDIR))
			result = remote;
		else
			result = "";
	}
	return result;
}

setsrcxs(xs)
	unsigned char xs;
{
	u.u_tsrcxs = xs;
}

setdstxs(xs)
	unsigned char xs;
{
	u.u_tdstxs = xs;
}

unsigned char
setpath(sp, path)
	struct wmsg_seg *sp;
	char *path;
{
	sp->addr = path;
	sp->len = MAXPATHLEN;
	return XS_KERN | XS_READ | XS_WRITE | XS_RMZSTR;
}

/* NOTE: credentials and user_ids have the same format!! */
setuser(id, rpid)
	register struct user_id *id;
{
	*id = *(struct user_id *)u.u_cred;

	/* root over the net becomes trfs_rootuid on server (uid -3) */
	if ((u.u_uid == 0) && (!diskless || (MID(rpid) != MID(servermid))))
		id->uid = trfs_rootuid;
}

getuser(id)
	register struct user_id *id;
{
	u.u_cred = crcopy(u.u_cred);	/* free old credentials, get new */
	*(struct user_id *)u.u_cred = *id;
	setsrcxs(XS_KERN | XS_READ);	/* get path from client kernel space */
}

FindCousin(rpid)
	register unsigned long rpid;
{
	register struct proc *p;
	register unsigned long *q = Client, *r = ClientNPROC;

	if (rpid == 0)
		return 0;
	do {
		if (*q++ == rpid) {
			p = &proc[q - 1 - Client];
			if (p->p_stat  &&  p->p_stat != SZOMB)
				return p->p_pid;
		}
	} while (q < r);
	return 0;
}

SetCousinRpid(pid, rpid)
	unsigned long rpid;
{
	register struct proc *p = u.u_procp;
	register int n;

	if (p->p_pid != pid) {
		p = (pid == 0) ? &proc[0] : pfind(pid);
		if (p == (struct proc *)0  ||  p->p_pid != pid  ||
		    p->p_stat == 0  ||  p->p_stat == SZOMB)
			return 0;
	}
	Client[p - proc] = rpid;
}

CousinNeeded(rmid)
	register unsigned long rmid;
{
	register struct file *fp, **fpp;

	if (rmid) {
		if (u.u_trrdirmid == rmid  ||  u.u_trcdirmid == rmid)
			return 1;
		fpp = u.u_ofile;
		while (fpp <= &u.u_ofile[u.u_lastfile])
			if ((fp = *fpp++)  &&  (fp->f_type == DTYPE_TRFS)  &&  
			    (fp->f_rmid == rmid))
				return 1;
	}
	return 0;
}

ResetCousinRpid()
{
	Client[u.u_procp - proc] = 0;
}

sysGetMachName(uap)
	register struct a {
		unsigned long	rpid;
		char		*mname;
		int		len;
	} *uap;
{
	extern char *RpidToName();
	char *name;
	int len;

#ifdef NPC
	if ((uap->rpid & MID_MASK) == NPC_MID) {
	    if((name = (char *)npc_getmachname(uap->rpid)) == (char *)0) {
		u.u_error = ESRCH;
		return;
	    }
	} else
#endif NPC
	if ((name = RpidToName(uap->rpid)) == (char *)0) {
		u.u_error = ESRCH;
		return;
	}
	len = MIN(uap->len, strlen(name) + 1);
	u.u_error = copyout((caddr_t)name, (caddr_t)uap->mname, len);
}

SwapInit(rpid)
	unsigned long rpid;
{
	struct vnode *vp;
	char swappath[MAXPATHLEN];
	char *name;
	extern char *RpidToName();
	int err = 0;

	if ((name = RpidToName(rpid)) == (char *)0)
		return EIO;
	sprintf(swappath, "/usr/spool/diskless/%s.swap", name);
	if (err = vn_open(swappath, UIOSEG_KERNEL, FWRITE|FCREAT,
		0600, &vp))
		printf("TRFS: can't create %s for swapping\n", swappath);
	else
		SetSwapVp(rpid, vp);
	return err;
}

/*
 * Make a cousin process a child of init.
 */
RunAway()
{
	register struct proc *q = u.u_procp, *p = &proc[1], *r;
	int s = splclock();

	/* detach from current siblings */
	if (r = q->p_ysptr)
		r->p_osptr = q->p_osptr;
	if (r = q->p_osptr)
		r->p_ysptr = q->p_ysptr;
	if ((r = q->p_pptr)->p_cptr == q)
		r->p_cptr = q->p_osptr;

	/* attach as a child of init */
	if (r = p->p_cptr)
		r->p_ysptr = q;
	q->p_ysptr = NULL;
	q->p_osptr = p->p_cptr;
	p->p_cptr = q;
	q->p_pptr = p;
	q->p_ppid = p->p_pid;
	spgrp(q);
	splx(s);
}

/* ------------------------- imported device support --------------------- */

#define	IMPORTED_DEV	makedev(DEV_TRFS_IMPORT,0)

ImportedDev(dev)
{
	return ((dev & IMPORTED_DEV) == IMPORTED_DEV);
}

ImportedDevice(imode, dev)
	unsigned short imode;
	dev_t dev;
{
	int imported = 0;

	if ((imode &= IFMT) == IFCHR  ||  (imode &= IFMT) == IFBLK)
		imported = ImportedDev(dev);
	return imported;
}

struct inode *
geti(fd)
{
	struct file *fp;
	struct vnode *vp;

/* THIS IS VERY BAD CODE!!! KLUDGE */
	if (fp = getf(fd)) {
		vp = (struct vnode *)fp->f_data;
		if (ISVDEV(vp->v_type))
			return(VTOI(VTOS(vp)->s_realvp));
	}
	return (struct inode *)0;
}

#define	INOHSZ	512
#if	((INOHSZ&(INOHSZ-1)) == 0)
#define	INOHASH(dev,ino)	(((dev)+(ino))&(INOHSZ-1))
#else
#define	INOHASH(dev,ino)	(((unsigned)((dev)+(ino)))%INOHSZ)
#endif

extern union ihead {			/* inode LRU cache, Chris Maltby */
	union  ihead *ih_head[2];
	struct inode *ih_chain[2];
} ihead[INOHSZ];

extern struct inode *ifreeh, **ifreet;

/*
 * look for a local vnode representing a device on the local machine which
 * was specified on the server.  If not found, allocate and initialize.
 * This code is similar to that in ../ufs/ufs_inode.c except that if it
 * is not found in the cache then we have enough information to create it
 * without reading it in from disk.
 */
struct vnode *
trfs_dev_set(dev, ino, mode, rdev)
	register dev_t dev, rdev;
	register ino_t ino;
	unsigned short mode;
{
	register struct inode *ip;
	register struct vnode *vp;
	register union  ihead *ih;
	register struct inode *iq;

	/*
	 * move the "imported device" bit from rdev to dev
	 */
	dev |= IMPORTED_DEV;
	rdev &= ~IMPORTED_DEV;

loop:	ih = &ihead[INOHASH(dev, ino)];
	for (ip = ih->ih_chain[0]; ip != (struct inode *)ih; ip = ip->i_forw)
		if (ino == ip->i_number && dev == ip->i_dev) {
			if ((ip->i_flag&ILOCKED) != 0) {
				ip->i_flag |= IWANT;
				sleep((caddr_t)ip, PINOD);
				goto loop;
			}
			/*
			 * If inode is on free list, remove it.
			 */
			if ((ip->i_flag & IREF) == 0) {
				if (iq = ip->i_freef)
					iq->i_freeb = ip->i_freeb;
				else
					ifreet = ip->i_freeb;
				*ip->i_freeb = iq;
				ip->i_freef = NULL;
				ip->i_freeb = NULL;
			}
			/*
			 * mark inode referenced and return it.
			 */
			ip->i_flag |= IREF;
			vp = ITOV(ip);

			/* If vnode is a device return special vnode instead */
			if (ISVDEV((vp)->v_type))
				vp = specvp(vp, rdev);
			else
				VN_HOLD(vp);
			return(vp);
		}

	/*
	 * Inode was not in cache. Get free inode slot for new inode.
	 */
	if (ifreeh == NULL)
		dnlc_purge();
	if ((ip = ifreeh) == NULL) {
		tablefull("irset: inode");
		u.u_error = ENFILE;
		return(NULL);
	}
	if (iq = ip->i_freef)
		iq->i_freeb = &ifreeh;
	ifreeh = iq;
	ip->i_freef = NULL;
	ip->i_freeb = NULL;
	/*
	 * Now to take inode off the hash chain it was on
	 * (initially, or after an iflush, it is on a "hash chain"
	 * consisting entirely of itself, and pointed to by no-one,
	 * but that doesn't matter), and put it on the chain for
	 * its new (ino, dev) pair
	 */
	remque(ip);
	insque(ip, ih);
	vp = ITOV(ip);
#ifdef QUOTA
	dqrele(ip->i_dquot);
	ip->i_dquot = NULL;
#endif
	ip->i_nlink = 1;	/* How do we know if it gets removed? */
	ip->i_dev = dev;
	ip->i_number = ino;
	ip->i_mode = mode;
	ip->i_rdev = rdev;
	ip->i_devvp = getmp(dev)->m_devvp;
	ip->i_flag = IREF;
	ip->i_diroff = 0;
	ip->i_lastr = 0;
	VN_INIT(vp, NULL, IFTOVT(mode), rdev);

	/* If vnode is a device return special vnode instead */
	if (ISVDEV((vp)->v_type)) {
		struct vnode *newvp;

		newvp = specvp(vp, rdev);
		VN_RELE(vp);
		vp = newvp;
	}
	return (vp);
}

trfs_dev_open(vp, mode)
	struct vnode *vp;
{
	return(VOP_OPEN(&vp, mode, u.u_cred));
}

RestartSys()
{
	u.u_eosys = RESTARTSYS;
}

#ifdef	SYSV
OptimizeCopy()
{
	register struct vnode *vp;

	if (u.u_procp->p_universe != UNIVERSE_SYSV)
		return 1;
	if ((vp = u.u_pdir) && vp->v_flag & VSVDIROP)
		return 0;
	return 1;
}
#endif	SYSV
