#include "../machine/pte.h"
#include "param.h"
#include "user.h"
#include "proc.h"
#include "seg.h"
#include "text.h"
#include "vnode.h"
#include "pathname.h"
#include "seg.h"
#include "uio.h"
#include "exec.h"
#include "vfs.h"
#include "vmmac.h"
#include "../wipc/wipc.h"
#include "../trfs/trfs.h"

#ifdef vax
#include "../vax/mtpr.h"
#endif vax

#ifdef SYSV
#include "../sysv/sys/filehdr.h"
#endif SYSV

/*
 * The exec system calls are NOT revectored system calls. For an exec of a TRFS
 * file the standard unix code in sys/kern_exec.c has been modified to 
 * reference the following code.  The first entry from the standard unix
 * code is to ("ClExec") to obtain header/setuid information from the server, 
 * the second entry ("ClGetxfile") actually loads the image.  As with system 
 * calls, the server side uses the "Co" prefix.  
 * NOTE: 413 images are NOT demmand paged and do NOT share text.
 */

#define	SHSIZE		32
union exdata {
	char	ex_shell[SHSIZE];	/* #! and name of interpreter */
	struct	exec ex_exec;		/* a.out header */
};

struct exinfo {
	union exdata exdata;
#ifdef	SYSV
	struct sysv_exdata sv_exdata;
#endif	SYSV
};

ClExec(rpidp, fname, fnamelen, exdatap,
#ifdef	SYSV
	sv_exdatap,
#endif	SYSV
	uidp, gidp)
	int	*rpidp;
	char	*fname;
	union exdata *exdatap;
#ifdef	SYSV
	struct sysv_exdata *sv_exdatap;
#endif	SYSV
	uid_t	*uidp;
	gid_t	*gidp;
{
	struct exinfo exinfo;
	int rpid = u.u_tcurrpid;

	bzero(&exinfo, sizeof (struct exinfo));
	if (((u.u_error = SdChexec(rpid, fname, fnamelen, &exinfo, 
	    sizeof(struct exinfo), uidp, gidp)) == 0 ) && 
	    (PID(rpid) == SERVER_PID) )
		rpid = RmidToRpid(MID(rpid));	/* was client, now cousin */
	*rpidp = rpid;

	if (u.u_error == 0) {
		bcopy(&exinfo.exdata, exdatap, sizeof(union exdata));
#ifdef	SYSV
		bcopy(&exinfo.sv_exdata, sv_exdatap,
						sizeof(struct sysv_exdata));
#endif	SYSV
	}

	return u.u_error;
}

CoChexec(fname, exdatap, exdatalen, uidp, gidp)
	char *fname;
	struct exinfo *exdatap;
	uid_t *uidp;
	gid_t *gidp;
{
	struct vnode	*vp = NULL;
	struct pathname	pn;
	struct vattr	vattr;
	struct exinfo 	exdata;
	int		resid;

	if (u.u_error = pn_get(fname, UIOSEG_USER, &pn))
		return u.u_error;
	if (u.u_tok = (setjmp(&u.u_tsave) == 0)) {
		u.u_error = lookuppn(&pn, FOLLOW_LINK, (struct vnode **)0, &vp);
		u.u_tok = 0;
	} else
		u.u_error = EMLINK;		/* indirection not supported */
	if (u.u_error) {
		pn_free(&pn);
		return u.u_error;
	}
	if (u.u_error = VOP_GETATTR(vp, &vattr, u.u_cred))
		goto bad;
	if ((vp->v_vfsp->vfs_flag & VFS_NOSUID) == 0) {
		if(uidp) {
		    if (vattr.va_mode & VSUID) {
			*uidp = vattr.va_uid;
		    } else if (*uidp == trfs_rootuid) {
			*uidp = 0;
		    }
		}
		if ((vattr.va_mode & VSGID) && gidp)
			*gidp = vattr.va_gid;
	} else  if (uidp && *uidp == trfs_rootuid) {
	    *uidp = 0;
	}
	if (u.u_error = VOP_ACCESS(vp, VEXEC, u.u_cred))
		goto bad;
	if ((u.u_procp->p_flag&STRC)
	    && (u.u_error = VOP_ACCESS(vp, VREAD, u.u_cred)))
		goto bad;
	if (vp->v_type != VREG ||
	   (vattr.va_mode & (VEXEC|(VEXEC>>3)|(VEXEC>>6))) == 0) {
		u.u_error = EACCES;
		goto bad;
	}

	exdata.exdata.ex_shell[0] = 0;		/* for zero length files */
	if (u.u_error = vn_rdwr(UIO_READ, vp, (caddr_t)&exdata.exdata, 
	    sizeof (union exdata), 0, UIOSEG_KERNEL, IO_UNIT, &resid))
	    goto bad;
#ifndef lint
	if (resid > sizeof (union exdata) - sizeof (exdata.exdata.ex_exec) &&
	    exdata.exdata.ex_shell[0] != '#') {
		u.u_error = ENOEXEC;
		goto bad;
	}
#endif
#ifdef	SYSV
	if (exdata.exdata.ex_exec.a_machtype == COFFMAGIC ||
			exdata.exdata.ex_exec.a_machtype == NCRMAGIC)
		if (sysv_gethead(vp, &exdata.exdata, &exdata.sv_exdata))
			goto bad;
#endif	SYSV

	if (u.u_error = xCopyTo(u.u_tclient, pn.pn_path, fname,
				pn.pn_pathlen, XS_KERN, XS_KERN))
		goto bad;
	if (u.u_error = xCopyTo(u.u_tclient, &exdata, exdatap,
				exdatalen, XS_KERN, XS_KERN))
		goto bad;
	u.u_texec = vp;			/* save pointer to vnode! */
	pn_free(&pn);
	return u.u_error;

bad:	pn_free(&pn);
	VN_RELE(vp);
	return u.u_error;
}

/* #define	RUN410 /* */

/*
 * Read in and set up memory for executed file.  The text is
 * merged into the data and bss, and loaded like a 407.  Not all this
 * code is TRFS specific, see ../sys/kern_exec.c for comments.
 */
ClGetxfile(rpid, ep,
#ifdef	SYSV
	sv_ep,
#endif	SYSV
	nargc, uid, gid, pn)
	register struct exec *ep;
#ifdef	SYSV
	struct sysv_exdata *sv_ep;
#endif	SYSV
	uid_t uid;
	gid_t gid;
	struct pathname *pn;
{
	size_t ts, ds, ids, uds, ss;
	register struct proc *p = u.u_procp;
	struct dmap *tdmap = 0;
#ifdef	SYSV
	register unsigned long datasize;
#endif	SYSV

	if (u.u_error = chkaout(ep->a_machtype))
		goto bad;
	ts = stoc(ctos(btoc(ep->a_text)));
	uds = clrnd(btoc(ep->a_bss));
	ss = clrnd(SSIZE + btoc(nargc));
#ifdef	SYSV
  	if (p->p_universe == UNIVERSE_SYSV) {
		if (ep->a_magic != OMAGIC
		    && stoc(ctos(btoc(ep->a_text + sv_ep->a_txtorg)))
					> btop(sv_ep->a_datorg)) {
			ep->a_magic = OMAGIC;
			datasize = ep->a_data + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_txtorg)));
			ts = 0;
			ds = clrnd(btoc(datasize + ep->a_bss));
			ids = clrnd(btoc(datasize));
			ep->a_text = 0;
		} else {
			ts = clrnd(btoc(ep->a_text + sv_ep->a_txtorg
				- (long)(ptob(btop(sv_ep->a_txtorg)))));
			ds = clrnd(btoc(ep->a_data + ep->a_bss + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_datorg)))));
			ids = clrnd(btoc(ep->a_data + sv_ep->a_datorg
				- (long)(ptob(btop(sv_ep->a_datorg)))));
			datasize = 0;
		}
	} else {
#endif	SYSV

#ifdef	RUN410
		ids = clrnd(btoc(ep->a_data));
		ds = clrnd(btoc(ep->a_data + ep->a_bss));
#else	RUN410
		ids = ts + clrnd(btoc(ep->a_data));
		ds = ts + clrnd(btoc(ep->a_data + ep->a_bss));
#endif	RUN410

#ifdef	SYSV
	}
  	if (p->p_universe == UNIVERSE_SYSV &&
	    chksize((unsigned)ts, (unsigned)ids, (unsigned)uds, (unsigned)ss))
		goto bad;
	else
#endif	SYSV

#ifdef	RUN410
	if (chksize((unsigned)ts, (unsigned)ids, (unsigned)uds, (unsigned)ss))
		goto bad;
#else	RUN410
	if (chksize((unsigned)0, (unsigned)ids, (unsigned)uds, (unsigned)ss))
		goto bad;
#endif	RUN410

	tdmap = (struct dmap *)kmem_alloc(2 * sizeof (struct dmap));
 	tdmap[0] = zdmap;
 	tdmap[1] = zdmap;
 	if (swpexpand(ds, ss, &tdmap[0], &tdmap[1]) == NULL)
		goto bad;
	if ((p->p_flag & SVFORK) == 0)
		vrelvm();
	else {
		p->p_flag &= ~SVFORK;
		p->p_flag |= SKEEP;
		wakeup((caddr_t)p);
		while ((p->p_flag & SVFDONE) == 0)
			sleep((caddr_t)p, PZERO - 1);
		p->p_flag &= ~(SVFDONE|SKEEP);
	}
	p->p_flag &= ~(SPAGV|SSEQL|SUANOM|SOUSIG);
	p->p_flag |= SKEEP;
	u.u_dmap = tdmap[0];
	u.u_smap = tdmap[1];
#ifdef	SYSV
	if (p->p_universe == UNIVERSE_SYSV) {
		p->p_tstart = btop(sv_ep->a_txtorg);
		vgetvm(ts, ds, ss);
	} else {
#endif	SYSV
	p->p_tstart = 0;

#ifdef	RUN410
	vgetvm(ts, ds, ss);
#else	RUN410
	vgetvm(0, ds, ss);
#endif	RUN410

#ifdef	SYSV
	}
#endif	SYSV
	if (u.u_error) {
		printf("TRFS: getxfile %d: ", u.u_error);
		swkill(p, "ClGetxfile");
	}

#ifdef	SYSV
  	if (p->p_universe == UNIVERSE_SYSV) {
		if (datasize)
			u.u_error = SdLoad(rpid, 
				(char *)sv_ep->a_txtorg,
				(u_long)sv_ep->a_tsize,
				(u_long)sv_ep->a_toffset);
		if (u.u_error == 0)
			u.u_error = SdLoad(rpid, 
				(char *)sv_ep->a_datorg,
				(u_long)ep->a_data,
				(u_long)sv_ep->a_doffset);
		if (u.u_error == 0 && ep->a_magic != OMAGIC)
			ClXalloc(rpid, ep, sv_ep);
	} else
#endif	SYSV

	if (ep->a_text) {
#ifdef	RUN410
		u.u_error = SdLoad(rpid, ctob(dptov(p, 0)), (int)ep->a_data, 
			(int)(ep->a_text + ((ep->a_magic==ZMAGIC) ? 0 :
				sizeof (struct exec))));
		if (u.u_error == 0)
#ifdef	SYSV
			ClXalloc(rpid, ep, sv_ep);
#else	SYSV
			ClXalloc(rpid, ep);
#endif	SYSV
#else	RUN410
		if ((u.u_error = SdLoad(rpid, ctob(dptov(p, ts)), ep->a_data,
			(int)(ep->a_text + ((ep->a_magic==ZMAGIC) ? 0 :
			sizeof (struct exec))))) == 0)	/* data */
		    u.u_error = SdLoad(rpid, ctob(tptov(p, 0)), ep->a_text,
			(int)((ep->a_magic==ZMAGIC) ? 0 : sizeof(struct exec)));
#endif	RUN410
	} else
		u.u_error = SdLoad(rpid, ctob(dptov(p, 0)), ep->a_data, 
			sizeof (struct exec));		/* text + data */
	SdLoad(rpid, 0, 0, 0);		/* release texec vnode on server */
	p->p_flag &= ~SKEEP;

	if (u.u_error) {
		printf("TRFS: load %d: ", u.u_error);
		swkill(p, "ClGetxfile");
	}
	if ((p->p_flag&STRC)==0) {
		if (uid != u.u_uid || gid != u.u_gid)
			u.u_cred = crcopy(u.u_cred);
		u.u_uid = uid;
		p->p_uid = uid;
		u.u_gid = gid;
#ifdef	SYSV
		p->p_suid = uid;
		p->p_sgid = gid;
#endif	SYSV
	} else
		psignal(p, SIGTRAP);
	u.u_prof.pr_scale = 0;
bad:
	if (tdmap)
 		kmem_free((caddr_t)tdmap, 2 * sizeof (struct dmap));
}

CoLoad(addr, len, offset, segflg)
	char *addr;
	unsigned long len, offset;
{
	register struct vnode *vp = u.u_texec;
	struct uio auio;
	struct iovec aiov;

	if (vp == NULL)
		return ENOEXEC;
	if (len == 0  &&  offset == 0) {
		u.u_texec = NULL;
		VN_RELE(vp);
		return 0;
	}
	auio.uio_iovcnt = 1;
	auio.uio_iov = &aiov;
	auio.uio_segflg = segflg;
	auio.uio_offset = offset;
	auio.uio_resid = aiov.iov_len = len;
	aiov.iov_base = addr;
	if (u.u_error = VOP_RDWR(vp, &auio, UIO_READ, IO_UNIT, u.u_cred))
		return(u.u_error);
	if (auio.uio_resid != 0)
		return EIO;
	return 0;
}

/*
 * Attach to a text segment.
 */
#ifdef	SYSV
ClXalloc(rpid, ep, sv_ep)
#else	SYSV
ClXalloc(rpid, ep)
#endif	SYSV
	register struct exec *ep;
#ifdef	SYSV
	register struct sysv_exdata *sv_ep;
#endif	SYSV
{
	register struct text *xp;
	register struct text *xp1;
	register struct proc *p = u.u_procp;

	if (ep->a_text == 0)
		return;
	xstats.alloc++;
	xp = xhead;
	if (xp == NULL) {
		tablefull("text");
		psignal(p, SIGKILL);
		return;
	}
	ALLOC(xp);
	if (xp->x_vptr) {
		xstats.alloc_cacheflush++;
		if (xp->x_flag & XUNUSED)
			xstats.alloc_unused++;
		xuntext(xp);
		xcache--;
	}
	xp->x_flag = XLOAD|XLOCK|XTRFS;
#ifdef	SYSV
  	if (p->p_universe == UNIVERSE_SYSV)
		xp->x_size = clrnd(btoc(ep->a_text + sv_ep->a_txtorg
				- (long)(ptob(btop(sv_ep->a_txtorg)))));
	else
#endif	SYSV
	xp->x_size = clrnd(btoc(ep->a_text));

	if (vsxalloc(xp) == NULL) {
		xp->x_flag &= ~XLOCK;
		swkill(p, "ClXalloc: no swap space");
		return;
	}
	xp->x_count = 1;
	xp->x_ccount = 0;
	xp->x_rssize = 0;
	xp->x_vptr = NULL;
	p->p_textp = xp;
	xlink(p);
	settprot(RW);
	p->p_flag |= SKEEP;
#ifdef	SYSV
	if (p->p_universe == UNIVERSE_SYSV) 
		u.u_error = SdLoad(rpid, 
			(char *)sv_ep->a_txtorg,
			(u_long)sv_ep->a_tsize,
			(u_long)sv_ep->a_toffset);
	else
#endif	SYSV
	u.u_error = SdLoad(rpid, 
			(caddr_t)ctob(tptov(p, 0)), (int)ep->a_text, 
			(int)((ep->a_magic==ZMAGIC) ? 0 : sizeof(struct exec)));
	p->p_flag &= ~SKEEP;
	settprot(RO);
	xp->x_flag |= XWRIT;
	xp->x_flag &= ~XLOAD;
	XUNLOCK(xp);
}

#ifdef	SYSV
/*
 * given a path and an rpid, read <len> bytes into <addr>
 * from <offset> bytes into the file pointed to by <path>
 * <path> must be in kernel space
 */

ClPload(rpid, path, addr, len, offset, segflg)
	unsigned long rpid;
	char *path;
	char *addr;
	unsigned long len, offset;
{
	return SdPload(rpid, path, addr, len, offset, segflg);
}

CoPload(path, addr, len, offset, segflg)
	char *path;
	char *addr;
	unsigned long len, offset;
{
	struct vnode *vp;
	struct uio auio;
	struct iovec aiov;
	struct pathname pn;
	register int error = 0;

	if (error = pn_get(path, UIOSEG_KERNEL, &pn))
		return error;
	if (setjmp(&u.u_tsave) == 0)
		error = lookuppn(&pn, FOLLOW_LINK, (struct vnode **)0, &vp);
	else
		error = EMLINK;		/* indirection not supported */
	pn_free(&pn);
	if (error)
		return error;
	if (vp == NULL)
		return ENOEXEC;
	if (len == 0  &&  offset == 0)
		goto out;
	auio.uio_iovcnt = 1;
	auio.uio_iov = &aiov;
	auio.uio_segflg = segflg;
	auio.uio_offset = offset;
	auio.uio_resid = aiov.iov_len = len;
	aiov.iov_base = addr;
	if (error = VOP_RDWR(vp, &auio, UIO_READ, IO_UNIT, u.u_cred))
		goto out;
	if (auio.uio_resid != 0)
		error = EIO;
out:
	VN_RELE(vp);
	return error;
}

struct headinfo {
	struct exec exdata;
	struct sysv_exdata sv_exdata;
};

ClGethead(rpid, path, execp, svexecp)
unsigned long rpid;
char *path;
struct exec *execp;
struct sysv_exdata *svexecp;
{
	struct headinfo headinfo;
	register int error;

	bcopy(execp, &headinfo.exdata, sizeof (struct exec));
	bcopy(svexecp, &headinfo.sv_exdata, sizeof (struct sysv_exdata));
	if (error = SdGethead(rpid, path, &headinfo, sizeof (struct headinfo)))
		return error;
	bcopy(&headinfo.exdata, execp, sizeof (struct exec));
	bcopy(&headinfo.sv_exdata, svexecp, sizeof (struct sysv_exdata));
	return 0;
}

CoGethead(path, addr)
char *path;
char *addr;
{
	struct headinfo headinfo;
	struct pathname	pn;
	struct vnode *vp;
	register int error = 0;

	if (error = pn_get(path, UIOSEG_KERNEL, &pn))
		return error;
	error = lookuppn(&pn, FOLLOW_LINK, (struct vnode **)0, &vp);
	pn_free(&pn);
	if (error)
		return error;
	if (vp == NULL)
		return ENOEXEC;
	if (error = xCopyFrom(u.u_tclient, addr, &headinfo,
				sizeof(struct headinfo), XS_KERN, XS_KERN))
		goto out;
	if (sysv_gethead(vp, &headinfo.exdata, &headinfo.sv_exdata)) {
		error = u.u_error;
		goto out;
	}
	headinfo.sv_exdata.vp = NULL;		/* meaningless for client */
	error = xCopyTo(u.u_tclient, &headinfo, addr, sizeof(struct headinfo),
						XS_KERN, XS_KERN);
out:
	VN_RELE(vp);
	return error;
}
#endif	SYSV
