/*
 * pseudo loop back driver for RFS test
 */

#if	defined(SYSV) && defined(RFS)
#include	"../h/types.h"
#include	"../h/param.h"
#include	"../h/user.h"
#include	"../sysv/sys/sysmacros.h"
#include	"../sysv/sys/stream.h"
#include	"../sysv/sys/stropts.h"
#include	"../sysv/sys/errno.h"
#include	"../sysv/sys/tihdr.h"
#include	"../sysv/sys/tiuser.h"
#include	"../sysv/sys/loop.h"
#include	"../sysv/sys/debug.h"

static unsigned char Req_to_event[] = {
/*T_CONN_REQ*/		TE_CONN_REQ,
/*T_CONN_RES*/		TE_CONN_RES,
/*T_DISCON_REQ*/	TE_DISCON_REQ,
/*T_DATA_REQ*/		TE_DATA_REQ,
/*T_EXDATA_REQ*/	TE_EXDATA_REQ,
/*T_INFO_REQ*/		-1,
/*T_BIND_REQ*/		TE_BIND_REQ,
/*T_UNBIND_REQ*/	TE_UNBIND_REQ,
/*T_UNITDATA_REQ*/	TE_UNITDATA_REQ,
/*T_OPTMGMT_REQ*/	TE_OPTMGMT_REQ,
/*T_ORDREL_REQ*/	TE_ORDREL_REQ
};

static struct module_info minfo = {
	0, "loop", 0, INFPSZ, 512, 128
};

/*static*/ int loopopen(), loopclose(), loopwput(), loopwsrv();
/*static*/ int /*looprput(),*/ looprsrv();

static struct qinit rinit = {
	NULL, looprsrv, loopopen, loopclose, NULL, &minfo, NULL
};

static struct qinit winit = {
	loopwput, loopwsrv, NULL, NULL, NULL, &minfo, NULL
};

struct streamtab loopinfo = { &rinit, &winit, NULL, NULL };

extern struct loop loop_loop[];
extern int loop_cnt;

/*static*/ void seterror(), setok();


/*ARGSUSED*/
/*static*/ int
loopopen(q, dev, flag, sflag)
queue_t	*q;
{
	struct loop *loop;

	/*
	 * If CLONEOPEN pick a minor device number to use.
	 * Otherwise, check the minor device range.
	 */
	if(sflag == CLONEOPEN) {
		for(dev=0; dev < loop_cnt; dev++) {
			if(!(loop_loop[dev].lup_flags & LOOP_INUSE))
				break;
		}
	} else
		dev = minor(dev);

	if(dev >= loop_cnt)
		return OPENFAIL;

	if(loop_loop[dev].lup_flags & LOOP_INUSE) /* already open */
		return(dev);
	loop_loop[dev].lup_flags = LOOP_INUSE;
	loop = &loop_loop[dev];
	loop->lup_flags = LOOP_INUSE;
	loop->lup_state = TS_UNBND;
	loop->lup_aqlen = loop->lup_qlen = 0;
	loop->lup_baddr[NETADDR] = loop->lup_baddr[PORTADDR] = 0;
	loop->oqptr = (queue_t *)NULL;
	WR(q)->q_ptr = (char *) loop;
	q->q_ptr = (char *)loop;
	loop->qptr = WR(q);
	/*
	 * Return minor device
	 */
	return dev;
}


/* Write put proc */

/*static*/ int
loopwput(q, mp)
queue_t	*q;
register mblk_t	*mp;
{
	register struct loop *loop;

	loop = (struct loop *)q->q_ptr;

	switch(mp->b_datap->db_type) {
	case M_IOCTL: {

		mp->b_datap->db_type = M_IOCNAK;
		((struct iocblk *)mp->b_rptr)->ioc_error = EINVAL;
		qreply(q, mp);
		break;
	}

	case M_FLUSH:

		/* Canonical flush processing */
		if(*mp->b_rptr & FLUSHW)
			flushq(q, FLUSHDATA);
		if(*mp->b_rptr & FLUSHR) {
			flushq(RD(q), FLUSHDATA);
			*mp->b_rptr & = ~FLUSHW;
			qreply(q, mp);
		} else
			freemsg(mp);
		break;

	case M_PCPROTO:	{
		register union T_primitives *ackp;
		register mblk_t *tmp;

		if((tmp = allocb(sizeof(struct T_info_ack), BPRI_MED)) == NULL) {
			mp->b_datap->db_type = M_ERROR;
			mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
			*mp->b_wptr++ = EPROTO;
			qreply(q, mp);
		} else if(((union T_primitives *)mp->b_rptr)->type ==
				T_INFO_REQ && (mp->b_wptr - mp->b_rptr) ==
				sizeof(struct T_info_req)) {
			ackp = (union T_primitives *)tmp->b_rptr;
			ackp->info_ack.PRIM_type = T_INFO_ACK;
			ackp->info_ack.TSDU_size = LOOP_MAXTSDUSIZE;
			ackp->info_ack.ETSDU_size = LOOP_MAXETSDUSIZE;
			ackp->info_ack.CDATA_size = LOOP_MAXCDATASIZE;
			ackp->info_ack.DDATA_size = LOOP_MAXDDATASIZE;
			ackp->info_ack.ADDR_size = LOOP_MAXADDRSIZE;
			ackp->info_ack.OPT_size	= LOOP_MAXOPTSIZE;
			ackp->info_ack.TIDU_size = LOOP_MAXTIDUSIZE;
			ackp->info_ack.SERV_type = LOOP_SERVTYPE;
			ackp->info_ack.CURRENT_state = loop->lup_state;
			tmp->b_wptr += sizeof(struct T_info_ack);
			tmp->b_datap->db_type = M_PCPROTO;
		} else {
			putq(q, mp);
			freemsg(tmp);
			break;
		}
		if(tmp != NULL)
			qreply(q, tmp);
		freemsg(mp);
		break;
	}

	default:
		putq(q, mp);
	}
}


/*static*/ int
loopwsrv(q)
queue_t *q;
{
	register mblk_t	*mp, *tmp;
	register struct loop *loop;
	register union T_primitives *pptr, *ackp;
	int event;

	loop = (struct loop *)q->q_ptr;
	ASSERT(loop->lup_flags & LOOP_INUSE);

	tmp = NULL;
	while((mp = getq(q)) != NULL) {
		event = -1;
		switch(mp->b_datap->db_type) {
		default:
passon:
			if(loop->oqptr != NULL && (loop->lup_flags & LOOP_CONNECTED)) {
			/* Pass the message to the other side */
			/* For now state changes are not being checked */
				if(mp->b_datap->db_type <= QPCTL &&
						!canput(loop->oqptr->q_next)) {
					putbq(q, mp); /* read side blocked */
					goto out;	/* out of while-loop */
				} else
					putnext(loop->oqptr, mp);
				/*****
				if(tmp != NULL)
					freemsg(tmp);
				******/
				continue; /* Don't break */
			} else {
garbage:
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EPROTO;
				qreply(q, mp);
				if(tmp != NULL)
					freemsg(tmp);
				break;
			}

		case M_DATA:	/* Send it as a data indication */
			if((tmp = allocb(sizeof(struct T_data_ind), BPRI_MED)) == NULL) {
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EAGAIN;
				qreply(q, mp);
				goto out1;
			}
			pptr = (union T_primitives *)tmp->b_rptr;
			pptr->data_ind.PRIM_type = T_DATA_IND;
			pptr->data_ind.MORE_flag = 0;	/* For now */
			tmp->b_wptr = tmp->b_rptr + sizeof(struct T_data_ind);
			linkb(tmp, mp);
			mp = tmp;
			tmp = NULL;
			goto passon;

		case M_PCPROTO:	
			/* Ideally these messages should not come here! */
		case M_PROTO:	{
			register len;
			register unsigned addr;
			int	error;

			pptr = (union T_primitives *)mp->b_rptr;
			if((tmp = allocb(MAX_REPLYSIZ, BPRI_MED)) == NULL) {
				mp->b_datap->db_type = M_ERROR;
				mp->b_rptr = mp->b_wptr = mp->b_datap->db_base;
				*mp->b_wptr++ = EAGAIN;
				qreply(q, mp);
				goto out1;
			}
			while(loop->lup_flags & LOOP_CHST)
				sleep(loop, PLOOP);
			loop->lup_flags |= LOOP_CHST;
			if((unsigned)pptr->type < T_CONN_IND) 
				event = Req_to_event[pptr->type];
			if(error = changestate(loop, event)) {
				loop->lup_flags &= ~LOOP_CHST;
				wakeup(loop);
				seterror(mp, tmp, hiword(error), loword(error));
				qreply(q, tmp);
				freemsg(mp);
				break;
			}
			switch(pptr->type) {
			default:
				loop->lup_flags &= ~LOOP_CHST;
				wakeup(loop);
				goto garbage;
		
			case T_BIND_REQ:

				ackp = (union T_primitives *)tmp->b_rptr;
				len = pptr->bind_req.ADDR_length; /* ?? */
				if(len)
					addr = *(int *)(mp->b_rptr + pptr->bind_req.ADDR_offset);
				if((mp->b_wptr - mp->b_rptr) < sizeof(struct T_bind_req) ||
					(len && ((mp->b_wptr - mp->b_rptr) != (len + pptr->bind_req.ADDR_offset))))
					seterror(mp, tmp, TBADDATA, 0);
				else if(len < 0 || len > LOOP_MAXADDRSIZE)
					seterror(mp, tmp, TBADADDR, 0);
				else if((addr = checkinuse(addr, len)) == -1 &&
						(addr = allocaddr(mp)) == -1)
					seterror(mp, tmp, TNOADDR, 0);
				else if(loop->lup_flags & LOOP_BOUND)
					seterror(mp, tmp, TOUTSTATE, 0);
				else {
					loop->lup_flags |= LOOP_BOUND;
					loop->lup_baddr[PORTADDR] = addr;
					loop->lup_aqlen = pptr->bind_req.CONIND_number;
					ackp->bind_ack.PRIM_type = T_BIND_ACK;
					ackp->bind_ack.ADDR_length = 4;
					ackp->bind_ack.ADDR_offset = sizeof(struct T_bind_ack);
					ackp->bind_ack.CONIND_number = loop->lup_aqlen;
					tmp->b_wptr += sizeof(struct T_bind_ack);
					*(int *)tmp->b_wptr++ = addr;
					tmp->b_datap->db_type = M_PROTO;
					event = TE_BIND_ACK;
				}
				if(event == -1) event = TE_ERROR_ACK;
				break;

			case T_UNBIND_REQ:

				if((mp->b_wptr - mp->b_rptr) != sizeof(struct T_unbind_req))
					seterror(mp, tmp, TBADDATA, 0);
				else if(!(loop->lup_flags & LOOP_BOUND))
					seterror(mp, tmp, TOUTSTATE, 0);
				else {
					/* Some more checks would be needed
					   before unbinding */
					loop->lup_flags &= ~LOOP_BOUND;
					loop->lup_baddr[PORTADDR] = 0;
					setok(mp, tmp);
					event = TE_OK_ACK1;
				}
				if(event == -1) event = TE_ERROR_ACK;
				break;

			case T_CONN_REQ:

				ackp = (union T_primitives *)tmp->b_rptr;
				len = pptr->conn_req.DEST_length; /* ?? */
				if(len)
					addr = *(int *)(mp->b_rptr + pptr->bind_req.ADDR_offset);
				if(len <= 0 || len > LOOP_MAXADDRSIZE)
					seterror(mp, tmp, TBADADDR, 0);
				else if(pptr->conn_req.OPT_length || pptr->conn_req.OPT_offset)
					seterror(mp, tmp, TNOTSUPPORT, 0);
				else if((mp->b_wptr - mp->b_rptr) < sizeof(struct T_conn_req) ||
					(mp->b_wptr - mp->b_rptr) != (len + pptr->conn_req.DEST_offset))
					seterror(mp, tmp, TBADDATA, 0);
				/* For now reject if somebody is not already
				   bound to the address
				 */
				else if(checkinuse(addr, len) != -1)
					seterror(mp, tmp, TNOADDR, 0);
				else if(!(loop->lup_flags & LOOP_BOUND))
					seterror(mp, tmp, TOUTSTATE, 0);
				else if(error = send_conn_ind(loop, mp, addr, len))
						seterror(mp, tmp, hiword(error), loword(error));
				else {
					setok(mp, tmp);
					event = TE_OK_ACK1;
				}
				if(event == -1) event = TE_ERROR_ACK;
				break;

			case T_CONN_RES:

				ackp = (union T_primitives *)tmp->b_rptr;
				if(pptr->conn_res.OPT_length || pptr->conn_res.OPT_offset)
					seterror(mp, tmp, TNOTSUPPORT, 0);
				else if((mp->b_wptr - mp->b_rptr) !=
						sizeof(struct T_conn_res))
					seterror(mp, tmp, TBADDATA, 0);
				else if(error = makeconnect(loop, mp, &event))
					seterror(mp, tmp, hiword(error), loword(error));
				else {
					loop->lup_qlen--;
					setok(mp, tmp);
				}
				if(event == -1) event = TE_ERROR_ACK;
				break;

			case T_DATA_REQ:
			case T_EXDATA_REQ:
				ASSERT(loop->oqptr != NULL);
				ASSERT(loop->lup_flags & LOOP_CONNECTED);
				pptr->type = (pptr->type == T_DATA_REQ) ?
						T_DATA_IND : T_EXDATA_IND;
				if(mp->b_datap->db_type <= QPCTL &&
						!canput(loop->oqptr->q_next)) {
					putbq(q, mp); /* read side blocked */
					goto out;	/* out of while-loop */
				} else
					putnext(loop->oqptr, mp);
				loop->lup_flags &= ~LOOP_CHST;
				wakeup(loop);
				/*
				 * Ideally the destination state need to be
				 * changed. But we are avoiding it because
				 * WE KNOW that a state change does not occur
				 * in data indications
				 */
				freemsg(tmp);
				continue; /* Don't break */

			case T_DISCON_REQ:

				ackp = (union T_primitives *)tmp->b_rptr;
				if((mp->b_wptr - mp->b_rptr) !=
						sizeof(struct T_discon_req))
					seterror(mp, tmp, TBADDATA, 0);
				else if(error = makedisconnect(loop, mp, &event))
					seterror(mp, tmp, hiword(error), loword(error));
				if(event == -1) event = TE_ERROR_ACK;
				break;

			case T_UNITDATA_REQ:
			case T_OPTMGMT_REQ:
			case T_ORDREL_REQ:
			/* The following messages should never occur! */
			case T_CONN_IND:
			case T_CONN_CON:
			case T_DISCON_IND:
			case T_DATA_IND:
			case T_EXDATA_IND:
			case T_INFO_ACK:
			case T_BIND_ACK:
			case T_ERROR_ACK:
			case T_OK_ACK:
			case T_UNITDATA_IND:
			case T_UDERROR_IND:
			case T_OPTMGMT_ACK:
			case T_ORDREL_IND:

				seterror(mp, tmp, TNOTSUPPORT, 0);
				break;
			} /* End switch */
			if(changestate(loop, event))
				panic("loopwsrv: can not change state\n");
			loop->lup_flags &= ~LOOP_CHST;
			wakeup(loop);
			qreply(q, tmp);
			freemsg(mp);
			break;
		} /* end case M_PROTO */
		} /* end switch */
	}
out1:
	if(mp) freemsg(mp);
out:
	return;
}

/*static*/ int
looprsrv(q)
queue_t	*q;
{
	struct loop *loop;

	loop = (struct loop *)q->q_ptr;
	if(loop->oqptr == NULL) return;
	qenable(WR(loop->oqptr));
}

/*static*/ int
loopclose(q)
queue_t	*q;
{
	register struct loop *loop;
	int	event;

	loop = (struct loop *)q->q_ptr;
	ASSERT(loop->lup_flags & LOOP_INUSE);
	loop->qptr = NULL;
	/*
	 * If we are connected to another stream, break the linkage.
	 * The disconnection is not well developed.
	 */

	if(loop->oqptr) {
		ASSERT(loop->lup_flags & LOOP_CONNECTED);
		ASSERT(loop->lup_flags & LOOP_BOUND);
		while(loop->lup_flags & LOOP_CHST)
			sleep(loop, PLOOP);
		loop->lup_flags |= LOOP_CHST;
		if(changestate(loop, TE_DISCON_REQ))
			printf("loopclose: changestate error for T_DISCON_IND\n");
		else if(makedisconnect(loop, NULL, &event))
			printf("loopclose: changestate error for makedisconnect\n");
		else if(changestate(loop,TE_OK_ACK1))
			printf("loopclose: changestate error for T_DISCON_IND\n");
		else if(changestate(loop, TE_UNBIND_REQ))
			printf("loopclose: changestate error for T_UNBND_REQ\n");
		else {
			/* Some more checks would be needed
			   before unbinding */
			loop->lup_flags &= ~(LOOP_BOUND|LOOP_CONNECTED);
			loop->lup_baddr[PORTADDR] = 0;
			changestate(loop,TE_OK_ACK1);
			((struct loop *)loop->oqptr->q_ptr)->oqptr = NULL;
			((struct loop *)loop->oqptr->q_ptr)->qptr = NULL;
			loop->oqptr = NULL;
		}
		loop->lup_flags &= ~LOOP_CHST;
		wakeup(loop);
	}
	loop->lup_flags = 0;
}

/* Sets an error reply */
/*static*/ void
seterror(mp, tmp, tlierror, unixerror)
mblk_t	*mp, *tmp;
{
	register union T_primitives *pptr;
	register struct T_error_ack *ackp;

	pptr = (union T_primitives *)mp->b_rptr;
	ackp = (struct T_error_ack *)tmp->b_rptr;
	ackp->PRIM_type = T_ERROR_ACK;
	ackp->ERROR_prim = pptr->type;
	ackp->TLI_error = tlierror;
	ackp->UNIX_error = unixerror;
	tmp->b_wptr = tmp->b_rptr + sizeof(struct T_error_ack);
	tmp->b_datap->db_type = M_PCPROTO;
}

/* Sets an ok reply */
/*static*/ void
setok(mp, tmp)
mblk_t	*mp, *tmp;
{
	register union T_primitives *pptr;
	register struct T_ok_ack *ackp;

	pptr = (union T_primitives *)mp->b_rptr;
	ackp = (struct T_ok_ack *)tmp->b_rptr;
	ackp->PRIM_type = T_OK_ACK;
	ackp->CORRECT_prim = pptr->type;
	tmp->b_wptr = tmp->b_rptr + sizeof(struct T_ok_ack);
	tmp->b_datap->db_type = M_PCPROTO;
}

/*
 * Used to check if an address is already in use while binding
 * If the address is in use -1 is returned, else the address.
 */
/*static*/ int
checkinuse(addr, len)
register unsigned addr;
register int len;
{
	register struct loop *loop;

	if(len != 4) return(-1);
		/* For now binding can be on a 4 byte address w.r.t
		   local machine (port!). This can change later on */
	for(loop = &loop_loop[0]; loop < &loop_loop[loop_cnt]; loop++)
		if((loop->lup_flags & LOOP_INUSE) && (loop->lup_baddr[PORTADDR] == addr))
			return(-1);
	return(addr);
}

/*
 * Allocate an used address for binding. The fields in the message are
 * directly updated. For now there is a bound for the search that is made.
 * It may not be there forever!!
 */
/*static*/ int
allocaddr()
{
	register unsigned addr;
	register struct loop *loop;
	register inuse;

	for(addr = LOOP_MINPORTNO; addr <= LOOP_MAXPORTNO; addr++) {
		inuse = 0;
		for(loop = &loop_loop[0]; loop < &loop_loop[loop_cnt]; loop++)
			if((loop->lup_flags & LOOP_INUSE) && (loop->lup_baddr[PORTADDR] == addr)) {
				inuse++;
				break;
			}
		if(!inuse) return(addr);
	}
	return(-1);
}

/* Send a connect indication on the destination message que */
/*static*/ int
send_conn_ind(srcloop, mp, destADDR, destLEN)
struct loop *srcloop;
mblk_t	*mp;
register unsigned destADDR;
int destLEN;
{
	register struct loop *destloop;
	register mblk_t	*tmp;
	register struct T_conn_ind *msgp;
	int	retval;

	for(destloop = &loop_loop[0]; destloop < &loop_loop[loop_cnt]; destloop++)
		if((destloop->lup_flags & LOOP_INUSE) &&
			(destloop->lup_baddr[PORTADDR] == destADDR))
			break;
	if(destloop >= &loop_loop[loop_cnt]) return(makerr(TBADADDR,0));
	if((tmp = allocb(sizeof(*msgp)+LOOP_MAXADDRSIZE, BPRI_MED)) == NULL)
		return(makerr(TSYSERR,EAGAIN));
	while(destloop->lup_flags & LOOP_CHST)
		sleep(destloop, PLOOP);
	destloop->lup_flags |= LOOP_CHST;
	if(destloop->lup_qlen >= destloop->lup_aqlen) {
		retval = makerr(TSYSERR,EAGAIN);
		goto err;
	} else
		destloop->lup_qlen++;
	if(retval = changestate(destloop, TE_CONN_IND))
		goto err;
	msgp = (struct T_conn_ind *)tmp->b_rptr;
	msgp->PRIM_type = T_CONN_IND;
	msgp->SRC_length = 4;	/* HARD-WIRED */
	msgp->SRC_offset = sizeof(*msgp);
	msgp->OPT_length = 0;	/* Not supported */
	msgp->OPT_offset = 0;
	msgp->SEQ_number = (int)(srcloop - loop_loop);
	tmp->b_wptr = tmp->b_rptr + sizeof(*msgp);
	*(int *)tmp->b_wptr++ = srcloop->lup_baddr[PORTADDR];
	tmp->b_datap->db_type = M_PROTO;
	/* Future plans for adhering to flow control!! */
	putnext(RD(destloop->qptr), tmp);
	destloop->lup_flags &= ~LOOP_CHST;
	wakeup(destloop);
	return(0);
err:
	destloop->lup_flags &= ~LOOP_CHST;
	wakeup(destloop);
	freemsg(tmp);
	return(retval);
}

/*
 * makeconnect : connects the two streams so that data transfer can continue.
 *		If connection succeeds a message is sent to the connected
 *		channel. It is upto the caller to update the qlen.
 */
/*static*/ int
makeconnect(msgloop, mp, rtneventp)
struct loop *msgloop;	/* Channel in which msg arrived	*/
mblk_t	*mp;		/* Acceptance message		*/
uint	*rtneventp;	/* Event to be considered for msgloop */
{
	register struct T_conn_con *conptr;
	register struct T_conn_res *resptr;
	register struct loop *srcloop;	/* Channel of acceptance	*/
	register struct loop *destloop;	/* Channel that requested conn.	*/
	mblk_t	*tmp;
	queue_t	*accq;			/* Q for srcloop		*/
	int	s;
	
	resptr = (struct T_conn_res *)mp->b_rptr;
	if(resptr->SEQ_number < 0 || resptr->SEQ_number > loop_cnt)
		return(makerr(TBADDATA,0));
	destloop = &loop_loop[resptr->SEQ_number];
	if(!(destloop->lup_flags & LOOP_INUSE))
		return(makerr(TBADDATA,0));
	accq = resptr->QUEUE_ptr;
	srcloop = (struct loop *)accq->q_ptr;
	if(srcloop < &loop_loop[0] || srcloop >= &loop_loop[loop_cnt])
		return(makerr(TBADDATA,0));
	if(WR(accq) != srcloop->qptr)
		return(makerr(TBADDATA,0));
	if(!(srcloop->lup_flags & LOOP_INUSE) || !(srcloop->lup_flags & LOOP_BOUND) ||
		(srcloop->lup_flags & LOOP_CONNECTED) || (destloop->lup_flags & LOOP_CONNECTED))
		return(makerr(TBADDATA,0));
	if(msgloop == srcloop && msgloop->lup_qlen > 1)
		return(makerr(TBADF,0));
	if((tmp = allocb(sizeof(*conptr)+LOOP_MAXADDRSIZE, BPRI_MED)) == NULL)
		return(makerr(TSYSERR,EAGAIN));
	while(srcloop->lup_flags & LOOP_CHST)
		sleep(srcloop, PLOOP);
	srcloop->lup_flags |= LOOP_CHST;
	/* There can be a deadlock here!! */
	while(destloop->lup_flags & LOOP_CHST)
		sleep(destloop, PLOOP);
	destloop->lup_flags |= LOOP_CHST;
	if(srcloop == msgloop) *rtneventp = TE_OK_ACK2;
	else {
		if((s = changestate(srcloop, TE_PASS_CONN)) ||
			(s = changestate(destloop, TE_CONN_CON))) {
			freemsg(tmp);
			srcloop->lup_flags &= ~LOOP_CHST;
			destloop->lup_flags &= ~LOOP_CHST;
			wakeup(srcloop);
			wakeup(destloop);
			return(s);
		}
		*rtneventp = (msgloop->lup_qlen == 1)? TE_OK_ACK3: TE_OK_ACK4;
	}
	conptr = (struct T_conn_con *)tmp->b_rptr;
	conptr->PRIM_type = T_CONN_CON;
	conptr->RES_length = sizeof(long);	/* HARD-WIRED */
	conptr->RES_offset = sizeof(*conptr);
	conptr->OPT_length = 0;	/* Not supported */
	conptr->OPT_offset = 0;
	tmp->b_wptr = tmp->b_rptr + sizeof(*conptr);
	*(int *)tmp->b_wptr++ = srcloop->lup_baddr[PORTADDR];
	tmp->b_datap->db_type = M_PROTO;
	/* Future plans for adhering to flow control!! */
	putnext(RD(destloop->qptr), tmp);
	srcloop->oqptr = RD(destloop->qptr);
	destloop->oqptr = RD(srcloop->qptr);
	srcloop->lup_flags |= LOOP_CONNECTED;
	destloop->lup_flags |= LOOP_CONNECTED;
	destloop->lup_flags &= ~LOOP_CHST;
	srcloop->lup_flags &= ~LOOP_CHST;
	wakeup(srcloop);
	wakeup(destloop);
	return(0);
}

/*
 * Changestate: Changes the state of the provider. If a valid transition
 *		return 0, else return an integer with t_errno in upper
 *		two bytes and errno in lower two bytes.
 *		All kinds of out the order behaviour needs to be taken
 *		care of in this routine!
 */
changestate(loop, event)
register struct loop *loop;
register unsigned event;
{
	extern unsigned char ti_statetbl[TE_NOEVENTS][TS_NOSTATES];
	register uint newstate;

	if(!(loop->lup_flags & LOOP_CHST))
		panic("loopdriver: bad change of state\n");
	if(event >= TE_NOEVENTS || loop->lup_state >= TS_NOSTATES)
		return(makerr(TBADDATA,0));
	newstate = ti_statetbl[event][loop->lup_state];
	if(newstate >= TS_NOSTATES) {
		/* All exceptions!! */
		/*1*/ if((event == TE_DATA_REQ || event == TE_EXDATA_REQ) &&
			loop->lup_state != TS_DATA_XFER && loop->lup_state != TS_IDLE)
			return(makerr(TSYSERR, EPROTO));
		return(makerr(TOUTSTATE,0));
	}
	loop->lup_state = newstate;
	return(0);
}

/*
 * Makedisconnect: Processing related to disconnect request.
 *	This request may come after/while data-xfer is going on or
 *	it may be a response to connect indication. Acknowledgement
 *	event would be different depending on this.
 */
/*static*/ int
makedisconnect(srcloop, mp, rtneventp)
register struct loop *srcloop;	/* Channel in which msg arrived	*/
mblk_t	*mp;		/* Acceptance message		*/
uint	*rtneventp;	/* Event to be considered for msgloop */
{
	register struct T_discon_ind *dindptr;
	register struct T_discon_req *dreqptr;
	register struct loop *destloop; 	/* Send indication to */
	register uint rtnevent;
	mblk_t	*tmp;
	int	s;
	
	if(mp)
		dreqptr = (struct T_discon_req *)mp->b_rptr;
	else dreqptr = NULL;
	rtnevent = TE_ERROR_ACK; /* default */
	if((tmp = allocb(sizeof(*dindptr), BPRI_MED)) == NULL)
		return(makerr(TSYSERR,EAGAIN));
	switch(srcloop->lup_state) {
	default:
	case TS_WACK_DREQ6:
		/* Waiting T_CONN_CON */
	case TS_WACK_DREQ10:
	case TS_WACK_DREQ11:
		freemsg(tmp);
		/* Don't know yet, what to do */
		return;

	case TS_WACK_DREQ7:
		/* In response to conn_ind */
		if(!dreqptr)
			return(makerr(TOUTSTATE,0));
		if(dreqptr->SEQ_number < 0 || dreqptr->SEQ_number > loop_cnt)
			return(makerr(TBADDATA,0));
		destloop = &loop_loop[dreqptr->SEQ_number];
		if(!(destloop->lup_flags & LOOP_INUSE) ||
			destloop->lup_state != TS_WCON_CREQ)
			return(makerr(TBADDATA,0));
		rtnevent = (srcloop->lup_qlen == 1)? TE_OK_ACK3: TE_OK_ACK4;
		break;

	case TS_WACK_DREQ9:
		ASSERT(srcloop->oqptr != NULL);
		destloop = (struct loop *)(srcloop->oqptr->q_ptr);
		if(destloop < &loop_loop[0] || destloop >= &loop_loop[loop_cnt])
			return(makerr(TBADDATA,0));
		rtnevent = TE_OK_ACK1;
		break;
	}
	/* There can be a deadlock here!! */
	while(destloop->lup_flags & LOOP_CHST)
		sleep(destloop, PLOOP);
	destloop->lup_flags |= LOOP_CHST;
	switch(destloop->lup_state) {

	case TS_WRES_CIND:	/* Awaiting T_CONN_RES in this channel */
		s = changestate(destloop, (destloop->lup_qlen == 1) ?
					TE_DISCON_IND2 : TE_DISCON_IND3);
		break;

	default:	/* Hopefully wrong states won't creep in */
		s = changestate(destloop, TE_DISCON_IND1);
		break;
	}
	if(s) {
		freemsg(tmp);
		destloop->lup_flags &= ~LOOP_CHST;
		wakeup(destloop);
		return(s);
	}
	dindptr = (struct T_discon_ind *)tmp->b_rptr;
	dindptr->PRIM_type = T_DISCON_IND;
	dindptr->DISCON_reason = 0;	/* Not supported */
	dindptr->SEQ_number = (srcloop - loop_loop);
	tmp->b_wptr = tmp->b_rptr + sizeof(*dindptr);
	tmp->b_datap->db_type = M_PROTO;
	/* Future plans for adhering to flow control!! */
	putnext(RD(destloop->qptr), tmp);
	srcloop->lup_flags |= LOOP_DISCONNECTED;
	destloop->lup_flags |= LOOP_DISCONNECTED;
	destloop->lup_flags &= ~LOOP_CHST;
	wakeup(destloop);
	*rtneventp = rtnevent;
	return(0);
}
#endif	defined(SYSV) && defined(RFS)
